/*
    sequence.d -- Sequence routines.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

#undef endp

#define	endp(obje)	(type_of(endp_temp = (obje)) == t_cons ? \
			 FALSE : endp_temp == Cnil ? TRUE : \
			 (FEwrong_type_argument(Slist, endp_temp), FALSE))

#ifdef MTCL
#define endp_temp  clwp->lwp_endp_temp
#else
object endp_temp;
#endif MTCL

/*
	I know the following name is not good.
*/
object
alloc_simple_vector(int l, enum aelttype aet)
{
	object x;

	x = alloc_object(t_vector);
	x->v.v_hasfillp = FALSE;
	x->v.v_adjustable = FALSE;
	x->v.v_displaced = Cnil;
	x->v.v_dim = x->v.v_fillp = l;
	x->v.v_self = NULL;
	x->v.v_elttype = (short)aet;
	return(x);
}

object
alloc_simple_bitvector(int l)
{
	object x;

	x = alloc_object(t_bitvector);
	x->bv.bv_hasfillp = FALSE;
	x->bv.bv_adjustable = FALSE;
	x->bv.bv_displaced = Cnil;
	x->bv.bv_dim = x->bv.bv_fillp = l;
	x->bv.bv_offset = 0;
	x->bv.bv_self = NULL;
	return(x);
}

Lelt(int narg, object x, object i)
{
	check_arg(2);
	VALUES(0) = elt(x, fixint(i));
	RETURN(1);
}

object
elt(object seq, int index)
{
	int i;
	object l;

	if (index < 0)
		FEerror("Negative index: ~D.", 1, MAKE_FIXNUM(index));
	switch (type_of(seq)) {
	case t_cons:
		for (i = index, l = seq;  i > 0;  --i)
			if (endp(l))
				goto E;
			else
				l = CDR(l);
		if (endp(l))
			goto E;
		return(CAR(l));

	case t_vector:
	case t_bitvector:
		if (index >= seq->v.v_fillp)
			goto E;
		return(aref(seq, index));

	case t_string:
		if (index >= seq->st.st_fillp)
			goto E;
		return(code_char(seq->ust.ust_self[index]));

	default:
		FEerror("~S is not a sequence.", 1, seq);
	}

E:
	FEerror("The index, ~D, is too large", 1, MAKE_FIXNUM(index));
}

siLelt_set(int narg, object seq, object index, object val)
{
	check_arg(3);
	VALUES(0) = elt_set(seq, fixint(index), val);
	RETURN(1);
}

object
elt_set(object seq, int index, object val)
{
	int i;
	object l;

	if (index < 0)
		FEerror("Negative index: ~D.", 1, MAKE_FIXNUM(index));
	switch (type_of(seq)) {
	case t_cons:
		for (i = index, l = seq;  i > 0;  --i)
			if (endp(l))
				goto E;
			else
				l = CDR(l);
		if (endp(l))
			goto E;
		return(CAR(l) = val);

	case t_vector:
	case t_bitvector:
		if (index >= seq->v.v_fillp)
			goto E;
		return(aset(seq, index, val));

	case t_string:
		if (index >= seq->st.st_fillp)
			goto E;
		if (type_of(val) != t_character)
			FEerror("~S is not a character.", 1, val);
		seq->st.st_self[index] = char_code(val);
		return(val);

	default:
		FEerror("~S is not a sequence.", 1, seq);
	}

E:
	FEerror("The index, ~D, is too large", 1, MAKE_FIXNUM(index));
}

@(defun subseq (sequence start &optional end &aux x)
	int s, e;
	int i, j;
@
	s = fixnnint(start);
	if (Null(end))
		e = -1;
	else
		e = fixnnint(end);
	switch (type_of(sequence)) {
	case t_symbol:
		if (Null(sequence)) {
			if (s > 0)
				goto ILLEGAL_START_END;
			if (e > 0)
				goto ILLEGAL_START_END;
			@(return Cnil)
		}
		FEwrong_type_argument(Ssequence, sequence);

	case t_cons:
		if (e >= 0)
			if ((e -= s) < 0)
				goto ILLEGAL_START_END;
		while (s-- > 0) {
			if (type_of(sequence) != t_cons)
				goto ILLEGAL_START_END;
			sequence = CDR(sequence);
		}
		if (e < 0)
			@(return `copy_list(sequence)`)
		{ object *z = &x;
		  for (i = 0;  i < e;  i++) {
		    if (type_of(sequence) != t_cons)
		      goto ILLEGAL_START_END;
		    z = &CDR(*z = CONS(CAR(sequence), Cnil));
		    sequence = CDR(sequence);
		  }
		}
		@(return x)

	case t_vector:
		if (s > sequence->v.v_fillp)
			goto ILLEGAL_START_END;
		if (e < 0)
			e = sequence->v.v_fillp;
		else if (e < s || e > sequence->v.v_fillp)
			goto ILLEGAL_START_END;
		x = alloc_simple_vector(e - s, sequence->v.v_elttype);
		array_allocself(x, FALSE);
		switch ((enum aelttype)sequence->v.v_elttype) {
		case aet_object:
		case aet_fix:
		case aet_sf:
			for (i = s, j = 0;  i < e;  i++, j++)
				x->v.v_self[j] = sequence->v.v_self[i];
			break;

		case aet_lf:
			for (i = s, j = 0;  i < e;  i++, j++)
				x->lfa.lfa_self[j] =
				sequence->lfa.lfa_self[i];
			break;
		}
		@(return x)

	case t_string:
		if (s > sequence->st.st_fillp)
			goto ILLEGAL_START_END;
		if (e < 0)
			e = sequence->st.st_fillp;
		else if (e < s || e > sequence->st.st_fillp)
			goto ILLEGAL_START_END;
		x = alloc_simple_string(e - s);
		x->st.st_self = alloc_contblock(e - s + 1);
		x->st.st_self[e-s] = '\0';
		for (i = s, j = 0;  i < e;  i++, j++)
			x->st.st_self[j] = sequence->st.st_self[i];
		@(return x)

	case t_bitvector:
		if (s > sequence->bv.bv_fillp)
			goto ILLEGAL_START_END;
		if (e < 0)
			e = sequence->bv.bv_fillp;
		else if (e < s || e > sequence->bv.bv_fillp)
			goto ILLEGAL_START_END;
		x = alloc_simple_bitvector(e - s);
		x->bv.bv_self = alloc_relblock((e-s+CHAR_BIT-1)/CHAR_BIT,
					       sizeof(char));
		s += sequence->bv.bv_offset;
		e += sequence->bv.bv_offset;
		for (i = s, j = 0;  i < e;  i++, j++)
			if (sequence->bv.bv_self[i/CHAR_BIT]&(0200>>i%CHAR_BIT))
				x->bv.bv_self[j/CHAR_BIT]
				|= 0200>>j%CHAR_BIT;
			else
				x->bv.bv_self[j/CHAR_BIT]
				&= ~(0200>>j%CHAR_BIT);
		@(return x)

	default:
		FEwrong_type_argument(Ssequence, x);
	}

ILLEGAL_START_END:
	FEerror("~S and ~S are illegal as :START and :END~%\
for the sequence ~S.", 3, start, end, sequence);
@)

Lcopy_seq(int narg, object x)
{
	check_arg(1);
	Lsubseq(2, x, MAKE_FIXNUM(0));
	RETURN(1);
}

int
length(object x)
{
	int i;

	switch (type_of(x)) {
	case t_symbol:
		if (Null(x))
			return(0);
		FEwrong_type_argument(Ssequence, x);

	case t_cons:
		for (i = 0;  !endp(x);  i++, x = CDR(x))
			;
		return(i);

	case t_vector:
	case t_string:
	case t_bitvector:
		return(x->v.v_fillp);

	default:
		FEwrong_type_argument(Ssequence, x);
	}
}

Llength(int narg, object x)
{
	check_arg(1);
	VALUES(0) = MAKE_FIXNUM(length(x));
	RETURN(1);
}

Lreverse(int narg, object x)
{
	check_arg(1);
	VALUES(0) = reverse(x);
	RETURN(1);
}

object
reverse(object seq)
{
	object x, y, v;
	int i, j, k;

	switch (type_of(seq)) {
	case t_symbol:
		if (Null(seq))
			return(Cnil);
		FEwrong_type_argument(Ssequence, seq);

	case t_cons:
		v = Cnil;
		for (x = seq;  !endp(x);  x = CDR(x))
			v = CONS(CAR(x), v);
		return(v);

	case t_vector:
		x = seq;
		k = x->v.v_fillp;
		y = alloc_simple_vector(k, x->v.v_elttype);
		array_allocself(y, FALSE);
		switch ((enum aelttype)x->v.v_elttype) {
		case aet_object:
		case aet_fix:
		case aet_sf:
			for (j = k - 1, i = 0;  j >=0;  --j, i++)
				y->v.v_self[j] = x->v.v_self[i];
			break;

		case aet_lf:
			for (j = k - 1, i = 0;  j >=0;  --j, i++)
				y->lfa.lfa_self[j] = x->lfa.lfa_self[i];
			break;
		}
		return(y);

	case t_string:
		x = seq;
		y = alloc_simple_string(x->st.st_fillp);
		y->st.st_self = alloc_contblock(x->st.st_fillp+1);
		for (j = x->st.st_fillp, i = 0;  j >=0;  --j, i++)
			y->st.st_self[j] = x->st.st_self[i];
		y->st.st_self[x->st.st_fillp] = '\0';
		return(y);

	case t_bitvector:
		x = seq;
		y = alloc_simple_bitvector(x->bv.bv_fillp);
		y->bv.bv_self
		  = alloc_relblock((x->bv.bv_fillp+CHAR_BIT-1)/CHAR_BIT,
				   sizeof(char));
		for (j = x->bv.bv_fillp - 1, i = x->bv.bv_offset;
		     j >=0;
		     --j, i++)
			if (x->bv.bv_self[i/CHAR_BIT]&(0200>>i%CHAR_BIT))
				y->bv.bv_self[j/CHAR_BIT] |= 0200>>j%CHAR_BIT;
			else
				y->bv.bv_self[j/CHAR_BIT] &= ~(0200>>j%CHAR_BIT);
		return(v);

	default:
		FEwrong_type_argument(Ssequence, seq);
	}
}

Lnreverse(int narg, object x)
{
	check_arg(1);
	VALUES(0) = nreverse(x);
	RETURN(1);
}

object
nreverse(object seq)
{
	object x, y, z;
	int i, j, k;

	switch (type_of(seq)) {
	case t_symbol:
		if (Null(seq))
			return(Cnil);
		FEwrong_type_argument(Ssequence, seq);

	case t_cons:
		for (x = Cnil, y = seq;  !endp(CDR(y));) {
			z = y;
			y = CDR(y);
			CDR(z) = x;
			x = z;
		}
		CDR(y) = x;
		return(y);

	case t_vector:
		x = seq;
		k = x->v.v_fillp;
		switch ((enum aelttype)x->v.v_elttype) {
		case aet_object:
		case aet_fix:
		case aet_sf:
			for (i = 0, j = k - 1;  i < j;  i++, --j) {
				y = x->v.v_self[i];
				x->v.v_self[i] = x->v.v_self[j];
				x->v.v_self[j] = y;
			}
			return(seq);

		case aet_lf:
			for (i = 0, j = k - 1;  i < j;  i++, --j) {
				longfloat y;
				y = x->lfa.lfa_self[i];
				x->lfa.lfa_self[i] = x->lfa.lfa_self[j];
				x->lfa.lfa_self[j] = y;
			}
			return(seq);
		}

	case t_string:
		x = seq;
		for (i = 0, j = x->st.st_fillp - 1;  i < j;  i++, --j) {
			k = x->st.st_self[i];
			x->st.st_self[i] = x->st.st_self[j];
			x->st.st_self[j] = k;
		}
		return(seq);

	case t_bitvector:
		x = seq;
		for (i = x->bv.bv_offset,
		     j = x->bv.bv_fillp + x->bv.bv_offset - 1;
		     i < j;
		     i++, --j) {
			k = x->bv.bv_self[i/CHAR_BIT]&(0200>>i%CHAR_BIT);
			if (x->bv.bv_self[j/CHAR_BIT]&(0200>>j%CHAR_BIT))
				x->bv.bv_self[i/CHAR_BIT]
				|= 0200>>i%CHAR_BIT;
			else
				x->bv.bv_self[i/CHAR_BIT]
				&= ~(0200>>i%CHAR_BIT);
			if (k)
				x->bv.bv_self[j/CHAR_BIT]
				|= 0200>>j%CHAR_BIT;
			else
				x->bv.bv_self[j/CHAR_BIT]
				&= ~(0200>>j%CHAR_BIT);
		}
		return(seq);

	default:
		FEwrong_type_argument(Ssequence, seq);
	}
}


init_sequence_function()
{
	make_function("ELT", Lelt);
	make_si_function("ELT-SET", siLelt_set);
	make_function("SUBSEQ", Lsubseq);
	make_function("COPY-SEQ", Lcopy_seq);
	make_function("LENGTH", Llength);
	make_function("REVERSE", Lreverse);
	make_function("NREVERSE", Lnreverse);
}

