/*
    package.d -- Packages.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

object lisp_package;
object user_package;
object keyword_package;
object system_package;
#ifdef CLOS
object clos_package;
#endif
object Vpackage;		/*  *package*  */

#ifndef MTCL
int intern_flag;
#endif

object uninterned_list;

/******************************* ------- ******************************/

#define	HASHCOEF	12345		/*  hashing coefficient  */

#define	INTERNAL	1
#define	EXTERNAL	2
#define	INHERITED	3

object Kinternal;
object Kexternal;
object Kinherited;
object Knicknames;
object Kuse;

bool
member_string_equal(x, l)
object x, l;
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (string_equal(x, CAR(l)))
			return(TRUE);
	return(FALSE);
}

/*
	Make_package(n, ns, ul) makes a package with name n,
	which must be a string or a symbol,
	and nicknames ns, which must be a list of strings or symbols,
	and uses packages in list ul, which must be a list of packages
	or package names i.e. strings or symbols.
*/
object
make_package(object n, object ns, object ul)
{
	object x, y;
	int i;

	if (type_of(n) == t_symbol) {
		y = alloc_simple_string(n->s.s_fillp);
		y->st.st_self = n->s.s_self;
		n = y;
	}
	if (find_package(n) != Cnil)
		package_already(n);
	x = alloc_object(t_package);
	x->p.p_name = n;
	x->p.p_nicknames = Cnil;
	x->p.p_shadowings = Cnil;
	x->p.p_uselist = Cnil;
	x->p.p_usedbylist = Cnil;
	x->p.p_internal = NULL;
	x->p.p_external = NULL;
	for (;  !endp(ns);  ns = CDR(ns)) {
		n = CAR(ns);
		if (type_of(n) == t_symbol) {
			y = alloc_simple_string(n->s.s_fillp);
			y->st.st_self = n->s.s_self;
			n = y;
		}
		if (find_package(n) != Cnil)
			package_already(n);
		x->p.p_nicknames = CONS(n, x->p.p_nicknames);
	}
	for (;  !endp(ul);  ul = CDR(ul)) {
		if (type_of(CAR(ul)) == t_package)
			y = CAR(ul);
		else {
			y = find_package(CAR(ul));
			if (Null(y))
				no_package(CAR(ul));
		}
		x->p.p_uselist = CONS(y, x->p.p_uselist);
		y->p.p_usedbylist = CONS(x, y->p.p_usedbylist);
	}
	x->p.p_internal
	= (object *)alloc_contblock(PHTABSIZE * sizeof(object));
	for (i = 0;  i < PHTABSIZE;  i++)
		x->p.p_internal[i] = Cnil;
	x->p.p_external
	= (object *)alloc_contblock(PHTABSIZE * sizeof(object));
	for (i = 0;  i < PHTABSIZE;  i++)
		x->p.p_external[i] = Cnil;
	x->p.p_link = pack_pointer;
	pack_pointer = &(x->p);
	return(x);
}

object
in_package(object n, object ns, object ul)
{
	object x, y;

	x = find_package(n);
	if (Null(x))
		x = make_package(n, ns, ul);
	else {
		for (;  !endp(ns);  ns = CDR(ns)) {
			n = CAR(ns);
			if (type_of(n) == t_symbol) {
				y = alloc_simple_string(n->s.s_fillp);
				y->st.st_self = n->s.s_self;
				n = y;
			}
			y = find_package(n);
			if (x == y)
				continue;
			if (y != Cnil)
				package_already(n);
			x->p.p_nicknames = CONS(n, x->p.p_nicknames);
		}
		for (;  !endp(ul);  ul = CDR(ul))
			use_package(CAR(ul), x);
	}
	Vpackage->s.s_dbind = x;
	return(x);
}

object
rename_package(object x, object n, object ns)
{
	object y;

	if (type_of(n) == t_symbol) {
		y = alloc_simple_string(n->s.s_fillp);
		y->st.st_self = n->s.s_self;
		n = y;
	}
	if (find_package(n) != Cnil)
		package_already(n);
	x->p.p_name = n;
	x->p.p_nicknames = Cnil;
	for (;  !endp(ns);  ns = CDR(ns)) {
		n = CAR(ns);
		if (type_of(n) == t_symbol) {
			y = alloc_simple_string(n->s.s_fillp);
			y->st.st_self = n->s.s_self;
			n = y;
		}
		y = find_package(n);
		if (x == y)
			continue;
		if (y != Cnil)
			package_already(n);
		x->p.p_nicknames = CONS(n, x->p.p_nicknames);
	}
	return(x);
}

/*
	Find_package(n) seaches for a package with name n,
	which is a string or a symbol.
	If not so, an error is signaled.
*/
object
find_package(object n)
{
	struct package *p;

	if ((type_of(n) != t_symbol) && (type_of(n) != t_string))
		FEwrong_type_argument(TSor_string_symbol, n);
	for (p = pack_pointer;  p != NULL;  p = p->p_link) {
		if (string_equal(p->p_name, n))
			return((object)p);
		if (member_string_equal(n, p->p_nicknames))
			return((object)p);
	}
	return(Cnil);
}

object
coerce_to_package(object p)
{
	object pp;

	if (type_of(p) == t_package)
		return(p);
	pp = find_package(p);
	if (Null(pp))
		no_package(p);
	return(pp);
}

object
current_package()
{
	object x;

	x = symbol_value(Vpackage);
	if (type_of(x) != t_package) {
		Vpackage->s.s_dbind = user_package;
		FEerror("The value of *PACKAGE*, ~S, was not a package.",
			1, x);
	}
	return(x);
}

/*
	Pack_hash(st) hashes string st
	and returns the index for a hash table of a package.
*/
int
pack_hash(object st)
{
	int h, i, l = st->st.st_fillp;
	char *str = st->st.st_self;

	for (h = 0, i = 0;  i < l;  i++)
		h += (str[i] & 0377) * HASHCOEF + 1;
	h &= 0x7fffffff;
	return(h %= PHTABSIZE);
}

/*
	Intern(st, p) interns string st in package p.
*/
object
intern(char *st, object p)
{
	int j, len;
	object x, *ip, *ep, l, ul;

	setup_string_register(st);
	j = pack_hash(string_register);
	ip = &p->p.p_internal[j];
	for (l = *ip;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register)) {
			intern_flag = INTERNAL;
			return(CAR(l));
		}
	ep = &p->p.p_external[j];
	for (l = *ep;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register)) {
			intern_flag = EXTERNAL;
			return(CAR(l));
		}
	for (ul=p->p.p_uselist; type_of(ul)==t_cons; ul = CDR(ul))
		for (l = CAR(ul)->p.p_external[j];
		     type_of(l) == t_cons;
		     l = CDR(l))
			if (string_eq(CAR(l), string_register)) {
				intern_flag = INHERITED;
				return(CAR(l));
			}
	x = make_symbol(string_register);
	if (p == keyword_package) {
		x->s.s_stype = (short)stp_constant;
		x->s.s_dbind = x;
		*ep = CONS(x, *ep);
		intern_flag = 0;
	} else {
		*ip = CONS(x, *ip);
		intern_flag = 0;
	}
	if (Null(x->s.s_hpack))
		x->s.s_hpack = p;
	return(x);
}

/*
	Find_symbol(st, len, p) searches for string st of length len in package p.
*/
object
find_symbol(char *st, int len, object p)
{
	int j;
	object *ip, *ep, l, ul;
	
	setup_string_register(st);
	j = pack_hash(string_register);
	ip = &p->p.p_internal[j];
	for (l = *ip;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register)) {
			intern_flag = INTERNAL;
			return(CAR(l));
		}
	ep = &p->p.p_external[j];
	for (l = *ep;  type_of(l) == t_cons;  l = CDR(l))
		if (string_eq(CAR(l), string_register)) {
			intern_flag = EXTERNAL;
			return(CAR(l));
		}
	for (ul=p->p.p_uselist; type_of(ul)==t_cons; ul = CDR(ul))
		for (l = CAR(ul)->p.p_external[j];
		     type_of(l) == t_cons;
		     l = CDR(l))
			if (string_eq(CAR(l), string_register)) {
				intern_flag = INHERITED;
				return(CAR(l));
			}
	intern_flag = 0;
	return(Cnil);
}

bool
unintern(object s, object p)
{
	object x, y, l, *lp;
	int j;

	j = pack_hash(s);
	x = find_symbol(s->st.st_self, s->st.st_fillp, p);
	if (intern_flag == INTERNAL && s == x) {
		lp = &p->p.p_internal[j];
		if (member_eq(s, p->p.p_shadowings))
			goto L;
		goto UNINTERN;
	}
	if (intern_flag == EXTERNAL && s == x) {
		lp = &p->p.p_external[j];
		if (member_eq(s, p->p.p_shadowings))
			goto L;
		goto UNINTERN;
	}
	return(FALSE);

L:
	x = OBJNULL;
	for (l = p->p.p_uselist; type_of(l) == t_cons; l = CDR(l)) {
		y = find_symbol(s->st.st_self, s->st.st_fillp, CAR(l));
		if (intern_flag == EXTERNAL) {
			if (x == OBJNULL)
				x = y;
			else if (x != y)
FEerror("Cannot unintern the shadowing symbol ~S~%\
from ~S,~%\
because ~S and ~S will cause~%\
a name conflict.", 4, s, p, x, y);
		}
	}
	delete_eq(s, &p->p.p_shadowings);

UNINTERN:
	delete_eq(s, lp);
	if (s->s.s_hpack == p)
		s->s.s_hpack = Cnil;
	if ((enum stype)s->s.s_stype != stp_ordinary)
		uninterned_list = CONS(s, uninterned_list);
	return(TRUE);
}

export(object s, object p)
{
	object x;
	int j;
	object *ep, *ip, l;

BEGIN:
	ip = NULL;
	j = pack_hash(s);
	x = find_symbol(s->st.st_self, s->st.st_fillp, p);
	if (intern_flag) {
		if (x != s) {
			import(s, p);	/*  signals an error  */
			goto BEGIN;
		}
		if (intern_flag == INTERNAL)
			ip = &p->p.p_internal[j];
		else if (intern_flag == EXTERNAL)
			return;
	} else
		FEerror("The symbol ~S is not accessible from ~S.", 2,
			s, p);
	for (l = p->p.p_usedbylist;
	     type_of(l) == t_cons;
	     l = CDR(l)) {
		x = find_symbol(s->st.st_self, s->st.st_fillp, CAR(l));
		if (intern_flag && s != x &&
		    !member_eq(x, CAR(l)->p.p_shadowings))
FEerror("Cannot export the symbol ~S~%\
from ~S,~%\
because it will cause a name conflict~%\
in ~S.", 3, s, p, CAR(l));
	}
	if (ip != NULL)
		delete_eq(s, ip);
	ep = &p->p.p_external[j];
	*ep = CONS(s, *ep);
}

unexport(object s, object p)
{
	object x, *ep, *ip;
	int j;

	if (p == keyword_package)
		FEerror("Cannot unexport a symbol from the keyword.", 0);
	x = find_symbol(s->st.st_self, s->st.st_fillp, p);
	if (intern_flag != EXTERNAL || x != s)
FEerror("Cannot unexport the symbol ~S~%\
from ~S,~%\
because the symbol is not an external symbol~%\
of the package.", 2, s, p);
	j = pack_hash(s);
	ep = &p->p.p_external[j];
	delete_eq(s, ep);
	ip = &p->p.p_internal[j];
	*ip = CONS(s, *ip);
}

import(object s, object p)
{
	object x;
	int j;
	object *ip;

	x = find_symbol(s->st.st_self, s->st.st_fillp, p);
	if (intern_flag) {
		if (x != s)
			FEerror("Cannot import the symbol ~S~%\
from ~S,~%\
because there is already a symbol with the same name~%\
in the package.", 2, s, p);
		if (intern_flag == INTERNAL || intern_flag == EXTERNAL)
			return;
	}
	j = pack_hash(s);
	ip = &p->p.p_internal[j];
	*ip = CONS(s, *ip);
	if (Null(s->s.s_hpack))
		s->s.s_hpack = p;
}

shadowing_import(object s, object p)
{
	object x, *ip;

	x = find_symbol(s->st.st_self, s->st.st_fillp, p);
	if (intern_flag && intern_flag != INHERITED) {
		if (x == s) {
			if (!member_eq(x, p->p.p_shadowings))
				p->p.p_shadowings
				= CONS(x, p->p.p_shadowings);
			return;
		}
		if(member_eq(x, p->p.p_shadowings))
			delete_eq(x, &p->p.p_shadowings);
		if (intern_flag == INTERNAL)
			delete_eq(x, &p->p.p_internal[pack_hash(x)]);
		else
			delete_eq(x, &p->p.p_external[pack_hash(x)]);
		if (x->s.s_hpack == p)
			x->s.s_hpack = Cnil;
		if ((enum stype)x->s.s_stype != stp_ordinary)
			uninterned_list = CONS(x, uninterned_list);
	}
	ip = &p->p.p_internal[pack_hash(s)];
	*ip = CONS(s, *ip);
	p->p.p_shadowings = CONS(s, p->p.p_shadowings);
}

shadow(object s, object p)
{
	int j;
	object *ip, x;

	find_symbol(s->st.st_self, s->st.st_fillp, p);
	if (intern_flag == INTERNAL || intern_flag == EXTERNAL)
		return;
	j = pack_hash(s);
	ip = &p->p.p_internal[j];
	x = make_symbol(s);
	x->s.s_hpack = p;
	*ip = CONS(x, *ip);
	p->p.p_shadowings = CONS(x, p->p.p_shadowings);
}

use_package(object x0, object p)
{
	object x = x0;
	int i;
	object y, l;

	if (type_of(x) != t_package) {
		x = find_package(x);
		if (Null(x))
			no_package(x0);
	}
	if (x == keyword_package)
		FEerror("Cannot use keyword package.", 0);
	if (p == x)
		return;
	if (member_eq(x, p->p.p_uselist))
		return;
	for (i = 0;  i < PHTABSIZE;  i++)
		for (l = x->p.p_external[i];
		     type_of(l) == t_cons;
		     l = CDR(l)) {
			 	y = CAR(l);
				y = find_symbol(y->st.st_self, y->st.st_fillp, p);
			if (intern_flag && CAR(l) != y)
FEerror("Cannot use ~S~%\
from ~S,~%\
because ~S and ~S will cause~%\
a name conflict.", 4, x, p, CAR(l), y);
		}
	p->p.p_uselist = CONS(x, p->p.p_uselist);
	x->p.p_usedbylist = CONS(p, x->p.p_usedbylist);
}

unuse_package(object x0, object p)
{
	object x = x0;

	if (type_of(x) != t_package) {
		x = find_package(x);
		if (Null(x))
			no_package(x0);
	}
	delete_eq(x, &p->p.p_uselist);
	delete_eq(p, &x->p.p_usedbylist);
}

@(defun make_package (pack_name
		      &key nicknames
			   (use `CONS(lisp_package, Cnil)`))
@
	check_type_or_string_symbol(&pack_name);
	@(return `make_package(pack_name, nicknames, use)`)
@)

@(defun in_package (pack_name &key nicknames (use Cnil use_sp))
@
	check_type_or_string_symbol(&pack_name);
	if (Null(find_package(pack_name)) && !(use_sp))
		use = CONS(lisp_package, Cnil);
	@(return `in_package(pack_name, nicknames, use)`)
@)

@(defun find_package (p)
@
	@(return `find_package(p)`)
@)

@(defun package_name (p)
@
	check_type_package(&p);
	@(return `p->p.p_name`)
@)

@(defun package_nicknames (p)
@
	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	@(return `p->p.p_nicknames`)
@)

@(defun rename_package (pack new_name &o new_nicknames)
@
	check_type_or_symbol_string_package(&pack);
	pack = coerce_to_package(pack);
	check_type_or_string_symbol(&new_name);
	@(return `rename_package(pack, new_name, new_nicknames)`)
@)

Lpackage_use_list(int narg, object p)
{
	check_arg(1);

	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	VALUES(0) = p->p.p_uselist;
	RETURN(1);
}

Lpackage_used_by_list(int narg, object p)
{
	check_arg(1);

	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	VALUES(0) = p->p.p_usedbylist;
	RETURN(1);
}

Lpackage_shadowing_symbols(int narg, object p)
{
	check_arg(1);

	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	VALUES(0) = p->p.p_shadowings;
	RETURN(1);
}

Llist_all_packages(int narg)
{
	struct package *p;
	object lp = Cnil;

	check_arg(0);
	for (p = pack_pointer;  p != NULL;  p = p->p_link)
		lp = CONS(p, lp);
	VALUES(0) = lp;
	RETURN(1);
}

@(defun intern (strng &optional (p `current_package()`) &aux sym)
@
	check_type_string(&strng);
	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	sym = intern(strng->st.st_self, p);
	if (intern_flag == INTERNAL)
		@(return sym Kinternal)
	if (intern_flag == EXTERNAL)
		@(return sym Kexternal)
	if (intern_flag == INHERITED)
		@(return sym Kinherited)
	@(return sym Cnil)
@)

@(defun find_symbol (strng &optional (p `current_package()`))
	object x;
@
	check_type_string(&strng);
	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	x = find_symbol(strng->st.st_self, strng->st.st_fillp, p);
	if (intern_flag == INTERNAL)
		@(return x Kinternal)
	if (intern_flag == EXTERNAL)
		@(return x Kexternal)
	if (intern_flag == INHERITED)
		@(return x Kinherited)
	@(return Cnil Cnil)
@)

@(defun unintern (symbl &optional (p `current_package()`))
@
	check_type_symbol(&symbl);
	check_type_or_symbol_string_package(&p);
	p = coerce_to_package(p);
	@(return `unintern(symbl, p) ? Ct : Cnil`)
@)

@(defun export (symbols &o (pack `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pack);
	pack = coerce_to_package(pack);
BEGIN:
	switch (type_of(symbols)) {
	case t_symbol:
		if (Null(symbols))
			break;
		export(symbols, pack);
		break;

	case t_cons:
		for (l = symbols;  !endp(l);  l = CDR(l))
			export(CAR(l), pack);
		break;

	default:
		check_type_symbol(&symbols);
		goto BEGIN;
	}
	@(return Ct)
@)

@(defun unexport (symbols &o (pack `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pack);
	pack = coerce_to_package(pack);
BEGIN:
	switch (type_of(symbols)) {
	case t_symbol:
		if (Null(symbols))
			break;
		unexport(symbols, pack);
		break;

	case t_cons:
		for (l = symbols;  !endp(l);  l = CDR(l))
			unexport(CAR(l), pack);
		break;

	default:
		check_type_symbol(&symbols);
		goto BEGIN;
	}
	@(return Ct)
@)

@(defun import (symbols &o (pack `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pack);
	pack = coerce_to_package(pack);
BEGIN:
	switch (type_of(symbols)) {
	case t_symbol:
		if (Null(symbols))
			break;
		import(symbols, pack);
		break;

	case t_cons:
		for (l = symbols;  !endp(l);  l = CDR(l))
			import(CAR(l), pack);
		break;

	default:
		check_type_symbol(&symbols);
		goto BEGIN;
	}
	@(return Ct)
@)

@(defun shadowing_import (symbols &o (pack `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pack);
	pack = coerce_to_package(pack);
BEGIN:
	switch (type_of(symbols)) {
	case t_symbol:
		if (Null(symbols))
			break;
		shadowing_import(symbols, pack);
		break;

	case t_cons:
		for (l = symbols;  !endp(l);  l = CDR(l))
			shadowing_import(CAR(l), pack);
		break;

	default:
		check_type_symbol(&symbols);
		goto BEGIN;
	}
	@(return Ct)
@)

@(defun shadow (symbols &o (pack `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pack);
	pack = coerce_to_package(pack);
BEGIN:
	switch (type_of(symbols)) {
	case t_symbol:
		if (Null(symbols))
			break;
		shadow(symbols, pack);
		break;

	case t_cons:
		for (l = symbols;  !endp(l);  l = CDR(l))
			shadow(CAR(l), pack);
		break;

	default:
		check_type_symbol(&symbols);
		goto BEGIN;
	}
	@(return Ct)
@)

@(defun use_package (pack &o (pa `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pa);
	pa = coerce_to_package(pa);
BEGIN:
	switch (type_of(pack)) {
	case t_symbol:
		if (Null(pack))
			break;

	case t_string:
	case t_package:
		use_package(pack, pa);
		break;

	case t_cons:
		for (l = pack;  !endp(l);  l = CDR(l))
			use_package(CAR(l), pa);
		break;

	default:
		check_type_package(&pack);
		goto BEGIN;
	}
	@(return Ct)
@)

@(defun unuse_package (pack &o (pa `current_package()`))
	object l;
@
	check_type_or_symbol_string_package(&pa);
	pa = coerce_to_package(pa);
BEGIN:
	switch (type_of(pack)) {
	case t_symbol:
		if (Null(pack))
			break;

	case t_string:
	case t_package:
		unuse_package(pack, pa);
		break;

	case t_cons:
		for (l = pack;  !endp(l);  l = CDR(l))
			unuse_package(CAR(l), pa);
		break;

	default:
		check_type_package(&pack);
		goto BEGIN;
	}
	@(return Ct)
@)

siLpackage_internal(int narg, object p, object index)
{
	int j;

	check_arg(2);
	check_type_package(&p);
	if (!FIXNUMP(index) ||
	    (j = fix(index)) < 0 || j >= PHTABSIZE)
		FEerror("~S is an illegal index to a package hashtable.",
			1, index);
	VALUES(0) = p->p.p_internal[j];
	RETURN(1);
}

siLpackage_external(int narg, object p, object index)
{
	int j;

	check_arg(2);
	check_type_package(&p);
	if (!FIXNUMP(index) ||
	    (j = fix(index)) < 0 || j >= PHTABSIZE)
		FEerror("~S is an illegal index to a package hashtable.",
			1, index);
	VALUES(0) = p->p.p_external[j];
	RETURN(1);
}

no_package(object n)
{
	FEerror("There is no package with the name ~A.", 1, n);
}

package_already(object n)
{
	FEerror("A package with the name ~A already exists.", 1, n);
}

siLpackage_size(int narg, object p)
{ 
  check_arg(1);
  check_type_package(&p);
  VALUES(0) = MAKE_FIXNUM(PHTABSIZE);
  VALUES(1) = MAKE_FIXNUM(PHTABSIZE);
  RETURN(2);
}

init_package()
{

	lisp_package
	= make_package(make_simple_string("LISP"),
		       Cnil, Cnil);
	user_package
	= make_package(make_simple_string("USER"),
		       Cnil,
		       CONS(lisp_package, Cnil));
	keyword_package
	= make_package(make_simple_string("KEYWORD"),
		       Cnil, Cnil);
	system_package
	= make_package(make_simple_string("SYSTEM"),
		       CONS(make_simple_string("SI"),
			         CONS(make_simple_string("SYS"),
					   Cnil)),
		       CONS(lisp_package, Cnil));
#ifdef CLOS
	clos_package
	= make_package(make_simple_string("CLOS"),
		       Cnil,
		       CONS(lisp_package, Cnil));
#endif

	Cnil->s.s_hpack = lisp_package;
	import(Cnil, lisp_package);
	export(Cnil, lisp_package);

	Ct->s.s_hpack = lisp_package;
	import(Ct, lisp_package);
	export(Ct, lisp_package);

	/*  There is no need to enter a package as a mark origin.  */

	Vpackage = make_special("*PACKAGE*", lisp_package);

	Kinternal = make_keyword("INTERNAL");
	Kexternal = make_keyword("EXTERNAL");
	Kinherited = make_keyword("INHERITED");
	Knicknames = make_keyword("NICKNAMES");
	Kuse = make_keyword("USE");

	uninterned_list = Cnil;
	enter_mark_origin(&uninterned_list);
}

init_package_function()
{
	make_function("MAKE-PACKAGE", Lmake_package);
	make_function("IN-PACKAGE", Lin_package);
	make_function("FIND-PACKAGE", Lfind_package);
	make_function("PACKAGE-NAME", Lpackage_name);
	make_function("PACKAGE-NICKNAMES", Lpackage_nicknames);
	make_function("RENAME-PACKAGE", Lrename_package);
	make_function("PACKAGE-USE-LIST", Lpackage_use_list);
	make_function("PACKAGE-USED-BY-LIST", Lpackage_used_by_list);
	make_function("PACKAGE-SHADOWING-SYMBOLS",
		      Lpackage_shadowing_symbols);
	make_function("LIST-ALL-PACKAGES", Llist_all_packages);
	make_function("INTERN", Lintern);
	make_function("FIND-SYMBOL", Lfind_symbol);
	make_function("UNINTERN", Lunintern);
	make_function("EXPORT", Lexport);
	make_function("UNEXPORT", Lunexport);
	make_function("IMPORT", Limport);
	make_function("SHADOWING-IMPORT", Lshadowing_import);
	make_function("SHADOW", Lshadow);
	make_function("USE-PACKAGE", Luse_package);
	make_function("UNUSE-PACKAGE", Lunuse_package);

	make_si_function("PACKAGE-SIZE", siLpackage_size);
	make_si_function("PACKAGE-INTERNAL", siLpackage_internal);
	make_si_function("PACKAGE-EXTERNAL", siLpackage_external);
}
