/*
    num_co.c -- Operations on floating-point numbers.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

/*
	IMPLEMENTATION-DEPENDENT

	This file contains those functions
	that know the representation of floating-point numbers.
*/

#include "config.h"
#include "num_include.h"

static object plus_half, minus_half;


#ifdef VAX
/*
	radix = 2

	SEEEEEEEEHHHHHHH	The redundant most significant fraction bit
	HHHHHHHHHHHHHHHH	is not expressed.
	LLLLLLLLLLLLLLLL
	LLLLLLLLLLLLLLLL
*/
#endif
#ifdef IEEEFLOAT
#if defined(ns32000) || defined(i386)
/*
	radix = 2

	LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL	The redundant most
	SEEEEEEEEEEEHHHHHHHHHHHHHHHHHHHH	significant fraction bit
						is not expressed.
*/
#else
/*
	radix = 2

	SEEEEEEEEEEEHHHHHHHHHHHHHHHHHHHH	The redundant most
	LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL	significant fraction bit
						is not expressed.
*/
#endif
#endif
#ifdef TAHOE
/*
        radix = 2

        SEEEEEEEEHHHHHHHHHHHHHHHHHHHHHHH       The redundant most significant
        LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL       fraction bit is not expressed.

*/
#endif
integer_decode_double(d, hp, lp, ep, sp)
double d;
int *hp, *lp, *ep, *sp;
{
	int h, l;

	if (d == 0.0) {
		*hp = *lp = 0;
		*ep = 0;
		*sp = 1;
		return;
	}
	h = *((int *)&d + HIND);
	l = *((int *)&d + LIND);
#ifdef VAX
	*ep = ((h >> 7) & 0xff) - 128 - 56;
	h = ((h >> 15) & 0x1fffe) | (((h & 0x7f) | 0x80) << 17);
	l = ((l >> 16) & 0xffff) | (l << 16);
#endif VAX
#ifdef IEEEFLOAT
	*ep = ((h & 0x7ff00000) >> 20) - 1022 - 53;
	h = (h & 0x000fffff | 0x00100000) << 1;
#endif IEEEFLOAT
#ifdef TAHOE
        *ep = ((h & 0x7f800000) >> 23) - 128 - 56;
        h = (h & 0x007fffff | 0x00800000) << 1;
#endif
	if (l < 0) {
		h++;
		l &= 0x7fffffff;
	}
	*hp = h;
	*lp = l;
	*sp = (d > 0.0 ? 1 : -1);
}

#ifdef VAX
/*
	radix = 2

	SEEEEEEEEMMMMMMM	The redundant most significant fraction bit
	MMMMMMMMMMMMMMMM	is not expressed.
*/
#endif VAX
#ifdef IEEEFLOAT
/*
	radix = 2

	SEEEEEEEEMMMMMMMMMMMMMMMMMMMMMMM	The redundant most
						significant fraction bit
						is not expressed.
*/
#endif IEEEFLOAT
#ifdef TAHOE
/*
        radix = 2
        
        SEEEEEEEEMMMMMMMMMMMMMMMMMMMMMMM        The redundant most significant
                                                fraction bit is not expressed.
*/
#endif
integer_decode_float(d, mp, ep, sp)
double d;
int *mp, *ep, *sp;
{
	float f;
	int m;

	f = d;
	if (f == 0.0) {
		*mp = 0;
		*ep = 0;
		*sp = 1;
		return;
	}
	m = *(int *)(&f);
#ifdef VAX
	*ep = ((m >> 7) & 0xff) - 128 - 24;
	*mp = ((m >> 16) & 0xffff) | (((m & 0x7f) | 0x80) << 16);
#endif VAX
#ifdef IEEEFLOAT
	*ep = ((m & 0x7f800000) >> 23) - 126 - 24;
	*mp = m & 0x007fffff | 0x00800000;
#endif IEEEFLOAT
#ifdef TAHOE
        *ep = ((m & 0x7f800000) >> 23) - 128 -24;
        *mp = m & 0x007fffff | 0x00800000;
#endif
	*sp = (f > 0.0 ? 1 : -1);
}

int
double_exponent(d)
double d;
{
	if (d == 0.0)
		return(0);
#ifdef VAX
	return(((*(int *)(&d) >> 7) & 0xff) - 128);
#endif VAX
#ifdef IEEEFLOAT
	return(((*((int *)(&d) + HIND) & 0x7ff00000) >> 20) - 1022);
#endif IEEEFLOAT
#ifdef TAHOE
        return(((*(int *)(&d) & 0x7f800000) >> 23) - 128);
#endif
}

double
set_exponent(d, e)
double d;
int e;
{
	double dummy;

	if (d == 0.0)
		return(0.0);
	*((int *)&d + HIND)
#ifdef VAX
	= *(int *)(&d) & 0xffff807f | ((e + 128) << 7) & 0x7f80;
#endif VAX
#ifdef IEEEFLOAT
	= *((int *)(&d) + HIND) & 0x800fffff | ((e + 1022) << 20) & 0x7ff00000;
#endif IEEEFLOAT
#ifdef TAHOE
        = *(int *)(&d) & 0x807fffff | ((e + 128) << 23) & 0x7f800000;
#endif
	dummy = d*d;
	return(d);
}


object
double_to_integer(d)
double d;
{
	int h, l, e, s;
	object x, y;
	object shift_integer();

	if (d == 0.0)
		return(MAKE_FIXNUM(0));
	integer_decode_double(d, &h, &l, &e, &s);

#ifdef VAX
	if (e <= -31) {
		h >>= (-e) - 31;
#endif VAX
#ifdef IEEEFLOAT
	if (e <= -31) {
		e = (-e) - 31;
		if (e >= 31)
			return(MAKE_FIXNUM(0));
		h >>= e;
#endif IEEEFLOAT
#ifdef TAHOE
        if (e <= -31) {
                h >>= (-e) - 31;
#endif

		return(MAKE_FIXNUM(s*h));
	}
	if (h != 0)
		x = bignum2(h, l);
	else
		x = MAKE_FIXNUM(l);

#ifdef VAX
	x = shift_integer(x, e);
#endif VAX
#ifdef IEEEFLOAT
	x = shift_integer(x, e);
#endif IEEEFLOAT
#ifdef TAHOE
        x = shift_integer(x, e);
#endif

	if (s < 0)
		x = number_negate(x);
	return(x);
}

object
remainder(x, y, q)
object x, y, q;
{
	object z;

	z = number_times(q, y);
	z = number_minus(x, z);
	return(z);
}

/* Coerce X to single-float if one arg,
   otherwise coerce to same float type as second arg */

Lfloat(int narg, object x, ...)
{
	double	d;
	enum type t;
	va_list args;
	object y;

	if (narg < 1)
		FEtoo_few_arguments(&narg);
	else if (narg > 2)
		FEtoo_many_arguments(&narg);
	va_start(args, x);
	y = va_arg(args, object);
	if (narg == 2) {
		check_type_float(&y);
		t = type_of(y);
	}
	switch (type_of(x)) {
	case t_fixnum:
		if (narg > 1 && t == t_shortfloat)
		  x = make_shortfloat((shortfloat)(fix(x)));
		else
		  x = make_longfloat((double)(fix(x)));
		break;

	case t_bignum:
	case t_ratio:
		d = number_to_double(x);
		if (narg > 1 && t == t_shortfloat)
		  x = make_shortfloat((shortfloat)d);
		else
		  x = make_longfloat(d);		
		break;

	case t_shortfloat:
		if (narg > 1 && t == t_shortfloat);
		  else
		    x = make_longfloat((double)(sf(x)));
		break;

	case t_longfloat:
		if (narg > 1 && t == t_shortfloat)
			x = make_shortfloat((shortfloat)(lf(x)));
		break;

	default:
		FEwrong_type_argument(TSor_rational_float, x);
	}
	VALUES(0) = x;
	RETURN(1);
}

Lnumerator(int narg, object x)
{
	check_arg(1);
	check_type_rational(&x);
	VALUES(0) = (type_of(x) == t_ratio) ? x->rat.rat_num : x;
	RETURN(1);
}

Ldenominator(int narg, object x)
{
	check_arg(1);
	check_type_rational(&x);
	VALUES(0) = (type_of(x) == t_ratio) ? x->rat.rat_den : MAKE_FIXNUM(1);
	RETURN(1);
}

Lfloor(int narg, object x, ...)
{
	object y, q, q1;
	double d;
	va_list args;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	if (narg > 1)
		goto TWO_ARG;
	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
		VALUES(0) = x;
		VALUES(1) = MAKE_FIXNUM(0);
		RETURN(2);

	case t_ratio:
		q = x;
		y = MAKE_FIXNUM(1);
		goto RATIO;

	case t_shortfloat:
		d = (double)(sf(x));
		q1 = double_to_integer(d);
		d -= number_to_double(q1);
		if (sf(x) < 0.0 && d != 0.0) {
			q1 = one_minus(q1);
			d += 1.0;
		}
		VALUES(0) = q1;
		VALUES(1) = make_shortfloat((shortfloat)d);
		RETURN(2);

	case t_longfloat:
		d = lf(x);
		q1 = double_to_integer(d);
		d -= number_to_double(q1);
		if (lf(x) < 0.0 && d != 0.0) {
			q1 = one_minus(q1);
			d += 1.0;
		}
		VALUES(0) = q1;
		VALUES(1) = make_longfloat(d);
		RETURN(2);

	default:
		FEwrong_type_argument(TSor_rational_float, x);
	}

TWO_ARG:
	if (narg > 2)
		FEtoo_many_arguments(&narg);
	va_start(args, x);
	y = va_arg(args, object);
	if ((FIXNUMP(x) || type_of(x) == t_bignum) &&
	    (FIXNUMP(y) || type_of(y) == t_bignum)) {
		if (number_zerop(x)) {
			VALUES(0) = MAKE_FIXNUM(0);
			VALUES(1) = MAKE_FIXNUM(0);
			RETURN(2);
		}
		integer_quotient_remainder_1(x, y, &q, &q1);
		VALUES(0) = q;
		VALUES(1) = q1;
		if (number_minusp(x) ? number_plusp(y) : number_minusp(y)) {
			if (!number_zerop(q1)) {
			  VALUES(0) = one_minus(q);
			  VALUES(1) = number_plus(q1, y);
			}
		}
		RETURN(2);
	}
	check_type_or_rational_float(&x);
	check_type_or_rational_float(&y);
	q = number_divide(x, y);
	switch (type_of(q)) {
	case t_fixnum:
	case t_bignum:
		VALUES(0) = q;
		VALUES(1) = MAKE_FIXNUM(0);
		break;
	
	case t_ratio:
	RATIO:
		q1 = integer_divide1(q->rat.rat_num, q->rat.rat_den);
		if (number_minusp(q))
			q1 = one_minus(q1);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
		break;

	case t_shortfloat:
	case t_longfloat:
		q1 = double_to_integer(number_to_double(q));
		if (number_minusp(q) && number_compare(q, q1))
			q1 = one_minus(q1);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
	}
	RETURN(2);
}


Lceiling(int narg, object x, ...)
{
	object y, q, q1;
	double d;
	va_list args;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	if (narg > 1)
		goto TWO_ARG;
	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
		VALUES(0) = x;
		VALUES(1) = MAKE_FIXNUM(0);
		RETURN(2);

	case t_ratio:
		q = x;
		y = MAKE_FIXNUM(1);
		goto RATIO;

	case t_shortfloat:
		d = (double)(sf(x));
		q1 = double_to_integer(d);
		d -= number_to_double(q1);
		if (sf(x) > 0.0 && d != 0.0) {
			q1 = one_plus(q1);
			d -= 1.0;
		}
		VALUES(0) = q1;
		VALUES(1) = make_shortfloat((shortfloat)d);
		RETURN(2);

	case t_longfloat:
		d = lf(x);
		q1 = double_to_integer(d);
		d -= number_to_double(q1);
		if (lf(x) > 0.0 && d != 0.0) {
			q1 = one_plus(q1);
			d -= 1.0;
		}
		VALUES(0) = q1;
		VALUES(1) = make_longfloat(d);
		RETURN(2);

	default:
		FEwrong_type_argument(TSor_rational_float, x);
	}

TWO_ARG:
	if (narg > 2)
		FEtoo_many_arguments(&narg);
	va_start(args, x);
	y = va_arg(args, object);
	if ((FIXNUMP(x) || type_of(x) == t_bignum) &&
	    (FIXNUMP(y) || type_of(y) == t_bignum)) {
		if (number_zerop(x)) {
			VALUES(0) = MAKE_FIXNUM(0);
			VALUES(1) = MAKE_FIXNUM(0);
			RETURN(2);
		}
		integer_quotient_remainder_1(x, y, &q, &q1);
		VALUES(0) = q;
		VALUES(1) = q1;
		if (number_plusp(x) ? number_plusp(y) : number_minusp(y)) {
			if (number_zerop(q1))
				RETURN(2);
			VALUES(0) = one_plus(q);
			VALUES(1) = number_minus(q1, y);
		}
		RETURN(2);
	}
	check_type_or_rational_float(&x);
	check_type_or_rational_float(&y);
	q = number_divide(x, y);
	switch (type_of(q)) {
	case t_fixnum:
	case t_bignum:
		VALUES(0) = q;
		VALUES(1) = MAKE_FIXNUM(0);
		break;
	
	case t_ratio:
	RATIO:
		q1 = integer_divide1(q->rat.rat_num, q->rat.rat_den);
		if (number_plusp(q))
			q1 = one_plus(q1);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
		break;

	case t_shortfloat:
	case t_longfloat:
		q1 = double_to_integer(number_to_double(q));
		if (number_plusp(q1) && number_compare(q, q1))
			q1 = one_plus(q1);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
	}
	RETURN(2);
}


Ltruncate(int narg, object x, ...)
{
	object y, q, q1;
	va_list args;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	if (narg > 1)
		goto TWO_ARG;
	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
		VALUES(0) = x;
		VALUES(1) = MAKE_FIXNUM(0);
		RETURN(2);

	case t_ratio:
		q1 = integer_divide1(x->rat.rat_num, x->rat.rat_den);
		VALUES(0) = q1;
		VALUES(1) = number_minus(x, q1);
		RETURN(2);

	case t_shortfloat:
		q1 = double_to_integer((double)(sf(x)));
		VALUES(0) = q1;
		VALUES(1) = number_minus(x, q1);
		RETURN(2);

	case t_longfloat:
		q1 = double_to_integer(lf(x));
		VALUES(0) = q1;
		VALUES(1) = number_minus(x, q1);
		RETURN(2);

	default:
		FEwrong_type_argument(TSor_rational_float, x);
	}

TWO_ARG:
	if (narg > 2)
		FEtoo_many_arguments(&narg);
	va_start(args, x);
	y = va_arg(args, object);
	if ((FIXNUMP(x) || type_of(x) == t_bignum) &&
	    (FIXNUMP(y) || type_of(y) == t_bignum)) {
		integer_quotient_remainder_1(x, y, &q, &q1);
		VALUES(0) = q;
		VALUES(1) = q1;
		RETURN(2);
	}
	check_type_or_rational_float(&x);
	check_type_or_rational_float(&y);
	q = number_divide(x, y);
	switch (type_of(q)) {
	case t_fixnum:
	case t_bignum:
		VALUES(0) = q;
		VALUES(1) = MAKE_FIXNUM(0);
		break;
	
	case t_ratio:
		q1 = integer_divide1(q->rat.rat_num, q->rat.rat_den);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
		break;

	case t_shortfloat:
	case t_longfloat:
		q1 = double_to_integer(number_to_double(q));
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
	}
	RETURN(2);
}


Lround(int narg, object x, ...)
{
	object y, q, q1, r;
	double d;
	int n, c;
	va_list args;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	if (narg > 1)
		goto TWO_ARG;
	switch (type_of(x)) {

	case t_fixnum:
	case t_bignum:
		VALUES(0) = x;
		VALUES(1) = MAKE_FIXNUM(0);
		RETURN(2);

	case t_ratio:
		q = x;
		y = MAKE_FIXNUM(1);
		goto RATIO;

	case t_shortfloat:
		d = (double)(sf(x));
		if (d >= 0.0)
			q = double_to_integer(d + 0.5);
		else
			q = double_to_integer(d - 0.5);
		d -= number_to_double(q);
		if (d == 0.5 && number_oddp(q)) {
			q = one_plus(q);
			d = -0.5;
		}
		if (d == -0.5 && number_oddp(q)) {
			q = one_minus(q);
			d = 0.5;
		}
		VALUES(0) = q;
		VALUES(1) = make_shortfloat((shortfloat)d);
		RETURN(2);

	case t_longfloat:
		d = lf(x);
		if (d >= 0.0)
			q = double_to_integer(d + 0.5);
		else
			q = double_to_integer(d - 0.5);
		d -= number_to_double(q);
		if (d == 0.5 && number_oddp(q)) {
			q = one_plus(q);
			d = -0.5;
		}
		if (d == -0.5 && number_oddp(q)) {
			q = one_minus(q);
			d = 0.5;
		}
		VALUES(0) = q;
		VALUES(1) = make_longfloat(d);
		RETURN(2);

	default:
		FEwrong_type_argument(TSor_rational_float, x);
	}

TWO_ARG:
	if (narg > 2)
		FEtoo_many_arguments(&narg);
	va_start(args, x);
	y = va_arg(args, object);
	check_type_or_rational_float(&x);
	check_type_or_rational_float(&y);
	q = number_divide(x, y);
	switch (type_of(q)) {
	case t_fixnum:
	case t_bignum:
		VALUES(0) = q;
		VALUES(1) = MAKE_FIXNUM(0);
		break;
	
	case t_ratio:
	RATIO:
		q1 = integer_divide1(q->rat.rat_num, q->rat.rat_den);
		VALUES(0) = q1;
		r = number_minus(q, q1);
		if ((c = number_compare(r, plus_half)) > 0 ||
		    (c == 0 && number_oddp(q1)))
			q1 = one_plus(q1);
		if ((c = number_compare(r, minus_half)) < 0 ||
		    (c == 0 && number_oddp(q1)))
			q1 = one_minus(q1);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
		break;

	case t_shortfloat:
	case t_longfloat:
		d = number_to_double(q);
		if (d >= 0.0)
			q1 = double_to_integer(d + 0.5);
		else
			q1 = double_to_integer(d - 0.5);
		d -= number_to_double(q1);
		if (d == 0.5 && number_oddp(q1))
			q1 = one_plus(q1);
		if (d == -0.5 && number_oddp(q1))
			q1 = one_minus(q1);
		VALUES(0) = q1;
		VALUES(1) = remainder(x, y, q1);
	}
	RETURN(2);
}


Lmod(int narg, object x, object y)
{
	check_arg(2);
	Lfloor(2, x, y);
	VALUES(0) = VALUES(1);
	RETURN(1);
}


Lrem(int narg, object x, object y)
{
	check_arg(2);
	Ltruncate(2, x, y);
	VALUES(0) = VALUES(1);
	RETURN(1);
}


Ldecode_float(int narg, object x)
{
	double d;
	int e, s;

	check_arg(1);
	check_type_float(&x);
	if (type_of(x) == t_shortfloat)
		d = sf(x);
	else
		d = lf(x);
	if (d >= 0.0)
		s = 1;
	else {
		d = -d;
		s = -1;
	}
	e = double_exponent(d);
	d = set_exponent(d, 0);
	if (type_of(x) == t_shortfloat) {
		VALUES(0) = make_shortfloat((shortfloat)d);
		VALUES(1) = MAKE_FIXNUM(e);
		VALUES(2) = make_shortfloat((shortfloat)s);
	} else {
		VALUES(0) = make_longfloat(d);
		VALUES(1) = MAKE_FIXNUM(e);
		VALUES(2) = make_longfloat((double)s);
	}
	RETURN(3);
}


Lscale_float(int narg, object x, object y)
{
	double d;
	int e, k;

	check_arg(2);
	check_type_float(&x);
	if (FIXNUMP(y))
		k = fix(y);
	else
		FEerror("~S is an illegal exponent.", 1, y);
	if (type_of(x) == t_shortfloat)
		d = sf(x);
	else
		d = lf(x);
	e = double_exponent(d) + k;
#ifdef VAX
	if (e <= -128 || e >= 128)
#endif VAX
#ifdef IEEEFLOAT
	if (type_of(x) == t_shortfloat && (e <= -126 || e >= 130) ||
	    type_of(x) == t_longfloat && (e <= -1022 || e >= 1026))
#endif IEEEFLOAT
#ifdef TAHOE
        if (e <= -128 || e >= 128)
#endif  
		FEerror("~S is an illegal exponent.", 1, y);
	d = set_exponent(d, e);
	VALUES(0) =
	  (type_of(x) == t_shortfloat) ?
	    make_shortfloat((shortfloat)d) : make_longfloat(d);
	RETURN(1);
}


Lfloat_radix(int narg, object x)
{
	check_arg(1);
	check_type_float(&x);
	VALUES(0) = MAKE_FIXNUM(2);
	RETURN(1);
}


Lfloat_sign(int narg, object x, object y)
{
	double d, f;

	if (narg < 1)
		FEtoo_few_arguments(&narg);
	else if (narg > 2)
		FEtoo_many_arguments(&narg);
	check_type_float(&x);
	if (type_of(x) == t_shortfloat)
		d = sf(x);
	else
		d = lf(x);
	if (narg == 1)
		f = 1.0;
	else {
		check_type_float(&y);
		x = y;
		if (type_of(x) == t_shortfloat)
			f = sf(x);
		else
			f = lf(x);
		if (f < 0.0)
			f = -f;
	}
	if (d < 0.0)
		f = -f;
	VALUES(0) = (type_of(x) == t_shortfloat) ?
	  make_shortfloat((shortfloat)f) : make_longfloat(f);
	RETURN(1);
}


Lfloat_digits(int narg, object x)
{
	check_arg(1);
	check_type_float(&x);
	VALUES(0) = (type_of(x) == t_shortfloat) ?
	  MAKE_FIXNUM(6) : MAKE_FIXNUM(14);
	RETURN(1);
}


Lfloat_precision(int narg, object x)
{
	check_arg(1);
	check_type_float(&x);
	if (type_of(x) == t_shortfloat)
		VALUES(0) = (sf(x) == 0.0) ? MAKE_FIXNUM(0) : MAKE_FIXNUM(24);
	else
		VALUES(0) = (lf(x) == 0.0) ? MAKE_FIXNUM(0) : MAKE_FIXNUM(53);
	RETURN(1);
}


Linteger_decode_float(int narg, object x)
{
	int h, l, e, s;

	check_arg(1);
	check_type_float(&x);
	if (type_of(x) == t_longfloat) {
		integer_decode_double(lf(x), &h, &l, &e, &s);
		VALUES(0) = (h != 0) ? bignum2(h, l) : MAKE_FIXNUM(l);
		VALUES(1) = MAKE_FIXNUM(e);
		VALUES(2) = MAKE_FIXNUM(s);
	} else {
		integer_decode_float((double)(sf(x)), &h, &e, &s);
		VALUES(0) = MAKE_FIXNUM(h);
		VALUES(1) = MAKE_FIXNUM(e);
		VALUES(2) = MAKE_FIXNUM(s);
	}
	RETURN(3);
}


Lcomplex(int narg, object r, object i)
{
	if (narg < 1)
		FEtoo_few_arguments(&narg);
	if (narg > 2)
		FEtoo_many_arguments(&narg);
	check_type_or_rational_float(&r);
	if (narg == 1)
		i = MAKE_FIXNUM(0);
	else
		check_type_or_rational_float(&i);
	VALUES(0) = make_complex(r, i);
	RETURN(1);
}


Lrealpart(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	VALUES(0) = (type_of(x) == t_complex) ? x->cmp.cmp_real : x;
	RETURN(1);
}


Limagpart(int narg, object x)
{
	check_arg(1);
	check_type_number(&x);
	switch (type_of(x)) {
	case t_fixnum:
	case t_bignum:
	case t_ratio:
		VALUES(0) = MAKE_FIXNUM(0);
		break;
	case t_shortfloat:
		VALUES(0) = shortfloat_zero;
		break;
	case t_longfloat:
		VALUES(0) = longfloat_zero;
		break;
	case t_complex:
		VALUES(0) = x->cmp.cmp_imag;
		break;
	}
	RETURN(1);
}

init_num_co()
{
	int l[2];
	float smallest_float, biggest_float;
	double smallest_double, biggest_double;
	float float_epsilon, float_negative_epsilon;
	double double_epsilon, double_negative_epsilon;
	double lf1, lf2;
	float sf1, sf2;
	object num;

#define LF_EQL(a,b) (lf1 = a, lf2 = b, lf1 == lf2)
#define SF_EQL(a,b) (sf1 = a, sf2 = b, sf1 == sf2)

#ifdef VAX
	l[0] = 0x80;
	l[1] = 0;
	smallest_float = *(float *)l;
	smallest_double = *(double *)l;
#endif VAX

#ifdef IEEEFLOAT
	((int *) &smallest_float)[0]= 1;
	((int *) &smallest_double)[HIND] = 0;
	((int *) &smallest_double)[LIND] = 1;
#endif IEEEFLOAT

#ifdef VAX
	l[0] = 0xffff7fff;
	l[1] = 0xffffffff;
	biggest_float = *(float *)l;
	biggest_double = *(double *)l;
#endif VAX

#ifdef IEEEFLOAT
	((int *) &biggest_float)[0]= 0x7f7fffff;
	((int *) &biggest_double)[HIND] = 0x7fefffff;
	((int *) &biggest_double)[LIND] = 0xffffffff;
#endif IEEEFLOAT

#ifdef TAHOE
        l[0] = 0x00800000;
        l[1] = 0;
        smallest_float = *(float *)l;
        smallest_double = *(double *)l;
#endif

/* We want the smallest number not satisfying something,
   and so we go quickly down, and then back up.  We have
   to use a function call for test, since in line code may keep
   too much precision, while the usual lisp eql,is not
   in line.
   We use SMALL as a multiple to come back up by.
*/

#define SMALL 1.05	

	for (float_epsilon = 1.0;
	     !SF_EQL((float)(1.0 + float_epsilon),(float)1.0);
	     float_epsilon /= 2.0)
		;
	while(SF_EQL((float)(1.0 + float_epsilon),(float)1.0))
	  float_epsilon=float_epsilon*SMALL;
	for (float_negative_epsilon = 1.0;
	     !SF_EQL((float)(1.0 - float_negative_epsilon) ,(float)1.0);
	     float_negative_epsilon /= 2.0)
		;
	while(SF_EQL((float)(1.0 - float_negative_epsilon) ,(float)1.0))
	  float_negative_epsilon=float_negative_epsilon*SMALL;
	for (double_epsilon = 1.0;
	     !(LF_EQL(1.0 + double_epsilon, 1.0));
	     double_epsilon /= 2.0)
		;
	while((LF_EQL(1.0 + double_epsilon, 1.0)))
	  double_epsilon=double_epsilon*SMALL;
	  ;
	for (double_negative_epsilon = 1.0;
	     !LF_EQL(1.0 - double_negative_epsilon , 1.0);
	     double_negative_epsilon /= 2.0)
		;
	while(LF_EQL(1.0 - double_negative_epsilon , 1.0))
	  double_negative_epsilon=double_negative_epsilon*SMALL;
	  ;

	num = make_shortfloat(biggest_float);
	make_constant("MOST-POSITIVE-SHORT-FLOAT", num);
	make_constant("MOST-POSITIVE-SINGLE-FLOAT", num);

	num = make_shortfloat(smallest_float);
	make_constant("LEAST-POSITIVE-SHORT-FLOAT", num);
	make_constant("LEAST-POSITIVE-SINGLE-FLOAT", num);

	num = make_shortfloat(-smallest_float);
	make_constant("LEAST-NEGATIVE-SHORT-FLOAT", num);
	make_constant("LEAST-NEGATIVE-SINGLE-FLOAT", num);

	num = make_shortfloat(-biggest_float);
	make_constant("MOST-NEGATIVE-SHORT-FLOAT", num);
	make_constant("MOST-NEGATIVE-SINGLE-FLOAT", num);

	num = make_longfloat(biggest_double);
	make_constant("MOST-POSITIVE-DOUBLE-FLOAT", num);
	make_constant("MOST-POSITIVE-LONG-FLOAT", num);

	num = make_longfloat(smallest_double);
	make_constant("LEAST-POSITIVE-DOUBLE-FLOAT", num);
	make_constant("LEAST-POSITIVE-LONG-FLOAT", num);

	num = make_longfloat(-smallest_double);
	make_constant("LEAST-NEGATIVE-DOUBLE-FLOAT", num);
	make_constant("LEAST-NEGATIVE-LONG-FLOAT", num);

	num = make_longfloat(-biggest_double);
	make_constant("MOST-NEGATIVE-DOUBLE-FLOAT", num);
	make_constant("MOST-NEGATIVE-LONG-FLOAT", num);

	num = make_shortfloat(float_epsilon);
	make_constant("SHORT-FLOAT-EPSILON", num);
	make_constant("SINGLE-FLOAT-EPSILON", num);
	num = make_longfloat(double_epsilon);
	make_constant("DOUBLE-FLOAT-EPSILON", num);
	make_constant("LONG-FLOAT-EPSILON", num);

	num = make_shortfloat(float_negative_epsilon);
	make_constant("SHORT-FLOAT-NEGATIVE-EPSILON", num);
	make_constant("SINGLE-FLOAT-NEGATIVE-EPSILON", num);
	num = make_longfloat(double_negative_epsilon);
	make_constant("DOUBLE-FLOAT-NEGATIVE-EPSILON", num);
	make_constant("LONG-FLOAT-NEGATIVE-EPSILON", num);

	plus_half = make_ratio(MAKE_FIXNUM(1), MAKE_FIXNUM(2));
	enter_mark_origin(&plus_half);

	minus_half = make_ratio(MAKE_FIXNUM(-1), MAKE_FIXNUM(2));
	enter_mark_origin(&minus_half);

	make_function("FLOAT", Lfloat);
	make_function("NUMERATOR", Lnumerator);
	make_function("DENOMINATOR", Ldenominator);
	make_function("FLOOR", Lfloor);
	make_function("CEILING", Lceiling);
	make_function("TRUNCATE", Ltruncate);
	make_function("ROUND", Lround);
	make_function("MOD", Lmod);
	make_function("REM", Lrem);
	make_function("DECODE-FLOAT", Ldecode_float);
	make_function("SCALE-FLOAT", Lscale_float);
	make_function("FLOAT-RADIX", Lfloat_radix);
	make_function("FLOAT-SIGN", Lfloat_sign);
	make_function("FLOAT-DIGITS", Lfloat_digits);
	make_function("FLOAT-PRECISION", Lfloat_precision);
	make_function("INTEGER-DECODE-FLOAT", Linteger_decode_float);
	make_function("COMPLEX", Lcomplex);
	make_function("REALPART", Lrealpart);
	make_function("IMAGPART", Limagpart);
}
