/*
    list.d -- List manipulating routines.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

object Ktest;
object Ktest_not;
object Kkey;

/******************************* ------- ******************************/

object Kinitial_element;

#ifdef MTCL
#define test_function   clwp->lwp_test_function
#define item_compared   clwp->lwp_item_compared
#define tf              clwp->lwp_tf
#define key_function    clwp->lwp_key_function
#define kf              clwp->lwp_kf
#else
object test_function;
object item_compared;
bool (*tf)();
object key_function;
object (*kf)();
#endif MTCL

#define TEST(x)         (*tf)(x)

#define saveTEST  \
	object old_test_function = test_function;  \
	object old_item_compared = item_compared;  \
	bool (*old_tf)() = tf;  \
	object old_key_function = key_function;  \
	object (*old_kf)() = kf;  \
	volatile bool eflag = FALSE; int r

#define protectTEST  \
	if (r = frs_push(FRS_PROTECT, Cnil)) {  \
		eflag = TRUE;  \
		goto L;  \
	}

#define restoreTEST  \
L:  \
	frs_pop();  \
	test_function = old_test_function;  \
	item_compared = old_item_compared;  \
	tf = old_tf;  \
	key_function = old_key_function;  \
	kf = old_kf;  \
	if (eflag) unwind(nlj_fr, nlj_tag, r);

bool
test_compare(object x)
{
	funcall(3, test_function, item_compared, (*kf)(x));
	return(VALUES(0) != Cnil);
}

bool
test_compare_not(object x)
{
	funcall(3, test_function, item_compared, (*kf)(x));
	return(VALUES(0) == Cnil);
}

bool
test_eql(object x)
{
	return(eql(item_compared, (*kf)(x)));
}

object
apply_key_function(object x)
{
	funcall(2, key_function, x);
	return(VALUES(0));
}

object
identity(object x)
{
	return(x);
}

setupTEST(object item, object test, object test_not, object key)
{
	item_compared = item;
	if (test != Cnil) {
		if (test_not != Cnil)
		    FEerror("Both :TEST and :TEST-NOT are specified.", 0);
		test_function = test;
		tf = test_compare;
	} else if (test_not != Cnil) {
		test_function = test_not;
		tf = test_compare_not;
	} else
		tf = test_eql;
	if (key != Cnil) {
		key_function = key;
		kf = apply_key_function;
	} else
		kf = identity;
}

#define PREDICATE2(f)  \
f ## _if(int narg, object pred, object arg, object key, object val) \
{  \
	if (narg < 2)  \
		FEtoo_few_arguments(&narg);  \
	RETURN(f(narg+2, pred, arg, Ktest, Sfuncall, key, val));  \
}  \
\
f ## _if_not(int narg, object pred, object arg, object key, object val) \
{  \
	if (narg < 2)  \
		FEtoo_few_arguments(&narg);  \
	RETURN(f(narg+2, pred, arg, Ktest_not, Sfuncall, key, val));  \
}

#define PREDICATE3(f)  \
f ## _if(int narg, object arg1, object pred, object arg3, object key, object val) \
{  \
	if (narg < 3)  \
		FEtoo_few_arguments(&narg);  \
	RETURN(f(narg+2, arg1, pred, arg3, Ktest, Sfuncall, key, val));  \
}  \
\
f ## _if_not(int narg, object arg1, object pred, object arg3, object key, \
	object val)  \
{  \
	if (narg < 3)  \
		FEtoo_few_arguments(&narg);  \
	RETURN(f(narg+2, arg1, pred, arg3, Ktest_not, Sfuncall, key, val));  \
}

bool
endp1(object x)
{
	if (type_of(x) == t_cons)
		return(FALSE);
	else if (Null(x))
		return(TRUE);
	FEwrong_type_argument(Slist, x);
}

object
car(object x)
{
	if (Null(x))
		return(x);
	if (type_of(x) == t_cons)
		return(CAR(x));
	FEwrong_type_argument(Slist, x);
}

object
cdr(object x)
{
	if (Null(x))
		return(x);
	if (type_of(x) == t_cons)
		return(CDR(x));
	FEwrong_type_argument(Slist, x);
}

object
kar(object x)
{
	if (type_of(x) == t_cons)
		return(CAR(x));
	FEwrong_type_argument(Scons, x);
}

object
kdr(object x)
{
	if (type_of(x) == t_cons)
		return(CDR(x));
	FEwrong_type_argument(Scons, x);
}

object list(int narg, ...)
{
	object p = Cnil, *z = &p;
	va_list args;

	va_start(args, narg);
	while (narg-- > 0)
		z = &CDR(*z = CONS(va_arg(args, object), Cnil));
	return(p);
}

object listA(int narg, ...)
{
	object p = Cnil, *z = &p;
	va_list args;

	va_start(args, narg);
	while (--narg > 0)
	  z = &CDR( *z = CONS(va_arg(args, object), Cnil));
	*z = va_arg(args, object);
	return(p);
}

bool
tree_equal(object x, object y)
{
	cs_check(x);

BEGIN:
	if (type_of(x) == t_cons)
		if (type_of(y) == t_cons)
			if (tree_equal(CAR(x), CAR(y))) {
				x = CDR(x);
				y = CDR(y);
				goto BEGIN;
			} else
				return(FALSE);
		else
			return(FALSE);
	else {
		item_compared = x;
		if (TEST(y))
			return(TRUE);
		else
			return(FALSE);
	}
}

object
append(object x, object y)
{
	object w, *z;

	z = &w;
	copy_list_to(x, &z);
/* i.e.
	for (; !ENDP(x); x = CDR(x))
		z = &CDR(*z = CONS(CAR(x), Cnil));
 */
	*z = y;
	return(w);
}

/*
	Copy_list(x) copies list x.
*/
object
copy_list(object x)
{
	object copy, y;

	if (type_of(x) != t_cons)
		return(x);
	copy = y = CONS(CAR(x), Cnil);
	for (x = CDR(x); type_of(x) == t_cons; x = CDR(x))
		y = CDR(y) = CONS(CAR(x), Cnil);
	CDR(y) = x;
	return(copy);
}

copy_list_to(object x, object **z)
{
	object *y;

	y = *z;
	for (; !ENDP(x); x = CDR(x))
		y = &CDR(*y = CONS(CAR(x), Cnil));
	*z = y;
}

/*
	Copy_alist(x) copies alist x.
*/
object
copy_alist(object x)
{
	object y, z;

	if (ENDP(x))
		return(Cnil);
	z = y = CONS(Cnil, Cnil);
	for (;;) {
		if (type_of(CAR(x)) == t_cons)
		   CAR(y) = CONS(car(CAR(x)), cdr(CAR(x)));
		else CAR(y) = CAR(x);
		x = CDR(x);
		if (ENDP(x))
			break;
		CDR(y) = CONS(Cnil, Cnil);
		y = CDR(y);
	}
	return(z);
}

/*
	Copy_tree(x) returns a copy of tree x.
*/
object
copy_tree(object x)
{
	cs_check(x);

	if (type_of(x) == t_cons) {
		return(CONS(copy_tree(CAR(x)), copy_tree(CDR(x))));
	} else
		return(x);
}

/*
	Subst(new, tree) returns
	the result of substituting new in tree.
*/
object
subst(object new, object tree)
{
	cs_check(new);

	if (TEST(tree))
		return(new);
	else if (type_of(tree) == t_cons) {
		return(CONS(subst(new, CAR(tree)), subst(new, CDR(tree))));
	} else
		return(tree);
}

/*
	Nsubst(new, treep) stores
	the result of nsubstituting new in *treep
	to *treep.
*/
nsubst(object new, object *treep)
{
	cs_check(new);

	if (TEST(*treep))
		*treep = new;
	else if (type_of(*treep) == t_cons) {
		nsubst(new, &CAR(*treep));
		nsubst(new, &CDR(*treep));
	}
}

/*
	Sublis(alist, tree) returns
	result of substituting tree by alist.
*/
object
sublis(object alist, object tree)
{
	object x;

	cs_check(alist);

	for (x = alist;  !ENDP(x);  x = CDR(x)) {
		item_compared = car(CAR(x));
		if (TEST(tree)) return(cdr(CAR(x)));
	}
	if (type_of(tree) == t_cons) {
		return(CONS(sublis(alist, CAR(tree)), sublis(alist, CDR(tree))));
	} else
		return(tree);
}

/*
	Nsublis(alist, treep) stores
	the result of substiting *treep by alist
	to *treep.
*/
nsublis(object alist, object *treep)
{
	object x;

	cs_check(alist);

	for (x = alist;  !ENDP(x);  x = CDR(x)) {
		item_compared = car(CAR(x));
		if (TEST(*treep)) {
			*treep = CDAR(x);
			return;
		}
	}
	if (type_of(*treep) == t_cons) {
		nsublis(alist, &CAR(*treep));
		nsublis(alist, &CDR(*treep));
	}
}

Lcar(int narg, object x)
{
	check_arg(1);

	if (type_of(x) == t_cons || Null(x)) {
		VALUES(0) = CAR(x);
		RETURN(1);
		}
	else
		FEwrong_type_argument(Slist, x);
}

Lcdr(int narg, object x)
{
	check_arg(1);

	if (type_of(x) == t_cons || Null(x)) {
		VALUES(0) = CDR(x);
		RETURN(1);
		}
	else
		FEwrong_type_argument(Slist, x);
}

object caar(object x)    {  return(car(car(x)));  }
object cadr(object x)    {  return(car(cdr(x)));  }
object cdar(object x)    {  return(cdr(car(x)));  }
object cddr(object x)    {  return(cdr(cdr(x)));  }
object caaar(object x)   {  return(car(car(car(x))));  }
object caadr(object x)   {  return(car(car(cdr(x))));  }
object cadar(object x)   {  return(car(cdr(car(x))));  }
object caddr(object x)   {  return(car(cdr(cdr(x))));  }
object cdaar(object x)   {  return(cdr(car(car(x))));  }
object cdadr(object x)   {  return(cdr(car(cdr(x))));  }
object cddar(object x)   {  return(cdr(cdr(car(x))));  }
object cdddr(object x)   {  return(cdr(cdr(cdr(x))));  }
object caaaar(object x)  {  return(car(car(car(car(x)))));  }
object caaadr(object x)  {  return(car(car(car(cdr(x)))));  }
object caadar(object x)  {  return(car(car(cdr(car(x)))));  }
object caaddr(object x)  {  return(car(car(cdr(cdr(x)))));  }
object cadaar(object x)  {  return(car(cdr(car(car(x)))));  }
object cadadr(object x)  {  return(car(cdr(car(cdr(x)))));  }
object caddar(object x)  {  return(car(cdr(cdr(car(x)))));  }
object cadddr(object x)  {  return(car(cdr(cdr(cdr(x)))));  }
object cdaaar(object x)  {  return(cdr(car(car(car(x)))));  }
object cdaadr(object x)  {  return(cdr(car(car(cdr(x)))));  }
object cdadar(object x)  {  return(cdr(car(cdr(car(x)))));  }
object cdaddr(object x)  {  return(cdr(car(cdr(cdr(x)))));  }
object cddaar(object x)  {  return(cdr(cdr(car(car(x)))));  }
object cddadr(object x)  {  return(cdr(cdr(car(cdr(x)))));  }
object cdddar(object x)  {  return(cdr(cdr(cdr(car(x)))));  }
object cddddr(object x)  {  return(cdr(cdr(cdr(cdr(x)))));  }

#define CXR(sel)        (int narg, object x) \
				{ check_arg(1); VALUES(0) = sel; RETURN(1); }

Lcaar   CXR(car(car(x)))
Lcadr   CXR(car(cdr(x)))
Lcdar   CXR(cdr(car(x)))
Lcddr   CXR(cdr(cdr(x)))
Lcaaar  CXR(car(car(car(x))))
Lcaadr  CXR(car(car(cdr(x))))
Lcadar  CXR(car(cdr(car(x))))
Lcaddr  CXR(car(cdr(cdr(x))))
Lcdaar  CXR(cdr(car(car(x))))
Lcdadr  CXR(cdr(car(cdr(x))))
Lcddar  CXR(cdr(cdr(car(x))))
Lcdddr  CXR(cdr(cdr(cdr(x))))
Lcaaaar CXR(car(car(car(car(x)))))
Lcaaadr CXR(car(car(car(cdr(x)))))
Lcaadar CXR(car(car(cdr(car(x)))))
Lcaaddr CXR(car(car(cdr(cdr(x)))))
Lcadaar CXR(car(cdr(car(car(x)))))
Lcadadr CXR(car(cdr(car(cdr(x)))))
Lcaddar CXR(car(cdr(cdr(car(x)))))
Lcadddr CXR(car(cdr(cdr(cdr(x)))))
Lcdaaar CXR(cdr(car(car(car(x)))))
Lcdaadr CXR(cdr(car(car(cdr(x)))))
Lcdadar CXR(cdr(car(cdr(car(x)))))
Lcdaddr CXR(cdr(car(cdr(cdr(x)))))
Lcddaar CXR(cdr(cdr(car(car(x)))))
Lcddadr CXR(cdr(cdr(car(cdr(x)))))
Lcdddar CXR(cdr(cdr(cdr(car(x)))))
Lcddddr CXR(cdr(cdr(cdr(cdr(x)))))

#define LENTH(n)        (int narg, object x) \
	{       check_arg(1);\
		VALUES(0) = nth(n, x);\
		RETURN(1);}

Lfifth  LENTH(4)
Lsixth  LENTH(5)
Lseventh        LENTH(6)
Leighth LENTH(7)
Lninth  LENTH(8)
Ltenth  LENTH(9)

Lcons(int narg, object car, object cdr)
{       object x;
	check_arg(2);
	x = alloc_object(t_cons);
	CAR(x) = car;
	CDR(x) = cdr;
	VALUES(0) = x;
	RETURN(1);
}

@(defun tree_equal (x y &key test test_not)
@
	setupTEST(Cnil, test, test_not, Cnil);
	if (tree_equal(x, y))
		@(return Ct)
	else
		@(return Cnil)
@)

Lendp(int narg, object x)
{
	check_arg(1);

	if (Null(x)) {
		VALUES(0) = Ct;
		RETURN(1);
	}
	if (type_of(x) == t_cons) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	FEwrong_type_argument(Slist, x);
}

Llist_length(int narg, object x)
{
	int n;
	object fast, slow, cdr_fast;

	check_arg(1);
	fast = slow = x;
	for (n = 0; ; n += 2) {
		if (ENDP(fast)) {
			VALUES(0) = MAKE_FIXNUM(n);
			RETURN(1);
		}
		cdr_fast = CDR(fast);
		if (ENDP(cdr_fast)) {
			VALUES(0) = MAKE_FIXNUM(n + 1);
			RETURN(1);
		}
		if (fast == slow && n > 0) {
			VALUES(0) = Cnil;
			RETURN(1);
		}
		fast = CDR(cdr_fast);
		slow = CDR(slow);
	}
}

Lnth(int narg, object n, object x)
{
	check_arg(2);
	VALUES(0) = nth(fixint(n), x);
	RETURN(1);
}

object
nth(int n, object x)
{
	if (n < 0)
		FEerror("Negative index: ~D.", 1, MAKE_FIXNUM(n));
	while (n-- > 0)
		if (ENDP(x)) {
			return(Cnil);
		} else
			x = CDR(x);
	if (ENDP(x))
		return(Cnil);
	else
		return(CAR(x));
}

Lnthcdr(int narg, object n, object x)
{
	check_arg(2);
	VALUES(0) = nthcdr(fixint(n), x);
	RETURN(1);
}

object
nthcdr(int n, object x)
{
	if (n < 0)
		FEerror("Negative index: ~D.", 1, MAKE_FIXNUM(n));
	while (n-- > 0 && !ENDP(x))
		x = CDR(x);
	return(x);
}

Llast(int narg, object x)
{       object y;
	check_arg(1);
	y = x;
	while (type_of(y) == t_cons) {
		x = y;
		y = CDR(y);
	}
	VALUES(0) = x;
	RETURN(1);
}

Llist(int narg, ...)
{
	object list = Cnil, z;
	va_list args;

	va_start(args, narg);
	if (narg-- != 0) {
		list = z = CONS(va_arg(args, object), Cnil);
		while (narg-- > 0) 
			z = CDR(z) = CONS(va_arg(args, object), Cnil);
		}
	VALUES(0) = list;
	RETURN(1);
}

LlistA(int narg, ...)
{
	object p = Cnil, *z=&p;
	va_list args;

	if (narg == 0)
		FEtoo_few_arguments(&narg);
	va_start(args, narg);
	while (--narg > 0)
	  z = &CDR( *z = CONS(va_arg(args, object), Cnil));
	*z = va_arg(args, object);
	VALUES(0) = p;
	RETURN(1);
}

@(defun make_list (size &key initial_element &aux x)
	int i;
@
	check_type_non_negative_integer(&size);
	if (!FIXNUMP(size))
		FEerror("Cannot make a list of the size ~D.", 1, size);
	i = fix(size);
	while (i-- > 0)
		x = CONS(initial_element, x);
	@(return x)
@)

Lappend(int narg, ...)
{
	object x, *lastcdr;
	va_list rest;

	if (narg == 0)
		VALUES(0) = Cnil;
	else {
		lastcdr = &x;
		va_start(rest, narg);
		while (narg-- > 1)
			copy_list_to(va_arg(rest, object), &lastcdr);
		*lastcdr = va_arg(rest, object);
		VALUES(0) = x;
	}
	RETURN(1);
}

Lcopy_list(int narg, object x)
{
	check_arg(1);
	VALUES(0) = copy_list(x);
	RETURN(1);
}

Lcopy_alist(int narg, object x)
{
	check_arg(1);
	VALUES(0) = copy_alist(x);
	RETURN(1);
}

Lcopy_tree(int narg, object x)
{
	check_arg(1);
	VALUES(0) = copy_tree(x);
	RETURN(1);
}

Lrevappend(int narg, object x, object y)
{
	check_arg(2);
	for (; !ENDP(x); x = CDR(x))
		y = CONS(CAR(x),y);
	VALUES(0) = y;
	RETURN(1);
}

object
nconc(object x, object y)
{
	object x1, cdr_x1;

	if (ENDP(x))
		return(y);
	for (x1 = x;  !(cdr_x1=CDR(x1), ENDP(cdr_x1));  x1 = cdr_x1)
		;
	CDR(x1) = y;
	return(x);
}

Lnconc(int narg, ...)
{
	object x, l,*lastcdr;
	va_list lists;
	
	if (narg < 1) { VALUES(0) = Cnil; RETURN(1); }
	lastcdr = &x;
	va_start(lists, narg);
	while (narg-- > 1) {
		l = va_arg(lists, object);
		if (ENDP(l))
			continue;
		*lastcdr = l;
		lastcdr = &CDR(l);
		while (!ENDP(*lastcdr)) lastcdr = &CDR(*lastcdr);
	}
	*lastcdr = va_arg(lists, object);
	VALUES(0) = x;
	RETURN(1);
}

Lreconc(int narg, object x, object y)
{
	object z;

	check_arg(2);
	for (; !ENDP(x);) {
		z = x;
		x = CDR(x);
		CDR(z) = y;
		y = z;
	}
	VALUES(0) = y;
	RETURN(1);
}

@(defun butlast (lis &optional (nn `MAKE_FIXNUM(1)`))
	int i, len; object res = Cnil, *fill = &res;
@
	check_type_non_negative_integer(&nn);
	if (!FIXNUMP(nn))
		@(return Cnil)
	len = length(lis) - fix(nn);
	for (i = 0; i < len ;  i++, lis = CDR(lis))
		fill = &CDR(*fill = CONS(CAR(lis), Cnil));
	@(return res)
@)

@(defun nbutlast (lis &optional (nn `MAKE_FIXNUM(1)`))
	int i;
	object x;
@
	check_type_non_negative_integer(&nn);
	if (!FIXNUMP(nn))
		@(return Cnil)
	for (i = 0, x = lis;  !ENDP(x);  i++, x = CDR(x))
		;
	if (i <= fix(nn))
		@(return Cnil)
	for (i -= fix(nn), x = lis;  --i > 0;  x = CDR(x))
		;
	CDR(x) = Cnil;
	@(return lis)
@)

Lldiff(int narg, object x, object y)
{       int i;
	object res = Cnil, *fill = &res;

	check_arg(2);
	for (i = 0;  !ENDP(x);  i++, x = CDR(x))
		if (x == y)
			break;
		else
			fill = &CDR(*fill = CONS(CAR(x), Cnil));
	VALUES(0) = res;
	RETURN(1);
}

Lrplaca(int narg, object x, object v)
{
	check_arg(2);
	check_type_cons(&x);
	CAR(x) = v;
	VALUES(0) = x;
	RETURN(1);
}

Lrplacd(int narg, object x, object v)
{
	check_arg(2);
	check_type_cons(&x);
	CDR(x) = v;
	VALUES(0) = x;
	RETURN(1);
}

@(defun subst (new old tree &key test test_not key)
	saveTEST;
@
	protectTEST;
	setupTEST(old, test, test_not, key);
	tree = subst(new, tree);
	restoreTEST;
	@(return tree)
@)

PREDICATE3(Lsubst)

@(defun nsubst (new old tree &key test test_not key)
	saveTEST;
@
	protectTEST;
	setupTEST(old, test, test_not, key);
	nsubst(new, &tree);
	restoreTEST;
	@(return tree)
@)

PREDICATE3(Lnsubst)

@(defun sublis (alist tree &key test test_not key)
	saveTEST;
@
	protectTEST;
	setupTEST(Cnil, test, test_not, key);
	tree = sublis(alist, tree);
	restoreTEST;
	@(return tree)
@)

@(defun nsublis (alist tree &key test test_not key)
	saveTEST;
@
	protectTEST;
	setupTEST(Cnil, test, test_not, key);
	nsublis(alist, &tree);
	restoreTEST;
	@(return tree)
@)

@(defun member (item list &key test test_not key)
	saveTEST;
@
	protectTEST;
	setupTEST(item, test, test_not, key);
	while (!ENDP(list)) {
		if (TEST(CAR(list)))
			goto L;
		list = CDR(list);
	}
	restoreTEST;
	@(return list)
@)

PREDICATE2(Lmember)

@(defun member1 (item list &key test test_not key)
	saveTEST;
@
	protectTEST;
	if (key != Cnil) {
		funcall(2, key, item);
		item = VALUES(0);
		}
	setupTEST(item, test, test_not, key);
	while (!ENDP(list)) {
		if (TEST(CAR(list)))
			goto L;
		list = CDR(list);
	}
	restoreTEST;
	@(return list)
@)

@(defun tailp (y x)
@
	for (;  !ENDP(x);  x = CDR(x))
		if (x == y)
			@(return Ct)
	@(return Cnil)
@)

Ladjoin(int narg, object item, object list, object k1, object v1,
	 object k2, object v2, object k3, object v3)
{
	if (narg < 2)
		FEtoo_few_arguments(&narg);
	Lmember1(narg, item, list, k1, v1, k2, v2, k3, v3);
	if (Null(VALUES(0)))
		VALUES(0) = CONS(item, list);
	else
		VALUES(0) = list;
	RETURN(1);
}

@(defun acons (x y z)
@
	@(return `CONS(CONS(x, y), z)`)
@)

@(defun pairlis (keys data &optional a_list)
	object k, d, res, *fill;
@
	fill = &res;
	k = keys;
	d = data;
	while (!ENDP(k)) {
		if (ENDP(d))
		 FEerror(
		  "The keys ~S and the data ~S are not of the same length",
		  2, keys, data);
		*fill = CONS(CONS(CAR(k), CAR(d)), Cnil);
		fill = &CDR(*fill);
		k = CDR(k);
		d = CDR(d);
	}
	if (!ENDP(d))
	    FEerror("The keys ~S and the data ~S are not of the same length",
		    2, keys, data);
	*fill = a_list;
	@(return `res`)
@)

@(defun `assoc_or_rassoc(object (*car_or_cdr)())`
	 (item a_list &key test test_not key)
	saveTEST;
@
	protectTEST;
	setupTEST(item, test, test_not, key);
	while (!ENDP(a_list)) {
	  if (!Null(CAR(a_list)) && TEST((*car_or_cdr)(CAR(a_list)))) {
	    a_list = CAR(a_list);
	    goto L;
	  }
	  a_list = CDR(a_list);
	}
	restoreTEST;
	@(return a_list)
@)

Lassoc(int narg, object item, object alist, object k1, object v1,
	 object k2, object v2)
	{ RETURN(Lassoc_or_rassoc(narg, car, item, alist, k1, v1, k2, v2)); }
Lrassoc(int narg, object item, object alist, object k1, object v1,
	 object k2, object v2)
	{ RETURN(Lassoc_or_rassoc(narg, cdr, item, alist, k1, v1, k2, v2)); }

PREDICATE2(Lassoc)
PREDICATE2(Lrassoc)

bool
member_eq(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (x == CAR(l))
			return(TRUE);
	return(FALSE);
}

siLmemq(int narg, object x, object l)
{
	check_arg(2);

	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (x == CAR(l)) {
			VALUES(0) = l;
			RETURN(1);
		}
	
	VALUES(0) = Cnil;
	RETURN(1);
}

void delete_eq(object x, object *lp)
{
	for (;  type_of(*lp) == t_cons;  lp = &CDR((*lp)))
		if (CAR((*lp)) == x) {
			*lp = CDR((*lp));
			return;
		}
}

/* Added for use by the compiler, instead of open coding them. Beppe */
object
assq(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (x == CAR(CAR(l)))
			return(CAR(l));
	return(Cnil);
}

object
assql(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (eql(x, CAAR(l)))
			return(CAR(l));
	return(Cnil);
}


object
assoc(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (equal(x, CAAR(l)))
			return(CAR(l));
	return(Cnil);
}

object
assqlp(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (equalp(x, CAR(CAR(l))))
			return(CAR(l));
	return(Cnil);
}

object
memq(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (x == CAR(l))
			return(l);
	return(Cnil);
}

object
memql(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (eql(x, CAR(l)))
			return(l);
	return(Cnil);
}

object
member(object x, object l)
{
	for (;  type_of(l) == t_cons;  l = CDR(l))
		if (equal(x, CAR(l)))
			return(l);
	return(Cnil);
}
/* End of addition. Beppe */

init_list_function()
{
	Ktest = make_keyword("TEST");
	Ktest_not = make_keyword("TEST-NOT");
	Kkey = make_keyword("KEY");

	Kinitial_element = make_keyword("INITIAL-ELEMENT");

	make_function("CAR", Lcar);
	make_function("CDR", Lcdr);

	make_function("CAAR", Lcaar);
	make_function("CADR", Lcadr);
	make_function("CDAR", Lcdar);
	make_function("CDDR", Lcddr);
	make_function("CAAAR", Lcaaar);
	make_function("CAADR", Lcaadr);
	make_function("CADAR", Lcadar);
	make_function("CADDR", Lcaddr);
	make_function("CDAAR", Lcdaar);
	make_function("CDADR", Lcdadr);
	make_function("CDDAR", Lcddar);
	make_function("CDDDR", Lcdddr);
	make_function("CAAAAR", Lcaaaar);
	make_function("CAAADR", Lcaaadr);
	make_function("CAADAR", Lcaadar);
	make_function("CAADDR", Lcaaddr);
	make_function("CADAAR", Lcadaar);
	make_function("CADADR", Lcadadr);
	make_function("CADDAR", Lcaddar);
	make_function("CADDDR", Lcadddr);
	make_function("CDAAAR", Lcdaaar);
	make_function("CDAADR", Lcdaadr);
	make_function("CDADAR", Lcdadar);
	make_function("CDADDR", Lcdaddr);
	make_function("CDDAAR", Lcddaar);
	make_function("CDDADR", Lcddadr);
	make_function("CDDDAR", Lcdddar);
	make_function("CDDDDR", Lcddddr);

	make_function("CONS", Lcons);
	make_function("TREE-EQUAL", Ltree_equal);
	make_function("ENDP", Lendp);
	make_function("LIST-LENGTH", Llist_length);
	make_function("NTH", Lnth);

	make_function("FIRST", Lcar);
	make_function("SECOND", Lcadr);
	make_function("THIRD", Lcaddr);
	make_function("FOURTH", Lcadddr);
	make_function("FIFTH", Lfifth);
	make_function("SIXTH", Lsixth);
	make_function("SEVENTH", Lseventh);
	make_function("EIGHTH", Leighth);
	make_function("NINTH", Lninth);
	make_function("TENTH", Ltenth);

	make_function("REST", Lcdr);
	make_function("NTHCDR", Lnthcdr);
	make_function("LAST", Llast);
	make_function("LIST", Llist);
	make_function("LIST*", LlistA);
	make_function("MAKE-LIST", Lmake_list);
	make_function("APPEND", Lappend);
	make_function("COPY-LIST", Lcopy_list);
	make_function("COPY-ALIST", Lcopy_alist);
	make_function("COPY-TREE", Lcopy_tree);
	make_function("REVAPPEND", Lrevappend);
	make_function("NCONC", Lnconc);
	make_function("NRECONC", Lreconc);

	make_function("BUTLAST", Lbutlast);
	make_function("NBUTLAST", Lnbutlast);
	make_function("LDIFF", Lldiff);
	make_function("RPLACA", Lrplaca);
	make_function("RPLACD", Lrplacd);
	make_function("SUBST", Lsubst);
	make_function("SUBST-IF", Lsubst_if);
	make_function("SUBST-IF-NOT", Lsubst_if_not);
	make_function("NSUBST", Lnsubst);
	make_function("NSUBST-IF", Lnsubst_if);
	make_function("NSUBST-IF-NOT", Lnsubst_if_not);
	make_function("SUBLIS", Lsublis);
	make_function("NSUBLIS", Lnsublis);
	make_function("MEMBER", Lmember);
	make_function("MEMBER-IF", Lmember_if);
	make_function("MEMBER-IF-NOT", Lmember_if_not);
	make_si_function("MEMBER1", Lmember1);
	make_function("TAILP", Ltailp);
	make_function("ADJOIN", Ladjoin);

	make_function("ACONS", Lacons);
	make_function("PAIRLIS", Lpairlis);
	make_function("ASSOC", Lassoc);
	make_function("ASSOC-IF", Lassoc_if);
	make_function("ASSOC-IF-NOT", Lassoc_if_not);
	make_function("RASSOC", Lrassoc);
	make_function("RASSOC-IF", Lrassoc_if);
	make_function("RASSOC-IF-NOT", Lrassoc_if_not);

	make_si_function("MEMQ", siLmemq);

}
