(in-package "PT")

(defun deferred-evaluation-handler (stream subchar arg)
  (declare (ignore subchar arg))
  `(push ',(read stream t nil t) *deferred-clauses*))
(defun evaluate-deferred-clauses ()
  (mapcar #'eval (reverse *deferred-clauses*))
  (setq *deferred-clauses* nil))
(defun clear-deferred-clauses ()
  (setf *deferred-clauses* nil))
(defun show-deferred-clauses ()
  *deferred-clauses*)

(eval-when (compile load eval)
  (set-dispatch-macro-character #\# #\{ #'deferred-evaluation-handler))

;;;
;;;     Maps events into default handlers
;;;
 
(defmacro defevents (class-name &rest specs)
   `(let ((mapping nil)
          (hdata nil)
          (temp nil)
          (func nil))
         (dolist (sp ',specs)
                 (setq mapping (parse-event-spec (car sp)))
                 (if (null mapping)
                     (warn "illegal default-event-mapping:  ~S for:  ~S"
                           sp ',class-name)
                     (progn
                      (setq func (cadr sp))
                      (setq hdata (gethash ',class-name *event-handler-table*))
                      (if (setq temp (member mapping hdata :test #'ev-map-test))
                         (setf (caar temp) func)
                          (setf (gethash ',class-name *event-handler-table*)
                                (nconc hdata
                                       (list (list func (list mapping)))))))))))

;;
;;      Registers event-handler with optional default mapping.
;;

(defmacro defhandler (name arglist &rest body
                           &aux class-name func-name mapping declares doc)
  (if (atom (car arglist))
      (error "bad first-argument-form:  ~S for event-handler:  ~S"
             (car `,arglist) `,name))
  (setq class-name `,(cadar arglist))
  (setq func-name
        `,(read-from-string
           (concatenate 'string (string class-name) "-" (string name))))
  (setq arglist `,(cons (caar arglist) (cdr arglist)))
  (setq mapping (cadr (member '&default arglist)))
  (when mapping
        (setq mapping
              (if (and (consp mapping) (listp (car mapping)))
                  (mapcar #'parse-event-spec mapping)
                  (list (parse-event-spec mapping))))
        (if (member nil mapping)
            (warn "illegal default-event-mapping:  ~S for:  ~S"
                  (cdr (member '&default arglist))
                  class-name)))
  (if (stringp (car body))
      (setq doc (car body)
            body (cdr body))
      (setq doc ""))

  (if (and (listp (car body))
           (eql (caar body) 'declare))
      (setq declares (car body)
            body (cdr body))
      (setq declares ""))

  ;;  Define function handler
  (if (evenp (length arglist))
      (remf arglist '&default)
      (remf (cdr arglist) '&default))

  ;;  Register handler
  `(progn
    (let ((hdata (gethash ',class-name *event-handler-table*))
          (temp nil))
          (if (setq temp (member ',name hdata :test #'ev-hand-test))
              (setf (cadar temp) ',mapping)
              (setf (gethash ',class-name *event-handler-table*)
                    (nconc hdata ',(list (list name mapping))))))
    (defun ,func-name ,arglist ,doc ,declares (block ,name (let nil ,@body)))))

(defun ev-hand-test (a b)
  (eql a (car b)))

(defun ev-map-test (a b)
  (equal a (cadr b)))

(defun parse-event-spec (spec)
  (when spec
        (if (atom spec)
            (setq spec (list spec)))
        (if (oddp (length spec))
            (if (or (member :state spec) (member :detail spec))
                (setq spec
                      (list (car spec)
                            (getf (cdr spec) :state)
                            (getf (cdr spec) :detail)))
                (let ((temp (make-list 3)))
                     (setf (first temp) (first spec)
                           (second temp) (convert-state (second spec))
                           (third temp) (convert-detail (first spec)
                                                        (third spec)))
                     (setq spec temp)))
            (if (or (member :state spec) (member :detail spec))
                (setq spec nil)
                (let ((temp (make-list 3)))
                     (setf (first temp) (first spec)
                           (second temp) (convert-state (second spec))
                           (third temp) (convert-detail (first spec)
                                                        (third spec)))
                     (setq spec temp))))
        spec))

(defun convert-state-key (key)
#+lucid
(if (stringp key)
    (setq key (intern key "KEYWORD"))
  (setq key (intern (symbol-name key) "KEYWORD")))
#+allegro
  (setq key (intern (symbol-name key) "KEYWORD"))
  (case key
        (:left-button :button-1)
        (:middle-button :button-2)
        (:right-button :button-3)
        (:meta :mod-1)
        (t key)))

(defun convert-state (state)
  (unless (listp state) (setq state (list state)))
  (mapcar #'convert-state-key state))

(defun convert-detail (event-type detail)
  (case event-type
	((:pointer-motion :button-motion :button-1-motion :button-2-motion
			  :button-3-motion :button-4-motion
			  :button-5-motion)
	 (case detail 
	       ((:left-button :button-1) (xlib:make-state-mask :button-1))
	       ((:middle-button :button-2) (xlib:make-state-mask :button-2))
	       ((:right-button :button-3) (xlib:make-state-mask :button-3))
	       ((:button-4 :button-5) 
		(xlib:make-state-mask detail))
	       (t nil)))
	(t
	 (typecase detail
		   (keyword 
		    (case detail 
			  ((:left-button :button-1) 1) 
			  ((:middle-button :button-2) 2) 
			  ((:right-button :button-3) 3)
			  (t detail)))
		   (character detail) 
		   (string
                    (if (= 1 (length detail)) (character detail) detail))
;; causes      	   (string
;; CLX 11R5	    (let ((char (character detail)))
;; to die		 (if char char detail)))
		   (null 0)
		   (t detail)))))


;;; from base/gc.cl

(defmacro make-shared-gc (window spec)
  `(make-gc ,window ,spec t))

;;; from base/synths.cl

(defmacro synth-p (obj)
  "Return t if obj is a synthetic-gadget, nil otherwise"
  `(and (listp ,obj)
        (or (= (length ,obj) 1)
            (keywordp (cadr ,obj)))))

(defmacro synth-exposed-p (synth)
  "Return whether the synthetic gadget is exposed (visible on screen)."
  `(getf ,synth :exposed))

(defmacro exposed-synths-of (synth-list)
  "Return a list of exposed synthetic gadgets"
  `(mapcan #'(lambda (ch) (if (synth-exposed-p ch) (list ch))) ,synth-list))

(defmacro synth-width (synth)
  "Return the 'desired' width of the synthetic gadget"
  `(let ((val (car ,synth)))
        (cond ((stringp val)
               (let ((font (getf (cdr ,synth) :font)))
                    (cond ((font-p font))
                          ((stringp font)
                           (setq font (get-font font)))
                          ((null font)
                           (setq font (get-font))))
                    (text-width val :font font)))
              ((not (listp val))
               (width val))
              ((null val) 0)
              #|(t (apply #'+
                        (mapcar #'(lambda (x) (intern-synth-width
                                               (cons x (cdr val))))
                                val)))|#
              (t (error "synth-width: can't compute width of list \`~s\`" val))
              )))

(defmacro synth-height (synth)
  "Return the 'desired' height of the synthetic gadget"
  `(let ((val (car ,synth)))
        (cond ((stringp val)
               (let ((font (getf (cdr ,synth) :font)))
                    (cond ((font-p font))
                          ((stringp font)
                           (setq font (get-font font)))
                          ((null font)
                           (setq font (get-font))))
                    (font-height font)))
              ((not (listp val))
               (height val))
              ((null val) 0)
              #|(t (apply #'+
                        (mapcar #'(lambda (x) (intern-synth-height
                                               (cons x (cdr val))))
                                val)))|#
              (t (error "synth-height: can't compute width of list \`~s\`" val))
              )))

;; geometry management macros

(defmacro initialize-gm (self)
  "Initialize a collection for a (new) geometry manager"
  `(gm-initialize (gm ,self) ,self))

(defmacro add-child (self child)
  "Add a child to a geometry manager"
  `(progn
    (setf (min-size ,self) nil)
    (gm-add-child (gm ,self) ,self ,child)))

(defmacro delete-child (self child)
  "Delete a child from a geometry manager"
  `(progn
    (setf (min-size ,self) nil)
    (gm-delete-child (gm ,self) ,self ,child)))

(defmacro calculate-min-size (self)
  "Recalculate the minimum size for a collection"
  `(gm-calculate-min-size (gm ,self) ,self))

;; general picasso macros 

(defmacro picasso-name-p (name)
  `(not (eq (name-type ,name) :error)))

(defmacro var-value-form-p (var)
  `(and (listp ,var) (eql (length ,var) 2) (eq (car ,var) 'value)))

(defmacro resolve-po (place)
  (let ((pl (eval place)))
       (cond ((typep pl 'picasso-object) `',pl)
             ((picasso-name-p pl) `(find-picasso-object-named ',pl))
             (t (setq pl (eval pl))
                (cond ((typep pl 'picasso-object) `',pl)
                      ((picasso-name-p pl) `(find-picasso-object-named ',pl))
                      (t (error "Bad picasso-object form")))))))

(defmacro resolve-and-set-po (place)
  (let ((pl (eval place)))
       (cond ((typep pl 'picasso-object) `',pl)
             ((picasso-name-p pl)
              (eval `(setf ,place (find-picasso-object-named ',pl)))
              (eval place))
             (t (setq pl (eval pl))
                (cond ((typep pl 'picasso-object) `',pl)
                      ((picasso-name-p pl) `(find-picasso-object-named ',pl))
                      (t (error "Bad picasso-object form")))))))

(defmacro link (variable)
  `(progn
    (unless (var-value-form-p ',variable)
            (error "link:  passed non variable form ~S" ',variable))
    (if (picasso-name-p ,variable)
        (setf ,variable (find-picasso-object-named ,variable)))))

(defmacro colonize (sym)
  `(intern (symbol-name ,sym) "KEYWORD"))

(defmacro remove-quote (thing)
  `(if (and (consp ,thing) (eq (car ,thing) 'quote))
       (setq ,thing (cadr ,thing))))

;;;
;;; determine if an object is a funcallable object
;;;

(defmacro function-p (value)
  "determine if the specified argument is a function or bound to a function"
  `(if (symbolp ,value)
      (fboundp ,value)
    (functionp ,value)))

(defmacro current-display ()
  "Return the current display instance or nil if no display"
  `*current-display*)

(defmacro default-display ()
  `(current-display))

(defmacro current-screen (&optional display)
  "Return the current screen instance or nil if no screen"
  `(cond ((null ,display) *current-screen*)
         ((display-p ,display)
          (primary-screen ,display))
         (,display
          (error "current-screen: invalid display \`~s\`." ,display))
         (t *current-screen*)))

(defmacro default-screen ()
  `(current-screen))

(defmacro root-window (&optional screen)
  "Return the current root-window or nil if no root-window"
  `(cond ((null ,screen) *current-root*)
         ((screen-p ,screen)
          (root ,screen))
         ((display-p ,screen)
          (root (primary-screen ,screen)))
         (t (error "root-window: invalid source \`~s\`." ,screen))))

(defmacro current-root ()
  `(root-window))

(defmacro default-root ()
  `(root-window))

(defmacro current-colormap (&optional source)
  `(cond ((null ,source)
          (when *current-root*
                (colormap *current-root*)))
         ((screen-p ,source)
          (colormap (root ,source)))
         ((display-p ,source)
          (colormap (root (primary-screen ,source))))
         ((window-p ,source)
          (colormap ,source))
         (t (error "current-colormap: invalid source \`~s\`." ,source))))

(defmacro default-colormap (&optional source)
  `(current-colormap ,source))

(defmacro resource-p (obj)
  `(member (class-name (class-of ,obj)) *resource-types*))

(defmacro paint-p (obj)
  `(member (class-name (class-of ,obj)) *paint-types*))


(defmacro current-font-table (&optional display)
  `(if ,display
       (font-table ,display)
       (font-table (current-display))))

(defmacro font-name-p (name &optional display)
  `(not (null
         (if ,display
             (xlib:list-font-names (res ,display) ,name)
             (xlib:list-font-names (res (current-display)) ,name)))))

(defmacro find-screen (num &optional display)
  `(if ,display
       (get-screen ,num ,display)
       (get-screen ,num)))

(defmacro find-font (name &optional display)
  `(if ,display
       (get-font ,name ,display)
       (get-font ,name)))

(defmacro find-color (name &optional colormap)
  `(if ,colormap
       (get-color ,name ,colormap)
       (get-color ,name)))

(defmacro find-paint (name &optional display)
  `(if ,display
       (get-paint ,name ,display)
       (get-paint ,name)))

(defmacro find-image (name)
  `(get-image ,name))

(defmacro white-pixel (&optional display)
  `(cond ((null ,display)
          (xlib:screen-white-pixel (res (current-screen))))
         ((display-p ,display)
          (xlib:screen-white-pixel
           (xlib:display-default-screen (res ,display))))
         ((screen-p ,display)
          (xlib:screen-white-pixel (res ,display)))
         (t (error "white-pixel: invalid argument \`~s\`." ,display))))

(defmacro black-pixel (&optional display)
  `(cond ((null ,display)
          (xlib:screen-black-pixel (res (current-screen))))
         ((display-p ,display)
          (xlib:screen-black-pixel
           (xlib:display-default-screen (res ,display))))
         ((screen-p ,display)
          (xlib:screen-black-pixel (res ,display)))
         (t (error "black-pixel: invalid argument \`~s\`." ,display))))

(defmacro find-window-table (display-res)
  `(gethash ,display-res *global-window-table-hashtab*))

(defmacro quick-find-window (window-res display-res)
  `(gethash ,window-res (find-window-table ,display-res)))

(defmacro find-window (window-res &optional display)
  "return the window object with the specified resource id"
  `(gethash
    ,window-res
    (cond ((null ,display)
           (find-window-table
            (xlib:window-display ,window-res)))
          ((display-p ,display) (window-hash-table ,display))
          ((screen-p ,display) (window-hash-table (display ,display)))
          (t (error "find-window: invalid argument \`~s\`." ,display)))))

(defmacro find-from-objid (objid)
  `(gethash ,objid *objid-window-table*))


;; color macros

(defmacro color-attach (self)
  `(do-attach ,self))

(defmacro color-detach (self)
  `(do-detach ,self))

(defmacro default-cursor ()
  `(get-cursor *default-cursor-name*))


(defmacro black-or-white-color-p (color)
  `(or (eq ,color (find-color "white"))
       (eq ,color (find-color "black"))))

(defmacro font-attach (self)
  `(do-attach ,self))

(defmacro font-detach (self)
  `(do-detach ,self))

(defmacro image-attach (self)
  `(do-attach ,self))

(defmacro image-detach (self)
  `(do-detach ,self))


;;; graphics macros

(defmacro nil-bbox-p (bb) `(null ,bb))
(defmacro 2d-bbox-p (bb) `(= 4 (length ,bb)))

(defmacro make-2d-bbox (xmin ymin xmax ymax) `(list ,xmin ,ymin ,xmax ,ymax))

(defmacro 2d-bb-xmin (bb) `(first ,bb))
(defmacro 2d-bb-ymin (bb) `(second ,bb))
(defmacro 2d-bb-xmax (bb) `(third ,bb))
(defmacro 2d-bb-ymax (bb) `(fourth ,bb))

(defmacro free-2d (v) `(vector-push-extend ,v *2d-vector-free-array*))

(defmacro copy-2d (v x y)
  `(setf (2d-point-x ,v) ,x (2d-point-y ,v) ,y))

(defmacro 2dv-copy (dst src)
  `(setf (2d-point-x ,dst) (2d-point-x ,src)
         (2d-point-y ,dst) (2d-point-y ,src)))

(defmacro 2dv-zerop (v1)
  "Is this vector the zero vector?"
  (and (= 0 (2d-point-x v1) (2d-point-y v1))))

(defmacro validate (dl mapper)
  `(unless (valid ,dl)
           (do-validation ,dl ,mapper)
           (setf (slot-value ,dl 'valid) t)))

(defmacro invalidate (dl)
  `(setf (slot-value ,dl 'valid) nil))

;;;
;;; dopoints is a macro to iterate through all the points of a line string.
;;; p1 and p2 are assigned succesive pairs of the line segments for the line
;;; string passed.
;;;
(defmacro dopoints ((p1 p2 linestr closed) &rest body)
  `(let* ((,p1 (car ,linestr))
          (,p2 nil))
         (dolist (,p2 (cdr ,linestr))
                 ,@body
                 (setq ,p1 ,p2))
         ;; Do wrap-around
         (when ,closed
               (setq ,p2 (car ,linestr))
               ,@body)))

(defmacro map-dc-to-wc (self dx dy wx wy)
  `(setf ,wx (/ (- ,dx (bx ,self)) (mx ,self))
         ,wy (/ (- ,dy (by ,self)) (my ,self))))

(defmacro map-wc-to-dc (self wx wy dx dy)
  `(setf ,dx (truncate (+ 0.5 (* (mx ,self) ,wx) (bx ,self)))
         ,dy (truncate (+ 0.5 (* (my ,self) ,wy) (by ,self)))))


(defmacro pw-make-point (x y)
  `(cons ,x ,y))

(defmacro x-crd (point)
  `(car ,point))

(defmacro y-crd (point)
  `(cdr ,point))

(defmacro set-x-crd (point val)
  `(setf (car ,point) ,val))

(defmacro set-y-crd (point val)
  `(setf (cdr ,point) ,val))

(defmacro make-sgg (&rest keys)
  `(make-simple-graphic-gadget ,@keys))

;; menu macros

(defmacro exposed-mes-of (synth-list)
  "Return a list of exposed menu-items gadgets"
  `(mapcan #'(lambda (ch) (if (exposed ch) (list ch))) ,synth-list))

(defmacro exposed-mis-of (synth-list)
  "Return a list of exposed menu-items gadgets"
  `(mapcan #'(lambda (ch) (if (exposed ch) (list ch))) ,synth-list))

(defmacro vertical-p (sb)
  `(eq (orientation ,sb) :vertical))

(defmacro clamp (min x max)
  `(min (max ,x ,min) ,max))

(defmacro buttons-top (sb)
  `(or (eq (button-pos ,sb) :top-left)
       (eq (button-pos ,sb) :top-right)))

(defmacro buttons-bottom (sb)
  `(or (eq (button-pos ,sb) :bottom-left)
       (eq (button-pos ,sb) :bottom-right)))

(defmacro buttons-right (sb)
  `(or (eq (button-pos ,sb) :bottom-right)
       (eq (button-pos ,sb) :top-right)))

(defmacro buttons-left (sb)
  `(or (eq (button-pos ,sb) :bottom-left)
       (eq (button-pos ,sb) :top-left)))

(defmacro mf-scroll-up (mf n)
  `(funcall (up-func ,mf) ,mf ,n))

(defmacro mf-scroll-down (mf n)
  `(funcall (down-func ,mf) ,mf ,n))

(defmacro mf-scroll-left (mf n)
  `(funcall (left-func ,mf) ,mf ,n))

(defmacro mf-scroll-right (mf n)
  `(funcall (right-func ,mf) ,mf ,n))

(defmacro un-current-fields-by-row (mf)
  `(let ((all-ch (mapcar #'geom-spec (children ,mf)))
         (cur-ch (slot-value ,mf 'current-indices)))
        ;;      Return list of all fields not current
        (sort (set-difference all-ch cur-ch :test #'pair-comp) #'car-comp)))

(defmacro un-current-fields-by-col (mf)
  `(let ((all-ch (mapcar #'geom-spec (children ,mf)))
         (cur-ch (slot-value ,mf 'current-indices)))
        ;;      Return list of all fields not current
        (sort (set-difference all-ch cur-ch :test #'pair-comp) #'cadr-comp)))

