;;;	(c) Copyright 1989, 1990, 1991 Sun Microsystems, Inc. 
;;;	Sun design patents pending in the U.S. and foreign countries. 
;;;	See LEGAL_NOTICE file for terms of the license.

;;;@(#)xview-input.lisp	3.34 10/11/91


(in-package "LISPVIEW")


;;; Setting the keyboard-focus requires a timestamp, typically the value of this
;;; timestamp comes from the ClientMessage WM_TAKE_FOCUS event.  See the Input Focus
;;; section of the ICCCM for more information.  Solo updates the variable below 
;;; each time an X11 event with a timestamp is received.

(defvar *x11-current-time* X11:CurrentTime)

(proclaim '(integer *x11-current-time*))

(defmacro update-x11-current-time (time)
  (let ((time-var (gensym)))
    `(let ((,time-var ,time))
       (declare (integer ,time-var))
       (when (> ,time-var *x11-current-time*)
	 (setq *x11-current-time* ,time-var))
       ,time-var)))


;;; These definitions are based on /usr/include/sys/types.h for SunOS 4.0.
;;; The return-type and width fields of select have been reduced from C ints
;;; (i.e. :signed-32bit integers) to :fixnums because in both cases a fixnum
;;; is big enough for this application.  Any changes to these defintions should
;;; be carefully sync'd with definition of notifier-loop.

(defconstant n-fd-bits 32)

(defconstant n-fd-set-ints 8)

(def-foreign-synonym-type fds-bits
  (:array :unsigned-32bit (8)))

(def-foreign-struct fd-set
  (fds-bits :type fds-bits))

(def-foreign-function (select (:return-type :fixnum))
  (width :fixnum)
  (readfds (:pointer fd-set))
  (writefds (:pointer fd-set))
  (exceptfds (:pointer fd-set))
  (timeout (:pointer XV:timeval)))



;;; Solo Global Input Processs - XView Notifier Interface
;;;
;;; The notifier interface is just a loop that waits until some input is 
;;; available on the X server connection and then calls XV:notify-dispatch
;;; repeatedly until the input has been exhausted.
;;;
;;; This function saves a little time by implementing FD_ZERO, FD_CLR in
;;; lisp code.  It should be checked carefully during a port.

(defconstant xview-notifier-waiting-whostate "Input Wait")


(defvar *xview-notifier-must-run* nil)
(defvar *xview-notifier-shouldnt-wait* nil)

(defvar *xview-notifier-timeout0* 100000) ;; .1 seconds
(defvar *xview-notifier-timeout1* 100000) ;; 

(defun xview-notifier-loop (dsp)
  (let* ((fd (X11:XConnectionNumber dsp))
	 (nullfds (make-foreign-pointer :type '(:pointer fd-set) :address 0))
	 (readfds (make-fd-set))
	 (bits (fd-set-fds-bits readfds))
	 (fd-set-elt (truncate fd n-fd-bits))
	 (fd-set-bit (expt 2 (mod fd n-fd-bits)))
	 (timeout0 (XV:make-timeval :tv-sec 0 :tv-usec *xview-notifier-timeout0*))
	 (timeout1 (XV:make-timeval :tv-sec 0 :tv-usec *xview-notifier-timeout1*)))
    (declare (fixnum fd-set-elt fd-set-bit))
    (macrolet 
     ((input-pending-p (timeout)
	`(or (/= 0 (logand XV:ndet-condition-change (foreign-value XV:*ndet-flags*)))
	     (progn
	       ;; FD_SET(readfds, fd)
	       (setf (typed-foreign-aref '(:pointer fds-bits) bits fd-set-elt)
		     fd-set-bit)
	       (= 1 (the fixnum (select n-fd-set-ints readfds nullfds nullfds ,timeout)))))))

     (dotimes (i n-fd-set-ints)             ;; FD_ZERO(readfds)
       (setf (foreign-aref bits i) 0))
     (X11:XSync dsp 0)
     (loop
       (process-wait xview-notifier-waiting-whostate
		     #'(lambda () (input-pending-p timeout0)))
       (loop 
	 (XV:with-xview-lock (XV:notify-dispatch))
	 (when *xview-notifier-shouldnt-wait*
	   (setq *xview-notifier-shouldnt-wait* nil)
	   (process-allow-schedule))
	 (unless (input-pending-p timeout1)
	   (return)))))))


;;; Start the notifier process and wait until it has starting waiting on the 
;;; X servers socket before returning.  This guarantees (modulo Lisp scheduler
;;; performance) that X will not timeout waiting for Lisp to accept all the events
;;; generated by starting XView.

(defvar *xview-notifier-process*)

(defconstant bad-disksave-warning 
  "The XView Notifier process appears to have been running when this image was
;;; disksaved.  It's likely that this LispView image will not work correctly.")

(defun start-xview-notifier (dsp)
  (when (or (not (boundp '*xview-notifier-process*))
	    (let ((state  (process-state *xview-notifier-process*)))
	      (when (eq state :flushed)
		(warn bad-disksave-warning))
	      (member state '(:flushed :killed))))
    (setq *xview-notifier-process*
	  (make-process 
	    :name "XView Notifier"
	    :function 'xview-notifier-loop
	    :args (list dsp)))
    (process-wait "Starting XView Notifier"
		  #'(lambda ()
		      (equal (process-whostate *xview-notifier-process*) 
			     xview-notifier-waiting-whostate)))))


;;; Return the X11 input mask (e.g. fixnum argument to XSelectInput) that corresponds
;;; to a particular Solo interest.

(defconstant x11-any-button-motion-mask 
  (logior X11:Button1MotionMask  
	  X11:Button2MotionMask  
	  X11:Button3MotionMask 
	  X11:Button4MotionMask 
	  X11:Button5MotionMask))
   

(defmacro x11-modifier-to-mask (m)
  `(case ,m
     (:button0 #.X11:Button1MotionMask)
     (:button1 #.X11:Button2MotionMask)
     (:button2 #.X11:Button3MotionMask)
     (:button3 #.X11:Button4MotionMask)
     (:button4 #.X11:Button5MotionMask)
     (t 0)))

	
(defmethod x11-input-mask ((i mouse-interest))
  (let* ((parsed-event-spec (slot-value i 'parsed-event-spec))
	 (action (nth 1 parsed-event-spec)))
    (cond
     ((member action '((or :enter :exit) (or :exit :enter)) :test #'equal)
      #.(logior X11:EnterWindowMask 
		X11:LeaveWindowMask))
     ((eq action :enter) #.X11:EnterWindowMask)
     ((eq action :exit) #.X11:LeaveWindowMask)
     ((eq action :move)
      (let ((down-mask 0))
	 (dolist (modifier-conjunct (cdar parsed-event-spec))
	   (let ((others nil)
		 (up-mask 0))
	     (dolist (modifier (cdr modifier-conjunct))
	       (cond
		((keywordp modifier) 
		 (setq down-mask (logior down-mask (x11-modifier-to-mask modifier))))
		((and (consp modifier) (eq (car modifier) :others))
		 (when (member (cadr modifier) '(:down (or :up :down) (or :down up)) :test #'equal)
		   (setq others t)))
		((consp modifier) 
		 (let ((mod-mask (x11-modifier-to-mask (car modifier))))
		   (if (eq (cadr modifier) :down)
		       (setq down-mask (logior down-mask mod-mask))
		     (setq up-mask (logior up-mask mod-mask)))))))
	     (when others
	       (setf down-mask (logior down-mask (logxor (logior up-mask down-mask) 
							 x11-any-button-motion-mask))))))
	 (if (= down-mask 0) 
	     X11:PointerMotionMask 
	   (logior down-mask #.X11:ButtonPressMask)))) ;; ButtonPressMask - defeats passive grab

     (t ;; button 
      (case (nth 1 action)
	(:down #.X11:ButtonPressMask)

	((:click1 :click2 :click3 :click4)
	 (let ((mask #.(logior #.X11:ButtonPressMask #.X11:ButtonReleaseMask)))
	   (dolist (x (cdar action) mask)
	     (setf mask (logior mask (x11-modifier-to-mask (cadr x)))))))

	(t #.(logior #.X11:ButtonPressMask #.X11:ButtonReleaseMask)))))))

			      
(defmethod x11-input-mask ((i keyboard-interest))
  (let ((action (cadr (slot-value i 'event-spec))))
    (case action
      (:down X11:KeyPressMask)
      (:up   X11:KeyReleaseMask)
      (t     (logior X11:KeyPressMask X11:KeyReleaseMask)))))


(defmethod x11-input-mask ((i damage-interest))
  #.X11:ExposureMask)

(defmethod x11-input-mask ((i keyboard-focus-interest))
  #.X11:FocusChangeMask)

(defmethod x11-input-mask ((i visibility-change-interest))
  #.X11:VisibilityChangeMask)





;;; Return a list of the X11 types (fixnum, see the type slot of XEvent union) that 
;;; correspond to a Solo interest.

(defmethod x11-event-types ((i mouse-interest))
  (let* ((parsed-event-spec (slot-value i 'parsed-event-spec))
	 (action (nth 1 parsed-event-spec)))
    (cond
     ((member action '((or :enter :exit) (or :exit :enter)) :test #'equal)
      '#.(list X11:EnterNotify X11:LeaveNotify))
     ((eq action :enter) '#.(list X11:EnterNotify))
     ((eq action :exit) '#.(list X11:LeaveNotify))
     ((eq action :move) '#.(list X11:MotionNotify))
     (t ;; button 
      (case (nth 1 action)
	((:down :click1 :click2 :click3 :click4) '#.(list X11:ButtonPress))
	(:up '#.(list X11:ButtonRelease))
	(t '#.(list X11:ButtonPress X11:ButtonRelease)))))))

(defmethod x11-event-types ((i keyboard-interest))
  (let ((action (cadr (slot-value i 'event-spec))))
    (case action
      (:down '#.(list X11:KeyPress))
      (:up   '#.(list X11:KeyRelease))
      (t     '#.(list X11:KeyPress X11:KeyRelease)))))

(defmethod x11-event-types ((i keyboard-focus-interest))
  '#.(list X11:FocusIn X11:FocusOut))

(defmethod x11-event-types ((i damage-interest))
  '#.(list X11:Expose X11:GraphicsExpose))

(defmethod x11-event-types ((i visibility-change-interest))
  '#.(list X11:VisibilityNotify))


;;; Multiple-value return two fixnums: mask and match.  For the state slot of an 
;;; XEvent to match the modifier expression (modifiers) the following expression
;;; must be non nil:   (= match (logand state mask))

(defun x11-event-state-target (modifiers)
  (let ((match 0)
	(mask x11-modifier-bits)
	(other nil))
    (labels 
     ((name-to-bit (modifier)        ;; assumes Sun mapping of Mod1 to :meta and
        (case modifier               ;; no :hyper or :super modifier keys
	  (:shift X11:ShiftMask)
	  (:control X11:ControlMask)
	  (:meta X11:Mod1Mask)
	  (:button0 X11:Button1Mask)
	  (:button1 X11:Button2Mask)
	  (:button2 X11:Button3Mask)
	  (:button3 X11:Button4Mask)
	  (:button4 X11:Button5Mask)
	  (t (if (consp modifier) (name-to-bit (car modifier)) 0))))
       
      (modifier-to-bit (modifier)
	(typecase modifier
	  ((member :others)
	   (setq other :down) 0)
	  (keyword
	   (name-to-bit modifier))
	  (cons
	   (if (eq (nth 0 modifier) :others)
	       (progn (setq other (nth 1 modifier)) 0)
	     (let ((bit (name-to-bit (nth 0 modifier))))
	       (case (nth 1 modifier)
		 (:up 0)
		 (:down bit)
		 (t (setq mask (logand mask (lognot bit))) 0)))))
	  (t 0))))

     (dolist (m modifiers)
       (setq match (logior match (modifier-to-bit m))))
     (case other
       ((nil :up))
       (:down 
	(dolist (m modifiers)
	  (setq match (logior match (name-to-bit m)))))
       (t 
	(setq mask 0)
	(dolist (m modifiers)
	  (setq mask (logior mask (name-to-bit m)))))))
    (values mask match)))
	  


;;; Create one x11-interest for each combination of modifiers and the action.
;;; For example if modifiers = (or (and :shift :left) (and :control :left)) and 
;;; action = (or :enter :exit) the we create x11-interests for 
;;; ((:shift :left) (or :enter :exit)) and ((:control :left) (or :enter :exit)).
;;;
;;; If the action is a button press or a button release then we additionally generate
;;; one x11-interest for each button.  For example if the action was 
;;; ((or (and :left) (and :right)) :down) then we generate one x11-interest for :left
;;; and one for :right.

(defmethod make-x11-interests ((i mouse-interest))
  (let* ((mask (x11-input-mask i))
	 (expr (slot-value i 'parsed-event-spec))
	 (types (x11-event-types i))
	 (buttons 
	  (if (or (= (car types) #.X11:ButtonPress) (= (car types) #.X11:ButtonRelease))
	      (mapcar #'(lambda (buttons)
			  (case (cadr buttons)          ;; no support for button chords
			    (:button0 #.X11:Button1)
			    (:button1 #.X11:Button2)
			    (:button2 #.X11:Button3)
			    (:button3 #.X11:Button4)
			    (:button4 #.X11:Button5)))
		      (cdaadr expr))))  
	 (nclicks
	  (if (consp (cadr expr))
	      (case (cadadr expr) (:click1 1) (:click2 2) (:click3 3) (:click4 4) (t 0))
	    0)))
    (macrolet 
     ((make-interest (x11-button)
	`(make-x11-mouse-interest
	  :object i
	  :input-mask mask
	  :types types
	  :state-mask state-mask
	  :state-match state-match
	  :button ,x11-button
	  :nclicks nclicks)))
     
     (mapcan #'(lambda (modifier)
		 (multiple-value-bind (state-mask state-match)
		     (x11-event-state-target (cdr modifier))
		   (if buttons
		       (mapcar #'(lambda (button)
				   (make-interest button))
			       buttons)
		     (list (make-interest 0)))))
	     (cdar expr)))))

(defmethod make-x11-interests ((i keyboard-interest))
  (let ((key-types (car (slot-value i 'event-spec))))
    (list (make-x11-keyboard-interest
	    :object i
	    :input-mask (x11-input-mask i)
	    :types (x11-event-types i)
	    :keysym-ranges (if (consp key-types)
			       (mapcan #'key-type-keysym-range key-types)
			     (key-type-keysym-range key-types))))))

(defmethod make-x11-interests ((i damage-interest))
  (list (make-x11-damage-interest
	  :object i
	  :input-mask (x11-input-mask i)
	  :types (x11-event-types i))))

(defmethod make-x11-interests (i)
  (list (make-x11-interest
	  :object i
	  :input-mask (x11-input-mask i)
	  :types (x11-event-types i))))




;;; Update the internal interest table and return an X11 input mask for this interest.
;;;
;;; Each xview-canvas contains a table that maps from X11 event type (a small integer
;;; code) to the interests for that event type.  The interests are respresented by a
;;; the x11-{mouse,keyboard,damage}-interest structures all of which :include x11-interest.
;;; Each x11-interest struct contains the real Solo interest as well as an (integer) X11
;;; input mask that selects the events that the interest matches.  One Solo interest can 
;;; map to several x11-interest structs, see make-x11-interests.

(defun xview-insert-interest (xvo interest relation relative canvas)
  (let* ((table (xview-canvas-interest-table xvo))
	 (x11-interests (make-x11-interests interest))
	 (input-mask 0))
    (dolist (xi x11-interests input-mask)
      (dolist (type (x11-interest-types xi))
	(let ((entry (svref table type)))
	  (cond 
	   ((null entry)
	    (setf (svref table type) (list xi)))
	   ((null relative)
	    (if (eq relation :before)
		(setf (svref table type) (nconc entry (list xi)))
	      (setf (svref table type) (cons xi entry))))
	   ((and (eq relation :at) (= relative 0))
	    (setf (svref table type) (cons xi entry)))
	   (t
	    (let* ((il (slot-value canvas 'interests))
		   (pl (mapcar #'(lambda (xi)
				   (cons (position (x11-interest-object xi) il) xi))
			       entry))
		   (after (case relation
			    (:at relative)
			    (:before (1- (position relative il)))
			    (:after (position relative il))))
		   (ip (dolist (x pl)
			 (when (>= (car x) after)
			   (return (cdr x))))))
	      (setf (svref table type) 
		    (list-insert xi :before ip entry)))))))
      (setf input-mask (logior input-mask (x11-interest-input-mask xi))))))



;;; Each xview-canvas contains a table that maps from X11 event type (a small integer
;;; code) to the interests for that event type.  The interests are respresented by a
;;; the x11-{mouse,keyboard,damage}-interest structures all of which :include x11-interest.
;;; Each x11-interest struct contains the real Solo interest as well as an (integer) X11
;;; input mask that selects the events that the interest matches.  One Solo interest can 
;;; map to several x11-interest structs, see make-x11-interests.

(defmethod dd-insert-interest ((p XView) interest relation relative canvas)
  (when (eq (status canvas) :realized)
    (XV:with-xview-lock 
     (let* ((xvo (device canvas))
	    (dsp (xview-object-dsp xvo))
	    (id (xview-object-id xvo)))
       (when id
	 (XV:xv-set id 
	   :win-consume-x-event-mask 
	     (xview-insert-interest xvo interest relation relative canvas)
	   :win-ignore-x-event-mask 
	     (if (svref (xview-canvas-interest-table xvo) X11:KeyRelease)
		 0
	       X11:KeyReleaseMask))
	 (defeat-xview-passive-grab dsp (xview-object-xid xvo))
	 (xview-maybe-XFlush (xview-object-xvd xvo) dsp))))))


(defmethod dd-withdraw-interest ((p XView) interest canvas)
  (XV:with-xview-lock 
    (let* ((xvo (device canvas))
	   (id (xview-object-id xvo)))
      (when id
	(let* ((table (xview-canvas-interest-table xvo))
	       (types (x11-event-types interest))
	       (ignore 0)
	       (consume 0))
	  (dolist (type types)
	    (let ((entry (svref table type)))
	      (dolist (xi entry)
		(when (eq interest (x11-interest-object xi))
		  (setf ignore (logior ignore (x11-interest-input-mask xi)))))
	      (setf (svref table type) 
		    (delete interest entry :test #'eq :key #'x11-interest-object))))
	  (dotimes (n X11:LastEvent)
	    (dolist (xi (svref table n))
	      (setf consume (logior consume (x11-interest-input-mask xi)))))

	  (XV:xv-set id :win-ignore-x-event-mask (logand ignore (lognot consume)))
	  (defeat-xview-passive-grab (xview-object-dsp xvo) (xview-object-xid xvo))
	  (xview-maybe-XFlush (xview-object-xvd xvo)))))))



;;; Convenient version of XInternAtom.  Atom can be a string or a symbol, 
;;; in either case the value of the like named X atom is cached.  

(defun x11-intern-atom (dsp atom &optional (only-if-exists 0))
  (or (and (symbolp atom) (get atom 'x11-atom-value))
      (let* ((atom (if (symbolp atom) atom (intern (string atom))))
	     (name (malloc-foreign-string (symbol-name atom))))
	(prog1
	    (setf (get atom 'x11-atom-value) 
		  (XV:with-xview-lock 
		    (X11:XInternAtom dsp name only-if-exists)))
	  (free-foreign-pointer name)))))


;;; DAMAGE events

(defmethod match-event (canvas (event damage-event))
  (let* ((table (xview-canvas-interest-table (device canvas)))
	 (x11-interest (or (car (svref table X11:Expose))
			   (car (svref table X11:GraphicsExpose)))))
    (if x11-interest
	(x11-interest-object x11-interest))))

(defun handle-xview-damage-event (canvas window xv-event x-event)
  (declare (ignore xv-event x-event))
  (let ((rl (XV:win-get-damage window)))
    (when (/= 0 (foreign-pointer-address rl))
      (let* ((last-rectnode (XV:rectlist-rl-tailp rl))
	     (r (XV:rectlist-rl-head rl))
	     (x-offset (XV:rectlist-rl-x rl))
	     (y-offset (XV:rectlist-rl-y rl))
	     (regions nil))
	(loop
	 (let ((rect (XV:rectnode-rn-rect r)))
	   (push (make-region :left (+ (XV:rect-r-left rect) x-offset)
			      :top (+ (XV:rect-r-top rect) y-offset)
			      :width (XV:rect-r-width rect)
			      :height (XV:rect-r-height rect))
		 regions)
	   (if (= (foreign-pointer-address r) 
		  (foreign-pointer-address last-rectnode))
	       (return)
	     (setq r (XV:rectnode-rn-next r)))))
	(let* ((event (make-damage-event 
		       :timestamp *x11-current-time*
		       :regions regions))
	       (interest (match-event canvas event)))
	  (when interest
	    (deliver-event canvas interest event)))))))



;;; KEYBOARD events

(defmethod match-event (canvas (event keyboard-event))
  (let ((keysym (keyboard-event-keysym event))
	(x11-interests
	 (svref (xview-canvas-interest-table (device canvas)) (keyboard-event-type event))))
    (dolist (i x11-interests NIL)
      (dolist (range (x11-keyboard-interest-keysym-ranges i))
	(when (and (>= keysym (car range)) (<= keysym (cdr range)))
	  (return-from match-event (x11-interest-object i)))))))
  

(let* ((length 64)
       (buffer-array 
	(make-foreign-pointer 
	  :type `(:pointer (:array :character (,length)))
	  :static t))
       (buffer-fp (foreign-array-to-pointer buffer-array))
       (keysym 
	(make-foreign-pointer 
	  :type '(:pointer X11:KeySym)
	  :static t))
       (null-compose-status (make-null-foreign-pointer 'X11:XComposeStatus)))

  (defun HANDLE-XVIEW-KEYBOARD-EVENT (canvas window xv-event x-event)
    (declare (ignore window xv-event)
	     (optimize (safety 1) (speed 3) (compilation-speed 0)))
    (let* ((x-event (X11:XEvent-XKey x-event))      
	   (n (X11:XLookupString x-event buffer-fp (1- length) keysym null-compose-status))
	   (string 
	    (when (> n 0) 
	      (if (eql (foreign-aref buffer-array 0) #\null)
		  #.(string #\null)
		(foreign-string-value buffer-array))))
	   (event
	    (make-keyboard-event 
	      :timestamp (update-x11-current-time (X11:XKeyEvent-time x-event))
	      :x (X11:XKeyEvent-x x-event)
	      :y (X11:XKeyEvent-y x-event)
	      :type (X11:XKeyEvent-type x-event)
	      :state (X11:XKeyEvent-state x-event)
	      :keysym (foreign-value keysym)
	      :string string))
	   (interest (match-event canvas event)))
	(when interest
	  (deliver-event (virtual-keyboard-focus canvas) interest event)
	  (setq *xview-notifier-shouldnt-wait* t)))))


(defmethod dd-keyboard-event-char ((p XView) event)
  (let ((string (keyboard-event-string event))
	(modifiers (keyboard-event-state event)))
    (if string
	(let ((char (schar string 0)))
	  (if (= (logand modifiers X11:Mod1Mask) X11:Mod1Mask)
	      (set-char-bit char :meta t)
	    char))
      #\null)))


(defmethod dd-keyboard-event-action ((p XView) event)
  (if (= (keyboard-event-type event) X11:KeyPress)
      :down
    :up))


(defmethod dd-keyboard-event-modifiers ((p XView) event)
  (x11-event-state-to-list (keyboard-event-state event)))


(defun key-type-keysym-range (type)
  (case type
    (:ascii                 (list '#.(cons X11:|XK-space|      X11:|XK-ydiaeresis|) 
				  '#.(cons X11:|XK-BackSpace|  X11:|XK-BackSpace|)
				  '#.(cons X11:|XK-Tab|        X11:|XK-Tab|)
				  '#.(cons X11:|XK-Linefeed|   X11:|XK-Linefeed|)
				  '#.(cons X11:|XK-Return|     X11:|XK-Return|)
				  '#.(cons X11:|XK-Escape|     X11:|XK-Escape|)
				  '#.(cons X11:|XK-Delete|     X11:|XK-Delete|)))
    (:modifier              (list '#.(cons X11:|XK-Shift-L|    X11:|XK-Hyper-R|)))
    (:cursor                (list '#.(cons X11:|XK-Home|   (1- X11:|XK-Select|))))
    (:function              (list '#.(cons X11:|XK-F1|         X11:|XK-F35|)))
    (:keypad                (list '#.(cons X11:|XK-KP-Space|   X11:|XK-KP-Equal|)))
    (:misc-function         (list '#.(cons X11:|XK-Select| (1- X11:|XK-KP-Space|))))
    (:programmable-function (list '#.(cons X11:|XK-KP-F1| X11:|XK-KP-F4|)))))



;;; MOUSE-BUTTON events

(defvar *click-interval-int* (* 0.3 internal-time-units-per-second))

(defvar *click-interval-ext* 0.3)

(defmethod click-interval () 
  *click-interval-ext*)

(defmethod (setf click-interval) (value)
  (check-type value number)
  (when (>  value 1.0)
    (warn "setting click-interval to unexpectedly large value ~S" value))
  (setf *click-interval-int* (* internal-time-units-per-second value)
	*click-interval-ext* value))


(defvar *click-travel-ext* 3)
(defvar *click-travel-int* 3)

(defmethod click-travel () 
  *click-travel-ext*)

(defmethod (setf click-travel) (value) 
  (check-type value number)
  (when (> value 64)
    (warn "setting click-travel to unexpectedly large value ~S" value))
  (setf *click-travel-int* (truncate value)
	*click-travel-ext* value))


(defun match-x11-click (click-interests button-event)
  ;; Incorporated "grab-patch" and "click2" patches.  -csp 8/8/91
  (let* ((queued-event (X11:make-XEvent))
	 (expiration-time nil)
	 (n-transitions 1) ;; initial down transition
	 (x0 (X11:XButtonEvent-x button-event))
	 (y0 (X11:XButtonEvent-y button-event))
	 (dsp (X11:XButtonEvent-display button-event))
	 (timer nil))

    (labels
     ((return-matching-interest ()
	(return-from match-x11-click 
           (if (oddp n-transitions)
	       nil
	     (let ((nclicks (truncate n-transitions 2)))
	       (dolist (xvi click-interests nil)
		 (when (= nclicks (x11-mouse-interest-nclicks xvi))
		   (return (x11-interest-object xvi))))))))
      ;; Call this to enable receipt of later events.  We call it after each
      ;; event, to be conservative.
      (xallow ()
	;; How amusing: if the mode is "sync", the server crashes.  -csp 7/7/91
	(X11:XAllowEvents dsp X11:AsyncPointer x11:CurrentTime))
      (start-timer ()
	(xallow)
	(setf expiration-time 
	        (+ (get-internal-real-time) *click-interval-int*)
	      timer 
		(make-process
		 :name "LispView Button Click Timer"
		 :function
		   #'(lambda ()
		       (process-wait "Click Timer Wait"  
				     #'(lambda () 
					 (> (get-internal-real-time) expiration-time)))))))

      (restart-timer ()
        (incf n-transitions)
	(xallow)
        (setf expiration-time (+ (get-internal-real-time) *click-interval-int*)))

      (timer-expired-p ()
	(not (and (MP:processp timer) (MP:process-alive-p timer)))))

     (LCL:let-globally ((LCL:*scheduling-quantum* 25))
       (start-timer)
       (loop
	(process-wait "Input Wait" 
		      #'(lambda () 
			  (or (/= 0 (X11:XPending dsp))
			      (timer-expired-p))))
	(if (timer-expired-p)
	    (return-matching-interest)
	  (X11:XPeekEvent dsp queued-event))

	(case (X11:XEvent-type queued-event)
	  ;; "click2" patch:
	  ;; Duplicate the motion check for button events too.  Previously,
	  ;; there was an (apparently false) assumption that if the mouse moved
	  ;; between two clicks, there would always be a motion event between the
	  ;; click events.  It turns out that this is not guaranteed.  So
	  ;; adding this extra check prevents mis-interpretation of a fast
	  ;; click-move-click as if it were a :click2.
	  ;;
	  ((#.X11:BUTTONPRESS #.X11:BUTTONRELEASE)
	   (LET* ((BUTTON-EVENT (MAKE-FOREIGN-POINTER 
				  :TYPE '(:POINTER X11:XBUTTONEVENT)
				  :ADDRESS (FOREIGN-POINTER-ADDRESS QUEUED-EVENT)))
		  (X1 (X11:XBUTTONEVENT-X BUTTON-EVENT))
		  (Y1 (X11:XBUTTONEVENT-Y BUTTON-EVENT)))
	     (IF (OR (> (ABS (- X0 X1)) *CLICK-TRAVEL-INT*)
		     (> (ABS (- Y0 Y1)) *CLICK-TRAVEL-INT*))
		 (RETURN-MATCHING-INTEREST)
	       (PROGN
		 (X11:XNEXTEVENT DSP QUEUED-EVENT)
		 (RESTART-TIMER)))))
	  ;;
	  ;; This is the original code upon which the above is modelled.
	  ;;
	  (#.X11:MotionNotify
	   (let* ((motion-event (make-foreign-pointer 
				  :type '(:pointer X11:XMotionEvent)
				  :address (foreign-pointer-address queued-event)))
		  (x1 (X11:XMotionEvent-x motion-event))
		  (y1 (X11:XMotionEvent-y motion-event)))
	     (if (or (> (abs (- x0 x1)) *click-travel-int*)
		     (> (abs (- y0 y1)) *click-travel-int*))
		 (return-matching-interest)
	       (X11:XNextEvent dsp queued-event))))
	  (t
	   (return-matching-interest))))))))


;;; X11 button events report the button that went up or down as a modifier.  We
;;; clear the corresponding bit in X11 state field because Solo doesn't 
;;; report the action button as a modifier.
;;;
;;; If the event matches an interest we set the events %gesture slot to a list
;;; (<type> <button> <list>).

(defmethod match-event (canvas (event mouse-button-event))
  (let ((x11-event (event-raw-event event)))
    (if x11-event
	(let* ((xvo (device canvas))
	       (type (X11:XButtonEvent-type x11-event))
	       (button (X11:XButtonEvent-button x11-event))
	       (state (logand (X11:XButtonEvent-state x11-event)
			      (case button
				(#.X11:Button1 #.(lognot X11:Button1Mask))
				(#.X11:Button2 #.(lognot X11:Button2Mask))
				(#.X11:Button3 #.(lognot X11:Button3Mask))
				(#.X11:Button4 #.(lognot X11:Button4Mask))
				(#.X11:Button5 #.(lognot X11:Button5Mask))
				(t #.(lognot 0)))))
	       (x11-interests (svref (xview-canvas-interest-table xvo) type)))
	  
	  (macrolet 
	   ((match-p (xvi)
	      `(and (= button (x11-mouse-interest-button ,xvi))
		    (= (logand (x11-mouse-interest-state-mask ,xvi) state)
		       (x11-mouse-interest-state-match ,xvi)))))

	   (let* ((click-interests 
		   (mapcan #'(lambda (xvi)
			       (when (and (> (x11-mouse-interest-nclicks xvi) 0) 
					  (match-p xvi))
				 (list xvi)))
			   x11-interests))
		  (interest 
		   (if click-interests (match-x11-click click-interests x11-event))))
	     (when interest
	       (setf (mouse-event-%gesture event) 
		     (list (cadadr (slot-value interest 'parsed-event-spec)) button state))
	       (return-from match-event interest)))

	   (dolist (xvi x11-interests)
	     (when (and (= 0 (x11-mouse-interest-nclicks xvi)) (match-p xvi))
	       (setf (mouse-event-%gesture event) (list type button state))
	       (return-from match-event (x11-interest-object xvi))))))

      (error "not implemented"))))


(defun handle-xview-button-event (canvas window xv-event x-event)
  (declare (ignore xv-event window))
  (let* ((x-event (X11:XEvent-XButton x-event))
	 (event (make-mouse-button-event 
		  :timestamp (update-x11-current-time (X11:XButtonEvent-time x-event))
		  :x (X11:XButtonEvent-x x-event)
		  :y (X11:XButtonEvent-y x-event)
		  :raw-event x-event))
	 (interest (match-event canvas event)))
    (when interest
      (setf (event-raw-event event) nil)
      (deliver-event canvas interest event))))



;;; MOUSE-MOVED events

(defmethod match-event (canvas (event mouse-moved-event))
  (declare (type-reduce number fixnum)
	   (optimize (safety 1) (speed 3) (compilation-speed 0)))
  (let ((x11-event (event-raw-event event)))
    (if x11-event
	(let* ((xvo (device canvas))
	       (type (X11:XMotionEvent-type x11-event))
	       (state (X11:XMotionEvent-state x11-event)))
	  (dolist (xvi (svref (xview-canvas-interest-table xvo) type))
	    (when (= (logand (x11-mouse-interest-state-mask xvi) state)
		     (x11-mouse-interest-state-match xvi))
	      (setf (mouse-event-%gesture event) state)
	      (return (x11-interest-object xvi)))))
      (error "not implemented"))))
  
(defun handle-xview-motion-event (canvas window xv-event x-event)
  (declare (ignore xv-event window)
	   (optimize (safety 1) (speed 3) (compilation-speed 0)))
  (let* ((x-event (X11:XEvent-XMotion x-event))
	 (event (make-mouse-moved-event 
		  :timestamp (update-x11-current-time (X11:XMotionEvent-time x-event))
		  :x (X11:XMotionEvent-x x-event)
		  :y (X11:XMotionEvent-y x-event)
		  :raw-event x-event))
	 (interest (match-event canvas event)))
    (when interest
      (setf (event-raw-event event) nil)
      (deliver-event canvas interest event)
      (setq *xview-notifier-shouldnt-wait* t))))



;;; MOUSE-CROSSING events

(defmethod match-event (canvas (event mouse-crossing-event))
  (let ((x11-event (event-raw-event event)))
    (if x11-event 
	(when (= (X11:XCrossingEvent-mode x11-event) X11:NotifyNormal)
	  (let* ((xvo (device canvas))
		 (type (X11:XCrossingEvent-type x11-event))
		 (state (X11:XCrossingEvent-state x11-event)))
	      (dolist (xvi (svref (xview-canvas-interest-table xvo) type))
		(when (= (logand (x11-mouse-interest-state-mask xvi) state)
			 (x11-mouse-interest-state-match xvi))
		  (setf (mouse-event-%gesture event) (list type state))
		  (return (x11-interest-object xvi))))))
      (error "not implemented"))))
  
(defun handle-xview-crossing-event (canvas window xv-event x-event)
  (declare (ignore xv-event window))
  (let* ((x-event (X11:XEvent-XCrossing x-event))
	 (event (make-mouse-crossing-event 
		  :timestamp (update-x11-current-time (X11:XCrossingEvent-time x-event))
		  :x (X11:XCrossingEvent-x x-event)
		  :y (X11:XCrossingEvent-y x-event)
		  :raw-event x-event))
	 (interest (match-event canvas event)))
    (when interest
      (setf (event-raw-event event) nil)
      (deliver-event canvas interest event))))



;;; FOCUS events

(defmethod match-event (canvas (event keyboard-focus-event))
  (let* ((table (xview-canvas-interest-table (device canvas)))
	 (x11-interest  (or (car (svref table X11:FocusIn))
			    (car (svref table X11:FocusOut)))))
    (if x11-interest 
	(x11-interest-object x11-interest))))


(defun handle-xview-focus-event (canvas window xv-event x-event)
  (declare (ignore x-event window))
  (with-keyboard-focus-lock 
    (let* ((action (if (= (XV:event-action xv-event) XV:action-take-focus)
		       :take
		     (if (= (XV:event-ie-code xv-event) XV:kbd-use)
			 :in
		       :out)))
	   (event (make-keyboard-focus-event
		   :timestamp *x11-current-time*
		   :focus action))
	   (interest (match-event canvas event)))
      (setf (xview-display-keyboard-focus (xview-object-xvd (device canvas)))
	    (if (eq action :in)
		canvas
	      nil))
      (when interest 
	(deliver-event canvas interest event))

      (unless (eq action :take)
	(let ((virtual-focus-canvas (virtual-keyboard-focus canvas)))
	  (when  (and virtual-focus-canvas (not (eq canvas virtual-focus-canvas)))
	    (let* ((event (let ((e (copy-keyboard-focus-event event)))
			    (setf (keyboard-focus-event-object e) virtual-focus-canvas
				  (keyboard-focus-event-virtual e) canvas)
			    e))
		   (interest (match-event virtual-focus-canvas event)))
	      (when interest
		(deliver-event virtual-focus-canvas interest event)))))))))



(defun handle-xview-reparent-event (canvas window xv-event x-event)
  (when (typep canvas 'top-level-window)
    (let* ((x-event (X11:XEvent-xreparent x-event))
	   (xvo (device canvas))
	   (root (XV:xv-get (xview-display-root (xview-object-xvd xvo)) :xv-xid))
	   (new-parent (X11:XReparentEvent-parent x-event)))
      (setf (xview-top-level-window-reparent xvo)
	    (if (= new-parent root) 
		nil
	      new-parent))))

  (XV:notify-next-event-func window (foreign-pointer-address xv-event) 0 0))


;;; CONFIGURE events
;;;
;;; Update the top-level windows bounding-region based on the X ConfigureNotify
;;; event.  
;;; - If our window has been reparented and the windows width and height 
;;; have not changed then we will receive both a real ConfigureNotify and a
;;; "synthetic" one.  We ignore ignore the real one in this case.  If the
;;; windows width or height changes then we will receive a real ConfigureNotify 
;;; event - but according to the ICCCM we will not get a synthetic ConfigureNotify 
;;; in this case.  In this case we ignore the windows x,y coordinates - since
;;; they're specified relative to the new parent.
;;; - If the window has not been reparented then we can just use the ConfigureNotify
;;; coordinates directly.

(defun handle-xview-configure-event (canvas window xv-event x-event)
  (declare (ignore window xv-event))
  (when (typep canvas 'top-level-window)
    (let* ((xvo (device canvas))
	   (old (xview-top-level-window-configuration xvo))
	   (x-event (X11:XEvent-xconfigure x-event))
	   (new-width (X11:XConfigureEvent-width x-event))
	   (new-height (X11:XConfigureEvent-height x-event)))
      (cond 
       ((or (null old)                                          ;; first ConfigureNotify
	    (null (xview-top-level-window-reparent xvo))        ;; haven't been reparented
	    (/= 0 (X11:XConfigureEvent-send-event x-event)))    ;; synthetic ConfigureNotify
	(setf (xview-top-level-window-configuration xvo)
	      (copy-foreign-pointer x-event)))
       ((or (/= new-width (X11:XConfigureEvent-width old))      
	    (/= new-height (X11:XConfigureEvent-height old)))
	(let ((new (setf (xview-top-level-window-configuration xvo)
			 (copy-foreign-pointer old))))
	  (setf (X11:XConfigureEvent-width new) new-width
		(X11:XConfigureEvent-height new) new-height))))

      (let* ((new (xview-top-level-window-configuration xvo))
	     (new-x (X11:XConfigureEvent-x new))
	     (new-y (X11:XConfigureEvent-y new)))
	(when (and (not (eq old new))
		   (or (null old)
		       (/= (X11:XConfigureEvent-x old) new-x)
		       (/= (X11:XConfigureEvent-y old) new-y)
		       (/= (X11:XConfigureEvent-width old) new-width)
		       (/= (X11:XConfigureEvent-height old) new-height)))
	  (let ((bw2 (* 2 (X11:XConfigureEvent-border-width new))))
	    (deliver-event canvas 
			   :bounding-region-notification
			   (make-bounding-region-notification-event
			     :timestamp *x11-current-time*
			     :region (make-region :left new-x
						  :top new-y
						  :width (+ bw2 new-width)
						  :height (+ bw2 new-height))))))))))


;;; VISIBILITY events

(defmethod match-event (canvas (event visibility-change-event))
  (let* ((table (xview-canvas-interest-table (device canvas)))
	 (x11-interest (car (svref table X11:VisibilityNotify))))
    (if x11-interest 
	(x11-interest-object x11-interest))))

(defun handle-xview-visibility-event (canvas window xv-event x-event)
  (declare (ignore window xv-event))
  (let* ((x-event (X11:XEvent-XVisibility x-event))
	 (event (make-visibility-change-event
		  :timestamp *x11-current-time*
		  :visibility (case (X11:XVisibilityEvent-state x-event)
				(#.X11:VisibilityUnobscured :unobscured)
				(#.X11:VisibilityPartiallyObscured :partially-obscured)
				(#.X11:VisibilityFullyObscured :fully-obscured)
				(t nil))))
	 (interest (match-event canvas event)))
    (when interest
      (deliver-event canvas interest event))))
		     


;;; CLIENT MESSAGE events
;;;
;;; XView makes top level windows :globally-active by default, Solo applications
;;; can also specify :globally-active.  According to the ICCCM: :globally-active
;;; top level windows are sent a ClientMessage (with a timestamp) when the window
;;; manager wants to turn over the keyboard focus.  XView handles this ClientMessage
;;; by setting the focus to the first XView panel that wants it  - XView doesn't 
;;; have any support for real keyboard focus management, maybe in 3.0.


(defun handle-xview-client-message-event (canvas window xv-event x-event)
  (let* ((x-event (X11:XEvent-xclient x-event))
	 (dsp (xview-object-dsp (device canvas))))
    (when (and dsp
	       (= (X11:XClientMessageEvent-format x-event) 32)
	       (= (X11:XClientMessageEvent-message-type x-event) 
		  (x11-intern-atom dsp 'wm_protocols)))
      (update-x11-current-time
       (foreign-aref (X11:bsl-union-l (X11:XClientMessageEvent-data x-event)) 1)))
	       
      (when (= (XV:event-action xv-event) XV:action-take-focus)
	(handle-xview-focus-event canvas window xv-event x-event))

      (XV:notify-next-event-func window (foreign-pointer-address xv-event) 0 0)))




;;; PROPERTY NOTIFY events

(defun handle-xview-property-notify-event (canvas window xv-event x-event)
  (declare (ignore canvas))
  (let ((x-event (X11:XEvent-xproperty x-event)))
    (update-x11-current-time (X11:XPropertyEvent-time x-event))

    (when (and (typep canvas 'root-canvas) 
	       (= (X11:XPropertyEvent-state x-event) X11:PropertyNewValue))
      (let* ((dsp (X11:XPropertyEvent-display x-event))
	     (xid (X11:XPropertyEvent-window x-event))
	     (atom (X11:XPropertyEvent-atom x-event))
	     (atom-name-fp (X11:XGetAtomName dsp atom))
	     (string (foreign-string-value atom-name-fp))
	     (match-length #.(length "LispView-IPC-input")))
	(when (and (>= (length string) match-length) 
		   (string-equal string  "LispView-IPC-input" 
				 :end1 match-length
				 :end2 match-length))
	  (deliver-event 
	    canvas
	    :file-manager
	    (make-lispview-ipc-event
	      :timestamp *x11-current-time*
	      :message (x11-get-window-property dsp xid atom :delete-p t :type X11:XA-String))))
	(X11:XFree atom-name-fp))))

  (XV:notify-next-event-func window (foreign-pointer-address xv-event) 0 0))



;;; MAP/UNMAP NOTIFY events
;;;
;;; In the table below (mapped w) means (xview-canvas-mapped (device w))
;;; and (closed w) means (xview-top-level-window-closed (device w)).
;;; The table defines how we handle MapNotify and UnmapNotify events.
;;; Note that the window manager can move a window between the normal and 
;;; iconic state but it never withdraws a window.  In other words a 
;;; canvas' "mapped" slot never has to be updated based on some action 
;;; taken by the window manager.  The base-window "closed" slot does.
;;;
;;; MapNotify   (mapped w) => NIL
;;; This shouldn't happen.  MapNotify events should only be delivered when 
;;; the window changes from Withdrawn or Iconic to Normal.  The window manager
;;; only controls the Iconic to Normal transition which is only possible
;;; if the client hasn't unmapped ("withdrawn") the window.
;;;
;;; MapNotify   (mapped w) => T
;;; If (closed w) is non nil then the window is no longer iconic, send a 
;;; closed-notification event to the window with closed = nil.  If (closed w)
;;; is nil then this is just the notification that the (setf mapped) has
;;; completed.  There is a small chance that just before window was programatically
;;; mapped the user expanded the windows icon  - so this MapNotify is a 
;;; consequence of the user action.
;;;
;;; UnMapNotify (mapped w) => NIL
;;; This event is probably due to the application programtically mapping
;;; the window.  There is a small chance that just before the application
;;; mapped the window, the user iconified the window and the UnmapNotify
;;; is a consequence of the user action.
;;;
;;; UnMapNotify (mapped w) => T  
;;; The window is now definitely iconic, if (closed w) => NIL then send a 
;;; closed-notification event to the window with closed = T. 

(defun handle-xview-mapped-event (canvas window xv-event x-event)
  (declare (ignore xv-event window))
  (flet 
    ((deliver-closed-notification (closed)
       (deliver-event canvas 
		      :closed-notification 
		      (make-closed-notification-event
			:timestamp *x11-current-time*
			:closed closed))))

    (when (typep canvas 'top-level-window)
      (let* ((xvo (device canvas))
	     (mapped (xview-canvas-mapped xvo))
	     (closed (xview-top-level-window-closed xvo)))
	(cond
	 ((and mapped closed (= (X11:XEvent-type x-event) X11:MapNotify))
	  (setf (xview-top-level-window-closed xvo) nil)
	  (deliver-closed-notification nil))
	 ((and mapped (null closed) (= (X11:XEvent-type x-event) X11:UnmapNotify))
	  (setf (xview-top-level-window-closed xvo) t)
	  (deliver-closed-notification t)))))))




;;; HANDLE-XVIEW-EVENT
;;;
;;; All events that occur in XView windows get funneled through the callback function
;;; below, xview-handle-window-event, which is called by the notifier. 

(defparameter xview-event-handlers 
  (let ((v (make-array X11:LastEvent))
	(handlers 
	 '((handle-xview-keyboard-event X11:KeyPress X11:KeyRelease)
	   (handle-xview-button-event X11:ButtonPress X11:ButtonRelease)
	   (handle-xview-crossing-event X11:EnterNotify X11:LeaveNotify)
	   (handle-xview-motion-event X11:MotionNotify)
	   (handle-xview-focus-event X11:FocusIn X11:FocusOut)
	   (handle-xview-damage-event X11:Expose X11:GraphicsExpose)
	   (handle-xview-reparent-event X11:ReparentNotify)
	   (handle-xview-configure-event X11:ConfigureNotify)
	   (handle-xview-visibility-event X11:VisibilityNotify)
	   (handle-xview-client-message-event X11:ClientMessage)
	   (handle-xview-property-notify-event X11:PropertyNotify)
	   (handle-xview-mapped-event X11:MapNotify X11:UnmapNotify))))
    (dolist (handler handlers v)
      (dolist (type (cdr handler))
	(setf (svref v (symbol-value type)) (car handler))))))

(defvar xview-event 
  (make-foreign-pointer :address 0 :type '(:pointer XV:event)))

(defvar x11-event 
  (make-foreign-pointer :address 0 :type '(:pointer X11:XEvent)))


(def-foreign-callable (handle-xview-event (:return-type :fixnum))
                      ((window :signed-32bit) (event :signed-32bit) (arg :signed-32bit) (type :signed-32bit))
  (declare (optimize (safety 1) (speed 3) (compilation-speed 0)))
  (when (/= event 0)
    (setf (foreign-pointer-address xview-event) event)
    (let ((x11-event-addr (XV:event-ie-xevent xview-event)))
      (if (= 0 x11-event-addr)
	  (handle-xview-internal-event xview-event window event arg type)
	(progn
	  (setf (foreign-pointer-address x11-event) x11-event-addr)
	  (let ((type (X11:XEvent-type x11-event)))
	    (when (< type #.X11:LastEvent)
	      (let ((handler (svref xview-event-handlers type))
		    (canvas (xview-id-to-object window)))
		(when (and handler (typep canvas 'canvas))
		  (handler-case
		   (funcall handler canvas window xview-event x11-event)
		   (error (condition)
		     (warn "An error was signalled within the XView notifier:~%~A" condition)))))))))))
  0)



;;; Internal XView events

(defun handle-xview-internal-event (xv-event window event arg type)
  (if (= (XV:event-ie-code xv-event) XV:scrollbar-request)
      (let ((canvas (xview-id-to-object window))
	    (scrollbar (xview-id-to-object arg)))
	(when (and (typep canvas 'canvas) (typep scrollbar 'scrollbar) (not (typep canvas 'root-canvas)))
	  (deliver-event canvas :scroll (make-scroll-event
					  :timestamp *x11-current-time*
					  :scrollbar scrollbar
					  :motion xview-scrollbar-motion
					  :view-start xview-scrollbar-view-start))))
    (XV:notify-next-event-func window event arg type))
  (setq *xview-notifier-shouldnt-wait* t))


;;; MOUSE-EVENT-GESTURE

(defun x11-event-state-to-list (state)
  (macrolet
   ((state-bit (bit keyword)
      `(if (= (logand state ,bit) ,bit) (list ,keyword)))
    (xlate-modifier (name)
      `(or (car (rassoc ,name *modifier-name-synonyms*)) ,name)))

   (case state
     (#.X11:ShiftMask (xlate-modifier :shift))
     (#.X11:ControlMask (xlate-modifier :control))
     (#.X11:Mod1Mask (xlate-modifier :meta))
     (t
       (nconc (state-bit X11:Button1Mask (xlate-modifier :button0))
	      (state-bit X11:Button2Mask (xlate-modifier :button1))
	      (state-bit X11:Button3Mask (xlate-modifier :button2))
	      (state-bit X11:ShiftMask (xlate-modifier :shift))
	      (state-bit X11:ControlMask (xlate-modifier :control))
	      (state-bit X11:Mod1Mask (xlate-modifier :meta)))))))   ;; Sun specific



(defmethod dd-mouse-event-gesture ((p XView) (event mouse-button-event))
  (let* ((gesture (mouse-event-%gesture event))
	 (state (nth 2 gesture))
	 (type (case (nth 0 gesture) 
		 (#.X11:ButtonPress :down)
		 (#.X11:ButtonRelease :up)
		 (t (nth 0 gesture))))
	 (button 
	  (case (nth 1 gesture)
	    (#.X11:Button1 :button0)
	    (#.X11:Button2 :button1)
	    (#.X11:Button3 :button2)
	    (#.X11:Button4 :button3)
	    (#.X11:Button5 :button4))))
    (values 
     (x11-event-state-to-list state)
     (list (or (car (rassoc button *button-name-synonyms*)) button) type))))

(defmethod dd-mouse-event-gesture ((p XView) (event mouse-moved-event))
  (values (x11-event-state-to-list (mouse-event-%gesture event)) :move))

(defmethod dd-mouse-event-gesture ((p XView) (event mouse-crossing-event))
  (let ((gesture (mouse-event-%gesture event)))
    (values (x11-event-state-to-list (cadr gesture)) 
	    (if (= (car gesture) X11:EnterNotify)
		:enter
	      :exit))))


;;; DD-MOUSE-STATE

(macrolet 
 ((make-int-fp ()
   `(make-foreign-pointer :type '(:pointer :signed-32bit) :static t)))

 (let ((root-return (make-int-fp))
       (child-return (make-int-fp))
       (root-x-return (make-int-fp))
       (root-y-return (make-int-fp))
       (win-x-return (make-int-fp))
       (win-y-return (make-int-fp))
       (mask-return (make-int-fp)))

   (defmethod dd-mouse-state ((p XView) display canvas)
     (XV:with-xview-lock 
       (let* ((xvd (device display))
	      (dsp (xview-display-dsp xvd))
	      (xid (xview-object-xid (device (or canvas (root-canvas display))))))
	 (when (and dsp 
		    xid 
		    (/= 0 (X11:XQueryPointer dsp xid 
					     root-return child-return 
					     root-y-return root-x-return 
					     win-x-return win-y-return
					     mask-return)))
	   (values 
	    (foreign-value win-x-return)
	    (foreign-value win-y-return)
	    (let ((mask (foreign-value mask-return)))
	      (macrolet 
	       ((modifier-bit (bit keyword)
		  `(if (= (logand mask ,bit) ,bit) 
		       (list (or (car (rassoc ,keyword *modifier-name-synonyms*)) ,keyword)))))

	      (nconc
	       (modifier-bit #.X11:ShiftMask :shift)
	       (modifier-bit #.X11:ControlMask :control)
	       (modifier-bit #.X11:Mod1Mask :meta)
	       (modifier-bit #.X11:Button1Mask :button0)
	       (modifier-bit #.X11:Button2Mask :button1)
	       (modifier-bit #.X11:Button3Mask :button2)
	       (modifier-bit #.X11:Button4Mask :button3)
	       (modifier-bit #.X11:Button5Mask :button4)))))))))))


(defmethod dd-warp-mouse ((p XView) relative-to x y)
  (XV:with-xview-lock 
    (let* ((xvo (device relative-to))
	   (xid (xview-object-xid xvo))
	   (dsp (xview-object-dsp xvo)))
      (when xid
	(X11:XWarpPointer dsp X11:None xid 0 0 0 0 x y)
	(xview-maybe-XFlush (xview-object-xvd xvo) dsp)))))

    
;;; DD-DISPLAY-KEYBOARD-FOCUS

(defmethod dd-display-keyboard-focus ((p XView) display)
  (xview-display-keyboard-focus (device display)))


;;; The keyboard focus can only be a mapped window or nil (X11:None).  If it's a window
;;; we force it to be mapped here rather on relying on the application to do it.

(defmethod (setf dd-display-keyboard-focus) (value (p XView) display)
  (when (typep value '(or canvas null))
    (XV:with-xview-lock 
      (let* ((xvd (device display))
	     (dsp (xview-display-dsp xvd))
	     (focus (if value
			(let* ((xvo (device value))
			       (id (xview-object-id xvo)))
			  (when id
			    (XV:xv-set id :win-map t)
			    (xview-object-xid xvo)))
		      #.X11:None)))
	(when (and focus dsp)
	  (X11:XSetInputFocus dsp focus #.X11:RevertToParent *x11-current-time*)
	  (xview-maybe-XFlush xvd dsp)))))
  value)



;;; DD-CANVAS-KEYBOARD-FOCUS

(defmethod dd-canvas-keyboard-focus ((p XView) canvas)
  (xview-canvas-to-focus (device canvas)))


;;; Return the canvas and the end of the to-focus chain that starts with root-canvas.

(defvar virtual-keyboard-focus-chain-limit 64)

(defun find-xview-focus (root-canvas level)
  (if (typep root-canvas 'canvas)
      (let ((focus (xview-canvas-to-focus (device root-canvas))))
	(cond
	 ((> level virtual-keyboard-focus-chain-limit)
	  (warn "More than ~D levels of keyboard focus indirection in ~S" level root-canvas)
	  root-canvas)
	 ((eq focus root-canvas)
	  root-canvas)
	 (t
	  (find-xview-focus focus (1+ level)))))))


(defmethod (setf dd-canvas-keyboard-focus) (to-canvas (p XView) from-canvas)
  (with-keyboard-focus-lock
    (let* ((from-xvo (device from-canvas))
	   (display-focus (xview-display-keyboard-focus (xview-object-xvd from-xvo)))
	   (old-virtual-focus (find-xview-focus display-focus 0))
	   (new-virtual-focus 
	    (progn
	      (setf (xview-canvas-to-focus from-xvo) to-canvas)
	      (find-xview-focus display-focus 0))))

      (unless (eq new-virtual-focus old-virtual-focus)
	(flet 
	 ((send-focus-event (object in/out)
	    (send-event object (make-keyboard-focus-event 
				 :object object
				 :timestamp *x11-current-time*
				 :focus in/out
				 :virtual display-focus))))
	 (when old-virtual-focus 
	   (send-focus-event old-virtual-focus :out))
	 (when new-virtual-focus 
	   (send-focus-event new-virtual-focus :in))))))
  to-canvas)

;;; Aaron's fix to avoid deadlock involving xv::*xview-lock* and
;;; *keyboard-focus-lock*
(defmethod (setf dd-canvas-keyboard-focus) :AROUND (to-canvas (p XView) from-canvas)
  (XV:with-xview-lock (call-next-method)))


;;; DD-VIRTUAL-KEYBOARD-FOCUS

(defmethod dd-canvas-virtual-keyboard-focus ((p XView) canvas)
  (find-xview-focus canvas 0))



;;; DD-VIRTUAL-KEYBOARD-FOCUS

(defmethod dd-display-virtual-keyboard-focus ((p XView) display)
  (let ((x11-focus (xview-display-keyboard-focus (device display))))
    (if x11-focus
	(find-xview-focus x11-focus 0))))


(proclaim '(inline x11-format-to-foreign-type))

(defun x11-format-to-foreign-type (format type)
  (case format
    (8   (case type
	   (#.X11:XA-string :character)
	   (#.X11:XA-cardinal :unsigned-8bit)
	   (t :signed-8bit)))
    (16  (case type 
	   (#.X11:XA-cardinal :unsigned-16bit)
	   (t :signed-16bit)))
    (t   (case type 
	   (#.X11:XA-cardinal :unsigned-32bit)
	   (t :signed-32bit)))))


;;; Return a foreign-pointer to an array nitems long of elements consistent
;;; with the specified actual-format and actual-type.  The array will contain
;;; the property value whose address is the the (foreign) value of prop-return.
;;;
;;; The value of a property returned by X11:XGetWindowProperty is an array
;;; of 8, 16, or 32 bit element.  The size of the array elements are specified
;;; by actual-format.  The client that set the property can also specify the 
;;; properties "actual type" with an X11 atom; we handle the predefined property
;;; types XA_string and XA_cardinal specially here, all others are treated as (signed)
;;; integers of the size specified by actual-format.

(defun x11-window-property-value (actual-format actual-type nitems prop-return)
  (let ((type (x11-format-to-foreign-type actual-format actual-type)))
    (if (> nitems 0)
	(make-foreign-pointer
	  :address (foreign-pointer-address (foreign-value prop-return))
	  :type `(:pointer (:array ,type (,nitems))))
      (make-foreign-pointer 
        :address 0
	:type `(:pointer ,type)))))


;;; Utility interface for XGetWindowProperty.  Returns a string if the properties
;;; type is X11:XA-string, otherwise returns a vector of integers.

(macrolet 
 ((make-fp (type)
   `(make-foreign-pointer :type '(:pointer ,type) :static t)))

 (let ((actual-type-return (make-fp X11:Atom))
       (actual-format-return (make-fp X11:int))
       (nitems-return (make-fp X11:unsigned-long))
       (bytes-after-return (make-fp X11:unsigned-long))
       (prop-return (make-fp (:pointer X11:char))))

   (defun X11-GET-WINDOW-PROPERTY (dsp xid atom 
				   &key 
				     (type X11:AnyPropertyType)
				     (delete-p nil)
				     (buffer-size 4096))
     (flet 
      ((x11-get (offset)
	 (let ((err
		(X11:XGetWindowProperty dsp xid atom offset buffer-size
					(if delete-p 1 0) type 
					actual-type-return actual-format-return
					nitems-return bytes-after-return prop-return)))
	   (if (= err X11:Success)
	       (let ((atom (foreign-value actual-type-return)))
		 (if (= atom X11:None) nil atom))
	     (return-from x11-get-window-property (values nil type nil))))))
      
      (XV:with-xview-lock 
       (let* ((offset 0)
	      (actual-type (x11-get 0))
	      (actual-format (foreign-value actual-format-return))
	      (value nil))

	 (when (and (null actual-type) (= 0 actual-format))  ;; no such property
	   (return-from x11-get-window-property (values nil nil T)))

	 (loop
	  (let* ((nitems (foreign-value nitems-return))
		 (prop (x11-window-property-value actual-format actual-type nitems prop-return)))
	    (push (if (= actual-type X11:XA-string)
		      (or (foreign-string-value prop) "")
		    (let ((vector (make-array (list nitems))))
		      (dotimes (i nitems vector)
			(setf (svref vector i) (foreign-aref prop i)))))
		  value)
	    (X11:XFree (foreign-value prop-return))
	    (when (= 0 (foreign-value bytes-after-return))
	      (return))
	    (x11-get (incf offset (truncate (* nitems actual-format) 32)))))

	 (values (apply #'concatenate (if (= actual-type X11:XA-string) 'simple-string 'simple-vector)
			(nreverse value))
		 actual-type
		 t)))))))

(defun x11-change-window-property (dsp xid property data type format 
				   &key 
				     (mode X11:PropModeReplace)
				     (transform #'identity)
				     (start 0)
				     (end (length data)))
  (let ((nitems (- end start)))
    (when (> nitems 0)
      (let* ((data-type (x11-format-to-foreign-type format type))
	     (data-fp (malloc-foreign-pointer 
		        :type `(:pointer (:array ,data-type (,nitems))))))
	(dotimes (i nitems)
	  (setf (foreign-aref data-fp i) (funcall transform (elt data (+ i start)))))
	(X11:XChangeProperty dsp 
			     xid
			     property
			     type
			     format
			     mode
			     (make-foreign-pointer 
			       :type '(:pointer X11:char)
			       :address (foreign-pointer-address data-fp))
			     nitems)
	(free-foreign-pointer data-fp)))))

	     
;;; Set the window manager hints and the WM_PROTOCOLS property as specified
;;; by the ICCCM "Input Focus" section.  Focus mode must be one of :passive, 
;;; :locally-active, :globally-active, null.

(defun  x11-set-keyboard-focus-mode (dsp xid mode)

  ;; If the focus mode (or "model") is :globally-active or :locally-active
  ;; then add the WM_TAKE_FOCUS X atom to the windows WM_PROTOCOLS property
  ;; (if it's not there already).  If the focus mode is :passive or null
  ;; then remove the WM_TAKE_FOCUS X atom.

  (let ((wm-protocols-atom (x11-intern-atom dsp 'wm-protocols))
	(wm-take-focus-atom (x11-intern-atom dsp 'wm-take-focus)))
    (multiple-value-bind (wm-protocols type success)
	(x11-get-window-property dsp xid wm-protocols-atom :type X11:XA-atom)
      (declare (ignore type))
      (if (null success) 
	  (warn "XGetProperty ~S ~S WM_PROTOCOLS failed" dsp xid)
	(macrolet 
	 ((change-wm-protocols (new)
	   `(x11-change-window-property dsp xid wm-protocols-atom ,new X11:XA-atom 32)))

	 (if (or (eq mode :globally-active) (eq mode :locally-active))
	     (unless (find wm-take-focus-atom wm-protocols :test #'=)
	       (change-wm-protocols (cons wm-take-focus-atom wm-protocols)))
	   (when (find wm-take-focus-atom wm-protocols :test #'=)
	     (change-wm-protocols (delete wm-take-focus-atom wm-protocols :test #'=))))))))

  ;; If the focus mode is :globally active or nil then set the window manager
  ;; input hint to 0 (False) otherwise 1.

  (let ((wmhints (let ((wmhints (X11:XGetWMHints dsp xid)))
		   (if (/= 0 (foreign-pointer-address wmhints))
		       wmhints
		     (X11:XAllocWMHints)))))
    (setf (X11:XWMHints-flags wmhints) (logior X11:InputHint (X11:XWMHints-flags wmhints))
	  (X11:XWMHints-input wmhints) (if (or (eq mode :globally-active) (null mode)) 0 1))
    (X11:XSetWMHints dsp xid wmhints)
    (setf (foreign-pointer-type wmhints) '(:pointer X11:char))
    (X11:XFree wmhints)))



(defmethod dd-keyboard-focus-mode ((p XView) canvas)
  (xview-top-level-window-keyboard-focus-mode (device canvas)))

(defmethod (setf dd-keyboard-focus-mode) (value (p XView) canvas)
  (XV:with-xview-lock 
    (let* ((xvo (device canvas))
	   (xid (xview-object-xid xvo))
	   (dsp (xview-object-dsp xvo)))
      (when xid
	(x11-set-keyboard-focus-mode dsp xid value)
	(xview-maybe-XFlush (xview-object-xvd xvo) dsp))
      (setf (xview-top-level-window-keyboard-focus-mode xvo) value)))
  value)


;;; Print methods for X11-interest and subtypes of X11-interest

(defmethod x11-input-mask-string (mask)
  (let ((x11-input-masks
	 '(X11:NoEventMask 
	   X11:KeyPressMask 
	   X11:KeyReleaseMask 
	   X11:ButtonPressMask 
	   X11:ButtonReleaseMask 
	   X11:EnterWindowMask 
	   X11:LeaveWindowMask 
	   X11:PointerMotionMask 
	   X11:PointerMotionHintMask
	   X11:Button1MotionMask 
	   X11:Button2MotionMask 
	   X11:Button3MotionMask 
	   X11:Button4MotionMask 
	   X11:Button5MotionMask 
	   X11:ButtonMotionMask 
	   X11:KeymapStateMask 
	   X11:ExposureMask 
	   X11:VisibilityChangeMask 
	   X11:StructureNotifyMask 
	   X11:ResizeRedirectMask 
	   X11:SubstructureNotifyMask 
	   X11:SubstructureRedirectMask
	   X11:FocusChangeMask 
	   X11:PropertyChangeMask
	   X11:ColormapChangeMask
	   X11:OwnerGrabButtonMask)))
    (with-output-to-string (stream)
      (if (= 0 (logcount mask))
	  (format stream "()")
	(dolist (bit (cdr x11-input-masks))
	  (let ((name (symbol-name bit))
		(value (symbol-value bit)))
	    (when (= value (logand value mask))
	      (format stream "~A " (subseq name 0 (- (length name) 4))))))))))


(defun x11-event-types-string (types)
  (flet 
   ((find-x11-event-type (type)
       (symbol-name (find type x11-event-types :test #'= :key #'symbol-value))))

   (format nil "~A" (or (mapcar #'find-x11-event-type types) "()"))))


(defun x11-state-target-string (mask match)
  (let ((bitmasks
	  (macrolet ((xlate-modifier (keyword)
		       `(or (car (rassoc ,keyword *modifier-name-synonyms*)) ,keyword)))
	    (list (cons X11:Button1Mask (xlate-modifier :button0))
		  (cons X11:Button2Mask (xlate-modifier :button1))
		  (cons X11:Button3Mask (xlate-modifier :button2))
		  (cons X11:ShiftMask (xlate-modifier :shift))
		  (cons X11:ControlMask (xlate-modifier :control))
		  (cons X11:Mod1Mask (xlate-modifier :meta))))))   ;; Sun specific
    (with-output-to-string (stream)
      (flet 
       ((print-bits (n)
	  (if (= n 0)
	      (format stream "()")
	    (dolist (x bitmasks)
	      (when (= (logand (car x) n) (car x))
		(format stream " ~A" (cdr x)))))))

	 (format stream "(:down ")
	 (print-bits match)
	 (format stream " :up ")
	 (print-bits (logand (lognot match) mask))
	 (format stream ")")))))


(defun print-x11-mouse-interest (interest stream level)
  (declare (ignore level))
  (let ((types (x11-mouse-interest-types interest)))
    (format stream "#<X11-mouse-interest ~A :input-mask ~A " 
	    (x11-interest-object interest)
	    (x11-input-mask-string (x11-mouse-interest-input-mask interest)))

    (when (intersection (list X11:ButtonPress X11:ButtonRelease) types :test #'=)
      (let ((x11-button-names 
	     (macrolet ((xlate-button-name (keyword)
			  `(or (car (rassoc ,keyword *button-name-synonyms*)) ,keyword)))
	       (list (cons X11:Button1 (xlate-button-name :button0))
		     (cons X11:Button2 (xlate-button-name :button1))
		     (cons X11:Button3 (xlate-button-name :button2))
		     (cons X11:Button4 (xlate-button-name :button3))
		     (cons X11:Button5 (xlate-button-name :button4))))))
	(format stream ":button ~S " (cdr (assoc (x11-mouse-interest-button interest) x11-button-names :test #'=)))))

    (format stream ":types ~A :modifiers ~A ~A ~X>"
	    (x11-event-types-string types)
	    (x11-state-target-string (x11-mouse-interest-state-mask interest)  
				     (x11-mouse-interest-state-match interest))
	    (if (/= (x11-mouse-interest-nclicks interest) 0)
		(format nil "n-clicks ~D" (x11-mouse-interest-nclicks interest))
	      "")
	    (SYSTEM:%pointer interest))))


(defun print-x11-keyboard-interest (interest stream level)
  (declare (ignore level))
  (format stream "#<X11-keyboard-interest ~A ~A ~X>"
	  (x11-interest-object interest)
	  (let ((types (x11-keyboard-interest-types interest)))
	    (cond
	     ((equal types '#.(list X11:KeyRelease)) "up")
	     ((equal types '#.(list X11:KeyPress)) "down")
	     (t "up/down")))
	  (SYSTEM:%pointer interest)))


(defun print-x11-damage-interest (interest stream level)
  (declare (ignore level))
  (format stream "#<X11-damage-interest ~A ~X>" 
	  (x11-interest-object interest)
	  (SYSTEM:%pointer interest)))


(defun print-x11-interest (interest stream level)
  (declare (ignore level))
  (format stream "#<X11 ~Ainterest ~A ~X>" 
	  (x11-input-mask-string (x11-interest-input-mask interest))
	  (x11-interest-object interest)
	  (SYSTEM:%pointer interest)))

