;;; -*- Mode:Lisp; Package:grapher; BASE:10; LOWERCASE:T; Syntax:Common-Lisp -*-
;;;
;;;	------ CLUE Grapher ------ 1990
;;;
;;;	by Tomoru Teruuchi (Fuji Xerox)
;;;	written as a part of a project under the Jeida Common Lisp Committee 1990
;;;
;;;	The source code is completely in public domain.
;;;
;;;	The author(s), the organization the author belongs, Jeida,
;;;	Jeida CLC, Masayuki Ida (the chairman), and Aoyama Gakuin University
;;;	have no warranty about the contents, the results caused by the use,
;;;	and any other possible situations.
;;;
;;;	This codes is written as an experimentation.
;;;	This codes do not mean the best functinality the system supports.

;;;
;;; file name : clue-grapher.lisp
;;;
(in-package 'grapher :use '(lisp cluei pcl xlib))

;;; Some Contant Data for Grapher

(defparameter *grapher-menu-title* "Grapher Commands")
(defparameter *set-popup-title* "Set Commands Submenu")

(defparameter *menu-callback* 
  '(("Add Arc" . add-an-arc)
    ("Create" . create-a-node)
    ("Delete Arc" . delete-an-arc)
    ("Delete Node" . delete-a-node)
    ("Move" . move-a-node)
    ("Reset" . clear-all-data)
    ("Set..." . set-popup)
    ("Store" . store-data)
    ("Load" . load-data)
    ("Quit" . quit-selected))
    "This is the association list of bottun label and callback
     functions")

(defparameter *set-popup*
  '(("Set Node Label" . set-label)
    ("Set Arc Mode" . set-arc-mode)
    ("Set Arc Label" . set-arc-label)))

(defvar *gcanvas* nil)
(defvar *my-menu* nil)
(defvar *set-menu* nil)
(defvar *i-stream* nil)
(defvar *current-mode* :other)
(defvar *message-out* nil)
(defvar *arc-label-mode* nil)

;;; definitions of Grapher's Contact Stuff
;;;
;;;

;;; definition of Grapher's Canvas
;;;
(defcontact canvas (composite)
  ((font
    :type font
    :initarg :font
    :initform 'fixed)
   (snode
    :type node
    :initarg :snode
    :initform nil
    :accessor canvas-snode)
   (cursor
    :initarg :cursor)
   (foreground
    :type pixel
    :accessor canvas-foreground
    :initarg :foreground
    :initform :black)
   (background
    :initarg :background
    :initform :white)
   (border-width
    :initarg :border-width
    :initform 1))
  (:resources
   font
   snode
   cursor
   foreground
   background
   border-width
   (event-mask :initform #.(make-event-mask :exposure)))
  (:documentation
   "This is Grapher's Vision Window"))

;;; definition of Grapher's wrapper (this children may be actual Windows)
;;;
(defcontact grapher-system (composite)
  ()
  (:documentation
   "This is Grapher's Wrapper contact"))

;;; 
;;; resize for wrapper
;;;

(defmethod resize :after ((top-shell top-level-shell) width height border-width)
  (resize (car (composite-children top-shell)) width height border-width)
)
 
(defmethod resize :after ((wrapper grapher-system) width height border-width)
  (dolist (child (composite-children wrapper))
	  (if (typep child 'fixed-menu)
	      (multiple-value-bind (m-width m-height m-border-width)
				   (preferred-size child)
				   (declare (ignore m-border-width))
				   (move child
					 (- width m-width (* 2 border-width))
					 (- height m-height
					    (* 2 border-width)))
				   (dolist (other-child 
					    (composite-children wrapper))
					   (if (typep other-child 'canvas)
					       (resize other-child
						       width 
						       (- height
							  m-height
							  (* 2 border-width))
						       (contact-border-width other-child))
					     )
					   (if (typep other-child 'interactive-stream)
					       (progn
						 (resize other-child
							 (- width
							    m-width
							    (* 2 border-width)
							    (* 2 (contact-border-width other-child)))
							 m-height
							 (contact-border-width other-child))
						 (move other-child
						       0
						       (- height
							  m-height
							  (* 2 border-width))))))
				   )
	    )
	  )
)


;;;
;;; Grapher system has other contact (fixed menu and interactive-stream).
;;; these contacts are defined in other files.
;;;

(defmethod display ((graph-canvas canvas) &optional x y width height &key)
  (declare (ignore x y width height))
  (map-over-nodes (a-node) 
		  (present-self a-node graph-canvas))
  (map-over-arcs (an-arc)
		 (present-self an-arc graph-canvas))
)

;;; Graphic Utilities
;;;    These functions depend on the Window Toolkit.
;;;    In this file, functions for CLUE are described.
;;;
(defun draw-circle (window x y radius &optional (erase-p nil) (gc nil) (fill-p nil))
  "drawing a circle using clx's xlib:draw-arc function"
  (let* ((left-up-x (- x radius))
	 (left-up-y (- y radius))
	 (width (+ radius radius))
	 (height width))
    (if (null gc)
	(using-gcontext (my-gc :drawable window
			       :foreground (if erase-p 0 1))
			(xlib:draw-arc window my-gc left-up-x left-up-y 
				       width height 0 (* 2 pi) fill-p))
      (xlib:draw-arc window gc left-up-x left-up-y 
		     width height 0 (* 2 pi) fill-p)))

)


(defun draw-string (window string x y &optional (erase-p nil) (gc nil))
  "drawing the indicated string contents (using xlib:draw-glyphs)"
  (if (null gc)
      (using-gcontext (my-gc :drawable window
			     :foreground (if erase-p 0 1)
			     :font 'fixed)
		      (xlib:draw-glyphs window my-gc x y string))
    (xlib:draw-glyphs window gc x y string))
)

(defun draw-box (window x y width height &optional (erase-p nil) (gc nil))
  "drawing a rectangle what is squere.(using xlib:draw-rectangle)"
  (let* ((left-up-x (- x (ceiling width 2)))
	 (left-up-y (- y (ceiling height 2))))
    (if (null gc)
	(using-gcontext (my-gc :drawable window
			   :foreground (if erase-p 0 1)
			   :font 'fixed)
		    (xlib:draw-rectangle window my-gc left-up-x left-up-y
					 width height ))
      (xlib:draw-rectangle window gc left-up-x left-up-y width height)))
)

;;;This function is modified for extensions.
;;;draw-my-arc may print arc's label (or arc unique-id) if *arc-label-mode* is t.
;;;
(defun draw-my-arc (contact x1 y1 x2 y2 arc &optional (erase-p nil) (gc nil))   
  "draw line between Node1 and Node2 (using xlib:draw-line)"
  (if (equal *arc-label-mode*  t)
      (if (null gc)
	  (using-gcontext (my-gc :drawable contact :foreground (if erase-p 0 1))
			  (multiple-value-bind (width height) 
					       (string-width *gcanvas* (if (arc-label arc)
									   (arc-label arc)
									 (arc-unique arc)) gc)
					       (setq width (+ width 8))(setq height (+ height 4))
					       (setf (arc-width arc) width)   ;added for extension
					       (setf (arc-height arc) height) ;added for extension
			   (multiple-value-bind (midx1 midy1 midx2 midy2)
						(find-edges-of-arcs width height x1 y1 x2 y2)
						(xlib:draw-line contact my-gc x1 y1 midx1 midy1)
						(draw-box  contact (+ x1 (ceiling (- x2 x1) 2))
							   (+ y1 (ceiling (- y2 y1) 2)) width height erase-p my-gc)
						(draw-my-label contact (+ x1 (ceiling (- x2 x1) 2))
							    (+ y1 (ceiling (- y2 y1) 2)) 
							    (if (arc-label arc) (arc-label arc)
							      (arc-unique arc)) erase-p my-gc)
						(xlib:draw-line contact my-gc midx2 midy2 x2 y2))))
	(multiple-value-bind (width height) 
			     (string-width *gcanvas* (if (arc-label arc) (arc-label arc)
						       (arc-unique arc)) gc)
			     (setq width (+ width 8))(setq height (+ height 4))
			     (setf (arc-width arc) width)   ;added for extension
			     (setf (arc-height arc) height) ;added for extension
			   (multiple-value-bind (midx1 midy1 midx2 midy2)
						(find-edges-of-arcs width height x1 y1 x2 y2)
						(xlib:draw-line contact gc x1 y1 midx1 midy1)
						(draw-box  contact (+ x1 (ceiling (- x2 x1) 2))
							   (+ y1 (ceiling (- y2 y1) 2)) width height erase-p gc)
						(draw-my-label contact (+ x1 (ceiling (- x2 x1) 2))
							    (+ y1 (ceiling (- y2 y1) 2)) 
							    (if (arc-label arc) (arc-label arc)
							      (arc-unique arc)) erase-p gc)
						(xlib:draw-line contact gc midx2 midy2 x2 y2))))
    (if (null gc)
	(using-gcontext (my-gc :drawable contact
			       :foreground (if erase-p 0 1))
			(xlib:draw-line contact my-gc x1 y1 x2 y2)
			)
      (xlib:draw-line contact gc x1 y1 x2 y2)))
)

(defun string-width (window string &optional (gc nil))
  "returns the width (pixels) of string (using text-extents)"
  (declare (ignore window))
  (if (null gc)
      (setq gc (create-gcontext :drawable window
				:font 'fixed)))
  (multiple-value-bind (width ascent descent)
		       (text-extents gc string)
		       (values width (+ ascent descent)))
)

(defun my-cursor (name display)
;;;  (let* ((number (cadr (assoc name cluei::*cursor-names* 
;;;			      :test #'equal))) for ver 7.1
  (let* ((number (if (find-package 'common-lisp) name
		   (cadr (assoc name cluei::*cursor-names*
				:test #'equal))))
	 cursor font)
    (setq font (open-font display "cursor"))
    (setq cursor
	  (create-glyph-cursor
	   :source-font font
	   :source-char number
	   :mask-font font
	   :mask-char (1+ number)
	   :foreground (make-color :red 0.0 :green 0.0 :blue 0.0)
	   :background (make-color :red 1.0 :green 1.0 :blue 1.0)))
    cursor)
)
  
;;;
;;; Make My Fixed Menu
;;;
(defun make-my-fmenu (&key parent)
  (let* ((fmenu (make-contact 'fixed-menu
			      :parent parent
			      :text *grapher-menu-title*
			      :background :white))
	 (display (contact-display parent))
	current-button callback-fn )
    (dolist (label *menu-callback*)
	    (setq current-button
		  (make-contact 'button
				:parent (menu-manager fmenu)
				:cursor (my-cursor (if (find-package 'common-lisp) hand2-cursor
						     "hand2") ;; solve the gap ver 7.1 and 7.2
						   display)
				:label (car label)))
	    (if (setq callback-fn
		      (cdr label))
		(if (listp callback-fn)
		    (dolist (cfunc callback-fn)
			    (add-callback current-button  :select  cfunc))
		  (if (equal "Set..." (car label))
		      (add-callback current-button :select
				callback-fn current-button)
		    (add-callback current-button :select callback-fn))))
	    )
    (multiple-value-bind (approve-p approve-x approve-y
				    approve-width approve-height
				    approve-border-width)
			 (fmenu-arrange fmenu 
					(contact-width parent)
					(contact-height parent))
			 (declare (ignore approve-p approve-x approve-y))
			 (values (+ approve-width
				    (* 2 approve-border-width))
				 (+ approve-height
				    (* 2 approve-border-width)))
			 )
    )
)

;;;
;;; make popup-menu
;;;
(defun make-my-popup-menu (&key parent title callback-list)
  (let* ((menu (make-contact 'menu :parent parent
			     :title title))
	 (menu-mgr (menu-manager menu))
	 callback-fn current-button)
    (dolist (label callback-list)
	    (setq current-button
		  (make-contact 'button
				:parent menu-mgr
				:label (car label)))
	    (if (setq callback-fn
		      (cdr label))
		(if (listp callback-fn)
		    (dolist (cfunc callback-fn)
			    (add-callback current-button :select  cfunc))
		  (if (equal "Set..." (car label))
		      (add-callback current-button :select
				callback-fn current-button)
		    (add-callback current-button :select callback-fn)))))
    menu)
)

;;;
;;; Pop-up My Menu
;;;

(defmethod popup-my-menu ((gcanvas canvas))
    (unwind-protect 
	(multiple-value-bind (menu-x menu-y)
			     (global-pointer-position 
			      (contact-display gcanvas))
			     (menu-choose *my-menu* menu-x menu-y)))
)

(defmethod popup-my-menu ((button button))
    (unwind-protect 
	(multiple-value-bind (menu-x menu-y)
			     (global-pointer-position 
			      (contact-display button))
			     (menu-choose *set-menu* menu-x menu-y)))
)

;;;
;;; Interactive Intaface
;;;
(defun prompt (stream)
  (stream-write-string stream
		       "[Grapher Command] ")
)

(defun query-data-msg (stream type)
  (prompt *i-stream*)
  (ecase type
	 (:create-node
	  (stream-write-string stream "Create New Node <x> : "))
	 (:add-arc
	  (stream-write-string stream "Add Arc <Node1> : "))
	 (:delete-arc
	  (stream-write-string stream "Delete Arc <Node1> : "))
	 (:move-node
	  (stream-write-string stream "Move Node <Node> : "))
	 (:delete-node
	  (stream-write-string stream "Delete Node <Node> : "))
	 (:reset )
	 (:set-label
	  (stream-write-string stream "Set Node Label <Node> : "))
	 (:arc-label
	  (stream-write-string stream "Set Arc Label <Node1> : "))
	 (:store-data )
	 (:load-data ))
)
	  
(defun query-data (stream type)
  (let* ((delim '(#\newline #\space))
	 (first nil)
	 (second nil)
	 (third nil))
    (query-data-msg stream type)
    (ecase type
	   (:create-node
	    (setq first (string-to-integer (echoback stream delim)))
	    (stream-write-string stream " <y> : ")
	    (setq second (string-to-integer 
			  (echoback stream delim))))
	   (:add-arc
	    (setq first (echoback stream delim))
	    (stream-write-string stream " <Node2> : ")
	    (setq second (echoback stream delim)))
	   (:delete-arc
	    (setq first (echoback stream delim))
	    (stream-write-string stream
				 " <Node2> : ")
	    (setq second (echoback stream delim)))
	   (:move-node
	    (setq first (echoback stream delim))
	    (stream-write-string stream
				 "New Position <x> : ")
	    (setq second (string-to-integer 
			  (echoback stream delim)))
	    (stream-write-string stream
				 " <y> : ")
    	    (setq  third (string-to-integer 
			  (echoback stream delim))))
	   (:delete-node
	    (setq first (echoback stream delim)))	    
	   (:reset
	     )
	   (:set-label
	    (setq first (catch 'node-name (echoback stream delim)))
	    (stream-write-string stream
				 " <New Label> : ")
	    (setq second (echoback stream delim)))
	   (:arc-label                                              ;;added for extension
	    (setq first (catch 'arc-name (echoback stream delim)))  ;;added for extension
	    (if (listp first)                                       ;;added for extension
		(progn                                              ;;added for extension
		  (setq second (node-name (cadr first)))            ;;added for extension
		  (setq first (node-name (car first)))              ;;added for extension
		  (stream-write-strings stream                      ;;added for extension
					(first " <Node2> : " second)))
	      (progn                                                ;;added for extension
		(stream-write-string stream " <Node2> : ")          ;;added for extension 
		(setq second (echoback stream delim))))             ;;added for extension
	    (stream-write-string stream " <New Label> : ")          ;;added for extension
	    (setq third (echoback stream delim))                    ;;added for extension
	    )                                                       ;;added for extension
	   (:store-data
	    (stream-write-string stream
				 "Input file name(newfile) : ")
	    (setq first (echoback stream delim)))
	   (:load-data
	    (stream-write-string stream
				 "Input file name(loadfile) : ")
	    (setq first (echoback stream delim)))
	   )
    (if (not (equal (cluei::stream-cursor-x stream) 0))
	(stream-fresh-line stream))
    (values first (if (null third) second
			    (list second third)))
    )
)

(defun echoback (stream &optional (delimiter '(#\newline)))
  (block nil
  (let ((string-p nil) char
	  (sbuf (make-array 255 :fill-pointer 0 :element-type 'string-char)))
    (if (null (listp delimiter))
	(setq delimiter (list delimiter)))
    (setq char (stream-read-char stream))
    (if (equal char #\") 
	(progn (delete #\space delimiter)
	       (setq string-p t)
	       (stream-write-char stream char))
      (progn
	(pushnew #\space delimiter)
	(if (null (or (equal char #\backspace) (equal char #\rubout)))
	    (progn 
	      (if (equal char #\return)
		  (setq char #\newline))
	      (vector-push-extend char sbuf)
	      (stream-write-char stream char))
	  (setq char #\backspace))))
    (if (member char delimiter)
	(return sbuf))
    (loop
     (setq char (stream-read-char stream))
     ;; Kludge for Sun Common Lisp(Lucid)
     (if (equal char #\return)
	 (setq char #\newline)
       (if (equal char #\rubout)
	   (setq char #\backspace)))
     (if (equal char #\backspace)
	 (if (> (length sbuf) 0)
	     (stream-write-char stream char))
       (stream-write-char stream char))
     (if (member char delimiter)
	 (if string-p (return (progn (vector-pop sbuf) sbuf))
	   (return sbuf))
       (if (eq char #\backspace)
	   (if (> (length sbuf) 0)
	       (vector-pop sbuf))
	 (vector-push-extend char sbuf)))))
))


(defun string-to-integer (string)
  (let* ((len (length string))
	 (retval 0))
    (dotimes (counter  len)
	    (setq retval (+ (* retval 10) 
			    (- (char-int (aref string counter))
			       48))))
    retval))
;;;
;;; Interface Macros used by callback functions
;;;

(defmacro new-node-pos ()
  `(query-data *i-stream* :create-node)
)

(defmacro delete-node-name ()
  `(query-data *i-stream* :delete-node)
)

(defmacro move-node-name ()
  `(query-data *i-stream* :move-node)
)

(defmacro new-label-name ()
  `(query-data *i-stream* :set-label)
)

(defmacro add-arc-nodes ()
  `(query-data *i-stream* :add-arc)
)

(defmacro delete-arc-nodes ()
  `(query-data *i-stream* :delete-arc)
)

(defmacro arcs-label-set ()            ;;added for extension
  `(query-data *i-stream* :arc-label)  ;;added for extension
)                                      ;;added for extension

;;;
;;; callback functions
;;;

(defmethod create-a-node-after ((node node) &key)
  (present-self node *gcanvas*)
  ;;print System Message
  (stream-write-string *i-stream* ";;;Created ")
  (print-object--stream node *i-stream*)
)


(defmethod create-a-node ()
  (catch :create-node
    (let (a-node)
      (setq *message-out* t)
      (multiple-value-bind (x y)
			   (new-node-pos)
			   (setq a-node (make-instance 'node))
			   (setf (node-xpos a-node) x)
			   (setf (node-ypos a-node) y))
      (create-a-node-after a-node)
      (setq *message-out* nil)
      )
    )
)

(defmethod delete-a-node-after ((node node) &key)
  (present-self node *gcanvas* t)
  (dolist (arc (node-arcs node))
	  (present-self arc *gcanvas* t)
	  (delete-self arc))
  (delete-self node)
  ;;print System Message
  (stream-write-string *i-stream* ";;;Deleted ")
  (print-object--stream node *i-stream*)
)

(defmethod delete-a-node ()
  (catch :delete-node
;    (let ((a-node (find-node (delete-node-name))))
    (let (a-node)
      (setq *message-out* t)
      (setq a-node (find-node (delete-node-name)))
      (if a-node
	  (delete-a-node-after a-node)
	)
      (setq *message-out* nil)
      )
    )
)

    
(defmethod move-a-node ()
  (catch :move-node
    (progn
      (setq *message-out* t)
      (multiple-value-bind (a-node  pos)
			   (move-node-name)
			   (if (setq a-node (find-node a-node))
			       (let ((x (car pos))
				     (y (cadr pos))
				     (arcs (node-arcs a-node)))
				 (stream-write-string *i-stream* ";;; Move Node ")
				 (print-object--stream a-node *i-stream* t)
				 (present-self a-node *gcanvas* t)
				 (dolist (an-arc arcs)
					 (present-self an-arc 
						       *gcanvas* t))
				 (move a-node x y)
				 (present-self a-node *gcanvas*)
				 (dolist (an-arc arcs)
					 (present-self an-arc
						       *gcanvas*))
				 (stream-write-string *i-stream*
						      " --> ")
				 (print-object--stream a-node *i-stream*))
			     nil)
			   )
      (setq *message-out* nil)))
)

(defmethod set-popup((button button))
  (if *set-menu*
      (popup-my-menu button)
    (progn 
      (setq *set-menu* (make-my-popup-menu :parent button :title *set-popup-title*
					    :callback-list *set-popup*))
      (popup-my-menu button))))

(defmethod set-arc-mode ()                             ;;added for extension
  (map-over-arcs (an-arc)                              ;;added for extension
		 (present-self an-arc *gcanvas* t))    ;;added for extension
  (if *arc-label-mode* (setq *arc-label-mode* nil)     ;;added for extension
    (setq *arc-label-mode* t))                         ;;added for extension
  (display *gcanvas*)                                  ;;added for extension
  (stream-write-strings *i-stream*                     ;;added for extension
			(";;;*arc-label-mode* is set " (format nil "~a" *arc-label-mode*)) t)
)                                                      ;;added for extension

(defmethod set-arc-label-after ((an-arc arc) &key)     ;;added for extension
  (stream-write-string *i-stream* "Arc's label is set : ") ;;added for extension
  (print-object--stream an-arc *i-stream*)             ;;added for extension
)                                                      ;;added for extension

(defmethod set-arc-label ()                            ;;added for extension
  (setq *message-out* t)                               ;;added for extension
  (multiple-value-bind (node1 node2)
		       (arcs-label-set)
		       (let ((arclabel (cadr node2))
			     an-arc)
			 (setq node2 (car node2))
			 (setq node1 (find-node node1)
			       node2 (find-node node2))
			 (if (and node1 node2)
			     (if (setq an-arc (find-arc node1 node2))
				 (progn
				   (present-self an-arc *gcanvas* t)
				   (setf (arc-label an-arc) arclabel)
				   (present-self an-arc *gcanvas*)
				   (set-arc-label-after an-arc))
			       (stream-write-strings *i-stream* ("*** Error *** There is such Arc!!!") t)))))
  (setq *message-out* nil)
)

(defmethod set-label ()
  (setq *message-out* t)
  (multiple-value-bind (node new-label)
		       (new-label-name)
		       (let ((a-node (find-node node)))
			      (if a-node
				  (progn
				    (stream-write-strings *i-stream*
							 (";;; Set label " new-label " to "))
				    (print-object--stream a-node *i-stream*)
				    (present-self a-node *gcanvas* t)
				    (with-slots (arcs) a-node
						(dolist (an-arc arcs)
							(present-self an-arc
								      *gcanvas* t))
						(setf (node-label a-node) new-label)
						(present-self a-node *gcanvas*)
						(dolist (an-arc arcs)
							(setf (arc-unique an-arc) (format nil "~a <--> ~a" 
										   (node-name (arc-node1 an-arc))
										   (node-name (arc-node2 an-arc))))
							(present-self an-arc
								      *gcanvas*))
						)
				    )
				nil)
			      )
		       )
  (setq *message-out* nil)
)

(defmethod add-an-arc-after ((an-arc arc) &key)
  (stream-write-string *i-stream* ";;;Added ")
  (print-object--stream an-arc *i-stream*)
)
  

(defmethod add-an-arc ()
  (catch :add-arc
    (progn
      (setq *message-out* t)
      (multiple-value-bind (node1 node2)
			   (add-arc-nodes)
			   (setq node1 (find-node node1)
				 node2 (find-node node2))
			   (if (and node1 node2)
			       (let ((an-arc 
				      (make-instance 'arc :node1 node1 
						     :node2 node2)))
				 (present-self an-arc *gcanvas*)
				 ;;print System Message
				 (add-an-arc-after an-arc)
				 )
			     )
			   )
      (setq *message-out* nil))
    )
)

(defmethod delete-an-arc-after ((an-arc arc) &key)
  (stream-write-string *i-stream* "Deleted ")
  (print-object--stream an-arc *i-stream*)
)

(defmethod delete-an-arc ()
  (catch :delete-arc
    (progn
      (setq *message-out* t)
      (multiple-value-bind (node1 node2)
			   (delete-arc-nodes)
			   (setq node1 (find-node node1)
				 node2 (find-node node2))
			   (if (and node1 node2)
			       (let ((an-arc (find-arc node1 node2)))
				 (if an-arc
				     (progn
				       (delete-self an-arc)
				       (present-self an-arc *gcanvas* t)
				       ;;print System Message
				       (delete-an-arc-after an-arc))
				   (progn 
				     (stream-write-string *i-stream* "*** Error *** There is No Arc!!!")
				     (stream-fresh-line *i-stream*))
				   )
				 )
			     nil)
			   )
      (setq *message-out* nil)))
)

(defmethod clear-all-data ()
  (stream-write-string *i-stream* "Clear all Data.")
  (map-over-nodes (a-node) 
		  (present-self a-node *gcanvas* t))
  (stream-write-char *i-stream* #\.)
  (map-over-arcs (an-arc)
		 (present-self an-arc *gcanvas* t))
  (stream-write-char *i-stream* #\.)
  (setq *all-the-nodes* nil 
	*next-node-index* 0)
  (stream-write-string *i-stream* "Done.")
  (stream-fresh-line *i-stream*)
)

;;;
;;; load and store functions
;;;

(defmethod store-data ()
  (let (fname p-node p-arc dest-node label
	      (*print-length* *print-length*)
	      (*print-level* *print-level*))
    (if (or *print-length* *print-level*)
	(setq *print-level* nil *print-length* nil))
    (with-open-file (file-id (setq fname (query-data *i-stream* :store-data))
			   :direction :io)
		  (format file-id ";;; This file is Grapher System's Data.~%")
		  (format file-id ";;; File name is ~s.~%" fname)
		  (dolist (node *all-the-nodes*)
			  (setq p-node
				(list 'node 
				      :xpos (node-xpos node)
				      :ypos (node-ypos node)
				      :label 
				      (if (setq label (node-label node))
					  label
					(format nil "stored-~d" (node-unique node)))
				      :radius 
				      (if label
					  (node-radius node)
					:needs-calculation)
				      :shape (node-shape node)
				      :arcs
				      (dolist (arc (node-arcs node) p-arc)
					      (if (equal node 
							 (setq dest-node
							       (arc-node1 arc)))
						  (setq dest-node 
							(arc-node2 arc)))
					      (pushnew 
					       (list (if (setq label (node-label dest-node))
						   label
						 (format nil "stored-~d"
							 (node-unique dest-node)))
						     (arc-label arc)) p-arc))
				      ))
			  (print p-node file-id)
			  (setq p-arc nil))
		  (print 'stop file-id))
    (stream-write-string *i-stream* "Storing Data is DONE.")
    (stream-fresh-line *i-stream*))
)

(defmethod load-data ()
  (let (loaded-arcs a-node)
    (with-open-file (file-id (truename (query-data *i-stream* :load-data))
			     :direction :input)
		   (do ((node (read file-id) (read file-id)))
		       ((equal node 'stop))
		       (if (and (listp node)(equal (car node) 'node))
			   (let ((xpos (cadr (member :xpos node :test #'equal)))
				 (ypos (cadr (member :ypos node :test #'equal)))
				 (radius (cadr (member :radius node :test #'equal)))
				 (label (cadr (member :label node :test #'equal)))
				 (shape (cadr (member :shape node :test #'equal)))
				 (arcs (cadr (member :arcs node :test #'equal))))
			     (if (find-node label t)
				 (progn
				   (stream-write-strings *i-stream*
							("Same label is already existing : " label) t)
				   (setq label (format nil "*stored-label(~s)*" label))
				   (setq radius :needs-calculation)))
			     (setq a-node
				   (make-instance 'node
						  :xpos xpos
						  :ypos ypos
						  :radius radius
						  :label label
						  :shape shape))
			     (pushnew (list a-node arcs) loaded-arcs))))
		   (dolist (arc loaded-arcs)
			   (dolist (dest-arc (cadr arc))
				   (if (listp dest-arc)
				       (setq label (cadr dest-arc) dest-arc (car dest-arc))
				     (setq label nil)) ;for upward compatibility in extension
				   (if (null (find-arc (car arc)
						       (or (find-node dest-arc)
							   (find-node 
							    (format nil "*storee-label(~s)" dest-arc)))))
				       (make-instance 'arc
						      :node1 (car arc)
						      :node2 
						      (or (find-node dest-arc)
							  (find-node
							   (format nil "*stored-label(~s)*" dest-arc)))
						      :label label))))
		   (display *gcanvas*)
		   )
    (stream-write-string *i-stream* "Loading Data is DONE.")
    (stream-fresh-line *i-stream*))
)
					       

;;;
;;; Inisizalize Grapher System
;;;
(defun init-gsystem ()
  (setq *all-the-nodes* nil *next-node-index* 0 *button-margin* 1)
  ;;; for Extension
  (setq *arc-label-mode* nil)
  ;;; Kludge Init
  (setq *my-menu* nil *set-menu* nil *gcanvas* nil *i-stream* nil
	*current-mode* :other *message-out* nil)
)
;;;
;;; main grapher
;;;

(defun grapher (&optional host)
  (let* ((disp (open-contact-display "Grapher" :host host))
	 (screen (contact-screen (display-root disp)))
	 (fg-color (screen-black-pixel screen))
;	 (bg-color (screen-white-pixel screen))
	 (top-shell (make-contact 'top-level-shell 
				  :parent disp :name 'Grapher
				  :width 1000 :height 800 
				  :background fg-color :icon-x 500 :icon-y 0))
	 (wrapper (make-contact 'grapher-system :parent top-shell :width 1000 :height 800))
	 grapher-canvas i-stream ffont-id
	 )
    (init-gsystem)
    (setq ffont-id (open-font disp 'fixed))
    (define-resources
      (* interactive-stream font) ffont-id
      (* canvas font) ffont-id
      (* fixed-menu font) ffont-id
      (* button font) ffont-id)
    (move-focus  top-shell :set :start wrapper)
    ;; make subwindows
    (multiple-value-bind (menu-width menu-height)
			 (make-my-fmenu :parent wrapper)
			 (setq grapher-canvas
			       (make-contact 'canvas :parent wrapper :x 0 :y 0
					     :cursor (my-cursor (if (find-package 'common-lisp) pencil-cursor
								  "pencil") ;;solve the gap ver 7.1 and ver 7.2
								disp)
					     :width (contact-width wrapper)
					     :height (- (contact-height wrapper) menu-height)))
			 (setq i-stream
			       (make-contact 'interactive-stream :parent wrapper
					     :x 0 :y (+ (contact-height grapher-canvas)
							(contact-border-width grapher-canvas))
					     :width (- (contact-width wrapper) menu-width)
					     :height menu-height))
			 (move-focus wrapper :set :start i-stream)
			 )
    (setq *my-menu* (make-my-popup-menu :parent grapher-canvas :title *grapher-menu-title*
					:callback-list *menu-callback*))
    (setq *gcanvas* grapher-canvas)
    (setq *i-stream* i-stream)
    ;; the layout of grapher system  is fixed
    ;; now add event and callback
    (add-event grapher-canvas '(:button-press :button-3) 
	       'popup-my-menu)
    (add-event wrapper ':destroy-notify 'throw-quit)
    ;; mapping contacts
    (update-state disp)
;    (set-input-focus disp *i-stream* :parent)
    ;;main event-loop
    (unwind-protect
		(catch :quit-selection
			(loop 
			 (process-next-event disp)))
	(close-display disp))
    )
)

(defun quit-selected ()
	(throw :quit-selection (print "Grapher is Over"))
)	
						       
