;;|=========================================================================|
;;|                         COPYRIGHT NOTICE                                |
;;|                                                                         |
;;|             Copyright 1990, 1991, 1992, 1993, 1994 Mark Tarver          |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun logic-difference
       (fp1 fp2 fp3 continuation)
       (cond ((and (rename '(var#1? var#2? var#3?))
                   (bind var#1? fp1)
                   (bind var#2? fp2)
                   (bind var#3? fp3)
                   (bind (lderef var#1?)
                         (my-set-difference (lderef var#2?)
                                            (lderef var#3?)))
                   (change-stacks)
                   (popstack continuation)))
             ((backtrack))))

(defun side-condition-help
       (fp1 continuation)
       (cond ((and (rename '(var#1?))
                   (bind var#1? fp1)
                   (evaluate-side-condition (lderef var#1?))
                   (change-stacks)
                   (popstack continuation)))
             ((backtrack))))

(defun compile-theory
       (fp1)
       (check-inbuilt-types fp1)
       (if (find-interactive-keyword fp1)
           (ct1 (rremove '@ (bracket fp1)))
           (ct1 (cons-form fp1))))

(defun check-inbuilt-types
       (fp1)
       (cond ((and (consp fp1)
                   (member (car fp1)
                           (list 'wff 'type)))
              (setq *autotypes* (union
                                  (list 'wff
                                        'type
                                        't-expr
                                        'proof-object
                                        'proof
                                        'sequent)
                                  *autotypes*)))
             (t t)))

(defun find-interactive-keyword
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq ':interactive (car fp1))
                   (eq 'yes (cadr fp1)))
              t)
             ((consp fp1) (find-interactive-keyword (cdr fp1)))
             (t nil)))

(defun ct1
       (fp1)
       (prog (sequent-structure)
             (parse-theory fp1)
             (setq *rewrites-auto* (list (list 'x '-> 'x)))
             (setq sequent-structure (compile-theory-to-sequent-structure
                                       (subst '<a>? '<a> fp1)))
             (test-overlap sequent-structure)
             (test-sequent sequent-structure)
             (if (interactive-structure sequent-structure)
                 (pushnew (car fp1) *theories*)
                 (pushnew (car fp1) *autotypes*))
             (if (pattern-matching-structure sequent-structure)
                 (setq *pattern* t))
             (fmakunbound (car fp1))
             (fmakunbound (concat (car fp1) 'rewrite))
             (put-prop (car fp1) 'inter nil)
             (if (interactive-structure sequent-structure)
                 (compile-inter-horn-clauses (car fp1)
                                             (compile-sequent-structure-to-horn-clauses
                                               sequent-structure))
                 (compile-horn-clauses
                   (compile-sequent-structure-to-horn-clauses sequent-structure)))
             (if (occurs (list 'iff 'yes)
                         sequent-structure)
                 (evalrldef (concat (car fp1) 'rewrite)
                            *rewrites-auto*))
             (if (interactive-structure sequent-structure)
                 (compile-tactics sequent-structure))
             (setq *pattern* nil)
             (return (car fp1))))

(defun interactive-structure
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 1)
                   (and (consp (cadar fp1))
                        (= (length (cadar fp1))
                           2))
                   (eq 'interactive
                       (caadr (car fp1)))
                   (eq 'yes
                       (cadar (cdar fp1))))
              t)
             (t nil)))

(defun pattern-matching-structure
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 2)
                   (and (consp (caddr fp1))
                        (= (length (caddr fp1))
                           2))
                   (eq 'pattern
                       (caadr (cdr fp1)))
                   (eq 'yes
                       (cadar (cddr fp1))))
              t)
             (t nil)))

(defun parse-theory
       (fp1)
       (prog (t2)
             (return
               (cond ((not (failure (setq t2 (non-interactive-theory fp1))))
                      t2)
                     (t (interactive-theory fp1))))))

(defun non-interactive-theory
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq ':interactive (cadr fp1))
                   (eq 'yes (caddr fp1)))
              (fail))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq ':interactive (cadr fp1))
                   (eq 'no (caddr fp1))
                   (consp (cdddr fp1)))
              (b-axioms (car fp1) (cdddr fp1)))
             ((and (consp fp1)
                   (consp (cdr fp1)))
              (b-axioms (car fp1) (cdr fp1)))
             (t (syntax-error fp1))))

(defun b-axioms
       (fp1 fp2)
       (cond ((null fp2) t)
             (t
              (prog (remainder)
                    (setq remainder (b-axiom fp1 fp2))
                    (if (eq 'fail remainder)
                        (syntax-error fp2)
                        (return (b-axioms fp1 remainder)))))))

(defun match-type (fp1 fp2)
  (cond ((equal fp2 fp1) t)
        ((and (consp fp2)
              (equal (car fp2) fp1))
         t)
        (t nil)))

(defun b-axiom (fp1 fp2)
  (cond ((and (consp fp2)
              (> (length fp2) 5)
              (eq 'thus (car fp2))
              (eq '<a> (cadr fp2))
              (eq '*
                  (caddr (cddr fp2)))
              (match-type fp1
                          (caddr (cdddr fp2)))
              (turnstile (caddr fp2)))
         (cdddr (cdddr fp2)))
        ((and (consp fp2)
              (> (length fp2) 10)
              (eq '<a> (car fp2))
              (eq '*
                  (caddr (cdr fp2)))
              (eq 'iff
                  (caddr (cdddr fp2)))
              (eq '<a>
                  (caddr (cdddr (cdr fp2))))
              (equal (caddr (cdddr (cddr fp2)))
                     (cadr fp2))
              (eq '*
                  (caddr (cdddr (cdddr (cdr fp2)))))
              (match-type fp1
                          (caddr (cdddr (cdddr (cddr fp2)))))
              (turnstile (cadr fp2)))
         (cdddr (cdddr (cdddr (cddr fp2)))))
        ((and (consp fp2)
              (> (length fp2) 4)
              (eq '<a> (car fp2))
              (eq '*
                  (caddr (cdr fp2)))
              (turnstile (cadr fp2)))
         (b-axiom fp1
                  (cdddr (cddr fp2))))
        (t 'fail)))

(defun interactive-theory
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq ':interactive (cadr fp1))
                   (eq 'yes (caddr fp1))
                   (eq ':pattern
                       (caddr (cdr fp1))))
              (and (id (car fp1))
                   (member (caddr (cddr fp1))
                           (list 'yes 'no))
                   (a-axioms (cdddr (cddr fp1)))))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq ':interactive (cadr fp1))
                   (eq 'yes (caddr fp1))
                   (id (car fp1)))
              (a-axioms (cdddr fp1)))
             (t (fail))))

(defun a-axioms
       (fp1)
       (cond ((null fp1) t)
             (t
              (prog (remainder)
                    (setq remainder (a-axiom fp1))
                    (if (eq 'fail remainder)
                        (syntax-error fp1)
                        (return (a-axioms remainder)))))))

(defun a-axiom
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 3)
                   (eq ':name (car fp1))
                   (eq ':side-condition
                       (caddr fp1))
                   (symbolp (cadr fp1))
                   (ok-side (caddr (cdr fp1))))
              (a-axiom (cdddr (cdr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 3)
                   (eq ':side-condition (car fp1))
                   (eq ':name (caddr fp1))
                   (ok-side (cadr fp1))
                   (symbolp (caddr (cdr fp1))))
              (a-axiom (cdddr (cdr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq ':name (car fp1))
                   (symbolp (cadr fp1)))
              (a-axiom-help (cddr fp1)))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq ':side-condition (car fp1))
                   (ok-side (cadr fp1)))
              (a-axiom (cddr fp1)))
             (t (a-axiom-help fp1))))

(defun ok-side
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'bind (car fp1))
                   (consp (caddr fp1))
                   (var (cadr fp1))
                   (every
                     #'(lambda (e)
                        (or (symbolp e) (integerp e) (stringp e)
                         (characterp e) (floatp e) (rationalp e)))
                     (cdadr (cdr fp1))))
              t)
             ((consp fp1) (every 'symbolp (cdr fp1)))
             (t nil)))

(defun a-axiom-help
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 5)
                   (eq 'thus (car fp1))
                   (eq '<a> (cadr fp1))
                   (eq '*
                       (caddr (cddr fp1)))
                   (turnstile (caddr fp1)))
              (cdddr (cdddr fp1)))
             ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq 'thus (car fp1))
                   (eq '* (caddr fp1))
                   (eq '|,|
                       (caddr (cddr fp1))))
              (a-axiom-help (cons 'thus
                                  (cdddr (cddr fp1)))))
             ((and (consp fp1)
                   (> (length fp1) 3)
                   (eq '* (cadr fp1))
                   (eq '|,| (caddr (cdr fp1))))
              (a-axiom-help (cdddr (cdr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq '<a> (car fp1))
                   (eq '* (caddr (cdr fp1)))
                   (turnstile (cadr fp1)))
              (a-axiom-help (cdddr (cddr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq '* (caddr (cdr fp1)))
                   (turnstile (cadr fp1))
                   (listp (car fp1)))
              (a-axiom-help (cdddr (cddr fp1))))
             (t 'fail)))

(defun syntax-error
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 2))
              (raise
                (format nil
                        "code 21: Syntax error near ... ~A ~A ~A ..."
                        (car fp1)
                        (cadr fp1)
                        (caddr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 1))
              (raise
                (format nil
                        "code 21: Syntax error near ... ~A ~A ..."
                        (car fp1)
                        (cadr fp1))))
             ((consp fp1)
              (raise
                (format nil
                        "Code 21: Syntax error near ... ~A ..."
                        (car fp1))))
             (t (raise (format nil
                               "code 21: Syntax error")))))

(defun compile-theory-to-sequent-structure
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq ':interactive (cadr fp1))
                   (eq ':pattern
                       (caddr (cdr fp1))))
              (cttss1 (cdddr (cddr fp1))
                      (list (list 'id (car fp1))
                            (list 'interactive
                                  (caddr fp1))
                            (list 'pattern
                                  (caddr (cddr fp1))))
                      (list (list 'alias nil)
                            (list 'side-conditions)
                            (list 'iff 'no)
                            (list 'i-sequent)
                            (list 'i-sequent-cond)
                            (list 'o-sequents)
                            (list 'o-sequents-cond))))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq ':interactive (cadr fp1)))
              (cttss1 (cdddr fp1)
                      (list (list 'id (car fp1))
                            (list 'interactive
                                  (caddr fp1))
                            (list 'pattern 'no))
                      (list (list 'alias nil)
                            (list 'side-conditions)
                            (list 'iff 'no)
                            (list 'i-sequent)
                            (list 'i-sequent-cond)
                            (list 'o-sequents)
                            (list 'o-sequents-cond))))
             ((consp fp1)
              (cttss1 (cdr fp1)
                      (list (list 'id (car fp1))
                            (list 'interactive 'no)
                            (list 'pattern 'no))
                      (list (list 'alias nil)
                            (list 'side-conditions)
                            (list 'iff)
                            (list 'i-sequent)
                            (list 'i-sequent-cond)
                            (list 'o-sequents)
                            (list 'o-sequents-cond))))
             (t
              (raise
                "code 13: No Patterns have Fired in compile-theory-to-sequent-structure"))))

(defun cttss1
       (fp1 fp2 fp3)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (> (length fp1) 5)
                   (eq 'thus (car fp1))
                   (eq '*
                       (caddr (cddr fp1)))
                   (and (consp fp3)
                        (> (length fp3) 3))
                   (and (consp (caddr (cdr fp3)))
                        (= (length (caddr (cdr fp3)))
                           1))
                   (eq 'i-sequent
                       (caadr (cddr fp3)))
                   (turnstile (caddr fp1)))
              (cons
                (append fp2
                        (cons (car fp3)
                              (cons (cadr fp3)
                                    (cons (list 'iff 'no)
                                          (cons
                                            (list 'i-sequent
                                                  (list (cadr fp1)
                                                        (caddr fp1)
                                                        (caddr (cdr fp1))
                                                        '*
                                                        (caddr (cdddr fp1))))
                                            (cdddr (cdr fp3)))))))
                (cttss1 (cdddr (cdddr fp1))
                        fp2
                        (list (list 'alias nil)
                              (list 'side-conditions)
                              (list 'iff)
                              (list 'i-sequent)
                              (list 'i-sequent-cond)
                              (list 'o-sequents)
                              (list 'o-sequents-cond)))))
             ((and (consp fp1)
                   (> (length fp1) 5)
                   (eq 'iff (car fp1))
                   (eq '* (caddr (cddr fp1)))
                   (and (consp fp3)
                        (> (length fp3) 3))
                   (and (consp (caddr (cdr fp3)))
                        (= (length (caddr (cdr fp3)))
                           1))
                   (eq 'i-sequent
                       (caadr (cddr fp3)))
                   (turnstile (caddr fp1)))
              (cons
                (append fp2
                        (cons (car fp3)
                              (cons (cadr fp3)
                                    (cons (list 'iff 'yes)
                                          (cons
                                            (list 'i-sequent
                                                  (list (cadr fp1)
                                                        (caddr fp1)
                                                        (caddr (cdr fp1))
                                                        '*
                                                        (caddr (cdddr fp1))))
                                            (cdddr (cdr fp3)))))))
                (cttss1 (cdddr (cdddr fp1))
                        fp2
                        (list (list 'alias nil)
                              (list 'side-conditions)
                              (list 'iff)
                              (list 'i-sequent)
                              (list 'i-sequent-cond)
                              (list 'o-sequents)
                              (list 'o-sequents-cond)))))
             ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq 'thus (car fp1))
                   (eq '* (caddr fp1))
                   (and (consp fp3)
                        (> (length fp3) 4))
                   (comma (caddr (cddr fp1))))
              (cttss1 (cons 'thus
                            (cdddr (cddr fp1)))
                      fp2
                      (cons (car fp3)
                            (cons (cadr fp3)
                                  (cons (caddr fp3)
                                        (cons (caddr (cdr fp3))
                                              (cons
                                                (append (caddr (cddr fp3))
                                                        (list
                                                          (list (cadr fp1)
                                                                '*
                                                                (caddr
                                                                  (cdr fp1)))))
                                                (cdddr (cddr fp3)))))))))
             ((and (consp fp1)
                   (> (length fp1) 3)
                   (eq '* (cadr fp1))
                   (and (consp fp3)
                        (= (length fp3) 7))
                   (comma (caddr (cdr fp1))))
              (cttss1 (cdddr (cdr fp1))
                      fp2
                      (list (car fp3)
                            (cadr fp3)
                            (caddr fp3)
                            (caddr (cdr fp3))
                            (caddr (cddr fp3))
                            (caddr (cdddr fp3))
                            (append (caddr (cdddr (cdr fp3)))
                                    (list (list (car fp1)
                                                '*
                                                (caddr fp1)))))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq ':name (car fp1))
                   (consp fp3))
              (cttss1 (cddr fp1)
                      fp2
                      (cons (list 'alias (cadr fp1))
                            (cdr fp3))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq ':side-condition (car fp1))
                   (and (consp fp3)
                        (> (length fp3) 1))
                   (consp (cadr fp3))
                   (eq 'side-conditions (caadr fp3)))
              (cttss1 (cddr fp1)
                      fp2
                      (cons (car fp3)
                            (cons
                              (cons 'side-conditions
                                    (append (cdadr fp3)
                                            (list (cadr fp1))))
                              (cddr fp3)))))
             ((and (consp fp1)
                   (> (length fp1) 4)
                   (eq '* (caddr (cdr fp1)))
                   (and (consp fp3)
                        (= (length fp3) 7))
                   (turnstile (cadr fp1)))
              (cttss1 (cdddr (cddr fp1))
                      fp2
                      (list (car fp3)
                            (cadr fp3)
                            (caddr fp3)
                            (caddr (cdr fp3))
                            (caddr (cddr fp3))
                            (append (caddr (cdddr fp3))
                                    (list
                                      (list (car fp1)
                                            (cadr fp1)
                                            (caddr fp1)
                                            '*
                                            (caddr (cddr fp1)))))
                            (append (caddr (cdddr (cdr fp3)))
                                    (list '+)))))
             (t (raise "code 13: No Patterns have Fired in cttss1"))))

(defun test-sequent
       (fp1)
       (mapcar 'test-sequent1
               (skolemise-rules fp1)))

(defun test-sequent1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 10)
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2))
                   (eq 'interactive (caadr fp1))
                   (eq 'yes
                       (cadar (cdr fp1)))
                   (and (consp (caddr (cdr fp1)))
                        (= (length (caddr (cdr fp1)))
                           2))
                   (eq 'alias
                       (caadr (cddr fp1)))
                   (consp (caddr (cddr fp1)))
                   (eq 'side-conditions
                       (caadr (cdddr fp1)))
                   (and (consp (caddr (cdddr (cdr fp1))))
                        (= (length (caddr (cdddr (cdr fp1))))
                           2))
                   (eq 'i-sequent
                       (caadr (cdddr (cddr fp1))))
                   (consp (caddr (cdddr (cddr fp1))))
                   (eq 'i-sequent-cond
                       (caadr (cdddr (cdddr fp1))))
                   (consp (caddr (cdddr (cdddr fp1))))
                   (eq 'o-sequents
                       (caadr (cdddr (cdddr (cdr fp1)))))
                   (consp (caddr (cdddr (cdddr (cdr fp1)))))
                   (eq 'o-sequents-cond
                       (caadr (cdddr (cdddr (cddr fp1))))))
              (if (not
                    (t-no-error (cadar (cdddr fp1))
                                (mk-seq-form (cadar (cdddr (cdddr fp1))))
                                (remove '+
                                        (cdadr (cdddr (cdddr fp1))))
                                (mapcar 'mk-seq-form
                                        (cdadr (cdddr (cdddr (cdr fp1)))))
                                (remove '+
                                        (cdadr (cdddr (cdddr (cddr fp1)))))
                                (cdadr (cdddr fp1))
                                (bld-context
                                  (list (cadar (cdddr (cdddr fp1)))
                                        (cdadr (cdddr (cdddr fp1)))
                                        (cdadr (cdddr (cdddr (cdr fp1))))
                                        (cdadr (cdddr (cdddr (cddr fp1))))
                                        (cdadr (cdddr fp1))))))
                  (raise
                    (format nil
                            "code 32: Type Error within axiom ~A"
                            (cadar (cdddr fp1))))))
             (t t)))

(defun mk-seq-form
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 5)
                   (eq '* (caddr (cdr fp1)))
                   (turnstile (cadr fp1)))
              (list (car fp1)
                    (cadr fp1)
                    (list (caddr fp1)
                          '*
                          (caddr (cddr fp1)))))
             (t (raise "code 13: No Patterns have Fired in mk-seq-form"))))

(defun t-no-error
       (fp1 fp2 fp3 fp4 fp5 fp6 fp7)
       (t-no-error-help fp1
                        (catch 'delay
                               (solve
                                 (list
                                   (list 'phi
                                         (put-conses fp2)
                                         'sequent
                                         fp7)
                                   (list 'phi
                                         (put-conses fp3)
                                         (list 'list 't-expr)
                                         fp7)
                                   (list 'phi
                                         (put-conses fp4)
                                         (list 'list 'sequent)
                                         fp7)
                                   (list 'phi
                                         (put-conses fp5)
                                         (list 'list 't-expr)
                                         fp7)
                                   (list 'phi-rpt fp6 'bool fp7))))))

(defun t-no-error-help
       (fp1 fp2)
       (cond ((eq 'delayed fp2)
              (format t
                      "~%==========================================================~%")
              (format t
                      " Axiom ~A contains an untypable side-condition.       ~%"
                      fp1)
              (format t
                      " Consult framework once more to eliminate this warning.   ~%")
              (format t
                      " Press <RETURN> to continue.                              ~%")
              (format t
                      "==========================================================~%")
              (read-char)
              t)
             (t fp2)))

(defun bld-context
       (fp1)
       (cond ((and (consp fp1)
                   (arbterm (car fp1)))
              (cons (list (car fp1) '* (newv))
                    (bld-context (subst t
                                        (car fp1)
                                        (cdr fp1)))))
             ((and (consp fp1)
                   (consp (car fp1)))
              (bld-context (append (car fp1) (cdr fp1))))
             ((consp fp1) (bld-context (cdr fp1)))
             (t nil)))

(defun test-overlap
       (fp1)
       (to1 (mapcar 'extract-i-sequent
                    (remove-if-not 'iff-structure fp1))))

(defun iff-structure
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 5)
                   (and (consp (caddr (cdddr fp1)))
                        (= (length (caddr (cdddr fp1)))
                           2))
                   (eq 'iff
                       (caadr (cdddr (cdr fp1))))
                   (eq 'yes
                       (cadar (cdddr (cddr fp1)))))
              t)
             (t nil)))

(defun extract-i-sequent
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 6)
                   (and (consp (caddr (cdddr (cdr fp1))))
                        (= (length (caddr (cdddr (cdr fp1))))
                           2))
                   (eq 'i-sequent
                       (caadr (cdddr (cddr fp1)))))
              (cadar (cdddr (cdddr fp1))))
             (t (raise "code 13: No Patterns have Fired in extract-i-sequent"))))

(defun to1
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (member (car fp1)
                           (cdr fp1)
                           ':test
                           'overlap))
              t)
             ((consp fp1) (to1 (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in to1"))))

(defun staprt (fp1) (staprt1 fp1 fp1))

(defun staprt1
       (fp1 fp2)
       (cond ((null fp1) fp2)
             ((and (consp fp1)
                   (var (car fp1)))
              (staprt1 (subst nil (car fp1) (cdr fp1))
                       (subst (newv) (car fp1) fp2)))
             ((and (consp fp1)
                   (consp (car fp1)))
              (staprt1 (append (car fp1) (cdr fp1))
                       fp2))
             ((consp fp1) (staprt1 (cdr fp1) fp2))
             (t (raise "code 13: No Patterns have Fired in staprt1"))))

(defun overlap
       (fp1 fp2)
       (cond ((solve (list (list 'logic-equal
                                 (staprt fp1)
                                 fp2)))
              (raise
                (format nil
                        "code 19: overlap between ~A and ~A"
                        (decons fp1)
                        (decons fp2))))
             (t nil)))

(defun decons
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'cons (car fp1)))
              (list (decons (cadr fp1))
                    '\|
                    (decons (caddr fp1))))
             ((consp fp1) (mapcar 'decons fp1))
             (t fp1)))

(defun compile-inter-horn-clauses
       (fp1 fp2)
       (put-prop fp1
                 'inter
                 (interprogram fp2)))

(defun interprogram
       (fp1)
       (mapcar 'ch-to-lambda-fun fp1))

(defun ch-to-lambda-fun
       (fp1)
       (clf1 (logiccode-of fp1)))

(defun clf1
       (fp1)
       (eval
         (list 'function
               (cons 'lambda
                     (cons (list 'fp1
                                 'fp2
                                 'fp3
                                 'continuation)
                           fp1)))))

(defun compile-sequent-structure-to-horn-clauses
       (fp1)
       (remove 'no-horn-clause
               (mapcar 'compile-sequent-structure-to-horn-clause
                       fp1)))

(defun compile-sequent-structure-to-horn-clause
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2))
                   (eq 'interactive (caadr fp1))
                   (eq 'no (cadar (cdr fp1))))
              (compile-sequent-structure-to-non-interactive-horn-clause
                (cons (car fp1)
                      (cons (list 'interactive 'no)
                            (cddr fp1)))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2))
                   (eq 'interactive (caadr fp1))
                   (eq 'yes (cadar (cdr fp1))))
              (compile-sequent-structure-to-interactive-horn-clause
                (cons (car fp1)
                      (cons (list 'interactive 'yes)
                            (cddr fp1)))))
             (t
              (raise
                "code 13: No Patterns have Fired in compile-sequent-structure-to-horn-clause"))))

(defun compile-sequent-structure-to-non-interactive-horn-clause (fp1)
  (cond ((and (consp fp1)
              (= (length fp1) 10)
              (and (consp (car fp1))
                   (= (length (car fp1))
                      2))
              (eq 'id (caar fp1))
              (and (consp (caddr (cdddr fp1)))
                   (= (length (caddr (cdddr fp1)))
                      2))
              (eq 'iff
                  (caadr (cdddr (cdr fp1))))
              (eq 'no
                  (cadar (cdddr (cddr fp1))))
              (and (consp (caddr (cdddr (cdr fp1))))
                   (= (length (caddr (cdddr (cdr fp1))))
                      2))
              (eq 'i-sequent
                  (caadr (cdddr (cddr fp1))))
              (and (consp (cadar (cdddr (cdddr fp1))))
                   (= (length (cadar (cdddr (cdddr fp1))))
                      5))
              (eq '*
                  (caddr (cdadr (caddr (cdddr (cdr fp1))))))
              (consp (caddr (cdddr (cdddr fp1))))
              (eq 'o-sequents
                  (caadr (cdddr (cdddr (cdr fp1)))))
              (turnstile (cadar (cdadr (cdddr (cddr fp1))))))
         (cons
           (list (cadar fp1)
                 (caddr (cadar (cdddr (cdddr fp1))))
                 'context?)
           (cons '<-
                 (compile-non-interactive-o-sequents-to-literals
                  (cdadr (cdddr (cdddr (cdr fp1))))))))
        ((and (consp fp1)
              (= (length fp1) 10)
              (and (consp (car fp1))
                   (= (length (car fp1))
                      2))
              (eq 'id (caar fp1))
              (and (consp (caddr (cdddr fp1)))
                   (= (length (caddr (cdddr fp1)))
                      2))
              (eq 'iff
                  (caadr (cdddr (cdr fp1))))
              (eq 'yes
                  (cadar (cdddr (cddr fp1))))
              (and (consp (caddr (cdddr (cdr fp1))))
                   (= (length (caddr (cdddr (cdr fp1))))
                      2))
              (eq 'i-sequent
                  (caadr (cdddr (cddr fp1))))
              (and (consp (cadar (cdddr (cdddr fp1))))
                   (= (length (cadar (cdddr (cdddr fp1))))
                      5))
              (eq '*
                  (caddr (cdadr (caddr (cdddr (cdr fp1))))))
              (consp (caddr (cdddr (cdddr fp1))))
              (eq 'o-sequents
                  (caadr (cdddr (cdddr (cdr fp1)))))
              (turnstile (cadar (cdadr (cdddr (cddr fp1))))))
         (add-auto-rewrite-rule
           (list (caddr (cadar (cdddr (cdddr fp1))))
                 '*
                 (caddr (cddar (cdadr (cdddr (cddr fp1))))))
           (rvin
             (list (caddr (cadar (cdddr (cdddr fp1))))
                   '*
                   (caddr (cddar (cdadr (cdddr (cddr fp1))))))
             (cdadr (cdddr (cdddr (cdr fp1))))))
         'no-horn-clause)
        (t
         (raise "code 13: No Patterns have Fired in
compile-sequent-structure-to-non-interactive-horn-clause"))))

(defun rvin (fp1 fp2) (rvin1 fp1 fp2 fp2))

(defun rvin1 (fp1 fp2 fp3)
  (cond ((null fp2) fp3)
        ((and (consp fp2)
              (consp (car fp2)))
         (rvin1 fp1
                (append (car fp2) (cdr fp2))
                fp3))
        ((and (consp fp2)
              (var (car fp2))
              (eigen (car fp2) fp1))
         (rvin1 fp1
                (cdr fp2)
                (subst (list '$ (gentemp))
                       (car fp2)
                       fp3)))
        ((consp fp2) (rvin1 fp1 (cdr fp2) fp3))
        (t (raise "code 13: No Patterns have Fired in rvin1"))))

(defun add-auto-rewrite-rule
       (fp1 fp2)
       (push (list fp1 '-> (mapcar 'aarr1 fp2))
             *rewrites-auto*))

(defun aarr1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 5)
                   (eq '* (caddr (cdr fp1)))
                   (turnstile (cadr fp1)))
              (list (caddr fp1)
                    '*
                    (caddr (cddr fp1))))
             (t (raise "code 13: No Patterns have Fired in aarr1"))))

(defun compile-non-interactive-o-sequents-to-literals
       (fp1)
       (mapcar 'compile-non-interactive-o-sequent-to-literal
               fp1))

(defun compile-non-interactive-o-sequent-to-literal
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 5)
                   (eq '* (caddr (cdr fp1)))
                   (turnstile (cadr fp1)))
              (list 'phi
                    (caddr fp1)
                    (caddr (cddr fp1))
                    'context?))
             (t
              (raise
                "code 13: No Patterns have Fired in compile-non-interactive-o-sequent-to-literal"))))

(defun compile-sequent-structure-to-interactive-horn-clause
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 10)
                   (and (consp (caddr fp1))
                        (= (length (caddr fp1))
                           2))
                   (eq 'pattern
                       (caadr (cdr fp1)))
                   (consp (caddr (cdr fp1)))
                   (eq 'alias
                       (caadr (cddr fp1)))
                   (consp (caddr (cddr fp1)))
                   (eq 'side-conditions
                       (caadr (cdddr fp1)))
                   (and (consp (caddr (cdddr (cdr fp1))))
                        (= (length (caddr (cdddr (cdr fp1))))
                           2))
                   (eq 'i-sequent
                       (caadr (cdddr (cddr fp1))))
                   (and (consp (cadar (cdddr (cdddr fp1))))
                        (= (length (cadar (cdddr (cdddr fp1))))
                           5))
                   (eq '*
                       (caddr (cdadr (caddr (cdddr (cdr fp1))))))
                   (consp (caddr (cdddr (cddr fp1))))
                   (eq 'i-sequent-cond
                       (caadr (cdddr (cdddr fp1))))
                   (consp (caddr (cdddr (cdddr fp1))))
                   (eq 'o-sequents
                       (caadr (cdddr (cdddr (cdr fp1)))))
                   (consp (caddr (cdddr (cdddr (cdr fp1)))))
                   (eq 'o-sequents-cond
                       (caadr (cdddr (cdddr (cddr fp1)))))
                   (turnstile (cadar (cdadr (cdddr (cddr fp1))))))
              (mk-side-conditions (cdadr (cdddr fp1))
                                  (cons
                                    (list 'junk
                                          (caadr (caddr (cdddr (cdr fp1))))
                                          (caddr
                                            (cadar (cdddr (cdddr fp1))))
                                          (caddr
                                            (cddar
                                              (cdadr (cdddr (cddr fp1))))))
                                    (cons '<-
                                          (append
                                            (compile-i-sequent-cond-to-literals
                                              (cdadr (cdddr (cdddr fp1))))
                                            (compile-mk-new-context
                                              (cdadr (cdddr (cdddr fp1))))
                                            (compile-interactive-os-sequents-to-literals
                                              (cdadr
                                                (cdddr (cdddr (cddr fp1))))
                                              (cdadr
                                                (cdddr (cdddr (cdr fp1))))))))))
             (t
              (raise
                "code 13: No Patterns have Fired in compile-sequent-structure-to-interactive-horn-clause"))))

(defun compile-i-sequent-cond-to-literals
       (fp1)
       (mapcar 'compile-i-sequent-cond-to-literal
               fp1))

(defun compile-i-sequent-cond-to-literal
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq '* (cadr fp1)))
              (list 'immediate
                    (list (car fp1) '* (caddr fp1))
                    '<a>?))
             (t
              (raise
                "code 13: No Patterns have Fired in compile-i-sequent-cond-to-literal"))))

(defun compile-mk-new-context
       (fp1)
       (cond ((null fp1) (list (list 'logic-equal
                                     '<a>?
                                     '<b>?)))
             (t (list (list 'logic-difference
                            '<b>?
                            '<a>?
                            fp1)))))

(defun my-set-difference
       (fp1 fp2)
       (cond ((null fp2) fp1)
             ((consp fp2)
              (my-set-difference (remove-first (car fp2) fp1)
                                 (cdr fp2)))
             (t (raise "code 13: No Patterns have Fired in my-set-difference"))))

(defun remove-first
       (fp1 fp2)
       (cond ((null fp2) nil)
             ((and (consp fp2)
                   (equal (car fp2) fp1))
              (cdr fp2))
             ((consp fp2) (cons (car fp2)
                                (remove-first fp1 (cdr fp2))))
             (t (raise "code 13: No Patterns have Fired in remove-first"))))

(defun compile-interactive-os-sequents-to-literals
       (fp1 fp2)
       (list (list 'return-sequents
                   (costl1 fp1 fp2))))

(defun return-sequents
       (fp1 fp2)
       (prog1 (append (eval-**append** fp1)
                      (deref fp2))
              (clrhash *binding-array*)))

(defun eval-**append**
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq '**append** (car fp1)))
              (append (cadr fp1) (caddr fp1)))
             ((consp fp1) (mapcar 'eval-**append** fp1))
             (t fp1)))

(defun costl1
       (fp1 fp2)
       (cond ((and (null fp1) (null fp2)) nil)
             ((consp fp2)
              (cons (costl2 (first-ocs fp1)
                            (car fp2))
                    (costl1 (rest-ocs fp1)
                            (cdr fp2))))
             (t (raise "code 13: No Patterns have Fired in costl1"))))

(defun first-ocs
       (fp1)
       (cond ((and (consp fp1)
                   (eq '+ (car fp1)))
              nil)
             ((consp fp1) (cons (car fp1)
                                (first-ocs (cdr fp1))))
             (t (raise "code 13: No Patterns have Fired in first-ocs"))))

(defun rest-ocs
       (fp1)
       (cond ((and (consp fp1)
                   (eq '+ (car fp1)))
              (cdr fp1))
             ((consp fp1) (rest-ocs (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in rest-ocs"))))

(defun costl2
       (fp1 fp2)
       (cond ((and (null fp1)
                   (consp fp2)
                   (= (length fp2) 5)
                   (null (car fp2))
                   (eq '* (caddr (cdr fp2)))
                   (turnstile (cadr fp2)))
              (list nil
                    (cadr fp2)
                    (list (caddr fp2)
                          '*
                          (caddr (cddr fp2)))))
             ((and (consp fp2)
                   (= (length fp2) 5)
                   (null (car fp2))
                   (eq '* (caddr (cdr fp2)))
                   (turnstile (cadr fp2)))
              (list (list '**append** fp1 nil)
                    (cadr fp2)
                    (list (caddr fp2)
                          '*
                          (caddr (cddr fp2)))))
             ((and (null fp1)
                   (consp fp2)
                   (= (length fp2) 5)
                   (eq '* (caddr (cdr fp2))))
              (list '<b>?
                    (cadr fp2)
                    (list (caddr fp2)
                          '*
                          (caddr (cddr fp2)))))
             ((and (consp fp2)
                   (= (length fp2) 5)
                   (eq '* (caddr (cdr fp2)))
                   (turnstile (cadr fp2)))
              (list (list '**append** fp1 '<b>?)
                    (cadr fp2)
                    (list (caddr fp2)
                          '*
                          (caddr (cddr fp2)))))
             (t (raise "code 13: No Patterns have Fired in costl2"))))

(defun mk-side-conditions
       (fp1 fp2)
       (cond ((null fp1) fp2)
             ((consp fp1)
              (mk-side-conditions (cdr fp1)
                                  (penultimate-call (csc (car fp1))
                                                    fp2)))
             (t
              (raise "code 13: No Patterns have Fired in mk-side-conditions"))))

(defun csc
       (fp1)
       (list 'side-condition fp1))

(defun side-condition
       (fp1 fp2)
       (popstack (cons (list 'side-condition-help fp1)
                       (deref fp2))))

(defun evaluate-side-condition
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'bind (car fp1))
                   (consp (caddr fp1)))
              (bind (cadr fp1)
                    (apply (caadr (cdr fp1))
                           (cdadr (cdr fp1)))))
             ((consp fp1) (apply (car fp1) (cdr fp1)))
             (t
              (raise
                "code 13: No Patterns have Fired in evaluate-side-condition"))))

(defun penultimate-call
       (fp1 fp2)
       (cond ((and (consp fp2)
                   (= (length fp2) 1))
              (list fp1 (car fp2)))
             ((consp fp2)
              (cons (car fp2)
                    (penultimate-call fp1 (cdr fp2))))
             (t (raise "code 13: No Patterns have Fired in penultimate-call"))))

(defun insert-@
       (fp1)
       (cond ((consp fp1) (cons '@
                                (mapcar 'insert-@ fp1)))
             (t fp1)))

(defun compile-tactics
       (fp1)
       (compile-tactics1 fp1 1))

(defun compile-tactics1
       (fp1 fp2)
       (cond ((null fp1) t)
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 3)
                   (and (consp (caddr (cdar fp1)))
                        (= (length (caddr (cdar fp1)))
                           2))
                   (eq 'alias
                       (caadr (cddar fp1)))
                   (null (cadar (cdddr (car fp1)))))
              (compile-tactics1 (cdr fp1) (1+ fp2)))
             ((and (consp fp1)
                   (consp (car fp1))
                   (> (length (car fp1)) 3)
                   (and (consp (caar fp1))
                        (= (length (caar fp1))
                           2))
                   (eq 'id (caar (car fp1)))
                   (and (consp (caddr (cdar fp1)))
                        (= (length (caddr (cdar fp1)))
                           2))
                   (eq 'alias
                       (caadr (cddar fp1))))
              (gentac (cadar (car fp1))
                      (cadar (cdddr (car fp1)))
                      fp2)
              (compile-tactics1 (cdr fp1) (1+ fp2)))
             (t (raise "code 13: No Patterns have Fired in compile-tactics1"))))

(defun gentac
       (fp1 fp2 fp3)
       (gentac1
         (list 'deftactic
               fp2
               '|{|
               'proof
               '->
               'proof
               '|}|
               'x
               '->
               (list 'refine fp1 fp3 'x))))

(defun gentac1
       (fp1)
       (typecheck fp1 (type-var fp1))
       (eval fp1))
