/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */

/*
 *  clause.c -- This file has routines assiciated with the clause data type.
 *
 */

#include "header.h"

#ifdef TURBO_C
#define CLAUSE_TAB_SIZE 100
#else
#ifdef THINK_C
#define CLAUSE_TAB_SIZE 100
#else
#define CLAUSE_TAB_SIZE 1000
#endif
#endif
    
/* hash table for accessing clauses by ID */
static struct clause_ptr *Clause_tab[CLAUSE_TAB_SIZE];

/* # of clauses integrated; used for clause id */
#ifndef ROO
/* In Glob for ROO. */
static int Int_clause_count;
#endif

/* back subsumed, demodulated put here, not deleted */
static struct clause *Hidden_clauses;

/* array to mark mapped literals during subsumption */
#define MAX_LITS 100
static char Map_array[MAX_LITS];

#ifdef ROO

/*************
 *
 *    init_clause_tab_for_roo()
 *
 *    Insert dummy nodes so that all slaves will share table with master.
 *
 *************/

init_clause_tab_for_roo()
{
    int i;
    
    for (i=0; i<CLAUSE_TAB_SIZE; i++) {
        Clause_tab[i] = get_clause_ptr(); /* insert dummy node */
        Clause_tab[i]->c = NULL;
	}
}  /* init_clause_tab_for_roo */

#endif

/*************
 *
 *    int next_cl_num()
 *
 *    What is the next clause number?  Do not increment the count.
 *
 *************/

int next_cl_num()
{
#ifdef ROO
    return(Glob->Int_clause_count + 1);
#else
    return(Int_clause_count + 1);
#endif
}  /* next_cl_num */

/*************
 *
 *    cl_integrate(c) -- integrate a clause
 *
 *    This routine integrates most subterms of the atoms. (Incoming clause must
 *    already have back pointers from literal to clause and atom to literal.)
 *
 *    The atoms are not shared, and arguments of positive equality literals
 *    are not shared.
 *
 *    A clause is integrated iff its ID is > 0.
 *
 *************/

void cl_integrate(c)
struct clause *c;
{
    struct literal *lit;
    struct term *atom;
    struct rel *r, *r1;

    if (c->id != 0) {
	fprintf(stdout, "WARNING, cl_integrate, called with integrated clause: ");
	print_clause(stdout, c);
	}
    else {
#ifdef ROO
	c->id = ++(Glob->Int_clause_count);
#else
	c->id = ++Int_clause_count;
#endif
	cl_insert_tab(c);
	lit = c->first_lit;
	while (lit != NULL) {
	    atom = lit->atom;
	    if (atom->varnum == POS_EQ || atom->varnum == LEX_DEP_DEMOD ||
		atom->varnum == CONDITIONAL_DEMOD) {
		/* do not share (condition), alpha, beta */
		r1 = atom->farg;
		while (r1 != NULL) {  /* for alpha and beta */
		    bd_kludge_insert(r1->argval);  /* put it where back demod can find it */
		    if (r1->argval->type == COMPLEX) {
			r = r1->argval->farg;
			while (r != NULL) {
			    r->argval = integrate_term(r->argval);
			    r->argof = r1->argval;
			    r->nocc = r->argval->occ.rel;
			    r->argval->occ.rel = r;
			    r = r->narg;
			    }
			}
		    r1->argof = atom;
		    r1->argval->occ.rel = r1;
		    r1 = r1->narg;
		    }
		}
	    else if (atom->type == COMPLEX) {
		r = atom->farg;
		while (r != NULL) {
		    r->argval = integrate_term(r->argval);
		    r->argof = atom;
		    r->nocc = r->argval->occ.rel;
		    r->argval->occ.rel = r;
		    r = r->narg;
		    }
		}
	    lit = lit->next_lit;
	    }
	}

}  /* cl_integrate */

/*************
 *
 *    cl_del_int(c) -- deallocate an integrated clause.
 *
 *************/

void cl_del_int(c)
struct clause *c;
{
    struct literal *lit, *plit;
    struct rel *r, *r2, *pr, *r1;
    struct term *atom;
    struct int_ptr *ip1, *ip2;

    lit = c->first_lit;
    while (lit != NULL) {
	atom = lit->atom;
	if (atom->varnum == POS_EQ || atom->varnum == LEX_DEP_DEMOD ||
		atom->varnum == CONDITIONAL_DEMOD) {
	    /* (condition), alpha, beta not shared */
	    r1 = atom->farg;
	    while(r1 != NULL) {  /* for alpha and beta */
		bd_kludge_delete(r1->argval);  /* back demod kludge */
		r = r1->argval->farg;
		while (r != NULL) {
		    r2 = r->argval->occ.rel;
		    pr = NULL;
		    while (r2 != NULL && r2 != r) {
			pr = r2;
			r2 = r2->nocc;
			}
		    if (r2 == NULL) {
			output_stats(stdout, 4);
			fprintf(stderr, "ABEND, cl_del_int, bad equality clause.\007\n");
			fprintf(stdout, "ABEND, cl_del_int, bad equality clause: ");
			print_clause(stdout, c);
			exit(1);
			}
		    if (pr == NULL)
			r->argval->occ.rel = r->nocc;
		    else
			pr->nocc = r->nocc;
		    if (r->argval->occ.rel == NULL)
			disintegrate_term(r->argval);
		    pr = r;
		    r = r->narg;
		    free_rel(pr);
		    }
		free_term(r1->argval);  /* alpha or beta */
		pr = r1;
		r1 = r1->narg;
		free_rel(pr);
		}
	    }
	else if (atom->type == COMPLEX) {
	    r = atom->farg;
	    while (r != NULL) {
		r2 = r->argval->occ.rel;
		pr = NULL;
		while (r2 != NULL && r2 != r) {
		    pr = r2;
		    r2 = r2->nocc;
		    }
		if (r2 == NULL) {
		    output_stats(stdout, 4);
		    fprintf(stderr, "ABEND, cl_del_int, bad clause.\007\n");
		    fprintf(stdout, "ABEND, cl_del_int, bad clause: ");
		    print_clause(stdout, c);
		    exit(1);
		    }
		if (pr == NULL)
		    r->argval->occ.rel = r->nocc;
		else
		    pr->nocc = r->nocc;
		if (r->argval->occ.rel == NULL)
		    disintegrate_term(r->argval);
		pr = r;
		r = r->narg;
		free_rel(pr);
		}
	    }
	free_term(atom);
	plit = lit;
	lit = lit->next_lit;
	free_literal(plit);
	}
    ip1 = c->parents;
    while (ip1 != NULL) {
	ip2 = ip1;
	ip1 = ip1->next;
	free_int_ptr(ip2);
	}
    cl_delete_tab(c);
    /* If there is other memory associated with clause, free it here */
    free_clause(c);
}  /* cl_del_int */

/*************
 *
 *    cl_del_non(c) -- deallocate a nonintegrated clause.
 *
 *************/

void cl_del_non(c)
struct clause *c;
{
    struct literal *lit, *plit;
    struct int_ptr *ip1, *ip2;

    lit = c->first_lit;
    while (lit != NULL) {
	lit->atom->occ.lit = NULL;
	zap_term(lit->atom);
	plit = lit;
	lit = lit->next_lit;
	free_literal(plit);
	}
    ip1 = c->parents;
    while (ip1 != NULL) {
	ip2 = ip1;
	ip1 = ip1->next;
	free_int_ptr(ip2);
	}
    /* If there is other memory associated with clause, free it here */
    free_clause(c);
}  /* cl_del_non */

/*************
 *
 *    cl_int_chk(c) -- check structure of clause -- for debugging
 *
 *************/

void cl_int_chk(c)
struct clause *c;
{
    struct literal *lit;

    printf("checking clause, address:%x " , c);
    print_clause(stdout, c);
    lit = c->first_lit;
    while (lit != NULL) {
	printf("    literal, address:%x sign:%d type:%d; atom:", lit, lit->sign, lit->atom->varnum);
	print_term(stdout, lit->atom); printf("\n");
	printf("    cont_cl:%x, atom container:%x\n", lit->container, lit->atom->occ.lit);
	lit = lit->next_lit;
	}
}  /* cl_int_chk */

/*************
 *
 *    struct list *read_cl_list(fp, errors_ptr)
 *
 *    Read clauses until EOF or the term `end_of_list' is reached.
 *
 *************/

struct list *read_cl_list(fp, ep)
FILE *fp;
int *ep;
{
    struct list *lst;
    struct clause *cl, *pcl;
    int rc;

    *ep = 0;
    lst = get_list();
    pcl = NULL;
    cl = read_clause(fp, &rc);
    while (rc == 0) {  /* while errors */
	(*ep)++;
	cl = read_clause(fp, &rc);
	}
    while (cl != NULL) {
	if (pcl == NULL)
	    lst->first_cl = cl;
	else
	    pcl->next_cl = cl;
	cl->prev_cl = pcl;
	cl->container = lst;
	pcl = cl;
	cl = read_clause(fp, &rc);
	while (rc == 0) {  /* while errors */
	    (*ep)++;
	    cl = read_clause(fp, &rc);
	    }
	}
    lst->last_cl = pcl;
    return(lst);
}  /* read_cl_list */
 
/*************
 *
 *    struct clause *read_clause(fp, retcd_ptr)
 *
 *************/

struct clause *read_clause(fp, rcp)
FILE *fp;
int *rcp;
{
    char buf[MAX_BUF];
    int p, go, sign, rc;
    struct term *t;
    struct clause *cl;
    struct literal *lit, *plit;

    rc = read_buf(fp, buf);
    if (rc == 0) {  /* error */
	*rcp = 0;
	return(NULL);
	}
    else if (buf[0] == '\0') {  /* ok. EOF */
	*rcp = 1;
	return(NULL);
	}
    else {
	cl = get_clause();
	plit = NULL;  /* prev lit */
	p = 0;
	skip_white(buf, &p);
	go = 1;
	while (go) {
	    sign = (buf[p] != '-');
	    if (sign == 0)
		p++;
	    t = str_atom(buf, &p, &rc);
	    if (rc == 0)
		sign = (sign ? 0 : 1);
	    if (t == NULL) {
		cl_del_non(cl);
		*rcp = 0;
		return(NULL);
		}
	    else if (t->type == NAME && str_ident(sn_to_str(t->sym_num),"end_of_list"))
		if (cl->first_lit == NULL) {
		    cl_del_non(cl);
		    free_term(t);
		    *rcp = 1;
		    return(NULL);
		    }
		else {
		    fprintf(stdout, "ERROR, bad literal:\n");
		    print_error(stdout, buf, p);
		    cl_del_non(cl);
		    free_term(t);
		    *rcp = 0;
		    return(NULL);
		    }
	    if (t->type == NAME && var_name(sn_to_str(t->sym_num))) {
		fprintf(stdout, "ERROR, literal is variable:\n");
		print_error(stdout, buf, p);
		cl_del_non(cl);
		free_term(t);
		*rcp = 0;
		return(NULL);
		}
	    lit = get_literal();
	    lit->atom = t;
	    lit->sign = sign;
	    lit->container = cl;  /* literal point up to clause */
	    t->occ.lit = lit;  /* atom point up to literal */
	    mark_literal(lit);  /* atoms have varnum > 0 */
	    if (plit == NULL)
		cl->first_lit = lit;
	    else
		plit->next_lit = lit;
	    plit = lit;
	    skip_white(buf, &p);
	    if (buf[p] == '|') {
		p++;
		skip_white(buf, &p);
		}
	    else if (buf[p] == '.')
		go = 0;
	    else {
		fprintf(stdout, "ERROR, | or . expected in clause:\n");
		print_error(stdout, buf, p);
		cl_del_non(cl);
		*rcp = 0;
		return(NULL);
		}
	    }
	if (set_vars_cl(cl)) {
	    for (lit = cl->first_lit; lit; lit = lit->next_lit) {
		if (contains_skolem_symbol(lit->atom)) {
		    fprintf(stdout, "\nERROR, input clause contains Skolem symbol:\n");
		    print_clause(stdout, cl); printf("\n");
		    cl_del_non(cl);
		    *rcp = 0;
		    return(NULL);
		    }
		}
	    Stats[CL_INPUT]++;
	    *rcp = 1;
	    return(cl);
	    }
	else {
	    fprintf(stdout, "ERROR, too many variables in clause, max is %d:\n%s\n",
			       MAX_VARS, buf);
	    cl_del_non(cl);
	    *rcp = 0;
	    return(NULL);
	    }
	}
}  /* read_clause */

/*************
 *
 *    int set_vars_cl(cl) -- decide which terms are variables
 *
 *************/

int set_vars_cl(cl)
struct clause *cl;
{
    struct literal *lit;
    char *varnames[MAX_VARS];
    int i;

    for (i=0; i<MAX_VARS; i++)
	varnames[i] = NULL;
    lit = cl->first_lit;
    while (lit != NULL) {
	if (set_vars_term(lit->atom, varnames))
	    lit = lit->next_lit;
	else
	    return(0);
	}
    return(1);
}

/*************
 *
 *    print_clause(fp, clause)
 *
 *************/

void print_clause(fp, cl)
FILE *fp;
struct clause *cl;
{
    struct literal *lit;
    struct int_ptr *ip;

    fprintf(fp, "%d ", cl->id);
    fprintf(fp,"[");
    if (cl->parents != NULL) {
	ip = cl->parents;
	while (ip != NULL) {
	    switch (ip->i) {
		case BINARY_RES_RULE  : fprintf(fp, "binary"); break;
		case HYPER_RES_RULE   : fprintf(fp, "hyper"); break;
		case NEG_HYPER_RES_RULE   : fprintf(fp, "neg_hyper"); break;
		case UR_RES_RULE      : fprintf(fp, "ur"); break;
		case PARA_INTO_RULE   : fprintf(fp, "para_into"); break;
		case PARA_FROM_RULE   : fprintf(fp, "para_from"); break;
		case FACTOR_RULE      : fprintf(fp, "factor"); break;
		case NEW_DEMOD_RULE   : fprintf(fp, "new_demod"); break;
		case BACK_DEMOD_RULE  : fprintf(fp, "back_demod"); break;
		case DEMOD_RULE       : fprintf(fp, "demod"); break;
		case UNIT_DEL_RULE    : fprintf(fp, "unit_del"); break;
		case NEW_FUNCTION_RULE: fprintf(fp, "new_function"); break;
		case FLIP_EQ_RULE     : fprintf(fp, "flip_eq"); break;
		case LINKED_UR_RES_RULE : fprintf(fp, "linked_ur_res"); break;
		case EVAL_RULE        : fprintf(fp, "eval"); break;
		default               : fprintf(fp, "%d", ip->i); break;
		}
	    ip = ip->next;
	    if (ip != NULL)
		fprintf(fp, ",");
	    }
	}
    fprintf(fp, "] ");
    lit = cl->first_lit;
    while (lit != NULL) {
	if (lit->sign == 0 && lit->atom->sym_num != Eq_sym_num)
	    fprintf(fp, "-");
	print_term(fp, lit->atom);
	lit = lit->next_lit;
	if (lit != NULL)
	    fprintf(fp, " | ");
	}
    fprintf(fp, ".\n");
}  /* print_clause */

/*************
 *
 *    p_clause(clause)
 *
 *************/

void p_clause(cl)
struct clause *cl;
{
    print_clause(stdout, cl);
}  /* p_clause */

/*************
 *
 *    print_cl_list(fp, lst)
 *
 *************/

void print_cl_list(fp, lst)
FILE *fp;
struct list *lst;
{
    struct clause *cl;

    if (lst == NULL)
	fprintf(fp, "(list nil)\n");
    else {
	cl = lst->first_cl;
	while (cl != NULL) {
	    print_clause(fp, cl);
	    cl = cl->next_cl;
	    }
	fprintf(fp, "end_of_list.\n");
	}
}  /* print_cl_list */

/*************
 *
 *    cl_merge(cl) -- merge identical literals (keep leftmost occurrence)
 *
 *************/

void cl_merge(c)
struct clause *c;
{
    struct literal *l1, *l2, *l_prev;

    l1 = c->first_lit;
    while (l1 != NULL) {
	l2 = l1->next_lit;
	l_prev = l1;
	while (l2 != NULL)
	    if (l1->sign == l2->sign && term_ident(l1->atom, l2->atom)) {
		l_prev->next_lit = l2->next_lit;
		l2->atom->occ.lit = NULL;
		zap_term(l2->atom);
		free_literal(l2);
		l2 = l_prev->next_lit;
		}
	    else {
		l_prev = l2;
		l2 = l2->next_lit;
		}
	l1 = l1->next_lit;
	}
}  /* cl_merge */

/*************
 *
 *     int tautology(c) -- Is clause c a tautology?
 *
 *************/

int tautology(c)
struct clause *c;
{
    struct literal *l1, *l2;
    int taut;

    taut = 0;
    l1 = c->first_lit;
    while (l1 != NULL && taut == 0) {
	l2 = l1->next_lit;
	while (l2 != NULL && taut == 0) {
	    taut = (l1->sign != l2->sign && term_ident(l1->atom, l2->atom));
	    l2 = l2->next_lit;
	    }
	l1 = l1->next_lit;
	}
    return(taut);
}  /* tautology */

/*************
 *
 *    int size_ancestor_bag(c)
 *
 *    Count the bag (not set) of ancestors.
 *
 *************/

int size_ancestor_bag(c)
struct clause *c;
{
    struct int_ptr *ip;
    int count, id;
    struct clause *parent;

    count = 0;
    for (ip = c->parents; ip != NULL; ip = ip->next) {
	id = ip->i;
	if (id > 0) {
	    parent = cl_find(id);
	    if (parent != NULL)
		count = count + 1 + size_ancestor_bag(parent);
	    }
	}
    return(count);
}  /* size_ancestor_bag */

/*************
 *
 *    int size_ancestor_set(c)
 *
 *    Count the set of ancestors.
 *
 *************/

int size_ancestor_set(c)
struct clause *c;
{
    struct clause_ptr *cp1, *cp2;
    int count;

    cp1 = NULL;
    get_ancestors(c, &cp1);

    for (count = -1; cp1 != NULL; count++) {
	cp2 = cp1;
	cp1 = cp1->next;
	free_clause_ptr(cp2);
	}

    return(count);
}  /* size_ancestor_set */

/*************
 *
 *    int proof_length(c)
 *
 *    Return length of proof.  If demod_history is clear, demodulation
 *    steps are not counted.  "new_demod" steps are not counted.
 *
 *************/

int proof_length(c)
struct clause *c;
{
    struct clause_ptr *cp1, *cp2;
    int count;

    cp1 = NULL;
    get_ancestors(c, &cp1);

    for (count = 0; cp1 != NULL; ) {
        if (cp1->c->parents && cp1->c->parents->i != NEW_DEMOD_RULE)
	    count++;
	cp2 = cp1;
	cp1 = cp1->next;
	free_clause_ptr(cp2);
	}

    return(count);
}  /* proof_length */

/*************
 *
 *    int subsume(c, d) -- does clause c subsume clause d?
 *
 *************/

int subsume(c,d)
struct clause *c;
struct clause *d;
{
    struct context *s;
    struct trail *tr;
    int subsumed;

    s = get_context();
    tr = NULL;
    subsumed = map_rest(c, d, s, &tr);
    if (subsumed)
	clear_subst_1(tr);
    free_context(s);
    return(subsumed);
}  /* subsume */

/*************
 *
 *    int map_rest(c, d, s, trp) - map rest of literals - for subsumption
 *
 *************/

int map_rest(c, d, s, trp)
struct clause *c;
struct clause *d;
struct context *s;
struct trail **trp;
{
    struct literal *c_lit, *d_lit;
    struct term *c_atom, *d_atom;
    struct trail *t_pos;
    int subsumed, i;

    /* get the first unmarked literal */
    c_lit = c->first_lit;
    i = 0;
    while (c_lit != NULL && Map_array[i] == 1) {
	c_lit = c_lit->next_lit;
	i++;
	}

    if (c_lit == NULL)
	return(1);  /* all lits of c mapped, so c subsumes d */
    else if (c_lit->atom->varnum == ANSWER) {   /* if answer literal, skip it */
	c_atom = c_lit->atom;
	Map_array[i] = 1;      /* mark as mapped */
	subsumed = map_rest(c, d, s, trp);
	Map_array[i] = 0;      /* remove mark */
	return(subsumed);
	}
    else {
	c_atom = c_lit->atom;
	Map_array[i] = 1;      /* mark as mapped */
	d_lit = d->first_lit;
	subsumed = 0;
	while (d_lit != NULL && subsumed == 0) {
	    d_atom = d_lit->atom;
	    t_pos = *trp;  /* save position in trail in case of failure */
	    if (c_lit->sign == d_lit->sign && match(c_atom, s, d_atom, trp))
		if (map_rest(c, d, s, trp))
		    subsumed = 1;
		else {
		    clear_subst_2(*trp, t_pos);
		    *trp = t_pos;
		    }
	    d_lit = d_lit->next_lit;
	    }
	Map_array[i] = 0;      /* remove mark */
	return(subsumed);
	}
}  /* map_rest */

/*************
 *
 *    int anc_subsume(c, d)
 *
 *    We already know that c subsumes d.  Check if d subsumes c and 
 *    ancestors(c) <= ancestors(d).
 *
 *************/

int anc_subsume(c,d)
struct clause *c;
struct clause *d;
{
    if (subsume(d,c))
	return(size_ancestor_set(c) <= size_ancestor_set(d));
    else
	return(1);
}  /* anc_subsume */

/*************
 *
 *    struct clause *forward_subsume(d)
 *
 *    Attempt to find a clause that subsumes d.
 *
 *************/

struct clause *forward_subsume(d)
struct clause *d;
{
    int subsumed;
    struct literal *d_lit;
    struct clause *c;
    struct term *c_atom, *d_atom;
    struct context *s;
    struct trail *tr;
    struct is_tree *is_db;
    struct fsub_pos *pos;
    struct fpa_head **fpa_db;
    struct fpa_tree *ut;
    int d_size, c_size, factor, i;
    struct literal *lit;

    subsumed = 0;
    s = get_context();

    factor = Flags[FACTOR].val;
    if (factor)  /* if factor don't let long clauses subsume short */
	d_size = num_literals(d);

    if (Flags[FOR_SUB_FPA].val == 0) {  /* if `is' indexing */
	d_lit = d->first_lit;

	while (d_lit != NULL && subsumed == 0) {
	    /* Is_pos_lits and Is_neg_lits are global variables */
	    is_db = d_lit->sign ? Is_pos_lits : Is_neg_lits;
	    c_atom = fs_retrieve(d_lit->atom, s, is_db, &pos);
	    while (c_atom != NULL && subsumed == 0) {
		c = c_atom->occ.lit->container;
		c_size = num_literals(c);
		if (c_size > MAX_LITS) {
		    output_stats(stdout, 4);
		    fprintf(stderr, "ABEND, forward_subsume, MAX_LITS.\007\n");
		    fprintf(stdout, "ABEND, forward_subsume, MAX_LITS.\n");
		    exit(1);
		    }
		if (factor == 0 || c_size <= d_size) {
		    for (i = 0, lit = c->first_lit;
                         lit->atom != c_atom; 
                         i++, lit = lit->next_lit);  /* empty body */
	            Map_array[i] = 1;      /* mark as mapped*/
		    tr = NULL;
		    subsumed = map_rest(c, d, s, &tr);
	            Map_array[i] = 0;      /* remove mark */
		    }

		if (subsumed && Flags[ANCESTOR_SUBSUME].val) {
		    renumber_vars(d);
		    subsumed = anc_subsume(c,d);
		    if (!subsumed)
			Stats[CL_NOT_ANC_SUBSUMED]++;
		    }

		if (subsumed == 0)
		    c_atom = fs_retrieve((struct term *) NULL, s, is_db, &pos);
		else {
		    clear_subst_1(tr);
		    canc_fs_pos(pos, s);
		    }
		}
	    d_lit = d_lit->next_lit;
	    }
	}
    else {  /* fpa indexing */

	d_lit = d->first_lit;
	while (d_lit != NULL && subsumed == 0) {
	    fpa_db = (d_lit->sign ? Fpa_pos_lits : Fpa_neg_lits);
	    d_atom = d_lit->atom;
	    ut = build_tree(d_atom, MORE_GEN, Parms[FPA_LITERALS].val, fpa_db);
	    c_atom = next_term(ut, 0);
	    while (c_atom != NULL && subsumed == 0) {
		tr = NULL;
	        c = c_atom->occ.lit->container;
		c_size = num_literals(c);
		if (c_size > MAX_LITS) {
		    output_stats(stdout, 4);
		    fprintf(stderr, "ABEND, forward_subsume, MAX_LITS.\007\n");
		    fprintf(stdout, "ABEND, forward_subsume, MAX_LITS.\n");
		    exit(1);
		    }
		if ((factor == 0 || c_size <= d_size) && 
		    match(c_atom, s, d_atom, &tr)) {
		    
		    for (i = 0, lit = c->first_lit;
			 lit->atom != c_atom; 
			 i++, lit = lit->next_lit);  /* empty body */
		    Map_array[i] = 1;      /* mark as mapped*/
		    subsumed = map_rest(c, d, s, &tr);
		    Map_array[i] = 0;      /* remove mark */
		    clear_subst_1(tr);
		    }

		if (subsumed && Flags[ANCESTOR_SUBSUME].val) {
		    renumber_vars(d);
		    subsumed = anc_subsume(c,d);
		    if (!subsumed)
			Stats[CL_NOT_ANC_SUBSUMED]++;
		    }

		if (subsumed == 0)
		    c_atom = next_term(ut, 0);
		else
		    zap_prop_tree(ut);
		}
	    d_lit = d_lit->next_lit;
	    }    
	}
    free_context(s);
    if (subsumed)
	return(c);
    else
	return(NULL);
}  /* forward_subsume */

/*************
 *
 *    struct clause_ptr *back_subsume(c)
 *
 *    Get the list of clauses subsumed by c.
 *
 *************/

struct clause_ptr *back_subsume(c)
struct clause *c;
{
    int subsumed, c_size, factor, i;
    struct literal *c_lit;
    struct clause *d;
    struct clause_ptr *subsumed_clauses;
    struct term *c_atom, *d_atom;
    struct context *s;
    struct fpa_tree *ut;
    struct trail *tr;

    factor = Flags[FACTOR].val;
    
    c_size = num_literals(c);

    if (c_size > MAX_LITS) {
	output_stats(stdout, 4);
	fprintf(stderr, "ABEND, back_subsume, MAX_LITS.\007\n");
	fprintf(stdout, "ABEND, back_subsume, MAX_LITS.\n");
	exit(1);
	}
    
    s = get_context();
    c_lit = c->first_lit;
    /* get first non-answer literal */
    i = 0;
    while (c_lit != NULL && c_lit->atom->varnum == ANSWER) {
	c_lit = c_lit->next_lit;
	i++;
	}

    if (c_lit == NULL) {
	fprintf(stdout, "\nNOTE: back_subsume called with empty clause.\n");
	return(NULL);
	}

    c_atom = c_lit->atom;
    ut = build_tree(c_atom, INSTANCE, Parms[FPA_LITERALS].val,
			 c_lit->sign ? Fpa_pos_lits : Fpa_neg_lits);
	/* Fpa_pos_lits and Fpa_neg_lits are global variables */

    subsumed_clauses = NULL;
    d_atom = next_term(ut, 0);
    while (d_atom != NULL) {
	d = d_atom->occ.lit->container;
	tr = NULL;
	if (c != d && (factor == 0 || c_size <= num_literals(d))
		   && match(c_atom, s, d_atom, &tr)) {
	    Map_array[i] = 1;  /* mark as mapped */
	    subsumed = map_rest(c, d, s, &tr);
	    Map_array[i] = 0;    /* remove mark */
	    clear_subst_1(tr);
	    if (subsumed)
		insert_clause(d, &subsumed_clauses);
	    }
	d_atom = next_term(ut, 0);
	}
    free_context(s);
    return(subsumed_clauses);
}  /* back_subsume */

/*************
 *
 *    struct clause_ptr *unit_conflict(c)
 *
 *    Search for unit conflict.  Return empty clause if found,
 *    return NULL if not found.
 *
 *    IT IS ASSUMED THAT c IS A UNIT CLAUSE!!
 *
 *************/

struct clause_ptr *unit_conflict(c)
struct clause *c;
{
    struct clause *d, *e;
    struct fpa_tree *ut;
    struct term *f_atom;
    struct literal *lit;
    int go, mp, ec;
    struct context *c1, *c2;
    struct trail *tr;
    struct clause_ptr *cp_return, *cp_prev, *cp_curr;

    c1 = get_context();
    c1->multiplier = 0;
    c2 = get_context();
    c2->multiplier = 1;
    lit = c->first_lit;
    while (lit->atom->varnum == ANSWER)  /* skip answer literals */
	lit = lit->next_lit;
    ut = build_tree(lit->atom, UNIFY, Parms[FPA_LITERALS].val, 
		    lit->sign ? Fpa_neg_lits : Fpa_pos_lits);
    f_atom = next_term(ut, 0);
    go = 1;
    cp_return = cp_prev = NULL;
    while (go && f_atom) {
	tr = NULL;
	d = f_atom->occ.lit->container;
	if (num_literals(d) == 1 && unify(lit->atom, c1, f_atom, c2, &tr)) {

	    e = build_bin_res(lit->atom, c1, f_atom, c2);
	    clear_subst_1(tr);
	    cl_merge(e);  /* answer literals */
	    cp_curr = get_clause_ptr();
	    cp_curr->c = e;
	    if (cp_prev)
		cp_prev->next = cp_curr;
	    else
		cp_return = cp_curr;
	    cp_prev = cp_curr;

	    ec = ++Stats[EMPTY_CLAUSES];
	    mp = Parms[MAX_PROOFS].val;
         
	    if (mp != 0 && ec >= mp)
		/* do not look for more proofs */
		go = 0;
	    }
	if (go)
	    f_atom = next_term(ut, 0);
	else
	    zap_prop_tree(ut);
	}
    free_context(c1);
    free_context(c2);
    return(cp_return);
}  /* unit_conflict */
	
/*************
 *
 *    int pos_clause(c)  --  Is this a positive clause (excluding answer lits) ?
 *
 *************/

int pos_clause(c)
struct clause *c;
{
    struct literal *lit;

    lit = c->first_lit;
    while (lit != NULL)
	if (lit->sign == 0 && lit->atom->varnum != ANSWER)
	    return(0);  /* fail because found negative non-anser literal */
	else
	    lit = lit->next_lit;
    return(1);
}  /* pos_clause */

/*************
 *
 *    int neg_clause(c)  --  Is this a negative clause (excluding answer lits) ?
 *
 *************/

int neg_clause(c)
struct clause *c;
{
    struct literal *lit;

    lit = c->first_lit;
    while (lit != NULL)
	if (lit->sign && lit->atom->varnum != ANSWER)
	    return(0);  /* fail because found positive non-answer literal */
	else
	    lit = lit->next_lit;
    return(1);
}  /* neg_clause */

/*************
 *
 *    int num_literals(c)  --  How many literals (excluding answer literals)?
 *
 *************/

int num_literals(c)
struct clause *c;
{
    int i;
    struct literal *lit;

    i = 0;
    lit = c->first_lit;
    while (lit != NULL) {
	if (lit->atom->varnum != ANSWER)  /* if not answer literal */
	    i++;
	lit = lit->next_lit;
	}
    return(i);
}  /* num_literals */

/*************
 *
 *    int unit_clause(c)  -- Is it a unit clause?
 *
 *************/

int unit_clause(c)
struct clause *c;
{
    return(num_literals(c) == 1);
}  /* unit_clause */

/*************
 *
 *    append_cl(lst, cl)
 *
 *************/

void append_cl(l, c)
struct list *l;
struct clause *c;
{
    c->next_cl = NULL;
    c->prev_cl = l->last_cl;

    if (l->first_cl == NULL)
	l->first_cl = c;
    else
	l->last_cl->next_cl = c;
    l->last_cl = c;
    c->container = l;
}  /* append_cl */

/*************
 *
 *    prepend_cl(lst, cl)
 *
 *************/

void prepend_cl(l, c)
struct list *l;
struct clause *c;
{
    c->prev_cl = NULL;
    c->next_cl = l->first_cl;
    if (l->last_cl == NULL)
	l->last_cl = c;
    else
	l->first_cl->prev_cl = c;
    l->first_cl = c;
    c->container = l;
}  /* prepend_cl */

/*************
 *
 *    insert_before_cl(c, c_new)
 *
 *************/

void insert_before_cl(c, c_new)
struct clause *c;
struct clause *c_new;
{
    struct list *l;

    l = c->container;

    c_new->next_cl = c;
    c_new->prev_cl = c->prev_cl;
    c->prev_cl = c_new;
    if (c_new->prev_cl == NULL)
	l->first_cl = c_new;
    else
	c_new->prev_cl->next_cl = c_new;

    c_new->container = l;

}  /* insert_before_cl */

/*************
 *
 *    insert_after_cl(c, c_new)
 *
 *************/

void insert_after_cl(c, c_new)
struct clause *c;
struct clause *c_new;
{
    struct list *l;

    l = c->container;

    c_new->prev_cl = c;
    c_new->next_cl = c->next_cl;
    c->next_cl = c_new;
    if (c_new->next_cl == NULL)
	l->last_cl = c_new;
    else
	c_new->next_cl->prev_cl = c_new;

    c_new->container = l;

}  /* insert_after_cl */

/*************
 *
 *    rem_from_list(c)
 *
 *************/

void rem_from_list(c)
struct clause *c;
{
    struct clause *p, *n;

    p = c->prev_cl;
    n = c->next_cl;
    if (n == NULL)
	c->container->last_cl = p; 
    else
	n->prev_cl = p;
    if (p == NULL)
	c->container->first_cl = n;
    else
	p->next_cl = n;
    c->container = NULL;
    c->prev_cl = NULL;
    c->next_cl = NULL;
}  /* rem_from_lists */

/*************
 *
 *    insert_clause(clause, *clause_ptr)
 *
 *    If not already there, insert clause into list of clause pointers.
 *
 *************/

void insert_clause(c, cpp)
struct clause *c;
struct clause_ptr **cpp;
{
    struct clause_ptr *curr, *prev, *new;

    curr = *cpp;
    prev = NULL;
    while (curr != NULL && curr->c->id > c->id) {
	prev = curr;
	curr = curr->next;
	}
    if (curr == NULL || curr->c->id != c->id) {
	new = get_clause_ptr();
	new->c = c;
	new->next = curr;
	if (prev != NULL) 
	    prev->next = new;
	else
	    *cpp = new;
	}
}  /* insert_clause */

/*************
 *
 *    int weight_cl(c, wt_index)  --  Weigh a clause.
 *
 *    Also weigh answer lits, which have default weight 0.
 *
 *************/

int weight_cl(c, wt_index)
struct clause *c;
struct is_tree *wt_index;
{
    int w, neg_weight;
    struct literal *lit;

    neg_weight = Parms[NEG_WEIGHT].val;
    w = 0;
    lit = c->first_lit;
    while (lit != NULL) {
	w += weight(lit->atom, wt_index);
	if (lit->atom->varnum != ANSWER && lit->sign == 0)
	    w += neg_weight;
	lit = lit->next_lit;
	}
    return(w);
}  /* weight_cl */

/*************
 *
 *    hide_clause(c) --  c must be integrated
 *
 *    Clauses can be hidden instead of deallocated so that they can
 *    be printed later on (mostly so that a child can know its parent).
 *
 *************/

void hide_clause(c)
struct clause *c;
{
    c->next_cl = Hidden_clauses;
    Hidden_clauses = c;
}  /* hide_clause */

/*************
 *
 *    del_hidden_clauses() -- deallocate all hidden clauses
 *
 *************/

void del_hidden_clauses()
{
    struct clause *c;

    while (Hidden_clauses != NULL) {
	c = Hidden_clauses;
	Hidden_clauses = Hidden_clauses->next_cl;
	cl_del_int(c);
	}
}  /* del_hidden_clauses */

/*************
 *
 *    struct clause *cl_copy(c)
 *
 *    Do not copy the list of parents.
 *
 *************/

struct clause *cl_copy(c)
struct clause *c;
{
    struct clause *d;
    struct literal *l, *l1, *l2;

    d = get_clause();
    d->type = c->type;
    l = c->first_lit;
    l2 = NULL;
    while (l != NULL) {
	l1 = get_literal();
	l1->target = l->target;
	l1->container = d;
	l1->sign = l->sign;
	l1->atom = copy_term(l->atom);
	remove_var_syms(l1->atom);  /* nullify variable symbols (if present) */
	l1->atom->occ.lit = l1;
	if (l2 == NULL)
	    d->first_lit = l1;
	else
	    l2->next_lit = l1;
	l2 = l1;
	l = l->next_lit;
	}
    return(d);
}  /* cl_copy */

/*************
 *
 *    remove_var_syms(t)
 *
 *    Variable terms normally do not have sym_nums.  This
 *    routine removes any that are present.
 *
 *************/

void remove_var_syms(t)
struct term *t;
{
    struct rel *r;

    if (t->type == VARIABLE)
        t->sym_num = 0;
    else if (t->type == COMPLEX)
	for (r = t->farg; r != NULL; r = r->narg)
	    remove_var_syms(r->argval);
}  /* remove_var_syms */

/*************
 *
 *    cl_insert_tab(c)
 *
 *************/

void cl_insert_tab(c)
struct clause *c;
{
    struct clause_ptr *cp1, *cp2, *cp3;
    int hashval, id;

    id = c->id;
    hashval = id % CLAUSE_TAB_SIZE;
    cp1 = Clause_tab[hashval];
    cp2 = NULL;

#ifdef ROO
    cp2 = cp1;
    cp1 = cp1->next;  /* skip dummy node */
#endif

    /* keep the chains ordered--increasing id */

    while (cp1 != NULL && cp1->c->id < id) {
	cp2 = cp1;
	cp1 = cp1->next;
	}
    if (cp1 != NULL && cp1->c->id == id) {
	output_stats(stdout, 4);
	fprintf(stderr, "ABEND, cl_insert_tab, clause already there.\007\n");
	fprintf(stdout, "ABEND, cl_insert_tab, clause already there: ");
	print_clause(stdout, c);
	exit(1);
	}
    else {
	cp3 = get_clause_ptr();
	cp3->c = c;
	cp3->next = cp1;
	if (cp2 == NULL) 
	    Clause_tab[hashval] = cp3;
	else
	    cp2->next = cp3;
	}
}  /* cl_insert_tab */

/*************
 *
 *    cl_delete_tab(c)
 *
 *************/

void cl_delete_tab(c)
struct clause *c;
{
    struct clause_ptr *cp1, *cp2;
    int hashval, id;

    id = c->id;
    hashval = id % CLAUSE_TAB_SIZE;
    cp1 = Clause_tab[hashval];
    cp2 = NULL;

#ifdef ROO
    cp2 = cp1;
    cp1 = cp1->next;  /* skip dummy node */
#endif

    /* chains are ordered--increasing id */

    while (cp1 != NULL && cp1->c->id < id) {
	cp2 = cp1;
	cp1 = cp1->next;
	}
    if (cp1 == NULL || cp1->c->id != id) {
	output_stats(stdout, 4);
	fprintf(stderr, "ABEND, cl_delete_tab, clause not found.\007\n");
	fprintf(stdout, "ABEND, cl_delete_tab, clause not found: ");
	print_clause(stdout, c);
	exit(1);
	}
    else {
	if (cp2 == NULL) 
	    Clause_tab[hashval] = cp1->next;
	else
	    cp2->next = cp1->next;
	free_clause_ptr(cp1);
	}
}  /* cl_delete_tab */

/*************
 *
 *    struct clause *cl_find(id)
 *
 *************/

struct clause *cl_find(id)
int id;
{
    struct clause_ptr *cp1;
    int hashval;

    hashval = id % CLAUSE_TAB_SIZE;
    cp1 = Clause_tab[hashval];

#ifdef ROO
    cp1 = cp1->next;  /* skip dummy node */
#endif

    /* lists are ordered--increasing id */

    while (cp1 != NULL && cp1->c->id < id)
	cp1 = cp1->next;
    if (cp1 == NULL || cp1->c->id != id)
	return(NULL);
    else
	return(cp1->c);
}  /* cl_find */

/*************
 *
 *    sort_lits(c)  --  sort literals
 *
 *************/

void sort_lits(c)
struct clause *c;
{
    struct literal *slits, *prev, *curr, *next, *inst;
    int sign, sym_num;

    /* this is an insertion sort */

    /* 1. negative before positive; */
    /* 2. non-answer literals before answer literals; */
    /* 3. lower symbol numbers first */

    slits = c->first_lit;
    if (slits == NULL)
	return;  /* c is empty clause */
    else {
	inst = slits->next_lit;
	slits->next_lit = NULL;
	while(inst != NULL) {
	    prev = NULL;
	    sign = inst->sign;
	    sym_num = inst->atom->sym_num;
	    if (inst->atom->varnum == ANSWER) {
		sym_num += 5000;
		}
	    curr = slits;
	    while (curr != NULL && 
		   (sign > curr->sign || 
		    (sign == curr->sign && 
		     sym_num >= curr->atom->sym_num))) {
		prev = curr;
		curr = curr->next_lit;
		}
	    next = inst->next_lit;
	    inst->next_lit = curr;
	    if (prev == NULL)
		slits = inst;
	    else
		prev->next_lit = inst;
	    inst = next;
	    }
	c->first_lit = slits;
	}
}  /* sort lits */

/*************
 *
 *    all_cont_cl(t, cpp) - insert containing clauses of t into *cpp
 *
 *************/

void all_cont_cl(t, cpp)
struct term *t;
struct clause_ptr **cpp;
{
    struct rel *r;
    struct clause *c;
    struct list *l;

    if (t->type != VARIABLE && t->varnum != 0) {  /* atom */
	c = t->occ.lit->container;
	l = c->container;
	if (l == Usable || l == Sos || l == Demodulators)
	    insert_clause(c, cpp);
	}
    else {  /* term */
	r = t->occ.rel;
	while (r != NULL) {
	    all_cont_cl(r->argof, cpp);
	    r = r->nocc;
	    }
	}
}  /* all_cont_cl */

/*************
 *
 *    zap_cl_list(lst)
 *
 *************/

void zap_cl_list(lst)
struct list *lst;
{
    struct clause *c1, *c2;

    c1 = lst->first_cl;
    while (c1 != NULL) {
	c2 = c1;
	c1 = c1->next_cl;
	cl_del_non(c2);
	}
    free_list(lst);
}  /* zap_cl_list */

/*************
 *
 *    mark_literal(lit)
 *
 *    Atoms have varnum > 0.  This routine inserts the appropriate code.
 *
 *************/

void mark_literal(lit)
struct literal *lit;
{
    char *name;
    struct term *a;

    a = lit->atom;

    /* atoms have varnum > 0 */
    name = sn_to_str(a->sym_num);

    if (initial_str("$ANS", name) || initial_str("$Ans", name) ||
				     initial_str("$ans", name))
	a->varnum = ANSWER;  /* answer literal */

    else if (sn_to_arity(a->sym_num) == 2 &&
			      (initial_str("EQ", name) ||
			       initial_str("Eq", name) ||
			       initial_str("eq", name) ||
			       a->sym_num == Eq_sym_num) )
	if (lit->sign)
	    a->varnum = POS_EQ;  /* positive equality */
	else
	    a->varnum = NEG_EQ;  /* negative equality */

    else if (a->sym_num == Conditional_demodulator_sym_num)
	a->varnum = CONDITIONAL_DEMOD;

    else if (name[0] == '$')
	a->varnum = EVALUABLE;  /* $ID, $LE, $AND, ... */

    else
	a->varnum = NORM_ATOM;  /* normal atom */

}  /* mark_literal */

/*************
 *
 *    get_ancestors(c, cpp)
 *
 *************/

void get_ancestors(c, cpp)
struct clause *c;
struct clause_ptr **cpp;
{
    struct clause_ptr *cp1, *cp2, *cp3;
    struct int_ptr *ip;
    struct clause *d;

    cp1 = *cpp;
    cp3 = NULL;
    while (cp1 != NULL && cp1->c->id < c->id) {
	cp3 = cp1;
	cp1 = cp1->next;
	}
    if (cp1 == NULL || cp1->c->id > c->id) {
	cp2 = get_clause_ptr();
	cp2->c = c;
	if (cp3 == NULL) {
	    cp2->next = *cpp;
	    *cpp = cp2;
	    }
	else {
	    cp2->next = cp3->next;
	    cp3->next = cp2;
	    }
	ip = c->parents;
	while (ip != NULL) {
	    if (ip->i >= 0) {  /* < 0 means it's a code for an inference rule */
		d = cl_find(ip->i);
		if (d == NULL)
		    printf("WARNING, clause %d has been deleted, proof is incomplete.\n", ip->i);
		else
		    get_ancestors(d, cpp);
		}
	    ip = ip->next;
	    }
	}
}  /* get_ancestors */

/*************
 *
 *    int proof_level(c)
 *
 *************/

int proof_level(c)
struct clause *c;
{
    struct int_ptr *ip;
    int id, max, l;
    struct clause *parent;
    
    if (c->parents == NULL)
        return(0);
    else {
	max = 0;
	for (ip = c->parents; ip != NULL; ip = ip->next) {
	    id = ip->i;
	    if (id > 0) {
		parent = cl_find(id);
		if (parent == NULL)
		    max = MAX_INT / 2;
		else {
		    l = proof_level(parent);
		    max = (l > max ? l : max);
		    }
		}
	    }
	return(max + 1);
	}
}  /* proof_level */

/*************
 *
 *    int renumber_vars(c)
 *
 *        Renumber the variables of a clause, starting with 0.  `c' must
 *    be nonintegrated.  return(0) if more than MAXVARS distinct variables.
 *
 *************/

int renumber_vars(c)
struct clause *c;
{
    struct literal *lit;
    int varnums[MAX_VARS];
    int i, ok;
    
    CLOCK_START(RENUMBER_TIME)

    ok = 1;
    for (i = 0; i < MAX_VARS; i++)
	varnums[i] = -1;
    
    lit = c->first_lit;
    while (lit != NULL) {
	if (renum_vars_term(lit->atom, varnums) == 0)
	    ok = 0;
	lit = lit->next_lit;
	}

    CLOCK_STOP(RENUMBER_TIME)

    return(ok);

}  /* renumber_vars */

/*************
 *
 *    int renum_vars_term(term, varnums) -- called from renumber_vars.
 *
 *************/

int renum_vars_term(t, varnums)
struct term *t;
int varnums[];
{
    struct rel *r;
    int i, ok;
    
    if (t->type == NAME)
	return(1);
    else if (t->type == COMPLEX) {
	ok = 1;
	r = t->farg;
	while (r != NULL) {
	    if (renum_vars_term(r->argval, varnums) == 0)
		ok = 0;
	    r = r->narg;
	    }
	return(ok);
	}
    else {
	i = 0;
	while (i < MAX_VARS && varnums[i] != -1 && varnums[i] != t->varnum)
	i++;
	if (i == MAX_VARS)
	    return(0);
	else {
	    if (varnums[i] == -1) {
		varnums[i] = t->varnum;
		t->varnum = i;
		}
	    else
		t->varnum = i;
	    return(1);
	    }
	}
}  /* renum_vars_term */

/*************
 *
 *    clear_var_names(t) -- set sym_num field of all variables to NULL
 *
 *************/

void clear_var_names(t)
     struct term *t;
{
    struct rel *r;
    
    if (t->type == VARIABLE)
	t->sym_num = 0;
    else {
	for (r = t->farg; r != NULL; r = r->narg)
	    clear_var_names(r->argval);
	}
}  /* clear_var_names */

/*************
 *
 *    cl_clear_vars(c)
 *
 *************/

void cl_clear_vars(c)
struct clause *c;
{
    struct literal *lit;

    for (lit = c->first_lit; lit != NULL; lit = lit->next_lit)
	clear_var_names(lit->atom);
}

/*************
 *
 *    void distinct_vars_rec(t, a)
 *
 *************/

void distinct_vars_rec(t, a)
struct term *t;
int a[];
{
    struct rel *r;
    
    if (t->type == VARIABLE)
	a[t->varnum]++;
    else if (t->type == COMPLEX)
	for (r = t->farg; r; r = r->narg)
	    distinct_vars_rec(r->argval, a);
}  /* distinct_vars_rec */

/*************
 *
 *    int distinct_vars(c)
 *
 *************/

int distinct_vars(c)
struct clause *c;
{
    struct literal *lit;
    int a[MAX_VARS], i, j;

    for (i = 0; i < MAX_VARS; i++)
	a[i] = 0;

    for (lit = c->first_lit; lit; lit = lit->next_lit)
	distinct_vars_rec(lit->atom, a);

    for (i = j = 0; i < MAX_VARS; i++)
	if (a[i] != 0)
	    j++;

    return(j);

}  /* distinct_vars */

 /*************
 *
 *    struct clause *find_first_cl(l)
 *
 *************/

struct clause *find_first_cl(l)
struct list *l;
{
    struct clause *c;

    if (l->first_cl == NULL)
	return(NULL);
    else {
	c = l->first_cl;
	return(c);
	}
}  /* find_first_cl */

/*************
 *
 *    struct clause *find_last_cl(l)
 *
 *************/

struct clause *find_last_cl(l)
struct list *l;
{
    struct clause *c;

    if (l->last_cl == NULL)
	return(NULL);
    else {
	c = l->last_cl;
	return(c);
	}
}  /* find_last_cl */

/*************
 *
 *    struct clause *find_random_cl(l)
 *
 *************/

struct clause *find_random_cl(l)
struct list *l;
{
    struct clause *c;
    int i, j;

    if (l->first_cl == NULL)
	return(NULL);
    else {
	j = (rand() % Stats[SOS_SIZE]) + 1;
        c = l->first_cl;
	i = 1;
        while (i < j && c) {
	    c = c->next_cl;
	    i++;
	    }
	if (!c) {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, find_random_cl, sos bad.\007\n");
	    fprintf(stdout, "ABEND, find_random_cl, sos bad.");
	    exit(1);
	    }
	return(c);
	}
}  /* find_random_cl */

/*************
 *
 *    struct clause *find_lightest_cl(l)
 *
 *    Input sos clauses might have weight field set to -MAX_INT so that
 *    they are returned first (in order).
 *
 *************/

struct clause *find_lightest_cl(l)
struct list *l;
{
    struct clause *c, *cm;
    int w, wm;

    if (l->first_cl == NULL)
	return(NULL);
    else {
	cm = l->first_cl;
	wm = cm->weight;
	c = cm->next_cl;
	while (c != NULL) {
	    w = c->weight;
	    if (w < wm) {
		wm = w;
		cm = c;
		}
	    c = c->next_cl;
	    }

	return(cm);
	}
}  /* find_lightest_cl */

/*************
 *
 *    struct clause *find_given_clause()
 *
 *************/

struct clause *find_given_clause()
{
    struct clause *giv_cl;

    if (Parms[PICK_GIVEN_RATIO].val != -1) {
	if (Stats[CL_GIVEN] % (Parms[PICK_GIVEN_RATIO].val + 1) == 0)
	    giv_cl = find_first_cl(Sos);
        else
	    giv_cl = find_lightest_cl(Sos);
	}
    else if (Parms[RANDOM_RATIO].val != -1) {
	if (Stats[CL_GIVEN] % (Parms[RANDOM_RATIO].val + 1) == 0)
	    giv_cl = find_random_cl(Sos);
        else
	    giv_cl = find_lightest_cl(Sos);
	}
    else if (Flags[SOS_QUEUE].val)
	giv_cl = find_first_cl(Sos);
    else if (Flags[SOS_STACK].val)
	giv_cl = find_last_cl(Sos);
    else
	giv_cl = find_lightest_cl(Sos);

    return(giv_cl);
}  /* find_given_clause */

/*************
 *
 *    struct clause *extract_given_clause()
 *
 *************/

struct clause *extract_given_clause()
{
    struct clause *giv_cl;

    giv_cl = find_given_clause();
    if (giv_cl) {
	Stats[SOS_SIZE]--;
	rem_from_list(giv_cl);
	}
    return(giv_cl);
}  /* extract_given_clause */

/*************
 *
 *    int unit_del(c)  --  unit deletion
 *
 *    Return 1 if any deletions occur.
 *
 *************/

int unit_del(c)
struct clause *c;
{
    struct clause *d;
    struct literal *prev, *curr, *next, *answers, *l1, *l2;
    struct term *d_atom;
    struct context *s;
    struct is_tree *is_db;
    struct fsub_pos *pos;
    struct int_ptr *ip0, *ip, *lp;
    int deleted, return_val;

    return_val = 0;
    s = get_context();

    /* first get last parent */
    lp = c->parents;
    if (lp != NULL)
	while (lp->next != NULL)
	    lp = lp->next;
    
    ip0 = lp;  /* save position to insert "ud" if any deleted */

    answers = NULL;
    prev = NULL;
    next = c->first_lit;
    while (next != NULL) {
	curr = next;
	next = next->next_lit;
	is_db = curr->sign ? Is_neg_lits : Is_pos_lits;
	d_atom = fs_retrieve(curr->atom, s, is_db, &pos);
	deleted = 0;
	while (d_atom != NULL && deleted == 0) {
	    d = d_atom->occ.lit->container;
	    if (num_literals(d) == 1) {
		return_val = 1;
		if (prev == NULL)
		    c->first_lit = next;
		else
		    prev->next_lit = next;
		ip = get_int_ptr();  /* append to history */
		ip->i = d->id;
		if (lp == NULL)
		    c->parents = ip;
		else
		    lp->next = ip;
		lp = ip;

		l2 = d->first_lit;  /* now append any answer literals */
		while (l2 != NULL) {
		    if (l2->atom->varnum == ANSWER) {
			l1 = get_literal();
			l1->container = c;
			l1->sign = l2->sign;
			l1->atom = apply(l2->atom, s);
			l1->atom->varnum = ANSWER;
			l1->atom->occ.lit = l1;
			l1->next_lit = answers;
			answers = l1;
			}
		    l2 = l2->next_lit;
		    }

		curr->atom->occ.lit = NULL;  /* so zap_term won't complain */
		zap_term(curr->atom);
		free_literal(curr);

		canc_fs_pos(pos, s);
		Stats[UNIT_DELETES]++;
		deleted = 1;
		}
	    else
		d_atom = fs_retrieve((struct term *) NULL, s, is_db, &pos);
	    }
	if (deleted == 0)
	    prev = curr;
	}
    if (prev == NULL)
	c->first_lit = answers;
    else
	prev->next_lit = answers;
    if (lp != ip0) {  /* at least one deletion occurred */
	ip = get_int_ptr();
	ip->i = UNIT_DEL_RULE;
	if (ip0 == NULL) {
	    ip->next = c->parents;
	    c->parents = ip;
	    }
	else {
	    ip->next = ip0->next;
	    ip0->next = ip;
	    }
	}
    free_context(s);
    return(return_val);
}  /* unit_del */

/*************
 *
 *    int n_resolution_check(c)
 *
 *    Return 1 if c satisfies the n-resolution restriction.
 *    c satisfies the restriction iff 
 *         1. c was not inferred by binary resolution, or
 *         2. one parent is a positive unit, or
 *         3. one parent is a negative clause.
 *
 *************/

int n_resolution_check(c)
struct clause *c;
{
    struct int_ptr *ip;
    struct clause *parent;

    if (c->parents == NULL || c->parents->i != BINARY_RES_RULE)
	return(1);  /* not generated by binary resolution */
    else {
	ip = c->parents->next;
        parent = cl_find(ip->i);
        if (pos_clause(parent) && num_literals(parent) == 1)
	    return(1);
	else if (neg_clause(parent))
	    return(1);
	else {
	    ip = ip->next;
	    parent = cl_find(ip->i);
	    if (pos_clause(parent) && num_literals(parent) == 1)
		return(1);
	    else if (neg_clause(parent))
		return(1);
	    else
		return(0);
	    }
	}
}  /* n_resolution_check  */

