;;|=========================================================================|
;;|                           COPYRIGHT NOTICE                              |
;;|                                                                         |
;;|                   Copyright 1992, Mark Tarver                           |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun print-answer
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '$ (car fp1)))
              (format t
                      "~A = ~A~%"
                      (create-external-variable (cadr fp1))
                      (deref (list '$ (cadr fp1)))))
             (t (raise "code 30: non-variable used in answer clause"))))

(defun value-of
       (fp1)
       (cond ((var fp1) (create-external-variables (deref (list '$ fp1))))
             (t fp1)))

(defun compile-horn-clauses
       (fp1)
       (cond ((null fp1) nil)
             ((eval '*outfile*)
              (pprint (horn-function fp1)
                      stream)
              (terpri stream))
             (t (eval (horn-function fp1)))))

(defun solve
       (fp1)
       (prog (*probvars*)
             (initstacks)
             (setq *probvars* (extrvars fp1))
             (store-problem-variables *probvars*)
             (return (popstack (ins$ *probvars* fp1)))))

(defun initstacks
       nil
       (setq *refinements* 0)
       (setq *varcounter* 0)
       (setq *trailstack* (list nil))
       (clrhash *binding-array*))

(defun ins$
       (fp1 fp2)
       (cond ((null fp1) fp2)
             ((consp fp1)
              (ins$ (cdr fp1)
                    (subst (list '$ (car fp1))
                           (car fp1)
                           fp2)))
             (t (raise "code 13: No Patterns have Fired in ins$"))))

(defun store-problem-variables
       (fp1)
       (cond ((null fp1) nil)
             ((consp fp1)
              (hashbind (car fp1) '-0-)
              (store-problem-variables (cdr fp1)))
             (t
              (raise
                "code 13: No Patterns have Fired in store-problem-variables"))))

(defun hashbind
       (fp1 fp2)
       (setf (gethash fp1 *binding-array*)
             fp2))

(defun imm-binding
       (fp1)
       (gethash (cadr fp1) *binding-array* '-0-))

(defun compile-horn-clauses-function
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1)))
              (compile (caar fp1)))
             (t
              (raise
                "code 13: No Patterns have Fired in compile-horn-clauses-function"))))

(defun horn-function
       (fp1)
       (compress (horn-clause-code (reverse fp1)
                                   (horn-function-shell fp1))))

(defun horn-function-shell
       (fp1)
       (cond ((null fp1) nil)
             ((consp fp1) (hfs1 (car fp1)))
             (t
              (raise "code 13: No Patterns have Fired in horn-function-shell"))))

(defun hfs1
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1)))
              (list 'defun
                    (caar fp1)
                    (append (fparams-of (find-log-arity (cdar fp1)))
                            (list 'continuation))
                    (list 'cond
                          (list (list 'backtrack)))))
             (t (raise "code 13: No Patterns have Fired in hfs1"))))

(defun horn-clause-code
       (fp1 fp2)
       (cond ((null fp1) fp2)
             ((and (consp fp1)
                   (consp fp2)
                   (= (length fp2) 4)
                   (eq 'defun (car fp2))
                   (consp (caddr (cdr fp2)))
                   (eq 'cond
                       (caadr (cddr fp2))))
              (horn-clause-code (cdr fp1)
                                (list 'defun
                                      (cadr fp2)
                                      (caddr fp2)
                                      (cons 'cond
                                            (cons (logiccode-of (car fp1))
                                                  (cdadr (cddr fp2)))))))
             (t (raise "code 13: No Patterns have Fired in horn-clause-code"))))

(defun add-backtrack
       nil
       (list (list 'backtrack)))

(defun backtrack
       nil
       (mapcar 'deallocate-values
               (pop *trailstack*))
       nil)

(defun deallocate-values
       (fp1)
       (remhash fp1 *binding-array*))

(defun iso-ids
       (fp1)
       (remove-duplicates (mapcar 'iso-id fp1)))

(defun iso-id
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1)))
              (caar fp1))
             (t (raise "code 13: No Patterns have Fired in iso-id"))))

(defun find-log-arity
       (fp1)
       (length (remove-if 'guard fp1)))

(defun logiccode-of
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq '-> (cadr fp1)))
              (cut-continuation
                (cons 'and
                      (ram-to-lisp (ram (list (car fp1)
                                              '<-
                                              (cddr fp1)))))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq '<- (cadr fp1)))
              (list
                (cons 'and
                      (ram-to-lisp (ram (list (car fp1)
                                              '<-
                                              (cddr fp1)))))))
             (t (raise "code 13: No Patterns have Fired in logiccode-of"))))

(defun cut-continuation
       (fp1)
       (cond ((and (consp fp1)
                   (eq 'and (car fp1)))
              (list (cons 'and
                          (butlast (cdr fp1)))
                    (list 'or
                          (car (last (cdr fp1)))
                          (list 'backtrack))))
             (t (raise "code 13: No Patterns have Fired in cut-continuation"))))

(defun ram
       (fp1)
       (ram1
         (provide-standard-vars fp1
                                fp1
                                '(var#1? var#2? var#3? var#4? var#5? var#6?
                                  var#7? var#8? var#9? var#10? var#11? var#12?
                                  var#13? var#14? var#15? var#16? var#17?
                                  var#18? var#19? var#20?))))

(defun provide-standard-vars
       (fp1 fp2 fp3)
       (cond ((null fp1) fp2)
             ((null fp3) fp2)
             ((and (consp fp1)
                   (consp (car fp1)))
              (provide-standard-vars (append (car fp1)
                                             (cdr fp1))
                                     fp2
                                     fp3))
             ((and (consp fp1)
                   (var (car fp1))
                   (consp fp3))
              (provide-standard-vars (rremove (car fp1)
                                              (cdr fp1))
                                     (subst (car fp3)
                                            (car fp1)
                                            fp2)
                                     (cdr fp3)))
             ((consp fp1) (provide-standard-vars (cdr fp1) fp2 fp3))
             (t
              (raise "code 13: No Patterns have Fired in provide-standard-vars"))))

(defun ram1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (consp (car fp1))
                   (deflog+-on))
              (append (findstvars (cdar fp1)
                                  (caddr fp1))
                      (um (cdar fp1))
                      (maintain-logic-stacks)
                      (rampush
                        (append (caddr fp1)
                                (list (list '-frame-
                                            (car fp1)))))))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (consp (car fp1)))
              (append (findstvars (cdar fp1)
                                  (caddr fp1))
                      (um (cdar fp1))
                      (maintain-logic-stacks)
                      (rampush (caddr fp1))))
             (t (raise "code 13: No Patterns have Fired in ram1"))))

(defun maintain-logic-stacks
       nil
       (list (list 'upstcks)))

(defun findstvars
       (fp1 fp2)
       (list (list 'sta
                   (extrvars (append fp1 fp2)))))

(defun extrvars
       (fp1)
       (remove-duplicates (extrvars1 fp1)))

(defun extrvars1
       (fp1)
       (cond ((null fp1) nil)
             ((consp fp1) (mapcan 'extrvars1 fp1))
             ((var fp1) (list fp1))
             (t nil)))

(defun um
       (fp1)
       (um1 nil fp1 (list 1) 'no))

(defun um1
       (fp1 fp2 fp3 fp4)
       (cond ((null fp2) nil)
             ((and (consp fp2)
                   (= (length fp2) 2)
                   (consp fp3)
                   (bar (car fp2)))
              (if (not (anon (cadr fp2)))
                  (list
                    (addunifinstr (list (car fp2)
                                        (cadr fp2))
                                  fp1
                                  fp3
                                  fp4))
                  nil))
             ((and (consp fp2)
                   (consp fp3)
                   (anon (car fp2)))
              (um1 fp1
                   (cdr fp2)
                   (cons (1+ (car fp3))
                         (cdr fp3))
                   fp4))
             ((and (consp fp2)
                   (consp fp3)
                   (not (var (car fp2)))
                   (atom (car fp2)))
              (cons (addunifinstr (car fp2) fp1 fp3 fp4)
                    (um1 (cons (car fp2) fp1)
                         (cdr fp2)
                         (cons (1+ (car fp3))
                               (cdr fp3))
                         'yes)))
             ((and (consp fp2)
                   (consp fp3)
                   (atom (car fp2)))
              (cons (addunifinstr (car fp2) fp1 fp3 fp4)
                    (um1 (cons (car fp2) fp1)
                         (cdr fp2)
                         (cons (1+ (car fp3))
                               (cdr fp3))
                         fp4)))
             ((and (consp fp2)
                   (guard (car fp2)))
              (cons (addunifinstr (car fp2) fp1 fp3 fp4)
                    (um1 fp1 (cdr fp2) fp3 fp4)))
             ((and (consp fp2)
                   (consp (car fp2))
                   (consp fp3))
              (cons
                (cons
                  (addunifinstr (car fp2)
                                fp1
                                fp3
                                (decide-deref (car fp2) fp1 fp4))
                  (um1 fp1 (car fp2) (cons 1 fp3) fp4))
                (um1 (append (flatten (car fp2))
                             fp1)
                     (cdr fp2)
                     (cons (1+ (car fp3))
                           (cdr fp3))
                     'yes)))
             (t (raise "code 13: No Patterns have Fired in um1"))))

(defun decide-deref
       (fp1 fp2 fp3)
       (cond ((eq 'yes fp3) 'yes)
             ((and (consp fp2)
                   (any-occurs fp1 fp2))
              'yes)
             (t 'no)))

(defun any-occurs
       (fp1 fp2)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (occurs (car fp1) fp2))
              t)
             ((consp fp1) (any-occurs (cdr fp1) fp2))
             (t (raise "code 13: No Patterns have Fired in any-occurs"))))

(defun addunifinstr
       (fp1 fp2 fp3 fp4)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (bar (car fp1)))
              (if (occurs (cadr fp1) fp2)
                  (list 'btv (cadr fp1) fp3 fp4)
                  (list 'ftv (cadr fp1) fp3 fp4)))
             ((null fp1) (list 'name nil fp3 fp4))
             ((guard fp1) (list 'guard fp1))
             ((consp fp1) (list 'cll fp1 fp3 fp4))
             ((and (var fp1) (occurs fp1 fp2)) (list 'bv fp1 fp3 fp4))
             ((var fp1) (list 'fv fp1 fp3 fp4))
             ((atom fp1) (list 'name fp1 fp3 fp4))
             (t (raise "code 13: No Patterns have Fired in addunifinstr"))))

(defun rampush
       (fp1)
       (list (list 'push fp1)))

(defun ram-to-lisp
       (fp1)
       (cond ((eval '*pattern*) (ram-to-lisp1 fp1))
             (t (ram-to-lisp2 fp1))))

(defun ram-to-lisp1
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (list-structure (car fp1)))
              (append (consp-check (car fp1))
                      (ram-to-lisp1 (append (cdar fp1)
                                            (cdr fp1)))))
             ((consp fp1)
              (cons (cnvram- (car fp1))
                    (ram-to-lisp1 (cdr fp1))))
             (t (raise "code 13: No Patterns have Fired in ram-to-lisp1"))))

(defun consp-check
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 4)
                   (member '\| (cadar fp1)))
              (list (list 'consp
                          (tterm (caddr (car fp1))))
                    (list '>
                          (list 'length
                                (tterm (caddr (car fp1))))
                          (- (length (cadar fp1))
                             3))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 4))
              (list (list 'consp
                          (tterm (caddr (car fp1))))
                    (list '=
                          (list 'length
                                (tterm (caddr (car fp1))))
                          (length (cadar fp1)))))
             (t (raise "code 13: No Patterns have Fired in consp-check"))))

(defun ram-to-lisp2
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (list-structure (car fp1)))
              (cons
                (insert-or+and (first (car fp1))
                               (ram-to-lisp2 (car fp1)))
                (ram-to-lisp2 (cdr fp1))))
             ((consp fp1)
              (cons (cnvram+ (car fp1))
                    (ram-to-lisp2 (cdr fp1))))
             (t (raise "code 13: No Patterns have Fired in ram-to-lisp2"))))

(defun list-structure
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1)))
              t)
             (t nil)))

(defun insert-or+and
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (= (length fp1) 4)
                   (consp fp2)
                   (member '\| (cadr fp1)))
              (cons 'and
                    (cons (list 'consp
                                (tterm (caddr fp1)))
                          (cons
                            (list '>
                                  (list 'length
                                        (tterm (caddr fp1)))
                                  (- (length (cadr fp1))
                                     3))
                            (cdr fp2)))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (consp fp2))
              (list 'or
                    (car fp2)
                    (cons 'and
                          (cons (list 'consp '*tempreg*)
                                (cons
                                  (list '=
                                        (list 'length
                                              '*tempreg*)
                                        (length (cadr fp1)))
                                  (cdr fp2))))))
             (t (raise "code 13: No Patterns have Fired in insert-or+and"))))

(defun cnvram+
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'sta (car fp1))
                   (null (cadr fp1)))
              (list 'rename nil))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'sta (car fp1)))
              (list 'rename
                    (list 'quote (cadr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'cll (car fp1))
                   (eq 'no (caddr (cdr fp1))))
              (list 'qtrybind
                    (list 'setq
                          '*tempreg*
                          (tterm (caddr fp1)))
                    (pushterms (cadr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'cll (car fp1))
                   (eq 'yes (caddr (cdr fp1))))
              (list 'trybind
                    (list 'setq
                          '*tempreg*
                          (list 'lderef
                                (tterm (caddr fp1))))
                    (callterms (cadr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'guard (car fp1)))
              (callterms (cadr fp1)))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'name (car fp1))
                   (eq 'no (caddr (cdr fp1)))
                   (stringp (cadr fp1)))
              (list 'or
                    (list 'equal
                          (cadr fp1)
                          (list 'setq
                                '*tempreg*
                                (tterm (caddr fp1))))
                    (list 'qtrybind
                          '*tempreg*
                          (cadr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'name (car fp1))
                   (eq 'no (caddr (cdr fp1))))
              (list 'or
                    (list 'eq
                          (list 'quote (cadr fp1))
                          (list 'setq
                                '*tempreg*
                                (tterm (caddr fp1))))
                    (list 'qtrybind
                          '*tempreg*
                          (list 'quote (cadr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'name (car fp1))
                   (eq 'yes (caddr (cdr fp1)))
                   (stringp (cadr fp1)))
              (list 'or
                    (list 'equal
                          (cadr fp1)
                          (list 'setq
                                '*tempreg*
                                (list 'lderef
                                      (tterm (caddr fp1)))))
                    (list 'qtrybind
                          '*tempreg*
                          (cadr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'name (car fp1))
                   (eq 'yes (caddr (cdr fp1))))
              (list 'or
                    (list 'eq
                          (list 'quote (cadr fp1))
                          (list 'setq
                                '*tempreg*
                                (list 'lderef
                                      (tterm (caddr fp1)))))
                    (list 'qtrybind
                          '*tempreg*
                          (list 'quote (cadr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'fv (car fp1)))
              (list 'bind
                    (cadr fp1)
                    (tterm (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'ftv (car fp1)))
              (list 'bind
                    (cadr fp1)
                    (ttail (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'btv (car fp1)))
              (list 'equal
                    (list 'lderef (cadr fp1))
                    (ttail (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'btv (car fp1)))
              (list 'equal
                    (list 'lderef (cadr fp1))
                    (list 'lderef
                          (ttail (caddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'bv (car fp1))
                   (eq 'no (caddr (cdr fp1))))
              (list 'unify
                    (list 'lderef (cadr fp1))
                    (tterm (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'bv (car fp1))
                   (eq 'yes (caddr (cdr fp1))))
              (list 'unify
                    (list 'lderef (cadr fp1))
                    (list 'lderef
                          (tterm (caddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1))
                   (deflog+-on))
              (list 'loop-and-pop
                    (list 'append
                          (pushterms (cadr fp1))
                          'continuation)))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1))
                   (null (cadr fp1)))
              (list 'popstack 'continuation))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1))
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           1)))
              (append (nextcall (caadr fp1))
                      (list 'continuation)))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1)))
              (append (nextcall (caadr fp1))
                      (list
                        (list 'append
                              (pushterms (cdadr fp1))
                              'continuation))))
             ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'upstcks (car fp1)))
              (list 'change-stacks))
             (t (raise "code 13: No Patterns have Fired in cnvram+"))))

(defun deflog+-on nil (eval '*deflog+*))

(defun cnvram-
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'sta (car fp1))
                   (null (cadr fp1)))
              (list 'rename nil))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'sta (car fp1)))
              (list 'rename
                    (list 'quote (cadr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'guard (car fp1)))
              (callterms (cadr fp1)))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'name (car fp1))
                   (selfeval (cadr fp1)))
              (list 'equal
                    (cadr fp1)
                    (tterm (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'name (car fp1)))
              (list 'eq
                    (list 'quote (cadr fp1))
                    (tterm (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'fv (car fp1)))
              (list 'bind
                    (cadr fp1)
                    (tterm (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'bv (car fp1)))
              (list 'equal
                    (list 'lderef (cadr fp1))
                    (list 'lderef
                          (tterm (caddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'ftv (car fp1)))
              (list 'bind
                    (cadr fp1)
                    (ttail (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'btv (car fp1)))
              (list 'equal
                    (list 'lderef (cadr fp1))
                    (ttail (caddr fp1))))
             ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'btv (car fp1)))
              (list 'equal
                    (list 'lderef (cadr fp1))
                    (list 'lderef
                          (ttail (caddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1))
                   (null (cadr fp1)))
              (list 'popstack 'continuation))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1))
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           1)))
              (append (nextcall (caadr fp1))
                      (list 'continuation)))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'push (car fp1)))
              (append (nextcall (caadr fp1))
                      (list
                        (list 'append
                              (pushterms (cdadr fp1))
                              'continuation))))
             ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'upstcks (car fp1)))
              (list 'change-stacks))
             (t (raise "code 13: No Patterns have Fired in cnvram-"))))

(defun nextcall
       (fp1)
       (cond ((and (consp fp1)
                   (symbolp (car fp1)))
              (cons (car fp1)
                    (mapcar 'callterms (cdr fp1))))
             (t (raise "code 10: non-symbol used as function name"))))

(defun callterms
       (fp1)
       (cond ((selfeval fp1) fp1)
             ((var fp1) (list 'lderef fp1))
             ((atom fp1) (list 'quote fp1))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (atsign (car fp1)))
              (if (symbolp (cadr fp1))
                  (cons (cadr fp1)
                        (mapcar 'callterms (cddr fp1)))
                  (raise "code 10: non-symbol used as function name")))
             ((and (consp fp1)
                   (member '\| fp1))
              (callterms (ins-cons@ fp1)))
             ((consp fp1) (cons 'list
                                (mapcar 'callterms fp1)))
             (t (raise "code 13: No Patterns have Fired in callterms"))))

(defun pushterms
       (fp1)
       (cond ((selfeval fp1) fp1)
             ((var fp1) fp1)
             ((atom fp1) (list 'quote fp1))
             ((and (consp fp1)
                   (atsign (car fp1)))
              (callterms fp1))
             ((and (consp fp1)
                   (member '\| fp1))
              (callterms (ins-cons@ fp1)))
             ((consp fp1) (cons 'list
                                (mapcar 'pushterms fp1)))
             (t (raise "code 13: No Patterns have Fired in pushterms"))))

(defun ins-cons@
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (bar (car fp1)))
              (cadr fp1))
             ((consp fp1) (list '@
                                'cons
                                (car fp1)
                                (ins-cons@ (cdr fp1))))
             (t (raise "code 13: No Patterns have Fired in ins-cons@"))))

(defun lderef
       (fp1)
       (cond ((eq *deref-flag* t) (lderef1 fp1))
             (t (imm-local-value fp1))))

(defun imm-local-value
       (fp1)
       (cond ((logical-var fp1) (ilv1 fp1
                                      (get-local-binding fp1)))
             (t fp1)))

(defun ilv1
       (fp1 fp2)
       (cond ((eq '-0- fp2) fp1)
             (t fp2)))

(defun lderef1
       (fp1)
       (cond ((atom fp1) fp1)
             ((logical-var fp1)
              (prog (*temp*)
                    (setq *temp* (get-local-binding fp1))
                    (if (eq *temp* '-0-)
                        (return fp1)
                        (return (lderef1 *temp*)))))
             (t (mapcar 'lderef1 fp1))))

(defun get-local-binding
       (fp1)
       (getf *local-array* (cadr fp1) '-0-))

(defun lderef2
       (fp1 fp2)
       (cond ((eq '-0- fp2) fp1)
             (t (lderef1 fp2))))

(defun unify
       (fp1 fp2)
       (cond ((equal fp2 fp1) t)
             ((logical-var fp1)
              (and (not (occurs fp1 (lderef fp2)))
                   (setq *deref-flag* t)
                   (bind fp1 fp2)))
             ((logical-var fp2)
              (and (not (occurs fp2 (lderef fp1)))
                   (setq *deref-flag* t)
                   (bind fp2 fp1)))
             ((and (consp fp1)
                   (consp fp2)
                   (equal (car fp2) (car fp1)))
              (unify (cdr fp1) (cdr fp2)))
             ((and (consp fp1)
                   (consp fp2)
                   (logical-var (car fp1))
                   (not (occurs (car fp1)
                                (lderef (car fp2)))))
              (setq *deref-flag* t)
              (bind (car fp1) (car fp2))
              (unify (lderef (cdr fp1))
                     (lderef (cdr fp2))))
             ((and (consp fp1)
                   (consp fp2)
                   (logical-var (car fp2))
                   (not (occurs (car fp2)
                                (lderef (car fp1)))))
              (setq *deref-flag* t)
              (bind (car fp2) (car fp1))
              (unify (lderef (cdr fp1))
                     (lderef (cdr fp2))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (consp fp2)
                   (consp (car fp2)))
              (and (unify (car fp1) (car fp2))
                   (unify (lderef (cdr fp1)) (lderef (cdr fp2)))))
             (t nil)))

(defun bind
       (fp1 fp2)
       (setf (getf *local-array* (cadr fp1))
             fp2)
       t)

(defun rename
       (fp1)
       (setf *local-array* nil)
       (setq *deref-flag* nil)
       (rename1 fp1))

(defun rename1
       (fp1)
       (cond ((null fp1) t)
             (t
              (setq *newvar* (incf *varcounter*))
              (set (car fp1)
                   (list '$ *newvar*))
              (setf (getf *local-array* *newvar*)
                    '-0-)
              (rename1 (cdr fp1)))))

(defun newvar
       nil
       (list '$ (incf *varcounter*)))

(defun trybind
       (fp1 fp2)
       (cond ((and (logical-var fp1)
                   (not (occurs fp1 fp2)))
              (setq *deref-flag* t) (bind fp1 fp2))
             (t nil)))

(defun qtrybind
       (fp1 fp2)
       (cond ((logical-var fp1) (setq *deref-flag* t) (bind fp1 fp2))
             (t nil)))

(defun logical-var
       (fp1)
       (cond ((and (consp fp1)
                   (eq '$ (car fp1)))
              t)
             (t nil)))

(defun popstack
       (fp1)
       (cond ((null fp1) t)
             ((and (consp fp1)
                   (consp (car fp1))
                   (eq '-frame- (caar fp1)))
              (popstack (cdr fp1)))
             ((and (consp fp1)
                   (consp (car fp1))
                   (eq 'answer (caar fp1)))
              (answer (cdar fp1)))
             ((consp fp1) (invoke (deref (car fp1))
                                  (cdr fp1)))
             (t (raise "code 13: No Patterns have Fired in popstack"))))

(defun loop-and-pop
       (fp1)
       (cond ((and (consp fp1)
                   (consp (car fp1))
                   (eq '-frame- (caar fp1)))
              (loop-and-pop (cdr fp1)))
             ((and (consp fp1)
                   (looping (deref (car fp1))
                            (cdr fp1)))
              (backtrack))
             (t (popstack fp1))))

(defun looping
       (fp1 fp2)
       (cond ((null fp2) nil)
             ((and (consp fp1)
                   (consp fp2)
                   (and (consp (car fp2))
                        (= (length (car fp2))
                           2))
                   (eq '-frame- (caar fp2))
                   (consp (cadar fp2))
                   (equal (caadr (car fp2))
                          (car fp1))
                   (subsumes (deref (cdadr (car fp2)))
                             (cdr fp1))))
             ((consp fp2) (looping fp1 (cdr fp2)))
             (t (raise "code 13: No Patterns have Fired in looping"))))

(defun subsumes
       (fp1 fp2)
       (cond ((and (null fp1) (null fp2)) t)
             ((and (consp fp1)
                   (consp fp2)
                   (equal (car fp2) (car fp1)))
              (subsumes (cdr fp1) (cdr fp2)))
             ((and (consp fp1)
                   (consp fp2)
                   (logical-var (car fp1)))
              (subsumes (subst (car fp2)
                               (car fp1)
                               (cdr fp1))
                        (cdr fp2)))
             (t nil)))

(defun invoke
       (fp1 fp2)
       (apply (car fp1)
              (append (cdr fp1) (list fp2))))

(defun change-stacks
       nil
       (setq *toptrail* nil)
       (change-bindarray *local-array*)
       (push *toptrail* *trailstack*)
       (incf *refinements*))

(defun change-bindarray
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (> (length fp1) 1))
              (push (car fp1) *toptrail*)
              (hashbind (car fp1) (cadr fp1))
              (change-bindarray (cddr fp1)))
             (t (raise "code 13: No Patterns have Fired in change-bindarray"))))

(defun deref
       (fp1)
       (cond ((atom fp1) fp1)
             ((logical-var fp1)
              (prog (*temp*)
                    (setq *temp* (imm-binding fp1))
                    (if (eq *temp* '-0-)
                        (return fp1)
                        (return (deref *temp*)))))
             (t (mapcar 'deref fp1))))

(defun deflog1
       (fp1)
       (cond ((consp fp1)
              (if (not *outfile*)
                  (format t
                          ";;SEQUEL compiling logic procedure ~A...~%"
                          (car fp1)))
              (deflog2 (car fp1) (cdr fp1))
              (car fp1))
             (t (raise "code 13: No Patterns have Fired in deflog1"))))

(defun deflog1-
       (fp1)
       (cond ((consp fp1)
              (format t
                      ";;SEQUEL compiling logic procedure ~A.....~%"
                      (car fp1))
              (setq *pattern* t)
              (deflog2 (car fp1) (cdr fp1))
              (setq *pattern* nil)
              (car fp1))
             (t (raise "code 13: No Patterns have Fired in deflog1-"))))

(defun deflog1+
       (fp1)
       (cond ((consp fp1)
              (format t
                      ";;SEQUEL compiling logic procedure ~A.....~%"
                      (car fp1))
              (setq *deflog+* t)
              (deflog2 (car fp1) (cdr fp1))
              (setq *deflog+* nil)
              (car fp1))
             (t (raise "code 13: No Patterns have Fired in deflog1+"))))

(defun deflog2
       (fp1 fp2)
       (compile-horn-clauses (rem-@ (canlog fp1
                                            (bracket fp2)))))

(defun rem-@
       (fp1)
       (mapcar 'rem-@1 fp1))

(defun rem-@1
       (fp1)
       (mapcar 'rem-@2 fp1))

(defun rem-@2
       (fp1)
       (cond ((and (consp fp1)
                   (atsign (car fp1)))
              (cdr fp1))
             (t fp1)))

(defun bracket-only-lists
       (fp1)
       (cond ((atom fp1) fp1)
             (t (bracket fp1))))

(defun bracket-clause
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (arrow (cadr fp1)))
              (cons (bracket-literal (car fp1))
                    (cons (cadr fp1)
                          (mapcar 'bracket-literal
                                  (cddr fp1)))))
             (t (raise "code 13: No Patterns have Fired in bracket-clause"))))

(defun bracket-literal
       (fp1)
       (cond ((consp fp1) (cons (car fp1)
                                (bracket (cdr fp1))))
             (t (raise "code 13: No Patterns have Fired in bracket-literal"))))

(defun canlog
       (fp1 fp2)
       (canlog1 (brlog fp2) fp1))

(defun brlog
       (fp1)
       (reverse (brlog1 (plsep fp1) nil)))

(defun plsep
       (fp1)
       (plsep1 (reverse fp1)
               (logarity fp1)))

(defun logarity
       (fp1)
       (length (isologhead fp1)))

(defun isologhead
       (fp1)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (arrow (car fp1)))
              nil)
             ((and (consp fp1)
                   (consp (car fp1))
                   (atsign (caar fp1)))
              (isologhead (cdr fp1)))
             ((consp fp1) (cons (car fp1)
                                (isologhead (cdr fp1))))
             (t (raise "code 13: No Patterns have Fired in isologhead"))))

(defun plsep1
       (fp1 fp2)
       (cond ((null fp1) nil)
             ((and (consp fp1)
                   (arrow (car fp1)))
              (cons (car fp1)
                    (plsep2 (cdr fp1) fp2 fp2)))
             ((consp fp1) (cons (car fp1)
                                (plsep1 (cdr fp1) fp2)))
             (t (raise "code 13: No Patterns have Fired in plsep1"))))

(defun plsep2
       (fp1 fp2 fp3)
       (cond ((null fp1) nil)
             ((equal 0 fp2) (plsep1 (cons 'br$$ fp1)
                                    fp3))
             ((and (consp fp1)
                   (guard (car fp1)))
              (cons (car fp1)
                    (plsep2 (cdr fp1) fp2 fp3)))
             ((consp fp1)
              (cons (car fp1)
                    (plsep2 (cdr fp1) (- fp2 1) fp3)))
             (t (raise "code 13: No Patterns have Fired in plsep2"))))

(defun brlog1
       (fp1 fp2)
       (cond ((null fp1) (list fp2))
             ((and (consp fp1)
                   (eq 'br$$ (car fp1)))
              (cons fp2
                    (brlog1 (cdr fp1) nil)))
             ((consp fp1) (brlog1 (cdr fp1)
                                  (cons (car fp1) fp2)))
             (t (raise "code 13: No Patterns have Fired in brlog1"))))

(defun canlog1
       (fp1 fp2)
       (cond ((null fp1) nil)
             ((and (consp fp1) (id fp2))
              (cons (canlog2 (car fp1) (list fp2))
                    (canlog1 (cdr fp1) fp2)))
             (t (raise "code 1: bad function name"))))

(defun canlog2
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (arrow (car fp1)))
              (cons fp2 fp1))
             ((consp fp1)
              (canlog2 (cdr fp1)
                       (append fp2
                               (list (car fp1)))))
             (t (raise "Code 12: missing <-, -> or arity mistake"))))

(defun answer
       (fp1)
       (mapcar 'print-answer fp1)
       (if (y-or-n-p "more? ")
           (backtrack)
           t))
