;  Copyright (c) 1983 by Gordon S. Novak Jr. 

;  This file defines a program for converting Interlisp code to
;  common lisp code.  It was originally written by Gordon Novak to
;  run under Interlisp, but was then adapted by Peter Karp to run
;  under common lisp.
;  


; The main flow of control in this program is:
;     ltrancoms
;     ltrandocoms
;     lisptrans 
;     pfndef
;     tprogn
;     tprint
;     transfm
;     dialecttransfm 
;     glcommonlisptransfm
;     glcommonlispfn



(proclaim '(special fexprflg filefexprs filefns fileglobalvars filespecvars 
		    fnname glambdaflg ltnatom ltcompatibilityfnsused 
		    outputdialect untranslatedfns gltypesubs lttimestamp 
		    ltgetpairs))

(proclaim '(special lisptranscompatibilityfns lisptransglispfns 
		    lisptraninitialized lisptranstandardfns 
		    lisptranstranslatedfns lisptransunglispify))





(setq userxforms '((clock get-internal-real-time)
		   (position il-position)))

(setq stop 'Done!)    ; Needed for loading Interlisp files

	      
	      
; edited: 17-Mar-89 15:09 =========================== glcommontranscond

; TRANSLATE A COND INTO IF, WHEN, UNLESS IF APPROPRIATE FOR COMMON 
;   LISP 

(defun glcommontranscond (x)
(cond ((null (cddr x))
       (cond ((and (consp (caadr x))
		   (member (caaadr x)
			   '(not null)))
	      (cons 'unless
		    (cons (cadr (caadr x))
			  (cdadr x))))
	     ((null (cddadr x))
	      (cons 'if
		    (cadr x)))
	     (t (cons 'when
		      (cadr x)))))
      ((and (null (cdddr x))
	    (eq (caaddr x)
		t))
       (cons 'if
	     (cons (caadr x)
		   (cons (if (cddadr x)
			     (cons 'progn
				   (cdadr x))
			     (cadadr x))
			 (cond ((null (cadr (caddr x)))
				nil)
			       ((null (cddr (caddr x)))
				(list (cadr (caddr x))))
			       (t (list (cons 'progn
					      (cdaddr x)))))))))
      (t x)))


; edited: 17-Mar-89 15:59 ============================ glcommontransprog

; TRANSLATE A PROG EXPRESSION FOR COMMON LISP, TURNING IT INTO A LET 
;   IF APPROPRIATE 

(defun glcommontransprog (x)
(prog ((lastex (car (last x)))
       nret res tmp)
      (setq res
	    (if (and (consp lastex)
		     (setq nret (glnoccurs 'return
					   x))
		     (or (and (eq (car lastex)
				  'return)
			      (eql nret 1))
			 (eql nret 0))
		     (not (some #'atom
				(cddr x)))
		     (not (gloccurs 'go
				    x)))
		(cons 'let
		      (mapcon #'(lambda (y)
				  (if (or (cdr y)
					  (eql nret 0))
				      (list (car y))
				      (list (cadar y))))
			      (cdr x)))
		x))
      (mapl #'(lambda (y)
		(cond ((null (cdr y)))
		      ((null (setq tmp (cadr y)))
		       (setf (cdr y)
			     (cddr y))
		       (setq tmp (cadr y)))
		      ((atom tmp))
		      ((and (eq (car tmp)
				'setq)
			    (consp (caddr tmp))
			    (eq (caaddr tmp)
				'cdr)
			    (eq (cadr tmp)
				(cadr (caddr tmp))))
		       (rplaca (cdr y)
			       (list 'pop
				     (cadr tmp))))
		      ((eq (car tmp)
			   'progn)
		       (rplacd y (nconc (cdr tmp)
					(cddr y))))))
	    (cdr res))
      (return res)))


; edited: 10-Sep-86 14:16 ============================== commonlisptranserror

; TRANSLATE A CALL TO ERROR INTO A FORM ACCEPTABLE TO COMMON LISP 

(defun commonlisptranserror (x)
(prog (args str origargs)
      (unless (cdr x)
	      (return (list (car x)
			    "NOMSG")))
      (setq origargs (cdr x))
      (if (and (consp (car origargs))
	       (null (cdr origargs))
	       (eq (caar origargs)
		   'list))
	  (setq origargs (cdar origargs)))
      (setq str "")
      (dolist (z origargs)
	      (if (stringp z)
		  (setq str (concat str z))
		  (progn (setq str (concat str " ~S "))
			 (push z args))))
      (return (cons (car x)
		    (cons str (nreverse args))))))


; edited: 19-May-89 12:09 ===================================== commonptmatch


(defun commonptmatch (x busy)
  (setq x (ltptmatch x 'ltuserpatterns))
  (setq x (ltptmatch x 'ltpatterns))
  (unless busy (setq x (ltptmatch x 'ltnbpatterns)))
  x)


; edited: 16-Mar-89 11:35 ================================ commontransglerror

; Translate GLERROR calls for Common Lisp. 

(defun commontransglerror (x)
(prog (str args l item stritem)
      (if (or (atom (caddr x))
	      (not (eq (caaddr x)
		       'list)))
	  (return x))
      (setq l (cdaddr x))
      lp
      (unless l (return (list (car x)
			      (cadr x)
			      (cons 'list
				    (cons str (nreverse args))))))
      (setq item (pop l))
      (if (stringp item)
	  (setq stritem item)
	  (progn (setq stritem " ~A ")
		 (push item args)))
      (setq str (if (null str)
		    stritem
		    (concat str " " stritem)))
      (go lp)))


; edited:  4-MAR-83 15:58 ======================================== countargs
;
;  This function counts the number of arguments in the definition of each
;  function to be converted, so that in all calls to that function we can
;  check if it is supplied with the right number of arguments.  If it is
;  not, we pad the call with NIL arguments (since Interlisp defaults
;  the value of unsupplied arguments to NIL).  
;
;  The problem with this approach is that Interlisp Nospread functions 
;  have variable numbers of arguments, which is indicated by defining
;  the function as (LAMBDA ARGS) rather than (LAMBDA (ARG1 .. ARGn)) .
;  For nospread functions we store their number of arguments as 'NOSPREAD
;  instead of as an integer; the code that does call checking ignores
;  nospread functions.

(defun countargs (fn)
(let (def tmp gltypesubs)
     (setq def (ltgetd fn))
     (cond ((and (eq (car def)
		     'nlambda)
		 (atom (cadr def)))
	    (setf (get fn 'nargs)
		  'fexpr)
	    (setq filefexprs (nconc1 filefexprs fn)))
	   ((member (car def)
		    '(lambda nlambda))
	    (setf (get fn 'nargs)
		  (if (consp (cadr def))
		      (length (cadr def))
		      'NOSPREAD)))
	   ((eq (car def)
		'glambda)
	    (setq tmp (gldecl (cadr def)
			      '(t nil)
			      (list nil)
			      nil nil))
	    (setf (get fn 'nargs)
		  (length tmp))))))


; edited: 20-Aug-86 10:56 =================================== dialecttransfm

; Transform an expression X for another Lisp dialect. 

(defun dialecttransfm (x)
(case outputdialect (commonlisp (glcommonlisptransfm x))
      (maclisp (glmaclisptransfm x))
      (franzlisp (glfranzlisptransfm x))
      (ucilisp (glucilisptransfm x))
      (psl (glpsltransfm x))
      (t (error "NOMSG"))))


; edited: 22-Aug-86 16:01 ==================================== dialecttransfn

; Transform A FUNCTION NAME X for another Lisp dialect. 

(defun dialecttransfn (x)
(case outputdialect (commonlisp (glcommonlispfn x))
      (maclisp (glmaclispfn x))
      (franzlisp (glfranzlispfn x))
      (ucilisp (glucilispfn x))
      (psl (glpslfn x))
      (t (error "NOMSG"))))


; edited: 22-Aug-86 15:42 ======================================= dialectinit

; Initialize for a particular output dialect. 

(defun dialectinit (outputdialect)
(let nil (dolist (x (case outputdialect (ucilisp '((terpri 0)
						   (print 1)
						   (prin1 1)
						   (prin2 1)))))
		 (setf (get (car x)
			    'nargs)
		       (cadr x)))
     (dolist (x (cdr (assoc outputdialect lisptransdialectstandardfns)))
	     (setf (get x 'lisptransstandardfn)
		   t))
     (setf (get 'gllispdialect
		'lisptransevalwhenconst)
	   t)
     (setf (get 'gllispdialect
		'lisptransconstantvalue)
	   outputdialect)))


; edited: 22-Mar-89 14:29 =============================== glcommonlispfn

; Transform a function name FN for COMMON LISP dialect. 

(defun glcommonlispfn (fn)
(let (tmp)
     (setq tmp (assoc fn '((memb member)
			   (fmemb member)
			   (fassoc assoc)
			   (map mapl)
			   (litatom symbolp)
			   (fixp integerp)
			   (getprop get)
			   (getproplist symbol-plist)
			   (listget getf)
			   (listp consp)
			   (nlistp consp)
			   (neq eq)
			   (igreaterp >)
			   (greaterp >)
			   (igeq >=)
			   (geq >=)
			   (ilessp <)
			   (lessp <)
			   (ileq <=)
			   (leq <=)
			   (iplus +)
			   (plus +)
			   (idifference -)
			   (difference -)
			   (iminus -)
			   (minus -)
			   (itimes *)
			   (times *)
			   (iquotient /)
			   (quotient /)
			   (add1 1+)
			   (* comment)
			   (sub1 1-)
			   (mapconc mapcan)
			   (apply* funcall)
			   (declare comment)
			   (unpack explode)
			   (pack readlist)
			   (dreverse nreverse)
			   (strequal string=)
			   (alphorder string<=)
			   (glstrgreaterp string>)
			   (glstrgep string>=)
			   (glstrlessp string<)
			   (dremove delete)
			   (antilog exp)
			   (print interlispprint)
			   (copy copy-tree)
			   (dsubst nsubst)
			   (eqp eql)
			   (ldifference set-difference)
			   (ltoccurs gloccurs)
			   (ltnoccurs glnoccurs)
			   (glfboundp fboundp))))
     (and tmp (cadr tmp))))


; edited: 23-May-89 12:30 ============================== glcommonlisptransfm

; Transform an expression X for COMMON LISP dialect. 

(defun glcommonlisptransfm (x)
(prog (tmp tmpb notflg radflg fn)
      
; first do argument reversals. 

      (cond ((atom x)
	     (return x))
	    ((member (car x)
		     '(map mapc mapcar mapconc maplist mapcon push some every 
			   subset))
	     (setq x (list (car x)
			   (caddr x)
			   (cadr x)))
	     (if (and (eq (car x)
			  'mapc)
		      (consp (cadadr x))
		      (eq (car (cadadr x))
			  'lambda))
		 (setq x (cons 'dolist
			       (cons (list (caadr (cadadr x))
					   (caddr x))
				     (cddr (cadadr x))))))))
      
; now see if the result will be negated. 

      (setq notflg (member (car x)
			   '(neq nlistp)))

;  Here is where we do the actual work of translating expressions from one
;  Lisp to another.  We check for calls to various different functions by
;  name and use specific translations for them.  If the current call
;  does not match any of the named functions, we use function GLCOMMONLISPFN
;  to see if the name of the function being called should be changed.

      (case (car x)
	    (prin1 (if (stringp (cadr x))
		       (setq x (cons 'princ
				     (cdr x)))))
	    (arctan2 (setq radflg (cadddr x))
		     (setq x (list 'atan
				   (cadr x)
				   (caddr x)))
		     (or radflg (setq x (list '*
					      x 57.29578))))
	    (equal (if (or (numberp (cadr x))
			   (numberp (caddr x)))
		       (setq x (cons 'eql
				     (cdr x)))))
	    ((sin cos)
	     (setq radflg (caddr x))
	     (setq x (if radflg (list (car x)
				      (cadr x))
			 (list (car x)
			       (list '*
				     (cadr x)
				     .01745329)))))
	    (append (unless (cddr x)
			    (setq x (list (car x)
					  (cadr x)
					  nil))))
	    (selectq (setq x (cons 'case
				   (copy-tree (cdr x))))
		     (setq tmp (nleft x 2))
		     (setf (cdr tmp)
			   (if (null (cadr tmp))
			       nil
			       (list (list t (cadr tmp)))))
		     (dolist (y (cddr x))
			     (when (consp (car y))
				   (setq tmp (l-casecopy (car y)))
				   (setf (car y)
					 (remove-duplicates tmp)))
			     (if (and (consp (car y))
				      (null (cdar y)))
				 (rplaca y (caar y)))))
	    (nth (setq x (list 'nthcdr
			       (if (integerp (caddr x))
				   (1- (caddr x))
				   (list '1-
					 (caddr x)))
			       (cadr x))))
;  Added by PKarp
	    (nchars (setq x (list 'length
				  (list 'string
					(cadr x)))))
;  Added by PKarp
	    (gensym (setq x `(intern (gensym (string ,(cadr x))))))
;  Added by PKarp
	    (sort (setq x (list 'sort
				(cadr x)
				'(function string-lessp))))
	    ((member memb fmemb)
	     (setq fn (car x))
	     (setq x (cons 'member
			   (copy-tree (cdr x))))
	     (cond ((and (consp (setq tmp (caddr x)))
			 (eq (car tmp)
			     'quote)
			 (consp (cadr tmp))
			 (every #'atom
				(cadr tmp)))
		    (setq tmpb (l-casecopy (cadr tmp)))
		    (setf (cadr tmp)
			  (remove-duplicates tmpb))
		    (unless (cdadr tmp)
			    (setq x (list (cond ((symbolp (caadr tmp))
						 'eq)
						((numberp (caadr tmp))
						 'eql)
						(t 'equal))
					  (cadr x)
					  (kwote (caadr tmp))))))
		   ((eq fn 'member)
		    (setq x (append x (list ':test
					    '(function equal)))))))
	    ((getprop putprop)
	     (cond ((and (consp (caddr x))
			 (eq (caaddr x)
			     'quote)
			 (setq tmp (cadr (assoc (cadr (caddr x))
						ltgetpairs))))
		    (setq x (if (eq (car x)
				    'getprop)
				(list tmp (cadr x))
				(list 'setf
				      (list tmp (cadr x))
				      (cadddr x)))))
		   ((eq (car x)
			'putprop)
		    (setq x (list 'setf
				  (list 'get
					(cadr x)
					(caddr x))
				  (cadddr x))))
		   (t (setq x (list 'get
				    (cadr x)
				    (caddr x))))))
	    (glerror (setq x (commontransglerror x)))
	    (putd (setq x (list 'setf
				(list 'symbol-function
				      (cadr x))
				(caddr x))))
	    (prog2 (if (and (cddr x)
			    (null (cdddr x)))
		       (setq x (cons 'progn
				     (cdr x)))))
	    (t (if (setq tmp (glcommonlispfn (car x)))
		   (setq x (cons tmp (cdr x))))))
      (return (if notflg (list 'not
			       x)
		  x))))


; edited: 16-Dec-83 13:17 ================================ glfranzlisptransfm

; Transform an expression X for FRANZ LISP dialect. 

(defun glfranzlisptransfm (x)
(prog (tmp notflg radflg)
      
; first do argument reversals. 

      (cond ((atom x)
	     (return x))
	    ((member (car x)
		     '(map mapc mapcar mapconc maplist mapcon push 
			   glstrgreaterp alphorder subset))
	     (setq x (list (car x)
			   (caddr x)
			   (cadr x))))
	    ((eq (car x)
		 'putprop)
	     (setq x (list (car x)
			   (cadr x)
			   (cadddr x)
			   (caddr x)))))
      
; now see if the result should be negated. 

      (setq notflg (member (car x)
			   '(alphorder geq leq glstrgep nlistp)))
      (cond ((and (eq (car x)
		      'prin1)
		  (stringp (cadr x)))
	     (setq x (cons 'princ
			   (cdr x))))
	    ((eq (car x)
		 'arctan2)
	     (setq radflg (cadddr x))
	     (setq x (list 'atan (cadr x)
			   (caddr x)))
	     (unless radflg (setq x (list 'times x 57.29578))))
	    ((member (car x)
		     '
		     (sin (* cos .01745329)))
	     (setq radflg (caddr x))
	     (setq x (if radflg (list (car x)
				      (cadr x))
			 (list (car x)
			       (list 'times (cadr x)
				     .01745329)))))
	    ((setq tmp (glfranzlispfn (car x)))
	     (setq x (cons tmp (cdr x))))
	    ((and (member (car x)
			  '(some every))
		  (null (cdddr x)))
	     (setq x (list (car x)
			   (cadr x)
			   (caddr x)
			   nil)))
	    ((and (eq (car x)
		      'append)
		  (null (cddr x)))
	     (setq x (list (car x)
			   (cadr x)
			   nil)))
	    ((eq (car x)
		 'nth)
	     (setq x (list 'nthcdr
			   (if (integerp (caddr x))
			       (1- (caddr x))
			       (list '1-
				     (caddr x)))
			   (cadr x))))
	    ((eq (car x)
		 'selectq)
	     (setf (car x)
		   'caseq)
	     (setq tmp (nleft x 2))
	     (setf (cdr tmp)
		   (if (null (cadr tmp))
		       nil
		       (list (list t (cadr tmp)))))))
      (return (if notflg (list 'not
			       x)
		  x))))


; edited: 22-Dec-83 11:20 ==================================== glfranzlispfn

; Transform a function name FN for FRANZ LISP dialect. 

(defun glfranzlispfn (fn)
(let (tmp)
     (setq tmp (assoc fn '((memb memq)
			   (fmemb memq)
			   (fassoc assq)
			   (litatom symbolp)
			   (getprop get)
			   (getproplist plist)
			   (igreaterp >)
			   (igeq >=)
			   (geq lessp)
			   (ilessp <)
			   (ileq <=)
			   (leq greaterp)
			   (iplus +)
			   (idifference -)
			   (itimes *)
			   (iquotient /)
			   (add1 1+)
			   (sub1 1-)
			   (* comment)
			   (eqp =)
			   (mapconc mapcan)
			   (apply* funcall)
			   (declare comment)
			   (nchars flatc)
			   (listp dtpr)
			   (nlistp dtpr)
			   (unpack explode)
			   (pack readlist)
			   (strequal equal)
			   (glstrlessp alphalessp)
			   (alphorder alphalessp)
			   (glstrgreaterp alphalessp)
			   (glstrgep alphalessp)
			   (dreverse nreverse)
			   (dremove delq)
			   (antilog exp)
			   (prin1 print)
			   (print interlispprint)
			   (concat uconcat))))
     (and tmp (cadr tmp))))


; edited: 16-Dec-83 13:18 ================================== glmaclisptransfm

; Transform an expression X for MACLISP dialect. 

(defun glmaclisptransfm (x)
(prog (tmp notflg radflg)
      
; first do argument reversals. 

      (cond ((atom x)
	     (return x))
	    ((member (car x)
		     '(map mapc mapcar mapconc maplist mapcon push some every 
			   subset glstrgreaterp alphorder))
	     (setq x (list (car x)
			   (caddr x)
			   (cadr x))))
	    ((eq (car x)
		 'putprop)
	     (setq x (list (car x)
			   (cadr x)
			   (cadddr x)
			   (caddr x)))))
      
; now see if the result will be negated. 

      (setq notflg (member (car x)
			   '(alphorder geq leq glstrgep neq nlistp)))
      (cond ((and (eq (car x)
		      'prin1)
		  (stringp (cadr x)))
	     (setq x (cons 'princ
			   (cdr x))))
	    ((eq (car x)
		 'arctan2)
	     (setq radflg (cadddr x))
	     (setq x (list 'atan (cadr x)
			   (caddr x)))
	     (unless radflg (setq x (list 'times x 57.29578))))
	    ((member (car x)
		     '
		     (sin (* cos .01745329)))
	     (setq radflg (caddr x))
	     (setq x (if radflg (list (car x)
				      (cadr x))
			 (list (car x)
			       (list 'times (cadr x)
				     .01745329)))))
	    ((setq tmp (glmaclispfn (car x)))
	     (setq x (cons tmp (cdr x))))
	    ((and (eq (car x)
		      'return)
		  (null (cdr x)))
	     (setq x (list (car x)
			   nil)))
	    ((and (eq (car x)
		      'append)
		  (null (cddr x)))
	     (setq x (list (car x)
			   (cadr x)
			   nil)))
	    ((eq (car x)
		 'selectq)
	     (setf (car x)
		   'caseq)
	     (setq tmp (nleft x 2))
	     (setf (cdr tmp)
		   (if (null (cadr tmp))
		       nil
		       (list (list t (cadr tmp))))))
	    ((eq (car x)
		 'nth)
	     (setq x (list 'nthcdr
			   (if (integerp (caddr x))
			       (1- (caddr x))
			       (list '1-
				     (caddr x)))
			   (cadr x)))))
      (return (if notflg (list 'not
			       x)
		  x))))


; edited:  8-Sep-86 15:17 ======================================= glmaclispfn

; Transform a function name FN for MACLISP dialect. 

(defun glmaclispfn (fn)
(let (tmp)
     (setq tmp (assoc fn '((memb memq)
			   (fmemb memq)
			   (fassoc assq)
			   (litatom symbolp)
			   (getprop get)
			   (getproplist plist)
			   (listp pairp)
			   (nlistp pairp)
			   (neq eq)
			   (igreaterp >)
			   (igeq >=)
			   (geq lessp)
			   (ilessp <)
			   (ileq <=)
			   (leq greaterp)
			   (iplus +)
			   (idifference -)
			   (iminus -)
			   (itimes *)
			   (iquotient //)
			   (add1 1+)
			   (sub1 1-)
			   (* comment)
			   (mapconc mapcan)
			   (apply* funcall)
			   (declare comment)
			   (nchars flatc)
			   (unpack explode)
			   (pack readlist)
			   (dreverse nreverse)
			   (strequal equal)
			   (alphorder alphalessp)
			   (glstrgreaterp alphalessp)
			   (glstrgep alphalessp)
			   (glstrlessp alphalessp)
			   (dremove delq)
			   (antilog exp)
			   (print interlispprint)
			   (concat concatl))))
     (and tmp (cadr tmp))))


; edited: 18-Aug-86 10:35 ====================================== glpsltransfm

; Transform an expression X for Portable Standard Lisp dialect. 

(defun glpsltransfm (x)
(prog (tmp notflg radflg)
      
; first do argument reversals. 

      (cond ((atom x)
	     (return x))
	    ((eq (car x)
		 'push)
	     (setq x (list (car x)
			   (caddr x)
			   (cadr x))))
	    ((member (car x)
		     nil)
	     (setq x (list (car x)
			   (cadr x)
			   (cadddr x)
			   (caddr x))))
	    ((eq (car x)
		 'apply*)
	     (setq x (list 'apply
			   (cadr x)
			   (cons 'list
				 (cddr x))))))
      
; now see if the result will be negated. 

      (setq notflg (member (car x)
			   '(nlistp boundp geq leq igeq ileq)))
      (cond ((and (eq (car x)
		      'prin1)
		  (stringp (cadr x)))
	     (setq x (cons 'princ
			   (cdr x))))
	    ((eq (car x)
		 'arctan2)
	     (setq radflg (cadddr x))
	     (setq x (list 'atan (cadr x)
			   (caddr x)))
	     (unless radflg (setf (car x)
				  'atand)))
	    ((member (car x)
		     '
		     (sin (* cos .01745329)))
	     (setq radflg (caddr x))
	     (setq x (list (if radflg (car x)
			       (cdr (assoc (car x)
					   '((sin . sind)
					     (cod . cosd)))))
			   (cadr x))))
	    ((setq tmp (glpslfn (car x)))
	     (setq x (cons tmp (cdr x))))
	    ((and (eq (car x)
		      'return)
		  (null (cdr x)))
	     (setq x (list (car x)
			   nil)))
	    ((and (eq (car x)
		      'append)
		  (null (cddr x)))
	     (setq x (list (car x)
			   (cadr x)
			   nil)))
	    ((eq (car x)
		 'error)
	     (setq x (list (car x)
			   0
			   (cond ((null (cdr x))
				  nil)
				 ((null (cddr x))
				  (cadr x))
				 (t (cons 'list
					  (cdr x)))))))
	    ((eq (car x)
		 'selectq)
	     (setf (car x)
		   'caseq)
	     (setq tmp (nleft x 2))
	     (setf (cdr tmp)
		   (if (null (cadr tmp))
		       nil
		       (list (list t (cadr tmp)))))))
      (return (if notflg (list 'not
			       x)
		  x))))


; edited: 22-Dec-83 11:22 ======================================== glpslfn

; Transform a function name FN for Portable Standard Lisp dialect. 

(defun glpslfn (fn)
(let (tmp)
     (setq tmp (assoc fn '((memb memq)
			   (fmemb memq)
			   (fassoc assoc)
			   (litatom idp)
			   (getprop get)
			   (getproplist prop)
			   (putprop put)
			   (listp pairp)
			   (nlistp pairp)
			   (neq ne)
			   (igreaterp greaterp)
			   (igeq lessp)
			   (geq lessp)
			   (ilessp lessp)
			   (ileq greaterp)
			   (leq greaterp)
			   (iplus plus)
			   (idifference difference)
			   (itimes times)
			   (iquotient quotient)
                           (* commentoutcode)
			   (mapconc mapcan)
			   (declare commentoutcode)
			   (nchars flatsize2)
			   (nthchar glnthchar)
			   (dreverse reversip)
			   (strequal string!=)
			   (alphorder string!<!=)
			   (glstrgreaterp string!>)
			   (glstrgep string!>!=)
			   (glstrlessp string!<)
			   (eqp eqn)
			   (last lastpair)
			   (nth pnth)
			   (nconc1 aconc)
			   (u-case glucase)
			   (dsubst substip)
			   (boundp unboundp)
			   (unpack explode)
			   (pack implode)
			   (dremove deletip)
			   (antilog exp)
			   (substring ilsubstring)
			   (getd getddd)
			   (putd putddd)
			   (concat concatl))))
     (and tmp (cadr tmp))))


; edited:  8-Aug-86 10:08 ================================== glucilisptransfm

; Transform an expression X for UCI LISP dialect. 

(defun glucilisptransfm (x)
(prog (tmp notflg radflg)
      
; first do argument reversals. 

      (cond ((atom x)
	     (return x))
	    ((member (car x)
		     '(map mapc mapcar mapconc maplist mapcon some every 
			   subset glstrgep glstrlessp))
	     (setq x (list (car x)
			   (caddr x)
			   (cadr x))))
	    ((eq (car x)
		 'putprop)
	     (setq x (list (car x)
			   (cadr x)
			   (cadddr x)
			   (caddr x)))))
      
; next see if the result should be negated. 

      (setq notflg (member (car x)
			   '(glstrgreaterp glstrlessp)))
      
; now do function renamings. 

      (cond ((and (eq (car x)
		      'prin1)
		  (stringp (cadr x)))
	     (setq x (cons 'princ
			   (cdr x))))
	    ((eq (car x)
		 'arctan2)
	     (setq radflg (cadddr x))
	     (setq x (list 'atan (cadr x)
			   (caddr x)))
	     (unless radflg (setq x (list 'times x 57.29578))))
	    ((member (car x)
		     '
		     (sin (* cos .01745329)))
	     (setq radflg (caddr x))
	     (setq x (if radflg (list (car x)
				      (cadr x))
			 (list (car x)
			       (list 'times (cadr x)
				     .01745329)))))
	    ((setq tmp (glucilispfn (car x)))
	     (setq x (cons tmp (cdr x))))
	    ((and (eq (car x)
		      'return)
		  (null (cdr x)))
	     (setq x (list (car x)
			   nil)))
	    ((and (eq (car x)
		      'append)
		  (null (cddr x)))
	     (setq x (list (car x)
			   (cadr x)
			   nil)))
	    ((eq (car x)
		 'apply*)
	     
; change apply* into apply. 

	     (setq x (list 'apply
			   (cadr x)
			   (cons 'list
				 (cddr x)))))
	    ((eq (car x)
		 'error)
	     
; make error have only a single argument. 

	     (setq x (list (car x)
			   (cons 'list
				 (cdr x))))))
      (return (if notflg (list 'not
			       x)
		  x))))


; edited: 16-Dec-83 13:20 ======================================= glucilispfn

; Transform a function name FN for UCILISP dialect. 

(defun glucilispfn (fn)
(let (tmp)
     (setq tmp (assoc fn '((memb memq)
			   (fmemb memq)
			   (fassoc assoc)
			   (getprop get)
			   (getproplist glgetproplist)
			   (eqp =)
			   (igreaterp >)
			   (igeq ge)
			   (geq ge)
			   (ilessp <)
			   (ileq le)
			   (leq le)
			   (iplus +)
			   (idifference -)
			   (itimes *)
			   (iquotient //)
			   (maplist mapl)
			   (mapcar mapcl)
			   (* comment)
			   (declare comment)
			   (nchars flatsizec)
			   (pack readlist)
			   (unpack explode)
			   (fixp inump)
			   (pop pop)
			   (push push)
			   (listp consp)
			   (alphorder lexorder)
			   (glstrgreaterp lexorder)
			   (glstrlessp lexorder)
			   (strequal eqstr)
			   (antilog exp)
			   (glstrgep lexorder))))
     (and tmp (cadr tmp))))


; edited: 29-JUL-82 09:43 ======================================== l-casecopy

; Make a lower-case copy of a structure. 

(defun l-casecopy (x)
(cond ((symbolp x)
       (l-case x))
      ((atom x)
       x)
      ((null (cdr x))
       (cons (l-casecopy (car x))
	     nil))
      (t (cons (l-casecopy (car x))
	       (l-casecopy (cdr x))))))


; edited: 21-Aug-86 17:10 ======================================== lisptrans

; Translate an INTERLISP function into another LISP dialect. 

(defun lisptrans (fnname)
(prog (args defn gotdate fexprflg glnatom glambdaflg)

   (format *terminal-io* "~A " fnname)
   (force-output *terminal-io*)

      (setq ltnatom 0)
      (cond ((atom (setq defn (ltgetd fnname)))
	     (prin1 fnname t)
	     (princ " is not defined as an EXPR." t)
	     (terpri t)
	     (return))
	    ((and (eq (car defn)
		      'nlambda)
		  (atom (cadr defn)))
	     (setq fexprflg t))
	    ((eq (car defn)
		 'nlambda)
	     (prin1 fnname t)
	     (princ " is NLAMBDA-spread, which is not translated properly." t)
	     (terpri t)
	     (setq fexprflg t))
	    ((eq (car defn)
		 'glambda)
	     (setq glambdaflg t))
	    ((not (eq (car defn)
		      'lambda))
	     (prin1 fnname t)
	     (princ " has bad form." t)
	     (terpri t)
	     (return)))
      (terpri)
      (terpri)
      (setq args (cadr defn))
      (setq defn (cddr defn))

;  Print a banner line before the function definition that includes the
;  name of the function.

      (princ ";  ")
      (princ (make-string (- 70 (length (string fnname)))
			  :initial-element '#\=))
      (princ "  ")
      (princ fnname)
      (terpri)  (terpri)
      
;  Loop through any leading comments in the function definition, and print
;  them before we print the definition.

      a
      (cond ((and defn (consp (car defn))
		  (eq (caar defn)
		      '*))
	     (if (or (string= (cadar defn)
			      "GSN: ")
		     (eq (cadar defn)
			 'gsn))
		 (setf (car (cdar defn))
		       'edited\:))
	     (when (and gotdate (eq (cadar defn)
				    'edited\:))
		   (pop defn)
		   (go a))
	     (setq gotdate (or gotdate (eq (cadar defn)
					   'edited\:)))
	     (prcomment (car defn))
	     (terpri)                ; Insert a little white space
	     (pop defn)              ; Go to next element in definition
	     (go a))
	    ((and defn (consp (car defn))
		  (eq (caar defn)
		      'declare))
	     (pop defn)
	     (go a)))
      
; print the start of the function definition. 

      (pfndef fnname args fexprflg defn)
      (if (and (eq outputdialect 'franzlisp)
	       (not glambdaflg))
	  (prin1 '\)))
      (terpri)))


; edited: 22-Aug-86 15:39 ===================================== lisptransinit


(defun lisptransinit nil
(let nil (dolist (x lisptranscompatibilityfns)
		 (setf (get x 'lisptranscompatibilityfn)
		       t))
     (dolist (x lisptransstandardfns)
	     (setf (get x 'lisptransstandardfn)
		   t))
     (dolist (x lisptranstranslatedfns)
	     (setf (get x 'lisptranstranslatedfn)
		   t))
     (dolist (x lisptransglispfns)
	     (setf (get x 'lisptransglispfn)
		   t))
     (setq lisptransunglispify nil)))


; edited: 20-Aug-86 10:58 ======================================== ltcase

; Translate an atom into the appropriate case. 

(defun ltcase (x)
(case outputdialect ((commonlisp maclisp franzlisp)
       (l-casecopy x))
      (t x)))


; edited: 24-Mar-89 10:04 ==================================== ltdefpatterns


(defun ltdefpatterns (l patwd)
(dolist (pat l)
	(pushnew pat (get (caar pat)
			  patwd)
		 :test #'equal)))


; edited:  4-FEB-83 13:56 ======================================== ltgetd

; Get the definition of FN. 

(defun ltgetd (fn)
;;;;pkarp  (or (and lisptransunglispify (get fn 'glcompiled))
;;;;pkarp    (getd fn)))
  (get fn 'interlisp-definition))    ;;;;pkarp

; edited: 18-APR-83 12:22 ======================================== ltmkvar

; Make a variable name for function translations. 

(defun ltmkvar nil
(let nil (incf ltnatom)
     (readlist (append '(l t v a r)
		       (explode ltnatom)))))


; edited:  4-FEB-83 13:03 ======================================== ltnnils

; Make a list of N NILs. 

(defun ltnnils (n)
(prog (lst)
      lp
      (if (<= n 0)
	  (return lst))
      (push nil lst)
      (decf n)
      (go lp)))


; edited:  6-Aug-86 16:42 ======================================== glnoccurs

; COUNT OCCURENCES OF ATM IN STR 

(defun glnoccurs (atm str)
(cond ((atom str)
       (if (eq atm str)
	   1 0))
      ((consp str)
       (+ (glnoccurs atm (car str))
	  (glnoccurs atm (cdr str))))
      (t 0)))


; edited: 22-Mar-89 14:46 ======================================== ltnoticefn

; Notice a call to a function FN. Check whether FN is undefined within 
;   this file. 

(defun ltnoticefn (fnlst)
(prog
  (lst fn)
  (setq fn (car fnlst))
  (cond
    ((not (symbolp fn))
     (when (and (not glambdaflg)
		(not (numberp fn)))
	   (princ "The function " t)
	   (prin1 fnname t)
	   (princ " contains a fn call whose CAR is non-atomic." t)
	   (terpri t)
	   (prin1 fn t)
	   (terpri t)))
    ((or (get fn 'lisptransstandardfn)
	 (get fn 'lisptranstranslatedfn)
	 (get fn 'lisptransuserfn)
	 (dialecttransfn fn))
     (return))
    ((get fn 'lisptranscompatibilityfn)
     (or (member fn ltcompatibilityfnsused)
	 (push fn ltcompatibilityfnsused)))
    ((and glambdaflg (get fn 'lisptransglispfn)))
    (t (setq lst (get fn 'ltreferences))
       (or (equal lttimestamp (get fn 'lttimestamp))
	   (if glambdaflg
	       (progn 
; see if this is a glisp expression rather than a function reference. 

		      (glsepinit fn)
		      (if (or (not (eq fn (glsepnxt)))
			      (and (atom (cadr fnlst))
				   (progn (glsepinit (cadr fnlst))
					  (gloperator? (glsepnxt)))))
			  (return)))
	       (progn (push fn untranslatedfns)
		      (setf (get fn 'lttimestamp)
			    lttimestamp))))
       (or (and lst (eq (car lst)
			fnname))
	   (setf (get fn 'ltreferences)
		 (cons fnname lst)))))))


; edited: 24-Mar-89 10:19 ======================================== ltobjtrans

; Translate a line of a GLISP object description. Code in the response 
;   slot is translated. 

(defun ltobjtrans (line)
(if (atom (cadr line))
    line
    (cons (car line)
	  (cons (if (atom (caadr line))
		    (transfm (cadr line)
			     t)
		    (mapcar #'(lambda (x)
				(transfm x t))
			    (cadr line)))
		(cddr line)))))


; edited: 18-APR-83 12:19 ======================================== gloccurs

; See if X occurs in STR, using EQ. 

(defun gloccurs (x str)
(cond ((eq x str)
       t)
      ((atom str)
       nil)
      (t (or (gloccurs x (car str))
	     (gloccurs x (cdr str))))))


; edited: 23-May-89 12:35 ======================================== ltpatinit

;;;; I removed these translations because they're not strictly
;;;; correct.  (not (not x)) always returns T or NIL, whereas
;;;; X can take on any random value.
;;;;
;;;;             ((not (not x))
;;;;		  x)
;;;;		 ((not (null x))
;;;;		  x)
;;;;		 ((null (not x))
;;;;		  x)
;;;;		 ((null (null x))
;;;;		  x)

(defun ltpatinit nil
(ltdefpatterns '(((not (consp x))
		  (atom x))
		 ((not (atom x))
		  (consp x))
		 ((if (not x)
		      a b)
		  (if x b a))
		 ((if (null x)
		      a b)
		  (if x b a))
		 ((if (not c)
		      a)
		  (unless c a))
		 ((if (null c)
		      a)
		  (unless c a))
		 ((if a b nil)
		  (if a b))
		 ((setq place (cons new place))
		  (push new place))
		 ((setf place (cons new place))
		  (push new place))
		 ((unless (member new place)
			  (push new place))
		  (pushnew new place))
		 ((unless (member new place :test tst)
			  (push new place))
		  (pushnew new place :test tst))
		 ((car (nthcdr n x))
		  (nth n x))
		 ((setq place (1+ place))
		  (incf place))
		 ((setf place (1+ place))
		  (incf place))
		 ((setq place (1- place))
		  (decf place))
		 ((setf place (1- place))
		  (decf place))
		 ((setq place (+ place n))
		  (incf place n))
		 ((setf place (+ place n))
		  (incf place n))
		 ((setq place (- place n))
		  (decf place n))
		 ((setf place (- place n))
		  (decf place n))
		 ((glgetassoc x l)
		  (cdr (assoc x l)))
		 ((u-case x)
		  x)
		 ((if p (setf x y)
		      (setf x z))
		  (setf x (if p y z)))
		 ((if p (setq x y)
		      (setq x z))
		  (setq x (if p y z)))
		 ((if p (return x)
		      (return y))
		  (return (if p x y)))
		 ((cond (u (return uu))
			(v (return vv))
			(w (return ww)))
		  (return (cond (u uu)
				(v vv)
				(w ww))))
		 ((cond (u (return uu))
			(v (return vv))
			(w (return ww))
			(x (return xx)))
		  (return (cond (u uu)
				(v vv)
				(w ww)
				(x xx))))
		 ((cond (u (return uu))
			(v (return vv))
			(w (return ww))
			(x (return xx))
			(y (return yy)))
		  (return (cond (u uu)
				(v vv)
				(w ww)
				(x xx)
				(y yy))))
		 ((cond (u (return uu))
			(v (return vv))
			(w (return ww))
			(x (return xx))
			(y (return yy))
			(z (return zz)))
		  (return (cond (u uu)
				(v vv)
				(w ww)
				(x xx)
				(y yy)
				(z zz)))))
	       'ltpatterns)
; Patterns to use when the result is not busy. 
(ltdefpatterns '(((rplaca x y)
		  (setf (car x)
			y))
		 ((rplaca (cdr x)
			  y)
		  (setf (cadr x)
			y))
		 ((rplaca (car x)
			  y)
		  (setf (caar x)
			y))
		 ((rplaca (cddr x)
			  y)
		  (setf (caddr x)
			y))
		 ((rplacd x y)
		  (setf (cdr x)
			y))
		 ((setq x (cdr x))
		  (pop x))
		 ((setf x (cdr x))
		  (pop x)))
	       'ltnbpatterns))


; edited:  4-FEB-83 11:41 =============================== ltprettyprintconst


(defun ltprettyprintconst (lst)
(let nil (terpri)
     (terpri)
     (prin1 '\()
     (prin1 (ltcase 'glispconstants))
     (dolist (x lst)
	     (terpri)
	     (prin1 '\()
	     (prin1 (ltcase x))
	     (spaces 1)
	     (printdef (ltcase (get x 'glisporigconstval)))
	     (spaces 1)
	     (printdef (ltcase (get x 'glispconstanttype)))
	     (prin1 '\)))
     (terpri)
     (prin1 '\))
     (terpri)
     (terpri)))


; edited:  4-FEB-83 11:44 ============================== ltprettyprintglobals


(defun ltprettyprintglobals (lst)
(let nil (terpri)
     (terpri)
     (prin1 '\()
     (prin1 (ltcase 'glispglobals))
     (dolist (x lst)
	     (terpri)
	     (prin1 '\()
	     (prin1 (ltcase x))
	     (spaces 1)
	     (printdef (ltcase (get x 'glispglobalvartype)))
	     (prin1 '\))
	     (terpri))
     (terpri)
     (prin1 '\))
     (terpri)
     (terpri)))


; edited: 18-APR-83 11:48 ================================= ltprettyprintstrs

; Pretty-print GLISP structure definitions for file package output. 

(defun ltprettyprintstrs (lst)
(prog
  (tmp obj)
  (terpri)
  (terpri)
  (prin1 '\()
  (interlispprint (ltcase 'glispobjects))
  lp
  (unless lst (terpri)
	  (prin1 '\))
	  (terpri)
	  (terpri)
	  (return))
  (setq obj (pop lst))
  (when (setq tmp (get obj 'glstructure))
	(terpri)
	(terpri)
	(prin1 '\()
	(prin1 (ltcase obj))
	(spaces 1)
	(printdef (ltcase (car tmp)))
	(mapl #'(lambda (rest)
		  (terpri)
		  (prin1 (ltcase (car rest)))
		  (spaces 1)
		  (printdef (ltcase (if (member (car rest)
						'(prop adj isa msg))
					(mapcar #'ltobjtrans
						(cadr rest))
					(cadr rest)))
			    8))
	      (cdr tmp))
	(prin1 '\))
	(terpri))
  (go lp)))


; edited:  5-May-89 18:09 ====================================== ltprintplain


(defun ltprintplain (form pos)
(prin1 '\()(prin1 (car form))(spaces 1)(printdef (cdr form)
						  (+ pos 8)
						  t t)(prin1 '\)))


; edited: 24-Mar-89 10:27 ======================================= ltprintprop

; Print a property value. FLG is T to print it even if NIL. 

(defun ltprintprop (atm prop val flg)
(when (or val flg)
      (terpri)
      (tprint (list 'putprop
		    (list 'quote
			  atm)
		    (list 'quote
			  prop)
		    (kwote val))
	      nil 1)))


; edited:  4-FEB-83 14:30 ====================================== ltprintprops

; Print specified PROPS for ATM. PROPS may be a single prop, list of 
;   props, or ALL. FLG is T to print props even when NIL. 

(defun ltprintprops (atm props flg)
(prog nil (cond ((eq props 'all)
		 (mapl #'(lambda (l)
			   (unless (member (car l)
					   sysprops)
				   (ltprintprop atm (car l)
						(cadr l)
						flg)))
		       (symbol-plist atm))
		 (return))
		((atom props)
		 (ltprintprop atm props (get atm props)
			      flg)
		 (return)))
      (dolist (x props)
	      (ltprintprop atm x (get atm x)
			   flg))))


; edited: 24-Mar-89 10:00 ======================================== ltptmatch

; Try to match INP against optimization patterns. If a match is found, 
;   the right-hand side of the pattern is returned with appropriate 
;   substitutions. 

(defun ltptmatch (inp patwd)
(prog (patterns)
      (setq ltptmatchbindings nil)
      top
      (and (consp inp)
	   (atom (car inp))
	   (setq patterns (get (car inp)
			       patwd)))
      lp
      (cond ((null patterns)
	     (return inp))
	    ((ltptmatcha (caar patterns)
			 inp)
	     (setq inp (sublis ltptmatchbindings (cadar patterns)))
	     (go top)))
      (setq ltptmatchbindings nil)
      (pop patterns)
      (go lp)))


; edited: 15-Mar-89 10:33 ======================================== ltptmatcha

; Match a pattern against an input. If PAT is a list, it is at the 
;   front of a Lisp function call. 

(defun ltptmatcha (pat inp)
(let (tmp)
     (cond

;  If (car pat) matches (car inp) then execute the pattern on the cdrs.
           ((consp pat)
	    (and (consp inp)
		 (eq (car pat)
		     (car inp))
		 (ltptmatchl (cdr pat)
			     (cdr inp))))
	   ((or (numberp pat)
		(null pat)
		(eq pat t))
	    (eql pat inp))
	   ((symbolp pat)
	    (cond ((setq tmp (assoc pat ltptmatchbindings))
		   (equal inp (cdr tmp)))
		  ((or (not (member pat '(m n)))
		       (numberp inp))
		   (push (cons pat inp)
			 ltptmatchbindings)))))))


; edited: 15-Mar-89 10:36 ======================================== ltptmatchl

; Match two lists of args 

(defun ltptmatchl (patl inpl)
(cond ((null patl)
       (null inpl))
      ((consp patl)
       (and (consp inpl)
	    (ltptmatcha (car patl)
			(car inpl))
	    (ltptmatchl (cdr patl)
			(cdr inpl))))))


; edited: 19-May-89 12:12 ======================================== ltrancoms

; Translate all the funtions on a FNS list from INTERLISP to another 
;   LISP dialect. 

(defun ltrancoms (outputdialect outfilename coms ltusertransforms ltgetpairs)
(let
  (normfile dialects *glnatom* filefns filespecvars fileglobalvars filefexprs 
	    untranslatedfns fullfilename ltcompatibilityfnsused glambdaflg 
	    fnname lttimestamp)
  (setq *glnatom* 0)
  
; make sure the output dialect is legal. 

  (setq dialects '(commonlisp franzlisp maclisp psl ucilisp))
  (unless (member outputdialect dialects)
	  (error "Dialect must be a member of ~S " dialects))
  (setq lttimestamp (get-internal-real-time))
  (terpri t)
  (terpri t)
  (terpri t)
  (format t "[Translating into file ~A in dialect ~A]" outfilename outputdialect)
  (terpri t)
  (terpri t)
  (ltpatinit)
  (if ltusertransforms (ltdefpatterns ltusertransforms 'ltuserpatterns))
  (dialectinit outputdialect)
  
  (ltraninterpretcoms coms)
  (dolist (x filefns)
	  (setf (get x 'lisptransuserfn)
		t))

  (if lisptransunglispify (dolist (fn filefns)
				  (if (and (consp (getd fn))
					   (eq (car (getd fn))
					       'glambda))
				      (glcc fn))))
  
; count number of function arguments for error checking. 

  (mapc #'countargs
	filefns)

  (setq normfile (open outfilename :direction :io))
  (output normfile)

; print a header on the file. 

    (terpri)
    (prcomment (list '*
		     fullfilename))

    (terpri)
    (terpri)
    (terpri)
    (ltranspecials filespecvars)
    (ltranglobals fileglobalvars)
    (ltranfexprs filefexprs)
    (mapc #'ltrandocoms        ; Translate functions
	  coms)
  (output t)
  (when ltcompatibilityfnsused (terpri)
	(princ 
	  "The following functions in the compatibility package are used:")
	(terpri)
	(interlispprint ltcompatibilityfnsused))
  (ltreporterrors untranslatedfns)
  fullfilename))


; edited:  4-FEB-83 13:36 =================================== ltrancomsvalue

; Get the value of an element of a COMS list; elements are of the form
; (KEYWORD VALUE1 .. VALUEn) or  (KEYWORD * VARIABLE) .  We return either
; (VALUE1 .. VALUEn) or the value of VARIABLE.

(defun ltrancomsvalue (comslist)
(if (eq (cadr comslist)
	'*)
    (eval (caddr comslist))
    (cdr comslist)))


; edited:  8-MAR-83 16:27 ======================================= ltrandocoms

; Process one item from a COMS list. 

(defun ltrandocoms (comslst)
(prog (lst)
      (when (eq (car comslst)
		'*)
	    (terpri)
	    (terpri)
	    (terpri)
	    (prcomment comslst)
	    (terpri)
	    (terpri)
	    (terpri)
	    (return))
      (setq lst (ltrancomsvalue comslst))

;  Dispatch according to the type of the current COMS element:

      (case (car comslst)

;         Translate function definitions

	    (fns (mapc #'lisptrans
		       lst))

;         Translate variable definitions

	    ((vars initvars)

;         For each definition, write out a form that SETQs the variable
;         to either its currently defined value, or to a value defined
;         here within the COMS.

	          (dolist (name lst)
			  (terpri)
			  (princ '\()
			  (prin1 (ltcase 'setq))
			  (spaces 1)
			  (if (atom name)
;                      Current value
			      (progn (prin1 (ltcase name))
				     (spaces 1)
				     (printdef (ltcase (kwote (eval name))
						       )))
;                      Value within COMS
			      (progn (prin1 (ltcase (car name)))
				     (spaces 1)
				     (printdef (ltcase (cadr name)))))
			  (princ '\))
			  (terpri)))
	    (glispobjects (ltprettyprintstrs lst))
	    (glispconstants (ltprettyprintconst lst))
	    (glispglobals (ltprettyprintglobals lst))
	    (p (dolist (x lst)
		       (terpri)
		       (printdef (ltcase x))
		       (terpri)))
	    (prop (dolist (x (cddr comslst))
			  (ltprintprops x (cadr comslst)
					t)))
	    (ifprop (dolist (x (cddr comslst))
			    (ltprintprops x (cadr comslst)
					  nil)))
	    (props (dolist (x (cdr comslst))
			   (ltprintprops (car x)
					 (cadr x)
					 t))))))


; edited:  4-FEB-83 14:09 ======================================= ltranfexprs

; Output declarations for FEXPR functions. 

(defun ltranfexprs (lst)
(if lst (case outputdialect ((maclisp franzlisp)
	       (terpri)
	       (terpri)
	       (printdef (list 'declare
			       (cons '*fexpr
				     (mapcar #'l-case
					     lst))))))))


; edited:  4-FEB-83 12:43 ======================================== ltranfns

; Translate all the funtions on a FNS list from INTERLISP to another 
;   LISP dialect. 

(defun ltranfns (outputdialect outfilename fns)
(ltrancoms outputdialect outfilename (list (cons 'fns
						 fns))
	   nil nil))


; edited: 20-Aug-86 10:59 ====================================== ltranglobals

; Output declarations for global variables. 

(defun ltranglobals (lst)
(when lst (terpri)
      (terpri)
      (printdef (case outputdialect ((maclisp franzlisp)
		       (list 'declare
			     (cons 'special
				   (mapcar #'l-case
					   lst))))
		      (commonlisp (list 'proclaim
					(list 'quote
					      (cons 'special
						    (mapcar #'l-case
							    lst)))))
		      (ucilisp (list 'declare
				     (cons 'special
					   lst)))
		      (psl (list 'global
				 (list 'quote
				       lst)))))))


; edited: 18-APR-83 11:31 ================================ ltraninterpretcoms

; Look through a COMS list to extract certain kinds of information 
; before translation begins.  Note that we don't extract ALL the information
; now; some we deal with later in LTRANDOCOMS.

(defun ltraninterpretcoms (coms)
(prog (comslst)
      lp
      (unless coms (return))
      (setq comslst (pop coms))
      (case (car comslst)
	    (fns (setq filefns (append (ltrancomsvalue comslst)
				       filefns)))
	    (specvars (setq filespecvars (append (ltrancomsvalue comslst)
						 filespecvars)))
	    (globalvars (setq fileglobalvars (union  (ltrancomsvalue comslst)
						     fileglobalvars)))
	    (initvars   (setq fileglobalvars
			      (union  (mapcar #'car
					      (ltrancomsvalue comslst))
				      fileglobalvars)))
	    ((glispobjects glispglobals glispconstants)
	     (pushnew (car comslst)
		      filefexprs)))
      (go lp)))


; edited: 20-Aug-86 11:00 ==================================== ltranspecials

; Output declarations for special variables. 

(defun ltranspecials (lst)
(when lst (terpri)
      (terpri)
      (printdef (case outputdialect ((maclisp franzlisp)
		       (list 'declare
			     (cons 'special
				   (mapcar #'l-case
					   lst))))
		      (commonlisp (list 'proclaim
					(list 'quote
					      (cons 'special
						    (mapcar #'l-case
							    lst)))))
		      (ucilisp (list 'declare
				     (cons 'special
					   lst)))
		      (psl (list 'fluid
				 (list 'quote
				       lst)))))))


; edited:  4-FEB-83 13:23 ==================================== ltremovecommas

; Remove top-level commas in a GLISP A function. 

(defun ltremovecommas (l)
(if (member '\,
	    l)
    (mapcan #'(lambda (x)
		(if (eq x '\,)
		    nil
		    (cons x nil)))
	    l)
    l))


; edited:  4-FEB-83 14:19 ==================================== ltreporterrors

; Report untranslated fns and where they were referenced. 

(defun ltreporterrors (fns)
(prog nil (or fns (return))
      (terpri)
      (terpri)
      (princ "The following functions are not in this file:")
      (terpri)
      (dolist (fn (sort fns #'string-lessp))
	      (terpri)
	      (interlispprint fn)
	      (interlispprint (sort (get fn 'ltreferences) #'string-lessp))
	      (setf (get fn 'ltreferences)
		    nil))))


; edited: 23-Mar-89 12:56 ====================================== lttransprog

; Translate places where a PROG variable is initialized to a value as 
;   allowed by Interlisp. This is done by adding a SETQ to set the 
;   value of each PROG variable which is initialized. In some cases, a 
;   change of variable name is required to preserve the same 
;   semantics. 

(defun lttransprog (x)
(let
  (tmp argvals setvars rest flg)
  (mapl
    #'(lambda (y)
	(cond
	  ((and (not flg)
		(consp (car y)))
	   
; if possible, use the same variable; otherwise, make a new one. 

	   (setq
	     tmp
	     (if (or (some #'(lambda (z)
			       (and (consp z)
				    (gloccurs (car z)
					      (cadar y))))
			   (cadr x))
		     (some #'(lambda (z)
			       (gloccurs (caar y)
					 z))
			   argvals))
		 (ltmkvar)
		 (caar y)))
	   (setq setvars (nconc1 setvars (list 'setq
					       tmp
					       (cadar y))))
	   (setq rest (nsubst tmp (caar y)
			      (cddr x)))
	   (push (cadar y)
		 argvals)
	   (rplaca y tmp))
	  ((and (atom (car y))
		(eq (nthchar (car y)
			     -1)
		    '\:)
		(consp (cadr y))
		(member (caadr y)*gltypenames*))
	   (setq flg t))
	  (t (setq flg nil))))
    (cadr x))
  (if setvars (setf (cdr (cdr x))
		    (nconc setvars rest)))
  x))


; edited: 10-Sep-86 13:36 ======================================== pfndef

; Print the definition of a function for another Lisp dialect. 

(defun pfndef (fnname args fexprflg defn)
(prog
  (newfnname)
  (princ '\()

;  Print the DEFUN and the function name and arguments.

  (case
    outputdialect
    (commonlisp
      (princ (cond (fexprflg 'defmacro)
		   (glambdaflg 'gldefun)
		   (t 'defun)))
      (spaces 1)
      (princ (l-case fnname))
      (spaces 1)
      (if
	fexprflg
	(progn (printdef (cons '&rest
			       (list (l-case args))))
	       (spaces 1)
	       (princ '\`)
	       (if (and defn (null (cdr defn))
			(null (cddar defn))
			(eq (cadar defn)
			    args))
		   (progn (printdef (list (l-case (caar defn))
					  (concat "',"
						  (format nil "~A" (l-case args)))))
			  (princ '\))
			  (terpri)
			  (return))
		   (progn (printdef (list (l-case (setq newfnname
							(concat (string fnname)
							       "-expr")))
					  (concat "',"
						  (format nil "~A" (l-case args)))))
			  (princ '\))
			  (terpri)
			  (pfndef newfnname (list args)
				  nil defn)
			  (return))))
	(printdef (if (atom args)
		      (l-case args)
		      (mapcar #'l-case
			      args)))))
    (maclisp (prin1 (if glambdaflg 'gldefun
			'defun))
	     (spaces 1)
	     (prin1 (l-case fnname))
	     (spaces 1)
	     (when fexprflg (prin1 'fexpr)
		   (spaces 1)
		   (if (and args (atom args))
		       (setq args (list args))))
	     (setq args (if (atom args)
			    (l-case args)
			    (mapcar #'l-case
				    args)))
	     (printdef args))
    (franzlisp (prin1 (if glambdaflg 'gldefun
			  'def))
	       (spaces 1)
	       (prin1 (l-case fnname))
	       (spaces 1)
	       (cond (fexprflg (prin1 '\()
			       (prin1 'nlambda))
		     (glambdaflg)
		     (t (prin1 '\()
			(prin1 'lambda)))
	       (spaces 1)
	       (if (and args (atom args))
		   (setq args (list args)))
	       (setq args (if (atom args)
			      (l-case args)
			      (mapcar #'l-case
				      args)))
	       (printdef args))
    ((ucilisp psl)
     (prin1 (cond (fexprflg 'df)
		  (glambdaflg 'dg)
		  (t 'de)))
     (spaces 1)
     (prin1 fnname)
     (spaces 1)
     (if (and fexprflg args (atom args))
	 (setq args (list args)))
     (printdef args))
    (t (error "NOMSG")))
;
;  Print the definition of the function
;
;;;;  (terpri)
  (tprogn defn nil)
  (princ '\))))


; edited:  8-Aug-86 17:35 ======================================== prcomment


; Print an Interlisp comment for another dialect. 


(defun prcomment (com)
(prog (col nc (firstele t))
      (if (> (il-position (output))
	     1)
	  (terpri))

      (case outputdialect
	    ((commonlisp maclisp franzlisp psl)
	     (princ (case outputdialect ((commonlisp maclisp franzlisp)
			   '\;)
			  (psl '%%)
			  (t (error "NOMSG"))))
	     (spaces 2)
	     (setq col 3)

;  If we have a comment like (* * Foo), get ride of the second asterisk.

	     (if (eq '* (cadr com))
		 (setq com (cdr com)))

; Loop through each element of the comment

	     (dolist (x (cdr com))
		     (setq nc (1+ (length (format nil "~A" x))))
; Emit a newline if we're at the edge of the page.

		     (when (> (+ col nc)
			      71)
		       (terpri)
		       (princ (case outputdialect
				((commonlisp maclisp franzlisp)
				 '\;)
				(psl '%%)
				(t (error "NOMSG"))))
				(spaces 2)
				(setq col 3))

; Emit current element (capitalize the first word in a comment).

		     (princ (if firstele
				(string-capitalize (format nil "~A" x))
				x))
		     (spaces 1)
		     (setq col (+ col nc))
		     (setq firstele nil)))

	    (ucilisp (prin1 '{)
		     (prin1 '\;)
		     (spaces 1)
		     (setq col 4)
		     (dolist (x (cdr com))
			     (prog (n)
				   lp
				   (cond ((setq n (strpos ";" x))
					  (setq x (rplstring x n "."))
					  (go lp))
					 ((setq n (strpos "{" x))
					  (setq x (rplstring x n "("))
					  (go lp))
					 ((setq n (strpos "}" x))
					  (setq x (rplstring x n ")"))
					  (go lp)))
				   (setq nc (1+ (length (format nil "~A" x))))
				   (if (> nc 60)
				       nil
				       (progn (when (> (+ col nc)
						       71)
						    (terpri)
						    (spaces 3)
						    (setq col 4))
					      (prin1 x)
					      (spaces 1)
					      (setq col (+ col nc))))))
		     (prin1 '})
		     (terpri))
	    (t (error "NOMSG")))
      (terpri)))


; edited: 22-Aug-86 16:07 ======================================== remove


(defun remove-duplicates (x)
(intersection x x))


; edited: 24-Mar-89 10:25 ======================================== tprint

; Print an expression, translated for another Lisp dialect. 

(defun tprint (x busy col)
(cond ((symbolp x)
       (printdef (case outputdialect ((commonlisp maclisp franzlisp)
			(l-case x))
		       (ucilisp x)
		       (t x))
		 col))
      ((atom x)
       (printdef x col))
      ((eq (car x)
	   '*)
       (prcomment x))
      ((eq (car x)
	   'quote)
       (princ '\')
       (printdef (case outputdialect ((commonlisp maclisp franzlisp)
			(l-casecopy (cadr x)))
		       (ucilisp (u-casecopy (cadr x)))
		       (t (cadr x)))
		 (1+ col)))
      (t
 (printdef (case outputdialect ((commonlisp maclisp franzlisp)
			  (l-casecopy (transfm x busy)))
			 (ucilisp (u-casecopy (transfm x busy)))
			 (t (transfm x busy)))
		   col))))


; edited: 24-Mar-89 10:26 ======================================== tprogn

; Output a list which is an implicit PROGN for another Lisp dialect. 

(defun tprogn (lst col)
(mapl #'(lambda (x)
	  (tprint (car x)
		  (null (cdr x))
		  col))
      lst))


; edited: 24-Mar-89 10:57 ======================================== transcond

; Translate a COND expression. 

(defun transcond (x busy)
(let
  (tmp)
  (setq
    tmp
    (cons
      'cond
      (mapcar
	#'(lambda (y)
	    (maplist #'(lambda (z)
			 (transfm (car z)
				  (or (eq z y)
				      (and busy (null (cdr z))))))
		     y))
	(cdr x))))
  (case outputdialect (commonlisp (glcommontranscond tmp))
	(t tmp))))


; edited: 24-Mar-89 10:18 ======================================== transfm

; Transform an expression X for another Lisp dialect. 

(defun transfm (x busy)
(prog
  (tmp nactual)
  (cond ((or (atom x)
	     (stringp x))
	 (return x))

; Special transformations for a few functions

	((member (car x)
		 '(quote  function cond selectq case setq prog * if))
	 (setq x
	       (case (car x)
		     (quote (transquote x busy))
		     (function (transfunction x busy))
		     (cond (transcond x busy))
		     (if (transif x busy))
		     ((selectq case)
		      (setq tmp (transselectq x busy))
		      
; the selectq may be optimized away, in which case we want to avoid 
;   transforming the result a second time. 

		      (if (and (consp tmp)
			       (eq (car tmp)
				   (car x)))
			  (dialecttransfm tmp)
			  tmp))
		     (setq (transsetq x busy))
		     (prog (transprog x busy))
        	     (* (cons 'comment
        		      (cdr x)))
		     (t (error "NOMSG")))))
	((and (eq outputdialect 'commonlisp)
	      (eq (car x)
		  'error))
	 (setq x (commonlisptranserror x)))
	((member (car x)
		 filefexprs))
	(t (setq x (mapcar #'(lambda (y)
			       (if (and (eq y '/)
					(eq outputdialect 'maclisp))
				   '//
				   (transfm y t)))
			   x))
	   
; Check for correct number of arguments to converted functions.  If
; If too few args are supplied we add NIL arguments.

	   (if (and (not glambdaflg)
		    (integerp (setq tmp (get (car x)
					     'nargs)))
		    (not (eql tmp (setq nactual (length (cdr x))))))
	       (if (> nactual tmp)
		   (progn (princ "***** " t)
			  (prin1 fnname t)
			  (princ " has too many args in call to " t)
			  (interlispprint (car x)
					  t))
		   (progn (nconc x (ltnnils (- tmp nactual)))
			  (prin1 fnname t)
			  (princ ": NILs added in call to " t)
			  (interlispprint (car x)
					  t))))
   
; now see if any transformations need to be made for the output 
;   dialect of lisp. 

	   (if (and glambdaflg (member (car x)
				       '(a an)))
	       (setq x (ltremovecommas x)))
	   (ltnoticefn x)
	   (setq x (dialecttransfm x))))
  (if (eq outputdialect 'commonlisp)
      (setq x (commonptmatch x busy)))
  (return x)))


; edited: 24-Mar-89 10:39 ================================ transfunction

; Translate a FUNCTION expression. 

(defun transfunction (x busy)
(if
  (atom (cadr x))
  (list (car x)
	(or (dialecttransfn (cadr x))
	    (cadr x)))
  (list (car x)
	(cons (caadr x)
	      (cons (cadadr x)
		    (maplist #'(lambda (y)
				 (transfm (car y)
					  (and busy (null (cdr y)))))
			     (cddadr x)))))))


; edited: 24-Mar-89 11:03 ======================================== transprog

; Transform a PROG expression. 

(defun transprog (x busy)
(let (tmp)
     (setq tmp (cons (mapcar #'(lambda (v)
				 (if (atom v)
				     v
				     (list (car v)
					   (transfm (cadr v)
						    t))))
			     (cadr x))
		     (mapcar #'(lambda (y)
				 (transfm y nil))
			     (cddr x))))
     (case outputdialect (commonlisp (glcommontransprog (cons 'prog
							      tmp)))
	   (t (lttransprog (cons 'prog
				 (if (some #'consp
					   (car tmp))
				     (copy-tree tmp)
				     tmp)))))))


; edited:  8-May-89 13:54 ======================================== transquote

; Transform a QUOTEd expression for another Lisp dialect. Atom 
;   substitutions are made in case single-character atoms require an 
;   escape character in the target dialect. 

(defun transquote (x busy)

  x

;;;;(list 'quote
;;;;      (sublis (case outputdialect (commonlisp (cons (cons (character 39)
;;;;							  '\')
;;;;						    '((\\ . \\)
;;;;						      (\: . \:)
;;;;						      (\:= . \:=)
;;;;						      (\, . \\\,)
;;;;						      (\; . \\\;)
;;;;						      (\  . \ )
;;;;						      (\. . \.)
;;;;						      (\( . \()
;;;;						      (\) . \))
;;;;						      (\~ . \~))))
;;;;		    (maclisp '((/ . //)
;;;;			       (\, . /\,)
;;;;			       (\' . /\')
;;;;			       (\  . /\ )
;;;;			       (\. . /\.)
;;;;			       (\( . /\()
;;;;			       (\) . /\))))
;;;;		    (franzlisp '((\\ . \\)
;;;;				 (\' . \')
;;;;				 (\, . \,)
;;;;				 (+_ . +\_)
;;;;				 (-_ . -\_)
;;;;				 (\  . \ )
;;;;				 (\. . \.)
;;;;				 (\( . \()
;;;;				 (\) . \))))
;;;;		    (ucilisp '((/ . //)
;;;;			       (\, . /\,)
;;;;			       (\' . /\')
;;;;			       (\  . /\ )
;;;;			       (\. . /\.)
;;;;			       (\( . /\()
;;;;			       (\) . /\))))
;;;;		    (psl '((! . !!)
;;;;			   (\, . !\,)
;;;;			   (\' . !\')
;;;;			   (\  . !\ )
;;;;			   (\. . !\.)
;;;;			   (\( . !\()
;;;;			   (\\ . !)
;;;;			   (\) . !\)))))
;;;;	      (cadr x)
;;;;	      t))
)


; edited: 24-Mar-89 11:05 ====================================== transselectq


(defun transselectq (x busy)
(prog
  (l sel)
  (if
    (or (consp (cadr x))
	(not (get (cadr x)
		  'lisptransevalwhenconst)))
    (return
      (cons
	(car x)
	(cons
	  (transfm (cadr x)
		   t)
	  (maplist
	    #'(lambda (y)
		(cond
		  ((atom (car y))
		   (transfm (car y)
			    t))
		  ((cdr y)
		   (cons
		     (caar y)
		     (maplist #'(lambda (z)
				  (transfm (car z)
					   (and busy (null (cdr z)))))
			      (cdar y))))
		  (t (transfm (car y)
			      busy))))
	    (cddr x))))))
  (setq sel (get (cadr x)
		 'lisptransconstantvalue))
  (setq l (cddr x))
  lp
  (cond
    ((null (cdr l))
     (return (transfm (car l)
		      busy)))
    ((or (eq sel (caar l))
	 (and (consp (caar l))
	      (member sel (caar l))))
     (return
       (if (cddar l)
	   (cons 'progn
		 (maplist #'(lambda (z)
			      (transfm (car z)
				       (and busy (null (cdr z)))))
			  (cdar l)))
	   (transfm (cadar l)
		    busy)))))
  (pop l)
  (go lp)))


; edited: 24-Mar-89 11:01 ======================================== transsetq

; Translate a SETQ expression. 

(defun transsetq (x busy)
(list (car x)
      (cadr x)
      (transfm (caddr x)
	       t)))


;  ===========================================================  transif
;  PKarp
;  
;  Convert Interlisp (IF a THEN (foo) (bar) ELSE (baz)) to
;                    (if a (progn (foo) (bar)) (baz))

(defun transif (x busy)
  (let ((c1 nil) (c2 nil))

;  Collect THEN clauses

    (dolist (y (cdddr x))
      (if (eq 'else y)
	  (return)
	  (push y c1)))
    (setq c1 (reverse c1))
    (if (> (length c1) 1)
	(push 'progn c1)
	(setq c1 (car c1)))

;  Collect ELSE clauses

    (setq c2 (cdr (member 'else (cdr (cdddr x)))))
    (if (> (length c2) 1)
	(push 'progn c2)
	(setq c2 (car c2)))

;  Construct new expression

    (if c2
	(list 'if
	      (transfm (cadr x) t)
	      (transfm c1 t)
	      (transfm c2 t))
	(list 'if
	      (transfm (cadr x) t)
	      (transfm c1 t)))
))



; edited: 26-AUG-82 15:32 ======================================== u-casecopy

; Make an UPPER-CASE copy of a structure. 

(defun u-casecopy (x)
(cond ((symbolp x)
       x)
      ((atom x)
       x)
      ((null (cdr x))
       (cons (u-casecopy (car x))
	     nil))
      (t (cons (u-casecopy (car x))
	       (u-casecopy (cdr x))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;            The following lists various Lisp function names and is 
;            used by the translator to identify functions that are
;            undefined in a given file, and hence may be problematic.

; Functions that are in an Interlisp-compatibility package that we in theory
; have.

(setq lisptranscompatibilityfns '(concat copy dsubst eqp every fixp floatp 
					 getd intersection kwote ldifference 
					 listget nconc1 nthchar pop pop push 
					 push putd remove some spaces stringp 
					 subatom subset union for mkatom
				         mkstring))

(setq lisptransglispfns '(a an an case case for for if if repeat repeat send 
			    sendprop send sendprop the those the those while 
			    while a an case for if repeat send sendprop the 
			    those while _))

;  Standard functions in all lisps

(setq lisptransstandardfns '(and apply assoc atom boundp caaaar caaadr caaar 
				 caadar caaddr caadr caar cadaar cadadr cadar 
				 caddar cadddr caddr cadr car cdaaar cdaadr 
				 cdaar cdadar cdaddr cdadr cdar cddaar cddadr 
				 cddar cdddar cddddr cdddr cddr cdr cond cons 
				 eq equal eval go last length list member 
				 minus nconc not null numberp
				   or print prin1 progn prog1 prog2 read 
				      rplaca rplacd set setq sublis subst 
				      terpri zerop))

;  Interlisp functions that we translate inline into other code.

(setq lisptranstranslatedfns '(add1 alphorder append apply* declare dremove 
				    dreverse eqp error every fassoc fmemb geq 
				    getprop getproplist glstrgep glstrgreaterp 
				    glstrlessp idifference igeq igreaterp ileq 
				    ilessp plus minus difference times 
				    quotient greaterp lessp iplus iquotient 
				    itimes leq listp litatom map mapc mapcar 
				    mapcon mapconc maplist memb nchars neq 
				    nlistp nth pack push putprop return 
				    selectq some strequal sub1 subset unpack 
				    push sin cos arctan2 nchars sort))

;  Functions specific to certain Lisp dialects

(setq lisptransdialectstandardfns '((commonlisp aref arrayp make-array princ 
						proclaim setf every stringp 
						floatp pop push gensym if
						remove-duplicates remprop)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



(setq lisptransunglispify nil)

(setq ltgetpairs nil)

(setq ltptmatchbindings nil)

(lisptransinit)

