;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                   Copyright (c) 1989 Peter D. Karp                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;  
;                  Interlisp compatibility package
;  
;  This set of functions is meant to be used both with a common lisp 
;  program that converts Interlisp code to common lisp, and with the 
;  programs that the converter produces.  These functions emulate
;  various Interlisp functions.
;  

(in-package :pkarp)
(export '(kwote strpos l-case output interlispprint printdef
	  il-position nleft ntail putassoc filecreated file-created
	  define-file-info prettycomprint noop copyright addtovar rpaqq
	  rpaq rpaq? defineq putprops mkatom memb) )



; These functions are called in forms written by the Interlisp file
; package.  We define many of them to null because the file package
; uses them to record information that we don't care about.

(defmacro filecreated (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro file-created (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro define-file-info (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro prettycomprint (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro noop (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro copyright (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro comment (&rest rest)
  (declare (ignore rest))
  nil)
(defmacro addtovar (&rest rest)
  (declare (ignore rest))
  nil)


;;;;;;;;; Variable-binding functions ;;;;;;;;;

(defmacro rpaqq (variable value)
  `(setq ,variable (quote ,value)))


(defmacro rpaq (variable value)
  `(setq ,variable ,value))


(defmacro rpaq? (variable value)
  (if (boundp variable)
      nil
      `(setq ,variable ,value)))



; Return a form that, when evaluated, yields X as its value.

(eval-when (load compile eval)
  (if (not (fboundp 'kwote))
      (defun kwote (X)
	(if (or (null X)
		(numberp X))
	    X
	  (list (quote quote) X)))
    ) )


;  The ARG function is used by Interlisp to select a parameter to a
;  function defined as (LAMBDA PARAMS) , which I believe they call a
;  Nospread definition.  So (ARG PARAMS 1) selects the value of the
;  first parameter from a function call with a variable number of
;  paramters.

(defmacro arg (param n)
  `(nth ,(- n 1) ,param))



; The DEFINEQ function is used to define a set of Interlisp functions.
; We define DEFINEQ so that it saves the function definition on a
; property of the function called INTERLISP-DEFINITION.

(defmacro defineq (&rest forms)
  (let ((value nil))
    (do ((forms forms (cdr forms)))
	((null forms))
	(setq value (cons `(setf (get (quote ,(car (car forms)))
				      'interlisp-definition)
				 (quote ,(cadr (car forms))))
			  value))
        (format t "~A " (caar forms))
        (force-output))
    (cons 'prog (cons nil value))))



(defmacro putprops (name macro value)
  `(setf (get (quote ,name) (quote ,macro))
	 (quote ,value)))


;;;;;;;;; String-manipulation functions ;;;;;;;;;



; Convert a string to an atom, e.g., (eq (mkatom "string") 'string)

(defmacro mkatom (atom)
  `(if (stringp ,atom)
       (intern ,atom)
       ,atom))


; Convert an object to a string, e.g.,  '(a b c) ==> "(a b c)"

(eval-when (load compile eval)
  (if (not (fboundp 'mkstring))
      (defmacro mkstring (object)
	`(if (stringp ,object)
	     ,object
	     (if (symbolp ,object)
		 (string  ,object)
	       (format nil "~A" ,object))))
  ) )

; Concatenate a variable list of strings.

(eval-when (load compile eval)
  (if (not (fboundp 'concat))
      (defmacro concat (&rest strings)
	(append '(concatenate 'string)
		(mapcar
		 #'(lambda (x)
		     (if (stringp x)
			 `(quote ,x)
		       `(mkstring ,x)))
		 strings))
	) ) )


; Determine if String contains Pat.
;
; Note: Interlisp uses an index origin of 1, whereas Common Lisp uses
; an origin of 0.  This function uses an origin of 1, for compatibility
; with existing Interlisp code that calls it.

(defun strpos (Pat String &optional
		          (Start 1) (Skip nil) (Anchor nil) (Tail nil))
  (let (match_index)

    (if Skip
	(error "The (strpos) emulator does not support the Skip argument"))

    (if (not (stringp Pat))
	(setq Pat (mkstring Pat)))
    (if (not (stringp String))
	(setq String (mkstring String)))
	
    (if (null Start) (setq Start 1))    ; Interlisp default is nil
    (setq Start (- Start 1))            ; Adjust index origin

    (setq match_index (if Anchor
			  (if (string= Pat String :start2 Start
				       :end2 (+ Start (length Pat)))
			      Start
			      nil)
			  (search  Pat String :start2 Start)))
    (if (and Tail match_index)
	(setq match_index (+ match_index (length Pat))))

;    Adjust index origin

    (if match_index (setq match_index (+ 1 match_index)))

    match_index))



; Return a specified substring of a string.

(defun substring (x start &optional (end (length x)) dummy)
  (declare (ignore dummy))

  (if (not (stringp x))
      (setq x (mkstring x)))

  (if (or (> start (length x))
          (> end (length x)))
      nil
      (progn
	(if (< start 0)
	    (setq start (+ 1 start (length x))))
	(if (< end 0)
	    (setq end (+ 1 end (length x))))

	(setq start (- start 1))

        (subseq x start end)))
)


; Interlisp: Converts X to lower case; if Flg is set then X should be
; capitalized, but I have not implemented this.  In Interlisp X can be
; an s-expr.

(defun l-case (X &optional (Flg nil))
  (declare (ignore flg))
  (read-from-string (string-downcase (prin1-to-string X))))


;;;;;;;;  I/O Functions  ;;;;;;;;

; Reset the standard output to argument  stream  if it is supplied, else
; return the current standard output.  If stream is T, then reset the
; standard output to the terminal.

(defun output (&optional (stream nil stream_flg))

  (if stream_flg

      (let (old-stdout)
	(setq old-stdout *standard-output*)
	(if (eq t stream)
	    (progn
	      (close *standard-output*)
	      (setq *standard-output* (make-synonym-stream '*terminal-io*)))
	    (setq *standard-output* stream))
	old-stdout)

      *standard-output*))
	


;  Write N space characters to Stream .

(defun spaces (N &optional (Stream *standard-output*))
  (dotimes (i N)
    (write-char '#\Space Stream)))



(defun interlispprint (x &optional (file *standard-output*))
  (prin1 x file)
  (terpri file))



(defun printdef (expr &optional left def tailflg fnslst file)
  (declare (ignore left def tailflg fnslst file))
  (write expr :pretty t)
  (terpri)
)


; Interlisp: Compute our current column position in the file stream, that is,
; the number of characters of our current position to the right of the last
; new line character.
  
(defun il-position (&optional (stream *standard-output*))
  (let (initial-position (column 0) filepos)

      (setq initial-position (file-position stream))

      (if (not (eq 0 initial-position))
	  (progn
	    (file-position stream (setq filepos (- initial-position 1)))
	    (setq column 1)
	    (do
	     nil
	     ((or
	       (eq 0 filepos)
	       (eq '#\Newline (read-char stream))))
	     (setq column (+ 1 column))
	     (file-position stream (setq filepos (- filepos 1))))))

      (file-position stream initial-position)
      column
))


;;;;;;;;  List Manipulation  ;;;;;;;;


(defmacro memb (atom list)
  `(member ,atom ,list))


; Interlisp: Returns the tail of L that contains N more elements than Tail.

(defun nleft (L N &optional (Tail nil))
  (let ()
    (if (null Tail)
	(ntail L N)
	(do ((x L (cdr x)))
	    ((or (null x)
		 (equal x Tail)) .
  	  ((if (null x)
 	       (ntail L N)
 	       (ntail L (+ N (length Tail))))))))))


; Returns the last N elements of list L.

(defun ntail (L N)
  (if (> N (length L))
      nil
      (do ((x L (cdr x))
	   (i 0 (+ 1 i)))
	  ((= i (- (length L) N)) .
	   (x)))))


; Searches Alst for a sublist CAR of which is EQ to Key.  If one
; is found, the CDR is replaced with Val.  If no such sublist is
; found, (CONS Key Val) is added at the CDR of Alst.  Returns
; Val.

(defun putassoc (Key Val Alst)
  (let (ass)
    (unless (consp Alst)
      (error "Arg not list ~S" Alst))

    (if (setq ass (assoc Key Alst))
	(rplacd ass Val)
	(rplacd Alst (cons (cons Key Val)
			   (cdr Alst))))
    Val
))
