;;; -*- Package: TRANSLISP; Mode: LISP; Syntax: Common-lisp; Base: 10 -*-
;;;_________________________________________________________________________________
;;;
;;;                       System: Translisp
;;;                       Module: TRANSLISP
;;;                       (Version 1.0)
;;;
;;; Copyright (c): Forschungsgruppe INFORM, Matthias Ressel
;;;                Universitaet Stuttgart
;;;
;;; File: ODIN:>matthias>translisp>translisp.lisp
;;; File Creation Date: 1/18/88 22:23:43
;;; Last Modification Time: 1/23/88 21:19:01
;;; Last Modification By: matthias
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;
;;;________________________________________________________________________________ 

(in-package 'translisp)

(export '(do-not-use rulesets rules transform-file transform-expr tf rule-untrace
	  deactivate-rule-ruleset activate-rule deactivate-rule delete-rule
	  delete-rule-ruleset delete-ruleset ActivateRules DeactivateRules
	  DeleteAllRules DeleteRules Change-Application InstallRule LoadRules
	  setup-application get-attribut *state* *default-rulefile-pathname*
	  *readtable-reset-hook* *readtable-hook* *read-function* *protocolport*
	  *prinlevel* *prinlength* *prot-lmar* ReadForm Commonist $commented-form$ 
	  transformfile transformstart transformfileform
	  rule-trace $left$ $result$ $separator$ 
	  Protocol NoPRotocol NeverProtocol All *read-into-package*))

#-system-translisp
(eval-when (compile eval)
  (load "pattern-macros")
  (load "record-macros")
  ;;(eval-when (compile load eval)
  (load "franz-macros"))


;;; semicolon-readln wird fuer *readform-readtable* benoetigt
(defun semicolon-readln (stream char)
  (declare (special *semicolon*)
	   (ignore char))
  (push (cons '$separator$ (list (cons #\; (readln stream)))) *semicolon*)
  (values))

;;; *readform-readtable* wird in readform benoetigt
(defvar *readform-readtable*
        (let ((readtable (copy-readtable)))
          (set-macro-character #\; #'semicolon-readln nil readtable)
          readtable))

;;; Readform wird als Initialisierungswert einer globalen Variable
;;; benoetigt und muss daher vor dieser stehen
(defun ReadForm (&optional port &aux (skip-chars '(#\Tab #\Space #\Return #\Newline #\;)))
  (do ((charn (peek-char nil port nil -1) (peek-char nil port nil -1))
       (s-list (list nil)))
      ((not (member charn skip-chars))
       (if (car s-list)
           (cons '$separator$ (list (car s-list)))
           (unwind-protect
	       (progn
		 (set-macro-character #\; #'semicolon-readln nil)
		 (let* ((*semicolon* nil)
			(input (read port nil)))
		   (declare (special *semicolon*))
		   (if *semicolon*
		       `($commented-form$ ,input ,.(nreverse *semicolon*))
		       input)))
	     (set-syntax-from-char #\; #\;))
	   ))
    (if (eql charn #\;)
        (lconc s-list (readln port))
        (tconc s-list (read-char port nil -1)))))

; Globale Variablen
; =================

(defvar *last-prot* 2)      ; gibt an, welche Protokollfunktion als letzte ausgefue.
(defvar *prot-lmar* 0)      ; Anfangsindentierung bei Vereinfachen von Ausdruecken
						; for use with dribble
(defvar *lmar* 0)           ; augenblickliche Indentierung fuer Protokollierung
(defvar *prinlength* 3)
(defvar *prinlevel* 2)            ; Default: Printtiefe bei Shortprot
(defvar *print-header* nil)         ; damit protocol-0 header ausgeben kann
(defvar *protocolport* nil)         ; Default: Protokoll auf Bildschirm
(defvar *read-function* 'ReadForm)  ; damit koennen auch Kommentare eingelesen.
(defvar *readtable-hook* nil)       ; damit kann man readtable vor dem Einlesen 
                                    ; eines Files aendern
(defvar *readtable-reset-hook* nil)
;(defvar *read-into-package* *package*
;  "in which package to read files to transform")

(defvar *rule-trace* t)             ; Moegliche Werte: nil t all
(defvar *state* (make-state))     ; Status ueber geladene Regeln

(defvar *default-rulefile-pathname*
	"rules.bin")

(setq *print-case* :downcase)			; auf besonderen Wunsch neuer Default

; Makros
; ======

;; if you want to hash
;; ===================
;
;(defun Mark (expr ruleset)
;  (pput expr ruleset))
;(defun Marked? (expr ruleset)
;  (member ruleset (pget expr) :test #'eq))
;
; Rule Loading
; ============

(defmacro get-rule (name)
  `(symbol-value ,name))

(defmacro get-attribut (rule attribut)
  `(assoc ,attribut (rule-attributes ,rule)))

(defmacro rule-supersets (rule)
  `(supersets-of-ruleset (rule-ruleset ,rule)))

(defun rule? (rule)
  (and (listp rule)
       (rulep rule)))

(defun rule-needed (rule)
  (get-attribut rule '$needed$))

(defun keys-of-pattern (pattern)
  (cond ((and (Elem? pattern)
	     (MemqRestr? pattern))
	 (MemSet pattern))
        ((Var? pattern) '($else$))
	((symbolp pattern) (list pattern))
        ((atom pattern) '($else$))
	((and (Elem? (car pattern))
              (MemqRestr? (car pattern)))
         (MemSet (car pattern)))
        ((symbolp (car pattern)) (list (car pattern)))
        (t '($else$))))
  
(defun supersets-of-ruleset (ruleset)
  (if (consp ruleset)
      ruleset
      (get ruleset '$supersets$)))
	
(defun reachable-nodes (nodes relation &optional (closure nodes))
  (cond ((null nodes) closure)
        (t (let* ((element (car nodes))
                  (closure-of-first (funcall relation element))
                  (new-items (nset-difference closure-of-first closure)))
             (reachable-nodes (union (cdr nodes) new-items)
                              relation
                              (union new-items closure))))))
	 
(defun setup-application (application &key no-warn)
  (when (state-goal *state*)
	(unless no-warn
	  (format t "~&%Note: Application already loaded: ~a~
                     ~%       will be deactivated" (state-goal *state*))))
  (deactivaterules)
  (setf (state-goal *state*) application))

(defun LoadRules (&optional (file "") (goal 'Commonist) &aux pathname)
  (setq pathname (merge-pathnames file *default-rulefile-pathname*))
  (cond ((probe-file pathname)
	 (setq *default-rulefile-pathname* pathname)
	 (setf (state-rulefile *state*) (namestring pathname))
	 (setf (state-rulefile-abbr *state*) (intern (pathname-name pathname)))
	 (setup-application goal)
	 (unwind-protect (load pathname) (reset-syntax)))
	(t (format t "~&I can't find ~a~%~
                        Keeping old rules" (namestring pathname)))))	; Fehlermeldung
  
(defun InstallRule (name ruleset priority match-function
		    pattern condition repl-function replacement action
                    optimization-goal protocol-flag attributes)
  (let ((rule (make-rule :name name
                           :ruleset ruleset
                           :priority priority
                           :match-function match-function
			   :pattern pattern
                           :condition condition
			   :replacement-function repl-function
                           :replacement replacement
                           :action action
                           :optimization-goal optimization-goal
                           :protocol-flag protocol-flag
                           :attributes attributes)))
    (cond ((boundp (rule-name rule))
           (delete-rule (get-rule (rule-name rule)))
           (msg N
                '|Rule with same name |
                (rule-name rule)
		'| will be overwritten|)))
    (set (rule-name rule) rule)
    (if (InstallRule? rule)
        (InstallRule-1 rule)
        (pushnew (rule-name rule) (state-deactivated-rules *state*)))))

(defun InstallRule-1 (rule)
  (push (rule-name rule) (state-rules *state*))
  (mapc #'(lambda (ruleset)
           (mapc #'(lambda (key)
                    (AddRule key rule ruleset))
                 (keys-of-pattern (rule-pattern rule))))
        (supersets-of-ruleset (rule-ruleset rule)))
  (princ "*")
  (force-output))
  
(defun common-element-p (list1 list2)
  (do ((restlist1 list1 (cdr restlist1))
       (result nil))
      ((or (null restlist1) result)
       result)
    (setq result (member (car restlist1) list2))))

(defun InstallRule? (rule)
  (or (eq (rule-optimization-goal rule) 'All)
      (common-element-p (get (state-goal *state*) :ruleclasses)
			(rule-optimization-goal rule))))  

(defun greater-prio (rule1 rule2)
  (> (rule-priority rule1) (rule-priority rule2)))

(defun AddRule (symbol rule ruleset)
  (let ((list-of-rules (get symbol ruleset)))
    (or (member rule list-of-rules :test #'equal)
        (progn (addruleset ruleset *state*)
               (if (null list-of-rules)
		   (setf (get symbol ruleset) (list rule))
		   (setf (get symbol ruleset)
			 (sort (cons rule list-of-rules)
			       #'greater-prio)))))))

(defun addruleset (ruleset state)
  (or (member ruleset (state-rulesets state) :test #'eq)
      (push ruleset (state-rulesets state))))

; Change Active Rules
; ===================

(defun Change-Application (application)
  (setup-application application :no-warn t)
  (activateRules)
  application)
	 
(defun DeleteRules ()				; CCC delete
  (mapc
    #'(lambda (rule-name)
       (cond ((not (eq (rule-optimization-goal (get-rule rule-name)) 'All))
              (setf (state-deactivated-rules *state*)
		    (delete rule-name (state-deactivated-rules *state*) :test #'eq))
              (delete-rule (get-rule rule-name)))))
    (append (state-deactivated-rules *state*) (state-rules *state*)))
  (setf (state-rulefile *state*) nil)
  (values))

(defun DeleteAllRules ()
  (mapc #'(lambda (rule-name)
           (delete-rule (get-rule rule-name)))
        (nconc (state-deactivated-rules *state*) (state-rules *state*)))
  (setf (state-rulefile *state*) nil)
  (setf (state-deactivated-rules *state*) nil)
  (setf (state-rulesets *state*) nil)
  (values))

(defun DeactivateRules (&key dont-ask-if-needed)
  (mapc #'(lambda (rule-name)
           (deactivate-rule (get-rule rule-name) :dont-ask-if-needed dont-ask-if-needed))
        (state-rules *state*))
  (values))

(defun ActivateRules ()
  (mapc #'(lambda (rule-name)
           (cond ((InstallRule? (get-rule rule-name))
                  (activate-rule (get-rule rule-name)))))
         (state-deactivated-rules *state*))
  (values))
	  
(defun delete-ruleset (ruleset)
  (mapc #'(lambda (rule-name)
	    (cond ((member ruleset
			   (rule-supersets (get-rule rule-name))
			   :test
			   #'eq)
		   (delete-rule (get-rule rule-name)))))
        (state-rules *state*)))

(defun delete-rule-ruleset (rule ruleset)
  (if (rulep rule)
      (if ruleset
          (mapc #'(lambda (key)
                   (setf (get key ruleset) (delete rule (get key ruleset) :test #'eq)))
                (keys-of-pattern (rule-pattern rule)))
          (delete-rule rule)))
  (values))

(defun delete-rule (rule)
  (cond ((rulep rule)
         (mapc #'(lambda (key)
                  (mapc #'(lambda (ruleset)
                           (setf (get key ruleset)
                               (delete rule (get key ruleset) :test #'eq)))
                        (supersets-of-ruleset (rule-ruleset rule))))
               (keys-of-pattern (rule-pattern rule)))
         (setf (state-rules *state*)
	       (delete (rule-name rule) (state-rules *state*) :test #'eq))
         (makunbound (rule-name rule))))
  (rule-name rule))
  
(defun deactivate-rule (rule &key dont-ask-if-needed)
  (if (rulep rule)
      (cond ((or dont-ask-if-needed
		 (not (rule-needed rule))
		 (y-or-n-p "Rule ~a is needed, deactivate anyway?" (rule-name rule)))
	     
	     (mapc #'(lambda (key)
		       (mapc #'(lambda (ruleset)
				 (setf (get key ruleset)
				       (delete rule (get key ruleset) :test #'eq)))
			     (supersets-of-ruleset (rule-ruleset rule))))
		   (keys-of-pattern (rule-pattern rule)))
	     (setf (state-rules *state*)
		   (delete (rule-name rule) (state-rules *state*) :test #'eq))
	     (pushnew (rule-name rule) (state-deactivated-rules *state*)))))
  (rule-name rule))

(defun activate-rule (rule)
  (if (rulep rule)
      (cond ((InstallRule? rule)
             (InstallRule-1 rule)
             (setf (state-deactivated-rules *state*)
		   (delete
		     (rule-name rule) (state-deactivated-rules *state*)
		     :test #'eq)))
            (t (format *error-output* "~&Did not install rule ~s" rule))))
  nil)

(defun deactivate-rule-ruleset (rule ruleset)
  (if (rulep rule)
      (cond ((rule-needed rule)
             (format *error-output* "~&Rule ~s is needed~%" (rule-name rule)))
            (t (mapc #'(lambda (key)
			 (setf (get key ruleset)
			       (delete rule (get key ruleset) :test #'eq)))
                     (keys-of-pattern (rule-pattern rule))))))
  nil)

; Rule Preselection

(defun possible-rules-1 (data ruleset)
  (cond ((symbolp data) (get data ruleset))
        ((atom data) nil)
	((symbolp (car data)) (get (car data) ruleset))))

(defun possible-rules-2 (ruleset)
  (get '$else$ ruleset))

; Rule Selection
; ==============

(defun do-once (ruleset expr rule-bdgs-gen)
  (protocol-0 (car rule-bdgs-gen) expr ruleset)
  (let ((new-expr (FireRule (car rule-bdgs-gen) (cadr rule-bdgs-gen))))
    (protocol-2 (car rule-bdgs-gen) new-expr)
    new-expr))

(defun do-forever (ruleset expr rule-bdgs-gen)
  (protocol-0 (car rule-bdgs-gen) expr ruleset)
  (do ((new-expr))
      ((null rule-bdgs-gen) expr)
    (setq new-expr (FireRule (car rule-bdgs-gen) (cadr rule-bdgs-gen)))
    (protocol-2 (car rule-bdgs-gen) new-expr)
    (setq expr new-expr)
    (setq rule-bdgs-gen (choose-first expr ruleset))
    (if rule-bdgs-gen
        (protocol-1 (car rule-bdgs-gen) expr ruleset))))

(defun do-until-equal (ruleset expr rule-bdgs-gen)
  (protocol-0 (car rule-bdgs-gen) expr ruleset)
  (do ((new-expr) (finished? nil))
      (finished? expr)
    (setq new-expr (FireRule (car rule-bdgs-gen) (cadr rule-bdgs-gen)))
    (protocol-2 (car rule-bdgs-gen) new-expr)
    (setq rule-bdgs-gen (choose-first new-expr ruleset))
    (setq finished? (or (null rule-bdgs-gen)
                        (equal expr new-expr)))
    (setq expr new-expr)
    (if (not finished?)
        (protocol-1 (car rule-bdgs-gen) expr ruleset))))

(defun rule-match? (rule data)
  (funcall (rule-match-function rule) data))

(defun tryrules (data rules)
  (and rules
       (let ((bdgs.gen (rule-match? (car rules) data)))
         (cond (bdgs.gen (cons (car rules) bdgs.gen))
               (t (tryrules data (cdr rules)))))))
  

(defun choose-first (data ruleset)
  (let ((rule.bdgs.gen-1 (tryrules data (possible-rules-1 data ruleset)))
        (rule.bdgs.gen-2 (tryrules data (possible-rules-2 ruleset))))
    (if rule.bdgs.gen-1
        (if (and rule.bdgs.gen-2
                 (> (rule-priority (car rule.bdgs.gen-2))
                    (rule-priority (car rule.bdgs.gen-1))))
            rule.bdgs.gen-2
            rule.bdgs.gen-1)
        rule.bdgs.gen-2)))

; Rule Execution
; ==============

#|| CCC - Alte Version
(defun FireRule (rule *bindings*)
  (declare (special *bindings*))
  (let* ((result (ReplaceVariables (rule-replacement rule) *bindings*))
         (*bindings* (cons `($result$ . ,result) *bindings*)))
    (declare (special *bindings*))
    (eval (rule-action rule))
    result))
||#

(defun FireRule (rule bindings)
  (funcall (rule-replacement-function rule) bindings))

(defun ListNonSplice (thing)
  (if (ToBeSpliced? thing)
      (copy-list (ExpressionToSplice thing))
      (list thing)))
  
; value erhaelt das Attribut zugeord-
; net, dass er in der naechsthoeheren
; Stufe eingespliced werd. soll
; wenn value in der naechsttieferen Stufe
; Attribut splice erhalten hat:
; hier einsplicen
; zu splicende Liste kann ebenfalls
; values mit Attribut splice enthalten

#|| CCC - Alte Version
(defun ReplaceVariables (pattern bindings)
  (declare (special *bindings*))
  (cond ((null pattern) nil)
        ((atom pattern) pattern)
        ((NamElem? pattern) (lookup pattern bindings))
        ((IsEvaluateAndInsert? pattern)
         (funcall (ExpressionToBeEvaled pattern)))
        ((IsEvaluateAndSplice? pattern)
         (CreateToBeSplicedExpression (funcall (ExpressionToBeEvaled pattern))))
        ((NamSegm? pattern)
         (CreateToBeSplicedExpression (lookup pattern bindings)))
        ((IsEvaluateAndInsert? (car pattern))
         (let ((value (funcall (ExpressionToBeEvaled (car pattern)))))
           (if (ToBeSpliced? value)
               (non-copy-append (ExpressionToSplice value)
                                (ReplaceVariables (cdr pattern) bindings))
               (cons value (ReplaceVariables (cdr pattern) bindings)))))
        ((IsEvaluateAndSplice? (car pattern))
         (non-copy-append
           (mapcan 'ListNonSplice
                   (funcall (ExpressionToBeEvaled (car pattern))))
           (ReplaceVariables (cdr pattern) bindings)))
        ((IsVariable? (car pattern))
         (let ((binding (lookup (car pattern) bindings)))
           (cond ((IsSpliceVariable? (car pattern))
                  (non-copy-append binding
                                   (ReplaceVariables (cdr pattern)
                                                     bindings)))
                 (t (cons binding
                          (ReplaceVariables (cdr pattern) bindings))))))
        (t (cons (ReplaceVariables (car pattern) bindings)
                 (ReplaceVariables (cdr pattern) bindings)))))
||#

(defun non-copy-append (first rest)
  (if rest
      (append first rest)
      first))

; Protocol Stuff
; ==============
 
(defun protocol? ()
  *rule-trace*)

(defun protocol-type (rule)
  (if (or (null *rule-trace*)
          (member (car (rule-protocol-flag rule))
                  '(NeverProtocol Never)
                  :test
                  #'eq))
      'NoProtocol
      (if (member *rule-trace* '(on all))
          'ShortProtocol
          (car (rule-protocol-flag rule)))))

(defun protocol-0 (rule expr ruleset &aux (p-type (protocol-type rule)))
  (case p-type
    (NoProtocol)
    ((Protocol LongProtocol ShortProtocol)
     (cond (*print-header* (print-header *print-header*)
                           (setf *print-header* nil)))
     (terpri *protocolport*)
     (tab *lmar* *protocolport*)
     (msg (P *protocolport*) "Rule: " (rule-name rule) B ruleset N)
     (if (eq p-type 'ShortProtocol)
         (print-tab-form expr *lmar* *protocolport*)
         (pp-tab-form expr *lmar* *protocolport*))
     (terpri *protocolport*)
     (setq *lmar* (+ *lmar* 2))
     (setq *last-prot* 1))))

(defun protocol-1 (rule expr ruleset)
  (declare (ignore expr))
  (case (protocol-type rule)
    ((Protocol LongProtocol ShortProtocol)
     (terpri *protocolport*)
     (tab *lmar* *protocolport*)
     (msg (P *protocolport*) "Rule: " (rule-name rule) B ruleset N)
     (setq *lmar* (+ *lmar* 2))
     (setq *last-prot* 1))))
  
(defun protocol-2 (rule expr &aux (p-type (protocol-type rule)))
  (case p-type
    ((Protocol LongProtocol ShortProtocol)
     (and (eql *last-prot* 2)
          (terpri *protocolport*))
     (setq *lmar* (- *lmar* 2))
     (tab *lmar* *protocolport*)
     (msg (P *protocolport*) "==>" N)
     (if (eq p-type 'ShortProtocol)
         (print-tab-form expr *lmar* *protocolport*)
         (pp-tab-form expr *lmar* *protocolport*))
     (terpri *protocolport*)
     (setq *last-prot* 2))))

(defun print-tab-form (expr lmar &optional port)
  (tab lmar port)
  (print-form expr port))

(defun print-form (expr &optional port)
  (let ((*print-level* *prinlevel*) (*print-length* *prinlength*))
    (prin1 expr port)))

(defun pp-tab-form (expr lmar &optional port)
  (tab lmar port)
  (let ((*print-pretty* t))
    (prin1 expr port)))

(defun print-header (expr)
  (msg (P *protocolport*)
       N
       ";________________________________________"
       "_________________________________________"
       N)
  (terpri *protocolport*)
  (princ "Expression: " *protocolport*)
  (print-form expr *protocolport*)
  (terpri *protocolport*))
	
;;; CCC Bug in this Functions
(defmacro rule-trace (&optional trace-flag)
  (if trace-flag
      `(setq *rule-trace* ',trace-flag)
      '*rule-trace*))

(defmacro rule-untrace ()
  `(setq *rule-trace* nil))

; Fileausgabe
; ===========

(defun pprin1 (expr &optional stream)
  (let ((*print-pretty* t))
    (prin1 expr stream)))

(defun pp-file (file &optional $outport$)
  (unwind-protect (mapc #'(lambda (expr)
			    (if (and (listp expr)
				     (eq (car expr) '$separator$))
				(print-$sep expr :stream $outport$)
				(pprin1 expr $outport$))
			    (if (not (eq *read-function* 'ReadForm))
				(terpri $outport$)))
			file)))

(defun print-$sep (expr &key stream)
  (cond ((and (consp (cadr expr))
              (characterp (caadr expr)))
         (mapc #'(lambda (i)
                  (write-char i stream)
                  (and (member i '(#\Newline #\Return))
                       (FORCE-output stream)))
               (cadr expr))
         t)))       

; User Functions
; ==============

(defun tf (file-or-expr
                  &optional
                  (thefile "")
                  (protocolfile ""))
  (cond ((listp file-or-expr)
	 (let ((*protocolport* *standard-output*))
	   (transform-expr file-or-expr)))
	(t (with-open-file (*protocolport* (merge-pathnames
					     protocolfile
					     (merge-pathnames ".PR" file-or-expr))
					   :direction :output)
	     (transform-file
               file-or-expr
	       (merge-pathnames
		 thefile
		 (merge-pathnames ".OPT" file-or-expr)))))))

(defun transform-expr (expr)
  (protocol-msg 
       N
       ";________________________________________"
       "_________________________________________"
       N)
  (protocol-pp-form expr)
  (protocol-msg N)
  (let* ((*lmar* *prot-lmar*) (result (TransformStart expr)))
    (protocol-msg N "======>")
    (protocol-pp-form result)
    (protocol-msg N)
    result))

#+lispm
(defun transform-file (file to-file)
  (declare (special *state*))
  (with-open-file (to-port to-file :direction :output)
    (msg (P *protocolport*)
	 "; Protocol of Transformation Process" N
	 "; ==================================" N
	 "; Transformed file: " (namestring to-file) N
	 "; Transformed version of file: " (namestring file) N
	 "; TransLisp version: " (get 'translisp 'version) N
	 "; Rules File: " (state-rulefile *state*) N
	 "; Rules version: " (get (state-rulefile-abbr *state*) 'version) N
	 "; Type of transformation: " (state-goal *state*) N
	 "; Date: " (date) N)
    (msg (P to-port)
	 "; Outputfile of Transformation Process" N
	 "; ====================================" N
	 "; Transformed version of file: " (namestring file) N
	 "; Rules File: " (state-rulefile *state*) N
	 "; Rules version: " (get (state-rulefile-abbr *state*) 'version) N
	 "; Type of transformation: " (state-goal *state*) N
	 "; Date: " (date) N)
    (let ((file-expr (read-commented-file file)))
      (pp-file (TransformFile file-expr) to-port)))
  t)

#-lispm
(defun transform-file (file to-file)
  (declare (special *state*))
  (let ((aux-file (merge-pathnames ".tmp-lisp" file)))
    (with-open-file (to-port aux-file :direction :output)
      (msg (P *protocolport*)
	   "; Protocol of Transformation Process" N
	   "; ==================================" N
	   "; Transformed file: " (namestring to-file) N
	   "; Transformed version of file: " (namestring file) N
	   "; TransLisp version: " (get 'translisp 'version) N
	   "; Rules File: " (state-rulefile *state*) N
	   "; Rules version: " (get (state-rulefile-abbr *state*) 'version) N
	   "; Type of transformation: " (state-goal *state*) N
	   "; Date: " (date) N)
      (msg (P to-port)
	   "; Outputfile of Transformation Process" N
	   "; ====================================" N
	   "; Transformed version of file: " (namestring file) N
	   "; Rules File: " (state-rulefile *state*) N
	   "; Rules version: " (get (state-rulefile-abbr *state*) 'version) N
	   "; Type of transformation: " (state-goal *state*) N
	   "; Date: " (date) N)
      (let ((file-expr (read-commented-file file)))
	(pp-file (TransformFile file-expr) to-port)
	))
    (qu*-filter-file aux-file to-file)
    (delete-file aux-file))
    t)
  
(defun rules (&optional ruleset car)
  (cond (car
	 (cond (ruleset (mapcan #'(lambda (rule)
                                    (cond (t (list (rule-name rule)))))
				(possible-rules-1 (list car) ruleset)))
	       (t (msg "Bitte ruleset angeben: " N) (rulesets))))
        (ruleset ((lambda (fn l)
                    (mapcan #'(lambda (x)
				(cond ((funcall fn x) (list x))
				      (t nil)))
                            l))
                  #'(lambda (rule-name)
		      (if (listp (rule-ruleset (get-rule rule-name)))
			  (member ruleset
				  (rule-ruleset (get-rule rule-name))
				  :test
				  #'eq)
			  (eq ruleset (rule-ruleset (get-rule rule-name)))))
                  (state-rules *state*)))
        (t (state-rules *state*))))

(defun rulesets ()
  (state-rulesets *state*))

(defmacro do-not-use (list-of-items)
  `(mapc
     #'(lambda (item)
	(let (((rulename ruleset)
	       (cond ((listp item) item)
		     (t (list item nil)))))
	  (if (boundp rulename)
	      (if (not (rule-needed (get-rule rulename)))
		  (if ruleset
		      (deactivate-rule-ruleset (get-rule rulename) ruleset)
		      (deactivate-rule (get-rule rulename)))
		  (msg (P *error-output*) "Rule " rulename " is needed" N))
	      (msg (P *error-output*) "Rule with name " rulename " unknown" N))))
     ',list-of-items))


; Filehandling
; ============

(defun read-commented-file (file)
  (declare (special *read-into-package*))
  (let ((*readtable* (copy-readtable))
	(*package* *read-into-package*))
  (declare (special *readtable* *package*))
  (with-open-file (port file :direction :input)
    (unwind-protect
	(progn
	  (if *readtable-hook*
	      (eval *readtable-hook*))
	  (do ((in (funcall *read-function* port)
		   (funcall *read-function* port))
	       (t-list '()))
	      ((null in) (nreverse t-list))
	    (push in t-list)))
      (if *readtable-reset-hook*
	  (eval *readtable-reset-hook*))))))

(defun readln (&optional port)
  (do ((charn (peek-char nil port nil -1) (peek-char nil port nil -1))
       (line '()))
      ((member charn '(#\Return #\Newline))
       (nreverse (cons (read-char port nil -1) line)))
    (push (read-char port nil -1) line)))

;;; Output-Filter
;;; =============

#-lispm

;;; Output

;;; CCC Bug: `(foo . ,x) is not filtered correct
;;; CCC      this is fixed
;;; another bug is not to be solved easily
;;;  (flr:qu* ...) in a data position (e.g. as key in a case form) is
;;;  erronously printed as `... which is not to read correctly

(progn

(defun qu*-filter-file (file1 file2)
  (with-open-file (in file1 :direction :input)
    (with-open-file (out file2 :direction :output)
      (qu*-filter-stream in out))))

(defun qu*-filter-stream (instream outstream)
  (let ((in-char))
    (loop (setq in-char (peek-char nil instream nil nil))
	  (unless in-char
	    (return "finished"))
	(qu*-filter-form in-char instream outstream))))

(defun qu*-filter-form (input instream outstream &optional (bq-level 0))	;expecting form
   (cond ((null input) (return-from qu*-filter-form nil))
	 ((eql Input #\))
		  (error "Just read a right paranthesis outside lisp expression"))
	 ((eql input #\;)
	     (write-line (read-line instream nil nil) outstream))
	 ((eql input #\\)			; this is for #\) etc.
	     (write-char (read-char instream) outstream)
	     (write-char (read-char instream) outstream))
	 ((member  input '(#\\ #\" #\|))
	     (prin1 (read-preserving-whitespace instream) outstream))
	 ((eql input #\( )
	     (read-char instream)
	     (cond ((eql (peek-char nil instream) #\))
		    (read-char instream)
		    (format outstream "()"))
		   (t (qu*-filter-list-car instream outstream bq-level))))
	 (t (write-char (read-char instream) outstream)))	;e.g. Blanks, NewLines
   t)

(defun qu*-filter-list-car (instream outstream bq-level)
  (let ((input (peek-char nil instream)))
    (cond ((eql input #\))
	   (error "Expecting an expression after comma but no rpar")
;	    (read-char instream)
;	    (format outstream "()")
	   )
	  ((eql input #\f)
	   (setq input (read-preserving-whitespace instream))	
	   (cond ((and (plusp bq-level)
		       (member input
			       '(flr:|,| flr:|,@| flr:|,.| flr:|,?| flr:|,*|)))
		  (qu*-filter-comma-args input instream outstream (1- bq-level)))
		 ((eq input 'flr:qu*)
		  (qu*-filter-qu*-args instream outstream (1+ bq-level)))
		 (t (write-char #\( outstream)
		    (prin1 input outstream)
		    (qu*-filter-up-to-rpar
		      (peek-char nil instream) instream outstream bq-level)
		    (write-char #\) outstream))
		 ))
	  (t (write-char #\( outstream)
	     (qu*-filter-up-to-rpar (peek-char nil instream) instream outstream bq-level)
	     (write-char #\) outstream)
	     ))))

(defun qu*-filter-up-to-rpar (input instream outstream bq-level)
  (cond ((eql input #\))
	 (read-char instream))
	((eql input #\f)
	 (setq input (read-preserving-whitespace instream))	
	 (cond ((and (plusp bq-level)
		     (member input
			     '(flr:|,| flr:|,@| flr:|,.| flr:|,?| flr:|,*|)))
		(format outstream ". ")
		(qu*-filter-comma-args input instream outstream (1- bq-level)))
	   ((eql input 'flr:qu*)
	    (format outstream ". ")
	    (qu*-filter-qu*-args instream outstream (1+ bq-level)))
	   (t 
	     (prin1 input outstream)
	     (qu*-filter-up-to-rpar (peek-char nil instream) instream outstream bq-level) 
	     )))
	(t (qu*-filter-form input instream outstream bq-level)
	   (qu*-filter-up-to-rpar (peek-char nil instream) instream outstream bq-level))))

(defun qu*-filter-after-qu* (input in out bq-level)
  (do ((input input (peek-char nil in)))
      ((eql input #\)) (read-char in))
    (qu*-filter-form input in out bq-level)))
    
(defun qu*-filter-comma-args (comma in out bq-level)	;expecting args to comma
  (princ (symbol-name comma) out)
  (let ((ch (peek-char t in t nil)))
    (cond ((eql ch #\.)
	   (read-char in)
	   (qu*-filter-after-qu* (peek-char t in nil nil) in out bq-level)
						; skip whitespace chars 
;	   (unless (eql (read-char in) #\))
;	     (error "Expecting a closing par after comma form just read"))
	  )
	  (t (qu*-filter-list-car  in out bq-level)))))

(defun qu*-filter-qu*-args (in out bq-level)		; expecting arg to qu*
  (write-char #\` out)
  (qu*-filter-after-qu* (peek-char t in nil nil) in out bq-level))
						;
) ;end #-lispm


(setf (get 'translisp 'version) '4.15)
