;;; -*- Mode: LISP; Syntax: Common-lisp; Package: User; Base: 10 -*-

;;; Rule Base: Franz to Common Lisp

;;;----------------------------------------------------------------------
;;; Package
;;;----------------------------------------------------------------------

; (in-package 'franz-to-common) ; in a version to come

;;;------------------------------------------------------------------------------------------
;;; Laden der Read-Macros und Rule-Macros
;;;------------------------------------------------------------------------------------------

(eval-when (compile eval)
  (unless (get 'tl::init-rule-handling 'tl::version)
    (load "../kernel/init-rule-handling")))

(init-rule-file)

;;;------------------------------------------------------------------------------------------
;;; Definition der Anwendung franz-to-common
;;;------------------------------------------------------------------------------------------

(setf (get 'franz-to-common :ruleclasses)
      '(franz-to-common franz-trigger		; die entsprechenden Regeln 
	filerules franz-objtalk-trigger))	; geladen

(setup-application 'franz-to-common)

(setf (state-rulefile *state*) "franz-to-common-rules")
(setf (state-rulefile-abbr *state*) 'franz-to-common-rules)

;;;------------------------------------------------------------------------------------------
;;; Zuladen der Regeln zur Fileverarbeitung
;;;------------------------------------------------------------------------------------------

(include-file-rules)

;;;------------------------------------------------------------------------------------------
;;; Definition der Reader Syntax zum Einlesen zu transformierender Files
;;;------------------------------------------------------------------------------------------

(setq *readtable-hook* '(flr:set-franz-input-syntax))	
(setq *readtable-reset-hook* '(flr:reset-franz-input-syntax))

(setq *read-into-package* (find-package 'fl)) ; in which package to
						; read files to transform	

;;;------------------------------------------------------------------------------------------
;;; Some global Variables
;;;------------------------------------------------------------------------------------------

(defvar *keyword-package* (find-package 'keyword))

(setq *colon-substitution-char* #\-)

;;;------------------------------------------------------------------------------------------
;;; Definition der Eigenschaften von FranzLisp Funktionen
;;;------------------------------------------------------------------------------------------

(eval-when (load eval)
  (mapc #'(lambda (foo)
	    (setf (get foo 'predicate) t))
        '(fl::|dtpr| fl::|listp| fl::|tailp| fl::|arrayp| fl::|atom|
	  fl::|bcdp| fl::|bigp| fl::|hunkp| fl::|stringp| fl::|symbolp| fl::|valuep|
	  fl::|type| fl::|typep| fl::|signp| fl::|eq| fl::|neq| fl::|equal|
	  fl::|eqstr| fl::|not| fl::|null| fl::|boundp| fl::|alphalessp| fl::|hunkp|
	  fl::|numberp| fl::|numbp| fl::|fixp| fl::|floatp| fl::|evenp| fl::|oddp|
	  fl::|zerop| fl::|onep| fl::|plusp| fl::|minusp| fl::|greaterp| fl::|>|
	  fl::|>&| fl::|lessp| fl::|<| fl::|<&| fl::|=| fl::|=&| fl::|purep|
	  fl::|portp|))
  (mapc '(lambda (foo)
           (setf (get foo 'discipline) 'lambda))
        '(fl::|$patom1| fl::|$prd1| fl::|$prdf| fl::|$prpr|
	  fl::|%:format-handler|           fl::|&:format-handler| fl::* fl::*$
	  fl::|*:format-handler| fl::|*array| fl::|*break|           fl::|*dif|
	  fl::|*invmod| fl::|*makhunk| fl::|*mod| fl::|*process|
	  fl::|*process-receive| fl::|*process-send| fl::|*quo| fl::|*rplacx|
	  fl::|*rset|           fl::|*throw| fl::+ fl::+$ fl::- fl::-$ fl::/ fl::/$
	  fl::1+ fl::1+$ fl::1- fl::1-$           fl::|;:format-handler| fl::< fl::<&
	  fl::= fl::=& fl::> fl::>& fl::|?th| fl::|Cnth|           fl::|Divide|
	  fl::|Emuldiv| fl::|I-throw-err| fl::|Sharpm'g00150| fl::|Sharpm'g00160|
	  fl::|Sharpm+g00125| fl::|Sharpm+g00139| fl::|Sharpm,g00155|
	  fl::|Sharpm,g00164|           fl::|Sharpm-g00132| fl::|Sharpm-g00145|
	  fl::|Sharpm/g00139| fl::|Sharpm/g00151|           fl::|SharpmMg00176|
	  fl::|SharpmMg00182| fl::|Sharpm\\g00160| fl::|Sharpm\\g00168|
	  fl::|Sharpm^g00144| fl::|Sharpm^g00155| fl::|Sharpmog00097|
	  fl::|Sharpmog00113|           fl::|Sharpmtg00171| fl::|Sharpmtg00178|
	  fl::|Sharpmxg00102| fl::|Sharpmxg00117|           fl::|[:format-handler|
	  fl::\\ fl::|]:format-handler| fl::^ fl::|^:format-handler|
	  fl::|a:format-handler| fl::|abs| fl::|absval| fl::|acos| fl::|add|
	  fl::|add-syntax-class| fl::|add1| fl::|aexplode| fl::|aexplodec|
	  fl::|aexploden|           fl::|allocate| fl::|allsym| fl::|alphalessp|
	  fl::|any-zerop|           fl::|append| fl::|append1| fl::|apply|
	  fl::|apply*| fl::|arg| fl::|argv|           fl::|arrac-nD| fl::|arrac-oneD|
	  fl::|arrac-twoD| fl::|array-type| fl::|arraydims|           fl::|arrayp|
	  fl::|arrayref| fl::|ascii| fl::|asin| fl::|assoc| fl::|assq| fl::|atan|
	  fl::|atom| fl::|attach| fl::|back-quote-ch-macro|
	  fl::|back-quote-comma-macro|           fl::|back=quotify|
	  fl::|baktraceprint| fl::|bcdad| fl::|bcdcall| fl::|bcdp|
	  fl::|bignum-leftshift| fl::|bignum-to-list| fl::|bigp| fl::|boole|
	  fl::|boundp|           fl::|bq-print-macro| fl::|break-err-handler|
	  fl::|c:format-handler| fl::|caaaar| fl::|caaadr| fl::|caaar| fl::|caadar|
	  fl::|caaddr| fl::|caadr| fl::|caar|           fl::|cadaar| fl::|cadadr|
	  fl::|cadar| fl::|caddar| fl::|cadddr| fl::|caddr| fl::|cadr| fl::|car|
	  fl::|cdaaar| fl::|cdaadr| fl::|cdaar| fl::|cdadar| fl::|cdaddr| fl::|cdadr|
	  fl::|cdar| fl::|cddaar| fl::|cddadr| fl::|cddar| fl::|cdddar| fl::|cddddr|
	  fl::|cdddr| fl::|cddr| fl::|cdr| fl::|cfasl| fl::|ch10:format-handler|
	  fl::|charcnt| fl::|charsinbuf|           fl::|chdir| fl::|close|
	  fl::|command-line-args| fl::|compile| fl::|concat|           fl::|concatl|
	  fl::|condclosefile| fl::|cons| fl::|construct-list*| fl::|copy|
	  fl::|copyint*| fl::|copysymbol| fl::|cos| fl::|cprintf| fl::|cputim|
	  fl::|cpy1|           fl::|cr| fl::|cvttofranzlisp| fl::|cvttointlisp|
	  fl::|cvttomaclisp|           fl::|cvttoucilisp| fl::|cxr|
	  fl::|d:format-handler| fl::|de-compose|           fl::|debug-err-handler|
	  fl::|debugging| fl::|defmcrooption| fl::|defmcrosrch|
	  fl::|defsharp-expand| fl::|delete| fl::|delq| fl::|deref| fl::|diff|
	  fl::|difference| fl::|double-to-float| fl::|drain| fl::|dsubst| fl::|dtpr|
	  fl::|english-print| fl::|english-print-thousand| fl::|eq| fl::|eq-tyimode|
	  fl::|eqstr| fl::|equal| fl::|err| fl::|err-with-message| fl::|error|
	  fl::|ev-arraycall| fl::|eval| fl::|eval1| fl::|evalframe| fl::|evalhook|
	  fl::|evenp|           fl::|exece| fl::|exit| fl::|exp| fl::|explode|
	  fl::|explodec| fl::|exploden|           fl::|expt| fl::|exvi|
	  fl::|f:format-handler| fl::|fact| fl::|fake| fl::|fasl|
	  fl::|fasl-a-file| fl::|fclosure| fl::|fclosure-alist|
	  fl::|fclosure-function|           fl::|fclosure-list| fl::|fclosurep|
	  fl::|feature-present| fl::|ferror| fl::|ffasl|           fl::|fileopen|
	  fl::|filepos| fl::|filestat| fl::|filestat:atime|
	  fl::|filestat:ctime| fl::|filestat:dev| fl::|filestat:gid|
	  fl::|filestat:ino|           fl::|filestat:mode| fl::|filestat:mtime|
	  fl::|filestat:nlink| fl::|filestat:rdev|           fl::|filestat:size|
	  fl::|filestat:type| fl::|filestat:uid| fl::|fillarray|
	  fl::|fillarrayarray| fl::|fix| fl::|fixp| fl::|flatc| fl::|flatsize|
	  fl::|float|           fl::|float-to-double| fl::|floatp| fl::|fork|
	  fl::|format| fl::|format-binpr|           fl::|format-binpr1|
	  fl::|format-ctl-ascii| fl::|format-ctl-justify|
	  fl::|format-ctl-list| fl::|format-ctl-octal| fl::|format-ctl-op|
	  fl::|format-ctl-repeat-char| fl::|format-ctl-string| fl::|format-prc|
	  fl::|format-test| fl::|format:anyradix-printer| fl::|format:fresh-line|
	  fl::|format:nsubstring| fl::|format:patom| fl::|format:print|
	  fl::|format:printorpatom| fl::|format:string-search-char| fl::|format:terpr|
	  fl::|format:tyo| fl::|framedump| fl::|franz-reset| fl::|franz-top-level|
	  fl::|freturn| fl::|frexp| fl::|fseek| fl::|funcall| fl::|funcallhook|
	  fl::|g00249::car| fl::|g00255::caar| fl::|g00261::cadr| fl::|g00267::cdr|
	  fl::|g00273::cdar| fl::|g00279::cddr| fl::|g00285::cxr| fl::|g00291::vref|
	  fl::|g00297::vrefi-byte| fl::|g00303::vrefi-word| fl::|g00309::vrefi-long|
	  fl::|g00315::nth| fl::|g00321::nthelem| fl::|g00327::nthcdr|
	  fl::|g00333::arraycall|           fl::|g00339::get| fl::|g00345::plist|
	  fl::|g00351::symeval| fl::|g00357::arg|           fl::|g00363::args|
	  fl::|g:format-handler| fl::|gensym| fl::|get| fl::|get_pname|
	  fl::|getaccess| fl::|getaddress| fl::|getaux| fl::|getchar| fl::|getcharn|
	  fl::|getd| fl::|getdata| fl::|getdelta| fl::|getdisc| fl::|getentry|
	  fl::|getenv|           fl::|getl| fl::|getlength| fl::|getsyntax|
	  fl::|greaterp| fl::|haipart|           fl::|hashtabstat| fl::|haulong|
	  fl::|hunk| fl::|hunk-to-list| fl::|hunkp|           fl::|hunksize|
	  fl::|implode| fl::|includef| fl::|infile| fl::|initsym|
	  fl::|initsym1| fl::|insert| fl::|int:cfasl| fl::|int:fasl|
	  fl::|int:fclosure-stack-stuff| fl::|int:fileopen| fl::|int:franz-call|
	  fl::|int:getsyntax| fl::|int:infile| fl::|int:memreport| fl::|int:outfile|
	  fl::|int:setsyntax| fl::|int:showstack| fl::|int:vector-range-error|
	  fl::|int:vref|           fl::|int:vset| fl::|int:vsize|
	  fl::|int:wrong-number-of-args-error| fl::|intern|           fl::|killcopy|
	  fl::|kwote| fl::|lambdacvt| fl::|last| fl::|lconc| fl::|ldiff|
	  fl::|length| fl::|lessp| fl::|lexpr-funcall| fl::|lineread| fl::|list|
	  fl::|list-to-bignum| fl::|listarray| fl::|listp| fl::|load|
	  fl::|load-autorunobject|           fl::|load-file| fl::|load-if-needed|
	  fl::|log| fl::|lsh| fl::|lsubst|           fl::|macroexpand|
	  fl::|make-fclosure-with-alist| fl::|make-list-array|
	  fl::|makereadtable| fl::|makhunk| fl::|maknam| fl::|maknum| fl::|makunbound|
	  fl::|map| fl::|mapc| fl::|mapcan| fl::|mapcar| fl::|mapcon| fl::|maplist|
	  fl::|marray| fl::|max| fl::|member| fl::|memq| fl::|merge| fl::|merge1|
	  fl::|mergelists| fl::|mfunction| fl::|min| fl::|minus| fl::|minusp|
	  fl::|mod|           fl::|monitor| fl::|msg-print| fl::|msg-tyo-char|
	  fl::|nconc| fl::|ncons|           fl::|nequal| fl::|new-sharp-sign-macro|
	  fl::|new-vector| fl::|new-vectori-byte|           fl::|new-vectori-long|
	  fl::|new-vectori-word| fl::|newsym| fl::|not| fl::|nreconc|
	  fl::|nreverse| fl::|nth| fl::|nthcdr| fl::|nthchar| fl::|nthelem| fl::|null|
	  fl::|numberp| fl::|numbp| fl::|nwritn| fl::|o:format-handler| fl::|oblist|
	  fl::|oddp| fl::|old-baktrace| fl::|old-showstack| fl::|oldsym| fl::|onep|
	  fl::|opval| fl::|outfile| fl::|p:format-handler| fl::|patom| fl::|plist|
	  fl::|plus|           fl::|plusp| fl::|pntlen| fl::|portp| fl::|pp-form|
	  fl::|pp-function| fl::|pp-prop|           fl::|pp-value| fl::|princ|
	  fl::|print| fl::|print-lambda| fl::|printaccross|
	  fl::|printblanks| fl::|printdef| fl::|printmacrochar| fl::|printprog|
	  fl::|probef|           fl::|product| fl::|prog1| fl::|prtpagesused|
	  fl::|ptime| fl::|ptr| fl::|purcopy|           fl::|purep| fl::|putaccess|
	  fl::|putaux| fl::|putd| fl::|putdata| fl::|putdelta|           fl::|putdisc|
	  fl::|putlength| fl::|putprop| fl::|pv%| fl::|q:format-handler|
	  fl::|quote!-expr-mac| fl::|quotient| fl::|quotify| fl::|quotify1|
	  fl::|r:format-handler| fl::|random| fl::|rassq| fl::|ratom| fl::|read|
	  fl::|read-in-lisprc-file| fl::|readc| fl::|readlist| fl::|remainder|
	  fl::|rematom|           fl::|remob| fl::|remove| fl::|removeaddress|
	  fl::|remprop| fl::|remq| fl::|remsym|           fl::|remsym1| fl::|replace|
	  fl::|reset| fl::|return| fl::|reverse| fl::|roman-char|
	  fl::|roman-step| fl::|rot| fl::|rplaca| fl::|rplacd| fl::|rplacx|
	  fl::|s:format-handler| fl::|scons| fl::|segment| fl::|set|
	  fl::|set-in-fclosure|           fl::|setarg| fl::|setf-check-cad+r|
	  fl::|setplist| fl::|setsyntax| fl::|shell|
	  fl::|showstack-baktrace| fl::|signal| fl::|sin| fl::|sizeof| fl::|sload|
	  fl::|small-segment| fl::|some| fl::|sort| fl::|sortcar| fl::|sortcarhelp|
	  fl::|sortmerge| fl::|splitlist| fl::|sprintf| fl::|sqrt|
	  fl::|sticky-bignum-leftshift| fl::|storeintern| fl::|stringp|
	  fl::|strip-extension|           fl::|sub1| fl::|sublis| fl::|sublishelp|
	  fl::|subpair| fl::|subpr| fl::|subst|           fl::|subst-eq|
	  fl::|subst-eqp| fl::|substeq| fl::|substequal| fl::|substring|
	  fl::|substringn| fl::|sum| fl::|symbolp| fl::|symeval|
	  fl::|symeval-in-fclosure|           fl::|symstat| fl::|sys:access|
	  fl::|sys:chmod| fl::|sys:fpeint-serv|           fl::|sys:gethostname|
	  fl::|sys:getpid| fl::|sys:getpwnam| fl::|sys:getpwnam-dir|
	  fl::|sys:int-serv| fl::|sys:link| fl::|sys:time| fl::|sys:unlink|
	  fl::|syscall|           fl::|tab| fl::|tailp| fl::|tconc| fl::|termcapexe|
	  fl::|termcapinit| fl::|terpr|           fl::|terpri| fl::|tilde-expand|
	  fl::|time-string| fl::|times| fl::|top-level|           fl::|trace-funp|
	  fl::|truename| fl::|tyi| fl::|tyimode| fl::|tyipeek| fl::|tyo|
	  fl::|type| fl::|typep| fl::|uconcat| fl::|uncompile|
	  fl::|undef-func-handler|           fl::|untyi| fl::|username-to-dir|
	  fl::|username-to-dir-flush-cache| fl::|valuep|           fl::|vector|
	  fl::|vector-dump| fl::|vectori-byte| fl::|vectori-long|
	  fl::|vectori-word| fl::|vectorip| fl::|vectorp| fl::|vget| fl::|vprop|
	  fl::|vputprop| fl::|vref| fl::|vrefi-byte| fl::|vrefi-long| fl::|vrefi-word|
	  fl::|vset| fl::|vseti-byte| fl::|vseti-long| fl::|vseti-word| fl::|vsetprop|
	  fl::|vsize| fl::|vsize-byte| fl::|vsize-word| fl::|wait| fl::|without-path|
	  fl::|x:format-handler| fl::|xcons| fl::|zapline| fl::|zerop|
	  fl::|{:format-handler| fl::|\|:format-handler| fl::|}:format-handler|
	  fl::|~:format-handler|))
  (mapc '(lambda (foo)
           (setf (get foo 'discipline) 'like-lambda))
        '(fl::|:=| fl::|<=| fl::|<=&| fl::|>=| fl::|>=&| fl::|*catch|
	  fl::|errset| fl::|list*| fl::|listify| fl::|sassoc| fl::|sassq| fl::|signp|
	  fl::|setq| fl::|arraycall| fl::|go| fl::|throw|))
  (mapc '(lambda (foo)
           (setf (get foo 'pure-function) t))
        '(fl::|cons| fl::|xcons| fl::|ncons| fl::|list| fl::|append|
	  fl::|append1| fl::|bignum-to-list| fl::|list-to-bignum| fl::|dtpr|
	  fl::|listp| fl::|tailp| fl::|length| fl::|car| fl::|cadr| fl::|caddr|
	  fl::|cadddr| fl::|caddddr| fl::|cadar| fl::|caar| fl::|caadr| fl::|cdr|
	  fl::|cdar| fl::|cddr| fl::|cdddr| fl::|cddddr| fl::|nthcdr| fl::|nthelem|
	  fl::|nth| fl::|last| fl::|ldiff| fl::|remove| fl::|subst| fl::|lsubst|
	  fl::|subpair| fl::|reverse| fl::|arrayp| fl::|atom| fl::|bcdp| fl::|bigp|
	  fl::|hunkp| fl::|stringp| fl::|symbolp| fl::|valuep| fl::|type| fl::|typep|
	  fl::|signp| fl::|eq| fl::|neq| fl::|equal| fl::|eqstr| fl::|not| fl::|null|
	  fl::|member| fl::|memq| fl::|uconcat| fl::|concat| fl::|concatl|
	  fl::|implode| fl::|maknam| fl::|copysymbol| fl::|ascii| fl::|boundp|
	  fl::|alphalessp| fl::|get_pname| fl::|plist| fl::|getd| fl::|getchar|
	  fl::|nthchar| fl::|getcharn| fl::|substring| fl::|substringn| fl::|aexplode|
	  fl::|explode| fl::|aexplodec| fl::|explodec| fl::|aexploden| fl::|exploden|
	  fl::|getaccess| fl::|getaux| fl::|getdelta| fl::|getdata| fl::|getlength|
	  fl::|arrayref| fl::|arraycall| fl::|arraydims| fl::|listarray| fl::|hunkp|
	  fl::|hunksize| fl::|cxr| fl::|getdisc| fl::|getentry| fl::|assoc| fl::|assq|
	  fl::|sublis| fl::|plist| fl::|get| fl::|getl| fl::|bcdad| fl::|copy|
	  fl::|copyint*| fl::|cpy1| fl::|getaddress| fl::|macroexpand| fl::|ptr|
	  fl::|kwote| fl::|scons| fl::|add| fl::|plus| fl::|sum| fl::|+| fl::|add1|
	  fl::|1+| fl::|diff| fl::|difference| fl::|-| fl::|sub1| fl::|1-| fl::|minus|
	  fl::|product| fl::|times| fl::|*| fl::|quotient| fl::|/| fl::|*quo|
	  fl::|Divide| fl::|Emuldiv| fl::|numberp| fl::|numbp| fl::|fixp| fl::|floatp|
	  fl::|evenp| fl::|oddp| fl::|zerop| fl::|onep| fl::|plusp| fl::|minusp|
	  fl::|greaterp| fl::|>| fl::|>&| fl::|lessp| fl::|<| fl::|<&| fl::|=|
	  fl::|=&| fl::|cos| fl::|sin| fl::|acos| fl::|asin| fl::|atan| fl::|haipart|
	  fl::|haulong| fl::|bignum-leftshift| fl::|sticky-bignum-leftshift|
	  fl::|boole| fl::|lsh| fl::|rot| fl::|abs| fl::|absval| fl::|expt| fl::|fact|
	  fl::|fix| fl::|float| fl::|log| fl::|max| fl::|min| fl::|mod|
	  fl::|remainder| fl::|*mod| fl::|sqrt| fl::|arg| fl::|freturn| fl::|frexp|
	  fl::|function| fl::|getdisc| fl::|listify| fl::|oblist| fl::|purep|
	  fl::|flatc| fl::|flatsize| fl::|nwritn| fl::|pntlen| fl::|portp|
	  fl::|probef| fl::|readlist| fl::|tyipeek| fl::|argv| fl::|fake| fl::|getenv|
	  fl::|hashtabstat| fl::|maknum| fl::|ptime| fl::|sizeof|
	  fl::|getsyntax|))
  (mapc '(lambda (function)
           (setf (get function 'cons-cell-generator) t))
        '(fl::|reverse| fl::|ncons| fl::|list| fl::|append1| fl::|remove|
	  fl::|mapcar| fl::|maplist| fl::|copy| fl::|subst| fl::|lsubst| fl::|Divide|
	  fl::|listify| fl::|aexplode| fl::|explode| fl::|aexplodec| fl::|explodec|
	  fl::|aexploden| fl::|exploden| fl::|listarray| fl::|oblist| fl::|opval|
	  fl::|ptime| fl::|bignum-to-list|)))

(setq *common-special-variables*
      '(* ** *** *applyhook* *break-on-warnings* *debug-io*
        *default-pathname-defaults* *error-output* *evalhook* *features*
        *load-verbose* *macroexpand-hook* *modules* *package* *print-array*
        *print-base* *print-case* *print-circle* *print-escape* *print-gensym*
        *print-length* *print-level* *print-pretty* *print-radix* *query-io*
        *random-state* *read-base* *read-default-float-format* *read-suppress*
        *readtable* *standard-input* *standard-output* *terminal-io*
        *trace-output* + ++ +++ - / // /// = applyhook array-dimension-limit
        array-rank-limit array-total-size-limit boole-1 boole-2 boole-and
        boole-andc1 boole-andc2 boole-c1 boole-c2 boole-clr boole-eqv boole-ior
        boole-nand boole-nor boole-orc1 boole-orc2 boole-set boole-xor
        call-arguments-limit char-bits-limit char-code-limit char-control-bit
        char-font-limit char-hyper-bit char-meta-bit char-super-bit
        double-float-epsilon double-float-negative-epsilon evalhook
        internal-time-units-per-second lambda-list-keywords
        lambda-parameters-limit least-negative-double-float
        least-negative-long-float least-negative-short-float
        least-negative-single-float least-positive-double-float
        least-positive-long-float least-positive-short-float
        least-positive-single-float long-float-epsilon
        long-float-negative-epsilon most-negative-double-float
        most-negative-fixnum most-negative-long-float most-negative-short-float
        most-negative-single-float most-positive-double-float
        most-positive-fixnum most-positive-long-float most-positive-short-float
        most-positive-single-float multiple-values-limit pi prin1 room
        short-float-epsilon short-float-negative-epsilon single-float-epsilon
        single-float-negative-epsilon))

;;;------------------------------------------------------------------------------------------
;;; Hilfsfunctions:
;;;------------------------------------------------------------------------------------------

(defun lambdap (foo)
  (or (and (listp foo)
           (eq (car foo) 'fl::|lambda|))
      (and (symbolp foo)
           (or (functionp foo)
	       (member (get foo 'discipline) '(lambda like-lambda))))))

(defun user-defined-p (foo)
  (and (symbolp foo)
       (get foo 'user-defined)))

(defun no-visible-comma-p (expr &optional (level 0))
  (cond ((atom expr))
	((member (car expr) '(flr::|,| flr::|,@| flr::|,.|) :test #'eq)
	 (unless (zerop level)
	   (no-visible-comma-p (cdr expr) (1- level))))
	((eq (car expr) 'flr::qu*) (no-visible-comma-p (cadr expr) (1+ level)))
	((no-visible-comma-p (car expr) level)
	 (no-visible-comma-p (cdr expr) level))))

(defun no-comma-splice (elem)
  (not (or (eq elem 'flr::|,|)
	   (and (listp elem)
		(member (car elem) '(flr::|,| flr::|,@| flr::|,.|))))))

(defun pure-function? (foo)
  (and (symbolp foo)
       (get foo 'pure-function)))

(defun clear-all-cl-symbol-properties (package)
  (do-symbols (sym package "cl-symbols cleared")
    (remprop sym :cl-symbol)))

(defun clear-cl-symbol-property (sym)
  (remprop sym :cl-symbol))

;;;------------------------------------------------------------------------------------------
;;; Wichtige Hilfsfunktion Symbol
;;;
;;;   fl::|foo| ==> foo
;;;   fl::|Foo| ==> |fOO|
;;;   fl::|FOO| ==> |foo|
;;;   fl::|:key| ==> :key
;;;   
;;; Resultierendes Symbol wird unter der Property :cl-user abgelegt.
;;;------------------------------------------------------------------------------------------

(defruleset symbol (expr))

(defrule convert-t Symbol 4
   fl::|t|
   ==>
   t
   (Franz-to-Common)
   (Neverprotocol)
 ((author "Matthias Ressel") (created "03.10.1987, 20:55")))

(defrule convert-nil Symbol 4
   fl::|nil|
   ==>
   nil
  (Franz-to-Common)
   (Neverprotocol)
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 18:54")
  (author "Matthias Ressel") (created "03.10.1987, 20:55")))

(defrule convert-quote Symbol 4
   fl::|quote|
   ==>
   quote
   (Franz-to-Common)
   (Neverprotocol)
 ((author "Matthias Ressel") (created "03.10.1987, 20:59")))

;;; Ersetzen von ':' durch '%' in Symbolen (nicht fuer das erste Zeichen)

(defrule  transform-keywords Symbol 3
	?sym:keyword-symbol-p
	==>
	?,(transform-keyword ?sym)
	(Franz-to-Common)
	(Neverprotocol)
((author "Matthias Ressel")))

(defrule transform-franzsymbols-with-colon Symbol 2
   ?sym:colon-symbol-p
   (locally (declare (special *colon-substitution-char*))
	    (not (eql *colon-substitution-char* #\:)))
   ==>
   ?,(transform-franz-symbol-with-colon ?sym)
   (Franz-to-Common)
   ((NeverProtocol))
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 17:24")
  (author "Matthias Ressel") (created "16.12.1987, 17:09")))

(defun transform-franz-symbol-with-colon (sym)
  (declare (special *colon-substitution-char*))
  (let ((cl-sym (get sym :cl-symbol)))
    (or cl-sym
	(setf (get sym :cl-symbol)
	      (intern (transform-symbol-name
			 (substitute *colon-substitution-char*
				     #\: (symbol-name sym) :start 1)))))))


(defrule  transform-franzsymbols Symbol 1
	?sym:franz-symbol-p
	==>
	?,(transform-franz-symbol ?sym)
	(Franz-to-Common)
	(Neverprotocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 13:37:34 1987")))

(defrule  MyTrigger-substitute-colons Symbol 1
	(?*args)
	==>
	?,(mapcar 'Symbol ?*args)
	(Franz-to-Common)
	(Neverprotocol)
((modified-by "Andreas Girgensohn") (modified "Fri Jan 23 17:47:41 1987")
 (author "Andreas Girgensohn") (created "Fri Jan 23 13:38:17 1987")))

(defrule  MyTrigger-substitute-colon-with-dot Symbol 0
	(?first . ?rest)
	==>
	(?,(Symbol ?first) . ?,(Symbol ?rest))
	(Franz-to-Common)
	(Neverprotocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 17:48:45 1987")))

(defun keyword-symbol-p (sym)
  (and (symbolp sym)
       (eq (symbol-package sym) *franz-package*)
       (> (length (symbol-name sym)) 1)
       (eql (schar (symbol-name sym) 0) #\:)))

(defun colon-symbol-p (sym)
  (and (symbolp sym)
       (eq (symbol-package sym) *franz-package*)
       (> (length (symbol-name sym)) 1)
       (find #\: (symbol-name sym) :start 1)))

(defun franz-symbol-p (sym)
  (and (symbolp sym)
       (eq (symbol-package sym) *franz-package*)))

(defun transform-keyword (sym)
  (let ((cl-sym (get sym :cl-symbol)))
    (or cl-sym
	(setf (get sym :cl-symbol)
	      (intern (transform-symbol-name
			(subseq (symbol-name sym) 1))
		      *keyword-package*)))))

(defun transform-symbol-name (sym-name)
  (cond ((notany #'upper-case-p sym-name)
	 (string-upcase sym-name))
	((notany #'lower-case-p sym-name)
	 (string-downcase sym-name))
	(t sym-name)))

(defun substitute-colons (sym)
  (intern (substitute #\% #\: (symbol-name sym) :start 1)))

(defun transform-franz-symbol (sym)
  (let ((cl-sym (get sym :cl-symbol)))
    (or cl-sym
	(setf (get sym :cl-symbol)
	      (intern (transform-symbol-name (symbol-name sym)))))))

;;;-------------------------------------------------------------------------------------------
;;; Definition einiger Regelmengen
;;;-------------------------------------------------------------------------------------------

(defruleset MyTrigger (form *outer*)
  :special (*outer* *fouter* *return*))

(setq *outer* 'Standard)

;;--------------------------------------------------------------------------------------------
;; Regelmengen Defaults
;;--------------------------------------------------------------------------------------------

(defruleset TransformStandard (item)
  :else (Standard (MyTrigger item 'Standard)))

(defruleset TransformForm (form *ruleset*)
  :special *ruleset*
  :else (TransformStandard form))

(defruleset TransformLetBody (body *last*)
  :special *last*
  :else (MyTriggerLetBody body *last*))

(defruleset TransformProgBody (body *return*)
  :special *return*
  :else body)

(defruleset TransformPrognBody (body *last*)
  :special *last*
  :else (MyTriggerPrognBody body *last*))

(defruleset TransformValuetype (type))
(defruleset TransformType (type))


(defruleset TransformDestructuringVariable (destruct-var)
  :else (MyTriggerFlMacroArgList destruct-var))


(defruleset TransformName (name)
  :else (Symbol name))
(defruleset TransformTag (tag)
  :else (Symbol tag))
						; und &whole ...
(defruleset TransformToBoolean (form)
  :else (TransformStandard form))
(defruleset TransformIgnoreResult (form)
  :else (TransformStandard form))
(defruleset TransformList (form)
  :else (TransformStandard form))
(defruleset TransformListIgnoreTopCopy (form)
  :else (TransformStandard form))
(defruleset TransformInteger (number)
  :else (TransformStandard number))

(defruleset TransformVariable (var)
  :else (Symbol var)
  :special (*variable-collision*  *common-special-variables*))
(defruleset TransformMyFunction (form *fouter*)
  :special *fouter*
  :else (MyFunction (MyTrigger form 'MyFunction)))

(defruleset TransformDeclaration (decl))

(defruleset TransformFunctionVariable (var)
  :else (MyLambda2 var)
  :special (*possible-function-collisions*))
(defruleset TransformKeyword (keyword)
  :else (transform-keyword keyword))
(defruleset TransformLambda1 (foo *louter*)	; foo ==> bar oder (lambda ..)
  :special *louter*
  :else (MyLambda1 (MyTriggerLambda foo)))
(defruleset TransformLambda2 (foo *louter*)	; nur foo ==> bar 
  :special *louter*
  :else (MyLambda2 (MyTriggerLambda foo)))
(defruleset TransformBody (body last)
  :else (MyTriggerPrognBody body last))

(defruleset TransformPlace (place)
  :else (TransformStandard place))	; Zwischenloesung -- Andreas

(defruleset TransformPort (port)
  :else (TransformStandard port))

(defruleset TransformMsgForms (form)
  :else (TransformStandard form))

(defruleset MyFunction (func)
  :else (Standard func))

;;;------------------------------------------------------------------------------------------
;;; Meta Regeln: Transform... etc...
;;;------------------------------------------------------------------------------------------

(defvar *assumed-lambdas* nil)
(defvar *assumed-user-functions* nil)


(defrule init-and-transform-file TransformFile 1
   (?*forms)
   ==>
   (?*,(locally
	 (declare (special
			   *possible-function-collisions*
			   *assumed-user-defined-collision*
			    *common-special-variables*
			   *variable-collision* ))
	 (prog2 (progn (setq *assumed-lambdas* nil
			     *assumed-user-functions* nil
			     *possible-function-collisions* nil
			     *assumed-user-defined-collision* nil
			     *variable-collision* nil)
		       (clear-all-cl-symbol-properties 'fl))
		(mapcar #'TransformFileForm
			?`(?*,(mapcar #'Classify ?*forms)))
		(let ((user-lambdas (intersection *assumed-lambdas* *assumed-user-functions*))
		      (user-else (set-difference *assumed-user-functions* *assumed-lambdas*))
		      (lambdas (set-difference *assumed-lambdas* *assumed-user-functions*))
		      (*package* *franz-package*))	; subpress package-prefix
		  (when *possible-function-collisions*
		    (protocol-format "~&~%The following functions would probably cause ~
                             a conflict~% with a CommonLisp Function: ~
                             (in package FL)~%~s~%~
                              It is possible that the CommonLisp function does exactly ~
                              what you want.~% Otherwise use another function name"
				     (nreverse *possible-function-collisions*)))
		  (when *assumed-user-defined-collision*
		    (protocol-format "~&~%The following functions were assumed to be ~
                                   user defined functions ~%and would probably cause~
                                   a name conflict with a CommonLisp function:~%~
                                   (in package FL)~%~s~%~
                                   It is possible that the CommonLisp function does exactly ~
                              what you want.~% Otherwise use another function name"
				     (nreverse *assumed-user-defined-collision*)))
		  (when *assumed-user-functions*
		    (protocol-format "~&~%The following functions were assumed to be ~
                                   user defined~%~
                                   and therefore their names were transformed: ~
                                   (in package FL)~%~s"
				     (nreverse *assumed-user-functions*))) 
		  (when *assumed-lambdas*
		    (protocol-format "~&~%The following function were assumed to be ~
				   lambdas (in package FL):~%~S~%~
                                   If not, the subexpressions might be falsely transformed"
				     (nreverse *assumed-lambdas*)))
		  (when *variable-collision*
		    (protocol-format "~&~%The following variables would probably cause ~
                                      a conflict~% with a CommonLisp special variable ~
                                      (or constant):~% (in package FL)~%~s"
				      (nreverse *variable-collision*)))))))
   (Franz-to-Common)
   (NeverProtocol)
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 19:52")
  (author "Matthias Ressel") (created "06.12.1987, 17:29")))

(defruleset Classify (commented-expr))

(defrule splice-and-classify-commented-condidtional-form Classify 2
   ($commented-form$ ?form:(flr::conditional-inclusion-p IT) ?*comments)
   ==>
   ?*,?`(?*comments ?,(progn (classify-expression
			       (flr::conditional-inclusion-form ?form))
			     ?form))
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 20:27")))

(defrule splice-and-classify-commented-form Classify 2
   ($commented-form$ ?form ?*comments)
   ==>
   ?*,?`(?*comments ?,(classify-expression ?form))
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 20:27")))

(defrule classify-conditional-form Classify 2
   ??:(flr::conditional-inclusion-p IT)
   ==>
   ?$left$
   (classify-expression (flr::conditional-inclusion-form ?$left$))
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 21:37")))

(defrule classify-other-expression Classify 1
   ??
   ==>
   ?,(classify-expression ?$left$)
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 20:30")))

(defruleset Classify-Expression (expr))

(defrule classify-expr Classify-Expression 2
   (??:{fl::|defun| fl::|de|} ?foo (?*) ?*)
   ==>
   ?$left$
   (progn (setf (get ?foo 'discipline) 'lambda)
	  (setf (get ?foo 'user-defined) t))
   (Franz-to-common)
   (NoProtocol)
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 22:24")
  (author "Matthias Ressel") (created "16.12.1987, 20:36")))

(defrule classify-expr-2 Classify-Expression 2
   (fl::|def| ?foo (??:{fl::|lambda| fl::|lexpr|} ?*))
   ==>
   ?$left$
   (progn (setf (get ?foo 'discipline) 'lambda)
	  (setf (get ?foo 'user-defined) t))
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 20:37")))

(defrule classify-expr-non-lambda Classify-Expression 1
   (??:{fl::|defun| fl::|de|} ?foo ?type:{fl::|macro| fl::|fexpr|} ?*)
   ==>
   ?$left$
   (progn (setf (get ?foo 'discipline) (translate-function-type ?type))
	  (setf (get ?foo 'user-defined) t))
   (Franz-to-common)
   (NoProtocol)
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 22:25")
  (author "Matthias Ressel") (created "16.12.1987, 20:40")))

(defrule classify-expr-non-lambda-2 Classify-Expression 1
   (fl::|def| ?foo (?type:{fl::|nlambda| fl::|macro|} ?*))
   ==>
   ?$left$
   (progn (setf (get ?foo 'discipline) (translate-function-type ?type))
	  (setf (get ?foo 'user-defined) t))
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 21:22")))

(defun translate-function-type (type)
  (case type
    (fl::|macro| 'macro)
    ((fl::|fexpr| fl::|nlambda|) 'nlambda)))

(defruleset TransformStart (expr))

(defrule transform-expr-with-init TransformStart 1
   ??
   ==>
   ?,(progn (clear-all-cl-symbol-properties 'fl)
	    (TransformSTandard ?$left$))
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 20:24")))

(defrule dont-touch-separators TransformStandard 10
	($separator$ ?chars)
	==>
	($separator$ ?chars)
	(Franz-to-Common Machine)
	(NoProtocol)
	((author "Matthias Ressel") (created "Sun Nov 30 20:44:58 1986")))

(defruleset mymacro (expr))

(defrule transform-variable (TransformStandard MyFunction MyMacro) 1
   ?var:symbolp
   ==>
   ?,(TransformVariable ?var)
   (Franz-to-Common)
   (NoProtocol)
 ((author "Andreas Girgensohn") (created "30.10.1987, 10:21")))

(defruleset data (expr))

(defrule transform-character (Data TransformStandard) 1
   ??:(flr::franz-char-p IT)
   ==>
   ?,(flr::franz-char-char ?$left$)
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 21:02")))

(defrule transform-special-char (DAta TransformStandard) 1
   ??:(flr::franz-special-char-p IT)	; In Klammern gilt normale
						; :-syntax
   ==>
   ?,(Convert-Special-Char ?$left$)
   (Franz-to-Common)
   (Protocol)
 ((author "Matthias Ressel") (created "02.11.1987, 19:36")))

(defun convert-Special-Char (special-char)
  (case (flr::franz-special-char-char special-char)
    (fl::|esc|
	 #+kcl #\^[
	 #-kcl #\escape)
    (fl::|sp| #\space)
    (fl::|rpar| #\))
    (fl::|lpar| #\()
    (otherwise special-char)
    ))

(defrule transform-conditional-inclusion (Data TransformStandard) 1
   ??:(flr::conditional-inclusion-p IT)
   ==>
   ?,(flr::make-conditional-inclusion
       :test (flr::conditional-inclusion-test ?$left$)
       :feature (Symbol (flr::conditional-inclusion-feature ?$left$))
       :form (TransformFileForm (flr::conditional-inclusion-form ?$left$)))
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 20:52")))

(defrule transform-readtime-eval (Data TransformStandard) 1
   ??:(flr::readtime-eval-p IT)
   ==>
   ?,(flr::make-readtime-eval
       :expression (TransformStandard (flr::readtime-eval-expression ?$left$)))
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 20:57")))



(defrule want-to-transform-lisp-expression TransformStandard 1
	?x:consp
	==>
	?,(Standard (MyTrigger ?x 'Standard))
	(Franz-to-Common)
	(NoProtocol)
	((author "Matthias Ressel") (created "Sat Jul 19 01:30:18 1986")
	 (modified-by "Matthias Ressel") (modified "Wed Nov 19 17:12:39 1986")))

(defrule Meta-Trigger-Macro TransformForm 2
	?x:consp
	(eq *ruleset* 'MyMacro)
	==>
	?,(MyMacro (MyTrigger ?x 'MyMacro))
	(franz-to-common)
	(NoProtocol)
((modified-by "Andreas Girgensohn")(modified "22.10.1987, 15:45")
 (author "Matthias Ressel") (created "Thu Apr 2 17:08:06 1987")))

(defrule Meta-Trigger TransformForm 1
	?x:consp
	==>
	?,(Standard (MyTrigger ?x *ruleset*))
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "08.09.1987, 18:15")
 (author "Matthias Ressel") (created "Thu Apr 2 16:57:40 1987")))

(defrule Meta-PrognBody TransformPrognBody 1
	(?*body ?last)
	==>
	(?*,(mapcar #'TransformIgnoreResult ?*body)
			?,(TransformForm ?last *last*))
	(Franz-to-common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "08.09.1987, 18:15")
 (author "Matthias Ressel") (created "Wed Apr 15 20:48:25 1987")))

(defrule Meta-ProgBody TransformProgBody 1
	(?*body)
	==>
	(?*,(mapcar #'MyProg ?*body))
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "08.09.1987, 18:15")
 (author "Matthias Ressel") (created "Wed Apr 15 21:40:25 1987")))

(defruleset transformmacroform (expr))

(defrule eval-macro-trigger TransformMacroForm 1
	?x:consp
	==>
	?,(MyMacro (MyTrigger ?x 'MyMacro))
	(FRanz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel") (modified "13.12.1987, 17:13")
 (author "Matthias Ressel") (created "Wed Feb 4 19:47:34 1987")))

(defrule trivial-destructuring-variable TransformDestructuringVariable 1
   ?x:symbolp
   ==>
   ?,(symbol ?x)
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "03.10.1987, 21:08")))

(defrule do-not-transform-to-common-foo TransformFunctionVariable 1
   ?foo:(common-function-p (symbol IT))
   ==>
   ?foo
   (progn (pushnew ?foo *possible-function-collisions*)
	  ;; (protocol-format "%Note: ~s could collide with CommonLisp function~%" ?foo)
	  )
   (Franz-to-Common)
   (NoProtocol)
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 22:42")
  (author "Matthias Ressel") (created "08.11.1987, 18:06")))

(defrule transform-declaration TransformDeclaration 1
   (fl::|declare| ?*decl-forms)
   ==>
   (fl::|declare| ?*,(mapcar #'TransformDeclForm ?*decl-forms))
   (FRanz-to-Common)
   (NoProtocol)
 ((modified-by "Matthias Ressel")(modified "16.12.1987, 19:52")
  (author "Matthias Ressel") (created "08.11.1987, 19:24")))

(defruleset transformdeclform (expr))

(defrule special-declaration TransformDeclForm 1
   (fl::|special| ?*vars)
   ==>
   (special ?*,(mapcar #'TransformVariable ?*vars))
   (FRanz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 19:26")))

(defrule function-declarations TransformDeclForm 1
   (?key:{fl::|lambda| fl::|nlambda| fl::|lexpr| fl::|*fexpr| fl::|*lexpr|
	  fl::|*expr| fl::|localf|} ?*foos)
   ==>
   (?,(Symbol ?key) ?*,(mapcar #'TransformFunctionVariable ?*foos))
   (FRanz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 19:31")))

(defrule variable-declarations TransformDeclForm 1
   (?key:{fl::|unspecial|} ?*vars)
   ==>
   (?,(Symbol ?key) ?*,(mapcar #'TransformVariable ?*vars))
   (protocol-format "%Warning: That's a CommonLips turkey: There's no undo ~%~
                     for a special declaration with defvar (or proclaim).")
   (FRanz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 19:34")))

(defrule else-declarations TransformDeclForm 0
   (?key ?*any)
   ==>
   (?key ?*,(mapcar #'Symbol ?*any))
   (FRanz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "08.11.1987, 19:39")))

;;;------------------------------------------------------------------------------------------
;;; Transformation von Macro Definitionen
;;;------------------------------------------------------------------------------------------

(defruleset MyMacro (macroform)			; CCC ??? MyMacro defined twice
  :else (Standard macroform))

(defrule macro-trigger-1 MyMacro 2
	'?x
	==>
	' ?,(let ((*dont-match-comma-splice* t)) ; verhindert (Match '?x ',@foo)
    (declare (special *dont-match-comma-splice*))
    (TransformStandard ?x))
	(Franz-to-Common Size Time Space Port)
	(NoProtocol)
((modified-by "Matthias Ressel") (modified "Wed Feb 4 19:14:28 1987")
 (author "Matthias Ressel") (created "Tue Feb 3 17:50:57 1987")))

(defrule macro-trigger-2 MyMacro 2
	`?x
	==>
	` ?,(let ((*dont-match-comma-splice* t))
    (declare (special *dont-match-comma-splice*))
    (TransformStandard ?x))
	(Franz-to-Common Size Time Space Port)
	(NoProtocol)
((modified-by "Matthias Ressel") (modified "Wed Feb 4 19:14:40 1987")
 (author "Matthias Ressel") (created "Tue Feb 3 17:52:28 1987")))

;;;------------------------------------------------------------------------------------------
;;; Ruecksetzen der Readersyntax
;;;------------------------------------------------------------------------------------------

(clean-rule-file)				; resets Reader-Syntax
						; if file compiles up to here
