;;; -*- Mode: Lisp; Package: gnode -*-

;;; $Header:$

;;; Copyright (c) 1990, 1993 Massachusetts Institute of Technology.
;;; All rights reserved.
;;; By the Athena Language Learning Project.
;;; "We also do windows."

(provide-module :mac-gnode)

(in-package :gnode)

#||(require-module :gnode-pkg)||#

(ccl:require 'quickdraw)

;;;==============
;;; Text boxes
;;;==============

(declaim (special *current-tree-window*))

(defclass text-box ()
  ((string :initarg :string :accessor box-string :type string)
   (font-spec :initarg :font-spec :reader font-spec)
   (position :accessor box-position)))

(defconstant times-12-plain '("Times" 12 :plain))
(defconstant geneva-12-plain '("Geneva" 12 :plain))
(defconstant sparta-12-plain '("Sparta" 12 :plain))
(defconstant times-12-italic '("Times" 12 :italic))
(defconstant times-12-bold '("Times" 12 :bold))
(defconstant times-10-plain '("Times" 10 :plain))

(defmethod initialize-instance ((self text-box) &rest init-plist
                                &key font &allow-other-keys)
  ;; Override NIL and convert GNODE-COMMAND-style font specification to
  ;; MCL-style font specification.
  (unless font (setq font #."a3"))
  (apply #'call-next-method self :font-spec
         (ecase (schar font 1)
           ;; Plain, for meanings and for special words, like "pro".
           (#\0 times-12-plain)
           ;; Special symbols.
           ;; Should display special symbols with Greek letters (tau,
           ;; epsilon) _unless_ language is Greek; plain t, e in Greek.
           ;; Tau and epsilon would be too confusing in Greek.
           (#\1
            (if (eq (current-language) :greek)
              geneva-12-plain
              sparta-12-plain))
           ;; Italic, for words, but display Greek in Greek characters.
           (#\2
            (if (eq (current-language) :greek)
              sparta-12-plain
              times-12-italic))
           ;; Bold, for nodes.
           (#\3 times-12-bold)
           ;; Small plain, for cornerscripts.
           (#\4 times-10-plain))
         init-plist))

(defmethod width ((self text-box))
  (ccl:string-width (box-string self) (font-spec self)))

(defmethod ascent ((self text-box))
  (multiple-value-bind (ascent descent) (ccl:font-info (font-spec self))
    (declare (ignore descent))
    ascent))

(defmethod descent ((self text-box))
  (multiple-value-bind (ascent descent) (ccl:font-info (font-spec self))
    (declare (ignore ascent))
    descent))

(defmethod center-adjustment ((self text-box))
  (multiple-value-bind (ascent descent) (ccl:font-info (font-spec self))
    (/ (- ascent descent) 2)))

(defmethod height ((self text-box))
  (multiple-value-bind (ascent descent) (ccl:font-info (font-spec self))
    (+ ascent descent)))

(defmethod set-position ((self text-box) h &optional v)
  (if v
    (setf (box-position self) (ccl:make-point (round h) (round v)))
    (setf (box-position self) h)))

(defmethod draw ((self text-box))
  (when (slot-boundp self 'position)
    (ccl:without-interrupts
     (ccl:move-to *current-tree-window* (box-position self))
     ;; Setting the view-font adds in any new properties but doesn't zero
     ;; out old ones.  That's not an issue for the font and size, which are
     ;; always explicitly set, but it is a problem with faces -- if the face
     ;; was bold, and we say italic, we _add_ italic.  So always reset to
     ;; plain before setting the view-font.
     ;; (&&& Seems very weird to need two quotes after #.!)
     (ccl:set-view-font *current-tree-window* #.''(:plain))
     (ccl:set-view-font *current-tree-window* (font-spec self))
     (princ (box-string self) *current-tree-window*))))

;;;==============
;;; Tree lines
;;;==============

(defclass tree-line ()
  ((from-node :initform nil :initarg :from :accessor from-node)
   (to-node :initform nil :initarg :to :accessor to-node)
   (pattern :initform ccl:*black-pattern* :initarg :pattern :reader line-pattern)
   (size :initform #.(ccl:make-point 1 1) :initarg :size :reader line-size)
   (shape :initform :straight :initarg :shape :reader line-shape)))

(defmethod initialize-instance ((self tree-line) &rest init-plist
                                &key fill shape &allow-other-keys)
  ;; Convert GNODE-COMMAND-style line fill specification to
  ;; MCL-style line fill specification.
  (ecase (schar fill 1)
    ;; The default, thin.
    (#\0 nil)
    ;; Dashed, for connecting to words.
    (#\1
     (setq init-plist (list* :pattern ccl:*gray-pattern* init-plist)))
    ;; Thick.
    (#\2
     (setq init-plist (list* :size #.(ccl:make-point 2 2) init-plist))))
  ;; "b3" is for invisible lines.  They should never be created (see
  ;; INITIALIZE-INSTANCE method for TREE-NODE.
  (setf (getf init-plist :shape)
        (ecase (schar shape 1)
          ;; The default, straight.
          (#\0 :straight)
          ;; Curved, for connecting arguments to normal predicates in GCF.
          (#\1 :curved)
          ;; Square, for connecting arguments to conjunctions in GCF.
          (#\2 :square)
          (#\4 :triangle)
          (#\5 :parallel)))
  (apply #'call-next-method self init-plist))

(defmethod draw ((self tree-line))
  (let ((from-node (from-node self))
        (to-node (to-node self))
        (shape (line-shape self)))
    (when (and from-node to-node)
      (let ((from-pos (box-position (main-box from-node)))
            (to-pos (box-position (main-box to-node)))
            from-point to-point)
        (setq from-point
              (if (> (ccl:point-v from-pos) (ccl:point-v to-pos))
                (top-connection from-node)
                (bottom-connection from-node))
              to-point
              (if (> (ccl:point-v to-pos) (ccl:point-v from-pos))
                (top-connection to-node)
                (bottom-connection to-node)))
        (ecase shape
          (:straight
           (draw-tree-line self from-point to-point))
          (:curved
           (let ((from-v (ccl:point-v from-point))
                 (to-v (ccl:point-v to-point))
                 (from-h (ccl:point-h from-point))
                 (to-h (ccl:point-h to-point))
                 start-angle1 start-angle2)
             (if (= from-h to-h)
               (draw-tree-line self from-point to-point)
               (let* ((top (min from-v to-v))
                      (left (min from-h to-h))
                      (right (max from-h to-h))
                      (height (abs (- from-v to-v)))
                      (half-height (round (/ height 2)))
                      (center-v (+ top half-height)))
                 (if (or (and (< from-v to-v)
                              (< from-h to-h))
                         (and (> from-v to-v)
                              (> from-h to-h)))
                   ;; FROM is above and to the left of TO, or
                   ;; FROM is below and to the right of TO.
                   (setq start-angle1 180 start-angle2 0)
                   ;; FROM is above and to the right of TO, or
                   ;; FROM is below and to the left of TO.
                   (setq start-angle1 90 start-angle2 270))
                 ;; Draw two 90 degree arcs, one from the from-node to the
                 ;; center, the other from the center to the to-node.  Arcs are
                 ;; drawn within rectangles.
                 (draw-tree-arc self start-angle1 left (- top half-height)
                                right center-v)
                 (draw-tree-arc self start-angle2 left center-v
                                right (+ center-v height))))))
          (:square
           (let ((from-v (ccl:point-v from-point))
                 (to-v (ccl:point-v to-point))
                 (from-h (ccl:point-h from-point))
                 (to-h (ccl:point-h to-point)))
             (if (= from-h to-h)
               (draw-tree-line self from-point to-point)
               (let* ((center (+ (min from-v to-v)
                                 (round (/ (abs (- from-v to-v)) 2))))
                      (from-center (ccl:make-point (ccl:point-h from-point) center))
                      (to-center (ccl:make-point (ccl:point-h to-point) center)))
                 (draw-tree-line self from-point from-center)
                 (draw-tree-line self from-center to-center)
                 (draw-tree-line self to-center to-point)))))
          (:triangle
           (let ((to-h (ccl:point-h to-point)))
             (if (= (ccl:point-h from-point) (ccl:point-h to-point))
               (draw-tree-line self from-point to-point)
               (let* ((to-v (ccl:point-v to-point))
                      (half-main-width (floor (/ (width (main-box to-node) 2))))
                      (left-to (ccl:make-point (- to-h half-main-width) to-v))
                      (right-to (ccl:make-point (+ to-h half-main-width) to-v)))
                 (draw-tree-line self from-point left-to)
                 (draw-tree-line self from-point right-to)
                 (draw-tree-line self left-to right-to)))))
          (:parallel
           (let ((from-h (ccl:point-h from-point))
                 (from-v (ccl:point-v from-point))
                 (to-h (ccl:point-h to-point))
                 (to-v (ccl:point-v to-point)))
             (draw-tree-line self (ccl:make-point (1- from-h) from-v)
                             (ccl:make-point (1- to-h) to-v))
             (draw-tree-line self (ccl:make-point (1+ from-h) from-v)
                             (ccl:make-point (1+ to-h) to-v)))))))))

(defmethod draw-tree-line ((self tree-line) from to)
  (ccl:without-interrupts
   (ccl:move-to *current-tree-window* from)
   (ccl:set-pen-size *current-tree-window* (line-size self))
   (ccl:set-pen-pattern *current-tree-window* (line-pattern self))
   (ccl:line-to *current-tree-window* to)))

(defmethod draw-tree-arc ((self tree-line) start-angle left top right bottom)
  (ccl:without-interrupts
   (ccl:set-pen-size *current-tree-window* (line-size self))
   (ccl:set-pen-pattern *current-tree-window* (line-pattern self))
   (ccl:frame-arc *current-tree-window* start-angle 90 left top right bottom)))


;;;===============
;;; Tree nodes
;;;===============

(defclass tree-node ()
  ((main-box :initarg :main-box :accessor main-box)
   (ne-box :initform nil :initarg :ne-box :accessor ne-box)
   (nw-box :initform nil :initarg :nw-box :accessor nw-box)
   (se-box :initform nil :initarg :se-box :accessor se-box)
   (sw-box :initform nil :initarg :sw-box :accessor sw-box)
   (children :initform nil :initarg :children :accessor children)
   (bottom-connection :accessor bottom-connection)
   (top-connection :accessor top-connection)
   (lines :initform nil :initarg :lines :accessor lines)
   ;; We need to store the line fill in the node because when using gnode,
   ;; the line fill command is issued within the scope of the child, and
   ;; indicates how to connect the child to the parent, but tree-nodes
   ;; store lines in the parent.  So we store the line fill when the node
   ;; is created, and fetch and use it when the parent and the connecting
   ;; line are created.
   (line-fill :initform #."l0" :initarg :line-fill :reader line-fill)
   ;; See doc. of line-fill.
   (line-shape :initform #."b0" :initarg :line-shape :reader line-shape)))

(defmethod initialize-instance ((self tree-node) &rest init-list
                                &key (default-lines t) &allow-other-keys)
  (call-next-method)
  (when default-lines
    (let ((l (lines self)))
      (dolist (child (children self))
        (let ((shape (line-shape child)))
          ;; The simplest way to make invisible lines is not to make them at all.
          (unless (string= shape #."b3")
            (push (make-instance 'tree-line :from child :to self
                                 :fill (line-fill child)
                                 :shape shape)
                  l))))
      (setf (lines self) l))))

;;; &&& Unused???
(defmethod find-node ((self tree-node) path)
  (etypecase path
    (string
     (if (string= path (box-string (main-box self)))
       self
       (dolist (child (children self))
         (when (find-node child path)
           (return child)))))
    (list
     (if (endp path)
       self
       (find-node (nth (first path) (children self)) (rest path))))))

(defmethod child-weight ((self tree-node))
  (reduce #'+ (children self) :key #'required-width :initial-value 0))

;;; This is for when a node has only one descendant at each level.  In this
;;; case, the nodes should fall directly under each other, connected by vertical
;;; lines.  The lines shouldn't wiggle left and right.  Therefore, the LEFT-WEIGHT
;;; of a node is the maximum width from the center to the left of the node and
;;; all its leftmost descendants.
(defmethod left-weight ((self tree-node))
  (max (+ (if (main-box self)
            (/ (width (main-box self)) 2)
            0)
          (if (nw-box self)
            (if (sw-box self)
              (max (width (nw-box self)) (width (sw-box self)))
              (width (nw-box self)))
            (if (sw-box self)
              (width (sw-box self))
              0)))
       (if (first (children self))
         (left-weight (first (children self)))
         0)))

;;; See comment at LEFT-WEIGHT.
(defmethod right-weight ((self tree-node))
  (max (+ (if (main-box self)
            (/ (width (main-box self)) 2)
            0)
          (if (ne-box self)
            (if (se-box self)
              (max (width (ne-box self)) (width (se-box self)))
              (width (ne-box self)))
            (if (se-box self)
              (width (se-box self))
              0)))
       (if (first (last (children self)))
         (right-weight (first (last (children self))))
         0)))

(defmethod ascent ((self tree-node))
  (+ (ascent (main-box self))
     (/ (max (if (ne-box self) (ascent (ne-box self)) 0)
             (if (nw-box self) (ascent (nw-box self)) 0))
        2)))

(defmethod descent ((self tree-node))
  (+ (descent (main-box self))
     (/ (max (if (se-box self) (descent (se-box self)) 0)
             (if (sw-box self) (descent (sw-box self)) 0))
        2)))

(defmethod levels ((self tree-node))
  (1+ (reduce #'max (children self) :key #'levels :initial-value 0)))

(defmethod maximum-ascent ((self tree-node))
  (reduce #'max (children self) :key #'maximum-ascent
          :initial-value (ascent self)))

(defmethod maximum-descent ((self tree-node))
  (reduce #'max (children self) :key #'maximum-descent
          :initial-value (descent self)))

;;; Nodes must be balanced, so the required width is twice the width of
;;; the wider side of the node.  The children are balanced without regard
;;; to the node itself, so their collective required width is simply the
;;; sum of their individual required widths.  The required with of the node
;;; itself is twice the width of its wider side, taking into account both
;;; the main box and the corner boxes.
(defmethod required-width ((self tree-node))
  (let ((half-main-width (/ (width (main-box self)) 2)))
    (ceiling
     (max (* (max (+ half-main-width
                     (if (nw-box self)
                       (if (sw-box self)
                         (max (width (nw-box self)) (width (sw-box self)))
                         (width (nw-box self)))
                       (if (sw-box self)
                         (width (sw-box self))
                         0)))
                  (+ half-main-width
                     (if (ne-box self)
                       (if (se-box self)
                         (max (width (ne-box self)) (width (se-box self)))
                         (width (ne-box self)))
                       (if (se-box self)
                         (width (se-box self))
                         0))))
             2)
          (reduce #'+ (children self) 
                  :key #'required-width :initial-value 0)))))

(defmethod natural-width ((self tree-node))
  (let ((half-main-width (/ (width (main-box self)) 2)))
    (+ (ceiling
        (max (* (max (+ half-main-width
                        (if (nw-box self)
                          (if (sw-box self)
                            (max (width (nw-box self)) (width (sw-box self)))
                            (width (nw-box self)))
                          (if (sw-box self)
                            (width (sw-box self))
                            0)))
                     (+ half-main-width
                        (if (ne-box self)
                          (if (se-box self)
                            (max (width (ne-box self)) (width (se-box self)))
                            (width (ne-box self)))
                          (if (se-box self)
                            (width (se-box self))
                            0))))
                2)
             (reduce #'+ (children self) 
                     :key #'natural-width :initial-value 0)))
       ;; Whitespace.
       8)))

(defmethod position-boxes ((self tree-node) left right center y yinc
                           &aux (main-box (main-box self)))
  (setf (top-connection self)
	(ccl:make-point (round center) (round (- y (ascent main-box)))))
  (setf (bottom-connection self)
	(ccl:make-point (round center) (round (+ y (descent main-box)))))
  (let ((width (- right left))
        (child-weight (child-weight self))
        (left-weight 0)
        (right-weight 0)
        (half-main-width (/ (width main-box) 2))
        (main-descent (descent main-box))
        (main-ascent (ascent main-box))
        (children (children self))
        l r c)
    (dolist (child children)
      (setq left-weight right-weight)
      (incf right-weight (required-width child))
      (setq l (+ left (/ (* width left-weight) child-weight)))
      (setq r (+ left (/ (* width right-weight) child-weight)))
      (setq c (/ (+ l r (- (left-weight child)
			   (right-weight child))) 2))
      (cond ((eq child (first children))
             (when (> c center)
               (setq c center)))
            ((eq child (first (last children)))
             (when (< c center)
               (setq c center))))
      (position-boxes child l r c (+ y yinc) yinc))
    (set-position main-box (- center half-main-width) y)
    (when (se-box self)
      (set-position (se-box self)
         (+ center half-main-width) 
         (+ (+ y main-descent) (center-adjustment (se-box self)))))
    (when (ne-box self)
      (set-position (ne-box self)
        (+ center half-main-width) 
        (+ (- y main-ascent) (center-adjustment (ne-box self)))))
    (when (sw-box self)
      (set-position (sw-box self)
         (- center half-main-width (width (sw-box self))) 
         (+ (+ y main-descent) (center-adjustment (sw-box self)))))
    (when (nw-box self)
      (set-position (nw-box self)
         (- center half-main-width (width (nw-box self)))
         (+ (- y main-ascent) (center-adjustment (nw-box self)))))))

(defmethod draw ((self tree-node))
  (draw (main-box self))
  (when (se-box self)
    (draw (se-box self)))
  (when (ne-box self)
    (draw (ne-box self)))
  (when (sw-box self)
    (draw (sw-box self)))
  (when (nw-box self)
    (draw (nw-box self)))
  (dolist (child (children self))
    (draw child))
  (dolist (line (lines self))
    (draw line)))

;;;===============
;;; Tree windows
;;;===============

(defclass tree-window (ccl:window)
  ((top-node :initarg :node :accessor top-node)
   (computed :initform nil :accessor computed))
  (:default-initargs :window-type :document-with-zoom :close-box-p t))

(defmethod initialize-instance ((self tree-window) &rest init-list)
  (call-next-method)
  (recompute self))

(defmethod ccl:view-default-size ((self tree-window))
  (ccl:make-point (natural-width (top-node self)) (natural-height self)))

(defmethod natural-height ((self tree-window))
  (let* ((top-node (top-node self))
         (heights (make-array (levels top-node) :initial-element 0)))
    (labels ((do-node (node level)
               (setf (aref heights level) 
                     (+ (ascent node) (descent node)))
               (dolist (child (children node))
                 (do-node child (1+ level)))))
      (do-node top-node 0))
    (ceiling (+ (reduce #'+ heights :initial-value 0)
                ;; Add in room for lines.
                (* (1- (levels top-node)) 20)))))

(defmethod recompute ((self tree-window))
  (setf (computed self) nil)
  (let* ((top-node (top-node self))
         (size (ccl:view-size self))
         (width (1- (ccl:point-h size)))
         (height (ccl:point-v size))
         (*current-tree-window* self)
         (max-ascent (maximum-ascent top-node))
         (max-descent (maximum-descent top-node))
         (levels (max 1 (1- (levels top-node))))
         (max-height (+ max-ascent max-descent)))
    (position-boxes top-node 0 width (/ width 2) max-ascent
                    (/ (- height max-height) levels)))
  ;; &&& If I call CCL:INVALIDATE-VIEW, then the old stuff doesn't get erased.
  ;; I guess I'd need to call CCL:VALIDATE-VIEW first.  What is the proper and
  ;; tasteful way to write this code?
  (ccl:with-port (ccl:wptr self)
    (#_InvalRect :ptr (ccl:rref (ccl:wptr self) :windowRecord.portRect)))
  (setf (computed self) t))

(defmethod ccl:set-view-size ((self tree-window) h &optional (v nil))
  ;; Can't find doc of how to update this to CLOS, so try call-next-method.
  ;; (usual-set-window-size self h v)
  (call-next-method)
  (recompute self))

(defmethod ccl:window-zoom-event-handler ((self tree-window) code)
  ;; Can't find doc of how to update this to CLOS, so try call-next-method.
  ;; (usual-window-zoom-event-handler self code)
  (call-next-method)
  (recompute self))

(defmethod ccl:view-draw-contents ((self tree-window))
  ;; Can't find doc of how to update this to CLOS, so try call-next-method.
  ;; (usual-window-draw-contents self)
  (call-next-method)
  (when (computed self)
    (let ((*current-tree-window* self))
      (draw (top-node self)))))


;;; Testing code.
#||
(defparameter node
;  (make-instance 'tree-node
;    :main-box (make-instance 'text-box :string "This")
 ;   :children
;    (list
     (make-instance 'tree-node
       :main-box (make-instance 'text-box :string "tree")
       :se-box (make-instance 'text-box :string "se" :font "a0")
       :ne-box (make-instance 'text-box :string "ne" :font "a0")
       :nw-box (make-instance 'text-box :string "nw" :font "a0")
       :sw-box (make-instance 'text-box :string "sw" :font "a0")
      #|| :children
       (list
        (make-instance 'tree-node
          :main-box (make-instance 'text-box :string "don't"))
        (make-instance 'tree-node
          :main-box (make-instance 'text-box :string "have")))||#)
;     (make-instance 'tree-node
;       :main-box (make-instance 'text-box :string "no"))
;     (make-instance 'tree-node
;       :main-box (make-instance 'text-box :string "video"))))
     )

(defun test ()
  (setq w (make-instance 'tree-window :node node)))
||#
