;;; -*- Mode: Lisp; Package: GNODE -*-

;;; $Header: /afs/athena.mit.edu/course/other/allp/nlp/tools/RCS/gnode.lisp,v 3.3 92/07/08 16:29:02 sfelshin Exp $

;;; Copyright (c) 1986-92 Massachusetts Institute of Technology.
;;; All rights reserved.
;;; There are three ways to get something done at the Athena Language
;;; Learning Project: do it yourself, ask Sue to do it, or forbid
;;; Stuart to do it.

(in-package :gnode)

;;; Display syntactic/morphological structure from nodes down to entries.
(provide-module :gnode)

#||(require-module :gnode-pkg)||#

#|| #+(and :ccl (not ccl-2)) (shadowing-import '(ccl:copy ccl:self ccl:ask))||#
#|| #+(and :ccl (not ccl-2)) (use-package :ccl)||#

#||
(global '(*gnode-kibbles* *gnode-entries*

	  gnode *gnode-all* *gnode-nord-snags* *gnode-word-features*
	  *gnode-cf-snags* *gnode-depth* *gnode-real-nodes*))
||#

(require-module :iter :macros t)
#||(use-package :iter)||#

(require-module :probes :macros t)	;for ENSURE-THAT
#||(use-package :probes)||#

(require-module :links :macros t)
#||(use-package :links)||#

(require-module :gnode-tools :macros t)
(require-module :gnode-support)
#+:ccl (require-module :mac-gnode)

(require-module :char-map)
(require-module :transliterate :macros t)
(require-module :std-chars)
#+:ccl (require-module :smarta-chars :when (eq (current-language) :greek))
#||(use-package :transliterate)||#

(require-module :feat-support :macros t)
(require-module :features :compile nil)
#||(use-package :features)||#

(require-module :snag-support)
(require-module :snags)
#||(use-package :snags)||#

(require-module :pstates :accessors t)
;;; We need pstate-support for at least PS-TOP, which is an inline.  If
;;; we compile module gnode in a non-parsing lisp, PS-TOP won't be compiled
;;; inline.  There must be a better way.
(require-module :pstate-support :inlines t :when (load-subsystem-p :parsing))
#||(use-package :parser)||#

(require-module :cuff-links)
(require-module :cf-structs :accessors t)
(require-module :cf-snags)		; for ako-drift
#||(use-package :cf)||#

(require-module :entries :accessors t)
#||(use-package :morphology)||#

(require-module :nords :accessors t)
(require-module :snord)
#||(use-package :grammar)||#

(require-module :transfer-functions)
#||(use-package :transfer-functions)||#

;;; For *GEN-BOX*.
(require-module :gen-structs :when (load-subsystem-p :generation))
#||(use-package :generation)||#


;;; ================================================================
;;; Global variables for control of default display behavior.

(defvar *gnode-kibbles* nil)
(defvar *gnode-entries* nil)

(defvar *gnode-all* nil)
(defvar *gnode-nord-snags* t)
(defvar *gnode-cf-snags* t)
(defvar *gnode-word-features* nil)
(defvar *gnode-depth* nil)
(defvar *gnode-real-nodes* nil)


;;; ================================================================
;;; Display words.  Can use GNODE for toplevel display of words.

(defun gnode-kibbles (word snord-box)
  (cond (*gnode-kibbles*
	 (dolist (kibble (word-derivation word))
	   (with-gnode-child (string-downcase (kibble-dmeaning kibble))
	     (gnode-command "l1")
	     (when *gnode-word-features*
	       (gnode-format "t~A" (kibble-features kibble)))
	     (when *gnode-entries*
	       (gnode-entries kibble)))))
	(*gnode-entries*
	 (dolist (kibble (word-derivation word))
	   (gnode-entries kibble))) 
	(t
	 (multiple-value-bind (string face)
	     (word-string-and-face word snord-box)
	   (with-gnode-child string
	     (gnode-command "l1")
	     (gnode-command face))))))

(defun gnode-entries (kibble)
  (dolist (entry (kibble-derivation kibble))
    (with-gnode-child (transliterate-string
                       (entry-string entry)
                       lexicon-internal
                       (if (eq (current-language) :greek)
                         *standard-symbol-character-set*
                         *standard-character-set*))
      (when *gnode-word-features*
        (gnode-format "t~A" (entry-features entry)))
      (gnode-command "a2")
      (gnode-command "l1"))))


;;; ================================================================
;;; Display nodes.

(defun assign-subscripts (node snord-box)
  (declare (type nord node) (type (or null snord-box) snord-box))
  (when (and *gnode-snords* snord-box)
    (setq node (snord node snord-box)))
  (let (binder)
    (when (and *gnode-new-bindings* snord-box)
      (setq binder (sb-immediate-binder node snord-box)))
    (when (and (not binder) *gnode-old-bindings* (node-p node)
	       (node-binding node))
      (setq binder (node-binding node)))
    (when binder
      (when (and *gnode-snords* snord-box)
	(setq binder (snord binder snord-box)))
      (if (and (gethash node *subscript-table*)
	       (gethash binder *subscript-table*))
	  ;; This solves a problem where four nodes are ordered
	  ;;   A B C D
	  ;; and D is bound to B, B to A, and C to D.  Before, when we
	  ;; hit B, it and A would get the subscript i.	 When we hit
	  ;; C, it and D would get j.  When we hit D, its subscript
	  ;; (and thus B's as well) would get bashed to j.
	  (let ((right (min (gethash node *subscript-table*)
			    (gethash binder *subscript-table*)))
		(wrong (max (gethash node *subscript-table*)
			    (gethash binder *subscript-table*))))
	    (maphash #'(lambda (key elt)
			 (when (eq elt wrong)
			   (setf (gethash key *subscript-table*) right)))
		     *subscript-table*))
	(let ((subscript (or (gethash node *subscript-table*)
			     (gethash binder *subscript-table*)
			     (incf *current-subscript*))))
	  (setf (gethash node *subscript-table*) subscript)
	  (setf (gethash binder *subscript-table*) subscript)))))
  (when (node-p node)
    (dolist (child (node-children node))
      (assign-subscripts child snord-box))))

(defiter pstate-nodes (ps)
  (if (and *gnode-real-nodes* (ps-real-children ps))
      (labels ((down (c)
		 (when c
		   (down (rest c))
		   (yield (first c)))))
	(down (ps-real-children ps)))
    (labels ((down (c)
	       (when c
		 (down (cddr c))
		 (yield (car c)))))
      (down (cdr (ps-children ps))))))

(defun pstate-top-node (ps)
  (if (and *gnode-real-nodes* (ps-real-children ps))
      (first (ps-real-children ps))
    (ps-top ps)))

(defun gnode (nodes &key
		    ((:all *gnode-all*) *gnode-all*)
		    ((:cf-snags *gnode-cf-snags*) *gnode-cf-snags*)
		    ((:nord-snags *gnode-nord-snags*) *gnode-nord-snags*)
		    ((:node-features *gnode-node-features*)
		     *gnode-node-features*)
		    ((:word-features *gnode-word-features*)
		     *gnode-word-features*)
		    ((:depth *gnode-depth*) *gnode-depth*)
		    ((:attend *gnode-attend*) *gnode-attend*)
		    ((:snords *gnode-snords*) *gnode-snords*)
		    ((:real-nodes *gnode-real-nodes*) *gnode-real-nodes*)
		    ((:underlying-strings *gnode-underlying-strings*)
		     *gnode-underlying-strings*)
		    ((:old-bindings *gnode-old-bindings*)
		     *gnode-old-bindings*)
		    ((:new-bindings *gnode-new-bindings*)
		     *gnode-new-bindings*)
		    ((:kibbles *gnode-kibbles*) *gnode-kibbles*)
		    ((:entries *gnode-entries*) *gnode-entries*)
		    ;; May be null, a snord-box, or a pstate.
		    (box (if (eq *direction* :generation)
			     *gen-box*))
		    (output-file nil))
  (let ((old-process *gnode-process*))
    (unwind-protect
	 (progn
	   (when output-file
	     #-:ccl
	     (setq *gnode-process*
		   (open (merge-pathnames
			  (canonical-pathname (pathname output-file))
			  ".gnode")
			 :direction :output))
	     ;; It shouldn't be too difficult to handle gnoding to a file,
	     ;; but I have no desire to write the code right now.
	     #+:ccl
	     (error
	      "Gnoding to a file is not supported under MCL."))
	   (unless (listp nodes)
	     (setq nodes (list nodes)))
	   (dolist (node nodes)
	     (let ((pbox box))
	       (when (pstate-p node)
		 (setq pbox node)
		 (when (and (= (length (ps-children pbox)) 3)
			    (ps-current-word pbox)
			    (eq (word-type (ps-current-word pbox))
				:end-of-input))
		   ;; The parse appears to be done, so only display the
		   ;; single node.
		   (setq node (pstate-top-node pbox))))
	       (reset-subscripts)
	       (let ((*reported-snags* ()))
		 (declare (special *reported-snags*))
		 (etypecase node
		   (nord
		    (assign-subscripts node pbox)
                    (let ((*current-node-plist* '()))
                      (declare (special *current-node-plist*))
		      (window-title (collect-leaves node pbox))
                      (gnode2 node 1 nil pbox t)
                      #+:ccl (make-instance 'tree-window
                               :window-title (getf *current-node-plist*
                                                   :window-title)
                               :node (first (getf *current-node-plist*
                                                  :children)))))
		   (pstate
		    (let ((leaves ()))
		      ;; Must assign subscripts in a separate pass from
		      ;; GNODE2ing.
		      (iterate (n (pstate-nodes node))
			(assign-subscripts n pbox)
			(setq leaves (nconc leaves (collect-leaves n pbox))))
                      (let ((*current-node-plist* '()))
                        (declare (special *current-node-plist*))
		        (with-gnode-child "PARSER-STACK"
			  (iterate (n (pstate-nodes node))
			    (gnode2 n 1 nil pbox nil "b0"))
			  (let ((n (ps-current-word node)))
			    (when n
			      (assign-subscripts n pbox)
			      (setq leaves (nconc leaves (list n)))
			      (gnode2 n 1 nil pbox nil "l1"))))
                        (window-title leaves)
		        #+:ccl
		        (make-instance 'tree-window
                          :window-title (getf *current-node-plist*
                                              :window-title)
                          :node (first (getf *current-node-plist*
                                             :children))))))))
               (finish-gnode-output)))
	   (reset-subscripts)
	   (when output-file
	     (close *gnode-process*)))
      (setq *gnode-process* old-process))
    nodes))


(defun boring-snag-p (snag)
  (and (not *gnode-attend*)
       (or (typep (cdr snag) 'ignore-snag)
	   (typep (cdr snag) 'ako-drift))))

(defun my-cf-snags (node snord-box)
  (declare (type nord node) (type (or null snord-box) snord-box))
  (when (and *gnode-snords* snord-box)
    (setq node (snord node snord-box)))
  (let ((cf (node-cf node)))
    (when (and cf (case-frame-tmappings cf))
      (remove-if
	#'boring-snag-p
	(snagstruct-snags (first (case-frame-tmappings cf)))))))

(defun my-nord-snags (node snord-box)
  (declare (type nord node) (type (or null snord-box) snord-box))
  (when (and *gnode-snords* snord-box)
    (setq node (snord node snord-box)))
  (remove-if #'boring-snag-p (nord-snags node)))

(defun interesting (node snord-box)
  (declare (type nord node) (type (or null snord-box) snord-box))
  (when (and *gnode-snords* snord-box)
    (setq node (snord node snord-box)))
  (or *gnode-all*
      (word-p node)
      (and (node-p node)
	   (or (node-subscript node)
	       (some #'(lambda (type)
			 (linked-p (node-type node) type 'subtype))
		       ;; Needn't include PRO (:PRO) or pro (:PWO) because
		       ;; they will be found interesting by other clauses,
		       ;; the former because it must be bound and the
		       ;; latter because it must be a word.
		       '(:comp :spec :subj :infl :aux))
	       (my-cf-snags node snord-box)
	       (my-nord-snags node snord-box)
	       (when (node-p node)
		 (dolist (child (node-children node))
		   (if (interesting child snord-box) (return t))))))))

#+:ccl
(defun feats-to-string (feats)
  (with-output-to-string (string-stream)
    (describe-features feats string-stream :compress-p t)))

(defun gnode2 (node depth current-pred snord-box &optional toplevel branch)
  (declare (type (or null snord-box) snord-box) (type nord node))
  (when (and *gnode-snords* snord-box)
    (setq node (snord node snord-box)))
  ;; Make sure that the node passed into GNODE gets displayed even if
  ;; it isn't "interesting".
  (when (or toplevel (interesting node snord-box))
    (with-gnode-child (if (word-p node)
	                (string-downcase (word-type node))
	                (nord-type node))
      (cond (branch
	     (gnode-command branch))
	    ((linked-p (nord-type node) :cp 'subtype)
	     (gnode-command "b1")))
      (when (node-subscript node)
	(gnode-command "s" (subscript-char node)))
      (typecase node
	(word
	 (gnode-kibbles node snord-box)
	 (when *gnode-word-features*
	   ;(gnode-command "t" (word-features node))
	   (gnode-format "t~A" (word-features node))
	   ))
	(node
	 (let* ((cf (node-cf node))
		(pred (if cf (case-frame-pred cf))))
	   (cond ((node-children node)
		  (if (and *gnode-depth* (>= depth *gnode-depth*))
		      (let ((leaves (collect-leaves node snord-box)))
			(cond (leaves
			       (with-gnode-child
				   (transliterate-string
				    (format
				     nil "~{~A~^ ~}"
				     (mapcar
				      #'(lambda (word)
					  (if *gnode-underlying-strings*
					      (word-underlying-string word)
					    (word-surface-string word)))
				      leaves))
				    lexicon-internal
				    (if (eq (current-language) :greek)
					*standard-symbol-character-set*
				      *standard-character-set*))
				 (gnode-command "a2")
				 (gnode-command "b4")))
			      (t
			       (with-gnode-child "[empty]"
				 (gnode-command "a1")))))
		    (dolist (child (node-children node))
		      (ensure-that () (nord-p child))
		      (gnode2 child (1+ depth) pred snord-box))))
		 ((cond ((and *gnode-new-bindings* snord-box)
			 (sb-immediate-binder node snord-box))
			(*gnode-old-bindings*
			 (node-binding node)))
		  (if (labeled-p node !trace)
		      (with-gnode-child "t"
			(gnode-command "l1"))
		    (with-gnode-child "PRO"
		      (gnode-command "l1"))))
		 (t
		  (with-gnode-child "[empty]"
		    (gnode-command "a1")
		    (gnode-command "l1"))))
	   (when *gnode-node-features*
	     (gnode-format "t~A" (node-features node)))
	   (when *gnode-cf-snags*
	     (when (and cf (not (eq pred current-pred)))
	       (local-penalty "e" (my-cf-snags node snord-box)))))))
      (when *gnode-nord-snags*
	(local-penalty "f" (my-nord-snags node snord-box))))))
