;;; define-interrupt-handler.lisp
;;;
;;; Support for interrupts in Lisp.
;;; One problem with MCL is that you can't write completion routines
;;; in Lisp. This used to mean that you needed to write them in C,
;;; Pascal, or assembler. This package adds support for completion
;;; routines written in Lisp.
;;;
;;; The latency for a Lisp completion routine may be too long for
;;; some applications. The interrupt won't be processed until the
;;; first function entry or backward branch after Lisp code exits
;;; all without-interrupts dynamic scopes. Also, the interrupt code
;;; won't run until the Mac process manager lets MCL run. The interrupt
;;; code calls #_WakeupProcess to ensure that this is will be soon,
;;; but since this is the Cooperative Multiprocessing Experience, it
;;; may take a while for other applications to give up the com.
;;;

;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Modification History
;;;
;;; ------------- 1.0
;;; bill 10/12/93 First released
;;;

#|
Documentation
-------------

DEFINE-INTERRUPT-HANDLER name&keys arglist &body body    { Macro }

  name&keys   either a symbol, name, or a list of the form:
                (name :queue-size queue-size)

  arglist     alternating types and argument names

  body        code to run when the interrupt happens

Similar to DEFPASCAL, but the resulting macptr can be called at
interrupt time. All that happens at interrupt time is that the
arguments are put into a queue. The body will be entered at the
first opportunity: as soon as your Lisp code does function entry
or a backward branch (after exiting all WITHOUT-INTERRUPTS).

Declares the name to be special and sets its value to a macptr that
can be used as the value for toolbox completion routines. You can
also FF-CALL this function, but see the warning in the example at the
end of this file. If queue-size is specified, it is the length of
the queue for values to this interrupt. That many interrupts can
happen before you process one of them. The default is 10.

The supported argument types are :word, :long, :ptr (or :pointer),
and the 16 register names as keywords (i.e. :d0 to :d7 and :a0 to :a7).
Data registers will be passed as integers. Address registers will be
passed as macptrs. :long parameters (e.g. data registers) may cons
bignums. No other consing will happen due to an interrupt unless your
body code conses. Macptr args passed to your body code have dynamic
extent. If you need to hold onto one of them beyond the dynamic
extent of the body code, copy it (with e.g. (%inc-ptr <macptr> 0)).

Returns the name, as do DEFPASCAL and DEFUN.

In order to support dynamically created interrupt handlers (e.g.
WITH-INTERRUPT-HANDLERS), the return value is different if the NIL is
passed for the name. In this case two values are returned:

1) The interrupt routine: a macptr
2) The routine's number. This can be passed to DELETE-INTERRUPT-NUMBER
   when you are finished with the routine.

If you specify a name of NIL, the macro expansion will not include a
compile & execute time require of LAPMACROS. In that case, you'll need
to include the following at top level in your source file before the
form containing the DEFINE-INTERRUPT-HANDLER:

(eval-when (:compile-toplevel :execute)
  (require "LAPMACROS"))

DEFINE-INTERRUPT with a non-NIL name includes this require for you.


INTERRUPT-OVERFLOW-COUNT                                 { Function }

Can be called from within the dynamic extent of the body of an interrupt
handler. Returns the number of interrupts that were missed since the
last time interrupt-overflow-count was called due to this interrupt's
queue being full. Will error if called from outside of the dynamic extent
of an interrupt handler. Note that this will return non-zero before
you get to the missed interrupts. There is currently no way to tell
where in the sequence of interrupts the missed ones fell. You will always
see the most recent interrupt, as the interrupt-time code overwrites
the last queue entry when the queue is full.


DELETE-INTERRUPT-HANDLER name &optional errorp           { Function }

name    a symbol naming the interrupt handler

errorp  true if an error should be signalled if there is no
        interrupt handler with the given name. The default is true.

Deletes the interrupt handler with the given name and frees all associated
Mac heap storage. It is a very bad idea to call delete-interrupt-handler
while there are outstanding interrupts on that handler (Crashville).


DELETE-INTERRUPT-NUMBER number &optional errorp          { Function }

same as DELETE-INTERRUPT-HANDLER, but takes an interrupt routine number
instead of a name. The number is returned as the second value from
DEFINE-INTERRUPT-HANDLER when the name is NIL.


WITH-INTERRUPT-HANDLERS handler-specs &body body         { Macro }

handler-specs  a list each of whose elements is of the form:
                 (name&keys args &body interrupt-body)

body           lisp forms to run in a context where the interrupt
               names are (lexically) bound to interrupt routines.

WITH-INTERRUPT-HANDLERS is to LET what DEFINE-INTERRUPT-HANDLER is to DEFVAR.
Conses some 240 bytes plus or minus for each binding (depends on how many args
there are).


See the example code at the end of this file.
                          
|#

;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; to do
;;;
#|
Add and use some DEFRESOURCE's to make WITH-INTERRUPT-HANDLERS cons less.
|#

(in-package :ccl)

(export '(define-interrupt-handler delete-interrupt-handler delete-interrupt-number
          with-interrupt-handlers interrupt-overflow-count))

(eval-when (:compile-toplevel :execute)
  (require "LAPMACROS")                 ; LAP spoken here
  (require "LISPEQU")                   ; ptask.state

(let ((*warn-if-redefine* nil))         ; the PREF lapop may eventually be part of LAP

(deflapop pref (reg record-accessor)
  `(,reg (get-field-offset ,record-accessor)))

))  ; end of let & eval-when

; This is totally evil, but the address of the interrupt pending
; flag is not exported to Lisp. This works for 2.0 & 3.0
(defconstant $ipending (+ $db_link 4))

; Here again, we know how MCL encodes the fact that there is an
; "interrupt" pending. If the word at (a5 $sp-eventch_jmp is not
; a NOP instruction, then periodic event processing will happen at
; the first backward branch of function entry.
(defconstant $nop-instruction #x4e75)

; should be #$invbl, but this seems to be an assembly-language-only constant
(defconstant $invbl-bit 6)

(defmacro define-interrupt-handler (name&keys arglist &body body)
  (let ((name name&keys)
        qsize)
    (unless (symbolp name)
      (destructuring-bind (nam &key queue-size) name
        (setq name nam qsize queue-size)))
    (multiple-value-bind (arg-encoding arg-count arg-names queue-filler-code
                                       stack-bytes entry-bytes decls)
                         (parse-interrupt-arglist arglist)
      `(progn
         ,@(when name
             ; Does this belong here. Probably yes, or some users will be confused.
             `((eval-when (:compile-toplevel :execute)
                 (require "LAPMACROS"))
               (defvar ,name)))
         (%define-interrupt-handler
          ',name ,qsize ,arg-encoding ,arg-count ,stack-bytes ,entry-bytes
          (nfunction ,name (lambda ,arg-names 
                             (declare ,@decls)
                             ,@body))
          (function ,(make-interrupt-lfun queue-filler-code)))))))

; This array holds an INTERRUPT structure instance for each
; DEFINE-INTERRUPT-HANDLER that has been evaluated.
(defvar *interrupt-routines* (make-array 100))

; Our Process Serial Number
(defvar *ccl-psn* nil)

; The address of the #_WakeUpProcess trap (dispatcher)
(defvar *wakeup-process-address* nil)

; Number of entries in *interrupt-routines*
(defvar *interrupt-routines-count* 0)

; This queue holds indices into *interrupt-routines*
(defvar *pending-interrupts* nil)

; Slots in *interrupt-routines* that are free due to delete-interrupt-handler
(defvar *free-interrupt-numbers* nil)

; Woe unto you if an interrupt happens for the routine that you remove here.
(defun delete-interrupt-handler (name &optional (errorp t))
  (let* ((interrupt (find-interrupt-named name)))
    (if (not interrupt)
      (when errorp
        (error "There is no interrupt named ~s" name))
      (%delete-interrupt-handler interrupt))))

(defun delete-interrupt-number (number &optional (errorp t))
  (let ((interrupt
         (and (< -1 number *interrupt-routines-count*)
              (svref *interrupt-routines* number))))
    (if (not interrupt)
      (when errorp
        (error "There is no interrupt number ~s" number))
      (%delete-interrupt-handler interrupt))))

(defun %delete-interrupt-handler (interrupt)
  (let* ((stub (interrupt-stub interrupt))
         (stub-ptr (interrupt-stub-pointer stub))
         (code (%get-ptr stub-ptr))
         (queue (interrupt-queue interrupt))
         (number (interrupt-routine-number interrupt)))
    (setf (svref *interrupt-routines* number) nil)
    (without-interrupts
     (if (eql (1+ number) *interrupt-routines-count*)
       (setq *interrupt-routines-count* number)
       (push number *free-interrupt-numbers*)))
    (#_DisposePtr stub-ptr)
    (#_DisposePtr (interrupt-code-pointer code))
    (#_DisposePtr queue))
  nil)

(defmacro with-interrupt-handler ((name&keys args &body interrupt-body)
                                  &body body)
  (let ((name name&keys)
        (number (gensym))
        keys)
    (when (listp name&keys)
      (setq name (car name&keys)
            keys (cdr name&keys)))
    `(let (,name ,number)
       (unwind-protect
         (progn
           (multiple-value-setq (,name ,number)
             (define-interrupt-handler (nil ,@keys) ,args
               ,@interrupt-body))
           ,@body)
       (delete-interrupt-number ,number nil)))))

; No telling how useful this will be. It conses 168 bytes.
(defmacro with-interrupt-handlers (handlers &body body)
  (if (null handlers)
    `(progn ,@body)
    `(with-interrupt-handler ,(car handlers)
       (with-interrupt-handlers ,(cdr handlers) ,@body))))

(defparameter *interrupt-encodings*
  '((:word . 0)
    (:long . 1)
    (:ptr . 2)
    (:pointer . 2)))

(defparameter *address-register-names*
  '(:a0 :a1 :a2 :a3 :a4 :a5 :a6 :a7))

(defparameter *register-names*
  `(:d0 :d1 :d2 :d3 :d4 :d5 :d6 :d7 ,@*address-register-names*))

(defparameter *valid-interrupt-arg-types*
  (append (butlast (mapcar 'car *interrupt-encodings*)) *register-names*))

(defun parse-interrupt-arglist (arglist)
  (let ((args arglist)
        names code word-args macptr-args
        (arg-encoding 0)
        (arg-count 0)
        (shift 0)
        (skip-count 0)
        (stack-bytes 0)
        (entry-bytes 0))
    (loop
      (when (null args) (return))
      (let* ((type (pop args))
             (name (pop args))
             (register-p (memq type *register-names*))
             (bytes (if (eq type :word) 2 4)))
        (unless (memq type *valid-interrupt-arg-types*)
          (error "~s is not one of ~s" type *valid-interrupt-arg-types*))
        (unless (or (listp name) (symbolp name))
          (setq name (require-type name '(or lisp symbol))))
        (unless register-p
          (incf stack-bytes bytes))
        (if (listp name)
          ; Skip this arg
          (unless register-p
            (incf skip-count bytes))
          ; Don't skip this arg
          (let* ((arg-type (if register-p
                             (if (memq type *address-register-names*)
                               :pointer
                               :long)
                             type))
                 (arg-type-code (cdr (assq arg-type *interrupt-encodings*))))
            (incf entry-bytes bytes)
            (incf arg-encoding (ash arg-type-code shift))
            (incf shift 2)
            (incf arg-count)
            (push name names)
            (push (if register-p
                    (progn
                      (unless (eq arg-type :long)
                        (push name macptr-args))
                      (cond ((eq type :a5) '(move.l (sp 4) a6@+))
                            ((eq type :a6) '(move.l @sp a6@+))
                            (t `(move.l ,(intern (symbol-name type) :ccl) a6@+))))
                    (progn
                      (unless (eql skip-count 0)
                        (push `(sub.w ($ ,skip-count) a5) code)
                        (setq skip-count 0))
                      (if (eq type :word)
                        (progn (push name word-args)
                               '(move.w -@a5 a6@+))
                        (progn
                          (unless (eq type :long)
                            (push name macptr-args))
                          '(move.l -@a5 a6@+)))))
                  code)))))
    (values arg-encoding arg-count (nreverse names) (nreverse code)
            stack-bytes
            (max entry-bytes 1)         ; 1 byte for no args
            `((type fixnum ,@(nreverse word-args))
              (type macptr ,@(nreverse macptr-args))))))

(defrecord simple-queue
  (overflow-count :long)
  (in :pointer)
  (out :pointer)
  (end :pointer)
  (data (array long 0)))

(defrecord interrupt-queue
  (entry-size :long)                    ; number of bytes per entry
  (stack-bytes :long)                   ; number of bytes to pop off the stack
  (routine-number :word)                ; index into *interrupt-routines*
  (pending-interrupts (:pointer :simple-queue))
  (q :simple-queue))

(defrecord pending-interrupts-queue
  (WakeUpProcess :ptr)                  ; (#_GetToolTrapAddress #_WakeUpProcess)
  (psn (:ptr :ProcessSerialNumber))
  (currenta5 :ptr)
  (ptaskstate (:ptr PTaskState))
  (q :simple-queue))

(defstruct interrupt
  routine                               ; the user function
  routine-name                          ; its name
  arg-encoding                          ; 2 bits per arg, :word, :long, or :ptr
  arg-count                             ; number of args to routine
  stub                                  ; interrupt code from make-interrupt-stub
  code-lfun                             ; pass this to make-interrupt-stub
  queue                                 ; an interrupt-queue record
  entry-size                            ; size of an entry in the queue
  stack-bytes                           ; size of stack on interrupt entry
  queue-size                            ; elements in queue
  routine-number)                       ; index of this record in *interrupt-routines*

; Here's where the interrupt time work happens.
; Check for overflow and increment the overflow-count if so.
; Overflow causes the most recent interrupt to get overwritten.
; I'd prefer to keep the n most recent interrupts, but this would
; require changing the output pointer, which could cause problems
; if we interrupt in the middle of the code below which pulls data out.
; The queue-filler-code is the third value returned by
; parse-interrupt-arglist above. It is a bunch of move instructions
; that expect the following setup:
;   a5 points just beyond the args on the stack
;   a6 points at the queue entry for this interrupt
;   (sp) = saved value of a6
;   (sp 4) = saved value of a5
(defmacro interrupt-code (&rest queue-filler-code)
  ; the PREF lapop is only defined at compile time above.
  ; I'm trying to avoid making it necessary to load LAP at run time.
  (unless (gethash 'pref *lapops*)
    (let ((*record-source-file* nil))
      (deflapop pref (reg record-accessor)
        `(,reg (get-field-offset ,record-accessor)))))
  `(new-lap
    interrupt-queue
    (dc.w 0 0)
    pending-interrupts
    (dc.w 0 0)
    ; Here's the entry point: 8 bytes into the lfun.
    (clr.l -@sp)                        ; for stack-bytes
    (spush a5)
    (spush a6)
    (spush d0)
    (move.l (^ interrupt-queue) a5)
    (move.l (pref a5 :interrupt-queue.stack-bytes) d0)
    (move.l d0 (sp 12))
    (move.l (pref a5 :interrupt-queue.q.in) a6)
    (move.l a6 d0)
    (add.l (pref a5 :interrupt-queue.entry-size) d0)
    (if# (eq (cmp.l (pref a5 :interrupt-queue.q.end) d0))
      (pea (pref a5 :interrupt-queue.q.data))
      (spop d0))
    (if# (ne (cmp.l (pref a5 :interrupt-queue.q.out) d0))
      (move.l d0 (pref a5 :interrupt-queue.q.in))
      ; Push on the *pending-interrupts* queue
      (spush a5)
      (spush a6)
      (move.w (pref a5 :interrupt-queue.routine-number) -@sp)
      (move.l (^ pending-interrupts) a5)
      (move.l (pref a5 :pending-interrupts-queue.q.in) a6)
      (move.l a6 d0)
      (add.l ($ 2) d0)
      (if# (eq (cmp.l (pref a5 :pending-interrupts-queue.q.end) d0))
        (pea (pref a5 :pending-interrupts-queue.q.data))
        (spop d0))
      (if# (eq (cmp.l (pref a5 :pending-interrupts-queue.q.out) d0))
        ; *pending-interrupts* queue is full.
        (add.l ($ 1) (pref a5 :pending-interrupts-queue.q.overflow-count))
        (add.w ($ 2) sp)
        (spop a6)
        (spop a5)
        (add.l ($ 1) (pref a5 :interrupt-queue.q.overflow-count))
        (move.l a6 (pref a5 :interrupt-queue.q.in))
        (spop d0)
        (bra @return)
        else#
        ; Put the routine-number in the *pending-interrupts* queue
        (move.l d0 (pref a5 :pending-interrupts-queue.q.in))
        (move.w sp@+ @a6)
        (move.l (pref a5 :pending-interrupts-queue.ptaskstate) a6)
        (move.l (@ #.#$ticks) (pref a6 ptaskstate.nexttick))
        ; Trust noone. Take no prisoners.
        (movem.l #(a0 a1 a2 a3 a4 d1 d2 d3 d4 d5 d6 d7) -@sp)
        (spush a5)                   ; save pending interrupts queue
        (move.l (pref a5 :pending-interrupts-queue.currenta5) a5)
        (move.w ($ #x4ef9) a6)       ; jmp absolute
        ; If necessary, call MCL's vbl routine to make the periodic-task
        ; code run as soon as possible.
        ; Without something like this, we may end up waiting a whole tick
        ; (16.6 milliseconds) if an interrupt happens during MCL without-interrupts.
        ; *interrupt-level* value cell is nilreg relative in 2.0
        (move.l (a5 $nil) nilreg)
        (lea (special *interupt-level*) nilreg)
        (tst.w @nilreg)
        (if# (eq (if# mi
                   (tst.w (a5 $ipending))
                   else#
                   (cmp.w ($ $nop-instruction) (a5 $eventch_jmp))))
          ; Fake a VBL interrupt, but only if either
          ; 1) Lisp interrupts are disabled and the $ipending flag is clear
          ; 2) or Lisp interrupts are enabled and there is not a
          ;    JMP instruction at $eventch_jmp
          (bset ($ $invbl-bit) (@ #.#$VBLQueue))
          (sne -@sp)
          (move.l (a5 $vbltask1) a0)
          (move.w (pref a0 :vblTask.vblcount) -@sp)
          (spush a0)
          (move.l (pref a0 :vblTask.vbladdr) a1)
          (jsr @a1)
          (spop a0)
          (move.w sp@+ (pref a0 :vblTask.vblcount))
          (if# (eq (tst.b sp@+))
            (bclr ($ $invbl-bit) (@ #.#$VBLQueue))))
        (spop a5)                    ; pending-interrupts-queue
        ; If supported, (#_WakeUpProcess (pref a5 :pending-interrupts-queue.psn))
        (move.l (pref a5 :pending-interrupts-queue.wakeupProcess) d0)
        (if# ne
          (spush (pref a5 :pending-interrupts-queue.psn))
          ; I don't think this is necessary, but it's certainly safe
          (move.l (pref a5 :pending-interrupts-queue.currenta5) a5)
          (move.w ($ 60) -@sp)
          (move.l d0 a0)
          (jsr @a0))
        (movem.l sp@+ #(a0 a1 a2 a3 a4 d1 d2 d3 d4 d5 d6 d7))
        (spop a6)
        (spop a5))
      else#
      (add.l ($ 1) (pref a5 :interrupt-queue.q.overflow-count))
      (move.l (pref a5 :interrupt-queue.entry-size) d0)
      (sub.l d0 a6)
      (pea (pref a5 :interrupt-queue.q.data))
      (if# (eq (cmp.l sp@+ a6))
        (move.l (pref a5 :interrupt-queue.q.end) a6)
        (sub.l d0 a6)))
    (move.l (sp 12) d0)              ; stack-bytes
    (lea (sp d0 20) a5)              ; 4 longs plus return address
    (spop d0)
    ,@queue-filler-code
    @return
    (spop a6)
    (lea (sp 8) a5)
    (add.l (a5 -4) a5)
    (move.l (sp 8) @a5)             ; return address
    (move.l @sp -@a5)                ; saved a5
    (move.l a5 sp)
    (spop a5)
    (rts)))

; interrupt-code is a macro instead of in-line so that people who
; macro-expand a define-interrupt-handler form will have less
; code to look at.
(defun make-interrupt-lfun (queue-filler-code)
  `(lambda (&lap 0)
     (interrupt-code ,@queue-filler-code)))

(defun make-interrupt-code (code-lfun interrupt-queue)
  (let ((p (lfun-to-ptr code-lfun)))
    (setf (%get-ptr p) interrupt-queue
          (%get-ptr p 4) *pending-interrupts*)
    (%incf-ptr p 8)))

(defun interrupt-code-pointer (interrupt-code)
  (%inc-ptr interrupt-code -8))

(defun find-interrupt-named (name)
  (dotimes (i *interrupt-routines-count*)
    (let ((ir (svref *interrupt-routines* i)))
      (when (eq name (interrupt-routine-name ir))
        (return ir)))))

; This stub is what is actually passed as the completion routine.
; The level of indirection is so that people can redefine their
; lisp code and automagically redefine
(defun make-interrupt-stub (interrupt-code)
  (let ((stub (lfun-to-ptr #'(lambda (&lap 0)
                               (new-lap
                                 code (dc.w 0 0)
                                 ; Done this way so we don't need to clear the cache
                                 (spush (^ code))
                                 (rts))))))
    (setf (%get-ptr stub) interrupt-code)
    (%incf-ptr stub 4)))

(defun interrupt-stub-pointer (interrupt-stub)
  (%inc-ptr interrupt-stub -4))

(defun install-interrupt (interrupt)
  (let* ((entry-size (interrupt-entry-size interrupt))
         (stack-bytes (interrupt-stack-bytes interrupt))
         (routine-number (interrupt-routine-number interrupt))
         (queue-size (interrupt-queue-size interrupt))
         (queue-bytes (* entry-size (1+ queue-size)))
         (queue-record-bytes (+ (record-length :interrupt-queue) queue-bytes))
         (old-queue (interrupt-queue interrupt))
         (stub (interrupt-stub interrupt))
         (code-lfun (interrupt-code-lfun interrupt))
         ; Need to cons a new queue so we can atomically replace the old one.
         (new-queue (#_NewPtr queue-record-bytes)))
    (initialize-interrupt-queue
     new-queue queue-bytes entry-size stack-bytes routine-number)
    (let* ((interrupt-code (make-interrupt-code code-lfun new-queue))
           (new-stub (if (macptrp stub)
                       (let ((stub-ptr (interrupt-stub-pointer stub)))
                         (with-macptrs ((old-code (%get-ptr stub-ptr)))
                           (setf (%get-ptr stub-ptr) interrupt-code)
                           (#_DisposePtr (interrupt-code-pointer old-code))
                           stub))
                       (setf (interrupt-stub interrupt)
                             (make-interrupt-stub interrupt-code))))
           (name (interrupt-routine-name interrupt)))
      (when name
        (set name new-stub)))
    (setf (interrupt-queue interrupt) new-queue)
    (when (macptrp old-queue)
      (#_DisposePtr old-queue))
    interrupt))

(defun initialize-interrupt-queue (iq q-bytes entry-size stack-bytes routine-number)
  (setf (pref iq :interrupt-queue.entry-size) entry-size
        (pref iq :interrupt-queue.stack-bytes) stack-bytes
        (pref iq :interrupt-queue.routine-number) routine-number
        (pref iq :interrupt-queue.pending-interrupts) *pending-interrupts*)
  (initialize-simple-queue (pref iq :interrupt-queue.q) q-bytes)
  iq)

(defun initialize-simple-queue (sq bytes)
  (let ((data (pref sq :simple-queue.data)))
    (setf (pref sq :simple-queue.overflow-count) 0
          (pref sq :simple-queue.in) data
          (pref sq :simple-queue.out) data
          (pref sq :simple-queue.end) (%incf-ptr data bytes)))
  sq)

(defun make-pending-interrupts-queue (periodic-task entry-count)
  (unless (eq (type-of periodic-task) 'periodic-task)
    ; can't use require-type as periodic-task is not a first class type
    (error "~s is not a ~s" periodic-task 'periodic-task))
  (let* ((q-bytes (* 2 (1+ entry-count)))
         (record-bytes (+ (record-length :pending-interrupts-queue) q-bytes))
         (q (#_NewPtr record-bytes)))
    (setf (pref q :pending-interrupts-queue.ptaskstate)
          (ptask.state periodic-task))
    (setf (pref q :pending-interrupts-queue.currenta5) (%currenta5))
    (setf (pref q :pending-interrupts-queue.psn) (or *ccl-psn* (%null-ptr)))
    (setf (pref q :pending-interrupts-queue.wakeUpProcess)
          (or *wakeup-process-address* (%null-ptr)))
    (initialize-simple-queue (pref q :pending-interrupts-queue.q) q-bytes)
    q))

(defun %define-interrupt-handler (name queue-size arg-encoding arg-count
                                       stack-bytes entry-size
                                       user-function code-lfun)
  (unless queue-size
    (setq queue-size 10))
  (without-interrupts
   (let ((interrupt (and name (find-interrupt-named name)))
         new-routine-number)
     (unless interrupt
       (setq interrupt (make-interrupt :routine-name name)
             new-routine-number (or (pop *free-interrupt-numbers*)
                                    *interrupt-routines-count*))
       (let ((routines *interrupt-routines*))
         (unless (> (length routines) new-routine-number)
           (let ((new-routines (make-array (ceiling (* new-routine-number 1.5)))))
             (dotimes (i new-routine-number)
               (setf (svref new-routines i) (svref routines i)))
             (setq *interrupt-routines*
                   (setq routines new-routines))))
         (setf (svref routines new-routine-number) interrupt
               (interrupt-routine-number interrupt) new-routine-number)))
     (setf (interrupt-routine interrupt) user-function
           (interrupt-arg-encoding interrupt) arg-encoding
           (interrupt-arg-count interrupt) arg-count
           (interrupt-code-lfun interrupt) code-lfun
           (interrupt-entry-size interrupt) entry-size
           (interrupt-stack-bytes interrupt) stack-bytes
           (interrupt-queue-size interrupt) queue-size)
     (install-interrupt interrupt)
     (when (and new-routine-number 
                (>= new-routine-number *interrupt-routines-count*))
       (setq *interrupt-routines-count* (1+ new-routine-number)))
     (or name
         (values (interrupt-stub interrupt) 
                 (interrupt-routine-number interrupt))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; A simple defresource
;;; The eval-when's & unless'es are here because DEFRESOURCE
;;; is built in to 3.0
;;;

(eval-when (:compile-toplevel :execute)
  (unless (fboundp '%cons-resource)

(def-accessors (resource) %svref
  nil                                   ; 'resource
  resource.constructor
  resource.destructor
  resource.initializer
  resource.pool)

(defmacro %cons-resource (constructor &optional destructor initializer)
  `(%gvector $v_istruct 'resource ,constructor ,destructor ,initializer (%cons-pool)))

(make-built-in-class 'resource *istruct-class*)

))

(eval-when (:compile-toplevel :execute :load-toplevel)
  (unless (fboundp 'defresource)

; Does NOT evaluate the constructor, but DOES evaluate the destructor & initializer
(defmacro defresource (name &key constructor destructor initializer)
  `(defparameter ,name (make-resource #'(lambda () ,constructor)
                                      ,@(when destructor
                                          `(:destructor ,destructor))
                                      ,@(when initializer
                                          `(:initializer ,initializer)))))

(defmacro using-resource ((var resource) &body body)
  (let ((resource-var (gensym)))
  `(let ((,resource-var ,resource)
         ,var)
     (unwind-protect
       (progn
         (setq ,var (allocate-resource ,resource-var))
         ,@body)
       (when ,var
         (free-resource ,resource-var ,var))))))

(defun make-resource (constructor &key destructor initializer)
  (%cons-resource constructor destructor initializer))

(defun allocate-resource (resource)
  (setq resource (require-type resource 'resource))
  (let ((pool (resource.pool resource))
        res)
    (without-interrupts
     (let ((data (pool.data pool)))
       (when data
         (setf res (car data)
               (pool.data pool) (cdr (the cons data)))
         (free-cons data))))
    (if res
      (let ((initializer (resource.initializer resource)))
        (when initializer
          (funcall initializer res)))
      (setq res (funcall (resource.constructor resource))))
    res))

(defun free-resource (resource instance)
  (setq resource (require-type resource 'resource))
  (let ((pool (resource.pool resource))
        (destructor (resource.destructor resource)))
    (when destructor
      (funcall destructor instance))
    (without-interrupts
     (setf (pool.data pool)
           (cheap-cons instance (pool.data pool)))))
  resource)

))  ; end of UNLESS & EVAL-WHEN


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Here the periodic task that invokes the user code
;;;

(defvar *interrupt-task* nil)

(defmacro processing-queue-entry ((entry-var q entry-size) &body body)
  (let ((thunk (gensym))
        (queue (gensym)))
    `(let ((,thunk #'(lambda (,entry-var) ,@body)))
       (declare (dynamic-extent ,thunk))
       (with-macptrs ((,queue ,q))
         (call-processing-queue-entry ,queue ,entry-size ,thunk)))))

(defun call-processing-queue-entry (q entry-size thunk)
  (unless (fixnump entry-size)
    (setq entry-size (require-type entry-size 'fixnum)))
  (unless (macptrp q)
    (setq q (require-type q 'macptr)))
  (locally (declare (type macptr q))
    (without-interrupts
     ; All these with-macptrs are to prevent consing. Really.
     (with-macptrs ((in (pref q :simple-queue.in))
                    (out (pref q :simple-queue.out)))
       (unless (eql in out)
         (unwind-protect
           (with-macptrs ((out out))    ; protect against %set-macptr on out
             (funcall thunk out))
           (%incf-ptr out entry-size)
           (with-macptrs ((end (pref q :simple-queue.end)))
             (if (eql out end)
               (with-macptrs ((data (pref q :simple-queue.data)))
                 (setf (pref q :simple-queue.out) data))
               (setf (pref q :simple-queue.out) out)))))))))

(defresource *macptr-resource*
  :constructor (%null-ptr))

(defvar *current-interrupt-queue*)

(defun interrupt-overflow-count ()
  ; LAP to make it very unlikely that an interrupt will
  ; go off between the time we sample the value and we clear it.
  ; Should really do this with hardware interrupts disabled, but that's
  ; too much of a pain.
  (lap-inline (*current-interrupt-queue*)
    (move.l arg_z atemp0)
    (move.l (atemp0 $macptr.ptr) atemp0)
    (lea (pref atemp0 :interrupt-queue.q.overflow-count) atemp0)
    (move.l @atemp0 acc)
    (clr.l @atemp0)
    (jsr_subprim $sp-mklong)))

(defun interrupt-task ()
  (declare (optimize (speed 3) (safety 0)))
  (let* ((q *pending-interrupts*)
         done)
    (declare (type macptr q))
    (when q                             ; startup transient
      (unless (eql 0 (pref q :pending-interrupts-queue.q.overflow-count))
        (unwind-protect
          (error "~s queue overflowed. Not good." '*pending-interrupts*)
          (setf (pref q :pending-interrupts-queue.q.overflow-count) 0)))
      (loop
        (setq done t)
        (processing-queue-entry (in
                                 (the macptr (pref q :pending-interrupts-queue.q))
                                 2)
          (declare (type macptr in))
          (setq done nil)
          (let* ((routine-number (%get-word in))
                 (interrupt (svref *interrupt-routines* routine-number))
                 (queue (interrupt-queue interrupt))
                 (*current-interrupt-queue* queue))
            (declare (type macptr queue))
            (processing-queue-entry (p
                                     (the macptr (pref queue :interrupt-queue.q))
                                     (interrupt-entry-size interrupt))
              (declare (type macptr p))
              (let* ((arg-encoding (interrupt-arg-encoding interrupt))
                     (arg-count (interrupt-arg-count interrupt))
                     (args (make-array arg-count))
                     (pointers (make-array arg-count))
                     (pointer-count 0))
                (declare (dynamic-extent args pointers))
                (declare (fixnum arg-count pointer-count arg-encoding)
                         (type simple-vector args pointers))
                (dotimes (i arg-count)
                  (let ((type (logand arg-encoding 3)))
                    (setf (svref args i)
                          (ecase type
                            (0 (%get-word p))
                            (1 (%get-long p))
                            (2 (with-macptrs ((ptr (%get-ptr p)))
                                 ; All this to avoid consing a macptr
                                 ; Maybe we should recurse instead
                                 (prog1
                                   (setf (svref pointers pointer-count)
                                         (%setf-macptr
                                          (allocate-resource *macptr-resource*)
                                          ptr))
                                   (incf pointer-count))))))
                    (setq arg-encoding (ash arg-encoding -2))
                    (%incf-ptr p (if (eql type 0) 2 4))))
                (applyv (interrupt-routine interrupt) args)
                (dotimes (i pointer-count)
                  (free-resource *macptr-resource* (svref pointers i)))))))
        (when done (return))))))      

(defun install-define-interrupt-handler ()
  (let ((psn (make-record :ProcessSerialNumber))
        (failed t))
    (unwind-protect
      (progn
        (setq *ccl-psn* nil
              *wakeup-process-address* nil)
        (when (getf *environs* :appleevents)      ; good test for process manager?
          (unless (eql 0 (#_GetCurrentProcess psn))
            (error "#_GetCurrentProcess failed."))
          (setq *ccl-psn* psn
                *wakeup-process-address* (#_GetToolTrapAddress #_WakeUpProcess)))
        (setq failed nil))
      (when failed
        (#_DisposePtr psn))))
  (setq *pending-interrupts* nil)
  (setq *interrupt-task*
        (%install-periodic-task 'interrupt-task 'interrupt-task 3600))
  (setq *pending-interrupts*
        (make-pending-interrupts-queue *interrupt-task* 200))
  (let ((routines *interrupt-routines*))
    (dotimes (i *interrupt-routines-count*)
      (install-interrupt (svref routines i)))))

(def-load-pointers install-define-interrupt-handler ()
  (install-define-interrupt-handler))

(provide :define-interrupt-handler)

#|

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Test interrupt latency
;;; Note that you should never ff-call an interrupt routine
;;; unless you know that no asynchronous interrupt can
;;; call it at the same time. If an interrupt goes off while the
;;; ff-call'ed interrupt code is running, the data structures
;;; will likely become inconsistent and your Lisp will likely crash.
;;;
;;; I timed this at 406 microseconds on a IIfx.
;;; Not blindingly fast, but 2463 interrupts/second is fast
;;; enough for many applications.
;;; Also, the latency depends on the time spent in the longest
;;; periodic task, as periodic tasks cannot be interrupted.
;;;
;;; Actual latency will be worse than this as this code doesn't need
;;; to wait long for a function entry, backward branch, or exiting from
;;; without-interrupts. MCL (especially Fred) has quite a bit of code that
;;; executes without-interrupts, and this will affect the real latency.
;;;
;;; Also, see the comment about the Macintosh process manager at
;;; the top of this file.
;;;

(defvar *x* nil)

(define-interrupt-handler latency-interrupt (:word x)
  (if (>= x 1000)
    (setq *x* x)
    (ff-call latency-interrupt :word (1+ x))))

; Return the interrupt latency in milliseconds
(defun compute-interrupt-latency ()
  (let ((start-time (get-internal-run-time)))
    (setq *x* nil)
    (ff-call latency-interrupt :word 1)
    (loop (when *x* (return)))
    (let ((end-time (get-internal-run-time)))
      (/ (- end-time start-time) 1000.0))))

(compute-interrupt-latency)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; A simple asynchronous file copy.
;;; Only copies the data fork.
;;; Always overwrites an existing file.
;;; Does not do double bufferring.
;;; Will not really run asynchronously except when copying
;;; to floppies (and network volumes?) until the file manager is
;;; updated to use the new reentrant SCSI manager (and even then
;;; it won't be really asynchronous except on IIfx'es and Quadras
;;; that have SCSI DMA).

;; This code calls DEFINE-INTERRUPT-HANDLER with a null name,
;; so we need to explicitly require LAPMACROS.
(eval-when (:compile-toplevel :execute)
  (require "LAPMACROS"))

(defun async-copy-file (from-file to-file done-thunk &optional (buffer-size 1024)
                                  debug)
  (setq to-file (merge-pathnames to-file from-file))
  (let ((from-namestring (mac-namestring from-file))
        (to-namestring (mac-namestring to-file)))
    (when (or (> (length from-namestring) 255)
              (> (length to-namestring) 255))
      (error "One of the file names is too long"))
    (unless (probe-file to-file)
      (create-file to-file))
    (let ((next-step nil)
          (from-pb (make-record (:ParamBlockRec :clear t)))
          (to-pb (make-record (:ParamBlockRec :clear t)))
          (buf (#_NewPtr :errchk buffer-size))
          (name (#_NewPtr :errchk 256))
          (listener (and debug (front-window :class 'listener)))
          handler handler-number error-code
          from-file-open to-file-open)
      (labels
        ((msg (msg)
           (when debug
             (format listener "~&~a~%" msg)
             (force-output listener)))
         (open-from-file ()
           (msg "Opening from file")
           (setq next-step #'open-to-file)
           (%put-string name from-namestring)
           (setf (pref from-pb :paramBlockRec.ioCompletion) handler
                 (pref from-pb :paramBlockRec.ioNamePtr) name
                 (pref from-pb :paramBlockRec.ioPermssn) #$fsRdPerm
                 (pref from-pb :paramBlockRec.iovRefnum)
                 (volume-number (mac-directory-namestring from-file)))
           (#_Open :async from-pb))
         (open-to-file ()
           (unless (check-error (pref from-pb :paramBlockRec.ioResult))
             (msg "Opening to file")
             (setq from-file-open t)
             (setq next-step #'start-io)
             (%put-string name to-namestring)
             (setf (pref to-pb :paramBlockRec.ioCompletion) handler
                   (pref to-pb :paramBlockRec.ioNamePtr) name
                   (pref to-pb :paramBlockRec.ioPermssn) #$fsWrPerm
                   (pref to-pb :paramBlockRec.iovRefnum)
                   (volume-number (mac-directory-namestring to-file)))
             (#_Open :async to-pb)))
         (start-io ()
           (unless (check-error (pref to-pb :paramBlockRec.ioResult))
             (msg "Starting IO")
             (setq to-file-open t)
             (setf (pref from-pb :paramBlockRec.ioBuffer) buf
                   (pref from-pb :paramBlockRec.ioReqCount) buffer-size
                   (pref from-pb :paramBlockRec.ioPosMode) #$fsFromStart
                   (pref to-pb :paramBlockRec.ioBuffer) buf
                   (pref to-pb :paramBlockRec.ioReqCount) buffer-size
                   (pref to-pb :paramBlockRec.ioActCount) buffer-size     ; fake out read-from-file
                   (pref to-pb :paramBlockRec.ioPosMode) #$fsFromStart)
             (read-from-file)))
         (read-from-file ()
           (unless (check-error (pref to-pb :paramBlockRec.ioResult))
             (msg "Reading")
             (if (< (pref to-pb :paramBlockRec.ioActCount) buffer-size)
               (close-to-file)
               (progn
                 (setq next-step #'write-to-file)
                 (#_Read :async from-pb)))))
         (write-to-file ()
           (unless (check-error (pref from-pb :paramBlockRec.ioResult))
             (msg "Writing")
             (let ((bytes (pref from-pb :paramBlockRec.ioActCount)))
               (when (< bytes buffer-size)
                 (when (eql 0 bytes)
                   (close-to-file))
                 (setf (pref to-pb :paramBlockRec.ioReqCount) bytes)))
             (setq next-step #'read-from-file)
             (#_Write :async to-pb)))
         (close-to-file ()
           (unless (check-error (pref from-pb :paramBlockRec.ioResult))
             (msg "Closing to file")
             (setq next-step #'flush-volume)
             (setq to-file-open nil)
             (#_Close :async to-pb)))
         (flush-volume ()
           (unless (check-error (pref to-pb :paramBlockRec.ioResult))
             (msg "Flushing to volume")
             (setq next-step #'close-from-file)
             (#_flushvol :async to-pb)))
         (close-from-file ()
           ; always called synchronously, so don't check-error
           (msg "Closing from file")
           (setq next-step #'finish-up)
           (setq from-file-open nil)
           (#_Close :async from-pb))
         (finish-up ()
           (unless (check-error (pref to-pb :paramBlockRec.ioResult))
             (msg "Finishing up")
             (#_DisposePtr from-pb)
             (#_DisposePtr to-pb)
             (#_DisposePtr buf)
             (#_DisposePtr name)
             (delete-interrupt-number handler-number)
             (funcall done-thunk error-code)))
         (check-error (code)
           ; Return non-NIL if there was an error
           (unless (or (eql code #$noErr) (eql code #$eofErr))
             (msg (format nil "Error code #~s" code))
             (unless error-code
               (setq error-code code))
             (setf (pref from-pb :paramBlockRec.ioResult) #$noErr
                   (pref to-pb :paramBlockRec.ioResult) #$noErr)
             (cond (to-file-open (close-to-file))
                   (from-file-open (close-from-file))
                   (t (finish-up)))
             t)))
        (multiple-value-setq (handler handler-number)
          (define-interrupt-handler nil ()
            (funcall next-step)))
        (when debug 
          (inspect (svref ccl::*interrupt-routines* handler-number))
          (locally (declare (special *open-from-file*))
            (setq *open-from-file* #'open-from-file)))
        (open-from-file)))))

(async-copy-file "ccl:examples;boyer-moore.lisp"
                 ; double ";" means :UP. This file is "ccl:temp.lisp", and it will stay
                 ; that way through merge-pathnames
                 "ccl:examples;;temp.lisp"
                 #'(lambda (error-code)
                     (format (front-window :class 'listener)
                             "~&Done: ~s~%" error-code)))

|#
