;;; -*- Mode:Common-Lisp; Package:ZWEI; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by members of the Large Knowledge Base project.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

(defvar *enable-lisp-type-completion-in-zmacs-p* t
"When true, commands like escape and super-/ wil do the right
sort of completion.
"
)

(defvar *zmacs-name-completion-hook* '(Default-Zmacs-Name-Completion-Function)
"A hook for the completion functions to use in zmacs.  The functions are
called in sequence until a non nil result is returned.
"
)

(defvar *Completion-Prefered-Replacement-Case* :Downcase
"The case in which completed text should be inserted."
)

(defvar *use-menu-even-if-only-one-completion-found-p* t
"When false causes completion to go ahead if only one completion is found
even if you picked a completion type that normally generates a menu.
"
)

(defun my-separate-name-from-package (word)
  "Separates the name from the package in word doing the right thing if
   word is a symbol."
  (if (symbolp word)
      (values (symbol-name word) (symbol-package word))
      (w:separate-name-from-package word)
  )
)

(defun default-zmacs-name-completion-function
       (word completion-type char function-position-p)
"The default zmacs lisp completion function.  Tries to find a variable or a
function name depending on function-position-p that matches the string Word
accordingto the completion type Completion-Type.  Char is the first char
in the word.
"
  (ignore char)
  (let ((old #'w:separate-name-from-package))
       (letf ((#'w:separate-name-from-package
	       #'(lambda (word)
		   (if (symbolp word)
		       (values (symbol-name word) (symbol-package word))
		       (funcall old word)
		   )
		 )
	      )
	     )
	 (ucl:with-otl-and-sg
	   nil
	   (w:get-symbol-completions
	     (if (lisp:find #\: word :test #'char=)
		 (or (catch-error (read-from-string word) nil) word)
		 word
	     )
	     completion-type
	     (if function-position-p #'fboundp #'ucl:boundp-and-not-keyword)
	   )
	 )
       )
  )
)

(defun complete-name-at-point (completion-type)
"Completes in Zmacs at the current point.  Completion-Type is the type of
completion we are going to do, e.g. :Recognition or :apropos.  Returns the
list of matches and the string that it tried to complete.
"
  (declare (values list-of-matches string-on-which-completion-was-attempted))
  (let ((saved-point (copy-bp (point))))
       (loop unless (bp-= (point) (send *interval* :First-Bp))
	     do (com-backward)
	     for char = (bp-char (point))
	     until (or (bp-= (point) (send *interval* :First-Bp))
		       (char-equal char #\()
		       (sys:whitespacep char)
		       (and (char-equal char #\#)
			    (char-equal (bp-char (forward-char (point))) #\%)
		       )
		   )
       )
       (let ((char (bp-char (point))))
	    (with-open-stream
	      (stream (interval-stream (if (or (char-equal char #\()
					       (sys:whitespacep char)
					   )
					   (forward-char (copy-bp (point)))
					   (copy-bp (point))
				       )
				       saved-point
		      )
	      )
	      (move-bp (point) saved-point)
	      (let ((string (read-line stream nil :Eof)))
		   (if (equal :Eof string)
		       (barf "Could not read something to complete.")
		       (let ((completions
			       (loop for function
				     in *Zmacs-Name-Completion-Hook*
				     for result = (Funcall function string
							   completion-type char
							   (char-equal char #\()
						  )
				     when result return result
				     finally (return nil)
			       )
			     )
			    )
			    (values completions String)
		       )
		   )
	      )
	    )
       )
  )
)


(defun find-longest-matching-name-1 (names index)
  "Given a list of names and the current index finds the largest substring
  in all of the names that match one another, i.e. if the names were
  (foo-bar foo-baz) then it would return foo-ba.
  "
  (if (rest names)
      (loop for name in (rest names)
	    when (>= index (length name))
	    do (return name)
	    when (>= index (length (first names)))
	    do (return (first names))
	    when (not (char-equal (aref (first names) index)
				  (aref name index)
		      )
		 )
	    do (return (subseq (first names) 0 index))
	    finally (return (find-longest-matching-name-1 names (+ 1 index)))
      )
      (first names)
  )
)

(defun stringify (thing)
  "Turns thing into a string."
  (if (stringp thing)
      thing
      (let ((*print-case* *Completion-Prefered-Replacement-Case*))
	   (Format nil "~S" thing)
      )
  )
)

(defun casify (string)
  "Returns String casified into the appropriate case."
  (ecase *Completion-Prefered-Replacement-Case*
    (:Capitalize (string-capitalize string))
    (:Upcase (string-upcase string))
    (:Downcase (string-downcase string))
  )
)

(defun get-package-prefix (package)
 "Returns a string for the package prefix for package."
  (string-append (package-prefix-print-name package)
		 (sys:pttbl-package-internal-prefix *readtable*)
  )
)

(Defun find-longest-matching-name (names index)
  "Given a list of names and the current index finds the largest substring
  in all of the names that match one another, i.e. if the names were
  (foo-bar foo-baz) then it would return foo-ba.
  "
  (let ((strings (mapcar 'My-Separate-Name-From-Package names)))
       (let ((best (Find-Longest-Matching-Name-1 strings index)))
	    (if best
		(casify
		  (multiple-value-bind (ignore package)
		      (My-Separate-Name-From-Package (first names))
		    (let ((symbol (find-symbol best package)))
			 (if symbol
			     ;;; Then we've completed to a real thing.
			     (Stringify symbol)
			     (if (equal package *package*)
				 best
				 (string-append
				   (get-package-prefix package)
				   best
				 )
			     )
			 )
		    )
		  )
		)
		nil
	    )
       )
  )
)

(defun recognition-matching-names (names prefix menu-p)
 "Given a list of names and the prefix string decides what string to complete
  to.  If menu-p is true then it uses a menu, otherwise it uses the longest
  matching substring.
 "
  (if menu-p
      (if names
	  (if (or *Use-Menu-Even-If-Only-One-Completion-Found-P* (rest names))
	      (W:menu-choose
		(mapcar #'(lambda (x) (list x :Value x))
			(mapcar 'Stringify names)
		)
		:Label (format nil "Possible completions of ~S" prefix)
	      )
	      (Stringify (first names))
	  )
	  (barf "No matches found.")
      )
      (Find-Longest-Matching-name names 0)
  )
)

(defun insert-char (char)
  "Inserts char in the current font at point."
  (let ((*last-command-char* char))
       (let ((*font* (char-font *last-command-char*)))
	    (com-self-insert)
       )
  )
)

(defun name-complete-1 (type completion-function &optional (menu-p nil))
  "Internal function for completion commands.  Completes the word at the point
   using the specified completion function.  If menu-p is true then it uses
   a menu to decide on multiple completions.
  "
  (multiple-value-bind (results prefix) (funcall completion-function type)
    (let ((best (ecase type
		  (:Recognition
		   (recognition-matching-names results prefix menu-p)
		  )
		  ((:Apropos :Spelling-Corrected)
		   (if results
		       (w:menu-choose
			 (mapcar #'(lambda (x) (list x :Value x))
				   (mapcar 'Stringify results)
			 )
			 :Label (format nil "Possible completions of ~S" prefix)
		       )
		       nil
		   )
		  )
		)
	  )
	 )
	 (if best
	     (if (> (length best) 0)
		 (progn
		   (loop for i from 1 to (length prefix) do (com-rubout))
		   (loop for i from 0 to (- (length best) 1) do
			 (insert-char (aref best i))
		   )
		 )
		 (beep)
	     )
	     (beep 'tv:notify)
	 )
    )
  )
  dis-text
)

;-------------------------------------------------------------------------------
;;; Define the commands themselves.
;-------------------------------------------------------------------------------

(defcom com-name-complete
  "Tries to complete the name, i.e. function or variable, at the point
using initial substring (recognition) completion."
  ()
  (if *Enable-Lisp-Type-Completion-In-Zmacs-P*
      (Name-complete-1 :Recognition 'complete-name-at-point)
      (com-ordinarily-self-insert)
  )
)

(tv:install-zmacs-commmand '(#\ com-name-complete))

(defcom com-name-menu-recognition-complete
  "Tries to complete the name, i.e. function or variable, at the point
using initial substring (recognition) completion giving a menu of the matches."
  ()
  (if *Enable-Lisp-Type-Completion-In-Zmacs-P*
      (Name-complete-1 :Recognition 'Complete-name-at-point t)
      (barf "Undefined command.")
  )
)

(tv:install-zmacs-commmand '(#\c-/ com-name-menu-recognition-complete))

(defcom com-name-apropos-complete
  "Tries to complete the name, i.e. function or variable, at the point
using apropos (any substring) completion giving a menu of the matches."
  ()
  (if *Enable-Lisp-Type-Completion-In-Zmacs-P*
      (name-complete-1 :Apropos 'complete-name-at-point)
      (barf "Undefined command.")
  )
)

(tv:install-zmacs-commmand '(#\s-/ com-name-apropos-complete))

(defcom Com-name-Spelling-Correcting-Complete
  "Tries to complete the name, i.e. function or variable, at the point
using spelling correcting completion on a substring, giving a menu of
the matches."
  ()
  (if *Enable-Lisp-Type-Completion-In-Zmacs-P*
    (name-complete-1 :Spelling-Corrected 'complete-name-at-point)
    (barf "Undefined command.")
  )
)

(tv:install-zmacs-commmand '(#\H- com-name-apropos-complete))

;-------------------------------------------------------------------------------
