;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

(defflavor task-server
	   ((task-stream nil))
	   (Clean-Break-Mixin tv:process-mixin tv:window)
  (:Default-Init-Plist
    :Expose-P nil
    :Activate-P t
    :Width  (min 200 (+ 301 (send tv:default-screen :Width)))
    :Height (min 700 (send tv:default-screen :Height))
    :Position (list (min 301 (send tv:default-screen :Width)) 0)
    :More-P nil
    :Save-Bits t
    :Deexposed-Typeout-Action :Expose
    :Process '(yw-task-server
		:special-pdl-size 4000
		:regular-pdl-size 10000
	      )
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
)

(defmethod (Task-Server :After :Init) (ignore)
  (pushnew self *all-daemons*)
  (setf (get tv:process :Initial-Priority) (send tv:process :Priority))
)

(defmethod (Task-Server :Before :kill) (&rest ignore)
  (setq *all-daemons* (remove self *all-daemons*))
)

(defun listify-name-with-dots (string &optional (start 0))
"Transforms a string like \"AI.Dexter\" into (\"AI\" #\. \"Dexter\")."
  (if (stringp string)
      (let ((dot-index (position #\. (the string string) :Start start)))
	   (if dot-index
	       (cons (subseq string start dot-index)
		     (cons #\. (listify-name-with-dots string (+ 1 dot-index)))
	       )
	       (if (equal 0 start)
		   (list string)
		   (list (subseq string start))
	       )
	   )
      )
      (if (listp string)
	  (apply #'append (mapcar #'Listify-Name-With-Dots string))
	  (list string)
      )
  )
)

(defun redotify-address-database ()
"Given an address database that is poluted with address local parts
 that are not in the form (\"AI\" #\. \"Dexter\"), cleans them out and fixes
 up the address database.
"
  (maphash #'(lambda (key address)
	       (ignore address)
	       (let ((local (get key :Local-Part)))
		    (if (and (consp local) (equal (length local) 1)
			     (position #\. (the string (first local)))
			)
			(progn (setf (getf (rest key) :Local-Part)
				     (Listify-Name-With-Dots (first local))
			       )
			       (remhash (yw-zwei:Address-Database-Key key)
					*address-database*
			       )
			       (print key)
			       (let ((*save-addresses-in-database-p* t)
				     (*force-into-address-database* t)
				    )
				    (declare
				      (special
					yw:*force-into-address-database*
				      )
				    )
				    (apply 'mail:get-address-object
				      (type-of address)
				      (loop for keyword in (rest key) by #'cddr
					    append
					    (list keyword
					      (if (equal keyword :Local-Part)
						  (Listify-Name-With-Dots
						    (send address :Local-Part)
						  )
						  (send address keyword)
					      )
					    )
				      )
				    )
			       )
			)
			(if (position #\.
			      (the string (first (send address :local-part)))
			    )
			    (setf (symeval-in-instance address 'mail:local-part)
				  (Listify-Name-With-Dots
				    (symeval-in-instance
				      address 'mail:local-part
				    )
				  )
			    )
			    nil
			)
		    )
	       )
	     )
	     *address-database*
  )
)

(defun parse-address-component (address)
  (typecase address
    (address
     (or (address-address-object address)
	 (let ((*save-addresses-in-database-p* t)
	       (*owning-address* address)
	      )
	      (declare (special *owning-address*))
	      (let ((object (if (address-personalname address)
				(mail:get-named-address
				  (address-personalname address)
				  (address-routelist address)
				  (Listify-Name-With-Dots
				    (address-mailbox address)
				  )
				  (address-host address)
				)
				(mail:get-basic-address
				  (Listify-Name-With-Dots
				    (address-mailbox address)
				  )
				  (address-host address)
				)
			    )
		    )
		   )
		   (or (address-address-object address)
		       (setf (address-address-object address) object)
		   )
		   object
	      )
	 )
     )
    )
    (cons (mapcar #'parse-address-component address))
  )
)

(defun get-address-object (address)
  (Parse-Address-Component address)
  (address-address-object address)
)

(defmethod (Task-Server :parse-envelope) (envelope cache-entry mailstream)
  (loop for (name . slot)
	in (sys:defstruct-description-slot-alist
	     (get 'envelope 'sys:defstruct-description)
	   )
	when (member (symbol-name name)
		     *header-field-keys-to-parse-for-address-database*
		     :Test #'string-equal
	     )
	do
	(let ((value (funcall (sys:defstruct-slot-description-ref-macro-name
				slot
			      )
			      envelope
		     )
	      )
	      (*owning-cache-entry* cache-entry)
	      (*owning-mailstream* mailstream)
	     )
	     (declare (special *owning-cache-entry* *owning-mailstream*))
	     (parse-address-component value)
	)
  )
)

(defmethod (Task-Server :Maybe-Filter-Header) (cache-entry header mailstream)
  (maybe-filter-header header cache-entry)
  (maybe-parse-multi-part-stuff cache-entry mailstream)
)


(defun yw-task-server (server)
  (let ((*terminal-io* server))
       (declare (special *terminal-io*))
       (loop do (send server :process-task))
; (with-metering ("lm:rice;meter.text") ()
;       (loop do (catch-all (catch-error (send server :process-task) t)))
;  )
 )
)

(defmethod (task-server :get-task) ()
  (without-interrupts
    (if task-stream
	(let ((task (first (last task-stream)))
	      (rest (butlast task-stream))
	     )
	     (setq task-stream rest)
	     (values task t)
	)
	(values nil nil)
    )
  )
)

(defmethod (task-server :snip-out-task) (entry)
  (without-interrupts
    (if task-stream  ;;; !!! Surgical ok here. !!!!
	(setq task-stream
	      (delete entry (the list task-stream) :Count 1 :Test #'eq)
	)
	nil
    )
  )
  (values nil nil)
)

(defmethod (task-server :peek-task) ()
  (not (equal nil task-stream))
)

(defmethod (task-server :maybe-put-task) (purpose task)
  (or (find task task-stream :Test #'equalp :Key 'task-queue-entry-task)
      (send self :put-task purpose task)
  )
)

(defmethod (task-server :put-task) (purpose task)
  (declare (optimize (speed 3) (safety 0)))
  (push (make-task-queue-entry :Purpose purpose
	  :Originator current-process :Task task
	)
	task-stream
  )
)

(defmethod (Task-Server :send-on) (to message &rest args)
  (yw:safe-lexpr-send to message args)
)

(defmethod (task-server :process-task) ()
  (process-wait "Await task" #'(lambda (me) (send me :Peek-Task)) self)
  (multiple-value-bind (task-object found-p) (send self :get-task)
    (let ((task (task-queue-entry-task task-object)))
	 (if found-p
	     (yw:safe-lexpr-send self (first task) (rest task))
	     (yw-error "Didn't find a task for some reason!!!")
	 )
    )
  )
)

(defmethod (task-server :process-next-message)
  (edit-window direction in-sequence-p owner sequence current-message
   increment continuation-method
  )
  (let ((message
	  (send sequence :Next-Message direction in-sequence-p current-message
		increment
	  )
        )
	(*owning-window* owner)
	(*process-messages-even-if-deleted*
	  (if in-sequence-p *process-messages-even-if-deleted* t)
	)
       )
       (declare (special *owning-window*))
       (if message
	   (letf (((symeval-in-instance self 'tv:io-buffer)
		   (send (send edit-window :superior) :Io-Buffer)
		  )
		 )
		 (let ((result
			 (send sequence continuation-method message edit-window)
		       )
		      )
		      (if (equal result :try-the-next)
			  (send self :Process-Next-Message edit-window direction
				in-sequence-p owner sequence message
				continuation-method
			  )
			  result
		      )
		 )
	   )
	   (if (and (send sequence :Superior)
		    (send (send sequence :mailbox) :Send-If-Handles :Superior)
	       )
	       ;;; Ppop back up to the superior sequence if we have one and
	       ;;; continue from here.  The superior of the mailstream
	       ;;; of the current sequence is the message from which it was
	       ;;; abstracted.
	       (progn (end-of-sequence-action sequence current-message
					      (cache-mailstream current-message)
		      )
		      (send self :Process-Next-Message edit-window direction
			    in-sequence-p owner (send sequence :Superior)
			    (send (send sequence :mailbox) :Superior)
			    increment continuation-method
		      )
	       )
	       (progn (if (and *reselect-mail-control-window-on-end-of-sequence*
			       (equal 1 increment)
			  )
			  (send owner :Mouse-Select)
			  nil
		      )
		      (end-of-sequence-action sequence current-message
					      (cache-mailstream current-message)
		      )
		      :no-message-found
	       )
	   )
       )
  )
)

(pushnew (string :Message-Read) *All-Event-Types*)

(defmethod (Task-Server :mark-message) (mailstream message)
  (flag/unflag-message Mailstream Message :Set :\\Seen)
  (Signal-Event mailstream message :Message-Read)
)

(defmethod (task-server :simple-major-task)
	   (sequence owner operation &rest args)
  (let ((*owning-window* owner))
       (declare (special *owning-window*))
       (yw:safe-lexpr-send sequence operation args)
  )
)

(defmethod (task-server :major-task) (sequence owner operation &rest args)
  (let ((next (send sequence :get-next-message))
	(*owning-window* owner)
       )
       (declare (special *owning-window*))
       (if next
	   (yw:safe-lexpr-send sequence operation next args)
	   (progn (send owner :Mouse-Select)
		  (if owner
		      (format-scroll-window owner
			      "~&No messages in \"~A.\""
			      (make-label-from-filter
				(send sequence :Sequence-Specifier)
			      )
		      )
		      (tv:notify (send owner :Prompt-Window)
			      "~&No messages in \"~A.\"."
			      (make-label-from-filter
				(send sequence :Sequence-Specifier)
			      )
		      )
		  )
	   )
       )
  )
)

(defmethod (task-server :read-sequence) (sequence owner)
  (send self :Major-Task sequence owner :Read-Message)
)

(defmethod (task-server :reply-to-sequence)
	   (sequence owner
	    &optional (all-p *reply-to-all-by-default*)
	              (inclusive-p *reply-inclusive-by-default*)
	   )
  (send self :Major-Task sequence owner :Reply-To-Message nil all-p inclusive-p)
)

(defmethod (task-server :Copy/move-sequence)
	   (sequence owner delete-p to &optional numbers)
  (send self :Major-Task sequence owner :Copy/move-Message
	to delete-p numbers
  )
)

(defmethod (task-server :Menu-Copy-sequence)
	   (sequence owner &optional numbers)
  (let ((to (zwei:read-defaulted-pathname-near-window
	      owner "Copy to file:"
	      (yw-zwei:copy/move-default-path
		(send sequence :Mailstream)
		yw:*default-copy-to-mailbox-name*
		yw:*default-copy-to-mailbox-type*
	      )
	    )
	)
       )
       (send self :Major-Task sequence owner :Copy/move-Message
	     to nil numbers
       )
  )
)

(defun andify-specifiers (specifiers)
  (if specifiers
      (if (rest specifiers)
	  (list :Sequence-And (first specifiers)
		(andify-specifiers (rest specifiers))
	  )
	  (first specifiers)
      )
      nil
  )
)

(defun read-date-or-null-string (stream)
  (let ((string (tv:read-string-and-trim stream)))
       (if (equal string "")
	   ""
	   (let ((val (time:parse-universal-time string)))
	        (ignore val)
		string
	   )
       )
  )
)

(setf (get :date-or-null-string 'tv:choose-variable-values-keyword)
      '(princ read-date-or-null-string nil nil nil
	      "Click left to input a new date from the keyboard."
       )
)

(defun read-sequence-with-menu
       (&key (just-specifier-p nil) (label "Define a sequence"))
  (multiple-value-bind
    (from to cc bcc subject text
     before on since
     mailbox-is
     answered-p deleted-p seen-p flagged-p
     pattern-string
    )
      (tv:values-using-menu
       (("" "From"       :String)
	("" "To"         :String)
	("" "CC"         :String)
	("" "BCC"        :String)
	("" "Subject"    :String)
	("" "Text"       :String)
	("" "Before"     :Date-Or-Null-String)
	("" "On"         :Date-Or-Null-String)
	("" "Since"      :Date-Or-Null-String)
        ("" "Mailbox-Is" :String)
	(nil "Answered" :Assoc
	    `(("Answered"   . ((:Sequence-Answered)))
	      ("Unanswered" . ((:Sequence-Unanswered)))
	      ("Neither"    . nil)
	     )
        )
	(nil "Deleted" :Assoc
	    `(("Deleted"   . ((:Sequence-Deleted)))
	      ("Undeleted" . ((:Sequence-~Deleted)))
	      ("Neither"   . nil)
	     )
        )
	(nil "Seen" :Assoc
	    `(("Seen"    . ((:Sequence-Seen)))
	      ("Unseen"  . ((:Sequence-Unseen)))
	      ("Neither" . nil)
	     )
        )
	(nil "Flagged" :Assoc
	    `(("Flagged"   . ((:Sequence-Flagged)))
	      ("Unflagged" . ((:Sequence-~flagged)))
	      ("Neither"   . nil)
	     )
        )
	("" "Pattern String"   :String)
      )
      :Label label
     )
    (let ((specifiers
	    (append (if (equal "" from)    nil `((:Sequence-From    ,from)))
		    (if (equal "" to)      nil `((:Sequence-To      ,to)))
		    (if (equal "" cc)      nil `((:Sequence-Cc      ,cc)))
		    (if (equal "" bcc)     nil `((:Sequence-Bcc     ,bcc)))
		    (if (equal "" subject) nil `((:Sequence-Subject ,subject)))
		    (if (equal "" text)    nil `((:Sequence-Text    ,text)))
		    (if (equal "" before)  nil `((:Sequence-before  ,before)))
		    (if (equal "" on)      nil `((:Sequence-on      ,on)))
		    (if (equal "" since)   nil `((:Sequence-since   ,since)))
		    (if (equal "" mailbox-is)
			nil
			`((:Sequence-mailbox-is   ,mailbox-is))
		    )
		    answered-p deleted-p seen-p flagged-p
		    (if (equal "" pattern-string)
			nil
			(let ((closure
				(let ((*make-sequence-with-no-mailbox-ok-p* t))
				     (parse-a-sequence-from-string
				       pattern-string (get-mail-control-window)
				     )
				)
			      )
			     )
			     (if (closurep closure)
				 (list
				   (send (funcall closure) :Sequence-Specifier)
				 )
				 (ferror nil "Error in pattern string ~S"
					 pattern-string
				 )
			     )
		        )
		    )
	    )
	  )
	 )
         (if specifiers
	     (if just-specifier-p
		 (andify-specifiers specifiers)
		 (let ((*make-sequence-with-no-mailbox-ok-p* t))
		      (simple-sequence (andify-specifiers specifiers))
		 )
	     )
	     nil
	 )
    )
  )
)

(defmethod (Task-Server :Select-Headers-Using-Menu)
	   (sequence owner &optional numbers)
  (ignore sequence numbers)
  (let ((*mailer* owner))
       (let ((sequence
	       (send owner :Eval-Inside-Yourself
		     '(read-sequence-with-menu :Label "Select Which Messages?")
	       )
	     )
	    )
	    (if sequence
		(send owner :Select-Headers-1 sequence nil nil)
		(beep)
	    )
       )
  )
)

(defmethod (task-server :Menu-Move-sequence)
	   (sequence owner &optional numbers)
  (let ((to (zwei:read-defaulted-pathname-near-window
	      owner "Move to file:"
	      (yw-zwei:copy/move-default-path
		(send sequence :Mailstream)
		yw:*default-move-to-mailbox-name*
		yw:*default-move-to-mailbox-type*
	      )
	    )
	)
       )
       (send self :Major-Task sequence owner :Copy/move-Message
	     to t numbers
       )
  )
)

(defmethod (task-server :hardcopy-sequence) (sequence owner &optional numbers)
;  (send self :Major-Task sequence owner :hardcopy-Message)
  (send self :Simple-major-Task sequence owner
	:hardcopy-Message nil numbers
  )
)

(defmethod (task-server :toggle-flagged-sequence)
	   (sequence owner &optional (numbers nil))
  (send self :Simple-major-Task sequence owner
	:flag-Message nil :Toggle numbers
  )
)

(defmethod (task-server :toggle-Seen-sequence)
	   (sequence owner &optional (numbers nil))
  (send self :Simple-major-Task sequence owner
	:Mark-Message nil :Toggle numbers
  )
)

(defmethod (task-server :flag-sequence) (sequence owner &optional (numbers nil))
  (send self :Simple-major-Task sequence owner
	:flag-Message nil t numbers
  )
)

(defmethod (task-server :unflag-sequence)
	   (sequence owner &optional (numbers nil))
  (send self :Simple-major-Task sequence owner
	:flag-Message nil nil numbers
  )
)

(defmethod (task-server :delete-or-undelete-sequence)
	   (sequence owner delete-p &optional numbers)
  (send self :Simple-Major-Task sequence owner
	:Delete-Message nil delete-p numbers
  )
)

(defmethod (task-server :mark-sequence-as-answered)
	   (sequence owner &optional (numbers nil))
  (send self :Simple-major-Task sequence owner
	:Mark-Message-as-answered nil t numbers
  )
)

(defmethod (task-server :delete-sequence) (sequence owner &optional numbers)
  (send self :Simple-Major-Task sequence owner
	:Delete-Message nil t (or numbers (send sequence :Numberise-Messages))
  )
)

(defmethod (task-server :undelete-sequence) (sequence owner &optional numbers)
  (send self :Simple-Major-Task sequence owner
	:Delete-Message nil nil (or numbers (send sequence :Numberise-Messages))
  )
)

(defmethod (task-server :menu-keyword-sequence)
	   (sequence owner &optional numbers)
  (ignore numbers owner)
  (let ((numbers (send sequence :Numberise-Messages)))
       (if (> (length numbers) 1)
	   (yw-error "There are too many messages in ~S" sequence)
	   (multiple-value-bind (set unset all)
	       (send sequence :Keyword-Data-For (first numbers))
	     (ignore set)
	     (let ((alist (mapcar #'(lambda (X) (list (second X) :Value X))
				    all
			  )
		   )
		  )
	          (let ((highlit
			  (remove-if #'(lambda (X) (member (third X) unset))
				     alist
			  )
			)
		       )
		       (let ((choices (w:multiple-menu-choose
					alist
					:Label
					 (format
					   nil
					   "Set/UnSet keywords for message ~D"
					   (first numbers)
					 )
					:Highlighted-Items highlit
				      )
			     )
			    )
			    (if choices
			        (loop for key in all do
				      (Flag/Unflag-Message
					(send sequence :Mailstream)
					numbers
					(if (member key choices) :Set :Clear)
					(first key)
				      )
				)
				nil
			    )
		       )
		  )
	     )
	   )
       )
  )
)

(defmethod (task-server :forward-sequence) (sequence owner)
  (send self :Major-Task sequence owner :Forward-Message)
)

(defmethod (task-server :remail-sequence) (sequence owner)
  (send self :Major-Task sequence owner :Remail-Message)
)

(defmethod (task-server :Flush-Display-Cache-For)
	   (message &optional (force-p nil))
  (let ((caches-to-flush
	  (loop with queue = (without-interrupts task-stream)
		for task-entry in queue
		for next in (rest queue)
		for (method cache) = (task-queue-entry-task task-entry)
		for (next-method next-cache) = (task-queue-entry-task next)
		for ok-to-collect = (equal method :Flush-Display-Cache-For)
		for ok = (and ok-to-collect
			      (equal next-method :Flags-Changed)
			      (eq cache next-cache)
			      (or force-p
				  (should-force-p
				    message (cache-mailstream message) force-p
				  )
			      )
			 )
		when ok
		do (send self :Snip-Out-Task next)
		when ok-to-collect
		do (send self :Snip-Out-Task task-entry)
		when ok-to-collect collect cache
	  )
	)
       )
       (send self :Flush-Display-Caches-For
	     (sort (cons message caches-to-flush) #'< :Key 'cache-msg#)
       )
  )
)


(defmethod (task-server :Flush-Display-Caches-For)
	   (messages &optional (force-p t))
  (with-delayed-redisplay-of-items
    (loop for message in messages
	  do (Flush-Display-Cache-For message force-p)
    )
  )
)


(defmethod (task-server :Maybe-Flush-Search-Cache-for-flags)
	   (mailstream cache-entry fetch/store)
  (send mailstream :Maybe-Flush-Search-Cache fetch/store :flags
	(cache-flags cache-entry) cache-entry
  )
)

(defmethod (Task-Server :Flags-Changed-Internal) (cache-entries mailstream)
  (let ((remove-from-windows nil)
	(add-to-windows nil)
	(earliest (first cache-entries))
       )
       (loop for cache-entry in cache-entries
	     for items = (all-items-for-message cache-entry)
	     do (loop for item in items
		      when (not (send (window item) :Accept-Message-P
				      cache-entry
		                )
			   )
		      do (pushnew item remove-from-windows)
		)
	        (loop for window in (set-difference
				      (send mailstream :Associated-Windows)
				      (mapcar 'window items)
				    )
		      when (send window :Accept-Message-P cache-entry)
		      do (pushnew window add-to-windows)
		)
       )
       (if add-to-windows
	   (send self :Flags-Changed-add-to-windows Cache-entries mailstream
		 add-to-windows earliest
	   )
	   nil
       )
       (if remove-from-windows
	   (send self :Flags-Changed-remove-from-windows Cache-entries
		 mailstream remove-from-windows earliest
	   )
	   nil
       )
  )
)

(defmethod (Task-Server :Flags-Changed-add-to-windows)
       (cache-entries mailstream add-to-windows earliest)
  (loop for window in add-to-windows
	when (send window :Filter) do
	(send window :Set-Up (send window :Owner)
	      (send window :Filter)
	      (send (send window :Owner)
		    :Find-Mailbox-Name (list-if-not mailstream)
	      )
	      (list-if-not mailstream) earliest t
	)
	(loop for cache-entry in cache-entries
	      for item
	          = (send window :Find-Display-Item-For-Message cache-entry)
	      when item do (Add-Item cache-entry item window)
	)
  )
  (loop for cache-entry in cache-entries
	do (send mailstream :maybe-add-to-Computed-Orders cache-entry)
  )
)


(defmethod (Task-Server :Flags-Changed-remove-from-windows)
       (cache-entries mailstream remove-from-windows earliest)
  (loop for item in remove-from-windows
	do (Remove-Item (object item) item)
  )
  (loop for window in (remove-duplicates (mapcar 'window remove-from-windows))
	do (send window :Set-Up (send window :Owner)
		 (send window :Filter)
		 (send (send window :Owner)
		       :Find-Mailbox-Name (list-if-not mailstream)
		 )
		 (list-if-not mailstream) earliest t
	   )
  )
  (loop for cache-entry in cache-entries
	do (send mailstream :Mark-Computed-Orders-For-Invalidation cache-entry)
  )
)

(defmethod (Task-Server :Flags-Changed)
	   (message mailstream &optional (force-p nil))
  ;;; Gather up all comparable flags-changed messages.
  (let ((cache-entries
	  (cons message
	    (loop for task-entry in (without-interrupts task-stream)
		  for (method cache stream) = (task-queue-entry-task task-entry)
		  for ok = (and (equal method :Flags-Changed)
				(equal (cache-mailstream cache) mailstream)
			   )
		  when ok
		  do (send self :Snip-Out-Task task-entry)
		  when ok
		  collect cache
	    )
	  )
	)
       )
       (send self :Flush-Display-Caches-For cache-entries
	     (if force-p :Flags-Changed nil)
       )
       (if cache-entries
	   (send self :Flags-Changed-Internal
		 (sort cache-entries 'message<)
		 mailstream
	   )
	   nil
       )
  )
)

(defmethod (task-server :preempt-header) (cache-entry)
  (or (is-present (cache-envelope cache-entry))
      (let ((mailstream (cache-mailstream cache-entry))
	    (message-number (cache-msg# cache-entry))
	    (max (send (cache-mailstream cache-entry) :Messagecnt))
	   )
	   (maybe-preempt-envelopes
	     mailstream
	     (loop for i
		   from (- message-number *daemon-header-read-grain-size*)
		   to (+ message-number *daemon-header-read-grain-size*)
		   when (and (> i 0) (<= i max))
		   collect i
	     )
	   )
	   (Prepare-For-Display cache-entry)
	   (send self :Flush-Display-Cache-For cache-entry)
      )
  )
)

;-------------------------------------------------------------------------------


(defun default-daemon-mailbox-open-operations ()
  *default-daemon-mailbox-open-operations*
)

(defun default-daemon-actions ()
"Returns a list of the actions that the yw daemon will perform in the
background.  Each element in the list consists of a method name on the
flavor yw:yw-daemon followed by its arguments.  This function uses
yw:*Default-Daemon-Actions* directly to compute the actions to perform. 
You can, therefore, change the background behaviour by changing either of these.
See the documentation for yw:*Default-Daemon-Actions* for the legal methods.
"
  *default-daemon-actions*
)

(defflavor yw-daemon
	   ((mailbox-open-operations (default-daemon-mailbox-open-operations))
	    (daemon-actions (default-daemon-actions))
	   )
	   (Task-Server tv:process-mixin tv:window)
  (:Default-Init-Plist
    :Process '(Yw-daemon
		:Priority -2
		:special-pdl-size 4000
		:regular-pdl-size 10000
	      )
    :Width  (min 300 (+ 201 (send tv:default-screen :Width)))
    :Height (min 700 (send tv:default-screen :Height))
    :Position '(0 0)
    :Deexposed-Typeout-Action :Permit
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
)

(defun yw-daemon (daemon)
  (let ((*terminal-io* daemon))
       (declare (special *terminal-io*))
       (loop do (send daemon :Daemon))
  )
)

(defmethod (Yw-daemon :eager-message-bodies) (sequence)
  (send sequence :Map-Over-Messages :eager-message)
)

(defun map-resource-return
   (function resource-name &optional (include-function #'(lambda (ignore) t))
    &rest extra-args &aux resource
   )
  "Call FUNCTION on each object created in resource RESOURCE-NAME.
FUNCTION gets three args at each call: the object, whether the resource
believes it is in use, and RESOURCE-NAME.  Return a list of the answers."
  (check-arg resource-name
	     (setq resource (si:get-resource-structure resource-name))
	     "the name of a resource")
  ;;Windows are the user's problem....
  (loop for i from 0 below (si:resource-n-entries resource)
	by si:resource-entry-size
	for object = (si:resource-entry resource i)
	when (and object
		  (or (not include-function)
		      (funcall include-function object)
		  )
	     )
	collect
	(apply function object (si:resource-inuse-p resource i) resource-name
		    extra-args
	)
  )
)

(defun remove-from-resource (resource-name object &aux resource n-entries)
"Remove OBJECT the resource RESOURCE-NAME.  OBJECT should have been returned
by a previous call to ALLOCATE-RESOURCE.
"
  (check-arg resource-name
	     (setq resource (sys:get-resource-structure resource-name))
	     "the name of a resource"
  )
  (unless (get resource-name 'sys:no-memory)
    (setq n-entries (sys:resource-n-entries resource))
    (loop for n
	  from (- n-entries sys:resource-entry-size)
	  downto 0 by sys:resource-entry-size
	  when (eq (sys:resource-entry resource n) object)
	  ;; Note that this doesn't need any locking.
	  do (let ((deallocator (sys:resource-deallocator resource)))
		  (when deallocator
		    (apply deallocator resource object
			   (sys:resource-parms resource n)
		    )
		  )
		  (return (setf (sys:resource-n-entries resource)
				(- (sys:resource-n-entries resource)
				   sys:resource-entry-size
				)
			  )
			  (setf (sys:resource-entry resource n) nil)
			  (setf (sys:resource-inuse-p resource n) nil)
		  )
	     )
	  finally (signal (make-condition
			    'sys:deallocate-non-resource-entry
			    "~S is not an object from the ~S resource"
			    object resource-name
			  )
			  :proceed-types '(:no-action)
		  )
    )
  )
)


(defun check-window-resource (resource flavor)
  (let ((changed
	  (map-yw-resource-return
	    #'(lambda (item &rest ignore) item)
	    resource
	    #'(lambda (item)
		(not (equal (sys:structure-total-size item)
			    (sys:flavor-instance-size (get flavor 'si:flavor))
		     )
		)
	      )
	  )
	)
	(non-empty
	  (map-yw-resource-return
	    #'(lambda (item &rest ignore) item)
	    resource #'(lambda (ignore) t)
	  )
	)
       )
       (if (or changed (and non-empty (si:mx-p)))
	   (progn (tv:notify tv:selected-window
			     "The flavor ~S, which defines the resource ~S has ~
                              changed.  Clearing the resource." flavor resource
                  )
		  (close-down-yw nil)
		  (clear-resource resource)
	   )
	   nil
       )
  )
)

(defun is-in-resource (something resource)
  (Map-Resource-Return #'(lambda (&rest ignore) t) resource
		       #'(lambda (X) (equal X something))
  )
)

(defun Active-Streams (window active-p resource-name)
  (ignore active-p resource-name)
  (let ((boxes (send window :All-Mailboxes)))
       boxes
  )
)

(defmethod (Yw-daemon :get-headers)
	   (name mailstream most-recent-n &optional (direction :forwards))
  (ignore name)
  (let ((TotalMsgs (send MailStream :MessageCnt)))
       (if (and TotalMsgs (not (send mailstream :Opened-With-Filter-P)))
	   (let ((missing
		   (loop for i
			 from (max 1 (- totalmsgs most-recent-n)) to TotalMsgs
			 unless (not (equal
				       :Unbound
				       (cache-envelope
					 (cache-entry-of i mailstream)
				       )
				     )
				)
			 collect i
		   )
		 )
		)
		(if missing
		    (progn (MAP-Fetch-Envelope MailStream
			       (remove nil
				 (safe-firstn *Daemon-Header-Read-Grain-Size*
					 (if (equal direction :forwards)
					     missing
					     (reverse Missing)
					 )
				 )
			       )
			   )
			   t
		    )
		    nil
		)
	   )
	   nil
       )
  )
)

(defun envelopes-present
       (mailstream &optional (report-stream *standard-output*))
"Tells you how many envelopes have been cached for mailstream reporting onto
report-stream.
"
  (let ((TotalMsgs (send MailStream :MessageCnt)))
       (if TotalMsgs
	   (let ((not-missing
		   (loop for i
			 from 1
			 to TotalMsgs
			 When (is-present
				(Cache-Envelope
				  (cache-entry-of i mailstream)
				)
			      )
			 collect i
		   )
		 )
		)
		(if not-missing
		    (format report-stream "~&Envelopes in ~A:- ~A"
			    (Print-Short-Mailbox-Name mailstream)
			    (colonify-numbers not-missing)
		    )
		    (format report-stream "~&No Envelopes")
		)
	   )
	   nil
       )
  )
)

(defun messages-present (mailstream &optional (report-stream *standard-output*))
"Tells you how many envelopes have been cached for mailstream reporting onto
report-stream.
"
  (let ((TotalMsgs (send MailStream :MessageCnt)))
       (if TotalMsgs
	   (let ((not-missing
		   (loop for i
			 from 1
			 to TotalMsgs
			 When (is-present
				(Cache-RFC822Text
				  (cache-entry-of i mailstream)
				)
			      )
			 collect i
		   )
		 )
		)
		(if not-missing
		    (format report-stream "~&Messages in ~A:- ~A"
			    (Print-Short-Mailbox-Name mailstream)
			    (colonify-numbers not-missing)
		    )
		    (format report-stream "~&No Messages")
		)
	   )
	   nil
       )
  )
)

(defmethod (Yw-daemon :bodies-present) (name mailstream &rest ignore)
  (ignore name)
  (messages-present mailstream self)
)

(defmethod (Yw-daemon :get-all-bodies)
	   (name mailstream &optional (direction :forwards))
  (ignore name)
  (let ((TotalMsgs  (send MailStream :MessageCnt))
	(RecentMsgs (send MailStream :RecentCnt))
       )
       (if (and TotalMsgs RecentMsgs)
	   (let ((missing
		   (loop for i
			 from 1
			 to TotalMsgs
			 unless (is-present
				  (Cache-RFC822Text
				    (cache-entry-of i mailstream)
				  )
				)
			 collect i
		   )
		 )
		)
		(if missing
		    (map-fetch-message MailStream
		      (if (equal direction :forwards)
			  (safe-firstn *daemon-body-read-grain-size* missing)
			  (reverse
			    (safe-firstn *daemon-body-read-grain-size* missing)
			  )
		      )
		    )
		    nil
		)
	   )
	   nil
       )
  )
)

(defun Get-Missing-envelopes (missing)
  (if missing
      (let ((target-stream (cache-mailstream (first missing)))
	    (failed nil)
	   )
	   (let ((to-fetch
		   (loop for message in missing
			 if (eq target-stream (cache-mailstream message))
			 collect message
			 else do (push message failed)
		   )
		 )
		)
;	     (cl:break)
	        (map-fetch-envelope target-stream to-fetch)
		(Get-Missing-Envelopes failed)
	   )
      )
      nil
  )
)

(defun Get-Missing-Headers
       (missing direction &optional (n *daemon-header-read-grain-size*))
  (if missing
      (let ((grain-to-get
	      (if (equal direction :forwards)
		  (safe-firstn n missing)
		  (reverse
		    (safe-firstn n missing)
		  )
	      )
	    )
	   )
	   (let ((target-stream (cache-mailstream (first grain-to-get))))
	        (map-fetch-header
		  target-stream
		  (loop for message in grain-to-get
			when (eq target-stream (cache-mailstream message))
			collect message
		  )
		)
	   )
      )
      nil
  )
)

(defun get-missing-messages (missing direction)
  (if missing
      (let ((grain-to-get
	      (if (equal direction :forwards)
		  (safe-firstn *daemon-body-read-grain-size* missing)
		  (reverse
		    (safe-firstn *daemon-body-read-grain-size* missing)
		  )
	      )
	    )
	   )
	   (let ((target-stream (cache-mailstream (first grain-to-get))))
	        (map-fetch-message
		  target-stream
		  (loop for message in grain-to-get
			when (eq target-stream (cache-mailstream message))
			collect message
		  )
		)
	   )
      )
      nil
  )
)

(defmethod (Yw-daemon :get-recent-bodies)
	   (name mailstream &optional (most-recent-n nil) (direction :forwards))
  (ignore name)
  (let ((TotalMsgs  (send MailStream :MessageCnt))
	(RecentMsgs (send MailStream :RecentCnt))
       )
       (if (and TotalMsgs RecentMsgs
		(not (send mailstream :Opened-With-Filter-P))
	   )
	   (let ((missing
		   (loop for i
			 from
			 (max 1 (- totalmsgs
				   (max RecentMsgs (or most-recent-n 1))
				)
			 )
			 to TotalMsgs
			 for cache = (cache-entry-of i mailstream)
			 unless (is-present (Cache-RFC822Text cache))
			 collect cache
		   )
		 )
		)
	        (get-missing-messages missing direction)
	   )
	   nil
       )
  )
)

(defmethod (Yw-daemon :get-nearby-bodies)
	   (name ignore
	    &optional (next-n *daemon-body-read-grain-size*)
	              (direction :forwards)
	   )
  (ignore name)
  (let ((messages-being-read
	  (loop for buffer in zwei:*zmacs-buffer-list*
		when (and (equal (get buffer :Buffer-Type) :Read)
			  (get buffer :Message-Sequence)
		     )
		collect (list (yw-zwei:get-message-number buffer)
			      (send (get buffer :Message-Sequence) :Mailstream)
			)
	  )
	)
	(all-message-entries nil)
       )
       (loop for (number stream) in messages-being-read
	     for total-messages = (send stream :messagecnt) do
	     (loop for i
		   from (max 1
			     (if (equal direction :Forwards)
				 number
				 (- number next-n)
			     )
			)
		   to   (min total-messages
			     (if (equal direction :Forwards)
				 (+ number next-n)
				 number
			     )
			)
		   for cache = (cache-entry-of i stream)
		   unless (is-present (Cache-RFC822Text cache))
		   do (let ((entry (assoc stream all-message-entries)))
			   (if entry
			       (if (member cache entry :Test #'eq)
				   nil
				   (setf (rest entry)
					 (cons cache (rest entry))
				   )
			       )
			       (push (list stream cache) all-message-entries)
			   )
		      )
	     )
       )
       (loop for (mailstream . missing) in all-message-entries do
	     (get-missing-messages missing  (complement-direction direction))
       )
  )
)

(defun safe-firstn (n list)
  (loop for i from 1 to n
	for element in list
	collect element
  )
)

(defmethod (Yw-daemon :Get-And-Cache-Directory-For-Completion) (&rest ignore)
  (let ((*cache-directory-lists-p* t)
	(*look-in-directory-cache-first-p* t)
       )
       (let ((path (fs:make-pathname :Host *user-host*
			   :Directory (default-mailbox-directory)
			   :Name :Wild
			   :Type :Wild
			   :Version :Newest
			   :Device
			     (send (net:parse-host *user-host*) :Default-Device)
	           )
	     )
	    )
	    (or (and (gethash path *directory-list-hash-table*) path)
		(fs:directory-list path)
	    )
       )
       
  )
)

(defun get-and-cache-all-newsgroups (&optional (mailstream nil))
  (loop for stream in *all-open-imap-streams*
	for boxes = (send stream :all-mailboxes)
	when (and (or (not mailstream)
		      (equal (send mailstream :Host) (send stream :Host))
		  )
		  (> (length boxes) 1)
	     )
	do (loop for str in *all-open-imap-streams*
		 when (equal (send mailstream :Host) (send stream :Host))
		 do (send str :Set-All-Mailboxes boxes)
	   )
	   (return boxes)
	finally (if mailstream
		    (progn (nntp-send mailstream :List nil)
			   (send mailstream :all-mailboxes)
		    )
		    nil
		)
  )
)

(defmethod (Yw-daemon :Get-And-Cache-NetNews-NewsGroups-For-Completion)
	   (name mailstream &rest ignore)
  (ignore name)
  (if (and (typep mailstream 'ip-nntp-stream)
	   (not (send mailstream :all-mailboxes))
      )
      (get-and-cache-all-newsgroups mailstream)
      nil
  )
)

(defmethod (Yw-daemon :Get-Some-Sort-Of-Flagged-Bodies)
  (name mailstream sequence-keyword
   &optional (most-recent-n nil) (direction :forwards) (bboards-too nil)
  )
  (let ((TotalMsgs  (send MailStream :MessageCnt))
	(dummy-sequence (Make-A-Dummy-Sequence name mailstream (Any-Mailer)))
       )
       (if (and (not (send mailstream :Opened-With-Filter-P))
		(or (not (send mailstream :Bboard-P)) bboards-too)
		TotalMsgs
	   )
	   (let ((missing
		   (loop for i
			 from (if (equal most-recent-n :All)
				  1
				  (max 1 (- totalmsgs (or most-recent-n 1)))
			      )
			 to TotalMsgs
			 for cache = (cache-entry-of i mailstream)
			 unless (or (send dummy-sequence sequence-keyword cache)
				    (is-present (Cache-RFC822Text cache))
				)
			 collect cache
		   )
		 )
		)
	        (get-missing-messages missing direction)
	   )
	   nil
       )
  )
)

(defmethod (Yw-daemon :Get-Unseen-Bodies)
  (name mailstream &optional (most-recent-n nil) (direction :forwards)
   (bboards-too nil)
  )
  (send self :Get-Some-Sort-Of-Flagged-Bodies name mailstream :Sequence-Seen
	most-recent-n direction bboards-too
  )
)

(defmethod (Yw-daemon :Get-Flagged-Bodies)
  (name mailstream &optional (most-recent-n nil) (direction :forwards)
  )
  (send self :Get-Some-Sort-Of-Flagged-Bodies name mailstream
	:Sequence-Unflagged most-recent-n direction nil
  )
)

(defmethod (Yw-daemon :Get-Selected-Bodies)
  (name mailstream &optional (direction :forwards) (bboards-too nil))
"Gets the messages selected by the user in a specific filter for this
 mailstream.  For instance, if the user selected Get <foo> <sequence>,
 then it wil fetch all of the matching messages for the sequence.
"
  (ignore name)
  (let ((TotalMsgs (send MailStream :MessageCnt)))
       (if (and (send mailstream :Opened-With-Filter-P)
		(or (not (send mailstream :Bboard-P)) bboards-too)
		TotalMsgs
		(send mailstream :Associated-Windows)
	   )
	   (let ((sequence
		   (send (first (last (send mailstream :Associated-Windows)))
			 :Filter
		   )
		 )
		)
	        (if sequence
		    (let ((messages (send sequence :Computed-Order-Safe)))
			 (let ((missing
				 (loop for message in messages
				       unless 
				       (is-present
					 (Cache-RFC822Text message)
				       )
				       collect message
				 )
			       )
			      )
			      (Get-Missing-Messages missing direction)
			 )
		    )
		    nil
		)
	   )
	   nil
       )
  )
)

(defun flush-display-cache-for (message &optional (force-p t))
  (let ((mailstream (cache-mailstream message)))
       (if (send mailstream :Open-P)
	   (without-recursion ()
	     (let ((window (send mailstream :Owning-Window)))
		  (setf (Cache-SubjectText message) :Unbound)
		  (setf (cache-fromtext    message) :Unbound)
		  (setf (Cache-ToText      message) :Unbound)
		  (let ((should-force-p
			  (should-force-p message mailstream force-p)
			)
		       )
		       (mapcar
			 #'(lambda (sum)
			     (if (and (member mailstream
					      (send sum :mailstreams) :Test #'eq
				      )
				      should-force-p
				 )
				 (if (or (not (send sum :Filter))
					 (send (send sum :Filter)
					       :Accept-Message-P message
					 )
				     )
				     (send sum :Flush-Display-For
					   message should-force-p
				     )
				     (send sum :Maybe-Redisplay-Selected-Item
					   message
				     )
				 )
				 nil
			     )
			   )
			   (and window (send window :All-Summary-Windows))
		       )
		  )
	     )
	   )
       )
  )
)


(defmethod (Yw-daemon :Daemon-Actions) ()
  (if (set-difference daemon-actions (default-daemon-actions))
      (setq daemon-actions (default-daemon-actions))
      nil
  )
  daemon-actions
)

(defmethod (yw-daemon :Daemonise-stream) (name mailstream)
  ;;; Only execute one daemon action per loop.  That way we shouldn't get
  ;;; bogged down.
  (loop for (method . args) in (send self :Daemon-Actions)
	for index from 0 to (- (length daemon-actions) 1)
	do (if (yw:safe-lexpr-send self :Send-If-Handles
				   method name mailstream args
               )
	       (setq daemon-actions
		     (append (nthcdr index daemon-actions)
			     (firstn index daemon-actions)
		     )
	       )
	       nil
	   )
  )
)

(defun any-mailer (&optional (no-error-p nil))
  "Finds any mail control window that it can.  IF no-error-p is true then it
returns nil if it can't find one, otherwise it barfs."
  (declare (special *owning-window*))
  (if (and (boundp '*mailer*)
	   *mailer*
      )
      *mailer*
      (if (and (boundp '*owning-window*)
	       *owning-window*
	  )
	  *owning-window*
	  (if *all-mail-control-windows*
	      (first *all-mail-control-windows*)
	      (if no-error-p
		  nil
		  (yw-error "Can't find any mail control windows.")
	      )
	  )
      )
  )
)

(defun Map-Over-Control-Windows
       (function &optional
	(applicable-function #'(lambda (window) (ignore window) t))
       )
  (loop for win in *all-mail-control-windows*
	when (funcall applicable-function win)
	collect (funcall function win nil nil)
  )
)

(defflavor close-worthy-error () (eh:error))

(defun close-worthy-error-handler (condition mailstream)
  (ignore condition mailstream)
  (tv:notify tv:selected-window "Error on stream ~A, closing it."
	     (send mailstream :Pretty-String)
  )
  (send mailstream :clean-up-after-close)
  (throw 'close-worthy-error :Close-Worthy-Error)
)

(defun no-closed-streams (in)
  (typecase in
    (cons (and (no-closed-streams (first in)) (no-closed-streams (rest in))))
    (imap-stream (send in :Open-P))
    (message-sequence (no-closed-streams (send in :Mailstream)))
    (otherwise t)
  )
)

(defun check-for-wedged-mail ()
  (if (or (null fs:user-id) (string-equal fs:user-id ""))
      nil
      (let ((dir (catch-error
		   (fs:directory-list
		     (make-pathname :host "LM"
				    :directory '("MAILER")
				    :Name :wild
				    :type "WORK"
				    :version :wild
				    :Device
				      (send net:local-host :default-device)
		     )
		   )
		   nil
		 )
	    )
	    (now (time:get-universal-time))
	   )
	   (let ((filtered
		   (loop for (path . plist) in (rest dir)
			 when (> (- now (getf plist :Creation-Date))
				 *delay-before-notification-of-wedged-mail*
			      ) ;; 4 minutes by default
			 collect (send path :String-For-Printing)
		   )
		 )
		)
		(if (and filtered
			 (loop for path in filtered collect (probe-file path))
		    )
		    (progn
		      (if (sys:mx-p)
			  (format-scroll-window nil
			    "There seems to be mail backed up in the ~
                             lm:mailer; directory!~%Perhaps either the mailer ~
                             daemon is wedged or your smtp~%server is not ~
                             listening.~%~
			     File~P: ~~{~A~^~%~}~~%This might also be due to ~
                             clock skew between your ~%Mac and your mX."
			    (length filtered)
			    filtered
			  )
			  (format-scroll-window nil
			    "There seems to be mail backed up in the ~
                             lm:mailer; directory!~%Perhaps either the mailer ~
                             daemon is wedged or your smtp~%server is not ~
                             listening.~%~
			     File~P: ~~{~A~^~%~}~"
			    (length filtered)
			    filtered
			  )
		      )
		      (beep :shoop)
		      t
		    )
		    nil
		)
	   )
      )
  )
)

(defmethod (yw-daemon :Daemon) ()
  (multiple-value-bind (task-object found-p) (send self :get-task)
    (let ((task (if task-object (task-queue-entry-task task-object) nil)))
	 (if (and found-p (no-closed-streams task))
	     (yw:safe-lexpr-send self (first task) (rest task))
	     (let ((all-streams
		     (remove-duplicates
		       (apply #'append
			      (map-over-control-windows	'active-streams)
		       )
		       :Test #'equalp
		     )
		   )
		  )
		  (loop for mailstream in all-streams
			for name = (send mailstream :mailbox)
			when (send mailstream :open-p)
			do (with-close-worthy-errors-handled (mailstream)
			     (send self :daemonise-stream name mailstream)
			   )
		  )
		  (loop for mailstream in all-streams
			for name = (send mailstream :mailbox)
			;;; !!!!! N.B. We should really just peek at the
			;;; streams, but that hasn't been implemented at
			;;; the other end yet [!].
			when (and (send mailstream :open-p)
				  (not (feature-enabled-p
					 :New.Mail.Notify mailstream
				       )
				  )
			     )
			do (with-close-worthy-errors-handled (mailstream)
			     (map-check-mailbox mailstream)
			   )
		  )
		  (map-over-control-windows
		    #'(lambda (window &rest ignore)
			 (send window :Recompute-Icon)
		       )
		  )
	      )
	 )
    )
  )
  (loop for action in *end-of-daemon-actions* do (funcall action))
  (sleep *yw-daemon-sleep-interval* "Sleep")
)

;-------------------------------------------------------------------------------


(defflavor rule-processor
	   ((agenda nil)
	    (event-list nil)
	   )
	   (task-server)
  :Gettable-Instance-Variables
  :Settable-Instance-Variables
  :Initable-Instance-Variables
  (:Default-Init-Plist
    :Process '(rule-processor-top-level
		:special-pdl-size 4000
		:regular-pdl-size 10000
	      )
  )
)

(defun rule-processor-top-level (window)
  (let ((*terminal-io* window)
	(*sequences-created-are-associated* nil)
       )
       (declare (special *terminal-io*))
       (loop do (send window :process-event))
  )
)

(defmethod (Rule-Processor :Pick-Next-Agenda-Entry) ()
  (without-interrupts (pop agenda))
)

(defmethod (Rule-Processor :Process-Agenda-Entry) ()
  (let ((Agenda-Entry (send self :pick-next-agenda-entry)))
       (send (Agenda-Entry-Rule-Set Agenda-Entry) :Apply-Self
	     (Agenda-Entry-Event Agenda-Entry) t
       )
  )
)

(defmethod (Rule-Processor :Process-Event) ()
  (declare (optimize (speed 3) (safety 0)))
  (declare (special *All-Rule-Sets*))
  (Process-wait "Await Event"
		#'(lambda (me) (or (send me :Agenda) (send me :event-list)))
		self
  )
  (if agenda
      (send self :process-agenda-entry)
      (let ((Event (without-interrupts (pop event-list))))
	   (loop for Rule-Set in *All-Rule-Sets*
		 when (send Rule-Set :Applicable-P Event)
		 do (without-interrupts
		      (push (make-agenda-entry :Rule-Set Rule-Set :Event  Event)
			    agenda
		      )
		    )
	   )
      )
  )
)

(Defmethod (rule-processor :add-event) (Event)
  (without-interrupts (push Event event-list))
)

(defun signal-event (mailbox message Event-Type)
  (declare (optimize (speed 3) (safety 0)))
  (declare (special *rule-processor*))
  (let ((Event (Make-Event :Mailbox mailbox
			   :Message message
			   :Type (string Event-Type)
	       )
	)
       )
       (send *rule-processor* :Add-Event event)
       event
  )
)

