;;; -*- Mode:Common-Lisp; Package:System-Internals; Base:10 -*-



(defun should-optimise ()
"Is true if optimisations should be made."
  (>= (compiler:opt-speed  compiler:optimize-switch)
      (+ (compiler:opt-safety compiler:optimize-switch) 1)
  )
)

(defsubst really-make-array-into-named-structure (array nss)
"A subst which, given an array will turn it into a generic named structure."
  (if (array-has-leader-p array)
      (setf (array-leader array 1) nss)
      (setf (aref array 0) nss)
  )
  (%p-dpb-offset 1 %%array-named-structure-flag array 0)
  array
)


(defsubst really-internal-make-named-structure (length leader-length)
"A subst, which open codes the production of a named structure."
  (let* ((array-header 
	   (%logdpb 1 %%array-leader-bit
		    (%logdpb 1 %%array-number-dimensions  
			     (%logdpb 7 %%array-type-field
				      (%logdpb 1 %%array-simple-bit length)
			     )
		    )
	   )
	 )
	 (qs-in-leader (+ 2 leader-length))
	 (vector
	   (%allocate-and-initialize-array 
	     array-header
	     length
	     leader-length
	     nil
	     (+ 1 qs-in-leader length)
	   )
	 )
	)
        (values vector length)
  )
)


(defsubst really-make-defstruct (length type)
" A subst, which makes a named defstruct instance in a totally open coded
 manner.
"
  (let ((array (really-internal-make-named-structure length 2)))
       (declare (unspecial array))
       (really-make-array-into-named-structure array type)
       array
  )
)

(defun turn-inits-into-lets (inits slot-alist)
" Given a list of slot initialisations for a defstruct and the slot alist from
 the defstruct definition it returns a list of let defintions, which will
 initialise the slots.
"
  (if inits
      (if (rest inits)
	  (if (and (keywordp (first inits))
		   (assoc (first inits) slot-alist :test #'string-equal)
	      )
	      (cons (list (first (assoc (first inits) slot-alist
					:test #'string-equal
				 )
			  )
			  (second inits)
		    )
		    (turn-inits-into-lets (rest (rest inits)) slot-alist)
	      )
	      (ferror nil "Unknown slot keyword ~S" (first inits))
	  )
	  (ferror nil "Not enough inits for make-* operation.")
      )
      nil
  )
)

(defun find-type-initialisation (type alist)
  (if alist
      (if (equal (list 'quote type)
		 (sys:defstruct-slot-description-init-code
		   (rest (first alist))
		 )
	  )
	  (first alist)
	  (find-type-initialisation type (rest alist))
      )
      (ferror nil "Cannot find the init for the type name.")
  )
)


(defun make-setfs (type array-name names alist named-array-p)
"Makes a collection of Setfs for an optimised defstruct definition."
  (using-defstruct-special-variables)
  (if (not named-array-p)
      (let ((code (defstruct-type-description-accessor-code
		    (get type :saved-type-description)
		  )
	    )
	    (init (find-type-initialisation type alist))
	   )
	   (cons (set-slot code array-name
			   (sys:defstruct-slot-description-number (rest init))
			   (sys:defstruct-slot-description-init-code
			     (rest init)
			   )
		  )
		  (make-setfs type array-name names (remove init alist) t)
	   )
      )
      (if names
	  (cons `(setf (,(sys:defstruct-slot-description-ref-macro-name
			   (rest (assoc (first names) alist))
			 )
			,array-name
		       )
		       ,(first names)
		 )
		 (make-setfs type array-name (rest names)
			     (remove (assoc (first names) alist) alist)
			     named-array-p
		 )
	  )
	  (if alist
	      (if (or (equal '%defstruct-empty%
			     (sys:defstruct-slot-description-init-code
			       (rest (first alist))
			     )
		      )
		      (defstruct-slot-description-name-slot-p
			(rest (first alist))
		      )
		  )
		  (make-setfs type array-name nil (rest alist) named-array-p)
		  (cons `(setf (,(sys:defstruct-slot-description-ref-macro-name
				   (rest (first alist))
				 )
				,array-name
			       )
			       ,(sys:defstruct-slot-description-init-code
				  (rest (first alist))
				)
			 )
			 (make-setfs
			   type array-name nil (rest alist) named-array-p
			 )
		  )
	      )
	      nil
	  )
      )
  )
)

(defun find-constructor (name constructors)
"Finds the defstruct constructor specification for Name in the list of
 constructors.  The match is either a symbol, which is a normal constructor,
 or it is a list of the form (Name (slot1 slot2...)), which is the result of
 (:Constructor Name (slot1 slot2...)) in the original defstruct.
"
  (if constructors
      (if (consp (first constructors))
	  (if (equal (first (first constructors)) name)
	      (second (first constructors))
	      (find-constructor name (rest constructors))
	  )
	  (if (equal (first constructors) name)
	      name
	      (find-constructor name (rest constructors))
	  )
      )
      (ferror nil "Cannot find a constructor description for ~S" name)
  )
)

(defun match-inits-with-constructor (name constructor inits)
"Given a constructor specifier and a list of inits being passed to that
 constructor, returns a list of the form (:slot1 init1 :slot2 init2...).
"
  (if (consp constructor)
      (if constructor
	  (if inits
	      (cons (intern (first constructor) "KEYWORD")
		    (cons (first inits)
			  (match-inits-with-constructor
			    name (rest constructor) (rest inits)
			  )
		    )
	      )
	      (ferror nil "Not enough inits for constructor ~S" name)
	  )
	  (if inits
	      (ferror nil "Too many inits for constructor ~S" name)
	      nil
	  )
      )
      inits
  )
)


(defun optimise-make-x (type form maker)
"Given a type, a constructor form and a maker form, which makes the base type
 for Type, it returns an optimised form of Form with the defstruct creation
 open coded fully.
"
  (let ((alist (sys:defstruct-description-slot-alist
		 (get type 'sys:defstruct-description)
	       )
	)
	(constructor
	  (find-constructor (first form)
			    (sys:defstruct-description-constructors
			      (get type 'sys:defstruct-description)
			    )
	  )
	)
       )
       (let ((inits (turn-inits-into-lets
		      (match-inits-with-constructor
			(first form) constructor (rest form)
		      )
		      alist
		    )
	     )
	     (named-array-p
	       (and (equal (first maker) 'make-array)
		    (get (cons nil maker) :named-structure-symbol)
	       )
	     )
	    )
	   `(let ,inits
	       (declare (unspecial ,@(mapcar #'first inits)))
	       (let ((structure
		      ,(if named-array-p
			   ;;; Makes a special case for named array type
			   ;;; defstructs.
		          `(si:really-make-defstruct ,(length alist) ',type)
		           maker
		       )
		     )
		    )
		    (declare (unspecial structure))
	          ,@(make-setfs type 'structure (mapcar #'first inits)
				alist named-array-p
		    )
		    structure
	       )
	    )
       )
  )
)


(defun make-optimiser-for-constructor-1 (cons-name keys)
"Makes and adds a compiler optimiser for constructor function being defined."
  (using-defstruct-special-variables)
  (let ((maker (apply (defstruct-type-description-bare-constructor
			type-description
		      )
		      name size subtype (name-offset) keys
	       )
	)
	(optimiser-name
	  (intern (string-append cons-name "-OPTIMISER")
		  (symbol-package cons-name)
	  )
	)
       )
       (let ((function `(lambda (form)
			  (if (should-optimise)
			      (optimise-make-x ',name form ',maker)
			      (list 'dont-optimize form)
			  )
			)
	     )
	    )
	    ;;; Install here and now so that we get the effect of
	    ;;; eval-when (compile load eval) even though we may be
	    ;;; nested a long way down.
	    (compile optimiser-name function)
	    (push `(defun ,optimiser-name (form)
		     (if (should-optimise)
			 (optimise-make-x ',name form ',maker)
			 (list 'dont-optimize form)
		     )
		   )
		   returns
	    )
       )
       (push `(compiler:defoptimizer ,cons-name ,optimiser-name) returns)
  )
)


(defun make-optimiser-for-constructor (args)
"Makes and adds a compiler optimiser for constructor function being defined."
  (using-defstruct-special-variables)
  (putprop name type-description :saved-type-description)
  (let ((cons-name
	  (typecase (first args)
	    (symbol (first args))
	    (cons (first (first args)))
	    (otherwise (ferror nil "I don't understand this constructor ~S."
			       (first args)
		       )
	    )
	  )
	)
       )
       (let ((keys nil)
	     (val nil)
	    )
	    (dolist (key (defstruct-type-description-defstruct-keywords
			   type-description
			 )
		    )
	      (unless (get key keys)
		(when (setf val (get-defstruct-property-value name key))
		  (push val keys)
		  (push key keys)
		)
	      )
	    )
	    (make-optimiser-for-constructor-1 cons-name keys)
       )
  )
)
 
(let ((compiler:compile-encapsulations-flag t))
     (advise make-callable-constructor :around :optimise nil
       :do-it
       (make-optimiser-for-constructor arglist)
     )
)


(let ((compiler:compile-encapsulations-flag t))
     (advise make-boa-constructor :around :optimise nil
       :do-it
       (make-optimiser-for-constructor arglist)
     )
)


