;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************


(defun ensure-nntp-database-read ()
  (or *netnews-flag-alist*
      (multiple-value-setq (*netnews-probe-time* *netnews-flag-alist*)
	 (with-open-file
	   (str *netnews-database-path*
		:Direction :Input
		:If-Does-Not-Exist nil
	   )
	   (if str
	       (values-list (read str))
	       nil
	   )
	 )
      )
  )
)

(defun save-nntp-database ()
  (Ensure-Nntp-Database-Read)
  (loop for stream in *all-open-imap-streams*
	for name = (send stream :Mailbox)
	for flags = (send stream :Flaglist)
	for entry = (assoc name *netnews-flag-alist* :Test #'string-equal)
	when (and name (typep stream 'ip-nntp-stream))
	do (setf *netnews-flag-alist*
		 (cons (list name flags (send stream :Generate-Flag-Database))
		       (remove entry *netnews-flag-alist*)
		 )
	   )
  )
  (with-open-file (str *netnews-database-path*
		       :Direction :Output
		       :If-Does-Not-Exist :Create
		       :If-Exists :Overwrite
	          )
    (print (list *netnews-probe-time* *netnews-flag-alist*) str)
  )
)

(defmethod imap-send :Around ((stream ip-nntp-Stream) Command
			      &Optional Args (reply-type nil)
			      (non-ok-reply-ok nil)
			     )
  (ignore args reply-type non-ok-reply-ok)
  (with-mailbox-locked (stream)
    (call-next-method)
  )
)

(defmethod (ip-nntp-stream :Resynch) ()
  (loop for line = (with-timeout (10 nil) (send self :Line-In))
	while line
	do (princ line) (terpri)
  )
)

(defmethod (Ip-Nntp-Stream :Ensure-Has-Map) ()
  (if (not readable-map)
      (setq readable-map (make-array number-of-articles :Initial-Element t))
      nil
  )
)

(defmethod (ip-nntp-stream :Print-Self) (stream depth slashify)
"A simple print method for imap streams."
  (ignore depth slashify)
  (catch-error
    (format stream "#<NNTP stream to ~A>" mailbox)
    nil
  )
)

(defmethod (ip-nntp-stream :Generate-Flag-Database) ()
  (Ensure-Nntp-Database-Read)
  (let ((specs nil)
	(current-spec nil)
	(current-messages nil)
       )
       (loop for i from 0 below messagecnt
	     for global-message-number = (+ first-article-number i)
	     for cache-entry = (aref messagearray i)
	     for temp-flags = (cache-flags cache-entry)
	     for flags = (if (is-present temp-flags) temp-flags nil)
	     do (if (or (set-difference current-spec flags)
			(set-difference flags current-spec)
		    )
		    (progn (push (list current-messages current-spec) specs)
			   (setq current-spec flags)
			   (setq current-messages (list global-message-number))
		    )
		    (push global-message-number current-messages)
		)
	     Finally (push (list current-messages current-spec) specs)
       )
       specs
  )
)

(defmethod (ip-nntp-stream :Update-Flags-From-Database) ()
  (declare (special *edit-server*))
  (Ensure-Nntp-Database-Read)
  (let ((entry (assoc mailbox *netnews-flag-alist* :Test #'string-equal)))
       (if entry
	   (progn
	     (setq flaglist (second entry))
	     (setf keywords (set-difference flaglist *system-flags*))
	     (loop for (numbers flags) in (third entry)
		   do (loop for global-message-number in numbers
			    for index = (- global-message-number
					   first-article-number
					)
			    for cache-entry
				= (if (and (>= index 0)
					   (< index
					      (array-active-length messagearray)
					   )
				      )
				      (cache-entry-of (+ 1 index) self)
				      nil
				  )
			    for existing-flags
			     = (if (and cache-entry
					(is-present (cache-flags cache-entry))
				   )
				   (cache-flags cache-entry)
				   nil
			       )
			    when (and cache-entry
				      (or (set-difference flags existing-flags)
					  (set-difference existing-flags flags)
				      )
				 )
			    do (setf (cache-flags cache-entry) flags)
			       (send *edit-server* :Put-Task
				     :Imap-Parse-Data-Flags
				     (list :Flags-Changed
					   (cache-entry-of (+ 1 index) self)
					   self t
				     )
			       )
			)
	     )
	   )
	   nil
       )
  )
)

(defmethod (ip-nntp-stream :After :Initialize-Mailstream) (TotalMsgs)
  (ignore TotalMsgs)
  (send self :Update-Flags-From-Database)
)

(defmethod at-least-imap-3-p ((mailstream ip-nntp-stream))
  t
)

(defun nntp-send (stream command arg-string/s)
  (format stream "~A ~{~A~^ ~}~%" command (list-if-not arg-string/s))
  (force-output stream)
  (let ((status-code (with-standard-io-environment (read stream))))
       (let ((string (progn (peek-char t stream)
			    (read-line stream)
		     )
	     )
	    )
	    (parse-nntp-status-code status-code string stream command)
       )
  )
)

(defmethod imap-send ((stream ip-nntp-stream) (Command (eql :Group))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore reply-type non-ok-reply-ok)
  (Nntp-Send stream command (first args))
  (setf (slot-value stream 'mailbox) (second args))
  (setf (slot-value stream 'flaglist) yw:*System-Flags*)
)

(defun decolonify-numbers (string)
  ;;; We remove 0 because this is generated by unreadbale messages.
  (remove 0 (with-input-from-string (stream (string-append string " "))
	      (imap:read-sequence stream)
	    )
  )
)

(defmethod imap-send ((stream ip-nntp-stream) (Command (eql 'yw:Fetch))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore reply-type non-ok-reply-ok)
  (destructuring-bind (messages type) args
    (loop for message in (decolonify-numbers messages)
	  do
	  (let ((*message-number* message)
		(*cache-entry*
		  (map-elt (send stream :Messagearray) message stream)
		)
		(*stream* stream)
	       )
	       (declare (special *stream* *message-number* *cache-entry*))
	       ;;; {!!!!}  Maybe we should pipeline these. !!!!
	       (loop for type in (list-if-not type)
		     do (ecase type
			  ((yw:All yw:rfc822.header)
			    (Nntp-Send stream :Head
			      (+ (slot-value stream 'first-article-number)
				 message -1
			      )
			    )
			  )
			  ((yw:RFC822.text* yw:RFC822.text)
			    (Nntp-Send stream :Body
			      (+ (slot-value stream 'first-article-number)
				 message -1
			      )
			    )
			  )
			)
	       )
	  )
    )
  )
)

(defun separate-criteria (criteria)
  (with-input-from-string (stream (with-output-to-string (*standard-output*)
				    (build-imap-arglist :search criteria)
				    (terpri)
				  )
			  )
    (imap:read-criteria stream)
  )
)

(defun implement-search-for-local-stream (stream Args)
  (let ((criteria (Separate-Criteria args))
	(array (send stream :Messagearray))
       )
       (send stream :Set-Selectedmsgs
	 (loop for index from 0 below (send stream :messagecnt)
	       for message-number = (+ 1 index)
	       for cache-entry = (aref array index)
	       for envelope = (cache-envelope cache-entry)
	       for body = (cache-rfc822text cache-entry)
	       when (loop for (criterion arg) in criteria
			  unless (Search-For-Criterion
				   stream criterion arg cache-entry envelope
				   body message-number
			         )
			  return nil
			  finally (return t)
		    )
	       collect message-number
	  )
       )
  )
)

(defmethod imap-send ((stream ip-nntp-stream) (Command (eql :search))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore args reply-type non-ok-reply-ok)
  (implement-search-for-local-stream stream Args)
 `(* :Ok "Search doesn't happen under NNTP.")
)

(defun not-present->-nil (x)
  (if (is-present x)
      x
      nil
  )
)

(defmethod search-for-criterion
	   ((stream ip-nntp-stream) (criterion t)
	    string cache-entry envelope body message
	   )
  (ignore envelope message string)
  (search-for-criterion-on-local-stream
    stream criterion string cache-entry envelope body message
  )
)

(defun search-for-criterion-on-local-stream
       (stream criterion string cache-entry envelope body message-number)
  (cond ((member criterion (send stream :Flaglist) :Test #'eq)
	 (member criterion (Not-Present->-Nil (cache-flags cache-entry))
		 :Test #'eq
         )
	)
	((get criterion :Inverse)
	 (not (Search-For-Criterion stream (get criterion :Inverse)
		string cache-entry envelope body message-number
	      )
	 )
	)
	((imap:~command-p criterion)
	 (not (Search-For-Criterion stream (imap:not-ify criterion)
		string cache-entry envelope body message-number
	      )
	 )
	)
	(t (if (and (is-present envelope) (is-present body))
	       nil
	       (progn (map-fetch-message stream message-number)
		      (throw :Try-Locally :Try-Locally)
	       )
	   )
	)
  )
)

;-------------------------------------------------------------------------------

(Defmethod imap-send ((stream ip-nntp-stream) (Command (eql 'yw:check))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore args reply-type non-ok-reply-ok)
  (let ((time (time:get-universal-time)))
       (nntp-send stream :Newnews
		  (cons (string (slot-value stream 'name-of-group))
			(Nntpify-Time
			  (or (slot-value stream 'Last-Check-Time) time)
			)
		  )
       )
       (setf (slot-value stream 'Last-Check-Time) time)
   )
  `(* :Ok "Check doesn't happen under NNTP.")
)

(defmethod imap-send ((stream ip-nntp-stream) (Command (eql 'yw:store))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (declare (special *edit-server*))
  (destructuring-bind (message-numbers type &rest flags) args
    (loop for message in (decolonify-numbers message-numbers)
	  do (loop for flag in flags
		   for cache-entry = (cache-entry-of message stream)
		   for existing = (cache-flags cache-entry)
		   do (setf (cache-flags cache-entry)
			    (ecase type
			      (yw:+flags (fs:cons-new flag existing))
			      (yw:-flags (remove flag existing))
			      (yw:flags flags)
			    )
		      )
		      (send stream :Maybe-Flush-Search-Cache :Store
			    (net:intern-as-keyword type)
			    (cache-flags cache-entry) message
		      )
		      (send *edit-server* :Put-Task :Imap-Parse-Data-Flags
			    (list :Flags-Changed cache-entry stream t)
		      )
	     )
    )
  )
  (ignore args reply-type non-ok-reply-ok)
  `(* :Ok "Store only happens locally under NNTP.")
)

(defmethod imap-send ((stream ip-nntp-stream) (Command (eql 'yw:logout))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore args reply-type non-ok-reply-ok)
  (Save-Nntp-Database)
  (Nntp-Send stream :Quit nil)
  `(* :Ok ,(format nil "NNTP connection to ~A closed." (send stream :Mailbox)))
)

(defmethod imap-send ((stream ip-nntp-stream) (Command (eql 'yw:set.flags))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore reply-type non-ok-reply-ok)
  (setf (slot-value stream 'flaglist) args)
  (setf (slot-value stream 'keywords)
	(set-difference args *system-flags*)
  )
)

(defmethod imap-send ((stream ip-nntp-stream) (Command t)
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore args reply-type non-ok-reply-ok)
  (ferror nil "Not implemented yet.")
)

(defresource string-lines ()
  :Constructor (make-array 512 :Element-Type 'character :Fill-Pointer t)
  :Initializer (setf (fill-pointer object) 0)
)

(defmethod (Ip-nntp-stream :Read-Nntp-Long-String) ()
  (let ((lines
	  (loop for temp-line = (send self :Line-In)
		until (equal temp-line ".")
		collect (let ((new-line (allocate-resource 'String-Lines))
			      (length (array-active-length temp-line))
		 	     )
			     (if (string= ".." temp-line :End2 2)
				 (progn 
				   (copy-array-portion temp-line 1 length
							new-line 0 (- length 1)
				   )
				   (setf (fill-pointer new-line) (- length 1))
				 )
				 (progn 
				   (copy-array-portion temp-line 0 length
							new-line 0 length
				   )
				   (setf (fill-pointer new-line) length)
				 )
			     )
			     new-line
			 )
	  )
	)
       )
       (let ((full-string
	       (make-string (loop for line in lines sum (+ 1 (length line))))
	     )
	     (start-index 0)
	    )
	    (loop for line in lines
		  for length = (array-active-length line)
		  do (copy-array-portion
		       line 0 length
		       full-string start-index (+ length start-index)
		     )
		     (incf start-index (+ 1 length))
		     (setf (aref full-string (- start-index 1)) #\newline)
		     (deallocate-resource 'string-lines line)
	    )
	    full-string
       )
  )
)

;-------------------------------------------------------------------------------


(defun parse-nntp-status-code (number string mailstream command-from-yw)
  (if (numberp number)
      (let ((class
	      (case (floor number 100)
		(1 :Information)
		(2 :Command-Ok)
		(3 :Command-Ok-So-Far-Send-The-Rest)
		(4 :Command-Failed)
		(5 :Command-Non-Existent)
		(otherwise nil)
	      )
	    )
	    (subclass
	      (case
		(floor (- number (* 100 (floor number 100)) (mod number 10)) 10)
		(0 :Connection)
		(1 :Newsgroup-Selection)
		(2 :Article-Selection)
		(3 :Distribution-Functions)
		(4 :Posting)
		(8 :Non-Standard)
		(9 :Debugging-Output)
	      )
	    )
	   )
	   (parse-nntp-status-code-1 class subclass (mod number 10)
				     mailstream command-from-yw string
	   )
      )
      (format-scroll-window (get-mail-control-window) "~A ~A" number string)
  )
)

(defmethod Parse-Nntp-Status-Code-1 :After
	   ((class (eql :Information))
	    (subclass (eql :Connection))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (format-scroll-window (get-mail-control-window) "~A" string)
  (values :Handled class subclass specific string)
)

(defmethod Parse-Nntp-Status-Code-1 :After
	   ((class (eql :Information))
	    (subclass (eql :Debugging-Output))
	    (specific t)
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (format-scroll-window (get-mail-control-window) "~A" string)
  (values :Handled class subclass specific string)
)

(defmethod Parse-Nntp-Status-Code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Connection))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (setf (slot-value mailstream 'posting-allowed-p) t)
  (format-scroll-window (get-mail-control-window) "~A" string)
  (values :Posting-Allowed class subclass specific string)
)

(defmethod Parse-Nntp-Status-Code-1 :After
	   ((class (eql :Command-Ok))
	    (subclass (eql :Connection))
	    (specific (eql 1))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (setf (slot-value mailstream 'posting-allowed-p) nil)
  (format-scroll-window (get-mail-control-window) "~A" string)
  (values :Posting-Not-Allowed class subclass specific string)
)

(defmethod Parse-Nntp-Status-Code-1 :After
	   ((class (eql :Command-Failed))
	    (subclass (eql :Connection))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (format-scroll-window (get-mail-control-window) "~A" string)
  (map-close mailstream)
  (signal 'sys:abort 'sys:abort)
  (values :Handled class subclass specific string)
)

(defmethod Parse-Nntp-Status-Code-1 :After
	   ((class (eql :Command-Non-Existent))
	    (subclass (eql :Connection))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (format-scroll-window (get-mail-control-window)
			"YW NNTP Error.  Error code was ~D: ~A" specific string
  )
  (values :Handled class subclass specific string)
)


;-------------------------------------------------------------------------------

;;; The :Article/:Body/:Head/:Stat commands.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Article-Selection))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Article))
	    string
	   )
  (ferror nil "This should never be called.")
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Article-Selection))
	    (specific (eql 1))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Head))
	    string
	   )
  (declare (special *message-number* *cache-entry*))
  (let ((header (send mailstream :Read-Nntp-Long-String)))
       (parse-envelope-from-header header mailstream)
       (action-for-new-header *cache-entry* header)
  )
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Article-Selection))
	    (specific (eql 1))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Body)) ;;; {!!!!}
	    string
	   )
  (declare (special *message-number* *cache-entry*))
  (let ((header (send mailstream :Read-Nntp-Long-String)))
       (parse-envelope-from-header header mailstream)
       (action-for-new-header *cache-entry* header)
  )
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Article-Selection))
	    (specific (eql 2))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Head)) ;;; {!!!!}
	    string
	   )
  (declare (special *cache-entry*))
  (let ((body (send mailstream :Read-Nntp-Long-String)))
       (Setf (Cache-RFC822Text *cache-entry*) body)
  )
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Article-Selection))
	    (specific (eql 2))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Body))
	    string
	   )
  (declare (special *cache-entry*))
  (let ((body (send mailstream :Read-Nntp-Long-String)))
       (Setf (Cache-RFC822Text *cache-entry*) body)
  )
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Article-Selection))
	    (specific (eql 2))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Stat))
	    string
	   )
  (declare (special *cache-entry*))
  (values :Handled class subclass specific string)
)

(defun parse-envelope-from-header (header mailstream)
  (declare (special *cache-entry*))
  (ignore mailstream)
  ;;; Don't remember any addresses we get off NNTP.
  (let ((*header-field-keys-to-parse-for-address-database* nil)
	(flags (Cache-flags *cache-entry*))
       )
       (Setf (Cache-RFC822Header *cache-entry*) header)
       (set-parsed-values *cache-entry* mailstream)
       ;;; Restore the flags, because we compute them in a manner other than
       ;;; from the header.
       (if (is-present flags)
	   (setf (Cache-flags *cache-entry*) flags)
	   nil
       )
  )
)

;------------ Error codes.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Failed))
	    (subclass (eql :Newsgroup-Selection))
	    (specific (eql 2))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (yw-error "No newsgroup selected. ~D: ~S" specific string)
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Failed))
	    (subclass (eql :Article-Selection))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (yw-error "No current article has been selected. ~D: ~S" specific string)
  (values :Handled class subclass specific string)
)

(defvar *corrupt-header*
"From: ***UnReadable***
Subject: ***UnReadable***
"
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Failed))
	    (subclass (eql :Article-Selection))
	    (specific (eql 3))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (declare (special *message-number* *cache-entry*))
  (send mailstream :Ensure-Has-Map)
  (setf (aref (slot-value mailstream 'readable-map) (- *message-number* 1))
	nil
  )
  (setf (cache-rfc822header *cache-entry*) *Corrupt-Header*)
  (setf (cache-rfc822text *cache-entry*) "")
  (parse-envelope-from-header *Corrupt-Header* mailstream)
  (action-for-new-header *cache-entry* *Corrupt-Header*)
  (Values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Failed))
	    (subclass (eql :Distribution-Functions))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw t)
	    string
	   )
  (yw-error "No such article found. ~D: ~S" specific string)
  (values :Handled class subclass specific string)
)

;-------------------------------------------------------------------------------

;;; The GROUP command.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Newsgroup-Selection))
	    (specific (eql 1))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Group))
	    string
	   )
  (with-standard-io-environment
    (with-input-from-string (*standard-input* string)
      (setf (slot-value mailstream 'number-of-articles)   (read))
      (setf (slot-value mailstream 'first-article-number) (read))
      (setf (slot-value mailstream 'last-article-number)  (read))
      (setf (slot-value mailstream 'name-of-group)        (read))
    )
  )
  ;;; {!!!!}  This should look in the user's record of the last read.
  (imap-recent mailstream (slot-value mailstream 'number-of-articles))
  (imap-exists mailstream (slot-value mailstream 'number-of-articles))
  (values :Handled class subclass specific string)
)

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Failed))
	    (subclass (eql :Newsgroup-Selection))
	    (specific (eql 1))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Group))
	    string
	   )
  (yw-error "No such news group. ~D: ~S" specific string)
  (map-close mailstream)
  (signal 'sys:abort 'sys:abort)
  (values :Handled class subclass specific string)
)

;-------------------------------------------------------------------------------

;;; The LIST command.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Newsgroup-Selection))
	    (specific (eql 5))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :List))
	    string
	   )
  (let ((groups-as-string (send mailstream :Read-Nntp-Long-String))
	(host (send (send mailstream :Host) :name))
       )
       (with-standard-io-environment
	 (with-input-from-string (*standard-input* groups-as-string)
	   (let ((groups
		   (loop for group = (read *standard-input* nil :eof)
			 for last = (read *standard-input* nil :eof)
			 for first = (read *standard-input* nil :eof)
			 for p = (read *standard-input* nil :eof)
			 until (equal :Eof group)
			 collect (string-append host ":" (string group))
		   )
		 )
		)
	        (send mailstream :Set-All-Mailboxes groups)
	   )
	 )
       )
  )
  (values :Handled class subclass specific string)
)

(defun complete-newsgroup (substring type &rest ignore)
  (with-standard-io-environment
    (letf ((#'sys:internal-read-char
	    (or *old-internal-read-char* #'sys:internal-read-char)
	   )
	  )
	  (let ((all-newsgroups (get-and-cache-all-newsgroups
				  (find-if #'(lambda (x)
					       (typep x 'ip-nntp-stream)
					     )
					     *all-open-imap-streams*
				  )
				)
		)
	       )
	       (if all-newsgroups
		   (multiple-value-bind (ignore host)
		       (mailbox-and-host-from-mailbox-name
			 (first all-newsgroups)
		       )
		     (multiple-value-bind (name ignore)
			 (mailbox-and-host-from-mailbox-name substring)
		       (simple-completer name all-newsgroups type
					 (+ 1 (length host))
		       )
		     )
		   )
		   (beep)
	       )
	  )
    )
  )
)

;-------------------------------------------------------------------------------

;;; The QUIT command.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Connection))
	    (specific (eql 5))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :Quit))
	    string
	   )
  (format-scroll-window (get-mail-control-window)
			"Closing NNTP connection ~A" string
  )
  (values :Handled class subclass specific string)
)


;-------------------------------------------------------------------------------

;;; The NEWGROUPS command.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Distribution-Functions))
	    (specific (eql 1))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :NewGroups))
	    string
	   )
  (let ((new-groups (send mailstream :Read-Nntp-Long-String)))
       (if (equal "" new-groups)
	   (format-scroll-window (get-mail-control-window)
				 "No new newsgroups."
	   )
	   (format-scroll-window
	     (get-mail-control-window)
	     "The following newsgroups are new:~%~A" new-groups
	   )
       )
  )
  (values :Handled class subclass specific string)
)


;-------------------------------------------------------------------------------

;;; The NEWNEWS command.

(defmethod parse-nntp-status-code-1
	   ((class (eql :Command-Ok))
	    (subclass (eql :Distribution-Functions))
	    (specific (eql 0))
	    (mailstream ip-nntp-stream)
	    (command-from-yw (eql :NewNews))
	    string
	   )
  (let ((new-groups (send mailstream :Read-Nntp-Long-String)))
       (if (equal "" new-groups)
	   nil
	   (Nntp-Send mailstream :Group
		      (slot-value mailstream 'name-of-group)
	   )
       )
  )
  (values :Handled class subclass specific string)
)

;-------------------------------------------------------------------------------

(defun open-nntp-stream (mailbox-name read-only-p)
  "Opens a possibly read-only mailbox called mailbox-name."
  (ignore read-only-p)
;  (condition-case (condition) ;;; {!!!!}
      (nntp-open mailbox-name)
;    (error (send condition :Report nil))
;  )
)


(defmethod open-yw-stream
	   (mailbox-name read-only-p (connection-type (eql :NNTP)))
  (open-nntp-stream mailbox-name read-only-p)
)

(defun nntpify-time (ut)
  (multiple-value-bind (secs minutes hours day month year)
      (time:decode-universal-time ut)
    (list (format nil "~2,'0,D~2,'0,D~2,'0,D" (mod year 100) month day)
	  (format nil "~2,'0,D~2,'0,D~2,'0,D" hours minutes secs)
    )
  )
)

(defun Nntp-Open (mailbox-name)
  (let ((MailHost (or (subseq MailBox-name 0 (position #\: MailBox-name))
		      *site-specific-nntp-server-host*
		  )
	)
       )
       (let ((mailstream (open-nntp-connection mailhost))
	     (completed-ok-p nil)
	    )
	    (unwind-protect
		(if mailstream
		    (progn
		      (let ((status-code
			      (with-standard-io-environment (read mailstream))
			    )
			   )
			   (let ((string (progn (peek-char t mailstream)
						(read-line mailstream)
		                          )
	                         )
	                        )
			        (Parse-Nntp-Status-Code
				  status-code string mailstream nil
				)
                           )
                      )
		      (imap-send mailstream :Group
				 (list (string-downcase
					 (mailbox-name-for-select mailbox-name)
				       )
				       mailbox-name
			         )
		      )
		      (setq completed-ok-p t)
		      (if *netnews-probe-time*
			  (nntp-send mailstream :Newgroups
				     (nntpify-time *netnews-probe-time*)
			  )
			  nil
		      )
		      (setq *netnews-probe-time* (time:get-universal-time))
		      mailstream
		    )
		    (progn (MAP-Close MailStream)		  ;;Else
			   nil
		    )
		)
	      (if completed-ok-p
		  nil
		  (progn ;;; Panic cleanup. 
			 (setq *all-open-imap-streams*
			       (remove mailstream *all-open-imap-streams*)
			 )
			 (map-close mailstream)
		  )
	      )	      
	    )
       )	      
  )

)
 
(defun open-nntp-connection (host)
  (let ((ip:*tcp-stream-instantiator* 'ip-nntp-stream-instantiator)
	(*make-imap-stream-read-only* t)
       )
       (declare (special ip:*tcp-stream-instantiator*
			 *make-imap-stream-read-only*
                )
       )
       (net:open-connection-on-medium (net:parse-host host) :Byte-Stream
				      "Generic-NNTP"
				      :Stream-Type
				      :Ascii-Translating-Character-Stream
       )
  )
)

(defun ip-nntp-stream-instantiator
       (connection timeout input-buffer-size number-of-input-buffers)
  "Makes an NNTP stream under IP."
  ;;RDA: This seems the only decent place to set these.  Apologies to
  ;;modularity...
  (setf input-buffer-size *yw-tcp-input-buffer-size*
	number-of-input-buffers *yw-tcp-number-of-input-buffers*)
  (yw:imap-stream-instantiator
    'yw:ip-nntp-stream connection timeout input-buffer-size
    number-of-input-buffers
  )
)