;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

;;; Command tables.

(defcommand-table Message-Sequence "IMAP Client message-sequence commands."
  "This is a command table that holds message-sequence commands for the IMAP
 client mailer.  The sort of commands that you'll find in this table are things
 like From, To, Subject..."
)

;-------------------------------------------------------------------------------

(defun get-message-sequence-arg (blip-extractor)
  (Read-String-Maybe-With-Blip
    *standard-input* nil nil blip-extractor
  )
)

;;; Message-sequence commands.

(Defimmediate-command (mail-control-window :All)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "All"
    :Description "Selects all headers ...."
    :Documentation '("Selects a sequence all headers into a new summary.  "
		     *message-sequence-documentation*
		    )
   )
  (Throw :Sequence (simple-sequence :Sequence-All))
)

(defcommand-short-form (mail-control-window :All) "A")

(Defimmediate-command (mail-control-window :Answered)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Answered"
    :Description "Selects a sequence of Answered headers ...."
    :Documentation
    '("Selects a sequence of answered message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-Answered))
)

(Defimmediate-command (mail-control-window :Bcc)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "bcc"
    :Description "Selects a sequence of headers bcc...."
    :Documentation
    '("Selects a sequence of message headers bcced to a particular source into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (Get-Message-Sequence-Arg 'canonical-bcc)))
       (Throw :Sequence
	      (simple-sequence
		:Sequence-Bcc
		(checking-named-sequences string 'canonical-bcc)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Before)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Before"
    :Description "Selects a sequence of headers sent before...."
    :Documentation
    '("Selects a sequence of message headers sent before a particular date/time into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((date-time (Get-Message-Sequence-Arg 'fetch-just-date)))
       (validate-date-time date-time)
       (Throw :Sequence
	      (simple-sequence
		:Sequence-Before
		(checking-named-sequences date-time 'fetch-just-date)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Cc)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "CC"
    :Description "Selects a sequence of headers cc...."
    :Documentation
    '("Selects a sequence of message headers cced to a particular source into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (Get-Message-Sequence-Arg 'canonical-cc)))
       (Throw :Sequence
	      (simple-sequence
		:Sequence-Cc
		(checking-named-sequences string 'canonical-cc)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Deleted)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Deleted"
    :Description "Selects a sequence of deleted headers ...."
    :Documentation
    '("Selects a sequence of message deleted headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-Deleted))
)

(Defimmediate-command (mail-control-window :Field)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Field"
    :Description
    "Selects a sequence of messages with specified Field name and value."
    :Documentation
    '("Selects a sequence of messages which have a specific value for the specified field into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((key (intern (string-upcase (read-string *standard-input*)) 'keyword))
	(search-string (read-string *standard-input*))
       )
       (Throw :Sequence (simple-sequence :Sequence-Field key search-string))
  )
)

(Defimmediate-command (mail-control-window :Flagged)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Flagged"
    :Description "Selects a sequence of Flagged headers ...."
    :Documentation
    '("Selects a sequence of flagged message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-Flagged))
)

(Defimmediate-command (mail-control-window :From)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "From"
    :Description "Selects a sequence of headers From...."
    :Documentation
    '("Selects a sequence of message headers from a particular source into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (Get-Message-Sequence-Arg 'canonical-from)))
       (throw :Sequence
	      (simple-sequence
		:Sequence-From
		(checking-named-sequences string 'canonical-from)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Id)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Id"
    :Description "Selects a sequence of headers with Id ...."
    :Documentation
    '("Selects a sequence of message headers with a particular Id substring.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (Get-Message-Sequence-Arg 'map-fetch-id)))
       (Throw :Sequence
	      (simple-sequence
		:Sequence-Id
		(checking-named-sequences string 'map-fetch-id)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Inverse)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Inverse"
    :Description "Selects all headers to be read backwards...."
    :Documentation
    '("Selects a sequence all headers into a new summary to be read backwards.  Equivalent to Reverse All.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence
	 (simple-sequence
	   :Sequence-Reverse
	   (make-a-sequence nil :Sequence-Specifier '(:Sequence-All))
	 )
  )
)

(defcommand-short-form (mail-control-window :Inverse) "I")

(Defimmediate-command (mail-control-window :Keyword)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Keyword"
    :Description "Selects a sequence of headers with Keyword......"
    :Documentation
    '("Selects a sequence of message headers for messages with a particular keyword into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((keyword (read-keyword *standard-input*
			       (or current-mailbox (first all-mailboxes))
	         )
        )
       )
       (throw :Sequence (simple-sequence :Sequence-Keyword keyword))
  )
)

(Defimmediate-command (mail-control-window :Last)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Last"
    :Description "Selects a sequence of the last N headers ...."
    :Documentation
    '("Selects a sequence of the last N message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((number (read-simple-number 0 *standard-input*)))
       (if (fixnump number)
	   (Throw :Sequence (simple-sequence :Sequence-Last number))
	   (parse-error "~S is not a legal \"Last <n>\" message number."
			number
	   )
       )
  )
)

(Defimmediate-command (mail-control-window :Length)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Length"
    :Description "Selects a sequence of headers whose length is < or > some value, i.e. Head Length > 20000"
    :Documentation
    '("Selects a sequence of headers whose length is < or > some value, i.e. Head Length > 20000.  "
      *message-sequence-documentation*
     )
   )
  (let ((op (with-standard-io-environment (read-a-token *standard-input*))))
       (case op
	 ((:> :< := :>= :<=)
	  (let ((length-string
		  (with-standard-io-environment
		    (read-string-maybe-with-blip *standard-input* nil nil
		       #'(lambda (mailstream messages)
			   (map-fetch-length mailstream messages)
			 )
		    )
		  )
		)
	       )
	       (let ((length (catch-error
			       (yw-read-from-string length-string nil :Eof)
			     )
		     )
		    )
		    (if (or (numberp length)
			    (assoc length-string *named-sequence-alist*
				   :Test #'string-equal
			    )
			)
			(Throw :Sequence
			       (simple-sequence :Sequence-Length
				 (intern (symbol-name op) 'lisp)
				 (checking-named-sequences
				   (or length length-string)
				   'map-fetch-length
				 )
			       )
			)
			(parse-error "~S is an illegal length specifier."
				     length
			)
		    )
	       )
	  )    
	 )
	 (otherwise
	   (parse-error "~%~S is an illegal length specifier operator." op)
	 )
       )
  )
)

(Defimmediate-command (mail-control-window :Mailbox-is)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Mailbox-is"
    :Description "Predicates on whether a mailbox is of some name."
    :Documentation
    '("Predicates on whether a mailbox is of some name."
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence
		     :Sequence-Mailbox-is
		     (string-trim *whitespace-chars*
				  (Read-Mailbox-Name *standard-input* t)
		     )
		   )
  )
)

(Defimmediate-command (mail-control-window :New)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "New"
    :Description "Selects a sequence of new headers ...."
    :Documentation
    '("Selects a sequence of new message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-New))
)

(Defimmediate-command (mail-control-window :Old)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Old"
    :Description "Selects a sequence of old headers ...."
    :Documentation
    '("Selects a sequence of old message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-Old))
)

(Defimmediate-command (mail-control-window :On)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "On"
    :Description "Selects a sequence of headers sent on...."
    :Documentation
    '("Selects a sequence of message headers sent on a particular date/time into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((date-time (get-message-sequence-arg 'fetch-just-date)))
       (validate-date-time date-time)
       (Throw :Sequence
	      (simple-sequence
		:Sequence-On
		(checking-named-sequences date-time 'fetch-just-date)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Recent)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Recent"
    :Description "Selects a sequence of Recent headers ...."
    :Documentation
    '("Selects a sequence of recent message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-Recent))
)

(Defimmediate-command (mail-control-window :Seen)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Seen"
    :Description "Selects a sequence of seen headers ...."
    :Documentation
    '("Selects a sequence of seen message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-Seen))
)

(Defimmediate-command (mail-control-window :Since)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Since"
    :Description "Selects a sequence of headers sent since...."
    :Documentation
    '("Selects a sequence of message headers sent since a particular date/time into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((date-time (get-message-sequence-arg 'fetch-just-date)))
       (validate-date-time date-time)
       (Throw :Sequence (simple-sequence :Sequence-Since date-time))
  )
)

(Defimmediate-command (mail-control-window :Subject)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Subject"
    :Description
    "Selects a sequence of headers with specified text in the subject field...."
    :Documentation
    '("Selects a sequence of message headers whose subject fields contain the specified text into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (get-message-sequence-arg 'canonical-subject)))
       (Throw :Sequence
	      (simple-sequence
		:Sequence-Subject
		(checking-named-sequences string 'canonical-subject)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :Text)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Text"
    :Description
    "Selects a sequence of messages with specified text in their bodies...."
    :Documentation
    '("Selects a sequence of messages whose bodies contain the specified text into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (read-string *standard-input*)))
       (Throw :Sequence (simple-sequence :Sequence-Text string))
  )
)

(Defimmediate-command (mail-control-window :To)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "To"
    :Description "Selects a sequence of headers To...."
    :Documentation
    '("Selects a sequence of message headers to a particular source into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((string (get-message-sequence-arg 'canonical-to)))
       (Throw :Sequence
	      (simple-sequence
		:Sequence-To
		(checking-named-sequences string 'canonical-to)
	      )
       )
  )
)

(Defimmediate-command (mail-control-window :UnAnswered)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "UnAnswered"
    :Description "Selects a sequence of UnAnswered headers ...."
    :Documentation
    '("Selects a sequence of unanswered message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-UnAnswered))
)

(Defimmediate-command (mail-control-window :UnDeleted)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "UnDeleted"
    :Description "Selects a sequence of undeleted headers ...."
    :Documentation
    '("Selects a sequence of message undeleted headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-~Deleted))
)

(Defimmediate-command (mail-control-window :UnFlagged)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "UnFlagged"
    :Description "Selects a sequence of UnFlagged headers ...."
    :Documentation
    '("Selects a sequence of unflagged message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :Sequence (simple-sequence :Sequence-~Flagged))
)

(Defimmediate-command (mail-control-window :UnKeyword)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "UnKeyword"
    :Description "Selects a sequence of headers with UnKeyword......"
    :Documentation
    '("Selects a sequence of message headers for messages with a particular keyword into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((keyword (read-keyword *standard-input* current-mailbox)))
       (throw :Sequence (simple-sequence :Sequence-~Keyword keyword))
  )
)

(Defimmediate-command (mail-control-window :UnSeen)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "UnSeen"
    :Description "Selects a sequence of unseen headers ...."
    :Documentation
    '("Selects a sequence of unseen message headers into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (Throw :sequence (simple-sequence :Sequence-UnSeen))
)

;-------------------------------------------------------------------------------

;;; Boolean keywords.

(Defimmediate-command (mail-control-window :Not)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Not"
    :Description "Selects a sequence of headers Not...."
    :Documentation
    '("Selects a sequence of message headers, which are not specified by some sequence into a new summary.  "
      *message-sequence-documentation*
     )
   )
  (let ((seq (Parse-a-message-sequence *standard-input* nil)))
       (validate-sequence seq nil)
       (Throw :Sequence (simple-sequence :Sequence-Not seq))
  )
)

(Defimmediate-command (mail-control-window :Reverse)
		      () (*All-Message-Sequence-Command-Names*)
  `(:Names "Reverse"
    :Description "Selects a sequence of headers to be read backwards...."
    :Documentation
    '("Selects a sequence of message headers, which are to be read backwards.  "
      *message-sequence-documentation*
     )
   )
  (let ((seq (Parse-a-message-sequence *standard-input* nil)))
       (validate-sequence seq nil)
       (Throw :Sequence (simple-sequence :Sequence-Reverse seq))
  )
)

;-------------------------------------------------------------------------------

;;; Sort commands.

(defcommand-table sort "IMAP Client sort-by subcommands."
"This is a command table that holds subcommands to the Sort message sequence
specifier."
)

(Defimmediate-command (mail-control-window :sorted-bcc)
		      () (*All-Sort-Command-Names*)
  `(:Names "Bcc"
    :Description "Sorts by the header's Bcc field"
   )
  (Throw :Sort :Get-bcc)
)

(Defimmediate-command (mail-control-window :sorted-cc)
		      () (*All-Sort-Command-Names*)
  `(:Names "CC"
    :Description "Sorts by the header's CC field"
   )
  (Throw :Sort :Get-cc)
)

(Defimmediate-command (mail-control-window :sorted-date)
		      () (*All-Sort-Command-Names*)
  `(:Names "Date"
    :Description "Sorts by the header's Date field"
   )
  (Throw :Sort :Get-date)
)

(Defimmediate-command (mail-control-window :sorted-from)
		      () (*All-Sort-Command-Names*)
  `(:Names "From"
    :Description "Sorts by the header's From field"
   )
  (Throw :Sort :Get-From)
)

(Defimmediate-command (mail-control-window :sorted-id)
		      () (*All-Sort-Command-Names*)
  `(:Names "Id"
    :Description "Sorts by the header's Id field"
   )
  (Throw :Sort :Get-Id)
)

(Defimmediate-command (mail-control-window :sorted-subject)
		      () (*All-Sort-Command-Names*)
  `(:Names "Subject"
    :Description "Sorts by the header's Subject field"
   )
  (Throw :Sort :Get-Canonical-Subject)
)

(Defimmediate-command (mail-control-window :sorted-text)
		      () (*All-Sort-Command-Names*)
  `(:Names "Text"
    :Description "Sorts by the message's text"
   )
  (Throw :Sort :Get-Text)
)

(Defimmediate-command (mail-control-window :sorted-to)
		      () (*All-Sort-Command-Names*)
  `(:Names "To"
    :Description "Sorts by the header's To field"
   )
  (Throw :Sort :Get-To)
)

;-------------------------------------------------------------------------------
