;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------

;;; Command tables.


(defcommand-table settable-thing "IMAP Client Settable thing commands."
  "This is a command table that holds Set commands for the IMAP
 client mailer.  The sort of commands that you'll find in this table are things
 like Keyword..."
)

;-------------------------------------------------------------------------------

(defun build-all-command-tables ()
  (mapcar #'funcall *all-command-table-builders*)
)

(defmethod (Mail-Control-Window :match-window-with-mailbox) ()
"Makes sure that the current summary window and the current mailbox
refer to the same thing.
"
  (if (and current-summary-window
	   (typep current-summary-window 'summary-window)
	   current-mailbox
	   (not (member current-mailbox
		       (send current-summary-window :Mailstreams)
		       :Test #'eq
		)
	   )
      )
      (if (and (send current-summary-window :Mailstreams)
	       (loop for stream in (send current-summary-window :Mailstreams)
		     always (send stream :Open-P)
	       )
	  )
	  (send current-summary-window :Select-Me)
	  (send self :Forget-Window current-summary-window nil t)
      )
      nil
  )
)

(defmethod (Mail-Control-Window :After :Expose) (&rest ignore)
  (without-recursion ()
    (if *try-to-expose-all-summary-windows-on-mailer-expose*
	(if current-mailbox
	    (let ((frame self))
		 (Expose-In-Order 'Summary-Windows
				  #'(lambda (win)
				      (member win
					      (send frame :All-Summary-Windows)
					      :Test #'eq
				      )
				    )
				  nil
		 )
	    )
	    nil
	)
	nil
    )
    (if (or *try-to-expose-current-summary-window-on-mailer-expose*
	    *try-to-expose-all-summary-windows-on-mailer-expose*
	)
	(if current-mailbox
	    (if current-summary-window
		(progn (send self :Match-Window-With-Mailbox)
		       (send current-summary-window :maybe-expose)
		)
		nil
	    )
	    nil
	)
    )
  )
)

(defmethod (Mail-Control-Window :change-constraints) (to)
  (let ((items (send Mailbox-Selector :Item-Mappings)))
       (setq tv:constraints to)
       (tv:constraint-frame-process-constraints)
       (setf `(,tv:configuration . ,tv:internal-constraints)
	     (if tv:configuration
		 (or (assoc tv:configuration tv:parsed-constraints :test #'eq)
		     (yw-error nil "Configuration ~S not found"
			       tv:configuration
		     )
		 )
		 (first tv:parsed-constraints))) ;Default initial configuration
       (tv:constraint-frame-recompute-configuration)
       (setq mailbox-selector (send self :Get-Pane :Mailbox-Selector))
       (setq Prompt-Window (send self :Get-Pane :Prompt-Window))
       (send Mailbox-Selector :Set-Item-Mappings items)
       (send Mailbox-Selector :Set-Display-Item
	     (cons nil (mapcar #'second items))
       )
       (send Mailbox-Selector :Update-All-Labels)
  )
)

(defmethod (yw:mail-control-window :Loop) ()
  "Begins the loop of fetching input, translating the input into a command,
and executing the command.  "
  (declare (values nil-or-return-value) (special ucl:key-sequence))
  (catch 'ucl:exit-command-loop
    (error-restart-loop
      ((sys:abort error) "Return to the ~a command loop."
       (or (send self :Send-If-Handles :Name) "current"))
      (condition-case ()
	  (ucl:my-ignore-errors-query-loop
	    (let ((yw:*line-ended* nil)
;		  (ucl:command-loop-typein? yw:prompt-window)
		 )
	         (declare (special yw:*line-ended*))
		 (if (not (yw:yw-command-loop-1 yw:prompt-window))
		     (progn (setq ucl:key-sequence nil))
		     nil
		 )
	    )
	  )
	;;Reprompt on aborts when we wouldn't normally
	(sys:abort
	 (send self :Set-Prefix-Argument nil)
	 (setq ucl:key-sequence nil)
	 (when (and (eq ucl:typein-handler :Handle-Typein-Input)
		    (neq ucl:input-mechanism 'ucl:typein)
	       )
	   (send self :Handle-Prompt)
	 )
        )
      )
    )
  )
)

(defun yw-command-loop-1 (prompt-window)
  (catch 'ucl:command-abort
    (let ((*this-application* self)
	  (ucl:this-application prompt-window)
	  (*prompt-window* prompt-window)
	  (*mailer* self)
	  (key-lookup #'tv:ucl-rh-lookup-character)
	 )
	 (declare (special *this-application* *prompt-window*))
	 (let ((try-yw
		 #'(lambda (ch) (Maybe-Try-Yw-For-Key-Command ch key-lookup))
	       )
	      )
	      (letf (((symeval-in-instance prompt-window 'ucl:typein-modes)
		      '(YW-Top-Level-Command-Names)
		     )
		     (#'tv:ucl-rh-lookup-character try-yw)
		     ((first (aref (send tv:rh-command-table :Key-Table) 26))
		      #\c-m-sh-s
		     )
		    )
		    (funcall prompt-window :Rubout-Handler
			     '((:activation = #\end))
			     'Parse-Command-Line prompt-window
		    )
	      )
	 )
    )
  )
)

(defun print-results-p ()
  nil ;ucl:abnormal-command?
)

(defmethod (mail-control-window :Copy-From-Address-To-Kill-Ring)
	   (sequence &rest ignore)
  (let ((message (first (send sequence :Sequence-Specifier)))
	(stream (send sequence :Mailstream))
       )
       (maybe-preempt-envelopes stream (number-of message))
       (let ((envelope (cache-envelope (cache-entry-of message sequence))))
	    (map-fetch-from stream message)
	    (let ((address (first (envelope-from envelope))))
	         (if address
		     (zwei:kill-string
		       (send (get-address-object address) :Address-String)
		     )
		     (beep)
		 )
	    )
       )
  )
)

(defmethod (Mail-Control-Window :mouse-l-hold) (&rest ignore)
  (declare (special *message*))
  (Send self :Message-As-Prefix-Arg *message* t)
)

(defmethod (Mail-Control-Window :Unselect-Prefix-Argument) (message)
  (Send self :Set-Prefix-Argument (remove message prefix-argument))
)

(defun print-prefix-args (process-window)
  (let ((string
	  (send (send process-window :superior) :Print-Any-Prefix-Args ", ")
	)
       )
       (if (equal "" string)
	   nil
	   (princ string process-window)
       )
  )
)

(defmethod (Mail-Control-Window :print-any-prefix-args) (&optional (suffix " "))
  (if prefix-argument
      (format nil "~A~A" (make-label-from-filter prefix-argument) suffix)
      ""
  )
)

(defmethod (Mail-Control-Window :get-current-summary-window) ()
  (if current-summary-window
      current-summary-window
      (if all-summary-windows
	  (progn (setq current-summary-window all-summary-windows)
		 current-summary-window
	  )
	  nil
      )
  )
)

(defun compute-prompt ()
  (let ((all-prompts (send self :All-Prompts))
	(table (first (send self :Active-Command-Tables)))
       )
       (string-append (if (assoc table all-prompts)
			  (second (assoc table all-prompts))
			  (second (first all-prompts))
		      )
;		      (send self :print-any-prefix-args)
       )
  )
)

(defun ideal-mail-control-window-position (window)
  (ignore window)
  (values-list *Default-Ideal-Mail-Control-Window-Position*)
)

(defmethod (Mail-Control-Window :reposition) (resize-p)
  (if resize-p
      (send self :Set-Edges 0 0 (mail-control-window-width)
	    (mail-control-window-height)
      )
      nil
  )
  (multiple-value-bind (x y) (ideal-mail-control-window-position self)
    (send self :Set-Position
	  (if (> (+ x (send self :Width)) (send (send self :Superior) :Width))
	      (- (send (send self :Superior) :Width) (send self :Width) 1)
	      x
	  )
	  (if (> (+ y (send self :Height)) (send (send self :Superior) :Height))
	      (- (send (send self :Superior) :Height) (send self :Height) 1)
	      y
	  )
    )
  )
)

(defun reposition-mail-control-windows (&optional (resize-p nil))
  (mapc #'(lambda (window) (send window :Reposition resize-p))
        *all-mail-control-windows*
  )
)

(Defmethod (Mail-Control-Window :After :Init) (ignore)
  (setq prompt-window (send self :Get-Pane :Prompt-Window))
  (setq mailbox-selector (send self :Get-Pane :mailbox-selector))
  (setq system-output-pane (send self :Get-Pane :System-Output))
  (setq process-status-pane (send self :Get-Pane :Process-Status))
  (send self :Reposition nil)
  (pushnew self *all-mail-control-windows*)
  (send (send self :Prompt-Window) :Mouse-Select)
)


(defun apply-to-mail-control-windows (function &rest args)
  (loop for window in *all-mail-control-windows* do
	(apply function window args)
  )
)

(defmethod (mail-control-window :recompute-icon) ()
  (if icon
      (if (type-specifier-p 'w:graphics-icon)
	  (let ((recentp (loop for stream in all-mailboxes
			       when (send stream :new-mail-p)
			       return t
			 )
		)
		(old (send icon :Send-If-Handles :Picture))
	       )
	       (send icon :Send-If-Handles :Set-Size-Of-Picture *size-of-icon*)
	       (send icon :Send-If-Handles :set-picture
		     (if recentp *new-mail-icon* *empty-mail-icon*)
	       )
	       (if (and (send icon :Exposed-P)
			(not (equal old (send icon :Send-If-Handles :Picture)))
		   )
		   (send icon :Refresh)
		   nil
	       )
	       (values icon recentp)
	  )
	  (values icon nil)
      )
      nil
  )
)

(defmethod (mail-control-window :Icon-For-Window) ()
  (if icon
      (send self :recompute-icon)
      (let ((new-icon
	      (if (type-specifier-p 'w:graphics-icon)
		  (make-instance 'w:graphics-icon
				 :Picture *new-mail-icon*
				 :Window-Being-Represented self
		  )
		  (make-instance 'w:text-icon
				 :Font-Map '(fonts:cptfont)
				 :Window-Being-Represented self
				 :text "IMAP Client"
		  )
	      )
	    )
	   )
	   (multiple-value-bind (x y) (send self :position)
	     (when (send new-icon :set-position x y :verify)
	       (send new-icon :center-around x y)
	     )
	   )
	   (push new-icon w:*windows-to-leave-exposed*)
	   (setq icon new-icon)
	   (send self :Icon-For-Window)
      )
  )
)


(defmethod (Mail-Control-Window :Before :kill) ()
  (setq *all-mail-control-windows* (remove self *all-mail-control-windows*))
  (if icon
      (progn (send icon :Kill)
	     (setq icon nil)
      )
      nil
  )
  (if (Is-In-yw-Resource self 'mailer-windows)
      (progn (remove-from-yw-resource 'Mailer-Windows self)
	     (send self :Clean-Up)
      )
      nil
  )
)

(defun reset-mcw (window &rest ignore)
  (send window :set-all-mailboxes nil)
  (setf (symeval-in-instance window 'current-mailbox) nil)
  (send window :set-current-summary-window nil)
  (mapcar #'(lambda (win)
	      (if (Is-In-yw-Resource win 'Summary-Windows)
		  (deallocate-yw-resource 'summary-windows win)
		  nil
	      )
	      (send win :Bury)
	      (send (send window :mailbox-selector) :Remove-Summary-Window win)
	    )
	  (send window :all-summary-windows)
  )
  (send (send window :mailbox-selector) :Forget-All)
  (send window :Set-All-Summary-Windows nil)
)

(defun warm-reset ()
  (with-daemons-reset-and-arrested (nil)
    (reset-all-locks)
    (Map-Over-Control-Windows 'reset-mcw)
    (loop for stream in *all-open-imap-streams*
	  do (send stream :Clean-Up-After-Close)
    )
    (map-yw-resource #'(lambda (window &rest ignore) (send window :Clean-Up))
		     'Mailer-Windows
    )
  )
  (reset-daemon-processes)
)

(defmethod (mail-control-window :Clean-Up) ()
  (loop for box in all-mailboxes do (send box :Set-Owning-Window nil))
  (setq all-mailboxes nil)
  (setq current-mailbox nil)
  (setq current-summary-window nil)
  (setq current-bboard nil)
  (setq prefix-argument nil)
  (setq current-sequence nil)
)

(defmethod (Mail-Control-Window :name-for-selection) ()
  (send self :name)
)

(defmethod (Mail-Control-Window :selection-substitute) ()
  (setq tv:selection-substitute
	(send self :Get-Pane :prompt-window)
  )
  tv:selection-substitute
)

(defmethod (Mail-Control-Window :associated-windows) (for-stream)
  (remove-if-not
    #'(lambda (win) (member for-stream (send win :Mailstreams) :Test #'eq))
    all-summary-windows
  )
)

(defmethod (Mail-Control-Window :A-Summary-Window-Named)
	   (this-stream &optional (no-filter-p t))
  (loop for win in all-summary-windows
	when (loop for stream in (send win :Mailstreams)
		   when (and (eq this-stream stream)
			     (or (not no-filter-p)
				 (not (send win :Filter))
			     )
			)
		   return t
	     )
	return win
  )
)

(defmethod (Mail-Control-Window :summary-window-named) (name)
  (find name all-summary-windows :Key 'summary-window-name)
)

(defmethod (Mail-Control-Window :New-Messages) (on-stream new-count)
  (let ((windows (send self :associated-windows on-stream)))
       (if windows
	   (loop for win in windows do
		 (let ((old-count (send win :old-number-of-items)))
		      (if (> new-count old-count)
			  ;;; More new messages have arrived
			  (send win :New-Messages new-count on-stream)
			  ;;; The mailbox has been expunged
			  (send win :Mailbox-Expunged on-stream)
		      )
		      (send Mailbox-Selector :Update-Label-For win)
		      (if (not (send win :Filter))
			  (send self :Set-Current-Summary-Window win)
			  nil
		      )
		  )
	   )
	   nil
       )
       (send self :set-current-sequence nil)
  )
)

(defmethod (Mail-Control-Window :Top-Level-Window-Blip)
	   (type body selector mouse-button)
  (declare (special *background-mouse-blip-methods*))
  (ignore type selector)
  (let ((entry
	  (find-if #'(lambda (X)
		       (char-equal mouse-button (Background-Blip-Mouse-Char X))
		     )
		     *background-mouse-blip-methods*
	  )
	)
       )
       (let ((*mailbox* body))
	    (declare (special *mailbox*))
	    (if entry
		(destructuring-bind (type-again summary-window) body
		  (ignore type-again)
		  (send summary-window (background-blip-method entry))
		)
		(beep)
	    )
       )
  )
)

(defmethod (Mail-Control-Window :Summary-Window)
	   (whole-blip solicited-p window &rest ignore)
  (if solicited-p
      (list (send window :mailbox))
      (yw:safe-lexpr-send self :top-level-window-blip whole-blip)
  )
)

(defmethod (Mail-Control-Window :Message-As-Prefix-Arg)
	   (message &optional (print-p t))
  (assert (typep message 'cache))
  (if (member message prefix-argument  :Test #'eq)
      nil
      (progn (if print-p
		 (send process-status-pane :Maybe-Display-Statuses t)
		 nil
	     )
	     (push message prefix-argument)
	     (add-selected-type message :prefix-arg)
      )
  )
  message
)


(defmethod (Mail-Control-Window :Menu-Of-Top-Level-Operations) (sequence ignore)
  (declare (special *right-button-options-menu*))
  (send *right-button-options-menu* :Expose-Near '(:Mouse))
  (let ((choice (unwind-protect
                    (progn (send *right-button-options-menu* :activate)
                           (send *right-button-options-menu* :Choose)
                    )
                  (send *right-button-options-menu* :deactivate)
                  (send *right-button-options-menu* :Bury)
                )
        )
       )
       (if choice
	   (send self :Read-Type-Command sequence choice)
	   (beep)
       )
  )
)

(defmethod message-from-blip-body ((key (eql :Message)) &rest args)
  (second args)
)

(defmethod (Mail-Control-Window :Top-Level-Message-Blip)
	   (type body Summary-Window mouse-button)
  (declare (special *top-level-mouse-blip-methods*))
  (ignore type Summary-Window)
  (let ((entry
	  (find-if #'(lambda (X)
		       (char-equal mouse-button (Top-Level-Blip-Mouse-Char X))
		     )
		     *top-level-mouse-blip-methods*
	  )
	)
       )
       (let ((*message* (apply 'Message-From-Blip-Body body))
	     (*message-window* summary-window)
	    )
	    (declare (special *message* *message-window*))
	    (if entry
		(destructuring-bind
		  (type-again mailbox cache-entry)
		  body
		  (ignore mailbox type-again)
		  (send self (Top-Level-Blip-Starter-Method entry)
			(Make-A-Sequence nil :Owner Self
			  :Mailbox (cache-mailstream cache-entry)
			  :Sequence-Specifier
			    (list (cache-msg# Cache-entry))
			  :User-Filter-Name nil
			  :Inits (list :Superior (send summary-window :Filter))
			)
			(Top-Level-Blip-Task-Daemon-Method entry)
		  )
		)
		(beep)
	    )
       )
  )
)

(defmethod (Mail-Control-Window :mouse-l-2-click) (&rest ignore)
  (declare (special *message*))
  (send self :Message-As-Prefix-Arg *message* t)
  (send self :fetch-and-execute)
)

(defmethod (Mail-Control-Window :Message)
	   (whole-blip solicited-p name message &rest ignore)
  (ignore name)
  (if solicited-p
      (list message)
      (if *use-top-level-mouse-clicks-as-prefix-arguments*
	  (send self :Message-As-Prefix-Arg message t)
	  (yw:safe-lexpr-send self :top-level-message-blip whole-blip)
      )
  )
)

(Defmethod (Mail-Control-Window :Can-Interpret-Input) (input)
  (and (and (consp (second input)) (equal (first input) (first (second input))))
       (send self :Operation-Handled-P (first input))
  )
)

(defmethod (Mail-Control-Window :Interpret-Input) (solicited-p input)
  (yw:safe-lexpr-send
    self (first (second input)) input solicited-p (rest (second input))
  )
)

(defmethod (mail-control-window :designate-io-streams) ()
  ;;Redefine this UCL method to set up the correct io bindings.
  (setq *terminal-io* (send self :get-pane :prompt-window))
  (setq *standard-input* *terminal-io*)
  (setq *standard-output* *terminal-io*)
  (setq *debug-io* *terminal-io*)
)

(Defmethod (Mail-Control-Window :around :handle-unknown-input)
	   (cont mt ignore)
;  (declare (special *command-stack*))
;  (if (and (boundp '*command-stack*) *command-stack*  ;;;; ????
;	   command-executed-immediately
;      )
;      (setq command-executed-immediately nil)
      (case ucl:input-mechanism
	((ucl:unknown ucl:key-or-button)
	 (if (and (consp ucl:kbd-input)
		  (send self :can-interpret-input ucl:kbd-input)
	     )
	     (send self :Interpret-Input nil Ucl:kbd-input)
	     (funcall-with-mapping-table cont mt :Handle-Unknown-Input)
	 )
	)
	(otherwise (funcall-with-mapping-table cont mt :Handle-Unknown-Input))
      )
;  )
)


(defun user-mail-control-window-inits ()
  *user-mail-control-window-inits*
)

(defun make-mail-control-window ()
"The official way to make a mail window."
  (apply 'make-instance 'Mail-Control-Window (user-mail-control-window-inits))
)

(defun (:Property Mailer-Windows constructor) ()
  (make-mail-control-window)
)

;(defwindow-resource mailer-windows ()
;  :Initial-Copies 0
;  :Constructor (make-mail-control-window)
;)

;-------------------------------------------------------------------------------

(defmethod (Mail-Control-Window :make-operator-sequence) (op arg1 arg2)
  (declare (special *current-mailbox*))
  (let ((mailbox (if (boundp '*current-mailbox*)
		     *current-mailbox*
		     current-mailbox
		 )
	)
       )
       (if (and (equal :sequence-then op)
		(equal op (first (send arg2 :Sequence-Specifier)))
	   )
	   (Make-A-Sequence nil :Owner self :Mailbox mailbox
	     :sequence-specifier
	       (cons op (cons arg1 (rest (send arg2 :Sequence-Specifier))))
	   )
	   (make-a-sequence nil :Owner self :Mailbox mailbox
			    :sequence-specifier (list op arg1 arg2)
	   )
       )
  )
)

(defun barf (&rest format-args)
  (mail-beep)
  (if format-args
      (apply 'format *error-output* format-args)
  )
  (signal 'sys:abort 'sys:abort)
)

(defmethod (Mail-Control-Window :read-defaulted-sequence)()
  (Parse-a-message-sequence *standard-input*
			    (or current-sequence *default-message-sequence*)
  )
)

(defun map-tree (function tree &rest args)
  (if (consp tree)
      (cons (apply #'map-tree function (first tree) args)
	    (apply #'map-tree function (rest  tree) args)
      )
      (apply function tree args)
  )
)


(defun copy-and-concretify-filter (sequence owner mailstream flavor &rest inits)
  (if (typep sequence 'message-sequence)
      (apply 'Make-A-Sequence flavor :Owner owner
	     :Mailbox mailstream
	     :Sequence-Specifier
	     (apply 'map-tree 'copy-and-concretify-filter
		    (send sequence :Sequence-Specifier)
		    owner mailstream flavor inits
	     )
	     inits
      )
      (if (typep sequence 'closure)
	  (funcall sequence)
	  sequence
      )
  )
)

(defun make-a-sequence (flavor &key
			(owner self)
			(mailbox (send owner :Current-Mailbox))
			sequence-specifier
			(User-Filter-Name nil)
			(inits nil)
		       )
  (if (and (not mailbox) (not *make-sequence-with-no-mailbox-ok-p*))
      (parse-error "No mailbox is open.")
      nil
  )
  (let ((real-flavor
	  (or flavor
	      (if (and (consp mailbox) (rest mailbox))
		  'multi-sequence
		  'message-sequence
	      )
	  )
	)
       )
       (let ((real-mailbox
	       (if *make-sequence-with-no-mailbox-ok-p*
		   nil
		   (if (subtypep real-flavor 'multi-sequence)
		       (list-if-not mailbox)
		       (progn (assert (equal (length (list-if-not mailbox)) 1))
			      (ucl:first-if-list mailbox)
		       )
		   )
	       )
	     )
	    )
	    (setq sequence-specifier (simplify-sequence sequence-specifier))
	    (Validate-Sequence sequence-specifier)
	    (let ((new
		   (apply #'make-instance real-flavor
			  :Owner owner
			  :Mailbox
			   real-mailbox
			  :Sequence-Specifier sequence-specifier
			  :Superior
			    (or (getf inits :Superior)
				(send (Get-Mail-Control-Window)
				      :Current-Sequence
				)
			    )
			  :Associated-P *sequences-created-are-associated*
			  (append (if owner
				      (list :owner owner)
				      nil
				  )
				  (if User-Filter-Name
				      (list :User-Filter-Name User-Filter-Name)
				      nil
				  )
				  inits
			  )
		   )
		  )
		 )
		 (if (and mailbox (not *disable-add-associated-filters*))
		     (loop for box in (list-if-not mailbox)
			   do (send box :add-associated-filter new)
		     )
		     nil
		 )
		 new
	    )
       )
  )
)

(defun simple-sequence (&rest specifiers)
  (if (typep self 'mail-control-window)
      (Simple-Sequence-1 specifiers)
      (send (Get-Mail-Control-Window) :Eval-Inside-Yourself
	    `(Simple-Sequence-1 ',specifiers)
      )
  )
)

(defun simple-disembodied-sequence (&rest specifiers)
  (let ((*make-sequence-with-no-mailbox-ok-p* t)
	(*disable-add-associated-filters* t)
	(*sequences-created-are-associated* nil)
       )
       (send (Get-Mail-Control-Window) :Eval-Inside-Yourself
	     `(Simple-Sequence-1 ',specifiers nil)
       )
  )
)

(defun-method simple-sequence-1 mail-control-window
	      (specifiers &optional (owner self))
  (declare (special *current-mailbox*))
  (if (and (equal (length specifiers) 1)
	   (typep (first specifiers) 'message-sequence)
      )
      (first specifiers)
      (make-a-sequence nil :Owner owner
		       :Mailbox (if (boundp '*current-mailbox*)
				    *current-mailbox*
				    current-mailbox
				)
		       :Sequence-Specifier specifiers
      )
  )
)

(defun make-a-dummy-sequence (name mailstream mailer)
  (ignore name)
  (let ((*mailer* mailer))
       (Make-A-Sequence nil :Owner mailer :Mailbox mailstream
			:Sequence-Specifier '(:Sequence-All)
       )
  )
)

;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------

(defmethod (Mail-Control-Window :find-mailbox) (box)
  (typecase box
    (string
     (find-if #'(lambda (x) (Stream-Name-Equal box (send x :Mailbox)))
	      all-mailboxes
     )
    )
    (imap-stream-mixin box)
    (cons (loop for item in box for result = (send self :Find-Mailbox item)
		when result collect result
	  )
    )
    (otherwise (yw-warn "~A is not the name of a currently open mailbox." box))
  )
)

(defmethod (Mail-Control-Window :find-mailbox-name) (box)
  (if (consp box)
      (loop for x in box append (send self :Find-Mailbox-Name x))
      (if (stringp box)
	  (list box)
	  (let ((mailboxes (send self :find-mailbox box)))
	       (loop for box in (list-if-not mailboxes)
		     collect (send box :Mailbox)
	       )
	  )
      )
  )
)

(defmethod (Mail-Control-Window :make-window-current) (window)
  (if window
      (progn (setq current-summary-window window)
	     (send window :Expose)
	     (send Mailbox-Selector :make-current window)
	     (send self :Set-Current-Mailbox (send window :Mailstreams))
	     (send self :Set-Current-Sequence (send window :Filter))
      )
      nil
  )
)

(defun find-mailbox-from-name (names)
  (loop for name in (list-if-not names)
	for result = (loop for Stream in *all-open-imap-streams*
			   when (string= (send stream :Mailbox) name)
			   return stream
	             )
	when result collect result
  )
)

(defun stream-name-equal (x y)
  (or (and (consp x) (consp y)
	   (= (length x) (length y))
	   (loop for a in x
		 for b in y
		 always (stream-name-equal a b)
	   )
      )
      (and (stringp x) (stringp y) (string= x y))
  )
)

(defmethod (Mail-Control-Window :find-or-open-mailbox)
	   (maybe-name &optional (window nil) (filter-for-new-window nil)
	    (read-only-p nil) (opened-by-filter-p nil)
	    (connection-type :IMAP) (create-window-p t)
	   )
  (declare (special *all-open-imap-streams*))
  (let ((name (loop for box in (list-if-not maybe-name)
	       	for result =
	       	  (Check-User-Mailbox box *user-host* (not window))
		when result collect result
	      )
	)
       )
       (let ((existing (or (find-mailbox-from-name name)
			   (and window (send window :Mailstreams))
		       )
	     )
	    )
	    (if (or (not existing)
		    (and (not window) (not (send self :Find-Mailbox name)))
		)
		(send self :Open-Mailbox-For
		      name existing filter-for-new-window read-only-p
		      opened-by-filter-p connection-type create-window-p
		)
		(if current-summary-window
		    (if (Stream-Name-Equal
			  (list-if-not existing)
			  (send current-summary-window :Mailstreams)
                        )
			(if window
			    (send self :Make-Window-Current window)
			    nil
			)
			(let ((w (send self :A-Summary-Window-Named existing)))
			     (or (and w (send self :Make-Window-Current w))
				 (if create-window-p
				     (send self :Create-Summary-Window-For
					   name filter-for-new-window
				     )
				     nil
				 )
			     )
			)
		    )
		    (if create-window-p
			(send self :Create-Summary-Window-For
			      name filter-for-new-window
			)
			nil
		    )
		)
	    )
	    (and current-summary-window (send current-summary-window :Expose))
	    (let ((box (send self :Find-Mailbox name)))
	         (if box
		     (progn (send self :Set-Current-Mailbox box)
			    (send self :Set-Current-Sequence
				  (and current-summary-window
				       (send current-summary-window :Filter)
				  )
			    )
		     )
		     nil
		 )
	    )
	    current-mailbox
       )
  )
)

(defun print-sequence-name (sequence)
  (if sequence
      (send sequence :Make-Label)
      ""
  )
)

(defmethod (Mail-Control-Window :Make-label) ()
  (if (and current-sequence current-mailbox)
      (let ((seq (print-sequence-name current-sequence)))
	   (if (equal seq "")
	       (format nil "Mailbox: ~A"
		       (print-short-mailbox-name current-mailbox)
	       )
	       (format nil "Sequence: ~A; Mailbox: ~A"
		       seq (print-short-mailbox-name current-mailbox)
	       )
	   )
      )
      (if current-mailbox
	  (format nil "Mailbox: ~A" (print-short-mailbox-name current-mailbox))
	  *no-mailbox-open-text*
      )
  )
)

(defmethod (Mail-Control-Window :remove-summary-window) (window)
  (setq all-summary-windows (remove window all-summary-windows))
)

(defmethod (Mail-Control-Window :existing-window-called) (streams the-filter)
  (loop for win in all-summary-windows
	when (and (not (set-difference streams (send win :mailstreams)))
		  (equal (send win :Make-Label)
			 (send win :Make-Label the-filter)
		  )
	     )
	return win
  )
)

(defmethod (Mail-Control-Window :create-summary-window-for)
	   (name &optional (the-filter nil) (inits nil))
  (send self :Resolve-Mailbox-Of-Sequence
	the-filter (send self :Find-Mailbox name)
  )
  (let ((real-filter (if (or (not the-filter)
			     (typep the-filter 'message-sequence)
			 )
			 the-filter
			 (apply-filter
			   the-filter
			   (get-filter-printed-representation the-filter)
			   self
			   (send self :Find-Mailbox name)
			 )
		     )
        )
       )
       (setq real-filter 
	     (if (and real-filter (not (associated-p real-filter)))
		 (Copy-And-Concretify-Filter
		   real-filter self (send self :Find-Mailbox name)
		   'message-sequence
		 )
		 real-filter
	     )
       )
       (let ((existing (send self :Existing-Window-Called
			     (list-if-not (send self :Find-Mailbox name))
			     real-filter
		       )
	     )
	    )
	    (if existing
		(progn (setq current-summary-window existing)
		       (send existing :Expose)
		       (send Mailbox-Selector :Make-Current existing)
		       (send existing :reinitialise inits)
		       (values existing t)
		)
		(let ((window (allocate-yw-resource 'Summary-Windows)))
		     (send window :Clean-Up)
		     (send window :reinitialise inits)
		     (multiple-value-bind (messages-found new-messages)
			     (send window :Set-Up self real-filter
				   (send self :Find-Mailbox-Name name)
				   (list-if-not (send self :Find-Mailbox name))
			     )
		       (loop for stream
			     in (list-if-not (send self :Find-Mailbox name))
			     do (send stream :Add-Associated-Window window)
		       )
		       (push window all-summary-windows)
		       (send Mailbox-Selector :New-Summary-Window window)
		       (setq current-summary-window window)
		       (send  window :Expose)
		       (values window messages-found new-messages)
		     )
		)
	    )
       )
  )
)

(defmethod (Mail-Control-Window :Initialize-Mailstream) (mailstream)
  (let ((TotalMsgs  (send MailStream :MessageCnt))
	(RecentMsgs (send MailStream :RecentCnt))
       )
    (send mailstream :Initialize-Mailstream TotalMsgs)
    (if (send mailstream :Mailbox)
        (Format-scroll-window *standard-output*
                "~&There ~[are~;is~:;are~] ~:*~D message~:P in ~A~
                 , ~:[~D~;none~*~] recent.~%"
                TotalMsgs
		(Print-Short-Mailbox-Name mailstream)
                (zerop RecentMsgs) RecentMsgs
        )
        nil
    )
    (loop for message-number from (max 1 (- TotalMsgs RecentMsgs)) to TotalMsgs
	  for message = (cache-entry-of message-number mailstream)
	  do (Signal-Event mailstream message :Recent-Message)
    )
  )
)

(defmethod (mail-control-window :resolve-mailbox-of-sequence) (sequence mailbox)
  (if (and sequence (typep sequence 'message-sequence)
	   (not (send sequence :Mailbox))
      )
      (etypecase sequence
	(multi-sequence (send sequence :Set-Mailbox (list-if-not mailbox)))
	(message-sequence (assert (equal (length (list-if-not mailbox)) 1))
	   (send sequence :Set-Mailbox (ucl:first-if-list mailbox))
	)
      )
      nil
  )
)

(defun open-imap-stream (mailbox-name read-only-p)
  "Opens a possibly read-only mailbox called mailbox-name."
  (condition-case (condition)
      (map-open mailbox-name nil read-only-p)
    (error (send condition :Report nil))
  )
)

(defmethod open-yw-stream (name read-only-p (connection-type (eql :Imap)))
  (Open-Imap-Stream name read-only-p)
)

;;;Edited by Tom Gruber            23 May 91  14:56
(defmethod (Mail-Control-Window :Open-Mailbox-For)
	   (names existing-streams &optional (filter-for-new-window nil)
	    (read-only-p nil) (opened-by-filter-p nil) (connection-type :Imap)
	    (window-p t)
	   )
"Opens a mailstream for the mailbox named Name.  Existing-stream is a stream
That's already open with that name.  Filter-for-new-window is a filter that
The user has provided in the form of a Get <foo> <filter> command or some such.
"
  (let ((streams
	  (if existing-streams
	      (loop for stream in existing-streams
		    do (pushnew stream all-mailboxes)
		    finally (return existing-streams)
	      )
	      (loop for name in names
		    collect
		    (let ((new-stream
			    (open-yw-stream name read-only-p connection-type)
			  )
			 )
			 (if (typep new-stream 'stream)
			     (push new-stream all-mailboxes)
			     (multiple-value-bind (mailbox host)
				 (mailbox-and-host-from-mailbox-name name)
				(yw-warn
				  "Sorry, I couldn't open a mail connection~
					 ~%to mailbox ~A on host ~A because:~
					 ~%~A" mailbox host new-stream
				)
			     )
			 )
			 (send self :initialize-mailstream new-stream)
			 (if opened-by-filter-p
			     (send new-stream :Set-Opened-With-Filter-P t)
			     nil
			 )
			 new-stream
		    )
	      )
	  )
	)
       )
       (send self :Resolve-Mailbox-Of-Sequence
	     filter-for-new-window (send self :Find-Mailbox names)
       )
       (let ((entries
	       (loop for name in names
		     collect
	            (assoc name *summary-windows-to-create-for-mailbox-alist*
		      :Test #'(lambda (x y)
		       ;; Note: The mailbox name passed has
		       ;; been expanded into hpp.stanford.edu:inbox e.g.
		       ;; we do a search here so that we get any substring
		       ;; match (maybe we should allow wildacarding some day???
		       ;; we use char-equal just in case there are
		       ;; case incompatibilities between the two.  Hosts
		       ;; are case insensitive.
				(search x (check-user-mailbox y)
					:Test #'char-equal
				)
			       )
		    )
	       )
	     )
	     (*current-mailbox* (first streams))
	    )
	    (declare (special *current-mailbox*))
	    (if window-p
	        (if (or filter-for-new-window (not entries))
		    (send self :create-summary-window-for names
			  filter-for-new-window
		    )
		    (loop for entry in entries do
		      (loop for (filter . inits) in (rest entry) do
			    (multiple-value-bind
			      (filt filt-name)
				(if filter
				    (parse-filter-expression filter)
				    nil
				)
			      (if (equal :no-window (first inits))
				  nil
				  (send self :create-summary-window-for names
					(if filt
					    (apply-filter
					      filt filt-name self streams
					    )
					    nil ;;; The generic inbox window.
					)
					inits
				  )
			      )
			    )
			    finally (if (not (assoc nil (rest entry)))
					(send self :Create-Summary-Window-For
					      names filter-for-new-window nil
					)
					nil
				    )
		      )
		    )
	        )
		nil
	    )
       )
  )
)

(defmethod (Mail-Control-Window :iconify-Window) (window)
  (send (second window) :iconify-me)
)

;;;Edited by Tom Gruber            8 Nov 90  15:02
(defmethod (Mail-Control-Window :Forget-Window)
	   (window &optional
	    (old-selected window)
	    (quiet-p nil)
	   )
  (send window :Forget-Items)
  (setq all-summary-windows (remove window all-summary-windows :Test #'eq))
  (let ((associated-streams (send window :Mailstreams)))
       (if (Is-In-yw-Resource window 'Summary-Windows)
	   (progn
		(if (and (send window :Mailstreams)
			 (send window :Filter)
		    )
		    (loop for stream in (send window :Mailstreams)
			  do (send stream :remove-associated-filter
				   (send window :Filter)
			     )
		    )
		    nil
		)
		(deallocate-yw-resource 'Summary-Windows window)
		(send window :clean-up)
		(send window :Reinitialise
		      (send window :Inits-To-Undo)
		)
		(send window :Set-Inits-To-Undo nil)
		(send window :bury)
	   )
	   nil
       )
       (send window :Bury)
       (send Mailbox-Selector :Remove-Summary-Window
	     window associated-streams quiet-p
       )
  )
  (setq current-summary-window
	(if old-selected
	    (if (eq old-selected window)
		(first all-summary-windows)
		old-selected
	    )
	    (first all-summary-windows)
	)
  )
  (send self :Set-Current-Sequence
	(if current-summary-window
	    (let ((filter (send current-summary-window :Filter)))
	         (if (equal :Recycled-Window filter)
		     nil
		     filter
		 )
	    )
	    nil
	)
  )
  (if current-summary-window (send current-summary-window :Expose) nil)
)

(defmethod (Mail-Control-Window :Read-type-command-with-selection)
	   (sequence operation &rest args)
  (declare (special *message-window*))
  (lexpr-send self :Read-Type-Command sequence operation args)
  (if (boundp '*message-window*)
      (send *message-window* :Select-Me)
      nil
  )
)

(defmethod (Mail-Control-Window :Read-type-command)
	   (sequence operation &rest args)
  (declare (special ucl:key-sequence *yw-daemon* *edit-server*))
  (send *Yw-daemon*   :Put-Task :Read-Type-Command
	(list :Eager-Message-Bodies sequence)
  )
  (send *edit-server* :Put-Task :Read-Type-Command
	(append (list operation sequence self) args)
  )
  (setq ucl:key-sequence nil)
)

(defmethod (Mail-Control-Window :Typein-Modes) ()
  (declare (special *new-typein-modes*))
  (if (and (boundp '*new-typein-modes*) *new-typein-modes*)
      *new-typein-modes*
      ucl:typein-modes
  )
)

(defun keyword-names (mailstream)
  (mapcar #'(lambda (x)
	      (list (intern (after-backslash X) :Keyword)
		    (string-capitalize (after-backslash X) :Spaces t)
		    (intern (symbol-name x) :Keyword)
	      )
	    )
	    (send mailstream :Keywords)
  )
)

(defun user-message-flags (mailstream number)
  (set-difference
    (cache-flags
      (MAP-Elt (send MailStream :MessageArray) number MailStream)
    )
    *system-flags*
  )
)

(defmethod (Mail-Control-Window :turn-into-bboard-path) (string)
  (let ((path (let ((fs:*merge-unix-types* nil))
		   (fs:merge-pathnames string
			  (fs:make-pathname
			    :Host *bboard-source-host*
			    :Name *default-bboard*
			    :Directory *Bboard-Source-Directory*
			    :Type *default-bboard-file-type*
			  )
		   )
	      )
	)
       )
       (send (if (stringp (send path :Type))
		 path
		 (fs:make-pathname :Type *Default-Bboard-File-Type*
				   :Defaults path
		 )
	      )
	      :String-For-Printing
       )
  )
)

(defmethod (Mail-Control-Window :get-next-bboard-name) (search-list)
  (setq current-bboard
	(if (and current-bboard
		 (member current-bboard search-list :Test #'string-equal)
	    )
	    (let ((index (position current-bboard search-list
				   :Test #'string-equal
			 )
		  )
		 )
	         (if index
		     (if (>= (+ 1 index) (length search-list))
			 (First search-list)
			 (nth (+ 1 index) search-list)
		     )
		     (First search-list)
		 )
	    )
	    (First search-list)
	)
  )
)

(defmethod (mail-control-window :Find-1) ()
  (if (and current-bboard
	   (member current-bboard *netnews-search-list*
		   :Test #'string-equal
	   )
      )
      (send self :Find-Netnews)
      (send self :Find-BBoard)
  )
)

(defmethod (mail-control-window :Find-bboard) ()
  (let ((*make-imap-stream-read-only* t))
       (declare (special *make-imap-stream-read-only*))
       (loop for count in *Bboard-Search-List*
	     Do (send self :Get-Next-Bboard-Name *Bboard-Search-List*)
		(Send self :Find-Or-Open-Mailbox
		      (send self :Turn-Into-Bboard-Path current-bboard)
		      nil
		      (send self :Get-Bboard-Filter current-bboard)
		)
	     until
	       (or (not count)
		   (not (zerop (send current-mailbox :Recentcnt)))
	       )
       )
  )
  (if (zerop (send current-mailbox :RecentCnt))
      (progn (format-scroll-window self "~&No new messages found in bboards")
	     (if (send self :Find-Mailbox
		       (Check-User-Mailbox *default-mailbox-name*)
		 )
		 (progn (format-scroll-window
			  self " - selecting default mailbox."
			)
			(send self :Find-Or-Open-Mailbox
			      (Check-User-Mailbox *default-mailbox-name*)
			)
		 )
		 (Format-scroll-window self ".")
	     )
      )
      (Format-scroll-window self "~&New messages found in ~A"
	      (print-short-mailbox-name current-mailbox)
      )
  )
)

(defmethod (mail-control-window :Find-NetNews) ()
  (let ((*make-imap-stream-read-only* t))
       (declare (special *make-imap-stream-read-only*))
       (loop for group in *NetNews-Search-List*
	     Do (send self :Get-Next-Bboard-Name *NetNews-Search-List*)
		(Send self :Find-Or-Open-Mailbox
		      (string-append *site-specific-nntp-server-host*
				     ":" current-bboard
                      )
		      nil
		      (send self :Get-Bboard-Filter current-bboard)
		)
	     until
	       (or (not group) (send current-mailbox :unseen-messages-p))
       )
  )
  (if (send current-mailbox :unseen-messages-p)
      (Format-scroll-window self "~&Unseen messages found in ~A"
	      (print-short-mailbox-name current-mailbox)
      )
      (progn (format-scroll-window
	       self "~&No new messages found in NetNews groups"
	     )
	     (if (send self :Find-Mailbox
		       (Check-User-Mailbox *default-mailbox-name*)
		 )
		 (progn (format-scroll-window
			  self " - selecting default mailbox."
			)
			(send self :Find-Or-Open-Mailbox
			      (Check-User-Mailbox *default-mailbox-name*)
			)
		 )
		 (Format-scroll-window self ".")
	     )
      )
  )
)

(defun just-bboard-name (name)
  (let ((index (position #\space name :Test #'char-equal)))
       (if index (subseq name (+ 1 index)) name)
  )
)

(defmethod (Mail-Control-Window :mark-messages-as-seen) (in)
  "I don't know how to do this for a bboard yet.?????"
;;; This will mark a normal messag as seen.  Will it work for a bboard?
  (let ((total (send (second in) :Messagecnt)))
       (Flag/Unflag-Message (second in) total :Set :\\Seen)
       (imap-select (second in) (Just-Bboard-Name (send (second in) :Mailbox)))
  )
)

(defmethod (Mail-Control-Window :bol) ()
  (multiple-value-bind (x y) (send prompt-window :read-Cursorpos)
    (ignore x)
;    (tv:mouse-confirm (format nil "~D, ~D" x y))
    (send prompt-window :set-cursorpos 0 y)
  )
)

(defmethod (Mail-Control-Window :apply-method-to-messages)
	   (method &optional (default *default-message-sequence*))
  (let ((sequence
	  (Parse-A-Message-Sequence
	    *standard-input* (or current-sequence default)
	  )
	)
       )
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation (Send sequence :Map-Over-Messages method))
  )
)

(defmethod (Mail-Control-Window :funcall-one-function-for-all-messages)
	   (function &optional (default *default-message-sequence*))
  (let ((sequence
	  (Parse-A-Message-Sequence
	    *standard-input* (or current-sequence default)
	  )
	)
       )
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation
	 (let ((messages (Send sequence :Numberise-Messages)))
	      (funcall function sequence messages)
	 )
       )
  )
)

(defun command-finished ()
  (declare (special *end-of-line-found*))
  (and (boundp '*end-of-line-found*) *end-of-line-found*)
)

(defun default-mail-server-user-directory ()
   *default-mail-server-user-directory*
)

(defun default-mailbox-directory ()
  (fs:force-user-to-login)
  *default-mailbox-directory*
)

(defwhopper (net:service-implementation-mixin :Complete-String)
	    (medium path string options)
  (let ((results (multiple-value-list
		   (let ((*look-in-directory-cache-first-p*
			   (and *cache-directory-lists-p*
				*use-caches-for-pathname-completion*
				(if (consp *use-caches-for-pathname-completion*)
				    (find-if #'(lambda (type) (typep self type))
					*use-caches-for-pathname-completion*
				    )
				    t
				)
			   )
			 )
			)
		        (continue-whopper medium path string options)
		   )
		 )
	)
       )
       (values-list results)
  )
)

(defflavor pseudo-directory-stream
	   (files all-files)
	   ()
  :Initable-Instance-Variables
)

(defmethod (pseudo-directory-stream :After :Init) (ignore)
  (setq all-files files)
)

(defmethod (pseudo-directory-stream :dir-list) ()
  all-files
)

(defmethod (pseudo-directory-stream :Next-Entry) ()
  (prog1 (first files) (setq files (rest files)))
)

(defwhopper (net:service-implementation-mixin :directory-stream)
	    (medium pathname &optional options)
  (flet ((_body_ (path)
	    (let ((stream (continue-whopper medium pathname options)))
	         (if (send stream :Operation-Handled-P :Next-Entry)
		     (let ((files (send stream :Dir-List)))
			  (setf (gethash path *directory-list-hash-table*)
				files
			  )
			  (if *look-in-directory-cache-first-p*
			      (make-instance 'Pseudo-Directory-Stream
					     :Files files
			      )
			      stream
			  )
		     )
		     stream
		 )
	     )
	 )
	)
     (if *look-in-directory-cache-first-p*
	 (let ((files (gethash pathname *directory-list-hash-table*)))
	      (if files
		  (make-instance 'pseudo-directory-stream :Files files)
		  (_body_ pathname)
	      )
	 )
	 (_body_ pathname)
     )
  )
)

;-------------------------------------------------------------------------------

(defun default-copy/move-path (mailstream name)
  (multiple-value-bind (ignore host)
      (mailbox-and-host-from-mailbox-name (send mailstream :mailbox))
    (let ((path (fs:parse-pathname
		  (string-downcase (string-append host ":" name))
		)
	  )
	 )
         (yw-zwei:copy/move-default-path
	   mailstream (send path :Name) (send path :Type) (send path :directory)
	 )
    )
  )
)

(defun canonical-mailbox-name-p (name)
"Is true if name is the name of a canonical mailbox."
  (declare (values alist-entry host?))
  (and (stringp name)
       (multiple-value-bind (name host)
	   (mailbox-and-host-from-mailbox-name name)
	 (ignore host)
	 (values (assoc name *canonical-mailbox-names* :Test #'string-equal)
		 host
	 )
       )
  )
)

(defun decanonicalize-mailbox-name (name wrt-host)
"Decanonicalizes a mailbox name with respect to a particular host."
  (multiple-value-bind (entry host?) (canonical-mailbox-name-p name)
    (let ((real-host (or (if (equal "" host?) wrt-host host?) wrt-host)))
	 (if entry
	     (if (and (third entry) real-host)
		 (string-append real-host ":" (second entry))
		 (second entry)
	     )
	     name
	 )
    )
  )
)

(defun canonicalize-mailbox-name (name)
"Canonicalizes a mailbox name if it can."
  (let ((entry (loop for (canon actual host-p) in *canonical-mailbox-names*
		     when (string-equal actual name)
		     return canon
	       )
	)
       )
       (if entry entry name)
  )
)

(defun sort-messages-by-mailstream (messages)
  (let ((streams-alist nil))
       (loop for message in messages
	     for stream = (cache-mailstream message)
	     for entry = (assoc stream streams-alist :Test #'eq)
	     if entry
	     do (setf (rest entry) (cons message (rest entry)))
	     else do (setf streams-alist
			   (cons (list stream message) streams-alist)
		     )
       )
       streams-alist
  )
)

(defun move-sequence-1 (delete-p sequence mailbox-name mailstream)
  (ignore mailstream)
  (multiple-value-bind (file-name host)
      (mailbox-and-host-from-mailbox-name mailbox-name)
    (let ((messages (send sequence :Numberise-Messages)))
         (if messages
	     (let ((real-file-name
		     (if (Canonical-Mailbox-Name-P mailbox-name)
			 (mailbox-and-host-from-mailbox-name
			   (Decanonicalize-Mailbox-Name file-name host)
			 )
			 (if (or yw:*copy/move-to-non-existent-files-ok-p*
				 (probe-file
				   (Default-Copy/Move-Path
				     (send sequence :Mailstream) file-name
				   )
				 )
				 (y-or-n-p
				   "~&The mailbox ~S does not exist.  Proceed?"
				   file-name
				 )
			     )
			     file-name
			     nil
			 )
		     )
		   )
		  )
		  (if real-file-name
		      (let ((sorted (Sort-Messages-By-Mailstream messages)))
			   (loop for (mailstream . messages) in sorted
				 do (funcall (if delete-p
						 'map-move-message
						 'map-copy-message
					     )
					     (or mailstream
						 (send *mailer*
						       :find-or-open-mailbox
						   (Decanonicalize-Mailbox-Name
						     *default-mailbox-name* host
						   )
						 )
					     )
					     messages real-file-name
				    )
			   )
		      )
		      (Barf " - Aborted")
		  )
	     )
	     nil
	 )
    )
  )
)

(defmethod (Mail-Control-Window :move-sequence) (delete-p)
  (let ((maybe-name (read-mailbox-name *standard-input*)))
       (let ((mailbox (Check-User-Mailbox maybe-name)))
	    (if mailbox
	        (let ((sequence (send self :Read-Defaulted-Sequence)))
		     (if (not sequence)
			 (parse-error "~&No message sequence found.")
			 nil
		     )
		     (continuation
		       (Move-Sequence-1
			 delete-p sequence mailbox current-mailbox
		       )
		     )
		)
		(parse-error "~&~A is not a valid mailbox name." maybe-name)
	    )
       )
  )
)

(Defmethod (mail-control-window :hardcopy-sequence) ()
  (let ((sequence (send self :Read-Defaulted-Sequence)))
       (if (not sequence) (parse-error "~&No message sequence found.") nil)
       (continuation (send sequence :hardcopy-self))
  )
)

(defun find-window-of-flavor (flavor &optional (current-window nil))
"Finds a window of the specified flavor.  If current-window is specified
Then this window is skipped.
"
  (or (tv:find-window-of-flavor flavor current-window)
      (labels ((search-inferiors (of)
		 (if (typep of flavor)
		     of
		     (loop for inf in (send of :Inferiors)
			   for match = (search-inferiors inf)
			   when (and match (not (equal match current-window)))
			   return match
		     )
		 )
	       )
	      )
	      (loop for screen in tv:all-the-screens
		    for match = (search-inferiors screen)
		    when (and match (not (equal match current-window)))
		    return match
	      )
      )
  )
)

(defun get-mail-control-window ()
  (or (and (boundp '*mailer*)
	   (typep *mailer* 'mail-control-window)
	   *mailer*
      )
      (find-window-of-flavor 'Mail-Control-Window)
      (first (map-yw-resource-return #'(lambda (window &rest ignore) window)
				     'Mailer-Windows
	     )
      )
      (allocate-yw-resource 'mailer-windows)
  )
)


(defun force-in-char (char mailer)
  (send (send mailer :Prompt-Window) :Force-Kbd-Input char)
  (sleep 0.05)
)

(defun wait-for-mailer-in-keyboard-state (mailer)
  (process-wait "Wait for reset"
    #'(lambda ()
	(string-equal(sys:process-whostate (send mailer :Process)) "Keyboard")
      )
  )
)

(defun apply-macro-command-string (string &optional (mailer *mailer*))
  (if (consp string)
      (mapcar #'apply-macro-command-string string)
      (let ((strings (split-into-lines string 0)))
	   (loop for str in strings
		 for short = (string-trim *whitespace-chars* str)
		 when (not (equal "" short))
		 do (Execute-Command-From-String short mailer)
;		    (Wait-For-Mailer-In-Keyboard-State mailer)
	   )
      )
  )
)

(defun find-frame (from flavor)
  (if (typep from flavor)
      from
      (find-frame (tv:sheet-superior from) flavor)
  )
)

(defun reset-daemon (daemon arrest-p priority-increment)
  (if (typep daemon 'process-status-pane)
      nil
     (let ((process (send daemon :Process)))
	  (send daemon :Send-If-Handles :Set-Task-Stream nil)
	  (send process :Set-Priority
		(min (+ (send process :Priority) priority-increment)
		     (or (get process :Initial-Priority) 0)
		)
	  )
	  (without-interrupts
	    (send process :Reset)
	    (if arrest-p (send process :Arrest-Reason arrest-p) nil)
	  )
     )
  )
)

(defun unarrest-daemon (daemon reason priority-decrement)
  (let ((process (send daemon :Process)))
       (send process :Set-Priority
	     (min (- (send process :Priority) priority-decrement)
		  (or (get process :Initial-Priority) 0)
	     )
       )
       (send process :Revoke-Arrest-Reason reason)
       (send process :Revoke-Arrest-Reason nil) ;; Just in case
       (send process :Arrest-Reasons)
  )
)


(defun unarrest-daemons (reason &optional (priority-decrement 0))
  (loop for daemon in *all-daemons*
	unless (typep daemon 'process-status-pane)
	do (Unarrest-Daemon daemon reason priority-decrement)
  )
)

(defun with-all-mailstream-locks-1 (streams function)
"Call function when we have acquired the locks for all of the streams."
  (if streams
      (with-mailbox-locked ((first Streams))
	(with-all-mailstream-locks-1 (rest streams) function)
      )
      (funcall function)
  )
)

(defun Reset-Daemons
       (&optional (arrest-p nil) (priority-increment 0)
	(delay-before-imperative nil)
       )
  (flet ((body ()
	       (loop for daemon in *all-daemons* do
		     (Reset-Daemon daemon arrest-p priority-increment)
	       )
	 )
	)
    (if delay-before-imperative
	(let ((result
		(with-timeout ((* 60 delay-before-imperative) nil)
		  (with-all-mailstream-locks (body) t)
		)
	      )
	     )
	     (if result nil (body))
	)
	(body)
    )
  )
)

(defun clear-all-mailstream-locks ()
  (loop for mailstream in *all-open-imap-streams* do
	(setf (imap.lock-location (send mailstream :imapLock)) nil)
  )
)

;-------------------------------------------------------------------------------

(defvar *old-y-or-n-p* #'y-or-n-p)

(defun yw-y-or-n-p (&rest args)
  (declare (special *prompt-window*))
  (send *prompt-window* :Eval-Inside-Yourself `(yw-y-or-n-p-1 ',args))
)

(tv:defun-rh yw-y-or-n-p-1 (args)
  (setf (tv:rh-fill-pointer) (tv:rh-typein-pointer))
  (setf (tv:rh-scan-pointer) (tv:rh-typein-pointer))
  (apply *old-y-or-n-p* args)
)

(defmethod (Mail-Control-Window :select-headers-1)
	   (sequence and-with old-window)
  (let ((forget-p t))
       (multiple-value-bind (window messages-found number-of-messages)
	   (send self :Create-Summary-Window-For
		 current-mailbox
		 (if and-with
		     (send self :Make-Operator-Sequence
			   :Sequence-And and-with sequence
		     )
		     sequence
		 )
	   )
	 (unwind-protect
	     (if (and (not messages-found)
		      (yw-y-or-n-p
			"No messages match this sequence.  ~
			Forget the sequence?"
		      )
		 )
		 nil
		 (progn (setq forget-p nil)
		        (if number-of-messages
			    (Format-scroll-window self
				    "~&~:[~D~;No~*~] message~:P selected~%"
				    (zerop number-of-messages)
				    number-of-messages
			    )
			)
		 )
	     )
	   (if forget-p
	       (send self :forget-window window old-window)
	       (send self :Find-Or-Open-Mailbox current-mailbox
		     window nil nil nil :Imap nil
	       )
	   )
	 )
       )
  )
)

(defmethod (Mail-Control-Window :select-headers) (&optional (and-with nil))
  (let ((sequence
	  (Parse-a-message-sequence *standard-input*
				    (if and-with nil current-sequence)
          )
	)
       )
       (if (not sequence) (Barf "~&No message sequence found.") nil)
       (continuation
	 (send self :Select-Headers-1 sequence and-with current-summary-window)
       )
  )
)

(defmethod (Mail-Control-Window :select-multiheaders-1)
	   (path/s sequence and-with old-window
	    &optional (connection-type :IMAP) (read-only-p nil)
	   )
  (let ((forget-p t)
	(directory
	  (loop for path in (list-if-not path/s)
		append (mapcar #'first
			(rest
			 (fs:directory-list
			   (fs:merge-pathnames
			     path
			     (if (and current-mailbox
				      (loop for box
					    in (list-if-not current-mailbox)
					    always (send box :Open-P)
				      )
				 )
				 (yw-zwei:copy/move-default-path
				   (ucl:first-if-list current-mailbox)
				   *default-copy-to-mailbox-name*
				   *default-mailbox-file-type*
				 )
				 (fs:make-pathname
				   :Directory (Default-Mailbox-Directory)
				   :Host *bboard-source-host*
				   :Type *default-mailbox-file-type*
				 )
			     )
			   )
			 )
			)
		       )
	  )
	)
       )
       (if (equal (length directory) 1)
	   (send self :Get-1 (send (first directory) :String-For-Host)
		 read-only-p sequence connection-type
		 (send (first directory) :Host)
	   )
	   (let ((streams
		   (loop for box in directory
			 unless (char-equal #\. (aref (send box :Name) 0))
			 append (list-if-not
				  (send self :Find-Or-Open-Mailbox
					(send box :string-for-printing)
					nil sequence read-only-p
					(not (equal nil sequence))
					connection-type nil
				  )
				)
		   )
		 )
		)
		(if streams
		    (let ((concrete
			    (Copy-And-Concretify-Filter sequence self streams
							nil
			    )
			  )
			 )
			 (multiple-value-bind
			   (window messages-found number-of-messages)
			     (send self :Create-Summary-Window-For
				   (loop for str in streams
					 collect (send str :Mailbox)
				   )
				   (if and-with
				       (send self :Make-Operator-Sequence
					     :Sequence-And and-with concrete
				       )
				       concrete
				   )
			     )
			   (send self :Set-Current-Mailbox streams)
			   (send self :Set-Current-Sequence
				 (and window (send window :Filter))
			   )
			   (unwind-protect
			       (if (and (not messages-found)
					(yw-y-or-n-p
					  "No messages match this sequence.  ~
					  Forget the sequence?"
					)
				   )
				   nil
				   (progn (setq forget-p nil)
					  (if number-of-messages
					      (Format-scroll-window self
					 "~&~:[~D~;No~*~] message~:P selected~%"
						 (zerop number-of-messages)
						 number-of-messages
					      )
					  )
				   )
			       )
			     (if forget-p
				 (send self :forget-window window old-window)
				 nil
			     )
			   )
			 )
		    )
		    (yw-warn "No mailboxes match {~S~^, ~}"
			     (list-if-not path/s)
		    )
		)
	   )
       )
  )
)


(defmethod (Mail-Control-Window :select-multiheaders) (&optional (and-with nil))
  (let ((paths
	  (loop for path = (string-trim
			     *whitespace-chars*
			     (let ((*eof-chars* '(#\,))
				   (*whitespace-chars*
				     (cons #\, *whitespace-chars*)
				   )
				  )
			          (declare (special *eof-chars*))
			          (Read-Mailbox-Name *standard-input* nil
						     nil
				  )
			     )
			   )
		collect path
		for commap = (equal #\, (peek-char t *standard-input*))
		when commap
		do (read-char *standard-input*)
		   (unread-char (skip-whitespace *standard-input*)
				*standard-input*
		   )
		until (not commap)
	  )
	)
	(sequence
	  (let ((*make-sequence-with-no-mailbox-ok-p* t))
	       (Parse-a-message-sequence *standard-input*
					 (if and-with nil current-sequence)
	       )
          )
	)
       )
       (if (not sequence) (Barf "~&No message sequence found.") nil)
       (continuation
	 (send self :Select-Multiheaders-1 paths sequence and-with
	       (and current-summary-window current-summary-window)
         )
       )
  )
)

(defmethod (Mail-Control-Window :get-1)
	   (string read-only-p sequence &optional (connection-type :IMAP)
	    (wrt-host *user-host*)
	   )
  ;;; {!!!!}  We'll have to put in a fix here for NNTP.
  (if (and (search "*" (the string string) :Test #'char-equal)
	   (y-or-n-p "Get all mailboxes matching ~S" string)
      )
      (let ((directory
	      (fs:directory-list
		(fs:merge-pathnames
		  string
		  (if current-mailbox
		      (yw-zwei:copy/move-default-path
			current-mailbox *default-copy-to-mailbox-name*
			*default-mailbox-file-type*
		      )
		      (fs:make-pathname
			:Directory (Default-Mailbox-Directory)
			:Host *bboard-source-host*
			:Type *default-mailbox-file-type*
		      )
		  )
		)
	      )
	    )
	   )
	   (loop for box in (mapcar #'first (rest directory))
		 unless (char-equal #\. (aref (send box :Name) 0))
		 do (send self :Find-Or-Open-Mailbox
			  (send box :string-for-printing)
			  nil sequence read-only-p
			  (not (equal nil sequence))
			  connection-type
		    )
	   )
      )
      (send self :Find-Or-Open-Mailbox
	    (if (equal "" string)
		*default-mailbox-name*
		(Decanonicalize-Mailbox-Name string wrt-host)
	    )
	    nil sequence read-only-p
	    (not (equal nil sequence))
	    connection-type
      )
  )
  (fully-unarrest-daemons)
  (load-databases-in-the-background)
)

(defun load-databases-in-the-background ()
  (process-run-function '(:Name "Load Databases" :Priority -3)
			#'(lambda () (Maybe-Load-Address-Database)
				     (maybe-load-rule-base)
			  )
  )
)


(defmethod (Mail-Control-Window :get-bboard-filter) (bboard)
  (let ((entry
	  (assoc bboard *bboard-filter-alist* :Test #'string-equal)
	)
       )
       (if entry
	   (second entry)
	   nil
       )
  )
)

;-------------------------------------------------------------------------------
;;; Initializations code.

(defun reset-reset-vars ()
  (loop for (var default) in (reverse *all-reset-vars*) do
	;; reverse in case we have serial dependencies.
	(setf (symbol-value var) (eval default))
  )
)

(defun clear-reset-vars ()
  (loop for (var default) in *all-reset-vars* do
	(makunbound var)
  )
)

(defun close-down-yw (&optional (reset-vars-too-p t))
  (let ((any 0))
       (Map-Over-Control-Windows 
	 #'(lambda (window ignore ignore)
	     (setq any (+ (length (send window :all-mailboxes)) any))
	   )
	 #'(lambda (X) (send X :all-mailboxes))
       )
       (if (and (> any 0)
		(progn (beep)
		       (y-or-n-p "~&There ~[are~;is~:;are~] ~D open ~
                                  mailbox~[es~;~:;es~].  Close ~
                                  ~[them~;it~:;them~]?"
				 any any any any
		       )
		)
	   )
	   (Map-Over-Control-Windows
	     #'(lambda (window ignore ignore)
		 (let ((*mailer* window))
		      (send window :Quit-Mailer-Internal)
		 )
		)
;	     #'(lambda (X) (send X :active-p))
	   )
	   nil
       )
       (maybe-save-address-database)
       (maybe-save-rule-base)
  )
  (if reset-vars-too-p (reset-reset-vars) nil)
)

(defmethod (mail-control-window :quit-mailer-internal) ()
  (with-daemons-reset-and-arrested ()
    (mapcar #'(lambda (win)
		(if (Is-In-yw-Resource win 'Summary-Windows)
		    (deallocate-yw-resource 'summary-windows win)
		    nil
		)
		(send win :Bury)
	      )
	      all-summary-windows
    )
    (setq current-summary-window nil)
    (setq current-mailbox nil)
    (send Mailbox-Selector :forget-all)
    (setq all-summary-windows nil)
    (mapcar #'(lambda (mbx)
		(MAP-Close mbx)
		(setq all-mailboxes (remove mbx all-mailboxes))
	      )
	      all-mailboxes
    )
    (setq all-mailboxes nil)
    (send prompt-window :Set-Label *no-mailbox-open-text*)
    (if (is-in-yw-resource self 'Mailer-Windows)
	(deallocate-yw-resource 'Mailer-Windows self)
	nil
    )
    (maybe-save-address-database)
    (maybe-save-rule-base)
    (tv:process-reset-and-enable (send self :Process))
  )
)

(defun start-up-yw ()
  (if (and fs:user-id
	   (not (member (string fs:user-id) *user-ids-no-to-start-up-for*
			:Test #'string-equal
		)
	   )
      )
      (progn (reset-reset-vars)
	     (if *automatically-start-mailer-on-login*
		 (get-mail-control-window)
		 nil
	     )
	     (load-databases-in-the-background)
	     (if (and (find-window-of-flavor 'Mail-Control-Window nil)
		      *automatically-open-connection-to-default-inbox*
		 )
		 (let ((*mailer*
			 (find-window-of-flavor 'Mail-Control-Window nil)
		       )
		      )
		      (send *mailer* :Find-Or-Open-Mailbox
			    *default-mailbox-name*
		      )
		      (send *mailer* :Mouse-Select)
		 )
		 nil
	     )
	     (if *read-yw-init-file-on-mailer-start-up*
		 (parse-yw-init-file)
		 nil
	     )
      )
      :not-started
  )
)

;-------------------------------------------------------------------------------
;;; Incremental search.

(defvar *saved-i-search-points*)

(defvar *last-search-string* "")

(defun i-search-1 (from-header backwards-p extract-and-match)
  (send (send *mailer* :Current-Summary-Window) :Maybe-Expose)
  (let ((*saved-i-search-points* '(("" nil)))
	(tv:rubout-handler nil)
       )
       (declare (special *Saved-I-Search-Points* tv:rubout-handler))
       (If backwards-p
	   (format *standard-output* "~&Reverse I-Search: ")
	   (format *standard-output* "~&I-Search: ")
       )
       (unwind-protect (i-search-2 from-header backwards-p extract-and-match)
	 (send *mailer* :deselect-searched-for-messages)
       )
  )
)

(defmethod (mail-control-window :deselect-searched-for-messages) ()
  (do-cache-entries
    (message current-mailbox (member :I-Search (selected-p message) :Test #'eq))
    (remove-selected-type message :I-Search)
  )
)

(defun coerce-to-char (x)
  (etypecase x
    (fixnum (int-char x))
    (character x)
  )
)

(defun increment-of (backwards-p)
  (if backwards-p -1 1)
)

(defun refresh-search-string (backwards-p)
  (send (send *standard-output* :Superior) :Bol)
  (send *standard-output* :Clear-Eol)
  (if backwards-p
      (format *standard-output* "~&Reverse I-Search: ~A"
	      (search-string)
      )
      (format *standard-output* "~&I-Search: ~A"
	      (search-string)
      )
  )
)

(defun Scroll-To-Saved-Point
       (header-number &optional (new-point-p nil) (point-to-remove nil))
  (if new-point-p
      (send *mailer* :deselect-searched-for-messages)
      nil
  )
  (if point-to-remove
      (let ((remove
	      (cache-entry-of point-to-remove (send *mailer* :Current-Mailbox))
	    )
	   )
	   (remove-selected-type remove :I-Search)
      )
      nil
  )
  (if (numberp header-number)
      (let ((message
	      (cache-entry-of header-number (send *mailer* :Current-Mailbox))
	    )
	    (found-p nil)
	   )
	   (do-summary-windows (window (send *mailer* :Current-Mailbox))
	     (setq found-p
		   (or found-p (send window :Scroll-To-show header-number))
	     )
	   )
	   (add-selected-type message :I-Search)
	   (if (not found-p)
	       (let ((scroller (send *mailer* :system-output-pane))
		     (font :default)
		    )
		    (tv:coerce-font font scroller)
		    (format-scroll-window *mailer*
		      "~&~A"
		      (without-tabs
			(get-and-format-header-display-string message)
			(floor (tv:sheet-inside-width scroller)
			       (tv:font-char-width font)
			)
			""
		      )
		    )
	       )
	       nil
	   )
      )
      (beep)
  )
  header-number
)

(defun extract-and-match-header (message extract-test)
  (let ((stream (send *mailer* :current-mailbox)))
       (case extract-test
	 (0 (maybe-preempt-envelopes stream message)
	    (let ((cache-entry (cache-entry-of message stream)))
	         (princ cache-entry 'si:null-stream)
		 (cache-header-display-string cache-entry)
	    )
	 )
	 (1 (map-fetch-header
	      stream message *daemon-header-read-grain-size*
	    )
	 )
	 (otherwise nil)
       )
  )
)

(defun extract-and-match-text (message extract-test)
  (let ((stream (send *mailer* :current-mailbox)))
       (case extract-test
	 ((0 1) (extract-and-match-header message extract-test))
	 (2 (map-fetch-message
	      stream message *daemon-body-read-grain-size*
	    )
	 )
	 (otherwise nil)
       )
  )
)

(defun find-new-string (new-string from-header backwards-p extract-and-match)
  (let ((header from-header)
	(extract-test 0)
	(max (send (send *mailer* :current-mailbox) :messagecnt))
       )
       (loop for thing-to-search
	     = (funcall extract-and-match header extract-test)
	     when (or (equal "" new-string)
		      (and thing-to-search
			   (search (the string new-string)
				   (the string thing-to-search)
				   :Test #'char-equal
			   )
		      )
		  )
	     return header ; Found a match
	     do (setq extract-test (+ 1 extract-test))
	        (if (not thing-to-search) ; Go to next message.
		    (progn (setq header (+ header (increment-of backwards-p)))
			   (setq extract-test 0)
		    )
		    nil
		)
	     until (or (< header 1) (> header max))
	     finally (return nil)
       )
  )
)

(defun search-string ()
  (declare (special *saved-i-search-points*))
  (first (first *saved-i-search-points*))
)

(defun i-search-c-s (from-header backwards-p char extract-and-match)
  (ignore backwards-p)
  (i-search-c-s-internal from-header nil char extract-and-match)
)

(defun i-search-c-r (from-header backwards-p char extract-and-match)
  (ignore backwards-p)
  (i-search-c-s-internal from-header t char extract-and-match)
)

(defun i-search-c-s-internal (from-header backwards-p char extract-and-match)
  (ignore char)
  (let ((*saved-i-search-points*
	  (cons (list (if (equal (search-string) "")
			  *Last-Search-String*
			  (search-string)
		      )
		      from-header
		)
		*saved-i-search-points*
	  )
	)
       )
       (refresh-search-string backwards-p)
       (let ((new-point
	       (find-new-string (search-string)
				(+ from-header (increment-of backwards-p))
				backwards-p extract-and-match
	       )
	     )
	    )
	    (if new-point
	        (I-Search-2
		  (Scroll-To-Saved-Point new-point) nil extract-and-match
		)
		(progn (beep)
		       (i-search-2 from-header nil extract-and-match)
		)
	    )
       )
  )
)

(defun i-search-rubout (from-header backwards-p char extract-and-match)
  (ignore char from-header)
  (Let ((old-point
	  (if from-header
	      from-header
	      (if (rest *saved-i-search-points*)
		  (second (first *saved-i-search-points*))
		  nil
	      )
	  )
	)
	(old-saved
	  (if *saved-i-search-points*
	      (second (first *saved-i-search-points*))
	      from-header
	  )
	)
       )
       (let ((*saved-i-search-points*
	       (if (rest *saved-i-search-points*)
		   (rest *saved-i-search-points*)
		   *saved-i-search-points*
	       )
	     )
	    )
	    (refresh-search-string backwards-p)
	    (I-Search-2
	      (Scroll-To-Saved-Point old-saved nil old-point)
	      backwards-p extract-and-match
	    )
       )
  )
)

(defun i-search-normal-char (from-header backwards-p char extract-and-match)
  (let ((new-string (string-append
		      (search-string) char
		    )
	)
       )
       (setq *Last-Search-String* new-string)
       (let ((new-point
	       (Find-New-String
		 new-string from-header backwards-p extract-and-match
	       )
	     )
	    )
	    (let ((*saved-i-search-points*
		    (cons (list new-string new-point) *saved-i-search-points*)
		  )
		 )
		 (refresh-search-string backwards-p)
		 (let ((position (scroll-to-saved-point
				   (second (first *saved-i-search-points*)) t
			         )
		       )
		      )
		      (if position
			  (i-search-2 position backwards-p extract-and-match)
			  (i-search-2 from-header backwards-p extract-and-match)
		      )
		 )
	    )
       )
  )
)

(defun i-search-2 (from-header backwards-p extract-and-match)
  (let ((char (coerce-to-char (send *standard-input* :Tyi))))
       (case char
	 (#\c-r
	  (if (equal "" (search-string))
	      nil
	      (i-search-c-r from-header backwards-p char extract-and-match)
	  )
	 )
	 (#\c-s
	  (if (equal "" (search-string))
	      nil
	      (i-search-c-s from-header backwards-p char extract-and-match)
	  )
	 )
	 (#\rubout
	  (i-search-rubout from-header backwards-p char extract-and-match)
	 )
	 ((#\ #\)
	  (setq *Last-Search-String* (search-string))
	  (values *Last-Search-String* from-header)
	 )
	 (Otherwise
	  (i-search-normal-char from-header backwards-p char extract-and-match)
	 )
       )
  )
)

;-------------------------------------------------------------------------------


;(defun x () (tv:window-under-mouse))
;(send foo :Kill)
;(send foo :active-command-tables)
;(clear-yw-resource 'mailer-windows)
;(clear-yw-resource 'summary-windows)
;(send (send foo :current-summary-window) :mailstream)
;(Build-Top-Level-Command-Table)
;(Build-Message-Sequence-Command-Table)
;(Build-Operator-Command-Table)
;(Build-Settable-Thing-Command-Table)
;(Build-All-Command-Tables)
;*all-open-imap-streams*
