;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;;;Mail Access Protocol routines -- interface between IMAP and YW

(defun close-error-handler (condition stream)
"An error handler for imap streams that's called when there is an error
during closing the stream.
"
  (ignore condition)
  (tv:notify tv:selected-window "Error closing stream ~A"
	     (send stream :Pretty-String)
  )
  (throw 'close-error :close-error)
)

(defun MAP-Close (MailStream)
"Closes an imap stream."
  "Logout user from IMAP server and break connection"
  (progn;with-close-errors-handled (MailStream)
     (prog1 (eq :Ok (second (IMAP-Logout MailStream)))
	    (if (send mailstream :Open-P)
		(send mailstream :Close)
		nil
	    )
     )
  )
)

(defun is-present (x)
"Is true of a cache entry slot if that cache entry slot is not :unbound."
  (not (equal :Unbound x))
)

(defmethod at-least-imap-3-p ((mailstream t))
  ;;; {!!!!}   Fix this when the unix server gets fixed.
  ;;; {!!!!}  I'm not sure that this is quite right.
  ;;; What happens if we are building message sequences from a rule base when
  ;;; we do not have a mailstream open.
  nil
)

(defmethod End-Of-Sequence-Action
	   (sequence message (mailstream imap-stream-mixin))
  (ignore sequence message)
  nil
)

(defmethod at-least-imap-3-p ((mailstream imap-stream-mixin))
  ;;; {!!!!}   Fix this when the unix server gets fixed.
  ;;; {!!!!}  I'm not sure that this is quite right.
  ;;; What happens if we are building message sequences from a rule base when
  ;;; we do not have a mailstream open.
  (and mailstream
       (send mailstream :Selected-Version)
       ;(version-less-p 3.0 (send mailstream :Selected-Version))
       (send mailstream :selected-features) ;;; !!!!
  )
)

(defun Map-Fetch-Message-internal (array messages mailstream preemptions)
  (let ((fetch-messages
	  (loop for msg
		in (if (listp messages)
		       messages
		       (loop for i from (number-of messages)
			     to (min (+ (number-of messages) preemptions)
				     (send mailstream :Messagecnt)
				)
			     collect i
		       )
		   )
		unless
		  (Is-Present
		    (Cache-RFC822Text (MAP-Elt array msg mailstream))
		  )
		collect msg
	  )
	)
       )
       (if fetch-messages
	   (let ((ip:*tcp-stream-whostate*
		   (if (equal (length fetch-messages) 1)
		       (format nil "Fetch Message ~D" (first fetch-messages))
		       (format nil "Fetch ~D Message~P"
			       (length fetch-messages) (length fetch-messages)
		       )
		   )
		 )
		)
		(let ((and-headers-too nil)
		      (just-bodies nil)
		     )
		     (loop for message in fetch-messages
			   do (if (Is-Present
				    (cache-rfc822header
				      (MAP-Elt array message mailstream)
				    )
				  )
				  (push message just-bodies)
				  (push message and-headers-too)
			      )
		     )
		     (if and-headers-too
			 (Imap-Send
			   MailStream 'FETCH
			   (list (colonify-numbers and-headers-too nil)
				 (if (At-Least-Imap-3-P mailstream)
				     '(RFC822.TEXT RFC822.HEADER)
				     ;;; Changed by JPR on
				     ;;; 07/19/89 14:30:46
				     ;;; to take advantage of
				     ;;; new unsupported
				     ;;; IMAP
				     ;;; get-without-setting-read-flag
				     ;;; rfc822 type.
				     '(RFC822.TEXT* RFC822.HEADER)
				 )
			   )
			 )
			 nil
		     )
		     (if just-bodies
			 (Imap-Send
			   MailStream 'FETCH
			   (list (colonify-numbers just-bodies nil)
				 (if (At-Least-Imap-3-P mailstream)
				     '(RFC822.TEXT)
				     ;;; Changed by JPR on
				     ;;; 07/19/89 14:30:46
				     ;;; to take advantage of
				     ;;; new unsupported
				     ;;; IMAP
				     ;;; get-without-setting-read-flag
				     ;;; rfc822 type.
				     '(RFC822.TEXT*)
				 )
			   )
			 )
			 nil
		     )
		)
	   )
	   nil
       )
  )
)

(defun Map-Fetch-Message
       (MailStream Messages &optional (preemptions 0) (depth 0))
"Fetch the text for the given message(s).  Messages can be either a number or
a list of numbers.  If Messages is a number then that number +
preemptions following messages are fetched.
"
  (declare (values message-body message-header))
  (let ((array (send MailStream :MessageArray)))
       (Map-Fetch-Message-internal array messages mailstream preemptions)
  )
  (let ((cache (cache-entry-of messages mailstream)))
       (assert (eq (cache-mailstream cache) MailStream))
       (if (Is-present (cache-RFC822Text cache))
	   (values
	     (cache-RFC822Text cache)
	     (let ((header (cache-rfc822header cache)))
		  (if (Is-Present header)
		      header
		      (Map-Fetch-Header mailstream messages)
		  )
	     )
	   )
	   (progn
	     (if (> depth 10)
		 (Imap-Select Mailstream
			      (mailbox-name-for-select
				(send Mailstream :mailbox)
			      )
		 )
		 nil
	     )
	     (loop for message in messages
		   when (> (number-of message) (send mailstream :messagecnt))
		   do (tv:notify tv:selected-window
				 "Inconsistent message counts found.  Resetting"
		      )
		      (signal 'sys:abort 'sys:abort)
	     )
	     (MAP-Fetch-Message MailStream Messages preemptions (+ 1 depth))
	   )
       )
  )
)

(defun MAP-Fetch-Flags (MailStream messages)
  "Fetch the flags for the given message(s)."
  (maybe-preempt-envelopes mailstream messages)
  (cache-flags (cache-entry-of messages mailstream))
)

(defun MAP-Fetch-Envelope (MailStream MsgSequence)
  "Fetch the envelope for the specified message(s)."
  ;;;MsgSequence is a single message number or a list of numbers
  (let ((MsgList (if (numberp MsgSequence)
		     (list MsgSequence)
		     MsgSequence)))
    (if msglist
        (let ((ip:*tcp-stream-whostate*
		(if (equal (length msglist) 1)
		    (format nil "Fetch Envelope ~D"
			    (first msglist)
		    )
		    (format nil "Fetch ~D Envelope~P"
			    (length msglist)
			    (length msglist)
		    )
		)
	      )
	     )
	     (IMAP-Send MailStream 'FETCH
		(list (colonify-numbers msglist nil) 'ALL)
	     )
	)
	nil
    )
    ;;;Return the envelope if single message specified
    (and (numberp MsgSequence)
	 (Cache-Envelope (MAP-Elt (send MailStream :MessageArray)
				  MsgSequence MailStream
			 )
	 )
    )
  )
)

(defun MAP-Fetch-Header (MailStream Messages &optional (preemptions 0))
"Fetch the header for the given message.  Messages can be either a number or
a list of numbers.  If Messages is a number then that number +
preemptions following headers are fetched.
"
  (let ((array (send MailStream :MessageArray)))
       (let ((fetch-messages
	       (loop for msg
		     in (if (listp messages)
			    messages
			    (loop for i from (number-of messages)
				  to (min (+ (number-of messages) preemptions)
					  (send mailstream :Messagecnt)
				     )
				  collect i
			    )
			)
		     when (equal :Unbound
				 (cache-rfc822header
				   (MAP-Elt array msg MailStream)
				 )
			  )
		     collect msg
	       )
	     )
	    )
	    (if fetch-messages
		(let ((ip:*tcp-stream-whostate*
			(if (equal (length fetch-messages) 1)
			    (format nil "Fetch Header ~D"
				    (first fetch-messages)
			    )
			    (format nil "Fetch ~D Header~P"
				    (length fetch-messages)
				    (length fetch-messages)
			    )
			)
		      )
		     )
		     (IMAP-Send  MailStream 'FETCH
				 (list (colonify-numbers fetch-messages nil)
				       'RFC822.Header
				 )
		     )
		)
		nil
	    )
       )
  )
  (let ((result
	  (cache-RFC822Header (cache-entry-of messages mailstream))
	)
       )
       (if (equal result :Unbound)
	   (MAP-Fetch-Header MailStream Messages preemptions)
	   result
       )
  )
)

(defun list-if-not (x)
"Returns (list x) if x is not a list."
  (if (listp x) x (list x))
)

(defun Maybe-Preempt-Envelopes
 (mailstream messages  &optional (preemptions *daemon-header-read-grain-size*))
"Preempts the envelopes in the (maybe list) Messages if they are not yet
present.  Messages can be either a number or a list of numbers.
If Messages is a number then that number + preemptions following messages
are fetched.
"
  (declare (special *potential-messages-to-preempt*))
  (process-wait "Wait for consistent mailbox"
		#'(lambda (stream)
		    (send stream :consistent-p)
		  )
		mailstream
  )
  (if (and (not (consp messages))
	   (is-present (Cache-Envelope (cache-entry-of messages MailStream)))
      )
      ;;; Just return this one, since we already have it.
      (Cache-Envelope (cache-entry-of messages MailStream))
      (let ((all-messages
	      (if (listp messages)
		   messages
		   (if (boundp '*potential-messages-to-preempt*)
		       (loop for mes in *potential-messages-to-preempt*
			     collect (number-of mes)
		       )
		       (loop for i from (number-of messages)
			     to (min (+ (number-of messages) preemptions)
				     (send mailstream :Official-Messagecnt)
				)
			     collect i
		       )
		   )
	      )
	    )
	   )
	   (let ((fetch-messages
		   (loop for msg in all-messages
			 unless
			 (is-present
			   (Cache-Envelope (cache-entry-of msg MailStream))
			 )
			 collect msg
		   )
		 )
		)
		(if fetch-messages
		    (let ((ip:*tcp-stream-whostate*
			    (if (equal (length fetch-messages) 1)
				(format nil "Fetch Envelope ~D"
					(first fetch-messages)
				)
				(format nil "Fetch ~D Envelope~P"
					(length fetch-messages)
					(length fetch-messages)
				)
			    )
			  )
			 )
			 (MAP-Fetch-Envelope MailStream fetch-messages)
		    )
		    nil
		)
	   )
      )
  )
)

(defun MAP-Fetch-To (MailStream Messages)
"Set the To string from the Address format of Message To slot, preempting
any relevant envelopes.
"
  (maybe-preempt-envelopes mailstream messages)
  (Do-Messages (cache mailstream messages Cache-ToText)
    (Format-Name 'envelope-to cache)
  )
  (cache-totext (cache-entry-of messages mailstream))
)

(defun format-name (cache-function cache-entry-or-address)
"Given a cache-function or a list of cache-functions to search that name an
 address holding slot, and a cache-entry it returns a string used to print
 out that slot value.
"
  (loop for cache-function in (List-If-Not cache-function)
	for result = (format-name-1 cache-function cache-entry-or-address)
	when result
	return result
	finally (return "????")
  )
)

(defun format-name-1 (envelope-accessor cache-entry-or-address)
"Given an envelope-accessor that names an address holding slot,
and a cache-entry it returns a string used to print out that slot value.
"
  (let ((address
	  (typecase cache-entry-or-address
	    (address cache-entry-or-address)
	    (envelope
	     (first
		(funcall envelope-accessor Cache-entry-or-address)
	     )
	    )
	    (cache
	     (or (Is-Present (Cache-envelope Cache-entry-or-address))
		 (MAP-Fetch-Envelope (cache-mailstream Cache-entry-or-address)
				     Cache-entry-or-address
	         )
	     )
	     (first
	       (funcall envelope-accessor
			(Cache-envelope Cache-entry-or-address)
	       )
	     )
	    )
	  )
	)
       )
       (if address
	   (or (let ((name (Address-PersonalName Address)))
		    (if (and name
			     *remove-dates-as-message-names-p*
			     (address-address-object address)
			     (not (typep (address-address-object address)
					 'mail:named-address
				  )
			     )
			     (not (send (address-address-object address)
					:Comments
				  )
			     )
			)
			(ucl:first-if-list
			  (send (address-address-object address) :Local-Part)
			)
			name
		    )
	       )
	       (and (address-address-object address)
		    (name-from-address-comment-field address)
	       )
	       (and (Address-MailBox Address)
		    (if (Address-Host Address)
			(format NIL "~A@~A"
				(Address-MailBox Address)
				(Address-Host Address)
			)
			(Address-MailBox Address)
		    )
	       )
	   )
	   nil
       )
  )
)

(defun name-from-address-comment-field (address)
  (if *use-comments-as-names-in-non-named-addresses*
      (let ((comments
	      (send (address-address-object address) :Comments)
	    )
	   )
	   (if comments
	       (string-trim *whitespace-chars-and-parens*
			    comments
	       )
	       nil
	   )
      )
      nil
  )
)

(defun map-fetch-bcc (mailstream messages)
"Gets the BCC fields of the messages denoted by Messages from Mailstream."
  (maybe-preempt-envelopes mailstream messages)
  (envelope-bcc (cache-envelope (cache-entry-of messages mailstream)))
)

(defun map-fetch-cc (mailstream messages)
"Gets the CC fields of the messages denoted by Messages from Mailstream."
  (maybe-preempt-envelopes mailstream messages)
  (envelope-cc (cache-envelope (cache-entry-of messages mailstream)))
)

(defun map-fetch-internaldate (mailstream messages)
"Gets the internal date fields of the messages denoted by Messages
from Mailstream.
"
  (maybe-preempt-envelopes mailstream messages)
  (cache-internaldate (cache-entry-of messages mailstream))
)

(defun MAP-Fetch-From (MailStream Messages)
"Set the From string from the Address format of Message From slot."
  (maybe-preempt-envelopes mailstream messages)
  (Do-Messages (cache mailstream messages Cache-FromText)
    (Format-Name 'envelope-from cache)
  )
  (Cache-FromText (cache-entry-of messages mailstream))
)

(defun canonical-address-1 (thing)
"An internal function used by Canonical-address.  Passed an address object
of some type, extracts the canonical address component.
"
  (typecase thing
    (mail:basic-address (send (yw-zwei:force-into-basic-address thing)
			      :local-part-string
			)
    )
    (address (canonical-address-1 (address-address-object thing)))
    (otherwise "????")
  )
)

(defun canonical-address (mailstream message address-extractor)
"Given a mailstream and a message extracts an address from the envelope of the
message using address-extractor and returns the canonical address.  Thus, 
the address James Rice <Rice@Sumex-aim.stanford.edu> will return \"Rice\".
"
  (maybe-preempt-envelopes mailstream message)
  (let ((addr (funcall address-extractor (cache-envelope message))))
       (parse-address-component addr)
       (typecase addr
	 (cons (Canonical-Address-1 (first addr)))
	 (address (Canonical-Address-1 addr))
	 (otherwise "????")
       )
  )
)

(defun canonical-bcc (mailstream message)
"Returns the canonical address of the BCC field of the specified message in the
mailbox denoted by Mailstream.
"
  (Canonical-Address mailstream message 'envelope-bcc)
)

(defun canonical-cc (mailstream message)
"Returns the canonical address of the CC field of the specified message in the
mailbox denoted by Mailstream.
"
  (Canonical-Address mailstream message 'envelope-cc)
)

(defun canonical-from (mailstream message)
"Returns the canonical address of the FROM field of the specified message in the
mailbox denoted by Mailstream.
"
  (Canonical-Address mailstream message 'envelope-from)
)

(defun canonical-to (mailstream message)
"Returns the canonical address of the TO field of the specified message in the
mailbox denoted by Mailstream.
"
  (Canonical-Address mailstream message 'envelope-to)
)

(defun MAP-Fetch-Length (MailStream Messages)
"Get and set the RFC822 size for the messages."
  ;;; Getting the from field will force us to get the rfc822 size.
  (Map-Fetch-From mailstream messages)
  (cache-rfc822size (cache-entry-of messages mailstream))
)

(defun MAP-Fetch-Id (MailStream Messages)
"Set the Id string from the Address format of Message From slot."
  (maybe-preempt-envelopes mailstream messages)
  (envelope-messageid
    (Cache-envelope (cache-entry-of messages mailstream))
  )
)

(defun MAP-Fetch-Subject (MailStream Messages)
"Set the Subject text of the message from the Subject slot of the Envelope."
  (if (and (not (consp messages))
	   (Is-Present (cache-subjecttext (cache-entry-of messages mailstream)))
      )
      (cache-subjecttext (cache-entry-of messages mailstream))
      (progn (maybe-preempt-envelopes mailstream messages)
	     (Do-Messages (cache mailstream messages Cache-SubjectText)
	       (let ((envelope (cache-envelope cache)))
		    (if (equal envelope :Unbound)
			""
			(let ((Subject (Envelope-subject envelope)))
			     (if (not (equal :Unbound subject))
				 (Envelope-subject envelope)
				 ""
			     )
			)
		    )
	       )
	     )
	     (Cache-SubjectText (cache-entry-of messages mailstream))
      )
  )
)

(defun canonicalise-subject (string)
"Given a string that denotes the subject of a message, canonicalises it.
This means stripping off any Re: strings or forwarding type text.
"
  (let ((string (string-trim *whitespace-chars* string)))
       (cond ((= (length string) 0) string)
	     ((search "Re:" string :End2 (length "Re:"))
	      (canonicalise-subject (subseq string (length "Re:")))
	     )
	     ((and (char= #\[ (aref string 0))
		   (char= #\] (aref string (- (length string) 1)))
		   (position #\: string :Test #'char=)
	      )
	      (let ((index (position #\: string :Test #'char=)))
		   (Canonicalise-Subject
		     (subseq string (+ index 1) (- (length string) 1))
		   )
	      )
	     )
	     (t string)
       )
  )
)

(defun canonical-subject (mailstream message)
"Returns the canonical subject of Message."
  (let ((subj (Map-Fetch-Subject mailstream message)))
       (Canonicalise-Subject subj)
  )
)

(defmethod MAP-Set-Flag ((MailStream imap-stream-mixin) messages Flag)
"Set a FLAG for the given messages."
  (let ((messages (List-If-Not messages))
	(ip:*tcp-stream-whostate* (format nil "Set ~A Flag" flag))
       )
       (and messages
	    (let ((SetFlagReply
		    (IMAP-Send Mailstream 'STORE
			   (list (colonify-numbers messages nil) '+FLAGS Flag)
		    )
		  )
		 )
		 (or (eq :Ok (second SetFlagReply))
		     (format-scroll-window mailstream
					   "~&Set flag ~A rejected: ~A"
					   flag (third SetFlagReply)
		     )
		 )
	    )
       )
  )
)

(defmethod MAP-Clear-Flag ((MailStream imap-stream-mixin) Messages Flag)
"Clear a flag for the given messages."
  (let ((messages (List-If-Not messages))
	(ip:*tcp-stream-whostate* (format nil "Clear ~A Flag" flag))
       )
       (let ((ClearFlagReply
	       (IMAP-Send MailStream 'STORE
			  (list (colonify-numbers messages nil)	'-FLAGS Flag)
	        )
	     )
	    )
	    (or (eq :Ok (second ClearFlagReply))
		(format-scroll-window mailstream "~&Clear flag ~A rejected: ~A"
			flag (third ClearFlagReply)
		)
	    )
       )
  )
)

(defun MAP-Search (MailStream &rest Criteria)
"Search for messages that satisfy the CRITERIA (i.e. Flagged, Answered, etc.)"
  (send MailStream :set-Selectedmsgs :No-Messages-Yet-Selected)
  (destructuring-bind (tag reply-type reply-arg)
		      (IMAP-Send MailStream :search Criteria :search)
    (ignore tag)
    (let ((messages-found (send MailStream :SelectedMsgs)))
	 (let ((return-value
		 (if (listp messages-found)
		     messages-found
		     (progn
		       (yw-warn "Search failed for search ~S because of ~S ~S"
				Criteria reply-type reply-arg
		       )
		       nil
		     )
		 )
	       )
	      )
	      (send MailStream :set-Selectedmsgs
		    :Error-You-Should-Never-Get-This
	      )
	      return-value
	 )
    )
  )
)

(defun MAP-Check-MailBox (MailStream &optional (notify-p nil))
"Check for new messages on Mailstream.  If Notify-p then the user is notified
of the new messages.
"
  (let ((old-message-count (or (send MailStream :MessageCnt) 0))
	(CheckReply (let ((ip:*tcp-stream-whostate*
			    (format nil "Check ~A"
				    (send mailstream :mailbox-name)
			    )
			  )
			  (result nil)
			 )
			 (with-timeout (1200)
			   (format (locally (declare (special *yw-daemon*))
					    *yw-daemon*) "~&Start Check at ")
			   (time:print-universal-time
			     (time:get-universal-time)
			     (locally (declare (special *yw-daemon*))
				      *yw-daemon*)
			   )
			   (setq result (IMAP-Send MailStream 'CHECK))
			   (format (locally (declare (special *yw-daemon*))
					    *yw-daemon*) "~&Finished Check at ")
			   (time:print-universal-time
			     (time:get-universal-time)
			     (locally (declare (special *yw-daemon*))
				      *yw-daemon*)
			   )	
			 )
			 result
	    )
	)
       )
       (if (neq :Ok (second CheckReply))
	   (if notify-p
	       (format-scroll-window mailstream "~&Check rejected: ~A"
		       (third CheckReply)))
	   ;;Else - report if no new msgs; Exists only reports if new msgs
	   (if (and notify-p
		    (equal old-message-count (send MailStream :MessageCnt))
	       )
	       (format-scroll-window
		 mailstream "~&There are no new messages."
	       )
	   )
       )
       (values CheckReply
	       (or (catch-error
		     (- (send MailStream :MessageCnt) old-message-count)
		     nil
		   )
		   0
	       )
       )
  )
)

(defun MAP-Expunge-MailBox (MailStream)
"Expunge the deleted messages from the mailbox."
  (let ((ExpungeReply
	  (let ((ip:*tcp-stream-whostate*
		  (format nil "Expunge ~A" (send mailstream :Mailbox-Name))
		)
	       )
	       (IMAP-Send MailStream 'EXPUNGE)
	  )
	)
       )
       (if (eq :Ok (second ExpungeReply))
	   ExpungeReply
	   ;;Else
	   (format-scroll-window mailstream
		   "~&Expunge rejected: ~A" (third ExpungeReply)
	   )
       )
  )
)

(defun With-Maybe-Reopening-Mail-File-1
       (from-mailbox name-from-path body-function)
"Executes Body-Function in such a way that current mailbox pointed to by
any stream pointing to name-from-path is temporarily reset to point to
some other file.  This allows copy and move operations to happen to mailboxes
that are already open on other streams.  Uses From-Mailbox as the source
mailbox.
"
  (declare (special *global-imap-lock*))
  (let ((existing (find-if
		    #'(lambda (stream)
			(multiple-value-bind (name host)
			    (and (send stream :Mailbox)
				 (mailbox-and-host-from-mailbox-name
				   (send stream :Mailbox)
				 )
			    )
			  (ignore host) ;; This will always be the same.
			  (string-equal name-from-path name)
			)
		      )
		      *all-open-imap-streams*
		  )
	)
       )
       (let ((current
	       (if existing
		   (multiple-value-bind (name host)
		       (mailbox-and-host-from-mailbox-name
			 (send existing :Mailbox)
		       )
		     (ignore host)
		     name
		   )
		   nil
	       )
	     )
	    )
	    (if existing
		(with-lock ((imap.lock-location *global-imap-lock*)) ;;; {!!!!}
		  (unwind-protect
		      (let ((*ignore-exists* t))
			   (imap-select existing
			      (mailbox-name-for-select
				(send from-mailbox :mailbox)
			      )
			   )
			   (funcall body-function)
		      )
		    (imap-select existing current)
		    (map-check-mailbox existing)
		  )
		)
		(funcall body-function)
	    )
       )
  )
)


(defun Map-Copy-Message
       (MailStream Message DestMailBox
	&optional (silent-p nil) (format-string "Copy")
       )
"Copy a message to the destination MailBox."
  ;;;Message is a list of the selected message numbers
  (Maybe-Mark-Message-As-Seen :Copy message mailstream)
  (with-maybe-reopening-mail-file (DestMailBox mailstream)
    (Map-Copy-Message-1
      MailStream Message DestMailBox silent-p format-string
    )
  )
)

(defun Map-Copy-Message-1
       (MailStream Message DestMailBox silent-p format-string)
"Copies the messages denoted by Message in Mailstream to the mailbox
named by DestMailbox.  If not silent-p then a notification is produced to this
effect.
"
  (declare (special *owning-window*))
  (let ((CopyReply
	  (let ((ip:*tcp-stream-whostate*
		  (format nil "~A ~A to ~A" format-string
			  (colonify-numbers message t) destmailbox
		  )
		)
	       )
	       (Imap-Send
		 MailStream 'copy
		 (list (colonify-numbers message nil) DestMailBox)
	       )
	  )
        )
       )
       (if (eq :Ok (second CopyReply))
	   (progn ;(Flag/Unflag-Message MailStream Message :Set :\\Seen)
	          ;;; I don't think we should mark as seen here.  I think this
	          ;;; could happen automatically.
		  (if silent-p
		      nil
		      (format-scroll-window mailstream
			      "~&Message~P ~A copied to ~A"
			      (length message)
			      (colonify-numbers message)
			      destmailbox
		      )
		  )
		  DestMailBox)
	   ;;Else
	   (format-scroll-window mailstream
				 "~&Copy rejected: ~A~%" (third CopyReply)
	   )
       )
  )
)

(defun MAP-Move-Message (MailStream Message DestMailBox)
"Move a message (i.e. Copy and Delete) to the destination MailBox."
  (Maybe-Mark-Message-As-Seen :Move message mailstream)
  (with-maybe-reopening-mail-file (DestMailBox mailstream)
    (map-move-message-1 MailStream Message DestMailBox)
  )
)

(defun known-to-be-seen-p (message mailstream)
"Is true is Message numbers a message which is known to be seen."
  (let ((cache-entry (cache-entry-of message mailstream)))
       (and cache-entry (is-present (cache-flags cache-entry))
	    (member :\\Seen (cache-flags cache-entry) :Test #'eq)
       )
  )
)

(Defun maybe-mark-message-as-seen (type messages mailstream)
"Marks the message as seen if the type matches one of the types specified
in *reasons-to-mark-messages-as-seen*.
"
  (if (member type *reasons-to-mark-messages-as-seen* :Test #'eq)
      (let ((mark-these
	      (loop for message in (list-if-not messages)
		    when (not (known-to-be-seen-p message mailstream))
		    collect message
	      )
	    )
	   )
	   (if mark-these
	       (Seen-Message MailStream mark-these)
	       nil
	   )
      )
      nil
  )
)

(defun map-move-message-1 (MailStream Message DestMailBox)
"Moves the messages denoted by Message in Mailstream to the mailbox
named by DestMailbox.  If not silent-p then a notification is produced to this
effect.
"
  (declare (special *owning-window*))
  (if (and (MAP-Copy-Message MailStream Message DestMailBox t  "Move")
	   (Flag/Unflag-Message MailStream Message :Set :\\Deleted)
      )
      (progn (format-scroll-window mailstream
		       "~&Message~P ~A moved to ~A"
		       (length message) (colonify-numbers message)
		       destmailbox
	     )
	     DestMailBox)))

(defun MAP-Elt (MessageArray Message MailStream &optional (error-p t))
"Returns an extant message cache entry or creates a new one for message number
message in the message array.
"
  (declare (optimize (speed 3) (safety 0)))
  (if (typep message 'cache)
      message
      (if messagearray
	  (let ((message-number (number-of message)))
	       (if (or (> message-number (array-active-length messagearray))
		       (< message-number 1)
		   )
		   (if error-p
		       (ferror nil
			 "Message number ~D is out of range for mailstream ~S"
			 message-number mailstream
		       )
		       (values nil :Message-Number-Out-Of-Range)
		   )
		   (let ((MessageRecord
			   (aref MessageArray (- Message-number 1))
			 )
			)
			(if (null MessageRecord)
			    (setf (cache-msg#
				    (setq MessageRecord
					  (setf (aref MessageArray
						      (- Message-number 1)
						)
						(make-cache
						  :Mailstream MailStream
						)
					  )
				    )
				  )
				  Message-number
			    )
			    nil
			)
			MessageRecord
		   )
	       )
	  )
	  ;;; Throw back out to the top.
	  (if error-p
	      (progn (if (member mailstream *all-open-imap-streams* :Test #'eq)
			 (progn (tv:notify tv:selected-window
					   "Mailstream ~S unexpectedly closed."
					   MailStream
				)
				(process-run-function
				  "Close" #'(lambda () (close mailstream))
				)
			 )
			 nil
		     )
		     (signal 'sys:abort 'sys:abort)
	      )
	      (values nil :No-Message-Array)
	  )
      )
  )
)

;===============================================================================

;;;IMAP support routines

(defun IMAP-Open (Host)
"The protocol-independent code to open a connection to the specified host."
  (let ((ip:*tcp-stream-instantiator* 'ip-imap-stream-instantiator)
	(net:*medium-stream-type-alist*
	  (cons (subst 'chaos-imap-stream
		       'chaos:ascii-translating-character-stream
		       (assoc :Ascii-Translating-Character-Stream
			      net:*medium-stream-type-alist*
		       )
		)
		net:*medium-stream-type-alist*
	  )
	)
       )
       (declare (special ip:*tcp-stream-instantiator*))
       (net:open-connection-on-medium (net:parse-host host) :Byte-Stream
				      "Generic-IMAP"
				      :Stream-Type
				      :Ascii-Translating-Character-Stream
       )
  )
)

(defun feature-enabled-p (feature mailstream)
"Is true when a feature is enabled for mailstream."
  (member feature (send mailstream :Selected-Features) :Test #'eq)
)

(defmethod (imap-stream-mixin :Initialize-Mailstream) (TotalMsgs)
  ;;This array is used to hold the Cache components for each message
  (setq MessageArray (make-array TotalMsgs :Adjustable T))
)

(defmethod (imap-stream-mixin :Consistent-P) ()
  (and messagearray
       (numberp messagecnt)
       (numberp recentcnt)
       (<= messagecnt (array-active-length messagearray))
       (>= recentcnt 0)
  )
)

(defmethod (imap-stream-mixin :Add-Mailbox) (new-mailbox)
"Adds a new mailbox to the list of mailboxes preserved."
  (pushnew new-mailbox all-mailboxes :Test #'string=)
)

(defmethod (imap-stream-mixin :Add-BBoard) (new-bboard)
"Adds a new bboard to the list of bboards preserved."
  (pushnew new-bboard all-bboards :Test #'string=)
)

(defmethod (imap-stream-mixin :Host) (&optional (no-error-p nil))
"Returns the host object for the host to which we are connected."
  (if mailbox
      (multiple-value-bind (name host)
	  (mailbox-and-host-from-mailbox-name mailbox)
	(ignore name)
	(net:parse-host host no-error-p)
      )
      nil
  )
)

(defmethod (imap-stream-mixin :mailbox-name) ()
"Returns the name of the mailbox that we represent."
  (multiple-value-bind (name host)
      (mailbox-and-host-from-mailbox-name Mailbox)
    (if (equal (net:parse-host host t) (net:parse-host *User-Host*))
	name
	mailbox
    )
  )
)

(defmethod (imap-stream-mixin :After :Set-Recentcnt) (to)
"Makes sure that any icons are updated if new mail arrives."
  (ignore to)
  (if (and owning-window
	   (send owning-window :Icon)
	   (type-specifier-p 'w:graphics-icon)
	   (typep (send owning-window :Icon) 'w:graphics-icon)
      )
      (send owning-window :recompute-icon)
      nil
  )
)

(defmethod (Imap-stream-mixin :new-mail-p) ()
"Is true if new mail has arrived on Self."
  (and (numberp recentcnt)
       (numberp messagecnt)
       (> Recentcnt 0)
       (arrayp messagearray)
       (send self :open-p)
       (loop for i from (max 1 (- messagecnt recentcnt)) to messagecnt
	     for entry = (map-elt messagearray i self)
	     when (or (not (Is-present (cache-envelope entry)))
		      (not (Is-present (cache-flags    entry)))
		      (not (member :\\Seen (cache-flags entry) :Test #'eq))
		  )
	     return t
	     finally (return nil)
       )
  )
)

(defun filter-equal (a b)
"Is true if the filters A and B are equivalent."
  (or (eq a b)
      (if (equal (type-of a) (type-of b))
	  (typecase a
	    (message-sequence
	     (and (equalp (send a :Mailbox) (send b :Mailbox))
		  (Filter-Equal (send a :Sequence-Specifier)
				(send b :Sequence-Specifier)
		  )
	     )
	    )
	    (cons (and (Filter-Equal (first a) (first b))
		       (Filter-Equal (rest a)  (rest b))
		  )
	    )
	    (otherwise (equalp a b))
	  )
	  nil
      )
  )
)

(defun trivial-filter-p (filter)
"Is true if FILTER just represents a message number."
  (typecase filter
    (message-sequence (trivial-filter-p (send filter :Sequence-Specifier)))
    (null t)
    (cons (and (equal 1 (length filter)) (numberp (first filter))))
  )
)

(defmethod (Imap-stream-mixin :Add-Associated-Filter) (filter)
"Adds filter to the list of associatewd filters."
  (if (trivial-filter-p filter)
      nil
      (pushnew filter associated-filters :Test 'filter-equal)
  )
)

(defmethod (Imap-stream-mixin :remove-Associated-Filter) (filter)
"Adds filter to the list of associatewd filters."
  (setq associated-filters
	(remove filter associated-filters :Test 'filter-equal)
  )
)

(defmethod (Imap-stream-mixin :Invalidate-Computed-Orders) ()
"Invalidates the computed orders of all of the filters associated with self."
  (loop for filter in associated-filters do
	(send filter :Invalidate-Computed-Order)
  )
)

(defmethod (Imap-stream-mixin :Maybe-Invalidate-Computed-Orders) ()
"Conditionally invalidates the computed orders of all of the filters
associated with self.
"
  (loop for filter in associated-filters do
	(send filter :Maybe-Invalidate-Computed-Order)
  )
)

(defmethod (Imap-stream-mixin :Mark-Computed-Orders-For-Invalidation) (descriptor)
"Marks the computed orders of all of the filters associated with self so that
they know that they should be invalidated later.
"
  (loop for filter in associated-filters do
	(send filter :Mark-Computed-Order-For-Invalidation descriptor)
  )
)

;(defmethod (Imap-stream-mixin :Maybe-Add-To-Computed-Orders) (descriptor)
;"Conditionally adds a message descriptor to the computed order of all of the
;filters associated with self.
;"
;  (loop for filter in associated-filters do
;	(send filter :Maybe-Add-To-Computed-Order descriptor)
;  )
;)

;;; This mod put in by JPR because, for some reason we are ending up with
;;; bogus associated filters that are pointing to closed streams.  We snip
;;; them out here.  This is a really bogus thing to do, but I don't have
;;; the time to fix it properly and I can't get the symptoms to appear
;;; right now. JPR.  {!!!!}  Hack alert. !!!!
(defmethod (Imap-stream-mixin :Maybe-Add-To-Computed-Orders) (cache)
"Conditionally adds a message cache to the computed order of all of the
filters associated with self.  If the stream of the filter is closed then
we snip it out (hack).
"
  (loop for filter in associated-filters
	for mailstreams = (send filter :mailstreams)
        do (if mailstreams
	       (if (loop for mailstream in mailstreams
			 always (send mailstream :open-p)
		   )
		   (send filter :Maybe-Add-To-Computed-Order cache)
		   (setq associated-filters (remove filter associated-filters))
	       )
	       nil
	   )
  )
)



(defmethod (Imap-stream-mixin :mark-as-read-only) ()
"Marks self as read-only."
  (setq read-only-p t)
)

(defmethod (Imap-stream-mixin :mark-as-read-write) ()
"Marks self as read-write."
  (setq read-only-p nil)
)

(defmethod (Imap-stream-mixin :After :Init) (ignore)
"Records self as an open imap stream and records who the owning window is."
  (declare (special *all-open-imap-streams* *mailer*))
  (pushnew self *all-open-imap-streams*)
  (setq owning-window *mailer*)
)

(defmethod (Imap-stream-mixin :Add-Associated-Window) (window)
"Adds Window to the list of associated windows of self."
  (pushnew window associated-windows)
)

(defmethod (Imap-stream-mixin :Remove-Associated-Window) (window)
"Adds Window to the list of associated windows of self."
  (setq associated-windows (remove window associated-windows))
)

(defmethod (Imap-stream-mixin :After :Close) (&rest ignore)
"Cleans up the stream and the things that it points to."
  (send self :Clean-Up-After-Close nil)
)

(defwhopper (imap-stream :Close) (&rest args)
"Makes sure that if we get a tcp buffer deallocate error because of a warm boot
we just proceed.
"
  (if (boundp '*dont-catch-close-errors-p*)
      (lexpr-continue-whopper args)
      (condition-case (condition)
	  (lexpr-continue-whopper args)
	(sys:deallocate-non-resource-entry nil)
	(error (if mailbox
		   (progn (tv:notify tv:selected-window "~&An error occurred ~
			   whilst closing an IMAP stream.  Closing the stream ~
			   surgically."
			  )
			  (setq mailbox nil)
			  (if messagearray
			      (loop for i
				    from 0
				    below (array-active-length messagearray)
				    do (setf (aref messagearray i) nil)
			      )
			      nil
			  )
			  (setq search-cache nil)
			  (setq associated-filters nil)
		   )
		   nil
	       )
	)
      )
  )
)

(defmethod (Imap-stream-mixin :clean-up-after-close) (&optional (abnormal-p t))
"Cleans up the stream and the things that it points to.  For instance, it
removes itself from the mailbox selector window and updates any open zmacs
buffers on this stream so that they are marked as closed.
"
  (declare (special *all-open-imap-streams*))
  (setq *All-Open-Imap-Streams* (remove self *all-open-imap-streams*))
  (if messagearray
      (loop for i from 0 to (- (array-active-length messagearray) 1) do
	    (if (aref messagearray i)
		(if (or abnormal-p
			(not *kill-buffers-when-mailstream-closed-by-user*)
		    )
		    (rename-associated-buffers
		      (aref messagearray i) "Closed" :Closed
		    )
		    (kill-associated-buffers (aref messagearray i))
		)
		nil
	    )
	    (setf (aref messagearray i) nil)
      )
      nil
  )
  (setq messagearray nil)
  (if associated-windows
      (loop for window in associated-windows do
	    (let ((owner (send window :Owner)))
		 (send owner :Forget-Window window nil t)
	    )
      )
      nil
  )
  (if owning-window
      (progn (send owning-window :Set-All-Mailboxes
		   (remove self
			   (send owning-window :All-Mailboxes)
			   :Test #'equal
		   )
	     )
	     (if (and (send owning-window :Current-Mailbox)
		      (equal self (send owning-window :Current-Mailbox))
		 )
		 (send owning-window :Set-Current-Mailbox
		       (first (send owning-window :All-Mailboxes))
		 )
		 nil
	     )
      )
      nil
  )
  (setq associated-windows nil)
  (setq owning-window nil)
  (setq mailbox nil)
)

(defmethod (Imap-stream-mixin :pretty-string) ()
"A short string to name self."
  (if mailbox (print-short-mailbox-name self) "Closed")
)

(defmethod (Imap-stream-mixin :bboard-p) ()
"Is true if self points to a bboard file."
  (and (host-object-from-mailbox-name mailbox)
       (equalp *bboard-source-directory*
	       (pathname-directory (fs:default-pathname mailbox))
       )
  )
)

(defun host-object-from-mailbox-name (string)
  (let ((index (or (search ":" (the string string) :Test #'char-equal) nil)))
       (let ((host-name (if index (subseq string 0 index) nil)))
	    (and host-name (net:parse-host host-name t))
       )
  )
)

(defun search-cache= (x y)
"Is true if two search cache entries X and Y are equivalent."
  (and (equalp (search-cache-entry-search-class x)
	       (search-cache-entry-search-class y)
       )
       (equalp (search-cache-entry-search-string x)
	       (search-cache-entry-search-string y)
       )
       (equalp (search-cache-entry-search-conjunction x)
	       (search-cache-entry-search-conjunction y)
       )
  )
)

(defmethod (imap-stream-mixin :Add-Search-Cache-Entry) (entry)
"Adds a search cache entry to the list of search caches."
  (setq search-cache
	(cons entry (remove entry (the list search-cache) :Test 'search-cache=))
  )
  entry
)

(defmethod (imap-stream-mixin :Flush-Search-Cache)
	   (&optional (classes nil supplied-p))
"Flushes the search cache.  If classes are supplied then only those search
caches are flushed.
"
  (if supplied-p
      (setq search-cache (set-difference search-cache classes))
      (setq search-cache nil)
  )
  (loop for sce in search-cache
	when (> messagecnt (array-total-size (search-cache-entry-mask sce)))
	do (adjust-array (search-cache-entry-mask sce)
			 (floor (* 1.5 messagecnt))
	   )
	when (not (= (fill-pointer (search-cache-entry-mask sce)) messagecnt))
	do (setf (fill-pointer (search-cache-entry-mask sce)) messagecnt)
  )
)

(defun element-matches (new-items element mailstream)
"Is true if the element in a search cache matches with the new flag items with
respect to mailstream.  It checks both the flags and the inverse flags.
"
  (let ((entry (assoc (first element) *simple-term-specifiers* :Test #'eq))
	(inverse (inverse-of-key (first element) mailstream))
       )
       (let ((inverse-entry
	       (assoc inverse
		      *simple-term-specifiers* :Test #'eq
               )
	     )
	    )
	    (if (third entry)
		;;; Then this sequence names a flag so this is required flag.
		(if (eq :\\Keyword (third entry))
		    (member (second element) new-items :test #'eq)
		    (member (third entry) new-items :test #'eq)
		)
		(if (third inverse-entry)
		    ;;; Then the inverse has a flag so we must not have this
		    (not (member (third inverse-entry) new-items :Test #'eq))
		    (ferror nil "???")
		)
	    )
       )
  )
)

(defun add-sorted (member list)
"Adds a new member to a list preserving the sorted order of the list."
  (let ((list-to-use (cons most-negative-fixnum list)))
       (loop for pointer on list-to-use
	     for value in list
	     when (<= member value)
	     do (progn (setf (rest pointer) (cons member (rest pointer)))
		       (return nil)
		)
	     finally (setf (rest pointer) (list member))
       )
       (rest list-to-use)
  )
)

(defun element-mentions-flag-p (element mailstream)
  "True if a conjunction element mentions a changable flag."
  (or (third (assoc (first element) *simple-term-specifiers* :Test #'eq))
      (third (assoc (inverse-of-key (first element) mailstream)
		    *simple-term-specifiers* :Test #'eq
	     )
      )
  )
)

(defun flags-mentioned-in-pattern-p (conjunction mailstream)
  "True if a search conjunction mentions any changable flag."
  (loop for element in conjunction
	when (element-mentions-flag-p element mailstream)
	return t
	finally (return nil)
  )
)

(defun non-flag-matches (message element mailstream)
"Is true if a non-flag element in a search cache matches message with
respect to the mailstream.
"
  (let ((seq (make-a-sequence nil :Owner (get-mail-control-window)
			      :mailbox mailstream
			      :sequence-specifier element
	     )
	)
       )
       (send seq :Accept-Message-P message)
  )
)

(defun reset-associated-filters (of-sce mailstream)
  (let ((filters (filters-that-refer-to-search-cache-entry mailstream of-sce)))
       (loop for filter in filters
	     do (send filter :Invalidate-Computed-Order)
       )
  )
)

(defun Search-Cache-Maybe-Update
       (matched-p present-p message sce)
"Possibly updates a search cache entry for message depending on
whether the message matched (matched-p) the search pattern of the cache entry
and whether it is already there or not.  It either adds, removes or does nothing
as appropriate.  If matched-p is :inverse then the match criterion is inverted.
"
  (let ((real-number (number-of message)))
       (if (and matched-p (not (equal :Inverse matched-p)))
	   (if present-p
	       (values nil :already-present)
	       (progn (setf (search-cache-entry-numbers sce)
			    (add-sorted real-number
			      (search-cache-entry-numbers sce)
			    )
		      )
		      (setf (aref (search-cache-entry-mask sce) real-number) 1)
		      (Reset-Associated-Filters
			sce (cache-mailstream message)
		      )
		      (values nil :added)
	       )
	   )
	   (if present-p
	       (progn (setf (search-cache-entry-numbers sce)
			    (remove real-number
				    (search-cache-entry-numbers
				      sce
				    )
			    )
		      )
		      (setf (aref (search-cache-entry-mask sce) real-number) 0)
		      (Reset-Associated-Filters
			sce (cache-mailstream message)
		      ) 
		      (values nil :removed)
	       )
	       (values nil :already-absent)
	   )
       )
  )
)

(defun Conjunction-Forces-Flush-P
         (new-items sce message mailstream)
"Is passed a set of new flag items and a search cache entry.  Returns true if
the new flag items for the message denoted by message will force the
search cache to be flushed.  As a side effect it checks to see whether it can/
should update the search cache for this conjunction.
"
  (if (search-cache-entry-search-conjunction sce)
      (let ((matched-p
	      (and (flags-mentioned-in-pattern-p
		     (search-cache-entry-search-conjunction
		       sce
		     )
		     mailstream
		   )
		   ;; the conjunction must mention a flag, otherwise we can't be
		   ;; sensitive to any changes.
		   (loop for element
			 in (search-cache-entry-search-conjunction
			      sce
			    )
			 for match = (if (Element-Mentions-Flag-P
					   element mailstream
					 )
					 (Element-Matches
					   new-items element mailstream
					 )
					 (non-flag-matches
					   message element mailstream
					 )
				     )
			 when (not match)
			 do (return nil)
			 finally (return t)
		   )
	      )
	    )
	    (present-p
	      (message-present-in-search-cache-entry-p sce (number-of message))
	    )
	   )
	   (Search-Cache-Maybe-Update
	     matched-p present-p message sce
	   )
      )
      nil
  )
)

(defun Simple-Cache-Forces-Flush-P
       (new-items search-cache-entry message flag-list)
"Is passed a set of new flag items and a search cache entry.  Returns true if
the new flag items for the message denoted by message will force the
search cache to be flushed.  As a side effect it checks to see whether it can/
should update the search cache for this simple entry.
"
  ;;; This is really only called for flags, not fields in general
  (let ((search-class (search-cache-entry-search-class search-cache-entry)))
       (if (and search-class (member search-class flag-list :Test #'eq))
	   (let ((matched-p
		  (loop for new-item in new-items do
			(cond ((equal search-class
				      new-item
			       )
			       (return t)
			      )
			      ((and (equal search-class :\\Keyword)
				    (string-equal
				      (string (search-cache-entry-search-string
						search-cache-entry
					      )
				      )
				      (string new-item)
				    )
			       )
			       (return t)
			      )
			      ((and (equal search-class :\\UnKeyword)
				    (string-equal
				      (string (search-cache-entry-search-string
						search-cache-entry
					      )
				      )
				      (string new-item)
				    )
			       )
			       (return :Inverse)
			      )
			      ;;; Maybe we should handle inverses here.
			      (t nil)
			)
		  )
		 )
		 (present-p
		   (member (number-of message)
			   (search-cache-entry-numbers search-cache-entry)
			   :Test #'eq ;;; These are fixnums.
		   )
		 )
		)
		(Search-Cache-Maybe-Update
		  matched-p present-p message search-cache-entry
		)
	   )
	   nil
       )
  )
)

;;; Record flags changed as an event type.
(pushnew (string :Flags-Changed) *All-Event-Types*)

(defmethod (imap-stream-mixin :Official-Messagecnt) ()
  (process-wait "Wait for Messagecnt"
		#'(lambda (stream)
		    (numberp (send stream :Messagecnt))
		  )
		self
  )
  messagecnt
)

(defmethod (imap-stream-mixin :Maybe-Flush-Search-Cache) (key &rest args)
"Conditionally flushes a search cache.  This only happens when certain IMAP
operations happen.  Key is the keyword denoting the imap response that we have
got.  This helps us compute whether any of our caches have been invalidate, for
instance dur to a change in flags.
"
  (ecase key
    (:Fetch nil) ;;; Nothing will have changed.
    (:Store ;;; Under IMAP3 we should probably do something here {!!!}
     (destructuring-bind (thing-stored value message) args
            (if (equal :Flags thing-stored)
		(let ((classes
		        (remove-if-not
			  #'(lambda (x)
			      (or (Simple-Cache-Forces-Flush-P
				    value x message
				    flaglist
				  )
				  (Conjunction-Forces-Flush-P
				    value x message self
				  )
			      )
			    )
			    search-cache
			)
		      )
		     )
		     (Signal-Event self message :Flags-Changed)
		     (send self :Flush-Search-Cache classes)
		)
		nil
	    )
     )
    )
    (:Exists
     (if (equal messagecnt (first args))
	 nil
	 (send self :Flush-Search-Cache)
     )
    )
    (:Recent nil) ;;; The recent message is purely informational and doesn't
                  ;;; indicate any particular messages.  Some other flag will
                  ;;; always change if we have a change in the recent count.
    (:Search nil) ;;; This must be a new search anyway.
  )
)


(defmethod (imap-stream-mixin :Matching-Search-Cache-Entry)
	   (search-class search-string &optional (conjunction nil))
"Given a search class returns any matching search cache entry.  For instance,
if we're searching for FROM Acuff then the cache entry corresponding to
this would be returned if there is one.
"
  (declare (optimize (speed 3) (safety 0)))
  (if conjunction
      (loop for sce in search-cache
	    when
	      (equalp conjunction (search-cache-entry-search-conjunction sce))
	    return sce
      )
;      (find-if #'(lambda (x)
;		   (equalp conjunction
;			   (search-cache-entry-search-conjunction x)
;		   )
;		 )
;		 search-cache
;      )
      (loop for sce in search-cache
	    when (and (eq search-class (Search-Cache-Entry-Search-Class sce))
		      (equal search-string
			     (Search-Cache-Entry-Search-String sce)
		      )
		 )
	    return sce
      )
;      (find-if #'(lambda (x)
;		   (and (equal search-class
;			       (Search-Cache-Entry-Search-Class x)
;			)
;			(equal search-string
;			       (Search-Cache-Entry-Search-String x)
;			)
;		   )
;		 )
;		 search-cache
;      )
  )
)

(defmethod (Imap-stream-mixin :Message-From-Number) (number)
"Returns the message descriptor for the message numbered Number."
  (cache-entry-of number self)
)
       
(defmethod (Imap-stream-mixin :open-p) ()
"Is true if self is still open."
  (member self *all-open-imap-streams* :Test #'eq)
)

(defmethod (Imap-stream-mixin :Closed-p) ()
"Is true if self has been closed."
  (not (send self :open-p))
)

(defmethod (Imap-stream-mixin :Print-Self) (stream depth slashify)
"A simple print method for imap streams."
  (ignore depth slashify)
  (catch-error
    (let ((*print-case* :Capitalize))
         (format stream "#<~A to ~A>" (type-of self) mailbox)
    )
    nil
  )
)

(defmethod new-mail-notify ((type (eql :Notification)) from stream)
"Causes a notification for new mail for the :notification notify type."
  (tv:notify tv:selected-window
	     "~&New mail from ~A has arrived in ~A"
	     from (print-short-mailbox-name stream)
  )
  (format-scroll-window stream
    "~&New mail from ~A has arrived in ~A"
    from (print-short-mailbox-name stream)
  )
)

(defmethod new-mail-notify ((type (eql nil)) from stream)
"Does nothing when new mail arrives for the null notify type."
  (ignore from stream)
  nil
)

(defmethod new-mail-notify ((type (eql t)) from stream)
"Prints out a message after new mail for the T notify type."
  (ignore from stream)
  (format-scroll-window stream
    "~&New mail from ~A has arrived in ~A"
    from (print-short-mailbox-name stream)
  )
)

(defmethod new-mail-notify ((type (eql :beep)) from stream)
"Beeps and prints out a message after new mail for the T notify type."
  (ignore from stream)
  (beep *new-mail-notify-beep-type*)
  (format-scroll-window stream
    "~&New mail from ~A has arrived in ~A"
    from (print-short-mailbox-name stream)
  )
)

;;; Records new mail as being an event type.
(pushnew (string :New-Mail) *All-Event-Types*)

(defmethod (Imap-stream-mixin :new-messages)
	   (n &optional (notify-p *notify-of-new-messages*))
"Is called when new messages arrive on Self."
  (if (> n (array-active-length (send self :MessageArray)))
      ;;; Then we have a consistency error caused by an expunge that happened
      ;;; after a task request was spawned so do nothing.
      nil
      (progn (process-wait "Wait for recentcount"
			   #'(lambda (stream)
			       (numberp (send stream :Recentcnt))
			     )
			   self
	     )
	     (Maybe-Preempt-Envelopes
	       self (loop for i from (- n recentcnt) to n collect i)
	     )
	     (let ((from (let ((val (cache-fromtext
				      (Map-Elt (send self :MessageArray) n self)
				    )
			       )
			      )
			      (or (and (not (equal val :Unbound)) val)
				  (MAP-Fetch-From self n)
			      )
			 )
		   )
		  )
	          (New-Mail-Notify notify-p from self)
	     )
	     (and owning-window (send owning-window :New-Messages self n))
	     (loop for i from (- n recentcnt) to n
		   when (not (member :\\Seen
				     (cache-flags (Map-Elt messagearray i self))
				     :Test #'eq
			     )
			)
		   do
		   (Signal-Event self i :New-Mail)
	     )
      )
  )
)

;-------------------------------------------------------------------------------

(defun yw:imap-stream-instantiator
       (flavor connection timeout input-buffer-size number-of-input-buffers)
"Instantiates an imap stream of a specific flavor.  This is used to make
stream instantiation protocol independent.
"
  (declare (special *make-imap-stream-read-only*))
  (make-instance
    flavor
    :connection connection :timeout timeout :input-buffer-size input-buffer-size
    :number-of-input-buffers number-of-input-buffers
    :Read-Only-P (if (boundp '*make-imap-stream-read-only*)
		     *make-imap-stream-read-only*
		     nil
		 )
  )
)

(defun yw:ip-imap-stream-instantiator
       (connection timeout input-buffer-size number-of-input-buffers)
"Makes an IMAP stream under IP."
  ;;RDA: This seems the only decent place to set these.  Apologies to
  ;;modularity...
  (setf input-buffer-size *yw-tcp-input-buffer-size*
	number-of-input-buffers *yw-tcp-number-of-input-buffers*)
  (yw:imap-stream-instantiator
    'yw:ip-imap-stream connection timeout input-buffer-size
    number-of-input-buffers
  )
)

;(defun yw:ip-imap-stream-instantiator
;       (connection timeout input-buffer-size number-of-input-buffers)
;"Makes an IMAP stream under IP."
;  (yw:imap-stream-instantiator
;    'yw:ip-imap-stream connection timeout input-buffer-size
;    number-of-input-buffers
;  )
;)

(defun yw:chaos-imap-stream-instantiator
       (connection timeout input-buffer-size number-of-input-buffers)
"Makes an IMAP stream under CHAOS."
  (yw:imap-stream-instantiator
    'yw:chaos-imap-stream connection timeout input-buffer-size
    number-of-input-buffers
  )
)

(defun IMAP-Login (stream Host NewUser)
  "LOGIN to the remote IMAP server."
  (let ((user-name (user-name)))
       (if NewUser
	   (fs:delete-password-etc User-Name Host)
	   nil
       )
       (let ((MaxLoginTry 4)
	     (ip:*tcp-stream-whostate*
	      (format nil "IMAP Login to ~A" (send (net:parse-host host) :Name))
	     )
	    )
	    (loop for LoginTry from 1 to MaxLoginTry
	          for LoginReply
		  = (IMAP-Send Stream 'login
		      (multiple-value-bind (user password)
			  (fs:file-get-password
			    user-name
			    (net:parse-host host)
			  )
			(list user password)
		      )
		    )
		  do (if (eq :Ok (second LoginReply))
			 (return LoginReply)
			 ;;Else
			 (format *error-output* "~&~A~%" (third LoginReply))
			 (fs:delete-password-etc
			   (fs:file-get-password user-name (si:parse-host Host))
			   Host
			 )
		     )
		  finally (progn (IMAP-Logout Stream)
				 (format *error-output*
					 "~&Too many LOGIN failures.~%"
				 )
			  )
	    )
       )
  )
)

;;;Edited by Tom Gruber            6 Jul 90  16:53
(defun IMAP-Logout (stream)
  "LOGOUT user from the remote IMAP server."
  (without-recursion (nil)
    (let ((LogoutReply (and (streamp Stream)
                            (send stream :Open-P)
                            (neq :Closed (send Stream :Send-If-Handles :status))
                            (let ((ip:*tcp-stream-whostate*
				    (let ((host (send stream :Host)))
					 (if host
					     (format nil "IMAP Logout from ~A"
						     (send host :Name)
					     )
					     "IMAP Logout"
					 )
                                    )
                                  )
                                 )
                                 (IMAP-Send Stream 'logout)
                            )
                       )
          )
         )
         (if LogoutReply
             (if (neq :Ok (second LogoutReply))
                 (format *error-output* "~&~A~%" (third LogoutReply))
                 ;;Else - LOGOUT succeeded
                 (progn (close Stream)
                        LogoutReply
                 )
             )
             nil
         )
    )
  )
)

(defun IMAP-Noop (stream)
  "NOOP to IMAP server (prevent it from timing out and closing stream)."
  (IMAP-Send Stream 'NOOP)
)

(defun IMAP-Select (stream MailBox)
  "SELECT a MailBox on the remote Host."
  (send Stream :set-MessageCnt NIL)
  (send Stream :set-RecentCnt NIL)
  (let ((SelectReply
	  (let ((ip:*tcp-stream-whostate*
		  (format nil "Select Mailbox ~A" mailbox)
		)
	       )
	       (IMAP-Send Stream 'select (list MailBox))
	  )
	)
       )
       (if (eq :Ok (second SelectReply))
	   (progn
	     (if (search "[READ-WRITE]" (the string (third selectreply))
			 :Test #'char-equal
		 )
		 (send stream :Mark-As-Read-Write)
		 nil
	     )
	     (if (search "[READ-ONLY]" (the string (third selectreply))
			 :Test #'char-equal
		 )
		 (send stream :Mark-As-Read-Only)
		 nil
	     )
	     SelectReply
	   )
	   ;;Else
	   (progn (IMAP-Logout Stream)
		  (signal 'error "~&~A~%" (third SelectReply))
	   )
       )
  )
)

(defun float->-number-list (x)
  "Turns a number like 3.42 into (3 42)."
  (if (consp x)
      x
      (let ((*read-base* 10.))
	   (let ((stringified (format nil "~D" x)))
		(list (floor x)
		      (or (catch-error
			    (read-from-string
			      (subseq stringified
				(+ 1 (position #\. stringified :Test #'char=))
			      )
			    )
			    nil
			  )
			  0
		      )
		)
	   )
      )
  )
)

(defun number-list->-float (x)
  "Turns (3 2) into 3.2 where necessary."
  (if (numberp x)
      x
      (let ((*read-base* 10.))
	   (read-from-string (format nil "~D.~D" (first x) (second x)))
      )
  )
)

(defun version-less-p (x y)
  "Is true if a the version x is less than y."
  (Version-Greater-P y x)
)

(defun version-greater-p (x y)
  "Is true if a the version x is greater than y."
  (let ((x (Float->-Number-List x))
	(y (Float->-Number-List y))
       )
       (or (> (first x) (first y))
	   (and (equal (first x) (first y)) (> (second x) (second y)))
       )
  )
)

(defun compute-features-argument (version-to-select features supported-features)
"Given a list of features we want under a given version and the supported
features of the server returns a string that encodes the features we're going
to select.
"
  (if features
      (let ((sorted (sort supported-features
			  'version-greater-p
			  :Key #'(lambda (x)
				   (if (numberp (second x))
				       x
				       (Float->-Number-List (first x))
				   )
				 )
		    )
	    )
	   )
	   (loop for entry in sorted
		 when (>= version-to-select (first entry))
		 do (return
		      (list (first entry) (second entry))
		      (intersection (rest entry) features :Test #'string-equal)
		    )
	   )
      )
      (values version-to-select nil)
  )
)

(defun IMAP-Version (stream version-command version-to-select features)
  "Pick a server version number and any required features."
  (multiple-value-bind (version selected-features)
      (if features
	  (let ((reply
		  (let ((ip:*tcp-stream-whostate* "Supported Versions"))
		       (imap-Send Stream :Supported.Versions nil nil t)
		  )
		)
	       )
	       (if (eq :Ok (second Reply))
		   (Compute-Features-Argument version-to-select features
					      (send stream :Supported-Versions)
		   )
		   (values version-to-select nil)
	       )
	  )
	  (values version-to-select nil)
      )
    (IMAP-Version-1 stream version-command version selected-features)
  )
)
      
(defun Imap-Version-1
       (stream version-command version-to-select selected-features)
"Internal function used by IMAP-version.  Deals with selecting features
and such if the server supports such things.
"
  (let ((Reply
	  (let ((ip:*tcp-stream-whostate* "Set Version"))
	       (let ((reply (Imap-Send
			      Stream version-command
			      (list (if (equal version-command :Select.Version)
					(Float->-Number-List version-to-select)
					(number-list->-float version-to-select)
				    )
			      )
			      nil t
			    )
		     )
		    )
		    ;;; Specifically check here to allow for the old
		    ;;; version syntax.
		    (if (equal version-command :Select.Version)
			(imap-Send Stream :Select.Features selected-features
			 nil t
			)
			reply
		    )
	       )
	  )
	)
       )
       (if (eq :Ok (second Reply))
	   (progn (send stream :Set-Selected-Version version-to-select)
		  (send stream :Set-Selected-Features selected-features)
		  (format-scroll-window
		    stream "Successfully selected version ~D"
		    (number-list->-float version-to-select)
		  )
		  t
	   )
	   (progn ;(format-scroll-window stream "~&~A~%" (third Reply))
		  t
	   )
       )
  )
)

(defun IMAP-Exists (stream MsgCount)
  "Unsolicited reply has notified us of new message(s)."
  (if (not (equal msgcount (send stream :Messagecnt)))
      (send stream :Flush-Search-Cache)
      nil
  )
  (Exists Stream MsgCount))

(defun IMAP-Recent (stream MsgCount)
  "Unsolicited reply has notified us of recent message(s)."
  (send Stream :set-RecentCnt MsgCount)
)

(defun IMAP-Expunged (stream Message)
  "Unsolicited reply has notified us of an expunged message."
  (declare (special *edit-server*))
  (without-interrupts ;;; Make sure that this is atomic.
   (with-daemons-reset-and-arrested ()
    (let ((NewTotalMsgs (1- (send Stream :MessageCnt)))
	  (MessageArray (send Stream :MessageArray)))
      (send stream :Flush-Search-Cache)
      (if *kill-buffers-when-messages-expunged*
	  (kill-associated-buffers Message)
	  (rename-associated-buffers message "Expunged" :Expunged)
      )
      (loop for item in (cache-all-items message)
	    do (remove-item message item)
      )
      (loop for low-index from (cache-msg# message) to (send Stream :MessageCnt)
	    for high-index from (+ 1 low-index)
	    for cache-entry-to-replace-current-index-with
	        = (cache-entry-of high-index stream nil)
	    do (if cache-entry-to-replace-current-index-with
		   (progn (setf (cache-msg#
				  cache-entry-to-replace-current-index-with
				)
				low-index
			  )
			  (rename-associated-buffers
			    cache-entry-to-replace-current-index-with
			  )
		   )
		   nil
	       )
	       (setf (cache-entry-of low-index stream)
		     cache-entry-to-replace-current-index-with
	       )
	    when cache-entry-to-replace-current-index-with
	    collect cache-entry-to-replace-current-index-with
	    into changed
	    finally (send *edit-server* :Put-Task :IMAP-Expunge
			  (list :flush-display-caches-for changed t)
	            )
      )
      (send Stream :Set-MessageCnt NewTotalMsgs)
      (loop for i from NewTotalMsgs below (array-active-length messagearray) do
	    (setf (aref messagearray i) nil)
      )
    )
   )
  )
)


(defun imap-stream-error-handler (condition stream)
"A simple error handler for IMAP streams."
  (ignore condition)
  (tv:notify tv:selected-window "Mailstream ~A got an error - closing."
	     (send stream :Pretty-String)
  )
  (map-close stream)
  (throw 'imap-stream-close :imap-stream-error)
)

(defun IMAP-Break-Chars (ArgString)
  "Look for certain break characters that control how args are sent
 to the IMAP server."
  (loop For Char In *IMAP.Arg-Breaks*
	do (if (find Char ArgString)
	       (return T))))

;-------------------------------------------------------------------------------

(defun Check-User-Mailbox (Mailbox &optional (wrt-host *User-Host*) (error-p t))
  "Check user-specified mailbox and return with expanded host name."
  ;;;If there's no host spec, the host defaults to *User-Host*.
  (if (typep mailbox 'imap-stream-mixin)
      (Check-User-Mailbox (send mailbox :Mailbox) wrt-host error-p)
      (if (find-if #'(lambda (test)
		       (search (the string test) (the string mailbox)
			       :Test #'char-equal
		       )
		     )
		     *literal-mailbox-name-strings*
	  )
	  mailbox
	  (let ((ColonPos (and (stringp mailbox) (position #\: Mailbox))))
	       (if ColonPos
		   (if (zerop ColonPos)
		       (and error-p
			    (format *error-output*
				    "~&Null host specification is illegal~%"
			    )
		       )
		       ;;Else - we have a Host:Mailbox specification
		       (let ((MailHost
			       (net:parse-host (subseq Mailbox 0 ColonPos) t)
			     )
			     (Mailfile
			       (string-upcase (subseq Mailbox (1+ ColonPos)))
			     )
			    )
			    (if MailHost
				(if (zerop (length MailFile))
				    (and error-p
					 (format *error-output*
				     "~&Null mailbox specification is illegal~%"
					 )
				    )
				    ;;Else - everything looks ok
				    (format nil "~A:~A" MailHost MailFile)
				)
				;;Else
				(and error-p
				     (format *error-output*
					     "~&\"~A\" is an unknown host~%"
					     (subseq MailBox 0 ColonPos)
				     )
				)
			    )
		       )
		   )
		   ;;Else - we just have a Mailbox w/o a Host
		   (if (stringp mailbox)
		       (format nil "~A:~A" wrt-host (string-upcase Mailbox))
		       (let ((path (make-pathname :Host wrt-host
						  :Directory mailbox
						  :Name nil
						  :Type nil
						  :Version nil
				   )
			     )
			    )
			    (remove-spaces (send path :String-For-Printing))
		       )
		   )
	       )
	  )
      )
  )
)

(defun mailbox-and-host-from-mailbox-name (name)
"Given a mailbox name returns values for the mailbox and host."
  (declare (values name host-name))
  (let ((stream (find-if #'(lambda (str) (equal name (send str :Mailbox)))
			 *all-open-imap-streams*
		)
	)
       )
       (if stream
	   (send stream :mailbox-and-host-from-mailstream)
	   (let ((index (position #\: name)))
	        (if index
		    (values (subseq name (+ index 1)) (subseq name 0 index))
		    (values name "")
		)
	   )
       )
  )
)

(defmethod (imap-stream :superior) ()
  nil
)

(defmethod (Imap-Stream :mailbox-and-host-from-mailstream) ()
"Given a mailbox returns values for the mailbox and host."
  (if mailbox
     (let ((index (position #\: mailbox)))
	  (values (subseq mailbox (+ index 1)) (subseq mailbox 0 index))
     )
     nil
  )
)

(defun Expunge-Mailbox (MailStream &optional ExitFlg)
  "Expunge the deleted messages and update the mailbox."
  (let* ((OldTotalMsgs (send MailStream :MessageCnt)))
    (MAP-Expunge-Mailbox MailStream)
    (let ((NewTotalMsgs (send MailStream :MessageCnt)))
      (let ((DeletedMsgs (- OldTotalMsgs NewTotalMsgs)))
	(format-scroll-window mailstream "~&~:[~D~;No~*~] message~:P expunged"
		(zerop DeletedMsgs) DeletedMsgs
	)
	(when (zerop NewTotalMsgs)
	  (format-scroll-window mailstream
				"~&There are no messages left, exiting"
          )
;	  (throw 'Empty-Mailbox (MAP-Close MailStream))
	)
	(when (and (plusp DeletedMsgs)
		   (not ExitFlg))
	  (send MailStream :set-MessageArray
		(adjust-array (send MailStream :MessageArray)
			      NewTotalMsgs)))))))


(defun Renumber-MailBox (mailbox renumber-type field-name)
  (Imap-Send mailbox :Renumber (list field-name renumber-type))
)

(defun special-recent-processor (message-flags)
"A special flag processor for the recent flag used in
printing headers displays.
"
  (if (member :\\Seen Message-Flags :Test #'eq) "R" "N")
)

(defun special-seen-processor (message-flags)
"A special flag processor for the seen flag used in
printing headers displays.
"
  (if (member :\\recent Message-Flags :Test #'eq) nil " ")
)

(defun special-unseen-processor (message-flags)
"A special flag processor for the (not seen) flag used in
printing headers displays.
"
  (if (member :\\recent Message-Flags :Test #'eq) " " "U")
)

(defun map-over-sequence (function message-sequence mailbox)
 "Maps the function over all of the messages denoted by message-sequence in
 mailbox."
  (loop for item in message-sequence append
	(if (consp item)
	    (loop for i from (first item) to (second item) collect
		  (funcall function i mailbox)
	    )
	    (list (funcall function item mailbox))
	)
  )
)

(defun flush-all-caches-containing (number mailstream)
"Flushes any caches that might refer to Number in the mailbox."
  (loop for sce in (send mailstream :Search-Cache)
	when (message-present-in-search-cache-entry-p sce number)
	do (send mailstream :Set-Search-Cache
		 (remove sce (send mailstream :Search-Cache))
	   )
  )
  (let ((old-cache-entry (cache-entry-of number mailstream))
	(message-array (send mailstream :Messagearray))
       )
       (ignore old-cache-entry)
       (setf (aref message-array (- number 1)) nil)
       (Map-Elt message-array number mailstream)
       (let ((interested-windows (cache-all-items old-cache-entry)))
	    (mapcar 'window interested-windows)
       )
  )
)

(defmethod Header-Flags-string (Message-Flags stream mailstream)
"Given a list of message flags and a stream onto which to print, prints out a
special string for the message flags.  It does this according to the specified
actions in *System-Flag-Display-Sequence*.
"
  (ignore mailstream)
  (loop for (key yes no) in *System-Flag-Display-Sequence* do
	(if (member key message-flags :Test #'eq)
	    (let ((str (if yes
			   (if (stringp yes)
			       yes
			       (funcall yes message-flags)
			   )
			   nil
		       )
		  )
		 )
	         (if str (princ str stream) nil)
	    )
	    (let ((str (if no
			   (if (stringp no)
			       no
			       (funcall no message-flags)
			   )
			   nil
		       )
		  )
		 )
	         (if str (princ str stream) nil)
	     )
	)
  )
)


(Defun header-keywords-string (keywords)
"Given a set of keywords returns a string to print in the header display for
the keywords.
"
  (if Keywords
      (format nil "~{~A~^ ~}" keywords)
;      (let ((UserFlags (format NIL "~A" keywords)))
;	   (format NIL "{~A}"
;		   (subseq UserFlags 1 (1- (length UserFlags)))
;	   )
;      )
      ""
  )
)

(defun Exists (MailStream MsgCount)
  "Called when unsolicited reply returns EXISTS (i.e. new MailBox size)."
  ;;;If :MessagCnt is NIL then MsgCount is initial size, not new size
  (declare (special *yw-daemon*))
  (if *ignore-exists*
      nil
      (if (send mailstream :messagecnt)
	  (let ((deltamsgs
		  (- msgcount
		     (min (send mailstream :set-messagecnt msgcount)
			  (if (send mailstream :messagearray)
			      (array-active-length
				(send mailstream :messagearray)
			      )
			      (progn (send (any-mailer) :initialize-mailstream
                                           mailstream
                                     )
                                     0
                              )
			  )
		     )
		  )
		)
	       )
	       (cond ((minusp deltamsgs)
		      (error "~&MailBox has shrunk by ~D message~:P~%"
			     (abs deltamsgs)))
		     ((zerop deltamsgs) nil)
		     (t (send mailstream :set-messagearray
			      (adjust-array (send mailstream :messagearray)
					    msgcount))
			(send *yw-daemon* :put-task :dont-recurse-on-mailstream
			      (list :send-on mailstream :new-messages msgcount)
			)
		     )
	       )
	  )
	  ;;Else - MsgCount is the initial (not new) size
	  (send mailstream :set-messagecnt msgcount)
      )
  )
)

(defun Flag/Unflag-Message
     (MailStream messages Set/ClearFlg &optional (Message-flag :\\Flagged))
  "Set or Clear a flag for the specified message(s)."
  (if (equal Message-flag :\\Deleted)
      (Maybe-Mark-Message-As-Seen :Delete messages mailstream)
      nil
  )
  (Funcall (case Set/ClearFlg
	     (:Set   'map-set-flag)
	     (:Clear 'map-clear-flag)
	   )
	   MailStream messages Message-flag
  )
)

(defun Seen-Message (MailStream SelectedMsg)
  "Set \Seen flag for message."
  (map-set-flag MailStream SelectedMsg :\\Seen)
)

;-------------------------------------------------------------------------------

;;; Fake streams....

;-------------------------------------------------------------------------------

(defun Find-Header-Of-Type
       (headers type accessor default &optional (post-processor #'identity))
"Given a list of headers, finds a header of type Type, defaulting to Default.
If it finds one it uses the method name Accessor to get the data we want and
then post-processor to get what we really want.
"
  (let ((header
	  (find-if #'(lambda (head) (equal type (send head :Type))) headers)
	)
       )
       (if header
	   (funcall post-processor (send header accessor))
	   default
       )
  )
)

(defun post-process-address-list (address-list)
"Given a list of address objects, turn them into a list of YW addresses."
  (mapcar 'make-address-from-explorer-address address-list)
)

(defun make-address-from-explorer-address (address)
"Given an Explorer Mail:Address, return an YW address."
  (let ((new-address (intern-address (send address :Name)
				     (send address :Route)
				     (first (send address :Local-Part))
				     (send address :Domain)
				     nil
		     )
        )
       )
       (or (address-address-object new-address)
	   (setf (address-address-object new-address) address)
       )
       (setf (address-comment new-address) ;;; Hack alert.
	     (and (send address :Comments)
		  (string-trim *whitespace-chars* (send address :Comments))
	     )
       )
       new-address
  )
)

(defmethod superior-envelope-of ((me imap-stream-mixin))
  (let ((superior (send me :Superior)))
       (and superior (superior-envelope-of superior))
  )
)

(defmethod superior-envelope-of ((me envelope))
  me
)

(defmethod superior-envelope-of ((me cache))
  (if (Is-Present (cache-envelope me))
      (cache-envelope me)
      nil
  )
)

(defmethod superior-cache-of ((me imap-stream-mixin))
  (and (send me :Superior)
       (superior-cache-of (send me :Superior))
  )
)

(defmethod superior-cache-of ((me cache))
  (or (superior-cache-of (cache-mailstream me)) me)
)

(defun ultimate-superior-cache-of (x)
  (if (eq x (Superior-Cache-Of x))
      x
      (ultimate-superior-cache-of (Superior-Cache-Of x))
  )
)

(defun parse-out-header-values (header body flags cache mailstream)
"Given a message header, a message body its flag list and a cache entry
returns a whole bunch of interesting values about the message.
"
  (declare (values internaldate flags envelope rfc822length from-text subject
		   to-text sender-text list-of-headers
           )
  )
  (let ((headers (parse-headers header cache))
	(superior-envelope (and mailstream (superior-envelope-of mailstream)))
       )
       (let ((internaldate (Find-Header-Of-Type headers :Date :Body "now"))
	     (subject (or (Find-Header-Of-Type headers :Subject :Body
					       (if superior-envelope nil "")
			  )
			  (Find-Header-Of-Type
			    headers :Content-Description :Body nil
			  )
			  (if superior-envelope
			      (envelope-subject superior-envelope)
			      ""
			  )
		      )
	     )
	     (messageid (Find-Header-Of-Type headers :message-id :Body ""))
	     (from
	       (or (Find-Header-Of-Type
		     headers :From :Address-List nil 'post-process-address-list
		   )
		   (and superior-envelope (envelope-from superior-envelope))
	       )
	     )
	     (sender
	       (Find-Header-Of-Type
		 headers :sender :Address-List nil 'post-process-address-list
	       )
	     )
	     (reply-to
	       (Find-Header-Of-Type
		 headers :reply-to :Address-List nil 'post-process-address-list
	       )
	     )
	     (to
	       (or (Find-Header-Of-Type
		     headers :to :Address-List nil 'Post-Process-Address-List
		   )
		   (and superior-envelope (envelope-to superior-envelope))
	       )
	     )
	     (cc
	       (or (Find-Header-Of-Type
		     headers :cc :Address-List nil 'Post-Process-Address-List
		   )
		   (and superior-envelope (envelope-cc superior-envelope))
	       )
	     )
	     (bcc
	       (or (Find-Header-Of-Type
		     headers :bcc :Address-List nil 'Post-Process-Address-List
		   )
		   (and superior-envelope (envelope-bcc superior-envelope))
	       )
	     )
	     (in-reply-to (Find-Header-Of-Type headers :In-Reply-To :Body ""))
	     (cache-to-look-in
	       (if (Is-Present (cache-envelope cache))
		   cache
		   (superior-cache-of mailstream)
	       )
	     )
	    )
	    (let ((new-envelope
		    (make-envelope internaldate subject from sender reply-to
				   to cc bcc in-reply-to messageid
		    )
		  )
		 )
		 (values internaldate
			 flags
			 new-envelope
			 ;;; {!!!!}  This would have to be updated when
			 ;;; we got the body.
			 (+ (length header)
			    (if (Is-Present body) (length body) 0)
			 )
			 (or (and from
				  (format-name '(envelope-from) new-envelope)
			     )
			     (if cache-to-look-in
				 (format-name
				   '(envelope-from envelope-reply-to
				     envelope-sender)
				   cache-to-look-in
				 )
				 nil
			     )
			 )
			 subject
			 (or (and to
				  (format-name '(envelope-to) new-envelope)
			     )
			     (if cache-to-look-in
				 (Format-Name 'envelope-to cache-to-look-in)
				 nil
			     )
			 )
			 (or (and sender
				  (format-name '(envelope-sender) new-envelope)
			     )
			     (if cache-to-look-in
				 (Format-Name 'envelope-sender cache-to-look-in)
				 nil
			     )
			 )
			 headers
		 )
	    )
       )
  )
)


(defun set-parsed-values (cache mailstream)
"Given a cache object, fill the cache slots by parsing the header that is
already in the cache header slot.
"
  (multiple-value-bind (internaldate flags envelope rfc822length
			from-text subject to-text sender-text
		       )
      (Parse-Out-Header-Values
	(cache-rfc822header cache) (cache-rfc822text cache) nil cache
	mailstream
      )
    (setf (cache-envelope cache) envelope)
    (setf (cache-internaldate cache) internaldate)
    (setf (cache-flags cache) flags)
    (setf (cache-rfc822size cache) rfc822length)
    (setf (cache-fromtext cache) from-text)
    (setf (cache-subjecttext cache) subject)
    (setf (cache-totext cache) to-text)
    (setf (cache-sendertext cache) sender-text)
  )
)

(defmethod Parse-Message-Into-Cache-Entry
	   (cache stream (message string) &optional (start 0))
  (ignore cache stream)
  (let ((end-of-header
	  (or (search (the string *blank-line*)
		      (the string message) :Test #'char=
		      :Start2 start
	      )
	      (loop with start = 0
		    for nl-index
		        = (sys:%string-search-char
			    #\newline message start (length message)
			  )
		    when (and nl-index
			      (loop for next-index from (+ 1 nl-index)
				    below (length message)
				    when (char= #\newline
						(aref message next-index)
					 )
				    return t
				    when (not (whitespace-p
						(aref message next-index)
					      )
					 )
				    return nil
				    finally (return t)
			      )
			 )
		    return nl-index
		    do (setq start (+ nl-index 1))
	      )
	  )
	)
       )
       (let ((header
	       (if end-of-header
		   (nsubstring message start (+ 1 end-of-header))
		   ""
	       )
	     )
	     (body
	       (if end-of-header
		   (nsubstring-with-fill-pointer message (+ 2 end-of-header))
		   message
	       )
	     )
	    )
	    (values header body)
       )
  )
)

(defmethod Parse-Message-Into-Cache-Entry
	   (cache stream (message cons) &optional (start 0))
  (ignore cache stream start)
  (values (first message) (second message))
)


(defmethod Parse-Message-Into-Cache-Entry :Around
	   (cache stream (message t) &optional (start 0))
  (ignore start)
  (multiple-value-bind (header body) (clos:call-next-method)
    (setf (cache-flags cache) nil)
    (setf (cache-rfc822header cache) header)
    (setf (cache-rfc822text cache) body)
    (setf (cache-parsed-headers cache) :Unbound)
    (setf (cache-content-type cache) :text)
    (Set-Parsed-Values cache stream)
  )
)

(defmethod close-in-superior ((me cache))
  (if (typep (cache-body-parts me) 'imap-stream-mixin)
      (setf (cache-body-parts me)
	    (loop for cache-entry being the array-elements
		  of (send (cache-body-parts me) :Messagearray)
		  collect (Maybe-Reconstruct-Message-From-Cache-Entry
			    cache-entry
			  )
	    )
      )
      nil
  )
)

(defun reconstruct-header-from-header-fields (cache-entry)
  (with-output-to-string (*standard-output*)
    (loop for header in (cache-parsed-headers cache-entry)
	  for tail on (cache-parsed-headers cache-entry)
	  do (format t "~A" (send header :String))
	  when (rest tail)
	  do (terpri)
    )
  )
)

(defun maybe-reconstruct-message-from-cache-entry (cache-entry)
  (let ((header (cache-rfc822header cache-entry))
	(body   (cache-rfc822text   cache-entry))
       )
      (cond ((cache-surgically-modified-p cache-entry)
	     (list (reconstruct-header-from-header-fields cache-entry) body)
	    )
	    ((and (array-indexed-p header)
		  (array-indexed-p body)
		  (eq (sys:array-indirect-to header)
		      (sys:array-indirect-to body)
		  )
	     )
	     ;;; These should be contiguous, so maybe we should
	     ;;; indirect a new displaced string over the whole
	     ;;; thing, but as we have the body separated out anyway
	     ;;; we'll just return the list.
	     (list header body)
	    )
	    (t (list header body))
      )
  )
)

(defmethod (Fake-Stream :Before :Close) (&rest ignore)
  (Close-In-Superior superior)
)

(defmethod (Fake-Stream :Parse-Messages) ()
"Parses the messages in Mesages so that they all have suitable cache entries
and such.
"
  (loop for message in messages
	for index from 0
	for message-index = (+ 1 index)
	for cache = (map-elt messagearray message-index self)
	do (parse-message-into-cache-entry cache self message)
  )
)

(defmethod (fake-stream :Maybe-Initialize) ()
  (if (and (not mailbox) superior)
      (setq mailbox (mailbox-name-from-superior-cache-entry superior))
      nil
  )
  (if (not owning-window)
      (setq owning-window (send stream :Owning-Window))
      nil
  )
  (if (not messagearray)
      (if owning-window
	  (send owning-window :Initialize-Mailstream self)
	  (send self :Initialize-Mailstream messagecnt)
      )
      nil
  )
  (if (and (> messagecnt 0) (not (aref messagearray 0)))
      (send self :parse-messages)
      nil
  )
)

(defmethod Imap-Send ((stream fake-Stream) (Command (eql 'check))
		      &Optional ArgList (reply-type nil) (non-ok-reply-ok nil)
		     )
"A null check method for fake streams."
  (ignore ArgList reply-type non-ok-reply-ok)
  nil
)

(defmethod (Fake-Stream :mailbox-and-host-from-mailstream) ()
"Makes the stream we really point to solve the problem."
  (send stream :Mailbox-And-Host-From-Mailstream)
)

(defun read-sequence-from-string (string)
"Given a string denoting a set of messages, returns a list of the message
 numbers.
"
  (let ((*readtable* *IMAP-ReadTable*))
       (yw-read-from-string (string-append "(" string ")"))
  )
)

(defmethod imap-send ((stream Fake-Stream) (Command (eql :search))
		      &Optional Args (reply-type nil) (non-ok-reply-ok nil)
		     )
  (ignore args reply-type non-ok-reply-ok)
  (implement-search-for-local-stream stream Args)
 `(* :Ok "Search doesn't happen for this type of stream.")
)

(defmethod search-for-criterion
	   ((stream fake-stream) (criterion t)
	    string cache-entry envelope body message
	   )
  (search-for-criterion-on-local-stream
    stream criterion string cache-entry envelope body message
  )
)

(defmethod store ((stream Fake-Stream) messages (type (eql '+flags)) name)
"Stores some flags for a fake stream."
  (declare (special *edit-server*))
  (Loop for message-number in (read-sequence-from-string messages)
	for cache-entry = (cache-entry-of message-number stream)
	for old-flags = (cache-flags cache-entry)
	for new = (fs:cons-new name old-flags)
	do (setf (cache-flags cache-entry) new)
	   (send *edit-server* :Put-Task :Imap-Parse-Data-Flags
		 ;;; !!!! I'm not quite sure what to do about this T  !
		 (list :flags-changed cache-entry stream t)
	   )
;	   (imap-unsolicited-reply
;	     stream `(:* ,message ,(format nil "Store (FLAGS ~A)" new))
;	   )
  )
)

(defmethod store ((stream Fake-Stream) messages (type (eql '-flags)) name)
"UnStores some flags for a fake stream."
  (declare (special *edit-server*))
  (Loop for message-number in (read-sequence-from-string messages)
	for cache-entry = (cache-entry-of message-number stream)
	for old-flags = (cache-flags cache-entry)
	for new = (remove name old-flags)
	do (setf (cache-flags cache-entry) new)
	   (send *edit-server* :Put-Task :Imap-Parse-Data-Flags
		 ;;; !!!! I'm not quite sure what to do about this T  !
		 (list :flags-changed cache-entry stream t)
	   )
;	   (imap-unsolicited-reply
;	     stream `(:* ,message ,(format nil "Store (FLAGS ~A)" new))
;	   )
  )
)

(defmethod Imap-Send ((stream fake-Stream) (Command (eql 'store))
		      &Optional ArgList (reply-type nil) (non-ok-reply-ok nil)
		     )
"Processes the store command for Fake streams."
  (ignore reply-type non-ok-reply-ok)
  (apply 'Store stream arglist)
  (list nil :Ok)
)

(defmethod Imap-Send ((stream fake-Stream) (Command (eql 'COPY))
		      &Optional ArgList (reply-type nil) (non-ok-reply-ok nil)
		     )
"Tells you that Copy is not a supported command on Fake streams.  This is
 because the message numbers don't mean anything.
"
  (ignore arglist reply-type non-ok-reply-ok)
  (tv:notify tv:selected-window "Copy/Move not supported for mailbox ~A."
	     (send stream :Mailbox)
  )
  nil
)

(defmethod MAP-Set-Flag ((MailStream fake-stream) messages Flag)
  (let ((superior (send mailstream :Superior)))
       (if superior
	   (Map-Set-Flag (cache-mailstream superior) superior flag)
	   (clos:call-next-method)
       )
  )
)

(defmethod MAP-Clear-Flag ((MailStream fake-stream) messages Flag)
  (let ((superior (send mailstream :Superior)))
       (if superior
	   (Map-Clear-Flag (cache-mailstream superior) superior flag)
	   (clos:call-next-method)
       )
  )
)

(defmethod Header-Flags-String :Around
  (Message-Flags stream (mailstream multipart-stream))
  (let ((*system-flag-display-sequence* nil))
       (clos:call-next-method)
  )
)

(defmethod end-of-sequence-action :After
  (sequence message (mailstream fake-stream))
  (ignore sequence message)
  (loop for window in (send mailstream :Associated-Windows)
	do (send (send window :Owner) :Forget-Window window)
  )
)

