;;; -*- Mode:common-Lisp; Package:System-Internals; Base:8; Patch-File:t -*-

;;; Make-fast-hash-table is 4.8X slower.
;;; Fast-Gethash hits are 4.8X faster.
;;; Fast-Gethash missed are 36% slower.
;;; Fast-puthash is 3.8X slower.

(eval-when (compile load eval)
(defflavor basic-fast-hash-table ((never-default-p nil)) ()
  :Abstract-Flavor
  (:Documentation "All fast hash tables are built on this.")
  ;;; Note: we have no vanilla flavor so that we can maphash over tables.
  :No-Vanilla-Flavor
  (:ordered-instance-variables never-default-p)
  (:outside-accessible-instance-variables never-default-p)
  :initable-instance-variables
)
)

(defsubst make-fast-hash-table (&optional (never-default-p nil))
"Conses up a new defflavor each time.  This is fairly expensive, but we
 hope to win with gethashes.
"
  (let ((name (intern (gensym "FAST-HASH-TABLE-") 'si)))
       (gensym "G") ;;; Reset gensym.
       (eval `(progn (eval '(defflavor ,name () (basic-fast-hash-table)))
		     (make-instance ',name :never-default-p ,never-default-p)
	      )
       )
  )
)

(defmethod clos:describe ((me Basic-Fast-Hash-Table))
  (format t "~&This is a fast hash table.")
  (if (basic-fast-hash-table-never-default-p me) (format t "~%No defaulting allowed!"))
  (format t "~%It's elements are:")
  (fast-maphash #'(lambda (k v) (format t "~&~S :	~30T~S" k v)) me)
)

(defmethod clos:print-object ((fast-hash-table Basic-Fast-Hash-Table) stream)
"Define a print method for fast hash tables.  They have no vanilla flavor
so we must do this.
"
  (format stream "#<~A " 'fast-hash-table) ;;; Get prefered case.
  (if (basic-fast-hash-table-never-default-p fast-hash-table)
      (format t "No defaulting ")
      nil
  )
  (LET ((*print-base* 8.)
	(*print-radix* nil)
	(*nopoint t)
       )
       (print-fixnum (%pointer fast-hash-table) stream)
  )
  (format stream ">")
)

(defun fast-gethash (key table &optional (default-value nil))
"Fast gethash is just like gethash only it operates on tables defined using
make-fast-hash-table.
"
  (multiple-value-bind (result found-p) (send table key)
    (if found-p result default-value)
  )
)

(defun fast-hash-table-count (hash-table)
  "Like hash-table-count only for fast-hash-tables."
  (hash-table-count
    (flavor-method-hash-table (get (type-of hash-table) 'flavor))
  )
)

(defun optimize-fast-gethash (form)
"A compiler optimizer for fast-gethash that knows about how to open code
out the default value if it isn't supplied-p.
"
  (destructuring-bind (ignore key table default-value) form
    (let ((k (gensym))
	  (h (gensym))
	  (d (gensym))
	 )
	 (if default-value
	     (let ((-result- (gensym))
		   (-found-p- (gensym))
		  )
		 `(let ((,k ,key)
			(,h ,table)
			(,d ,default-value)
		       )
		       (declare (optimize (safety 0) (speed 3)))
		       (multiple-value-bind (,-result- ,-found-p-) (send ,h ,k)
			 (if ,-found-p- ,-result- ,d)
		       )
		  )
	     )
	    `(let ((,k ,key)
		   (,h ,table)
		  )
		  (declare (optimize (safety 0) (speed 3)))
		  (send ,h ,k)
	     )
	 )
    )
  )
)

;;; Install the compiler optimizer.
(compiler:defoptimizer Fast-Gethash Optimize-Fast-Gethash)

(defvar *optimize-fast-hash-tables-for-get-speed-p* nil
"When true will make fast-gethash faster still at the cost of more time
at puthash time.
"
)
(setq *optimize-fast-hash-tables-for-get-speed-p* nil)

(defun fast-maphash (function fast-hash-table)
"Map the function over the hash-table just like maphash."
  (maphash #'(lambda (key method-function mapping-table)
	       (ignore method-function mapping-table)
	       (funcall function key (Fast-Gethash key fast-hash-table))
	     )
	   (flavor-method-hash-table (get (type-of fast-hash-table) 'flavor))
  )
  fast-hash-table
)

(defun fast-maphash-return
       (function fast-hash-table &optional (return-function 'list))
"Map the function over the hash-table just like maphash-return ."
  (maphash-return
    #'(lambda (key method-function mapping-table)
	(ignore method-function mapping-table)
	(funcall function key (Fast-Gethash key fast-hash-table))
      )
      (flavor-method-hash-table (get (type-of fast-hash-table) 'flavor))
      return-function
  )
)

(defun fast-remhash (key fast-hash-table)
  "Just like remhash only works on fast-hash-tables."
  (remhash key
	   (flavor-method-hash-table (get (type-of fast-hash-table) 'flavor))
  )
)

(defun fast-clrhash (fast-hash-table)
  "Just like clrhash only works on fast-hash-tables."
  (clrhash (flavor-method-hash-table (get (type-of fast-hash-table) 'flavor)))
)

(defun fast-puthash (key value table)
"Puts values in fast-hash-tables. (not quite as fast as puthash on normal hash
 tables.
"
  (let ((fl (get (type-of table) 'flavor)))
       (or fl (ferror nil "Cannot find flavor for ~S" table))
       (let ((ht (flavor-method-hash-table fl)))
	    (let ((mt (or (getf (flavor-plist fl) 'mapping-table)
			  (setf (getf (flavor-plist fl) 'mapping-table)
				(make-array 4 :Leader-Length 3 :Type 'art-16b
					    :Initial-Element 0
				)
			  )
		      )
		  )
		 )
		 ;;; mt is a pseudo self-mapping table for the instance.
		 ;;; smash these things in to make it look right.
		 (setf (array-leader mt 0) 0)
		 (setf (array-leader mt 1) fl)
		 (setf (array-leader mt 2) fl)
		 ;;; Gross me out.  Smash a closure or function depending on the
		 ;;; tradeoff selection into the flavor method hash table
		 ;;; for the type of Table.  When we send Key messages to the
		 ;;; instance, this function/closure will be called thinking
		 ;;; it was a method and will just return the value we wanted
		 ;;; really.
		 (let ((function (if (basic-fast-hash-table-never-default-p table)
				     (if *Optimize-Fast-Hash-Tables-For-Get-Speed-P*
					 (compile nil `(lambda (&rest ignore) ',value))
					 #'(lambda (&rest ignore) value)
				     )
				     (if *Optimize-Fast-Hash-Tables-For-Get-Speed-P*
					 (compile nil
						  `(lambda (&rest ignore)
						     (values ',value t)
						   )
					 )
					 #'(lambda (&rest ignore) (values value t))
				     )
				 )
		       )
		      )
		      ;;; Note: We would like to put the value in as an extra
		      ;;; value in the hash table to use in maphash but it
		      ;;; doesn't work.
		      (puthash key (locf (second (list :Foo function))) ht mt)
		 )
		 value
	    )
       )
  )
)

(defun fast-puthash-1 (key table value)
"Just like fast-puthash only with a different arg order.  This gives the
 fast-gethash setf method the right arg evaluation order semantics.
"
  (Fast-Puthash key value table)
)

;;; Make (setf (fast-gethash .... work right.
(defsetf Fast-Gethash Fast-Puthash-1)


;;; TI code.
;;; We patch this so that when we send keys to fast-hash-table instances and we
;;; get a hash miss this function gets called, so we want to do the right thing
;;; we don't want to signal an error for instance hash failure.

(defun instance-hash-failure (op &rest args &aux (ht (%function-inside-self)) fn-location func)
  (cond
    ((/= (dont-optimize (hash-table-gc-generation-number ht)) %gc-generation-number)
     (let ((newht (with-lock ((hash-table-lock ht))
		     (funcall (dont-optimize (hash-table-rehash-function ht)) ht ()))))
	   ;; Some %POINTER's may have changed, try rehashing
       ;(set-in-instance (dont-optimize (hash-table-instance ht)) 'hash-array newht)
       (setf (instance-function self) (FOLLOW-STRUCTURE-FORWARDING newht)))))
  (cond ;((and (typep self 'Basic-Fast-Hash-Table) (not (rehash-for-gc ht)))
	; (values nil nil))
	(t (setq fn-location
 ;; In case a GC has happened or the hash table has been rehashed and forwarded,
 ;; search it again using GETHASH to find out if the operation is really there.
		 ;; GETHASH does follow forwarding, and rehashes if nec.
		 (gethash op ht )) ;(dont-optimize (hash-table-instance ht))))
	   (when     fn-location
	     ;; In case GETHASH rehashed, snap out forwarding. 
	     (setf (instance-function self)
		   (FOLLOW-STRUCTURE-FORWARDING (instance-function self))))
 ; (symeval-in-instance (dont-optimize (hash-table-instance ht)) 'hash-array))))
	   (cond
	     ((setq func (or (car fn-location)		;Found a definition
			     (flavor-default-handler (instance-flavor self))))
	      (apply func op args))
	     ;;; Patched here by JPR.
	     ((typep self 'Basic-Fast-Hash-Table) (values nil nil))
	     ((setq func (and (neq op :unclaimed-message) ;user defined handler
			      (get-handler-for self :unclaimed-message)))
	      (apply func :unclaimed-message op args))
	     (t (apply 'flavor-unclaimed-message op args))))))



;===============================================================================

(defflavor show-fast-hash-table
	   ()
	   (tv:generic-middle-button-mixin tv:inspection-data)
  (:documentation
"Shows a hash tables hash array elements.
"
  )
)

(defmethod (show-fast-hash-table :format-concisely) (stream)
"Just prints it out."
  (format stream "~S" tv:data)
)

(defmethod (show-fast-hash-table :generate-item) (&aux result)
"Makes the inspector items for a hash-table."
  (push '("") result)
  (push '("Fast-Hash-Table Elements") result)
  (push '("") result)
  (loop for element
	in (tv:Make-Window-Items-For-Hash-Table
	     (flavor-method-hash-table (get (type-of tv:data) 'flavor))
	     #'(lambda (val)
		 (if (locativep val)
		     (funcall (contents val) nil)
		     nil
		 )
	       )
	       nil
	   )
	do (push element result)
  )
  (push '("") result)
  (values (nreverse result)
	 `(:font fonts:hl12bi :string ,(format nil "~s" tv:data))
  )
)

(defmethod (show-fast-hash-table :help) ()
"Gives help when you middle button on a defstruct."
  (format nil "
The inspection pane you just selected is currently displaying the defstruct
instance ~S simply as a defstruct instance.  Mousing L2 on it should show it
to you in some other way.
"
	  tv:data))

(tv:Defperspective :fast-hash-table (x show-x)
  :show-x-type-for-perspective Show-fast-hash-table
  :This-Perspective-Applicable-Function
    (and (typep x 'basic-fast-hash-table)
	 (not (typep show-x 'show-fast-hash-table))
    )
  :menu-item-name "Fast Hash Table elements"
  :New-Inspect-Function (tv:allocate-data 'Show-Fast-Hash-Table x)
  :Priority 20
)

;-------------------------------------------------------------------------------


(export '(basic-fast-hash-table make-fast-hash-table fast-gethash
	  fast-maphash fast-maphash-return fast-remhash fast-clrhash
	  fast-puthash fast-hash-table-count
	  *optimize-fast-hash-tables-for-get-speed-p*
	 )
	'ticl
)

;-------------------------------------------------------------------------------

