;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

(defstruct IMAP.Lock
  "IMAP Send/Receive lock to insure synchronous communication."
  ;;;Can't send again until received matching tag from previous send.
  ;;;Uses DEFSTRUCT so WITH-LOCK can do LOCF on it.
  (Location NIL))

(defun make-an-imap-lock ()
  (declare (optimize (speed 3) (safety 0)))
  (declare (special *all-locks*))
  (let ((new-lock (Make-Imap.Lock)))
       (push new-lock *all-locks*)
       new-lock
  )
)

(defun reset-all-locks ()
  (loop for lock in *all-locks* do (setf (Imap.Lock-Location lock) nil))
)
	
(defstruct-safe (Cache :Named)
  "The components of each Message cache entry."
  (Msg# :Unbound)			; The message number
  (InternalDate :Unbound)		; The message date
  (Flags :Unbound)			; Any flags on that message
  (Envelope :Unbound)			; The envelope object for that message
  (RFC822Size :Unbound)			; The size of the message
  (FromText :Unbound)			; The from text for the message
  (SubjectText :Unbound)		; The subject of the message
  (RFC822Header :Unbound)		; The header of the message
  (RFC822Text :Unbound)			; The body of the message
  (RFC822-All-text :Unbound)		; All of the text of the message
  (totext :Unbound)			; The to field of the message
  (associated-zmacs-buffers nil)	; The buffers that are currently reading
  (sendertext :Unbound)			; The sender of the message
  (parsed-headers :Unbound)             ; The headers parsed a la ZMail
  (body-parts :Unbound)                 ; RFC-822+ parts.
  (content-type *default-content-type*) ; Nil means nothing needs to be done.
  (content-subtype  *default-content-subtype*)
  (content-type-parameters nil)
  (content-transfer-encoding nil)
  (content-encoded-p nil)               ; True for non-trivial transfer-encoding
  (content-id :Unbound)
  (old-content-transfer-encoding :Unbound) ; What our encoding was before decode
  (surgically-modified-p nil)           ; True when we hack on the body etc.
  (header-display-string nil)           ; The header-display-string for
                                        ; the message.
  (all-items nil)		        ; A list of window items for message
  mailstream                            ; The stream pointing to the mailbox
  (flags-used-for-display nil)          ; The flag list used to compute
                                        ; header-display-string
  (ready-to-recompute nil)	        ; True when we must recompute display
  (selected-p nil)		        ; Is true if the message is selected
  (decoded-body :Unbound)		; For richtext etc.
)

(defun (:Property cache named-structure-invoke)
       (message-name thing &Rest arguments)
"A print method for cache entries."
  (ecase message-name
    (:Print-Self
     (Print-Self thing (first arguments) (second arguments) (third arguments))
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
  )
)

(defstruct-safe (Envelope :Named
			 (:Constructor make-envelope
				       (date subject from sender reply-to
					to cc bcc In-Reply-to messageid
				       )
		         )
		)
  "Used as a template to access the Cache-Envelope slot."
  Date
  Subject
  From
  Sender
  Reply-to
  To
  Cc
  Bcc
  In-Reply-to
  MessageId
  $CC
  $From
  $Subject
  $To
  canonicalized-p
)

(defun non-null-string (x)
  (if (equal x "")
      nil
      x
  )
)

(defun (:Property envelope named-structure-invoke)
       (message-name thing &Rest arguments)
"A print method for envelopes."
  (ecase message-name
    (:Print-Self
      (format (first arguments) "#<Envelope ~A>"
	      (or (Non-Null-String (envelope-messageid thing))
		  (Non-Null-String (envelope-from thing))
		  (Non-Null-String (envelope-sender thing))
		  (Non-Null-String (envelope-reply-to thing))
		  "????"
	      )
      )
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
  )
)

(defstruct-safe (Address :named)
  "Used as a template to access the Message recipient slots."
  ;;;The recipients are slots like To, From, Cc, etc.
  PersonalName
  RouteList
  MailBox
  Host
  address-object
  comment
)

(defun (:Property Address named-structure-invoke)
       (message-name thing &Rest arguments)
"A print method for addresses."
  (ecase message-name
    (:Print-Self
      (format (first arguments) "#<Address ~A@~A>" (Address-Mailbox thing)
	      (Address-Host thing)
      )
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
  )
)

(defstruct-safe (top-level-blip :Named)
"A structure to represent the thing that controls mouse blips on summary
windows at the top level (not as args to a command).
"
  mouse-char		; The mouse char that this record refers to
  starter-method	; A method to call to start off the processing
  task-daemon-method	; The method that is invoked by the task server
  doc-string		; Who-line doc string for this mouse combination.
  mouse-key		; Keyword for the mouse char
  applicable-if		; Predicate to determine whether this is applicable
  undo-method		; A method to undo things for mouse-hold
)

(defstruct-safe (background-blip :Named)
"A structure to represent the thing that controls mouse blips on summary
windows at the top level that blip on the background of the
window (not an item).
"
  mouse-char		; The mouse char that this record refers to
  method		; The method that is invoked to process the blip
  doc-string		; Who-line doc string for this mouse combination.
  mouse-key		; Keyword for the mouse char
  applicable-if		; Predicate to determine whether this is applicable
  undo-method		; A method to undo things for mouse-hold
)

(defstruct-safe (message-highlight-spec :Named)
"A structure that specifies how to highlight a certain type of message
in a summary window.
"
  name				; The name of th highlight type
  (stipple tv:100%-black)	; The stipple pattern
  (alu tv:alu-xor)		; The alu to use
  start				; Start position (see size-from-spec)
  end				; End position (see size-from-spec)
)


(defstruct-safe (search-cache-entry :Named)
"Entries in the search cache."
  search-class		; The class of things searched for e.g. FROM
  search-string		; The string used in the search e.g. "Acuff"
  numbers		; The message numbers that matched the search
  search-conjunction    ; If there's no class then the conjunction is a list
                        ; of the following form: ((:sequence-from "RICE")...)
  mailstream            ; backpointer to mailbox.
  mask                  ; bit mask mapping numbers to presence in cache
)

(defsubst message-present-in-search-cache-entry-p (sce message-number)
  (not (zerop (aref (search-cache-entry-mask sce) message-number)))
)

(defun (:Property search-cache-entry named-structure-invoke)
       (message-name thing &Rest arguments)
"A print method for search cache entries."
  (ecase message-name
    (:Print-Self
     (catch-error
      (flet ((numbers ()
	      (mouse-sensitively-colonify-numbers
		(Search-Cache-Entry-Mailstream thing)
		(first arguments) (Search-Cache-Entry-Numbers thing) t
	      )
	     )
	    )
	    (if (Search-Cache-Entry-Search-Conjunction thing)
		(format (first arguments) "#<Search-Cache ~{~A~^, ~}"
			(Search-Cache-Entry-Search-Conjunction thing)
		)
		(if (Search-Cache-Entry-Search-String thing)
		    (format (first arguments) "#<Search-Cache ~A, ~A"
			    (Search-Cache-Entry-Search-Class thing)
			    (Search-Cache-Entry-Search-String thing)
		    )
		    (format (first arguments) "#<Search-Cache ~A"
			    (Search-Cache-Entry-Search-Class thing)
		    )
		)
	    )
	    (format (first arguments) ": ")
	    (numbers)
	    (format (first arguments) ">")
      )
      nil
     )
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
  )
)

(defstruct-safe (task-queue-entry :Named)
"An entry in the task queue.  Knows who put the entry there, why and how to
do it.
"
  task		; The task to perform
  originator	; The originator of the task
  purpose	; The purpose of the task
)

(defun (:Property task-queue-entry named-structure-invoke)
       (message-name thing &Rest arguments)
"A print method for task queue entries."
  (ecase message-name
    (:Print-Self
      (format (first arguments) "#<Task ~A, ~: ~>"
	      (Task-Queue-Entry-Purpose thing)
	      (list (Task-Queue-Entry-Originator thing) nil)
	      (list (Task-Queue-Entry-Task thing) t)
      )
    )
    (:Which-Operations '(:Which-Operations :Print-Self))
  )
)

;-------------------------------------------------------------------------------

(defstruct-safe (event :Named)
"Events in the rule system"
  mailbox   ;;; The source mailbox that caused the event.
  message   ;;; The message that triggered the event.
  type      ;;; The type of the message.
)

(defstruct-safe (agenda-entry :Named)
"An entry in the rule system's agenda."
  rule-set  ;;; The rule set that is to be invoked on the event.
  Event     ;;; The event that caused this agenda entry.
)

;-------------------------------------------------------------------------------
