;;; -*- Mode:Common-Lisp; Package:W; Base:10; Patch-File:T -*-

;;; This software developed by:
;;;	Rich Acuff
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in Jun '85-Dec '87.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15
;;;	DARPA Grant N00039-86- C-0033

;;;----------------------------------------------------------------------
;;; Portions of this code indicated by the comment line:
;;;	;TI Code:
;;; are derived from code licensed from Texas Instruments Inc.
;;; KSL's changes are noted by comment lines beginning with:
;;;	;RDA:
;;;	;JPR:
;;;  The following restrictions apply to the TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;;  Most of this code was developed in smaller pieces between June 1985 and
;;;  April 1987, and consolidated to this file for conversion to Explorer
;;;  release 3.0 in April 1987.

#|
   This file contains many pieces of generally useful window system related
utilities.  With a few execptions, they don't affect the environment
directly.  Those that can are under the control of user settable variables.
In general, things later in the file might use thing earlier in the file.
The following facilities are defined:

- WINDOW-SYSTEM-UTILITIES: A set of utilities, including:
  -- TEMP-PIXEL-ARRAYS, a resource of pixel arrays
  -- MAKE-BITMAP, makes a 1 bit pixel array, rounding up the width to a
     multiple of 32, as the Explorer requires (now moved to KSL-patches)
  -- BITMAP-WIDTH and BITMAP-HEIGHT, (now moved to KSL-Patches)
  -- CLEAR-BITMAP which zeros a bitmap (now moved to KSL-Patches).
  -- Functions to read and write bitmaps written from InterLisp-D
  -- WITH-MOUSE-SHEET to execute some code with MOUSE-SHEET temporarily
     set to some window.
  -- LEFT-DOWN?, MIDDLE-DOWN?, and RIGHT-DOWN? for testing mouse button
     states.

- A definition for <Term> P so that it can be used to easily change the
  priority of the process associated with the selected window.

- DEEXPOSED-MOUSE: Redefinitions of some system functions so that
  :DEEXPOSED-MOUSE-BUTTONS and :DEEXPOSED-WHO-LINE-DOCUMENTATION-STRING
  get called if a window is deexposed, under the mouse, and handles
  them.

- SNAPSHOT-WINDOWS: "Snapshots" of the screen--the user specifies a
  rectangle, and a new  window is created with that part of the screen
  image in it.  SNAP-REGION  or the window oriented system menu are the
  usual interfaces.

- BACKGROUNDS: A "background" or "curtain" window, providing an easy way
  to hide temporarily unwanted windows if the user is employing a
  desk-top window style.  See doc strings for *WINDOWS-TO-LEAVE-EXPOSED*,
  CREATE-BACKGROUND, SET-BACKGROUND, and CHOOSE-BACKGROUND.

- TRANSPARENT-WINDOW: the flavor W:TRANSPARENT-WINDOW-MIXIN which allows
  the image underneath a window to bleed through, giving the illusion of
  non-rectangular windows.

- WINDOW-MANAGER-SYSTEM-MENU: A window manager oriented system menu,
  under the control of  W:*USE-WINDOW-MANAGER-SYSTEM-MENU*

- ANALOG-CLOCK: An analog clock accessable via the W:ANALOG-CLOCK
  function

- VERTICALLY-ORDERED-MENU-COLUMNS: a mixin flavor which allows the ordering
  of items in multi column menus to be vertical, like a phone book, as opposed
  to the default, which is horizontal.

- CHOICE-FACILITY-ENHANCEMENTS: Two macros W:Assign-Using-Menu and
  W:Values-Using-Menu.  These provide a more friendly interface to the
  w:choose-variable-values function.  They do not require the use of
  Special variables.  The former Setfs the value denoting expressions
  with the new values, the latter returns multiple values.

- RUBBER-BAND-RECTANGLES: If *USE-RUBBER-BANDS?* is non-NIL then
  rectangles (eg. for new window shapes) are specified with a mouse
  following box.  (NOTE: This is now a system patch)

- WINDOW-DRAGGING: If W:*DRAGGING-ENABLED?* is non-NIL most windows that
  don't already have some action associated with the Middle mouse button
  can be "dragged".  "Grab" a window by pressing and holding the Middle
  mouse button, move the ghost box to the desired new position, and
  release.  This works with LISP-LISTENERs, SINGLE-WINDOW-VT100s, Zmacs
  frames (in the mini-buffer area), ANALOG-CLOCK, Snap windows,
  non-sensative areas of Inspectors, and  Window Debuggers, and
  GRAPHICAL-VALUE-MONITORs.

- The variable w:*MOUSE-WARP-GRATUITOUSLY?* which when non-NIL causes the
  system to warp the mouse less (eg. after menu selection or during menu
  pop-up).

- The function ticl:WUM , which is a synonym for tv:window-under-mouse.

- The macros tv:with-window-ops-on-bit-array
  and tv:without-window-ops-on-bit-array  Execute a BODY with a SHEET
  pretending to be deexposed so that any window ops happen on its bit
  save array, not directly onto the screen.  This can be much faster
  than normal output because the screen memory into which the normal
  screen array is mapped is very slow wrt the machine's normal memory.
|#

;;;=============================================================================

;;; Utilities


;;; Bitmap manipulation functions
(defun clear-bitmap (bm)
  "Sets all the bits in the bitmap bm to zero."
  (check-arg bm (typep bm '(array bit (* *))) "a bitmap")
  (bitblt alu-setz
	  (bitmap-width bm)
	  (bitmap-height bm)
	  bm 0 0 bm 0 0)
  )

(defun make-bitmap (width height &rest make-array-options)
  "Makes a bitmap (1 bit pixel array).  WIDTH is rounded up to a multiple
   of 32 if necessary.  MAKE-ARRAY-OPTIONS is passed to MAKE-ARRAY."
  (apply #'make-array (list height (* (ceiling width 32) 32))
	 :element-type 'bit
	 make-array-options)
  )

(defun bitmap-width (bitmap)
  "The width of BITMAP."
  (array-dimension bitmap 1)
  )


(defun bitmap-height (bitmap)
  "The height of BITMAP."
  (array-dimension bitmap 0)
  )

(defresource temp-pixel-arrays (width height type)
  :constructor (make-array (list height (* (ceiling width 32) 32)) :type type)
  :matcher (and (>= (array-dimension object 1) width)
		(>= (array-dimension object 0) height))
  )


(defmacro with-mouse-sheet ((sheet) &body body)
  "Executes BODY with MOUSE-SHEET set to SHEET"
  `(let ((.old-mouse-sheet. mouse-sheet))
     (unwind-protect
	 (progn
	   (unless (eq mouse-sheet ,sheet) (mouse-set-sheet ,sheet))
	   ,@body
	   )
       (unless (eq .old-mouse-sheet. mouse-sheet)
	 (mouse-set-sheet .old-mouse-sheet.))
       )
     )
  )

(defun left-down? (buttons)
  "Non-NIL if the left button is down in the button mask BUTTONS."
  (logtest 1 buttons))

(defun middle-down? (buttons)
  "Non-NIL if the middle button is down in the button mask BUTTONS."
  (logtest 2 buttons))

(defun right-down? (buttons)
  "Non-NIL if the right button is down in the button mask BUTTONS."
  (logtest 4 buttons))

;;;----------------------------------------------------------------------

;;; InterLisp bitmaps

(defun read-interlisp-bm (file &optional bm)
  "Reads a bitmap written by InterLisp-D from FILE.  The first 16 bits of
   FILE is the width of the bitmap, the second 16 bits is the height,
   followed by raster lines, padded to fit in an integral number of 16 bit
   chunks.  The bits in the 16 bit chunks are reversed from what the
   Explorer would like, so we take a lot of time moving them.  FILE can be a
   stream open for reading, or something acceptable to OPEN.  If the latter,
   the file is left closed.  BM must be a pixel array large enough to hold
   the bitmap being read, or NIL, in which case a new one is created."
  (if (streamp file)
      (read-interlisp-bm-from-stream file bm)
      (with-open-file
	(s file :direction :input :characters nil :byte-size 16)
	(read-interlisp-bm-from-stream s bm)
	)
      )
  )

(defun read-interlisp-bm-from-stream (stream &optional bm)
  "Reads a bitmap from STREAM.  See READ-INTERLISP-BM for format details.  BM is
   either a pixel array to put the bits into, or NIL, in which case a pixel
   array is created."
  (let* ((width (tyi stream))
	 (height (tyi stream))
	 (row (make-array (list 1 (* (ceiling width 32) 32))
			  :element-type 'bit
			  :initial-element 0))
	 (row16b (make-array (ceiling width 16)
			     :element-type '(unsigned-byte 16)
			     :displaced-to row))
	 )
    (if bm
	;; validity check on BM
	(unless (and (typep bm '(array bit (* *)))
		     (>= (bitmap-height bm) height)
		     (>= (bitmap-width bm) width)
		     (zerop (rem (bitmap-width bm) 32))
		     )
	  (error nil "The object ~A cannot be used to store a bitmap that is ~D by ~D" bm width height)
	  )
	;; no BM given, make one
	(setf bm (make-array (list height (* (ceiling width 32) 32))
			     :element-type 'bit
			     :initial-element 0)
	      )
	)
    (dotimes (y height)
      ;; read a ROW
      (send stream :string-in nil row16b)
      ;; swap bits
      (dotimes (i (length row16b))
	(setf (aref row16b i) (reverse-16b (aref row16b i)))
	)
      ;; move to BM
      (bitblt alu-seta width 1 row 0 0 bm 0 y)
      )
    bm
    )
  )

(defun write-interlisp-bm (file bm &optional width height)
  "Write BM to FILE in a format InterLisp-D can read.  See
   READ-INTERLISP-BM for format details.  WIDTH and HEIGHT default to
   the width and height of BM.  FILE is as in READ-INTERLISP-BM."
  (if (streamp file)
      (write-interlisp-bm-to-stream file bm width height)
      (with-open-file
	(s file :direction :output :characters nil :byte-size 16)
	(write-interlisp-bm-to-stream s bm width height)
	)
      )
  )

(defun write-interlisp-bm-to-stream (stream bm width height)
  "Does the work of WRITE-INTERLISP-BM"
  (check-arg bm (typep bm '(array bit (* *))) "a bitmap")
  (if width
      (unless (and (numberp width) (<= width (bitmap-width bm)))
	(error nil
	       "The width ~D is too big for the bitmap ~A" width bm))
      (setf width (bitmap-width bm))
      )
  (if height
      (unless (and (numberp height) (<= height (bitmap-height bm)))
	(error nil
	       "The height ~D is too big for the bitmap ~A" height bm))
      (setf height (bitmap-height bm))
      )
  (write-char width stream)
  (write-char height stream)
  (let* ((row (make-array (list height (* (ceiling width 32) 32))
			  :element-type 'bit
			  :initial-element 0))
	   ;;  This must be a mult of 16 instead of 32
	   (row16b (make-array (ceiling width 16)
			       :element-type '(unsigned-byte 16)
			       :displaced-to row))
	   )
    (dotimes (y height)
      (bitblt alu-seta width 1 bm 0 y row 0 0)
      (dotimes (i (length row16b))
	(setf (aref row16b i) (reverse-16b (aref row16b i))))
      (send stream :string-out row16b)
      )
    )
  )

(eval-when (compile load eval)
  (defun reverse-8b (n)
    "N is an 8 bit number.  Return a number with the same 16 bits, but in
   the opposite order."
    (let ((k 0))
      (dotimes (i 8)
	(setf k (ash k 1))
	(incf k (ldb (byte 1 0) n))
	(setf n (ash n -1))
	)
      k
      )
    )
  )

(defconstant Reverse-8-Bit-Table
	     (let ((table (make-array 256
				      :element-type '(unsigned-byte 16))))
	       (dotimes (i 256)
		 (setf (aref table i) (reverse-8b i))
		 )
	       table
	       )
  "Table mapping the 8 bit number N to a number with the same bits in
the opposite order."
  )

(defun reverse-16b (n)
  "N is a 16 bit number.  Return a number with the same 16 bits, but in
   the opposite order."
  (dpb (aref Reverse-8-Bit-Table (ldb (byte 8 0) n))
       (byte 8 8)
       (aref Reverse-8-Bit-Table (ldb (byte 8 8) n)))
  ) 

;;;-----------------------------------------------------------------------------
  

(provide 'window-system-utilities)

;;;=============================================================================

;;;  Easy <Term> key way of adjusting priorities.

(defun term-adjust-priority (arg)
  (unless arg (setf arg 1))
  (when tv:selected-window
    (let ((p (send tv:selected-window :process)))
      (when p
	(send p :set-priority (+ arg (send p :priority)))
	(tv:notify nil 
		   "~A Priority = ~D"
		   (send p :name)
		   (send p :priority))
	)
      )
    )
  )

(tv:add-terminal-key #\p #'term-adjust-priority 
    ''("Increase selected process' priority by arg."
       "   No arg ==> 1.  0 arg => print priority on selected window")
    :KEYBOARD-PROCESS		 ;This so we run at high priority
    ) 

;;;=============================================================================

;;; Deexposed mouse handling

(if (intersection '(:release-4 :Release-5) *features*)
    ;; TI Code from SYS:WINDOW;MOUSE.LISP#22
    TV: ;;; Updated by JPR for Rel 5. and for Rel 6 on 7/14/89
    (DEFUN MOUSE-DEFAULT-HANDLER (WINDOW &OPTIONAL SCROLL-BAR-FLAG
				  &AUX
				  (WINDOW-X-OFFSET 0) (WINDOW-Y-OFFSET 0)
				  WINDOW-X WINDOW-Y)
      (unless (SYMBOLP WINDOW)
	(MULTIPLE-VALUE-SETQ (WINDOW-X-OFFSET WINDOW-Y-OFFSET)
			     (SHEET-CALCULATE-OFFSETS SELF MOUSE-SHEET)))
    
      ;; Be careful not to do the :update method when the who line documentation window
      ;; isn't there (which is the case during a window system build).
      (WHEN (AND (BOUNDP 'WHO-LINE-DOCUMENTATION-WINDOW) WHO-LINE-DOCUMENTATION-WINDOW)
	;; Update who-line when entering new handlers.
	(send who-line-documentation-window :update))
      
      (DO ((DX) (DY) (BU) (BD) (HAND) (X) (Y)
	   (OLD-OWNER WINDOW-OWNING-MOUSE WINDOW-OWNING-MOUSE)
	   (LEFT-OFFSET 0)
	   (right-offset 0)
	   (WAIT-FLAG NIL T))
	  (MOUSE-RECONSIDER)
	;; Wait until the mouse moves
	(MULTIPLE-VALUE-SETQ (DX DY BD BU X Y) (MOUSE-INPUT WAIT-FLAG))
	;; If asked to reconsider, do so immediately.
	;; Don't bother updating blinker since it is likely to change soon, and
	;; in any case we are going to be called back shortly.
	(when MOUSE-RECONSIDER (RETURN NIL))
	;; Update console-idle time when buttons pushed
	(unless (ZEROP BD) (SETQ KBD-LAST-ACTIVITY-TIME (TIME)))
	;; Approximate speed of the mouse in inches per second
	(SETQ MOUSE-SPEED (/ (ISQRT (+ (* MOUSE-X-SPEED MOUSE-X-SPEED)
				       (* MOUSE-Y-SPEED MOUSE-Y-SPEED)))
			     100.0s0))
	;; If the mouse is moving incredibly fast, flash up something to
	;; help the user find it.  Thus if you can't find the mouse, you must whip it.
	(when (> MOUSE-SPEED MOUSE-FAST-MOTION-SPEED)
	  (if mouse-fast-track-bitmap-mouse-p
	      (draw-bitmap-mouse-cursor mouse-speed)
	      ;;ELSE
	      (draw-mouse-fast-motion-cursor)))
    
	(SETQ WINDOW-X (- X WINDOW-X-OFFSET)        ; X offset of mouse within window
	      WINDOW-Y (- Y WINDOW-Y-OFFSET))       ; Y offset of mouse within window
    
	;; Consider entering the scroll bar.  [Perhaps this should be changed so that
	;; it is in the move-handler rather than here.  Problem with that is LEFT-OFFSET.]
	;; If there is a scroll bar and we are entering it, activate it.
	;; However, the mouse must move at least a certain distance past the edge
	;; of the window in order to qualify for scrolling (this is set by
	;; the SCROLL-BAR-RELUCTANCE variable in the window).  Before entering
	;; scroll bar, send a :MOUSE-MOVES message in order to let the window know
	;; what's happening.
    
	;; LEFT-OFFSET is how far out the left side of the window the mouse has moved,
	;; or 0 if the mouse is inside the window.
	;; If the window is at the left edge of the screen, MOUSE-X will not itself
	;; move out the left edge of the window, but DX will.  When the mouse reaches
	;; the left edge of the window, accumulate leftward motion into LEFT-OFFSET.  
	;; RIGHT-OFFSET does the same thing for when the scroll-bar is on the right.
	(COND ((< WINDOW-X 0)	;; may 9-7-88 was <= see HACK below
	       (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      1)))	   		;First time, don't use all of DX
	      ((and (typep window 'sheet)
		    (>= window-x (sheet-width window))
		    (send window :send-if-handles :scroll-bar-on-right))
	       (setq right-offset (if (plusp right-offset)
				      (max (+ right-offset dx) 1)
				      1)))
	      ;; [may] 9-7-88
	      ;; HACK : the left edge is a special case. Its possible to hang up the
	      ;;         the system when scroll bar on right and mouse on left edge
	      ;;         looping in :handle-mouse and here with wait-flag always nil
	      ;;         which allows #'mouse-input to be process greedy! We COULD always set
	      ;;         left-offset to 0 in/of 1 BUT left edge of scroll-bar would not make
	      ;;         double-arrow for scrolling sometimes.
	      ;; cases : 1. left-offset=0 AND no scroll-bar ( fixed lying :enable-scrolling-p )
	      ;;         2. left-offset=0 AND scroll-bar on right - force left-offset = 0 to check right-offset
	      ;;         3. left-offset=0 AND left scroll-bar - this case only happens SOMETIMES as scroll-bar is being
	      ;;            added. The other two cases happen EVERY time.
	      ((= window-x 0)
	       ;; we are called with scroll-bar-flag indicating a scroll bar but if
	       ;; scroll bar is on RIGHT we really should check right-offset FIRST in COND below
	       (COND ((and (typep window 'sheet)
			   (send window :send-if-handles :scroll-bar-on-right)) ;; scroll bar is on RIGHT
		      (SETQ left-offset 0)) ;; force skipping of left-offset in COND below
		     (t
		      (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      0))))) ;First time, don't use all of DX [may 9-13-88 was 1 - fix case 3]
	      (t
	       (SETQ LEFT-OFFSET 0
		     right-offset 0)))
	(COND ((or old-owner window-owning-mouse)) ; These disable scroll-bar.
	      ((EQ SCROLL-BAR-FLAG 'T)
	       (cond ((AND SCROLL-BAR-MAX-SPEED
			  (> MOUSE-SPEED SCROLL-BAR-MAX-SPEED)))
		     ((PLUSP LEFT-OFFSET)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar))
			     (return nil))
			    ((> LEFT-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X 0))))	;Don't escape the window yet
		     ((plusp right-offset)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar-on-right))
			     (return nil))
			    ((> right-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X (1- (sheet-width window))))))))	;Don't escape the window yet
	      
	      ((EQ SCROLL-BAR-FLAG :IN)
	       ;; We are in the scroll bar.  Moving the mouse faster than the exit
	       ;; speed, or moving it into the interior of the window more than 40. 
	       ;; will escape.  
	       (COND ((AND SCROLL-BAR-MAX-EXIT-SPEED
			   (> MOUSE-SPEED SCROLL-BAR-MAX-EXIT-SPEED))
		      ;; Moving like a bat, let the guy out of the scroll bar
		      (RETURN NIL))
		     ((MINUSP WINDOW-X)		;Trying to go out left, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X 0)
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X WINDOW-X-OFFSET))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((>= window-x (sheet-width window))	;Trying to go out right, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X (- (sheet-width window) 8.))  ; hack
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X (+ WINDOW-X-OFFSET window-x)))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((and (> WINDOW-X (+ 40. (sheet-inside-left window)))	; Escape scroll-bar
			   (< window-x (- (sheet-inside-right window) 40.)))
		      (RETURN NIL)))))
	;; Update the position of the mouse before checking for button clicks, so
	;; that button clicks get processed with knowledge of where the mouse
	;; was when the button was first clicked.  The arguments to the move handler
	;; may be where the mouse was when the button was clicked, whereas the
	;; mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.   
	(SETQ MOUSE-WARP NIL)
	(SEND (IF (SYMBOLP WINDOW)
		     'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
		 :mouse-moves WINDOW-X WINDOW-Y)
	;; Check for all the ways of losing control of the mouse.
	(IF (COND ;; The move handler may have decided to warp the mouse so that it will not
		  ;; move out of the window.  This test is a crock but should work.
		  (MOUSE-WARP NIL)
		  ;; Check for mouse ceasing to be grabbed.
		  ((EQ WINDOW T)
		   (NEQ WINDOW-OWNING-MOUSE T))
		  ;; Check for window becoming grabbed.
		  ((EQ WINDOW-OWNING-MOUSE T)
		   (NEQ WINDOW T))
		  ;; Check for some other window (not above this one) being greedy.
		  (WINDOW-OWNING-MOUSE
		   (NOT (SHEET-ME-OR-MY-KID-P WINDOW WINDOW-OWNING-MOUSE)))
		  ;; Check for moving into a window when not in any
		  ((NULL WINDOW)
		   (WINDOW-OWNING-MOUSE X Y))
		  ;; Check for leaving the boundaries of the current window
		  ;; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning
		  ((NOT (AND (SHEET-EXPOSED-P WINDOW)
			     (>= WINDOW-X 0)
			     (<  WINDOW-X (SHEET-WIDTH WINDOW))
			     (>= WINDOW-Y 0)
			     (<  WINDOW-Y (SHEET-HEIGHT WINDOW))))
		   WAIT-FLAG)
		  ;; Check for moving into an inferior of the current window
		  ((NEQ (LOWEST-SHEET-UNDER-POINT WINDOW WINDOW-X WINDOW-Y
						  :HANDLE-MOUSE :EXPOSED)
			WINDOW)
		   T))
	    ;; Return to overseer, saving any pending button click.
	    (RETURN (MOUSE-DEFER-BUTTONS BU BD)))
	;; Now process button pushes if mouse is not seized.
	(COND ((OR (ZEROP BD) (EQ WINDOW T) (EQ OLD-OWNER T)))	;; may 9-7-88 
	      ;; If over an exposed window, send it the button-push
	      (WINDOW (SEND WINDOW :mouse-buttons BD WINDOW-X WINDOW-Y))
	      ;;RDA:- call :DEEXPOSED-MOUSE-BUTTONS if we're over a
	      ;; deexposed window that has a handler for it
	      ((setq hand (window-under-mouse :deexposed-mouse-buttons :active x y))
	       (multiple-value-bind (xoff yoff)
		   (sheet-calculate-offsets hand mouse-sheet)
		 (send hand :deexposed-mouse-buttons bd (- x xoff) (- y yoff))))
	      ;; Default action for left button is to select what mouse is pointing at
	      ((LOGTEST 1 BD)
	       (AND (SETQ HAND (WINDOW-UNDER-MOUSE :MOUSE-SELECT :ACTIVE X Y))
		    ;; Next line temporarily papers over a bug with :MOUSE-SELECT
		    (GET-HANDLER-FOR HAND :SELECT)
		    (MOUSE-SELECT HAND)))
	      ;; Default action for middle button is to switch to the main screen
	      ((LOGTEST 2 BD)
	       (when (TYPEP MOUSE-SHEET 'SCREEN)
		 (if (mac-screen-p mouse-sheet)
		     (give-mouse-ownership-to-the-Explorer)
		     (PROCESS-RUN-FUNCTION "Set mouse sheet" #'MOUSE-SET-SHEET DEFAULT-SCREEN))))
	      ;; Default action for right button is to call the system menu
	      ((LOGTEST 4 BD)
	       (MOUSE-CHARACTER-BUTTON-ENCODE BD)      ;Satisfy those who double-click out of habit
	       (MOUSE-CALL-SYSTEM-MENU)))))
 (if (intersection '(:release-6) *features*)
    ;; TI Code from SYS:WINDOW;MOUSE.LISP#34
    TV: ;;; Updated by JPR for Rel 6 on 7/14/89
    (DEFUN MOUSE-DEFAULT-HANDLER (WINDOW &OPTIONAL SCROLL-BAR-FLAG
				  &AUX
				  (WINDOW-X-OFFSET 0) (WINDOW-Y-OFFSET 0)
				  WINDOW-X WINDOW-Y)
      (unless (SYMBOLP WINDOW)
	(MULTIPLE-VALUE-SETQ (WINDOW-X-OFFSET WINDOW-Y-OFFSET)
			     (SHEET-CALCULATE-OFFSETS SELF MOUSE-SHEET)))
    
      ;; Be careful not to do the :update method when the who line documentation window
      ;; isn't there (which is the case during a window system build).
      (WHEN (AND (BOUNDP 'WHO-LINE-DOCUMENTATION-WINDOW) WHO-LINE-DOCUMENTATION-WINDOW)
	;; Update who-line when entering new handlers.
	(send who-line-documentation-window :update))
      
      (DO ((DX) (DY) (BU) (BD) (HAND) (X) (Y)
	   (OLD-OWNER WINDOW-OWNING-MOUSE WINDOW-OWNING-MOUSE)
	   (LEFT-OFFSET 0)
	   (right-offset 0)
	   (WAIT-FLAG NIL T))
	  (MOUSE-RECONSIDER)
	;; Wait until the mouse moves
	(MULTIPLE-VALUE-SETQ (DX DY BD BU X Y) (MOUSE-INPUT WAIT-FLAG))
	;; If asked to reconsider, do so immediately.
	;; Don't bother updating blinker since it is likely to change soon, and
	;; in any case we are going to be called back shortly.
	(when MOUSE-RECONSIDER (RETURN NIL))
	;; Update console-idle time when buttons pushed
	(unless (ZEROP BD) (SETQ KBD-LAST-ACTIVITY-TIME (TIME)))
	;; Approximate speed of the mouse in inches per second
	(SETQ MOUSE-SPEED (/ (ISQRT (+ (* MOUSE-X-SPEED MOUSE-X-SPEED)
				       (* MOUSE-Y-SPEED MOUSE-Y-SPEED)))
			     100.0s0))
	;; If the mouse is moving incredibly fast, flash up something to
	;; help the user find it.  Thus if you can't find the mouse, you must whip it.
	(when (> MOUSE-SPEED MOUSE-FAST-MOTION-SPEED)
	  (if mouse-fast-track-bitmap-mouse-p
	      (draw-bitmap-mouse-cursor mouse-speed)
	      ;;ELSE
	      (draw-mouse-fast-motion-cursor)))
    
	(SETQ WINDOW-X (- X WINDOW-X-OFFSET)	; X offset of mouse within window
	      WINDOW-Y (- Y WINDOW-Y-OFFSET))	; Y offset of mouse within window
    
	;; Consider entering the scroll bar.  [Perhaps this should be changed so that
	;; it is in the move-handler rather than here.  Problem with that is LEFT-OFFSET.]
	;; If there is a scroll bar and we are entering it, activate it.
	;; However, the mouse must move at least a certain distance past the edge
	;; of the window in order to qualify for scrolling (this is set by
	;; the SCROLL-BAR-RELUCTANCE variable in the window).  Before entering
	;; scroll bar, send a :MOUSE-MOVES message in order to let the window know
	;; what's happening.
    
	;; LEFT-OFFSET is how far out the left side of the window the mouse has moved,
	;; or 0 if the mouse is inside the window.
	;; If the window is at the left edge of the screen, MOUSE-X will not itself
	;; move out the left edge of the window, but DX will.  When the mouse reaches
	;; the left edge of the window, accumulate leftward motion into LEFT-OFFSET.  
	;; RIGHT-OFFSET does the same thing for when the scroll-bar is on the right.
	(COND ((< WINDOW-X 0)	;; may 9-7-88 was <= see HACK below
	       (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      1)))	   		;First time, don't use all of DX
	      ((and (typep window 'sheet)
		    (>= window-x (sheet-width window))
		    (send window :send-if-handles :scroll-bar-on-right))
	       (setq right-offset (if (plusp right-offset)
				      (max (+ right-offset dx) 1)
				      1)))
	      ;; [may] 9-7-88
	      ;; HACK : the left edge is a special case. Its possible to hang up the
	      ;;	    the system when scroll bar on right and mouse on left edge
	      ;;	    looping in :handle-mouse and here with wait-flag always nil
	      ;;	    which allows #'mouse-input to be process greedy! We COULD always set
	      ;;	    left-offset to 0 in/of 1 BUT left edge of scroll-bar would not make
	      ;;	    double-arrow for scrolling sometimes.
	      ;; cases : 1. left-offset=0 AND no scroll-bar ( fixed lying :enable-scrolling-p )
	      ;;	    2. left-offset=0 AND scroll-bar on right - force left-offset = 0 to check right-offset
	      ;;	    3. left-offset=0 AND left scroll-bar - this case only happens SOMETIMES as scroll-bar is being
	      ;;	      added. The other two cases happen EVERY time.
	      ((and (= window-x 0) (eq scroll-bar-flag t)) ;; may 01/30/89 ADDED (eq t scroll-bar-flag) check I left out.
	       ;; we are called with scroll-bar-flag indicating a scroll bar but if
	       ;; scroll bar is on RIGHT we really should check right-offset FIRST in COND below
	       (COND ((and (typep window 'sheet)
			   (send window :send-if-handles :scroll-bar-on-right)) ;; scroll bar is on RIGHT
		      (SETQ left-offset 0)) ;; force skipping of left-offset in COND below
		     (t
		      (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      0))))) ;First time, don't use all of DX [may 9-13-88 was 1 - fix case 3]
	      (t
	       (SETQ LEFT-OFFSET 0
		     right-offset 0)))
	(COND ((or old-owner window-owning-mouse)) ; These disable scroll-bar.
	      ((EQ SCROLL-BAR-FLAG 'T)
	       (cond ((AND SCROLL-BAR-MAX-SPEED
			  (> MOUSE-SPEED SCROLL-BAR-MAX-SPEED)))
		     ((PLUSP LEFT-OFFSET)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar))
			     (return nil))
			    ((> LEFT-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X 0))))	;Don't escape the window yet
		     ((plusp right-offset)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar-on-right))
			     (return nil))
			    ((> right-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X (1- (sheet-width window))))))))	;Don't escape the window yet
	      
	      ((EQ SCROLL-BAR-FLAG :IN)
	       ;; We are in the scroll bar.  Moving the mouse faster than the exit
	       ;; speed, or moving it into the interior of the window more than 40. 
	       ;; will escape.  
	       (COND ((AND SCROLL-BAR-MAX-EXIT-SPEED
			   (> MOUSE-SPEED SCROLL-BAR-MAX-EXIT-SPEED))
		      ;; Moving like a bat, let the guy out of the scroll bar
		      (RETURN NIL))
		     ((MINUSP WINDOW-X)		;Trying to go out left, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X 0)
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X WINDOW-X-OFFSET))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((>= window-x (sheet-width window))	;Trying to go out right, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X (- (sheet-width window) 8.))  ; hack
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X (+ WINDOW-X-OFFSET window-x)))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((and (> WINDOW-X (+ 40. (sheet-inside-left window)))	; Escape scroll-bar
			   (< window-x (- (sheet-inside-right window) 40.)))
		      (RETURN NIL)))))
	;; Update the position of the mouse before checking for button clicks, so
	;; that button clicks get processed with knowledge of where the mouse
	;; was when the button was first clicked.  The arguments to the move handler
	;; may be where the mouse was when the button was clicked, whereas the
	;; mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.   
	(SETQ MOUSE-WARP NIL)
	(SEND (IF (SYMBOLP WINDOW)
		     'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
		 :mouse-moves WINDOW-X WINDOW-Y)
	;; Check for all the ways of losing control of the mouse.
	(IF (COND ;; The move handler may have decided to warp the mouse so that it will not
		  ;; move out of the window.  This test is a crock but should work.
		  (MOUSE-WARP NIL)
		  ;; Check for mouse ceasing to be grabbed.
		  ((EQ WINDOW T)
		   (NEQ WINDOW-OWNING-MOUSE T))
		  ;; Check for window becoming grabbed.
		  ((EQ WINDOW-OWNING-MOUSE T)
		   (NEQ WINDOW T))
		  ;; Check for some other window (not above this one) being greedy.
		  (WINDOW-OWNING-MOUSE
		   (NOT (SHEET-ME-OR-MY-KID-P WINDOW WINDOW-OWNING-MOUSE)))
		  ;; Check for moving into a window when not in any
		  ((NULL window)
		   (OR (window-owning-mouse x y)
		       ;; Also check for mouse escaping MOUSE-SHEET.  CJJ 05/11/88.
		       ;;; For Mouse-Transport support.  Added by KJF for CJJ on 08/16/88 for
		       ;;; Multiple Monitor (MMON) support.
		       (AND (mmon-p) (NOT (mouse-in-mouse-sheet-p)))))
		  ;; Check for leaving the boundaries of the current window
		  ;; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning
		  ((NOT (AND (SHEET-EXPOSED-P WINDOW)
			     (>= WINDOW-X 0)
			     (<  WINDOW-X (SHEET-WIDTH WINDOW))
			     (>= WINDOW-Y 0)
			     (<  WINDOW-Y (SHEET-HEIGHT WINDOW))
			     ;; Also check for mouse escaping MOUSE-SHEET.  CJJ 05/11/88.
			     ;;; For Mouse-Transport support.  Added by KJF for CJJ on 08/16/88 for
			     ;;; Multiple Monitor (MMON) support.
			     (IF (mmon-p)
				 (OR (EQ scroll-bar-flag :in) (mouse-in-mouse-sheet-p))
				 ;; else
				 t)))
		   WAIT-FLAG)
		  ;; Check for moving into an inferior of the current window
		  ((NEQ (LOWEST-SHEET-UNDER-POINT WINDOW WINDOW-X WINDOW-Y
						  :HANDLE-MOUSE :EXPOSED)
			WINDOW)
		   T))
	    ;; Return to overseer, saving any pending button click.
	    (RETURN (MOUSE-DEFER-BUTTONS BU BD)))
	;; Now process button pushes if mouse is not seized.
	(COND ((OR (ZEROP BD) (EQ WINDOW T) (EQ OLD-OWNER T))	;; may 9-7-88
	       ;;; The following added by JPR to mirror Acuff's deexposed mouse
	       ;;; buttons.
	       (setq hand (window-under-mouse :deexposed-mouse-moves :active x y))
	       (if hand
		   (multiple-value-bind (xoff yoff)
		       (sheet-calculate-offsets hand mouse-sheet)
		     (send hand :deexposed-mouse-moves (- x xoff) (- y yoff))))
	      )
	      ;; If over an exposed window, send it the button-push
	      (WINDOW (SEND WINDOW :mouse-buttons BD WINDOW-X WINDOW-Y))
	      ;;RDA:- call :DEEXPOSED-MOUSE-BUTTONS if we're over a
	      ;; deexposed window that has a handler for it
	      ((setq hand (window-under-mouse :deexposed-mouse-buttons :active x y))
	       (multiple-value-bind (xoff yoff)
		   (sheet-calculate-offsets hand mouse-sheet)
		 (send hand :deexposed-mouse-buttons bd (- x xoff) (- y yoff))))
	      ;; Default action for left button is to select what mouse is pointing at
	      ((LOGTEST 1 BD)
	       (AND (SETQ HAND (WINDOW-UNDER-MOUSE :MOUSE-SELECT :ACTIVE X Y))
		    ;; Next line temporarily papers over a bug with :MOUSE-SELECT
		    (GET-HANDLER-FOR HAND :SELECT)
		    (MOUSE-SELECT HAND)))
	      ;; Default action for middle button is to switch to the main screen
	      ((LOGTEST 2 BD)
	       (when (TYPEP MOUSE-SHEET 'SCREEN)
		 (if (mac-screen-p mouse-sheet)
		     (give-mouse-ownership-to-the-Explorer)
		     (PROCESS-RUN-FUNCTION "Set mouse sheet" #'MOUSE-SET-SHEET DEFAULT-SCREEN))))
	      ;; Default action for right button is to call the system menu
	      ((LOGTEST 4 BD)
	       (MOUSE-CHARACTER-BUTTON-ENCODE BD)	   ;Satisfy those who double-click out of habit
	       (MOUSE-CALL-SYSTEM-MENU)))))
    ;; Else Rel 3
    ;;;TI Code: From SYS:WINDOW;MOUSE.LISP#11
    TV:
    (DEFUN MOUSE-DEFAULT-HANDLER (WINDOW &OPTIONAL SCROLL-BAR-FLAG
				  &AUX
				  (WINDOW-X-OFFSET 0) (WINDOW-Y-OFFSET 0)
				  WINDOW-X WINDOW-Y)
      (unless (SYMBOLP WINDOW)
	(MULTIPLE-VALUE-SETQ (WINDOW-X-OFFSET WINDOW-Y-OFFSET)
	  (SHEET-CALCULATE-OFFSETS SELF MOUSE-SHEET)))
      
      ;; Be careful not to do the :update method when the who line documentation window
      ;; isn't there (which is the case during a window system build).
      (WHEN (AND (BOUNDP 'WHO-LINE-DOCUMENTATION-WINDOW) WHO-LINE-DOCUMENTATION-WINDOW)
	;; Update who-line when entering new handlers.
	(send who-line-documentation-window :update))
      
      (DO ((DX) (DY) (BU) (BD) (HAND) (X) (Y)
	   (OLD-OWNER WINDOW-OWNING-MOUSE WINDOW-OWNING-MOUSE)
	   (LEFT-OFFSET 0)
	   (right-offset 0)
	   (WAIT-FLAG NIL T))
	  (MOUSE-RECONSIDER)
	;; Wait until the mouse moves
	(MULTIPLE-VALUE-SETQ (DX DY BD BU X Y) (MOUSE-INPUT WAIT-FLAG))
	;; If asked to reconsider, do so immediately.
	;; Don't bother updating blinker since it is likely to change soon, and
	;; in any case we are going to be called back shortly.
	(when MOUSE-RECONSIDER (RETURN NIL))
	;; Update console-idle time when buttons pushed
	(unless (ZEROP BD) (SETQ KBD-LAST-ACTIVITY-TIME (TIME)))
	;; Approximate speed of the mouse in inches per second
	(SETQ MOUSE-SPEED (/ (ISQRT (+ (* MOUSE-X-SPEED MOUSE-X-SPEED)
				       (* MOUSE-Y-SPEED MOUSE-Y-SPEED)))
			     100.0s0))
	;; If the mouse is moving incredibly fast, flash up something to
	;; help the user find it.  Thus if you can't find the mouse, you must whip it.
	(when (> MOUSE-SPEED MOUSE-FAST-MOTION-SPEED)
	  (if mouse-fast-track-bitmap-mouse-p
	      (draw-bitmap-mouse-cursor mouse-speed)
	      ;;ELSE
	      (draw-mouse-fast-motion-cursor)))
	
	(SETQ WINDOW-X (- X WINDOW-X-OFFSET)	       ; X offset of mouse within window
	      WINDOW-Y (- Y WINDOW-Y-OFFSET))	       ; Y offset of mouse within window
	
	;; Consider entering the scroll bar.  [Perhaps this should be changed so that
	;; it is in the move-handler rather than here.  Problem with that is LEFT-OFFSET.]
	;; If there is a scroll bar and we are entering it, activate it.
	;; However, the mouse must move at least a certain distance past the edge
	;; of the window in order to qualify for scrolling (this is set by
	;; the SCROLL-BAR-RELUCTANCE variable in the window).  Before entering
	;; scroll bar, send a :MOUSE-MOVES message in order to let the window know
	;; what's happening.
	
	;; LEFT-OFFSET is how far out the left side of the window the mouse has moved,
	;; or 0 if the mouse is inside the window.
	;; If the window is at the left edge of the screen, MOUSE-X will not itself
	;; move out the left edge of the window, but DX will.  When the mouse reaches
	;; the left edge of the window, accumulate leftward motion into LEFT-OFFSET.  
	;; RIGHT-OFFSET does the same thing for when the scroll-bar is on the right.
	(COND ((<= WINDOW-X 0)
	       (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      1)))	       ;First time, don't use all of DX
	      ((and (typep window 'sheet)
		    (>= window-x (sheet-width window))
		    (send window :send-if-handles :scroll-bar-on-right))
	       (setq right-offset (if (plusp right-offset)
				      (max (+ right-offset dx) 1)
				      1)))
	      (t
	       (SETQ LEFT-OFFSET 0
		     right-offset 0)))
	(COND ((or old-owner window-owning-mouse))     ; These disable scroll-bar.
	      ((EQ SCROLL-BAR-FLAG 'T)
	       (cond ((AND SCROLL-BAR-MAX-SPEED
			   (> MOUSE-SPEED SCROLL-BAR-MAX-SPEED)))
		     ((PLUSP LEFT-OFFSET)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar))
			     (return nil))
			    ((> LEFT-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X 0))))    ;Don't escape the window yet
		     ((plusp right-offset)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar-on-right))
			     (return nil))
			    ((> right-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X (1- (sheet-width window))))))))  ;Don't escape the window yet
	      
	      ((EQ SCROLL-BAR-FLAG :IN)
	       ;; We are in the scroll bar.  Moving the mouse faster than the exit
	       ;; speed, or moving it into the interior of the window more than 40. 
	       ;; will escape.  
	       (COND ((AND SCROLL-BAR-MAX-EXIT-SPEED
			   (> MOUSE-SPEED SCROLL-BAR-MAX-EXIT-SPEED))
		      ;; Moving like a bat, let the guy out of the scroll bar
		      (RETURN NIL))
		     ((MINUSP WINDOW-X)		       ;Trying to go out left, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X 0)
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X WINDOW-X-OFFSET))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((>= window-x (sheet-width window))       ;Trying to go out right, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X (- (sheet-width window) 8.))    ; hack
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X (+ WINDOW-X-OFFSET window-x)))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((and (> WINDOW-X (+ 40. (sheet-inside-left window)))     ; Escape scroll-bar
			   (< window-x (- (sheet-inside-right window) 40.)))
		      (RETURN NIL)))))
	;; Update the position of the mouse before checking for button clicks, so
	;; that button clicks get processed with knowledge of where the mouse
	;; was when the button was first clicked.  The arguments to the move handler
	;; may be where the mouse was when the button was clicked, whereas the
	;; mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.   
	(SETQ MOUSE-WARP NIL)
	(SEND (IF (SYMBOLP WINDOW)
		  'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
	      :mouse-moves WINDOW-X WINDOW-Y)
	;; Check for all the ways of losing control of the mouse.
	(IF (COND ;; The move handler may have decided to warp the mouse so that it will not
              ;; move out of the window.  This test is a crock but should work.
              (MOUSE-WARP NIL)
              ;; Check for mouse ceasing to be grabbed.
              ((EQ WINDOW T)
               (NEQ WINDOW-OWNING-MOUSE T))
              ;; Check for window becoming grabbed.
              ((EQ WINDOW-OWNING-MOUSE T)
               (NEQ WINDOW T))
              ;; Check for some other window (not above this one) being greedy.
              (WINDOW-OWNING-MOUSE
               (NOT (SHEET-ME-OR-MY-KID-P WINDOW WINDOW-OWNING-MOUSE)))
              ;; Check for moving into a window when not in any
              ((NULL WINDOW)
               (WINDOW-OWNING-MOUSE X Y))
              ;; Check for leaving the boundaries of the current window
              ;; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning
              ((NOT (AND (SHEET-EXPOSED-P WINDOW)
                         (>= WINDOW-X 0)
                         (<  WINDOW-X (SHEET-WIDTH WINDOW))
                         (>= WINDOW-Y 0)
                         (<  WINDOW-Y (SHEET-HEIGHT WINDOW))))
               WAIT-FLAG)
              ;; Check for moving into an inferior of the current window
              ((NEQ (LOWEST-SHEET-UNDER-POINT WINDOW WINDOW-X WINDOW-Y
                                              :HANDLE-MOUSE :EXPOSED)
                    WINDOW)
               T))
	    ;; Return to overseer, saving any pending button click.
	    (RETURN (MOUSE-DEFER-BUTTONS BU BD)))
	;; Now process button pushes if mouse is not seized.
	(COND ((OR (ZEROP BD) (EQ WINDOW T) OLD-OWNER))
	      ;; If over an exposed window, send it the button-push
	      (WINDOW (SEND WINDOW :mouse-buttons BD WINDOW-X WINDOW-Y))
	      ;;RDA:- call :DEEXPOSED-MOUSE-BUTTONS if we're over a
	      ;; deexposed window that has a handler for it
	      ((setq hand (window-under-mouse :deexposed-mouse-buttons :active x y))
	       (multiple-value-bind (xoff yoff)
		   (sheet-calculate-offsets hand mouse-sheet)
		 (send hand :deexposed-mouse-buttons bd (- x xoff) (- y yoff))))
	      ;; Default action for left button is to select what mouse is pointing at
	      ((LOGTEST 1 BD)
	       (AND (SETQ HAND (WINDOW-UNDER-MOUSE :MOUSE-SELECT :ACTIVE X Y))
		    ;; Next line temporarily papers over a bug with :MOUSE-SELECT
		    (GET-HANDLER-FOR HAND :SELECT)
		    (MOUSE-SELECT HAND)))
	      ;; Default action for middle button is to switch to the main screen
	      ((LOGTEST 2 BD)
	       (IF (TYPEP MOUSE-SHEET 'SCREEN)
		   (PROCESS-RUN-FUNCTION "Set mouse sheet" #'MOUSE-SET-SHEET DEFAULT-SCREEN)))
	      ;; Default action for right button is to call the system menu
	      ((LOGTEST 4 BD)
	       (MOUSE-CHARACTER-BUTTON-ENCODE BD)      ;Satisfy those who double-click out of habit
	       (MOUSE-CALL-SYSTEM-MENU)))))
    )
 )

;;; The following lines added by JPR to get around a problem on the uX.
(defparameter tv:*has-been-booted* t)

(Add-Initialization "Say that I've booted."     '(setq tv:*has-been-booted* t)   '(:User-Application :Normal))
(Add-Initialization "Say that I've booted."     '(setq tv:*has-been-booted* t)   '(:Login :Normal))
(Add-Initialization "Say that I've logged out." '(setq tv:*has-been-booted* nil) '(:Logout :Normal))


;;;TI Code:  From SYS:WINDOW-MX;MAC-WHOLIN.LISP#3
TV:
(DEFUN WHO-LINE-DOCUMENTATION-FUNCTION-MAC ()
  "This displays the who line documentation for the currently selected
window.  If the selected window is locked an error message is blinked
in the who line area.  Who line documentation may be either a string
of text which is to be displayed or a list of keyword value pairs.  To
see what keywords are accepted see the documentation for the
tv:process-who-line-documentation-list function."
  (declare (special last-mac-mouse-doc-1 newline-seq current-doc-line mac-line))
  (LET* ((W MOUSE-WINDOW)
	 (MSG  "*** Error with window locked; try TERMINAL C-CLEAR-INPUT or TERMINAL C-C ***")
	 (MSG1 "    Error with window locked; try TERMINAL C-CLEAR-INPUT or TERMINAL C-C")
	 (NEW-STATE (COND (LOCKED-ERROR-WINDOWS
			    ;; To attract attention, make this message blink.
			    ;; The following EQ test should remain as EQ in spite of what the
			    ;; compiler says.  In this case EQ is both faster and correct.
			    (IF (compiler-let ((inhibit-style-warnings-switch t))
				  (EQ last-mac-mouse-doc-1 MSG))	;Compiler-let added by PMH 7/2/87
				MSG1 MSG))
			  ;;RDA:- ask about deexposed doc
			  ((and *has-been-booted* (null w))
			   (let ((handling-window
				   (window-under-mouse
				     :deexposed-who-line-documentation-string)))
			     (when handling-window
			       (send handling-window
				     :deexposed-who-line-documentation-string))
			     )
			   )
                          ((SYMBOLP W)
                           (AND W WHO-LINE-MOUSE-GRABBED-DOCUMENTATION))
                          (T (MULTIPLE-VALUE-BIND (DOC ERROR)
                                 (CATCH-ERROR
                                   (FUNCALL W :WHO-LINE-DOCUMENTATION-STRING)
                                   NIL)
                               (IF ERROR
                                   "Error getting documentation string" DOC))))))
    (COND ((AND (NEQ last-mac-mouse-doc-1  NEW-STATE)
                (NOT (EQUAL last-mac-mouse-doc-1  NEW-STATE)))
           (SETQ last-mac-mouse-doc-1  NEW-STATE)
	   (COND ((TYPEP NEW-STATE 'STRING)
		  (let ((newline-search-loc -1))
		    (loop
		      (if (setf newline-search-loc (search newline-seq (the string new-state) :start2 (incf newline-search-loc))) 
			  (setf (aref new-state newline-search-loc) #\space)
			  (return))))
		  (display-mac-mouse-documentation t new-state))
		 ((CONSP NEW-STATE)

		  (PROCESS-WHO-LINE-DOCUMENTATION-LIST-MAC NEW-STATE)
		  )
		 (T (CATCH 'PAGE-OVERFLOW
		      (PROCESS-WHO-LINE-DOCUMENTATION-LIST-MAC
			'(:MOUSE-R-1 "Bring up the System Menu."))
		      ))))))
  (unless (ZEROP (LENGTH current-doc-line))
    (display-mac-mouse-documentation t))
  (SETF mac-line 1)
  )

;;;TI Code:  From SYS:WINDOW;WHOLIN.LISP#12
TV:
(DEFUN WHO-LINE-DOCUMENTATION-FUNCTION (WHO-SHEET)
  "This displays the who line documentation for the currently selected
window.  If the selected window is locked an error message is blinked
in the who line area.  Who line documentation may be either a string
of text which is to be displayed or a list of keyword value pairs.  To
see what keywords are accepted see the documentation for the
tv:process-who-line-documentation-list function."
  (DECLARE (:SELF-FLAVOR WHO-LINE-SHEET))
  (LET* ((W MOUSE-WINDOW)
	 (MSG  "*** Error with window locked; try TERMINAL C-CLEAR-INPUT or TERMINAL C-C ***")
	 (MSG1 "    Error with window locked; try TERMINAL C-CLEAR-INPUT or TERMINAL C-C")
	 (NEW-STATE (COND (LOCKED-ERROR-WINDOWS
			    ;; To attract attention, make this message blink.
			    ;; The following EQ test should remain as EQ in spite of what the
			    ;; compiler says.  In this case EQ is both faster and correct.
			   (IF (compiler-let ((inhibit-style-warnings-switch t))
				  (EQ WHO-LINE-ITEM-STATE MSG))	;Compiler-let added by PMH 7/2/87
				MSG1 MSG))
			  ;;RDA:- ask about deexposed doc
			  ((and *has-been-booted* (null w))
			   (let ((handling-window
				   (window-under-mouse
				     :deexposed-who-line-documentation-string)))
			     (when handling-window
			       (send handling-window
				     :deexposed-who-line-documentation-string))
			     )
			   )
                          ((SYMBOLP W)
                           (AND W WHO-LINE-MOUSE-GRABBED-DOCUMENTATION))
                          (T (MULTIPLE-VALUE-BIND (DOC ERROR)
                                 (CATCH-ERROR
                                   (FUNCALL W :WHO-LINE-DOCUMENTATION-STRING)
                                   NIL)
                               (IF ERROR
                                   "Error getting documentation string" DOC))))))
    (COND ((AND (NEQ WHO-LINE-ITEM-STATE NEW-STATE)
                (NOT (EQUAL WHO-LINE-ITEM-STATE NEW-STATE)))
           (SETQ WHO-LINE-ITEM-STATE NEW-STATE)
           (SHEET-CLEAR WHO-SHEET)
           (SHEET-SET-CURSORPOS WHO-SHEET 2 1)
           (CATCH-ALL
             (COND ((TYPEP NEW-STATE 'STRING)
                    (CATCH 'PAGE-OVERFLOW
                      (SHEET-STRING-OUT WHO-SHEET NEW-STATE)))
                   ((CONSP NEW-STATE)
                    (PROCESS-WHO-LINE-DOCUMENTATION-LIST WHO-SHEET NEW-STATE))
                   (T (CATCH 'PAGE-OVERFLOW
                        (PROCESS-WHO-LINE-DOCUMENTATION-LIST
                          WHO-SHEET
                          '(:MOUSE-R-1 "Bring up the System Menu."))))))))))

(provide 'deexposed-mouse)

;;;=============================================================================

;;;  Snapshot support

;;;  Works best when used with DEEXPOSED-MOUSE

(defflavor simple-bit-window
	   ()
	   (stream-mixin
	    borders-mixin
	    box-label-mixin
	    label-mixin
	    minimum-window
	    )
  (:default-init-plist :blinker-p nil :save-bits t :label nil)
  )

(defmethod (simple-bit-window :deexposed-mouse-buttons) (mask x y)
  "Call standard :MOUSE-BUTTONS method."
  (send self :mouse-buttons mask x y)
  )

;;;Edited by Acuff                 31 Oct 87  12:20
;;;Edited by Acuff                 16 Nov 87  15:43
(defmethod (simple-bit-window :mouse-click) (button x y)
  "L: Expose, else Pop up System Menu."
  (case button
	((#\mouse-L-1 #\mouse-L-2) (send self :expose))
	((#\mouse-M-1 #\mouse-M-2)
	 (if (and (boundp '*dragging-enabled?*)
		  *dragging-enabled?*
		  (char= #\mouse-m-1 button))
	     (drag-window-from self x y)
	     (process-run-function
	       '(:name "Set Window Position" :priority 1)
	       #'(lambda (window) (mouse-set-window-position window))
	       self)
	     )
	 )
	((#\mouse-R-1 #\mouse-R-2) (mouse-call-system-menu))
	)
  )

(defmethod (simple-bit-window :deexposed-who-line-documentation-string) ()
  "Call normal :WHO-LINE-DOCUMENTATION-STRING."
  (send self :who-line-documentation-string)
  )

(defmethod (simple-bit-window :who-line-documentation-string) ()
  `(:mouse-L-1 "Expose This Window" :mouse-M-1 "Move This Window"
    :mouse-R-1 "System Menu")
  )

;;; Have to refresh from saved bits always
(defwhopper (simple-bit-window :refresh) (&optional ignore)
  "Refresh from our bit-array"
  (declare (ignore ignore))
  (continue-whopper :some-old-bits))

(defun snap-region (&optional (window mouse-sheet) edges)
  "Take a snapshot of a region of the screen into a simple window."
  (let* ((screen (sheet-get-screen window))
	 (screen-bits (send screen :screen-array)))
    (unless (consp edges)
      (setf edges (multiple-value-list
		    (mouse-specify-rectangle nil nil nil nil screen 0 0 t))))
    (when (integerp (first edges))
      (let ((new-window (make-instance 'simple-bit-window :superior screen
				       :edges edges :borders 2))
	    (width (- (third edges) (first edges)))
	    (height (- (fourth edges) (second edges))))
	(prepare-sheet (screen)
	  (bitblt alu-seta (- width 4)
		  (- height 4)
		  screen-bits
		  (+ (first edges) 2)
		  (+ (second edges) 2)
		  (send new-window :bit-array)
		  2 2)
	  )
	new-window
	)
      )
    )
  )

(compile-flavor-methods simple-bit-window)

(provide 'snapshot-windows)

;;;=============================================================================

;;;  Rich Acuff, Stanford KSL, Feb 87

;;;  Provides a `background' window so that uninteresting windows can be
;;;  hidden, making it possible to use a ``desk-top'' screen management
;;;  metaphore.  Evaluate (W:CREATE-BACKGROUND) after loading.  Push any
;;;  windows that you always want up (eg. from Graphical-Monitors) onto
;;;  W:*WINDOWS-TO-LEAVE-EXPOSED*, and then use <Term> Z to `reset' the
;;;  screen.  Clicking left on the background will get you a `background
;;;  menu' that offers to let you clean up your windows (same as <Term> Z),
;;;  or choose a new background image, which can be a gray level, or one of
;;;  several images stored on IMAGES:IMAGES;*.BITMAP.  Use <TERM> 1 C to
;;;  complement the background's video.

(require '36xx-Explorer)			;for gray values

;;;-----------------------------------------------------------------------------

;;;  Variables and parameters

(defvar *background* nil
  "An instance of W:BACKGROUND-WINDOW")

(defvar *background-bitmaps-files* (parse-namestring "IMAGES:IMAGES;*.BITMAP#>")
  "Pathname describing files containing images suitable for use on
   backgrounds, in `fast' format.")

(defparameter *background-grays* 'tv:(6%-gray 7%-gray 8%-gray 12%-gray 25%-gray
				      33%-gray 50%-gray 66%-gray 75%-gray
				      88%-gray)
  "Gray arrays that can be used for backgrounds.")

(defvar *windows-to-leave-exposed* nil "Windows left up by CLEAN-UP-WINDOWS")

#|
;;; RDA: have to leave this out until W:READ-BIT-ARRAY-FILE takes a
;;; scratch bm arg.
(defvar *background-scratch-bitmap* (make-array (list 1024 1024)
						:type 'art-1b ;'Bit
						:initial-element 0)
  "Scratch bitmap used by CHOOSE-BACKGROUND")
|#
;;;----------------------------------------------------------------------

;;;  Background windows

(defflavor background-window
	   ()
	   (select-mixin
	    stream-mixin
	    minimum-window
	    )
  (:default-init-plist :save-bits t :blinker-p nil
		       :deexposed-typeout-action :permit)
  (:documentation
    "A window to serve as a curtain of sorts between a working set of
windows and other actiavted but currently not in use window, ie.  a
background.  Clicking on it causes a menu of background functions to pop
up, even when it's deexposed.")
  )

(defmethod (background-window :screen-manage-deexposed-visibility) ()
  "Make sure the screen manager updates our image, even when we're
   buried."
  t
  )

;;;Edited by Acuff                 4 Dec 87  19:09
;;;Edited by Acuff                 4 Dec 87  19:32
(defmethod (background-window :mouse-click) (char x y)
  "Don't select with left click.  Give a menu."
  (declare (ignore x y))
  (case char
    ((#\mouse-R-1 #\mouse-R-2) 
     (mouse-call-system-menu))
    ((#\mouse-M-1 #\mouse-M-2)
     (mouse-call-standard-system-menu))
    ((#\mouse-L-1 #\mouse-L-2)
     (mouse-call-background-menu))
    )
  )

(defmethod (background-window :deexposed-mouse-buttons) (buttons x y)
  "Call normal :MOUSE-BUTTONS method to pop up right menu."
  (declare (ignore x y))			;these will be screen coords
  (send self :mouse-buttons buttons 0 0)	;x and y don't matter
  )

(defmethod (background-window :deexposed-who-line-documentation-string) ()
  "Call normal :WHO-LINE-DOCUMENTATION-STRING."
  (send self :who-line-documentation-string)
  )

;;;Edited by Acuff                 4 Dec 87  19:09
;;;Edited by Acuff                 4 Dec 87  19:32
(defmethod (background-window :who-line-documentation-string) ()
  `(:mouse-L-1 "Background Menu"
    :mouse-M-1 "Large System Menu"
    :mouse-R-1 "System Menu")
  )

;;;----------------------------------------------------------------------

;;;  Background menus

(defun mouse-call-background-menu (&optional (sup mouse-sheet))
  "Like MOUSE-CALL-SYSTEM-MENU, but for background menus."
  (process-run-function '(:name "Background menu" :priority 0)
			#'(lambda (sup)
			    (using-resource (menu background-menu sup)
			      (send menu :choose)))
			sup))

(defparameter *background-menu-items*
	       `(("New Background" :eval (w:choose-background)
		  :documentation "Choose a new image for the background")
		 ("Random Background" :eval (w:choose-random-background)
		  :documentation
		  "Choose a new image at random for the background")
		 ("Bury Windows" :eval (w:clean-up-windows)
		  :documentation
		  "Bury all windows not on USER:*WINDOWS-TO-LEAVE-EXPOSED* behind the background."
		  )
		 ("Screen Saver" :eval
		  (process-run-function "Screen Saver" 'tv:screen-saver))
		 )
  )

(defwindow-resource background-menu ()
	:make-window
        (menu
	  :dynamic t
	  :item-list-pointer '*background-menu-items*
	  :pop-up t
          :save-bits nil)
	:reusable-when :deexposed)

;;;Edited by Acuff                 4 Dec 87  18:45
(defun choose-background ()
  "Choose a new image for the background window in w:*BACKGROUND*"
  (let (items choice)
    (setf items (loop for f in
		      (ignore-errors (directory *background-bitmaps-files*))
		      collect
		      `(,(string-capitalize (pathname-name f))
			:value ,f
			:documentation
			"Load this image and paint it on the background.")
		      )
	  )
    (push `("Gray Value" :value :gray :font fonts:cptfontb
	    :documentation
	    "Choose a gray value and paint it on the background")
	  items)
    (setf choice (menu-choose items :label "Backgrounds"))
    (case choice
	  (nil)					;do nothing
	  (:gray (choose-gray-background))
	  (otherwise
	   (set-background
	     (read-bit-array-file choice))
;	     (read-fast-bm choice *background-scratch-bitmap* t))
	   )
	  )
    )
  )

(defun choose-random-background ()
  "Pick one of the images in IMAGES:IMAGES; at randome and make it the new
   background."
  (set-background (random-background)))

(defun random-background ()
  (let ((images (directory "IMAGES:IMAGES;*.BITMAP#>")))
    (read-bit-array-file
      (elt images (random (length images) (make-random-state t))))
    )
  )
(defun choose-gray-background ()
  "Choose a gray value and paint it on the background"
  (let ((choice
	  (menu-choose (loop for g in *background-grays*
			     collect
			     `(,g :value ,g :documentation
			       "Paint the background with this gray shade"))
		       )
		)
	)
    (when choice (set-background (eval choice)))
    )
  )

;;;----------------------------------------------------------------------

;;; Functions for controlling backgrounds

(defun create-background ()
  "Create and initialize a new W:BACKGROUND-WINDOW and put it into
   W:*BACKGROUND*.  Creates it in reverse video since most images are done
   in black on white.  Use <Term> 1 C to change it's video."
  (setf *background* (make-instance 'background-window
				    :reverse-video-p t))
  (set-background 50%-gray)
  )

(defun set-background (bm &optional (window *background*))
  "Change the image in WINDOW to show the contents of the bitmap BM."
  (let ((old-video-mode (send window :reverse-video-p)))
    (send window :set-reverse-video-p nil)
    (send window :clear-screen)
    (send window :bitblt alu-seta		;new image
	  (send window :inside-width)
	  (send window :inside-height)
	  bm
	  0 0 0 0)
    (send window :set-reverse-video-p old-video-mode)
    )
  (prepare-sheet ((sheet-get-screen window))
    (send (sheet-get-screen window) :screen-manage)
    )
  )

;;;Edited by Acuff                 4 Dec 87  19:32
(defun clean-up-windows (&optional arg)
  "Exposes the background, W:*BACKGROUND*, and then exposes all the
   windows on W:*WINDOWS-TO-LEAVE-EXPOSED*.  NIL ARG means expose
   the selected window as well."
  (if *background*
      (let ((w selected-window))
	(send *background* :expose)
	(loop for w in *windows-to-leave-exposed* doing
	      (when (send w :active-p)
		(send w :expose))
	      )
	(when (and (not arg) w)
	  (send w :select))
	)
      (notify nil "There is no background window in W:*BACKGROUND*")
      )
  )

(add-terminal-key #\z 'clean-up-windows
   ''("Bury all windows not on W:*WINDOWS-TO-LEAVE-EXPOSED*."
      "   No arg means leave W:SELECTED-WINDOW exposed as well.")
   )

(defun kbd-complement-with-background (&optional arg)
  "If ARG = 1, complement the background window, else call
   KDB-COMPLEMENT."
  (if (eql arg 1)
      (ignore-errors
	(send *background* :set-reverse-video-p
	      (not (send *background* :reverse-video-p)))
	(prepare-sheet (*background*)
	  (send (sheet-get-screen *background*) :screen-manage))
	)
      (tv::kbd-complement arg)
      )
  )

(add-terminal-key #\c 'kbd-complement-with-background
		  ''("Complement video black-on-white state"
		     "    an arg of 1 means complement the background's video"
		     "    any other arg means complement the who line's video")
		  :system)

(defun top-level-window (window)
  "Returns the highest level window in the inferior tree that is at or
   above WINDOW and a direct inferior of WINDOW's screen.  Return WINDOW
   if it's a screen."
  (if (or (eq (sheet-get-screen window) (send window :superior))
	  (null (send window :superior)))
      window
      (top-level-window (send window :superior))))

(defun term-hack-windows-to-leave-exposed (&optional arg)
  "If ARG is NIL, push the window under the mouse onto
   W:*WINDOWS-TO-LEAVE-EXPOSED*, if ARG is 1, remove the window under the
   mouse from W:*WINDOWS-TO-LEAVE-EXPOSED*, and if ARG is anything else,
   set W:*WINDOWS-TO-LEAVE-EXPOSED* to NIL."
  (case arg
      (nil (push (top-level-window (window-under-mouse))
		 *windows-to-leave-exposed*))
      (1 (setf *windows-to-leave-exposed*
	       (delete (top-level-window (window-under-mouse))
		       *windows-to-leave-exposed*)))
      (otherwise (setq *windows-to-leave-exposed* nil))
      )
  )

(add-terminal-key #\e 'term-hack-windows-to-leave-exposed
   ''("Push the window under the mouse onto W:*WINDOWS-TO-LEAVE-EXPOSED*"
      "   an arg of one deletes from W:*WINDOWS-TO-LEAVE-EXPOSED*"
      "   any other arg sets W:*WINDOWS-TO-LEAVE-EXPOSED* to NIL")
   )

;;;-----------------------------------------------------------------------------

;;;Don't select a random window if there is no selected window
(unless (or (not (boundp 'tv:screen-saver )) (sys:mx-p))
  (advise tv:screen-saver :around 'fix-selected-window nil
    (let ((w tv:selected-window))
      :do-it
      (unless (or w
		  (not (fboundp 'w:clean-up-windows)))
	(w:clean-up-windows 1))
      )
    )
  )

(compile-flavor-methods background-window)

(provide 'backgrounds)

;;;=============================================================================

;;;  Windows that let what's under them show through

;;;  Rich Acuff, Stanford KSL, Mar-87

(require '36xx-Explorer)
;(require 'window-system-utilities)

(defflavor transparent-mixin
	   ((already-refreshing? nil))
	   ()
  (:required-methods :mask)
  (:required-flavors stream-mixin)
  (:documentation :mixin
"Provides for a window mixin that makes all the bits in itself other than
those specified by the bitmap returned by the :MASK message be just like
the image that would be shown if the window were not exposed.  Ie., the
underneath image ``bleeds'' through.  Works by shadowing :REFRESH.  Causes
:REFRESH to be called after :CHANGE-OF-SIZE-OR-MARGINS or :SET-POSITION.")
  )

(defun sheet-superior-screen-array (window)
  "Return the screen array of the superior window."
  (or (sheet-screen-array (sheet-superior window))
      (sheet-bit-array (sheet-superior window))
      ;;; I think that bit array is the one to use here.
      ;;; This has changed with rel 3
      )
  )


(defmethod (transparent-mixin :refresh) (&rest ignore)
  "Copy our screen array from what's underneath us, and then erase the :MASK"
  (declare (ignore ignore))
  (unless (or already-refreshing?
	      (not exposed-p)
	      (not (send superior :exposed-p)))
    (using-resource (underneath-pixels
		      temp-pixel-arrays
		      (send superior :width)
		      (send superior :height)
		      (array-type
			(sheet-superior-screen-array self)))
      ;; Don't get into a loop incase BURY re-exposes us.
      (letf ((already-refreshing? t))
         ;; Get us out of the way of our underneath
	 (send self :bury)
	 (send (sheet-get-screen self) :screen-manage)
	 (multiple-value-bind (x y)
	     (send self :position)
	   ;; Stash the bits
	   (bitblt alu-seta
		   (send self :width)
		   (send self :height)
		   (sheet-superior-screen-array self)
		   x y
		   underneath-pixels 0 0)
	   )
	 (send self :expose :noop)
	 ;; Put in the underneath bits
	 (send self :bitblt alu-seta
	       (send self :width)
	       (send self :height)
	       underneath-pixels
	       0 0 0 0)
	 ;; Clear the masked bits
	 (let ((mask (send self :mask)))
	   (when (typep mask '(array fixnum (* *)))
	     (send self :bitblt alu-setz
		   (send self :width)
		   (send self :height)
		   mask
		   0 0 0 0)
	     )
	   )
	 )
      )
    )
  )

(defmethod (transparent-mixin :after :change-of-size-or-margins)
	   (&rest ignore)
  "Make sure we're updated"
  (declare (ignore ignore))
  (send self :refresh)
  )

(defmethod (transparent-mixin :after :set-position) (x y &optional verify?)
  "Make sure we're updated"
  (declare (ignore x y))
  (unless verify?
    (send self :refresh))
  )

(provide 'Transparent-Window)

;;;=============================================================================

;;;  The window manager system menu

;(require 'snapshot-windows)

;;;  Control vars

(defvar *window-manager-system-menu-items*
	'(("Move" :window-op move-window-op
	   :documentation "Move this window")
	  ("Reshape" :buttons ((nil :window-op reshape-window-op)
			       (nil :window-op move-edge-window-op)
			       (nil :window-op move-multi-edge-window-op))
	   :documentation
	   "L: New edges from mouse, M: Move one edge; R: Move multiple edges")
	  ("Expand" :window-op expand-window-op
	   :documentation "Expand this window")
	  ("Create" :window-op create-window-op
	   :documentation "Create a new window")
	  ("Snap" :window-op snap-window-op
	   :documentation "Take a snapshop copy of a region of the screen")
	  ("Kill" :window-op kill-window-op
	   :documentation "Kill this window")
	  ("Bury" :window-op bury-window-op
	   :documentation "Bury this window")
	  ("Sys Menu" :funcall mouse-call-standard-system-menu
	   :documentation "Pop up the Explorer System Menu"))
  "Menu items for the Window Manager System Menu")

(defvar *use-window-manager-system-menu* nil
  "T means use a menu from the W:WINDOW-MANAGER-SYSTEM-MENU resource
instead of W:SYSTEM-MENU.  The former allows easier manipulation of windows
on the screen.")

(export '(*use-window-manager-system-menu* *window-manager-system-menu-items* ))

profile:
(define-profile-variable w:*use-window-manager-system-menu* (:display :mouse :input)
  :cvv-type :t-or-nil)

;;;----------------------------------------------------------------------

;;;TI Code:  Cribbed from SYS:WINDOW;MOUSE.LISP#11
(defun mouse-call-system-menu (&optional (sup mouse-sheet))
  ;;RDA: call the right system menu
  (if *use-window-manager-system-menu*
      (process-run-function '(:name "Window Manager Menu" :priority 10.)
			#'(lambda (sup)
			    (using-resource (menu window-manager-system-menu sup)
			      (funcall menu ':choose)))
			sup)
      (PROCESS-RUN-FUNCTION '(:NAME "System Menu" :PRIORITY 10.)
                        #'(LAMBDA (SUP)
                            (USING-RESOURCE (MENU SYSTEM-MENU SUP)
                              (SEND MENU :CHOOSE)))
                        SUP)
      )
  )

(defwindow-resource window-manager-system-menu ()
  :make-window
  (menu
    :dynamic t
    :pop-up t
    :item-list *window-manager-system-menu-items*
    :save-bits nil)
  :reusable-when :deexposed)

;;;----------------------------------------------------------------------
;;; Actual operations

(defun  move-window-op (window m-x m-y)
  "Window-op handler to move a window."
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (mouse-set-window-position window))

(defun reshape-window-op (window m-x m-y)
  "Window-op handler to reshape a window."
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (mouse-set-window-size window))

(defun create-window-op (window m-x m-y)
  "Window-op handler to create a new window."
  (declare (ignore window m-x m-y))
  (system-menu-create-window))

(defun kill-window-op (window m-x m-y)
  "Window-op handler to kill a window."
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (send window :kill))

(defun bury-window-op (window m-x m-y)
  "Window-op handler to bury a window"
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (send window :bury))

(defun expand-window-op (window m-x m-y)
  "Window-op handler to expand a window"
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (expand-window window))

(defun move-edge-window-op (window m-x m-y &aux window-edge-alist)
  "Window op handler to move one edge of a window"
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (setf window (or (send window :superior) window))
  (setf window-edge-alist (get-window-edge-alist window))
  (update-edges (tv:sec-single-move window window-edge-alist)
		window-edge-alist))

(defun move-multi-edge-window-op (window m-x m-y &aux window-edge-alist)
  "Wdinow op handler to move a group of edges."
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (setf window (or (send window :superior) window))
  (setf window-edge-alist (get-window-edge-alist window))
  (update-edges (tv:sec-multiple-move window window-edge-alist)
		window-edge-alist))

(defun snap-window-op (window m-x m-y)
  "Wdinow op handler to make a snapshot of a screen region."
  (declare (ignore m-x m-y))
  (setf window (find-right-sheet window))
  (setf window (snap-region window))
  (when window
    (send window :expose)
    (mouse-set-window-position window)))

;;;----------------------------------------------------------------------
;;;  Helper functions

(defun update-edges (new-edges old-edges)
  "Change the windows in NEW-EDGES to reflect their new shapes, if
   changed from OLD-EDGES."
  (dolist (new new-edges)
    (let ((old (assoc (first new-edges) old-edges)))
      (or (equal (cddr old) (cddr new))	   ;Edges not the same?
	  (multiple-value-bind (win lose)
	      (send (first new) :set-edges (third new) (fourth new)
		    (fifth new) (sixth new) :verify)
	    (if win (apply (car new) :set-edges (cddr new))
		(beep)
		(tv:pop-up-format "Illegal edges for ~S:~%~A"
			       (car new) lose)))))))

(defun mouse-call-standard-system-menu ()
  "Invoke the standard Explorer System Menu."
  (process-run-function '(:name "System Menu" :priority 10.)
			#'(lambda ()
			    (using-resource (menu system-menu mouse-sheet)
			      (funcall menu ':choose)))
			)
  )

(defvar *types-of-top-level-windows* '(tv:screen))

;;;Edited by LOW                   12 Jan 90  18:02
(defun find-right-sheet (window)
  "Get the right sheet to operate on.  If HYPER is down, window (which
   may be a pane in a frame) is right, else the frame itself (inferior to
   mouse sheet) is it."
  (if (not (key-state :hyper))		   ;climb up the window tree
      (loop for parent = (tv:sheet-superior window)
	    until (find-if #'(lambda (x) (typep parent x)) *types-of-top-level-windows*)
	    do (setq window parent)))
  window)

(provide 'window-manager-system-menu)

;;;=============================================================================

;;; Rich Acuff, Stanford KSL, March 87

;;;  Analog clock windows.  ANALOG-CLOCK is probably the most interesting
;;;  function.

;(require 'transparent-window)
;(require 'deexposed-mouse)
;(require 'window-system-utilities)

;;;Edited by Acuff                 18 Mar 88  12:13
(defflavor analog-clock-mixin
	   (x-pos
	    y-pos
	    (dont-update? nil)
	    (number-font fonts:tvfont)
	    (face-bitmap nil)
	    (radius -1)
	    (fmt-string "~D")
	    (show-seconds? nil)
	    (line-xs (make-array 60 :element-type 'fixnum))
	    (line-ys (make-array 60 :element-type 'fixnum))
	    (scratch-x-array (make-array 2 :element-type 'fixnum))
	    (scratch-y-array (make-array 2 :element-type 'fixnum))
	    )
	   ()
  (:documentation :mixin
  "Analog time display.  Requires that the methods :CLOCK-FACE-RADIUS
   (returns radius of face) and :CLOCK-FACE-POSITION (returns two
   values, the x and y position of the center of the clock face in
   SELF).  Someone must call either the :UPDATE-TIME or the
   :UPDATE-HANDS methods.  Stashes a `virgin' clock face bitmap, and then
   copies it to the window and draws the hands on each refresh.")
  (:required-methods :clock-face-radius :clock-face-position)
  (:required-flavors graphics-mixin stream-mixin)
  (:initable-instance-variables)
  (:gettable-instance-variables)
  (:settable-instance-variables)
  )

;;; Clock face methods

(defmethod (analog-clock-mixin :display-clock-face) ()
  "Draw the clock face"
  (let ((new-radius (send self :clock-face-radius)))
    (multiple-value-setq (x-pos y-pos)
      (send self :clock-face-position))
    (unless (= new-radius radius)
      (send self :best-font))
    (setf radius new-radius)
    (let-globally ((dont-update? t))
      (send self :draw-filled-circle x-pos y-pos radius white)
      (send self :draw-circle x-pos y-pos radius)
      (send self :draw-numbers)
      ;; Now save this "virgin" face
      (send self :save-face)
      )
    )
  )

(defmethod (analog-clock-mixin :draw-numbers) ()
  "Put the numbers on the face of the clock."
  (do ((i 1 (1+ i)))
      ((> i 12))
    (send self :draw-one-number i)
    )
  )

(defmethod (analog-clock-mixin :draw-one-number) (n)
  "Put a number on the face of the clock."
  (let* ((string (format nil fmt-string n))
	 (font (send self :best-font))
	 (box-height (font-baseline font))
	 (slope (tan (clock-to-rads (* 5 (mod n 12)))))
	 box-width x y r
	 )
    (setf box-width (send self :string-length string 0 nil nil font))
    ;; Use pythag to compute how far out the radius to put the box (sort
    ;; of treat the box like a circle with radius = 1/2 diagonal)
    (setf r (- radius (sqrt (+ (expt (/ box-width 2) 2)
			       (expt (/ box-height 2) 2)))
	       (* 1/20 radius)				;fudge
	       )
	  )
    (setf x (sqrt (/ (expt r 2) (+ (expt slope 2) 1))))
    (setf y (abs (* x slope)))
    ;; fix up quadrants
    (setf y (+ y-pos  
	       (case n
		     ((1 2 3 9 10 11 12) (- y))
		     (otherwise y))))
    (setf x (+ x-pos
	       (case n
		     ((1 2 3 4 5) x)
		     (otherwise (- x)))))
    ;; now from center of box in float to screen coords to draw
    (setf y (+ 0 (ceiling (- y (/ box-height 2)))))
    (setf x (+ 0 (ceiling (- x (/ box-width 2)) 1)))
    ;; write the number
    (send self :string-out-explicit string x y nil nil font alu-seta)
    )
  )

(defmethod (analog-clock-mixin :best-font) ()
  ;; Should compute best font based on size, but we'll force user to do
  ;; :SET-NUMBER-FONT.
  (if (typep number-font 'font)
      ;; Then check that it's a good font
      (unless (numberp (tv:font-char-width number-font))
	(load (concatenate 'string "SYS:FONTS;" (string (tv:font-name number-font)))
	      :verbose nil)
	(setq number-font (eval (tv:font-name number-font)))   ;set and return
	)
      ;; Else try again
      (progn
	(setq number-font (eval number-font))
	(send self :best-font)
	)
      )
  number-font
  )

(defmethod (analog-clock-mixin :save-face) ()
  "Stash the just drawn face into FACE-BITMAP, creating or growing it if
   necessary."
  ;; create or make bigger
  (unless (and (typep face-bitmap '(array bit (* *)))
	       ;; height = width, so only check one
	       (>= (array-dimension face-bitmap 0) (* radius 2)))
    (setf face-bitmap (make-bitmap (* 2 radius) (* 2 radius)))
    )
  (send self :bitblt-from-sheet alu-seta
	(* 2 radius) (* 2 radius)
	(- x-pos radius)
	(- y-pos radius)
	face-bitmap 0 0 )
  )

;;;  Hands methods

(defmethod (analog-clock-mixin :update-time) ()
  "Update SELF's display with the current time."
  ;; Wait until exposed
  #+foo(process-wait "Output Hold" #'(lambda (w)
				  (= (sheet-output-hold-flag w) 0))
		self)
  (multiple-value-bind (seconds minutes hours)
      (time:get-time)
    (if seconds
	;; All cool, do it
	(send self :update-hands hours minutes seconds)
	;; No time, so don't do anything
	nil
	)
    )
  )

(defmethod (analog-clock-mixin :update-hands) (h m s)
  "Change the display on SELF to reflect the new values H M and S (S is
   only shown if the i.v. SHOW-SECONDS? is non-NIL)."
  ;; map hour to the clock position it points at
  (setf h (+ (* (mod h 12) 5) (round (* m 5/60))))
  (unless dont-update?
    ;; erase old
    (send self :bitblt alu-seta (* 2 radius) (* 2 radius)
	  face-bitmap 0 0
	  (- x-pos radius) (- y-pos radius))
    (if show-seconds?
	(progn
	  (send self :draw-hand s 1 1)
	  (send self :draw-hand m .8 1)
	  (send self :draw-hand h .4 1)
	  )
	(progn
	  (send self :draw-hand m .9 1)
	  (send self :draw-hand h .6 1)
	  )
	)
    )
  )

(defmethod (analog-clock-mixin :draw-hand) (value ratio line-width)
  "Draw a hand pointing at VALUE that is RAITO * RADIUS long, and
   LINE-WIDTH wide."
  (multiple-value-bind (x y)
      (send self :endpoint value ratio)
    (send self :draw-line x-pos y-pos x y line-width)
    )
  )

(defmethod (analog-clock-mixin :endpoint) (value ratio)
  "Return X and Y of the endpoint of the line going from (X-POS, Y-POS)
   toward the clock value VALUE, RATIO * RADIUS of the distance out."
  (let ((x (abs (round (* (cos (clock-to-rads value)) (* radius ratio)))))
	(y (abs (round (* (sin (clock-to-rads value)) (* radius ratio))))))
    (unless (<= 15 value 45)
      (setf y (- y)))
    (unless (<= 0 value 30)
      (setf x (- x)))
    (values (+ x-pos x) (+ y y-pos))
    )
  )

;;;----------------------------------------------------------------------

;;;  Helper fns

(defun clock-to-rads (n)
  "Map from clock coordinates (0 to 60 starting straight up) to radians
   (0 to 2pi starting straight right)."
  ;; rotate clock face so that 12 o'clock is at 3 o'clock, make the
  ;; numbers go up counterclockwise, and convert to radians
  (* (mod (- 60 (mod (- n 15) 60)) 60) pi 1/30)
  )

(defun run-analog-clock (clock)
  "Run the analog clock CLOCK."
  (send clock :run-clock)
  )

(defun analog-clock (&optional (show-seconds? nil)
		               (edges '(0 0 100 100))
			       (priority 0))
  "Run an analog clock window.  If SHOW-SECONDS? is non-NIL, there will be
   a second hand.  EDGES are the edges of the clock window.  PRIORITY is
   the process priority of the update process that gets created."
  (make-instance 'w:analog-clock
		 :edges edges
		 :process `(run-analog-clock :priority ,priority
					     :name "Analog Clock")
		 :show-seconds? show-seconds?
		 :expose-p t
		 )
  )

(defun square-window (window)
  "Make WINDOW square based on it's minimum size measurement."
  (let ((size (min (send window :inside-height)
		   (send window :inside-width))))
    (send window :set-size size size)
    )
  )

(defun analog-clock-menu-set-parameters (window)
  "Menu of things to do to an analog clock."
  (menu-choose `(("Redraw" :eval (send ,window :refresh) :documentation
		  "Clear and redraw the clock face")
		 ,(if (send window :show-seconds?)
		      `("No Seconds" :eval
			(send ,window :set-show-seconds? nil)
			:documentation "Don't show seconds")
		      `("Seconds" :eval
			(progn
			  (send ,window :set-show-seconds? t)
			  ;; start it happening
			  (when (send ,window :process)
			    (send (send ,window :process) :reset)))
			:documentation "Do show seconds")
		      )
		 ("Square" :eval (square-window ,window) :documentation
		  "Make the window square")
		 ("New Font" :eval
		  (progn
		    (send ,window :set-number-font
			  (or (choose-font) (send ,window :number-font)))
		    (send ,window :refresh))
		  :documentation
		  "Pick a new font for the numbers")
		 )	       
	       )
  )

(defun choose-font ()
  (eval (menu-choose
	  (let ((item-list nil))
	    (do-symbols (sym (find-package "FONTS") item-list)
	      (when (and (boundp sym)
			 (typep (eval sym) 'font))
		(push sym item-list)))))))

;;;----------------------------------------------------------------------

;;;  Instantiable clock

(defflavor analog-clock
	   ((mask nil))
	   (analog-clock-mixin
	    transparent-mixin
	    stream-mixin
	    graphics-mixin
	    process-mixin
	    minimum-window
	    )
  (:default-init-plist :save-bits nil :blinker-p nil)
  (:gettable-instance-variables mask)
  )

(defmethod (analog-clock :clock-face-radius) ()
  (round (truncate (- (min (send self :inside-height)
			   (send self :inside-width))
		      1)
		   2
		   )
	 )
  )

(defmethod (analog-clock :clock-face-position) ()
  (values (floor (/ (send self :inside-width) 2))
	  (floor (/ (send self :inside-height) 2))
	  )
  )

;;; Make sure we stay up to date

;;;Edited by Acuff                 31 Oct 87  14:30
(defmethod (analog-clock :after :refresh) (&rest ignore)
  "Redraw the clock face and time."
  (declare (ignore ignore))
  (unless (or already-refreshing? (not (send self :exposed-p)))
    (send self :display-clock-face)
    (send self :update-time)
    )
  )

(defmethod (analog-clock :run-clock) ()
  (loop
    ;; Might get errors because of async changes--ignore and fix next
    ;; time around
    (if (send self :exposed-p)
	(progn
	  (ignore-errors (send self :update-time))
	  (sleep (if show-seconds? 1 60))
	  )
	;; not exposed, so be mellow; don't show seconds, and cause an
	;; update once a minute
	(let-globally ((show-seconds? nil))
	  (send (sheet-get-screen self) :screen-manage)
	  (sleep 60)
	  )
	)
    )
  )

(defmethod (analog-clock :after :expose) (&rest ignore)
  "Wake up our process if we have one."
  (if (typep (send self :process) 'si:process)
      (send (send self :process) :reset))
  )

;;; Mouse clicks

(defmethod (analog-clock :deexposed-mouse-buttons) (button-mask x y)
  "Call standard :MOUSE-CLICK method."
  (send self :mouse-buttons button-mask x y)
  )

;;;Edited by Acuff                 31 Oct 87  12:20
;;;Edited by Acuff                 16 Nov 87  15:43
(defmethod (analog-clock :mouse-click) (button x y)
  (case button
    ((#\mouse-L-1 #\mouse-L-2)
     (if (send self :exposed-p)
         (process-run-function
	   '(:name "Clock Menu" :priority 0)
	   #'analog-clock-menu-set-parameters self)
	 (send self :expose)
	 )
     )
    ((#\mouse-M-1 #\mouse-M-2)
     (if (and (boundp '*dragging-enabled?*)
	      *dragging-enabled?*
	      (char= #\mouse-m-1 button))
	 (drag-window-from self x y)
	 (process-run-function
	   '(:name "Set Window Position" :priority 1)
	   #'(lambda (window) (mouse-set-window-position window))
	   self)
	 )
     )
    ((#\mouse-R-1 #!z #\mouse-R-2) (mouse-call-system-menu))
    )
  )

(defmethod (analog-clock :deexposed-who-line-documentation-string) ()
  `(:mouse-L-1 "Expose this clock"
    :mouse-M-1 "Move this clock"
    :mouse-R-1 "System Menu")
  )

(defmethod (analog-clock :who-line-documentation-string) ()
  `(:mouse-L-1 "Menu of analog clock operations"
    :mouse-M-1 "Move this clock"
    :mouse-R-1 "System Menu")
  )

(defmethod (analog-clock :screen-manage-deexposed-visibility) ()
  t
  )

(compile-flavor-methods analog-clock)

(provide 'Analog-Clock)

;;;=============================================================================


;;; James Rice, Stanford KSL, 1985

;;; This file contains the definition of the flavor
;;; tv:vertically-ordered-menu-columns.  This is a mixin, which reorders menu
;;; items, which are in a given order so that when the menu splits into t
;;; a number of columns the items are split into columns and not into rows
;;; first.




(defun every-nth (a-list result-list number-of-rows index)
"This function takes a list of menu items, an accumulating result list, the
 number of rows that are going to be displayed and an index, which is used as
 the function iterates along the list.  What the function does is it counts
 along the list until the index reaches one.  As it does so it accumulates all
 of the elements that it passes in the list into the result parameter and
 removes them from the a-list parameter.  When it gets down to an index of one
 it means that it has reached an nth element.  It returns a values pair of this
 element consed onto a list of these values returned from doing the same
 process again on the rest of the list and a list which contains all of the
 elements left over, which were accumulated into the result paramater and then
 reversed, when the a-list argument runs out.  What this means therefore is
 that the value of this function is a pair of lists.  The first is a list of
 all of the elements, which have indexes in the list, which are modulo number
 of rows, i.e. every nth element, and the second returned value is the original
 a-list, in the same order as it was before, only with the elements of the
 first list removed.
"
  (declare (optimize (speed 3) (safety 0)))
  (if (equal nil a-list)
      (Values nil (reverse result-list))
      (if (equal index 1)
	  (multiple-value-bind (list-from-tail result-list-from-tail)
	      (every-nth (rest a-list)
			 result-list
			 number-of-rows
			 number-of-rows
	      )
	      (Values (Cons (First a-list) list-from-tail)
		      result-list-from-tail
	      )
	  )
	  (every-nth (rest a-list) (Cons (First a-list) result-list)
		     number-of-rows (- index 1)
	  )
      )
  )
)


(defun reorder-for-menu-items (a-list number-of-rows)
"This function takes a list and a number of rows and reorders the list so that
 the list will be in columns major order when it is displayed.  It calls
 every-nth, which returns two answers; the first is a list which has every nth
 element in the list and the second is the original list with all of the
 elements in the first list removed.  This function simply calls every-nth
 until it has processed each of the rows in the menu.  It returns a list with
 all of these row sublists glued together.
"
  (if (equal 0 number-of-rows)
      a-list
      (Multiple-value-bind (list-for-first rest-of-list)
   	        (every-nth a-list nil number-of-rows 1)
	   (Append list-for-first
		   (reorder-for-menu-items rest-of-list (- number-of-rows 1))
	   )
      )
  )
)


(defun pad-to-a-multiple (a-list number-of-columns pad-item)
"This function takes a list of menu items and a number of columns into which
 that menu is to be displayed and it then pads that list out with unselectable
 menu items at the end so that the returned list then has a length which is an
 integer multiple of the number of columns in the menu.
"
    (Append a-list
	    (Make-List (rem (- number-of-columns
			       (rem (length a-list) number-of-columns)
			    )
			    number-of-columns
		       )
		       :Initial-Value pad-item
	    )
    )
)


(defun reorder-into-n-columns (a-list number-of-columns pad-item)
"This function takes a list of menu items and the number of columns into which
 they items are to be displayed and it returns a rearranged list, for which the
 items are in such an order that they will appear in a menu of the defined
 number of columns in order in columns from left to right.  The function does
 this by working on a list, which is padded out with unselectable items so that
 it will form a rectangle with no gaps.  This is easier to deal with.
"
    (let ((full-list (pad-to-a-multiple a-list number-of-columns pad-item)))
         (reorder-for-menu-items full-list
		  (quotient (length full-list) number-of-columns)
	 )
    )
)


(defflavor tv:vertically-ordered-menu-columns (tv:old-item-list) ()
    :Abstract-Flavor
    (:Required-Instance-variables tv:Item-List tv:Columns)
    (:Documentation :Mixin
"This is a mixin for menus, which makes sure that itemlists stay in vertical
 alphabetical order when the menus are displayed in multiple columns.
 The normal behaviour is for the itemlist to be displayed in the original order
 but going from left to right and then from top to bottom.  This mixin makes
 the columns more like those in a phone book.
"
    )
)

(export 'tv:vertically-ordered-menu-columns 'tv)

(defmethod (tv:vertically-ordered-menu-columns :After :Init) (init-plist)
"This method takes the itemlist, which has been set up somewhere in the
 initialisation of the menu and rearranges it so that if the items were in,
 let us say, alphabetical order as defined by the user they then end up in a
 modified order so that they will appear in alphabetical order in columns from
 left to right.  This is unlike the normal behaviour, which makes them come out
 in alphabetical order across the columns, not down.
"
  (ignore init-plist)
  (Send Self :Set-Item-List (Send Self :Items-Reordered tv:Item-List))
)


(defmethod (tv:vertically-ordered-menu-columns :Before :Update-item-list) ()
"Remembers the item list so that it can check to see whether it has been changed
 by the :Update-item-list method.
"
  (Setq tv:Old-Item-list tv:Item-List)
)


(defmethod (tv:vertically-ordered-menu-columns :After :Update-item-list) ()
"This method takes the itemlist, which has been set in the update-item-list
 method if this is a dynamic menu and rearranges it so that if the items were
 in, let us say, alphabetical order as defined by the user they then end up in a
 modified order so that they will appear in alphabetical order in columns from
 left to right.  This is unlike the normal behaviour, which makes them come out
 in alphabetical order across the columns, not down.
"
  (if (not (equal tv:Item-List tv:Old-Item-List))
      (Send Self :Set-Item-List (Send Self :Items-Reordered tv:Item-List))
  )
)


(defmethod (tv:vertically-ordered-menu-columns :Items-Reordered) (items)
"This method is passed a list of items in the order that the user thinks that
 they should appear.  It returns a reordered version of the list.
"
  (reorder-into-n-columns items tv::Columns (List "" :No-select nil))
)


;;;-----------------------------------------------------------------------------

(provide 'vertically-ordered-menu-columns)

;;;=============================================================================



;;; This file contains the definitions of two macros called Assign-Using-Menu
;;; and Values-using-Menu.  They are designed to provide a cleaner interface to
;;; choose-variable-values.

tv:
(defun clean-up-choices (margin-choices)
"Taken from the body of choose-variable-values."
  (mapcar #'(lambda (x)
	      (list (if (atom x) x (car x)) nil
		    'choose-variable-values-choice-box-handler
		    nil nil (if (atom x) nil (cadr x))))
	  margin-choices)
)

tv:
(defun just-call-window (window-location superior near-mode margin-choices osw)
"Taken from the body of choose-variable-values."
  (let ((window (symbol-value window-location))
	(processed-message nil)
	(margin-choices (clean-up-choices margin-choices))
       )
       (let ((current-window
	       (if (send superior :operation-handled-p
			 :set-selection-substitute)
		   superior
		   (or osw mouse-sheet)))
	     (old-substitute (send superior :selection-substitute)))
	 (unwind-protect
	     (progn
	       (clear-input window)
	       (delaying-screen-management
		 (expose-window-near window near-mode)
		 (send window :select)  
		 (send current-window :send-if-handles
		       :set-selection-substitute window))
	       (do () (nil)
		 ;; wait for something from the keyboard.
		 (let ((kbd-intercepted-characters
			 choose-variable-values-intercepted-characters))
		   (process-wait "Choose" #'listen window)
		   (and (setq processed-message
			      (choose-variable-values-process-message
				window (read-any window)))
			(return)))))
	   (delaying-screen-management
	     (send window :deactivate)
	     (send current-window :send-if-handles
		   :set-selection-substitute old-substitute)
	     (and osw (send osw :select nil))))
	 (if (eq processed-message 'exit)
	     (execute-margin-choice margin-choices
	       w:margin-choice-completion-string
	       #'ignore)
	     ;;else
	     (if (eq processed-message 'abort)
		 (execute-margin-choice
		   margin-choices
		   w:margin-choice-abort-string
		   #'(lambda ()
		       (signal-condition eh:*abort-object*)))))))
)

tv:
(defmacro w:maybe-ephemeral-cvv-menu
	  (variables &key
	   (permanent t)
	   (function nil)
	   (near-mode ''(:mouse))
	   (label "Choose Variable Values")
	   (width nil)
	   (extra-width 10.)
	   (height nil)
	   (margin-choices nil)
	   (superior nil)
	   (reverse-video-p nil)
	   (value-tab t)
	   (force-permanent nil)
	   (selected-io nil)
	   (foreground-color *default-menu-foreground*)
	   (background-color *default-menu-background*)
	   (label-color      *default-menu-label-foreground*)
	   (label-background *default-menu-label-background*)
	  )
"Just like choose-variable-values, only has a :Permanent arg.  If this is true
 then it organises things so that the cvv menu is only consed once.
"
  (if permanent
     `(let ((window-location ',(prog1 (gensym "CVV-WINDOW-" t) (gensym "G")))
	    (the-superior ,superior)
	    (near-mode ,near-mode)
	    (margin-choices ,margin-choices)
	    (old-allocate #'allocate-resource)
	    (old-deallocate #'deallocate-resource)
	   )
	   (if (boundp window-location)
	       (let ((osw selected-window))
		    (just-call-window window-location the-superior near-mode
				      margin-choices osw
		    )
	       )
	       (letf ((#'allocate-resource
		       #'(lambda (.name. &rest args)
			   (if (equal .name.
				   'tv:temporary-choose-variable-values-window)
			       (progn (setf (symbol-value window-location)
					    (make-window
					      'temporary-choose-variable-values-window
					      :superior
					      (or (first args)
						  tv:default-screen)
					      :edges-from
					      (list
						(sheet-inside-left
						  default-screen)
						(sheet-inside-top
						  default-screen)
						(sheet-inside-right
						  default-screen)
						(sheet-inside-bottom
						  default-screen))
					      :foreground-color
					      *default-menu-foreground*
					      :background-color
					      *default-menu-background*))
				      (symbol-value window-location))
			       (apply old-allocate .name. args))))
		      (#'deallocate-resource
		       #'(lambda (.name. &rest args)
			   (if (equal .name.
				   'tv:temporary-choose-variable-values-window)
			       nil
			       (apply old-deallocate .name. args))))
		     )
		     (choose-variable-values
		       ,variables
		       :function ,function
		       :label ,label
		       :width ,width
		       :superior the-superior
		       :near-mode ,near-mode
		       :margin-choices ,margin-choices
		       :extra-width ,extra-width
		       :height ,height
		       :reverse-video-p ,reverse-video-p
		       :value-tab ,value-tab
		       :force-permanent t;,force-permanent
		       :selected-io ,selected-io
		       :foreground-color ,foreground-color
		       :background-color ,background-color
		       :label-color ,label-color
		       :label-background ,label-background
		     )
	       )
	   )
      )
     `(choose-variable-values
	,variables
	:function ,function
	:label ,label
	:width ,width
	:superior ,superior
	:near-mode ,near-mode
	:margin-choices ,margin-choices
	:extra-width ,extra-width
	:height ,height
	:reverse-video-p ,reverse-video-p
	:value-tab ,value-tab
	:force-permanent ,force-permanent
	:selected-io ,selected-io
	:foreground-color ,foreground-color
	:background-color ,background-color
	:label-color ,label-color
	:label-background ,label-background
      )
  )
)

(defun Process-Variable-Values-Items (an-item)
"This function is called by the choose variable values macros.  It is passed
 an item that the user wants to go into the menu.  The item is one of two
 things.  Either it is a list which has the structure (expression .rest), where
 the expression denotes the initial value/location of the initial value for the
 menu and the rest has an optional string, a keyword and optional arguments. 
 These are defined in the windows manual for choose variable values menus.  The
 other option is for the item to be a string.  This causes a separator to be
 produced in the menu.  This function returns a list, which has six elements
 and can be of two forms.  If the item that this function is passed is not a
 list ie. it is a string then
 (:Throw-Away :Throw-Away :Throw-Away :Throw-Away item :Throw-Away) is
 returned.  This causes the calling macro to throw everything away except the
 string.  If the item is a list then the six items that are returned are as
 follows:-  The first and second are generated symbols.  These will be declared
 special in the macro that calls this function so that Choose-variable-values
 can change its value and so that a special variable holds onto the initial
 value so that the quit option can be Evaled correctly.  Why the code in this
 choice box has to be interpretted I do not know but it's ridiculous in my
 opinion.  The third and fourth are lists denoting Setfs to the generated
 symbols of the initial value passed on by the user.  This will be displayed
 as the initial value by the menu.  The fourth is used as a reinitialisation
 clause in the quit option.  The Fifth is a list that actually goes into the
 argument list of choose variable values.  It specifies the generated symbol as
 the destination for the value returned by the user.  The Sixth value is a
 list denoting a Setf, which reassigns the value returned by the menu to the
 initial expression provided by the user.  This is only used in the side effect
 causing cases such as Assign-Using-Menu.
"
    (if (Consp an-item)
	(let ((Temporary-name-1 (Gensym))
	      (Temporary-name-2 (Gensym))
	     )
	     (List Temporary-name-1
		   Temporary-name-2
	           (List 'Setf Temporary-name-2 (First an-item))
	           (List 'Setf Temporary-name-1 Temporary-name-2)
	           (Cons 'List
			 (Cons (List 'Quote Temporary-name-1)
			       (if (Stringp (Second an-item))
			           (Cons (Second an-item)
				         (Rest (Rest an-item))
			           )			           
			           (Rest an-item)
			       )
			 )
		   )
	           (List 'Setf (First an-item) Temporary-name-1)
	     )
	)
	(List :Throw-Away :Throw-Away :Throw-Away
	      :Throw-Away an-item     :Throw-Away
	)
    )
)


(defun defaultise-choose-variable-values-keywords (keywords)
"Gives a useful set of defaults for the CVV keywords."
  (let ((selected-io-to-add (if (member :Selected-IO keywords)
				nil
				#+TI '(:Selected-io nil)
				#+Symbolics nil
			    )
	)
	(superior-to-add (if (member :Superior keywords)
			     nil
			     #+TI '(:Superior tv:main-screen)
			     #+Symbolics nil
			 )
	)
       )
       (append selected-io-to-add superior-to-add keywords)
  )
)


(defmacro tv:Assign-using-Menu (Items &Rest Keywords)
"This is a macro that parcels up a call to Choose-Variable-Values so that the
 user does not have to worry about defining any Special symbols to hold the
 values used by the menu.  It takes a menu specification like
 tv:choose-variable-values ie. a list of items &rest option keywords.  Calling
 this construct has the side effect of assigning the chosen values back to the
 items that as the specifications for the initial values, ie. the heads of the
 items within the item list.  The item list should not be quoted. This is done
 by mapping the user's arguments into the following code :-
    (Assign-using-Menu ((Item-one \"Item One\" :Expression)
			'A separator'
			(Item-two \"Item Two\" :String)
		       )
		       :Label \"A Name\"
    )

==>

   (Funcall #'(Lambda ()
		(Declare (Special #Genedsym1 #Genedsym1-2
				  #Genedsym2 #Genedsym2-2
			 )
		)
	        (Setf #Genedsym1-2 Item-one)
	        (Setf #Genedsym1 #Genedsym1-2)
	        (Setf #Genedsym2-2 Item-two)
	        (Setf #Genedsym2 #Genedsym2-2)
		(maybe-ephemeral-cvv-menu
		    (List (List '#Genedsym1 \"Item One\" :Expression)
			  'A separator'
			  (List '#Genedsym2 \"Item Two\" :String)
		    )
		    :Margin-Choices
			'((\"Abort []\"
			   '(Progn (Setf #Genedsym1 #Genedsym1-2)
				   (Setf #Genedsym2 #Genedsym2-2)
			    )
			  )
                          \"Do it []\"
			 )			
		    :Label \"A Name\"
		)
	        (Setf Item-one #Genedsym1)
	        (Setf Item-two #Genedsym2)
	    )
    )
"
  (let ((Processed-Items (Mapcar #'Process-Variable-Values-Items Items)))
     `(Funcall ;; Call the automatically generated closure.
	#'(Lambda() ;; Define a closure in which to put the special variables
	 (Declare (Special ;; Construct a special declaration for the
			   ;; generated symbols.
		     ,@(Remove :Throw-Away ;; Throw away values for separators
				(Mapcar #'(Lambda (an-item) (First an-item))
				        Processed-Items
				)
		       )
		     ,@(Remove :Throw-Away ;; Throw away values for separators
				(Mapcar #'(Lambda (an-item) (Second an-item))
				        Processed-Items
				)
		       )
		  )
	 )
	 ;; Construct a set of initialisation Setfs
	 ,@(Remove :Throw-Away ;; Throw away values for separators
		  (Mapcar #'(Lambda (an-item) (Third  an-item)) Processed-Items)
	   )
	 ,@(Remove :Throw-Away ;; Throw away values for separators
		  (Mapcar #'(Lambda (an-item) (Fourth an-item)) Processed-Items)
	   )
	 ;; Get the user to choose from the menu.
	 (let ((result
		(condition-case nil
		 (Catch :Abort-Menu
		   (maybe-ephemeral-cvv-menu
		    ,(Cons 'List
			   (Mapcar #'(Lambda (an-item) (Fifth an-item))
				   Processed-Items
			   )
		     )
		    :Margin-Choices
			(List (List "Abort []"
				    '(progn
				       ;; Reset the values of the #1 symbols.
					,@(Remove :Throw-Away
						(Mapcar #'(Lambda (an-item)
								(Fourth an-item)
							)
							Processed-Items
						)
					   )
					   (throw :Abort-Menu :Abort-Menu)
				    )
			      )
			      "Do it []"
			)
		    ,@(defaultise-choose-variable-values-keywords Keywords)
		   )
		 )
		 (sys:abort :Abort-Menu)
		)
	       )
	      )
	      ;; Construct a set of terminating Setfs
	      ,@(Remove :Throw-Away ;; Throw away values for separators
			(Mapcar #'(Lambda (an-item) (Sixth an-item))
				Processed-Items
		        )
	        )
	      result
	 )
        )
      )
  )
)

(export 'tv:Assign-using-Menu 'tv)

(defmacro tv:Values-using-Menu (Items &Rest Keywords)
"This is a macro that parcels up a call to Choose-Variable-Values so that the
 user does not have to worry about defining any Special symbols to hold the
 values used by the menu.  It takes a menu specification like
 tv:choose-variable-values ie. a list of items &rest option keywords.  Calling
 this construct causes a value to be returned, which is a set of multiple
 values in the same order as the assignable items, ie. non-separators in the
 item list.  The item list should not be quoted. This is done by mapping the
 user's arguments into the following code :-
    (Values-using-Menu ((Item-one \"Item One\" :Expression)
			'A separator'
			(Item-two \"Item Two\" :String)
		       )
		       :Label \"A Name\"
    )

==>

   (Funcall #'(Lambda ()
		(Declare (Special #Genedsym1 #Genedsym1-2
				  #Genedsym2 #Genedsym2-2
			 )
		)
	        (Setf #Genedsym1-2 Item-one)
	        (Setf #Genedsym1 #Genedsym1-2)
	        (Setf #Genedsym2-2 Item-two)
	        (Setf #Genedsym2 #Genedsym2-2)
		(maybe-ephemeral-cvv-menu
		    (List (List '#Genedsym1 \"Item One\" :Expression)
			  'A separator'
			  (List '#Genedsym2 \"Item Two\" :String)
		    )
		    :Margin-Choices
			'((\"Abort []\"
			   '(Progn (Setf #Genedsym1 #Genedsym1-2)
				   (Setf #Genedsym2 #Genedsym2-2)
			    )
			  )
                          \"Do it []\"
			 )			
		    :Label \"A Name\"
		)
	        (Values #Genedsym1 #Genedsym2)
	    )
    )
"
  (let ((Processed-Items (Mapcar #'Process-Variable-Values-Items Items)))
     `(Funcall ;; Call the automatically generated closure.
	#'(Lambda() ;; Define a closure in which to put the special variables
	 (Declare (Special ;; Construct a special declaration for the
			   ;; generated symbols.
		     ,@(Remove :Throw-Away ;; Throw away values for separators
				(Mapcar #'(Lambda (an-item) (First an-item))
				        Processed-Items
				)
		       )
		     ,@(Remove :Throw-Away ;; Throw away values for separators
				(Mapcar #'(Lambda (an-item) (Second an-item))
				        Processed-Items
				)
		       )
		  )
	 )
	 ;; Construct a set of initialisation Setfs
	 ,@(Remove :Throw-Away ;; Throw away values for separators
		  (Mapcar #'(Lambda (an-item) (Third  an-item)) Processed-Items)
	   )
	 ,@(Remove :Throw-Away ;; Throw away values for separators
		  (Mapcar #'(Lambda (an-item) (Fourth an-item)) Processed-Items)
	   )
	 ;; Get the user to choose from the menu.
	 (condition-case nil
	   (Catch :Abort-Menu
	     (maybe-ephemeral-cvv-menu
	      ,(Cons 'List
		     (Mapcar #'(Lambda (an-item) (Fifth an-item))
			     Processed-Items
		     )
	      )
	      :Margin-Choices
		  (List (List "Abort []"
			      '(Progn ;; Reset the values of the #1 symbols.
				  ,@(Remove :Throw-Away
					  (Mapcar #'(Lambda (an-item)
							  (Fourth an-item)
						  )
						  Processed-Items
					  )
				     )
				     (Throw :Abort-Menu :abort-menu)
			      )
			)
			"Do it []"
		  )
	       ,@(defaultise-choose-variable-values-keywords Keywords)
	     )
	   )
	   (sys:abort ,@(Remove :Throw-Away
				(Mapcar #'(Lambda (an-item)
					    (Fourth an-item)
					  )
					Processed-Items
			        )
		        )
		        :Abort-Menu
           )
	 )
	 ;; Return the values generated by the menu as multiple values.
	 (Values ,@(Remove :Throw-Away ;; Throw away values for separators
			   ;; This is the same as the list of specials.
			   (Mapcar #'(Lambda (an-item) (First an-item))
			     	   Processed-Items
			   )
		   )
	 )
        )
      )
  )
)

(export 'tv:Values-using-Menu 'tv)

;;;-----------------------------------------------------------------------------

(provide 'Choice-Facility-Enhancements)

;;;======================================================================

;;; Rich Acuff, Stanford KSL, Oct '87

(defvar *dragging-enabled?* nil
  "   When non-NIL enables dragging of windows.  Ie., 'grab' a window by
   pressing and holding a button, move the ghost box that will pop up to
   the new position, and release the button.")

(defun mouse-drag-window (window &optional (move-p t) (mx mouse-x) (my mouse-y))
  "   Drag a window around using the mouse.  As long as a button is down
   a ghost box tracks the mouse, and, if MOVE-P is non-NIL, moves WINDOW
   to the position of the ghost box when the button is released.  WINDOW
   is not moved if the position is illegal.  The values returned are the
   new X and Y values of the new upper left corner.  MX and MY are the
   position of the mouse that should be used as a starting point."
  (declare (values x y))
  (let ((wx (send window :x-offset))
	(wy (send window :y-offset))	;WINDOW's coords in SUPERIOR
	sx sy			       	;SUPERIOR's coords in MOUSE-SHEET
	mwx mwy				;WINDOW's coords in MOUSE-SHEET
	mxoff myoff			; Mouse offset inside ghost boxx
	(superior (send window :superior))
	)
    ;; Assumes MOUSE-SHEET doesn't change during execution
    (unless (sheet-me-or-my-kid-p superior mouse-sheet)
      (error "Attempt to set position of ~S, which is not inferior to MOUSE-SHEET"
	     window))
    (multiple-value-setq (sx sy)	; (0,0) most of the time
      (sheet-calculate-offsets superior mouse-sheet))
    (multiple-value-setq (mwx mwy)	; Where on the MOUSE-SHEET is WINDOW?
      (sheet-calculate-offsets window mouse-sheet))
    ;; Offsets so ghost exposes on top of WINDOW if mouse hasn't been moved
    (setq mxoff (- mx mwx)   myoff (- my mwy))
    (with-mouse-grabbed
      (without-interrupts
	;; Create a mouse blinker that is the same size as WINDOW 
	(mouse-set-blinker-definition
	  :box-stay-inside-blinker mxoff myoff nil
	  :set-size (send window :width) (send window :height))
	(setq who-line-mouse-grabbed-documentation
	      "Release mouse button to set position"))
      (mouse-warp 0 0 t)		;Get the blinker to the mouse
      (blinker-set-visibility mouse-blinker t)
      ;; Now wait for the mouse button to come up
      (process-wait "Button Up" #'(lambda () (zerop mouse-last-buttons)))
      (multiple-value-setq (mx my) (send mouse-blinker :read-cursorpos))  ; New position
      (setq wx (+ sx mx) wy (+ sy my))	;Convert to SUPERIOR's coords
      (blinker-set-visibility mouse-blinker nil)	; Turn off the ghost box
      (cond ((send window :set-position wx wy :verify)	; Ok to move?
	     (when move-p
	       (send window :set-position wx wy))
	     (return-from mouse-drag-window wx wy))
	    (t						; Invalid postion
	     (setq who-line-mouse-grabbed-documentation
		   (format nil "Can't move ~A to that spot." window))
	     (beep)
	     (sleep 1))			; Let user see error msg
	    )
      )
    )
  )

;;;----------------------------------------------------------------------
;;; System functions hacked (RDA) to know about drag moving

;;;TI Code: Do drag move by default
tv:
;;;Edited by Acuff                 16 Nov 87  15:43
(DEFMETHOD (ESSENTIAL-MOUSE :MOUSE-CLICK) (BUTTONS x y)
  ; BUTTONS can be character or integer.
  (COND ((AND (= BUTTONS #\MOUSE-L-1)
	      (NOT (SEND (SEND SELF :ALIAS-FOR-SELECTED-WINDOWS)
			 :SELF-OR-SUBSTITUTE-SELECTED-P))
	      (GET-HANDLER-FOR SELF :SELECT))	;paper over a bug
	 (MOUSE-SELECT SELF)
	 T)
	(T
	    (OR (SEND SELF :SEND-IF-HANDLES :FORCE-KBD-INPUT
		   `(:MOUSE-BUTTON ,(MERGE-SHIFT-KEYS BUTTONS) ,SELF ,X ,Y))
	     (AND (= BUTTONS #\MOUSE-R-1)
		  (MOUSE-CALL-SYSTEM-MENU)
		  T)
	     ;; This clause added by RDA
	     (and w:*dragging-enabled?*
		  (= buttons #\mouse-m-1)	;Drag move
		  (w:drag-window-from self x y))
	     (BEEP)))))				;click not handled

;;;TI Code: Drag on Mouse-M-H in UCL
tv:
(if (intersection '(:release-4 :release-5) *features*)
;;; TI Code for rel 4 from SYS:PATCH.WINDOW;PATCH-4-67.LISP#5
 (DEFUN-RH alternate-rubout-handler ()
  (LET ((CH) (CH-CHAR) (CH-CONTROL-META) (COMMAND) (keys-read)
	(FILL-POINTER (RH-FILL-POINTER))
	(TYPEIN-POINTER (RH-TYPEIN-POINTER))
	(INITIAL-ENTRY (RHB-INITIAL-ENTRY))
	(RUBBED-OUT-SOME? NIL)
	(INITIAL-INPUT (CADR (ASSOC :INITIAL-INPUT RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	(INITIAL-INPUT-POINTER (CADR (ASSOC :INITIAL-INPUT-POINTER RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	(NUMERIC-ARG NIL)
	(NUMERIC-ARG-NEGATIVE NIL))
    (SETF (RHB-INITIAL-ENTRY) NIL)
    ;; Kludge #1.  If this is the first time this rubout handler has been invoked
    ;; in this stream, then we must create the input history.
    (OR (RH-INPUT-RING)
	(WHEN (FBOUNDP 'ZWEI:MAKE-HISTORY)
	  (SETF (RH-INPUT-RING) (RH-MAKE-INPUT-RING))))
    (WHEN INITIAL-ENTRY
      ;; save the previous input on the input history,
      ;; unless the previous read said not to save it.
      (COND ((AND (NOT (RH-DONT-SAVE-FLAG))
		  TYPEIN-POINTER
		  (NEQ INITIAL-ENTRY :RESTORED)
		  (NOT (ZEROP TYPEIN-POINTER)))
	     ;; only add the contents if it is different from the last entry, and
	     ;; the entry is at least 2 characters long.
	     (SETF (FILL-POINTER RUBOUT-HANDLER-BUFFER) TYPEIN-POINTER)
	     (WHEN (AND (> TYPEIN-POINTER 1)
			(FBOUNDP 'ZWEI:HISTORY-LATEST-ELEMENT)
			(MISMATCH RUBOUT-HANDLER-BUFFER
				  (ZWEI:HISTORY-LATEST-ELEMENT (RH-INPUT-RING))))
	       (ZWEI:PUSH-ON-HISTORY (SUBSEQ RUBOUT-HANDLER-BUFFER 0 TYPEIN-POINTER)
				     (RH-INPUT-RING)))
	     (SETF (FILL-POINTER RUBOUT-HANDLER-BUFFER) FILL-POINTER)))
      ;; Then initialize the typein pointer.
      (SETF (RH-TYPEIN-POINTER) FILL-POINTER)
      (SETQ TYPEIN-POINTER FILL-POINTER)
      ;; Gobble the initial input if any.
      (WHEN (AND INITIAL-INPUT (NEQ INITIAL-ENTRY :RESTORED))
	(RH-INSERT-STRING INITIAL-INPUT 0 NIL NIL NIL)
	(SETQ RUBBED-OUT-SOME? T FILL-POINTER (RH-FILL-POINTER))
        (SETQ INITIAL-INPUT-POINTER
	      (MAX (MIN (OR INITIAL-INPUT-POINTER TYPEIN-POINTER)
			(LENGTH RUBOUT-HANDLER-BUFFER)) 0))
	(SETF TYPEIN-POINTER INITIAL-INPUT-POINTER
	      (RH-TYPEIN-POINTER) INITIAL-INPUT-POINTER)
 	(RH-SET-POSITION TYPEIN-POINTER))
      ;; Record whether this unit of input should be saved on the history.
      (SETF (RH-DONT-SAVE-FLAG)
	    (OR (CADR (ASSOC :DONT-SAVE RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
		(CADR (ASSOC :NO-INPUT-SAVE RUBOUT-HANDLER-OPTIONS :TEST #'EQ))))
      )
    ;;; Can this ever go off? :pass-though now only allows non-bucky. -- mly ;;;
    ;; Kludge #5.  We can't echo or rub out a bucky char or a bli,
    ;; so if the last char inserted was a either of those
    ;; and it did not terminate the input, flush it.
    (AND (NOT (ZEROP TYPEIN-POINTER))
	 (OR (CONSP (AREF RUBOUT-HANDLER-BUFFER (1- TYPEIN-POINTER)))
	     (Not (Zerop (CHAR-BITS (Aref RUBOUT-HANDLER-BUFFER (1- Typein-Pointer))))))
	 (SETQ TYPEIN-POINTER (SETF (RH-TYPEIN-POINTER) (DECF (RH-FILL-POINTER)))))
    ;; Kludge #4.  After resuming a Break, the stream's cursorpos is wrong.
    ;; In fact, the cursor is at the end of the string in that case.
    ;; So, if it is supposed to be elsewhere, move it.
    ;; This condition also avoids wasting time when we are reading typein
    ;; at the end of the string.
    (OR (= FILL-POINTER TYPEIN-POINTER)
	(RH-CURSOR-MOTION TYPEIN-POINTER))
    ;; In case we had to return to the caller with a EDITING-COMMAND char
    ;; while RUBBED-OUT-SOME? was T, make things consistent again
    ;; by causing a rescan now.
    (WHEN (AND (NOT INITIAL-ENTRY)
	       (= (RH-SCAN-POINTER) MOST-POSITIVE-FIXNUM))
      (SETF (RH-SCAN-POINTER) 0)
      (THROW 'RUBOUT-HANDLER T)) 
    (CATCH 'RETURN-CHARACTER
      (WHEN RUBOUT-HANDLER-ACTIVATION-CHARACTER
	(THROW 'RETURN-CHARACTER
		(PROG1 RUBOUT-HANDLER-ACTIVATION-CHARACTER
		       (SETQ RUBOUT-HANDLER-ACTIVATION-CHARACTER NIL))))
      ;; Read characters.  If an ordinary character is typed and nothing has been rubbed out,
      ;; return immediately.  Otherwise, let all editing operations complete
      ;; before returning. 
      (DO (*LAST-COMMAND-TYPE*
	   *CURRENT-COMMAND-TYPE*
	   *RUBOUT-HANDLER-MARK*
	   (EDITING-COMMAND    (CDR (ASSOC :EDITING-COMMAND RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	   (DO-NOT-ECHO        (CDR (ASSOC :DO-NOT-ECHO     RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	   (PASS-THROUGH       (CDR (ASSOC :PASS-THROUGH    RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	   (COMMAND-HANDLER    (ASSOC :COMMAND     RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
	   (PREEMPTABLE        (ASSOC :PREEMPTABLE RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
	   (ACTIVATION-HANDLER (ASSOC :ACTIVATION  RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	  (NIL)
	;; Read a character from the stream after bypassing ourself.
	(SETQ CH
	      (LET ((RUBOUT-HANDLER NIL)
		    (TV:*DEFAULT-READ-WHOSTATE* (IF NUMERIC-ARG
						    (FORMAT NIL "Arg = ~d" numeric-arg)
						  "Keyboard")))
		(READ-ANY SELF)))
	(IF (LISTP CH)
	    (COND ((EQ (CAR CH) 'REDISPLAY-RUBOUT-HANDLER)
		   (SEND SELF :SET-CURSORPOS
			 PROMPT-STARTING-X PROMPT-STARTING-Y)
		   (SEND SELF :CLEAR-EOL)
		   (RH-REPRINT-INPUT NIL T))
		  (PREEMPTABLE
		   (MULTIPLE-VALUE-BIND (STRING INDEX)
		       (SEND SELF :SAVE-RUBOUT-HANDLER-BUFFER)
		     (SETQ OLD-TYPEAHEAD (LIST STRING INDEX)))
		   ;; Save the text, rub it all out, and unread the blip.
		   ;; The :FULL-RUBOUT option will cause the RH to return
		   ;; to the caller who will then read the blip.
		   (UNREAD-ANY CH SELF)         ;Should this be UNREAD-CHAR??  It was :UNTYI.
		   (RH-COM-CLEAR-INPUT NIL)
		   (SETF (RH-SCAN-POINTER) 0)
		   (THROW 'RETURN-FROM-RUBOUT-HANDLER
			   (VALUES CH (CADR PREEMPTABLE))))
		  ((AND (EQ (CAR CH) :MOUSE-BUTTON)
			(EQL (CADR CH) #\MOUSE-3-1))
		   (MOUSE-CALL-SYSTEM-MENU))
		  ;; This clause added by RDA for dragging
		  ((and w:*dragging-enabled?*
			(eq (first ch) :mouse-button)
			(char= (second ch) #\mouse-2-1))
		   (w:drag-window-from self (fourth ch) (fifth ch))))
	  (PROGN                                ;CH is a character, not a blip.
            (WHEN (AND (EQL CH #\)) *RH-MATCHING-PAREN-OPTIONS*
                       (< (RH-TYPEIN-POINTER) (CADDR *RH-MATCHING-PAREN-OPTIONS*)))
              (RH-BLINK-RIGHT-PAREN))             ;Cause matching parens to blink.
            ;;This code handles auto-completion on UCL command names and special typein modes.
            ;;Auto-completion used to be implemented as an Input Edtor command, but
            ;;this was inefficient on large typed expressions because each SPACE would cause 
            ;;re-reading of the input.
            (WHEN (AND (EQL CH #\SPACE)
                       ucl:command-loop-typein?  ;;Auto-complete only during ucl:READ-FOR-UCL.
                       user:auto-complete?  ;;Users can turn off auto-completion.
                       (>= (RH-TYPEIN-POINTER) (RH-FILL-POINTER))
                       (PLUSP (RH-TYPEIN-POINTER))
                       (NOT (CHAR= #\SPACE (AREF RUBOUT-HANDLER-BUFFER (1- (RH-TYPEIN-POINTER))))))
              (FUNCALL (complete-word-function rh-completion-handler)
                       :RECOGNITION (LIST 'complete-lisp-atom :auto-complete-p) T)
              (SETQ RUBBED-OUT-SOME? T))  ;;Force rescan of completed input.
            
            (SETQ CH-CHAR (CHAR-CODE CH))       ;WARNING - CH-CHAR is a Fixnum !!
            (SETQ CH-CONTROL-META (CHAR-BITS CH))
            (IF (AND (ZEROP ch-control-meta) (MEMBER CH PASS-THROUGH))
                (SETQ command NIL)
                ;;Look up the key in the RH command table, fetching more keys as necessary (in
                ;;case the user has bound RH commands to keystroke sequences greater than 1).
                ;;If the look-up fails, we have only read one key, and we are doing type-in for a
                ;;UCL aplication, try looking up the key in the UCL's tables.  If a command is found
                ;;or there are several commands found starting with the key, let the UCL
                ;;application preempt us to process the command.
                (PROGN
                  (MULTIPLE-VALUE-SETQ (command keys-read)
                    (ucl-rh-lookup-character ch))
                  (WHEN (AND (NULL command) ucl:command-loop-typein? preemptable (NULL (CDR keys-read)))
                    (WHEN (AND (SETQ command (SEND ucl:command-loop-typein? :lookup-keys keys-read))
                               (NEQ command :FETCH-MORE-KEYS)
                               (SEND command :GET :CANT-PREEMPT))
                      ;;If this command can't preempt typed expressions, don't try to.
                      (SETQ command NIL)))))
            (COND
              ((AND COMMAND-HANDLER
                    (APPLY (CADR COMMAND-HANDLER) CH (CDDR COMMAND-HANDLER)))
               (SETF (RH-SCAN-POINTER) 0)
               (THROW 'RETURN-FROM-RUBOUT-HANDLER
                      (VALUES
                        `(:COMMAND ,CH ,(* (OR NUMERIC-ARG 1)
                                           (IF NUMERIC-ARG-NEGATIVE -1 1)))
                        :COMMAND)))
              ((OR (MEMBER CH DO-NOT-ECHO)
                   (AND ACTIVATION-HANDLER
                        (APPLY (CADR ACTIVATION-HANDLER) CH (CDDR ACTIVATION-HANDLER))))
               (RH-SET-POSITION (RH-FILL-POINTER))
               (LET ((VALUE
                       (IF (MEMBER CH DO-NOT-ECHO) CH
                           `(:ACTIVATION ,CH ,(* (OR NUMERIC-ARG 1)
                                                 (IF NUMERIC-ARG-NEGATIVE -1 1))))))
                 (COND (RUBBED-OUT-SOME?
                        ;; Why isn't this done in the :RUBOUT-HANDLER method loop?
                        (SETQ RUBOUT-HANDLER-ACTIVATION-CHARACTER VALUE)
                        (SETF (RH-SCAN-POINTER) 0)
                        (THROW 'RUBOUT-HANDLER T))
                       (T (THROW 'RETURN-CHARACTER VALUE)))))
              ;; Don't touch this character, just return it to caller.
              ((OR (MEMBER CH EDITING-COMMAND)
                   (SI:ASSQ-CAREFUL CH EDITING-COMMAND))
               ;; Cause rubout handler rescan next time the user does :TYI.
               (If RUBBED-OUT-SOME? (SETF (RH-SCAN-POINTER) MOST-POSITIVE-FIXNUM))
               (RETURN CH))
              ;;Handle Control-number and Control-U specially.  This code was moved forward in
              ;;order to catch numeric args before the UCL application's num.  arg command claimes
              ;;the keystroke.  Note that CH-CHAR is a fixnum, CH is a Character.
              ((AND (NOT (ZEROP CH-CONTROL-META)) (<= (Char-Code #\0) CH-CHAR (Char-Code #\9)))
               (SETQ NUMERIC-ARG (+ (* (OR NUMERIC-ARG 0) 10.) (- CH-CHAR (Char-Code #\0)))))
              ((CHAR= CH #\CONTROL-U)
               (SETQ NUMERIC-ARG (* (OR NUMERIC-ARG 1) 4)))
              ((AND (NOT (ZEROP CH-CONTROL-META)) (= CH-CHAR (Char-Code #\-)))
               (IF NUMERIC-ARG
                   (SEND SELF :BEEP)
                   (SETQ NUMERIC-ARG-NEGATIVE (NOT NUMERIC-ARG-NEGATIVE))))
              (command
               (IF (NOT (TYPEP command 'rh-command))
                   ;;Keystroke was found in a UCL application's command-table.  Have the read
                   ;;function return the values '(:KEYSTROKE <ch>) :MOUSE-CHAR to indicate that
                   ;;we are interrupting typein to execute another command.  <ch> will be
                   ;;processed as if it were typed at top level.
                   (IO-BUFFER-PUSH IO-BUFFER (LIST :KEYSTROKE (CHAR-INT ch)))
                   ;; A standard rh editing command of some sort. The RUBBED-OUT-SOME bit can only
                   ;; be cleared by entering this function again.  The function is passed the
                   ;; numeric argument, and returns T if we are going to need to throw out (like
                   ;; DIS-ALL in the editor).
                   (PROGN
                     (SETQ *LAST-COMMAND-TYPE* *CURRENT-COMMAND-TYPE*
                           *CURRENT-COMMAND-TYPE* NIL)
                     (SETQ RUBBED-OUT-SOME?
                           (OR (SEND command :execute self
                                     (* (OR numeric-arg 1) (IF numeric-arg-negative -1 1))
                                     NIL)
                               RUBBED-OUT-SOME?))
                     (SETQ NUMERIC-ARG NIL NUMERIC-ARG-NEGATIVE NIL)
                     ;; If the buffer is empty and the :FULL-RUBOUT option is active, then throw now
                     ;; This will throw if the user types Rubout or ClearScreen immediately after
                     ;; entering the read function.  It is important that we check for this here
                     ;; and not in RH-DELETE-STRING since some commands, such as Yank-Pop, may
                     ;; temporarily empty the buffer.  It wouldn't be the right thing to throw
                     ;; if the buffer only contained whitespace since it is the responsibility
                     ;; of the caller to discard whitespace when looking for special characters.
                     (COND ((AND (ZEROP (RH-FILL-POINTER))
                                 (ASSOC :FULL-RUBOUT RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
                            ;; This SETF should be done in the :RUBOUT-HANDLER method loop.
                            (SETF (RH-SCAN-POINTER) 0)
                            (THROW 'RUBOUT-HANDLER T))))))
              ;;Beep and ignore.
              ((NOT (ZEROP ch-control-meta))
               (SEND SELF :beep)
               (SETQ numeric-arg nil numeric-arg-negative nil))
              
              ;; Self-inserting character.  Set RUBBED-OUT-SOME since if we return,
              ;; we were typing in the middle of the line.  Typing at the end of the
              ;; line throws to RETURN-CHARACTER.
              (T (UNLESS NUMERIC-ARG-NEGATIVE
                   (RH-INSERT-CHAR CH (OR NUMERIC-ARG 1) RUBBED-OUT-SOME?)
                   (SETQ RUBBED-OUT-SOME? T))  
                 (SETQ *LAST-COMMAND-TYPE* *CURRENT-COMMAND-TYPE*
                       *CURRENT-COMMAND-TYPE* NIL
                       *RUBOUT-HANDLER-MARK* NIL)
                 (SETQ NUMERIC-ARG NIL NUMERIC-ARG-NEGATIVE NIL)))
	    ;; on some occasions we just want to know that something changed
	    ;;  PMH 3/25
	    (when (and RUBBED-OUT-SOME? rh-rescan-any-change)
	      (SETF (RH-SCAN-POINTER) 0)
	      (THROW 'RUBOUT-HANDLER T))
	    ))))))

 (DEFUN-RH alternate-rubout-handler ()
  (LET ((CH) (CH-CHAR) (CH-CONTROL-META) (COMMAND) (keys-read)
	(FILL-POINTER (RH-FILL-POINTER))
	(TYPEIN-POINTER (RH-TYPEIN-POINTER))
	(INITIAL-ENTRY (RHB-INITIAL-ENTRY))
	(RUBBED-OUT-SOME? NIL)
	(INITIAL-INPUT (CADR (ASSOC :INITIAL-INPUT RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	(INITIAL-INPUT-POINTER (CADR (ASSOC :INITIAL-INPUT-POINTER RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	(NUMERIC-ARG NIL)
	(NUMERIC-ARG-NEGATIVE NIL))
    (SETF (RHB-INITIAL-ENTRY) NIL)
    ;; Kludge #1.  If this is the first time this rubout handler has been invoked
    ;; in this stream, then we must create the input history.
    (OR (RH-INPUT-RING)
	(WHEN (FBOUNDP 'ZWEI:MAKE-HISTORY)
	  (SETF (RH-INPUT-RING) (RH-MAKE-INPUT-RING))))
    (WHEN INITIAL-ENTRY
      ;; save the previous input on the input history,
      ;; unless the previous read said not to save it.
      (COND ((AND (NOT (RH-DONT-SAVE-FLAG))
		  TYPEIN-POINTER
		  (NEQ INITIAL-ENTRY :RESTORED)
		  (NOT (ZEROP TYPEIN-POINTER)))
	     ;; only add the contents if it is different from the last entry, and
	     ;; the entry is at least 2 characters long.
	     (SETF (FILL-POINTER RUBOUT-HANDLER-BUFFER) TYPEIN-POINTER)
	     (WHEN (AND (> TYPEIN-POINTER 1)
			(FBOUNDP 'ZWEI:HISTORY-LATEST-ELEMENT)
			(MISMATCH RUBOUT-HANDLER-BUFFER
				  (ZWEI:HISTORY-LATEST-ELEMENT (RH-INPUT-RING))))
	       (ZWEI:PUSH-ON-HISTORY (SUBSEQ RUBOUT-HANDLER-BUFFER 0 TYPEIN-POINTER)
				     (RH-INPUT-RING)))
	     (SETF (FILL-POINTER RUBOUT-HANDLER-BUFFER) FILL-POINTER)))
      ;; Then initialize the typein pointer.
      (SETF (RH-TYPEIN-POINTER) FILL-POINTER)
      (SETQ TYPEIN-POINTER FILL-POINTER)
      ;; Gobble the initial input if any.
      (WHEN (AND INITIAL-INPUT (NEQ INITIAL-ENTRY :RESTORED))
	(RH-INSERT-STRING INITIAL-INPUT 0 NIL NIL NIL)
	(SETQ RUBBED-OUT-SOME? T FILL-POINTER (RH-FILL-POINTER))
        (SETQ INITIAL-INPUT-POINTER
	      (MAX (MIN (OR INITIAL-INPUT-POINTER TYPEIN-POINTER)
			(LENGTH RUBOUT-HANDLER-BUFFER)) 0))
	(SETF TYPEIN-POINTER INITIAL-INPUT-POINTER
	      (RH-TYPEIN-POINTER) INITIAL-INPUT-POINTER)
 	(RH-SET-POSITION TYPEIN-POINTER))
      ;; Record whether this unit of input should be saved on the history.
      (SETF (RH-DONT-SAVE-FLAG)
	    (OR (CADR (ASSOC :DONT-SAVE RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
		(CADR (ASSOC :NO-INPUT-SAVE RUBOUT-HANDLER-OPTIONS :TEST #'EQ))))
      )
    ;;; Can this ever go off? :pass-though now only allows non-bucky. -- mly ;;;
    ;; Kludge #5.  We can't echo or rub out a bucky char or a bli,
    ;; so if the last char inserted was a either of those
    ;; and it did not terminate the input, flush it.
    (AND (NOT (ZEROP TYPEIN-POINTER))
	 (OR (CONSP (AREF RUBOUT-HANDLER-BUFFER (1- TYPEIN-POINTER)))
	     (Not (Zerop (CHAR-BITS (Aref RUBOUT-HANDLER-BUFFER (1- Typein-Pointer))))))
	 (SETQ TYPEIN-POINTER (SETF (RH-TYPEIN-POINTER) (DECF (RH-FILL-POINTER)))))
    ;; Kludge #4.  After resuming a Break, the stream's cursorpos is wrong.
    ;; In fact, the cursor is at the end of the string in that case.
    ;; So, if it is supposed to be elsewhere, move it.
    ;; This condition also avoids wasting time when we are reading typein
    ;; at the end of the string.
    (OR (= FILL-POINTER TYPEIN-POINTER)
	(RH-CURSOR-MOTION TYPEIN-POINTER))
    ;; In case we had to return to the caller with a EDITING-COMMAND char
    ;; while RUBBED-OUT-SOME? was T, make things consistent again
    ;; by causing a rescan now.
    (WHEN (AND (NOT INITIAL-ENTRY)
	       (= (RH-SCAN-POINTER) MOST-POSITIVE-FIXNUM))
      (SETF (RH-SCAN-POINTER) 0)
      (THROW 'RUBOUT-HANDLER T)) 
    (CATCH 'RETURN-CHARACTER
      (WHEN RUBOUT-HANDLER-ACTIVATION-CHARACTER
	(THROW 'RETURN-CHARACTER
		(PROG1 RUBOUT-HANDLER-ACTIVATION-CHARACTER
		       (SETQ RUBOUT-HANDLER-ACTIVATION-CHARACTER NIL))))
      ;; Read characters.  If an ordinary character is typed and nothing has been rubbed out,
      ;; return immediately.  Otherwise, let all editing operations complete
      ;; before returning. 
      (DO (*LAST-COMMAND-TYPE*
	   *CURRENT-COMMAND-TYPE*
	   *RUBOUT-HANDLER-MARK*
	   (EDITING-COMMAND    (CDR (ASSOC :EDITING-COMMAND RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	   (DO-NOT-ECHO        (CDR (ASSOC :DO-NOT-ECHO     RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	   (PASS-THROUGH       (CDR (ASSOC :PASS-THROUGH    RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	   (COMMAND-HANDLER    (ASSOC :COMMAND     RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
	   (PREEMPTABLE        (ASSOC :PREEMPTABLE RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
	   (ACTIVATION-HANDLER (ASSOC :ACTIVATION  RUBOUT-HANDLER-OPTIONS :TEST #'EQ)))
	  (NIL)
	;; Read a character from the stream after bypassing ourself.
	(SETQ CH
	      (LET ((RUBOUT-HANDLER NIL)
		    (TV:*DEFAULT-READ-WHOSTATE* (IF NUMERIC-ARG
						    (FORMAT NIL "Arg = ~d" numeric-arg)
						  "Keyboard")))
		(READ-ANY SELF)))
	(IF (LISTP CH)
	    (COND ((EQ (CAR CH) 'REDISPLAY-RUBOUT-HANDLER)
		   (SEND SELF :SET-CURSORPOS
			 PROMPT-STARTING-X PROMPT-STARTING-Y)
		   (SEND SELF :CLEAR-EOL)
		   (RH-REPRINT-INPUT NIL T))
		  (PREEMPTABLE
		   (MULTIPLE-VALUE-BIND (STRING INDEX)
		       (SEND SELF :SAVE-RUBOUT-HANDLER-BUFFER)
		     (SETQ OLD-TYPEAHEAD (LIST STRING INDEX)))
		   ;; Save the text, rub it all out, and unread the blip.
		   ;; The :FULL-RUBOUT option will cause the RH to return
		   ;; to the caller who will then read the blip.
		   (UNREAD-ANY CH SELF)         ;Should this be UNREAD-CHAR??  It was :UNTYI.
		   (RH-COM-CLEAR-INPUT NIL)
		   (SETF (RH-SCAN-POINTER) 0)
		   (THROW 'RETURN-FROM-RUBOUT-HANDLER
			   (VALUES CH (CADR PREEMPTABLE))))
		  ((AND (EQ (CAR CH) :MOUSE-BUTTON)
			(EQL (CADR CH) #\MOUSE-3-1))
		   (MOUSE-CALL-SYSTEM-MENU))
		  ;; This clause added by RDA for dragging
		  ((and w:*dragging-enabled?*
			(eq (first ch) :mouse-button)
			(char= (second ch) #\mouse-2-1))
		   (w:drag-window-from self (fourth ch) (fifth ch))))
	  (PROGN                                ;CH is a character, not a blip.
            (WHEN (AND (EQL CH #\)) *RH-MATCHING-PAREN-OPTIONS*
                       (< (RH-TYPEIN-POINTER) (CADDR *RH-MATCHING-PAREN-OPTIONS*)))
              (RH-BLINK-RIGHT-PAREN))             ;Cause matching parens to blink.
            ;;This code handles auto-completion on UCL command names and special typein modes.
            ;;Auto-completion used to be implemented as an Input Edtor command, but
            ;;this was inefficient on large typed expressions because each SPACE would cause 
            ;;re-reading of the input.
            (WHEN (AND (EQL CH #\SPACE)
                       ucl:command-loop-typein?  ;;Auto-complete only during ucl:READ-FOR-UCL.
                       user:auto-complete?  ;;Users can turn off auto-completion.
                       (>= (RH-TYPEIN-POINTER) (RH-FILL-POINTER))
                       (PLUSP (RH-TYPEIN-POINTER))
                       (NOT (CHAR= #\SPACE (AREF RUBOUT-HANDLER-BUFFER (1- (RH-TYPEIN-POINTER))))))
              (FUNCALL (complete-word-function rh-completion-handler)
                       :RECOGNITION (LIST 'complete-lisp-atom :auto-complete-p) T)
              (SETQ RUBBED-OUT-SOME? T))  ;;Force rescan of completed input.
            
            (SETQ CH-CHAR (CHAR-CODE CH))       ;WARNING - CH-CHAR is a Fixnum !!
            (SETQ CH-CONTROL-META (CHAR-BITS CH))
            (IF (AND (ZEROP ch-control-meta) (MEMBER CH PASS-THROUGH))
                (SETQ command NIL)
                ;;Look up the key in the RH command table, fetching more keys as necessary (in
                ;;case the user has bound RH commands to keystroke sequences greater than 1).
                ;;If the look-up fails, we have only read one key, and we are doing type-in for a
                ;;UCL aplication, try looking up the key in the UCL's tables.  If a command is found
                ;;or there are several commands found starting with the key, let the UCL
                ;;application preempt us to process the command.
                (PROGN
                  (MULTIPLE-VALUE-SETQ (command keys-read)
                    (ucl-rh-lookup-character ch))
                  (WHEN (AND (NULL command) ucl:command-loop-typein? preemptable (NULL (CDR keys-read)))
                    (WHEN (AND (SETQ command (SEND ucl:command-loop-typein? :lookup-keys keys-read))
                               (NEQ command :FETCH-MORE-KEYS)
                               (SEND command :GET :CANT-PREEMPT))
                      ;;If this command can't preempt typed expressions, don't try to.
                      (SETQ command NIL)))))
            (COND
              ((AND COMMAND-HANDLER
                    (APPLY (CADR COMMAND-HANDLER) CH (CDDR COMMAND-HANDLER)))
               (SETF (RH-SCAN-POINTER) 0)
               (THROW 'RETURN-FROM-RUBOUT-HANDLER
                      (VALUES
                        `(:COMMAND ,CH ,(* (OR NUMERIC-ARG 1)
                                           (IF NUMERIC-ARG-NEGATIVE -1 1)))
                        :COMMAND)))
              ((OR (MEMBER CH DO-NOT-ECHO)
                   (AND ACTIVATION-HANDLER
                        (APPLY (CADR ACTIVATION-HANDLER) CH (CDDR ACTIVATION-HANDLER))))
               (RH-SET-POSITION (RH-FILL-POINTER))
               (LET ((VALUE
                       (IF (MEMBER CH DO-NOT-ECHO) CH
                           `(:ACTIVATION ,CH ,(* (OR NUMERIC-ARG 1)
                                                 (IF NUMERIC-ARG-NEGATIVE -1 1))))))
                 (COND (RUBBED-OUT-SOME?
                        ;; Why isn't this done in the :RUBOUT-HANDLER method loop?
                        (SETQ RUBOUT-HANDLER-ACTIVATION-CHARACTER VALUE)
                        (SETF (RH-SCAN-POINTER) 0)
                        (THROW 'RUBOUT-HANDLER T))
                       (T (THROW 'RETURN-CHARACTER VALUE)))))
              ;; Don't touch this character, just return it to caller.
              ((OR (MEMBER CH EDITING-COMMAND)
                   (SI:ASSQ-CAREFUL CH EDITING-COMMAND))
               ;; Cause rubout handler rescan next time the user does :TYI.
               (If RUBBED-OUT-SOME? (SETF (RH-SCAN-POINTER) MOST-POSITIVE-FIXNUM))
               (RETURN CH))
              ;;Handle Control-number and Control-U specially.  This code was moved forward in
              ;;order to catch numeric args before the UCL application's num.  arg command claimes
              ;;the keystroke.  Note that CH-CHAR is a fixnum, CH is a Character.
              ((AND (NOT (ZEROP CH-CONTROL-META)) (<= (Char-Code #\0) CH-CHAR (Char-Code #\9)))
               (SETQ NUMERIC-ARG (+ (* (OR NUMERIC-ARG 0) 10.) (- CH-CHAR (Char-Code #\0)))))
              ((CHAR= CH #\CONTROL-U)
               (SETQ NUMERIC-ARG (* (OR NUMERIC-ARG 1) 4)))
              ((AND (NOT (ZEROP CH-CONTROL-META)) (= CH-CHAR (Char-Code #\-)))
               (IF NUMERIC-ARG
                   (SEND SELF :BEEP)
                   (SETQ NUMERIC-ARG-NEGATIVE (NOT NUMERIC-ARG-NEGATIVE))))
              (command
               (IF (NOT (TYPEP command 'rh-command))
                   ;;Keystroke was found in a UCL application's command-table.  Have the read
                   ;;function return the values '(:KEYSTROKE <ch>) :MOUSE-CHAR to indicate that
                   ;;we are interrupting typein to execute another command.  <ch> will be
                   ;;processed as if it were typed at top level.
                   (IO-BUFFER-PUSH IO-BUFFER (LIST :KEYSTROKE (CHAR-INT ch)))
                   ;; A standard rh editing command of some sort. The RUBBED-OUT-SOME bit can only
                   ;; be cleared by entering this function again.  The function is passed the
                   ;; numeric argument, and returns T if we are going to need to throw out (like
                   ;; DIS-ALL in the editor).
                   (PROGN
                     (SETQ *LAST-COMMAND-TYPE* *CURRENT-COMMAND-TYPE*
                           *CURRENT-COMMAND-TYPE* NIL)
                     (SETQ RUBBED-OUT-SOME?
                           (OR (SEND command :execute self
                                     (* (OR numeric-arg 1) (IF numeric-arg-negative -1 1))
                                     NIL)
                               RUBBED-OUT-SOME?))
                     (SETQ NUMERIC-ARG NIL NUMERIC-ARG-NEGATIVE NIL)
                     ;; If the buffer is empty and the :FULL-RUBOUT option is active, then throw now
                     ;; This will throw if the user types Rubout or ClearScreen immediately after
                     ;; entering the read function.  It is important that we check for this here
                     ;; and not in RH-DELETE-STRING since some commands, such as Yank-Pop, may
                     ;; temporarily empty the buffer.  It wouldn't be the right thing to throw
                     ;; if the buffer only contained whitespace since it is the responsibility
                     ;; of the caller to discard whitespace when looking for special characters.
                     (COND ((AND (ZEROP (RH-FILL-POINTER))
                                 (ASSOC :FULL-RUBOUT RUBOUT-HANDLER-OPTIONS :TEST #'EQ))
                            ;; This SETF should be done in the :RUBOUT-HANDLER method loop.
                            (SETF (RH-SCAN-POINTER) 0)
                            (THROW 'RUBOUT-HANDLER T))))))
              ;;Beep and ignore.
              ((NOT (ZEROP ch-control-meta))
               (SEND SELF :beep)
               (SETQ numeric-arg nil numeric-arg-negative nil))
              
              ;; Self-inserting character.  Set RUBBED-OUT-SOME since if we return,
              ;; we were typing in the middle of the line.  Typing at the end of the
              ;; line throws to RETURN-CHARACTER.
              (T (UNLESS NUMERIC-ARG-NEGATIVE
                   (RH-INSERT-CHAR CH (OR NUMERIC-ARG 1) RUBBED-OUT-SOME?)
                   (SETQ RUBBED-OUT-SOME? T))  
                 (SETQ *LAST-COMMAND-TYPE* *CURRENT-COMMAND-TYPE*
                       *CURRENT-COMMAND-TYPE* NIL
                       *RUBOUT-HANDLER-MARK* NIL)
                 (SETQ NUMERIC-ARG NIL NUMERIC-ARG-NEGATIVE NIL)))))))))
)

;;;TI Code: Mouse-M-H in mini-buffer allows dragging
ZWEI:
(DEFUN PROCESS-SPECIAL-COMMAND (OP &REST ARGS)
  ;;this was implemented previously as a defselect - rpm 6/23/86
  (CASE OP
    (REDISPLAY					;NO ARGS 
     ;;The window is presumably on our list of windows and will get redisplayed
     ;;in the normal course of events when buffered input had been processed.
     NIL)
    (SELECT-WINDOW				;ARGS = (WINDOW)
     (PROG1 (NEQ (CAR ARGS) *WINDOW*)
	    (MAKE-WINDOW-CURRENT (CAR ARGS))))
    (CONFIGURATION-CHANGED			;NO ARGS
     (LET ((FEW (FRAME-EXPOSED-WINDOWS)))
       (UNLESS (MEMBER *WINDOW* FEW :TEST #'EQ)
	 (MAKE-WINDOW-CURRENT (CAR FEW))))
     NIL)
    (SCROLL					;ARGS = (WINDOW NLINES TYPE)
     (LET ((WINDOW (FIRST ARGS))
	   (NLINES (SECOND ARGS))
	   (TYPE (THIRD ARGS)))
       (IF (EQ TYPE :RELATIVE)
	   (RECENTER-WINDOW-RELATIVE WINDOW NLINES)
	   (RECENTER-WINDOW WINDOW :START
			    (LET-IF (NEQ WINDOW *WINDOW*)
				    ((*INTERVAL* (WINDOW-INTERVAL WINDOW)))
			      (FORWARD-LINE (INTERVAL-FIRST-BP (WINDOW-INTERVAL WINDOW)) NLINES T))))
       (UNLESS (EQ WINDOW *WINDOW*)
	 ;; Scrolling nonselected window => flush typeout on it
	 ;; because the main loop won't do it except for the selected window.
	 (PREPARE-WINDOW-FOR-REDISPLAY WINDOW))
       T))
    (:MOUSE-BUTTON				;ARGS = (CH WINDOW *MOUSE-X* *MOUSE-Y*)
     (LET ((CH (INT-CHAR (FIRST ARGS)))		;do int-char to get around :any-tyi bug - remove when fixed
	   (WINDOW (SECOND ARGS))
	   (*MOUSE-X* (THIRD ARGS))
	   (*MOUSE-Y* (FOURTH ARGS)))
       (IF (NOT (TYPEP WINDOW 'ZWEI))
	   ;; The following changed by RDA to add dragging
;	   (WHEN (CHAR= CH #\MOUSE-R)
;	     (W:MOUSE-CALL-SYSTEM-MENU))
	   (if (char= ch #\mouse-r)
	       (w:mouse-call-system-menu)
	       (if (and w:*dragging-enabled?* (char= ch #\mouse-m))
		   (w:mouse-drag-window (w:find-right-sheet window))))
	   (PROGN
	     (DECF *MOUSE-X* (W:SHEET-INSIDE-LEFT (WINDOW-SHEET WINDOW)))
	     (DECF *MOUSE-Y* (W:SHEET-INSIDE-TOP (WINDOW-SHEET WINDOW)))	
	     (AND (MEMBER :RECORD (SEND *STANDARD-INPUT* :WHICH-OPERATIONS) :TEST #'EQ)
		  (SEND *STANDARD-INPUT* :RECORD CH))
	     (IF *MOUSE-HOOK*
		 (FUNCALL *MOUSE-HOOK* WINDOW CH *MOUSE-X* *MOUSE-Y*)
		 (IF (NEQ WINDOW *WINDOW*)	;Given in another window,
		     (LET ((*COMTAB* (IF (EQ *WINDOW* *MINI-BUFFER-WINDOW*)
					 *STANDARD-COMTAB*
					 *COMTAB*))
			   (*LAST-COMMAND-TYPE* NIL)	;dont confuse mouse mark thing, and
			   *CURRENT-COMMAND-TYPE*	;temporarily act there (mini-buffer)
			   (*WINDOW* WINDOW)
			   (*INTERVAL* (WINDOW-INTERVAL WINDOW)))	
		       (PROCESS-COMMAND-CHAR CH))
		     (PROCESS-COMMAND-CHAR CH)))
	     T))))
    ((:TYPEOUT-EXECUTE :EXECUTE)		;ARGS = (FUNCTION &REST ARGS)
     (LET ((*MINI-BUFFER-DONT-RECORD* T))
       ;; We would not be able to repeat the command anyway.
       (NOT (APPLY (CAR ARGS) (CDR ARGS)))))
    ;;these last three used to be auto-generated by the defselect - keep them up-to-date!!
    (:WHICH-OPERATIONS 
     '(REDISPLAY SELECT-WINDOW CONFIGURATION-CHANGED SCROLL :MOUSE-BUTTON :TYPEOUT-EXECUTE :EXECUTE))
    (:OPERATION-HANDLED-P
     (AND (MEMBER OP '(REDISPLAY SELECT-WINDOW CONFIGURATION-CHANGED
		       SCROLL :MOUSE-BUTTON :TYPEOUT-EXECUTE :EXECUTE)
		  :TEST #'EQ)
	  T))
    (:SEND-IF-HANDLES
     (AND (MEMBER OP '(REDISPLAY SELECT-WINDOW CONFIGURATION-CHANGED
		       SCROLL :MOUSE-BUTTON :TYPEOUT-EXECUTE :EXECUTE)
		  :TEST #'EQ)
	  (PROCESS-SPECIAL-COMMAND (CAR ARGS) (CDR ARGS))))
    (OTHERWISE
      (UNKNOWN-SPECIAL-COMMAND OP ARGS))))

;;;----------------------------------------------------------------------
;;; Other helpers and hooks

(defun mouse-drag-wum ()
  ;;FIND-RIGHT-SHEET from WINDOW-SYSTEM-ADDITIONS
  (when *dragging-enabled?*
    (mouse-drag-window (find-right-sheet (window-under-mouse)))))

(defun drag-window-from (window x y)
  "Can be called from mouse process.  Drags WINDOW starting from X, Y
   inside WINDOW."
  (when *dragging-enabled?*
    (multiple-value-bind (xoff yoff)
	(sheet-calculate-offsets window mouse-sheet)
      (process-run-function
	"Drag Window" #'mouse-drag-window window t
	(+ x  xoff) (+ y yoff))
      )
    )
  )

ucl:
(add-command 'drag-window
  '(:DESCRIPTION "Allows the window to be dragged to a different position"
    :KEYS #\MOUSE-m
    :definition w:mouse-drag-wum)
  'universal-commands)

;; Allow dragging deexposed Listeners
(defmethod (lisp-listener :deexposed-mouse-buttons) (mask x y)
  "Move or Select"
  (let ((buttons (mouse-character-button-encode mask)))
    (case buttons
      ((#\mouse-r-2 #\mouse-r-1) (mouse-call-system-menu))
      (#\mouse-m-1 (when *dragging-enabled?* (drag-window-from self x y)))
      (#\mouse-l-1 (mouse-select self))
      )
    )
  )

(defmethod (w::lisp-listener :deexposed-who-line-documentation-string) ()
  `(:mouse-L-1 "Expose This Window" :mouse-M-1 "Drag This Window"
    :mouse-R-1 "System Menu")
  )

(provide 'window-dragging)

;;;======================================================================

(defvar *mouse-warp-gratuitously?* t
"If NIL then don't warp the mouse in menus, etc.  Will still move the
mouse for some operations, like window positioning.")

;;;TI Code: changes marked with RDA
(when (>= (si:get-system-version) 4)
 (DEFMETHOD (MENU :CHOOSE) NIL
  (LET (RETURN-RESULT)
    (WHEN (PLUSP (SHEET-MENU-DYNAMIC))
      (SEND SELF :UPDATE-ITEM-LIST))
    (PROG ((X MOUSE-X)
	   (Y MOUSE-Y)
	   (SUCCESS T))
      (UNWIND-PROTECT (PROGN
		       (CATCH 'ABORT
			 (RETURN
			  (PROGN
			    (COND
			      ((AND (PLUSP (SHEET-MENU-POP-UP)) (NOT EXPOSED-P))
			       (MULTIPLE-VALUE-BIND (X-OFF Y-OFF) (SHEET-CALCULATE-OFFSETS SUPERIOR MOUSE-SHEET)
				 (MULTIPLE-VALUE-BIND (X Y) (SEND SELF :CENTER-AROUND (- MOUSE-X X-OFF) (- MOUSE-Y Y-OFF))
				   (when *mouse-warp-gratuitously?*	;RDA: Added WHEN check
				   (MOUSE-WARP (+ X X-OFF) (+ Y Y-OFF)))))
			       (WITH-MOUSE-GRABBED (SEND SELF :SELECT)
				  (COND
				    ((NEQ SELF
					  (LOWEST-SHEET-UNDER-POINT MOUSE-SHEET MOUSE-X MOUSE-Y
								    NIL :EXPOSED))
				     (SEND SELF :DEACTIVATE) (THROW 'ABORT
								    ()))))))
			    (SETQ CHOSEN-ITEM NIL)
			    (OR (EQ (SEND SELF :STATUS) :SELECTED) (SEND SELF :SELECT))
			    (LOOP DO
			       (PROCESS-WAIT "Menu choose" #'MENU-CHOOSE-INTERNAL-WAIT-FUNCTION
					     SELF (LOCF CHOSEN-ITEM) (LOCF EXPOSED-P))
			       UNTIL (OR CHOSEN-ITEM (NOT EXPOSED-P)) WHEN (LISTEN SELF) DO
			       (LET ((KBD-INTERCEPTED-CHARACTERS MENU-INTERCEPTED-CHARACTERS))
				 (SEND SELF :PROCESS-CHARACTER (READ-ANY SELF))))
			    (UNWIND-PROTECT (SETQ
					     RETURN-RESULT
					     (SEND SELF :EXECUTE CHOSEN-ITEM))
			      (SETQ CHOSEN-ITEM NIL)))))
		       (SETQ SUCCESS NIL))
	(WHEN (and SUCCESS *mouse-warp-gratuitously?*)	;RDA: Check *MOUSE-WARP-GRATUITOUSLY?* too
	  (MOUSE-WARP X Y))))
    RETURN-RESULT))
)


;-------------------------------------------------------------------------------

;;; Added by JPR.
(defun ticl:wum (&optional (superiors 0))
"The window currently under the mouse, or Superiors windows up from it."
  (loop with win = (tv:window-under-mouse)
	for supers from superiors downto 1
	until (not (tv:sheet-superior win))
	do (setq win (tv:sheet-superior win))
	finally (return win)
  )
)

(export 'ticl:wum 'ticl)

;-------------------------------------------------------------------------------

(defvar *already-messing-with-screen-arrays* nil
"A list of windows that the current process is hacking on to swap the
bit arrays for bacground drawing etc.
"
)

(defmacro tv:with-window-ops-on-bit-array
	  ((sheet &optional (save-bits-p t)) &body body)
"Execute BODY with SHEET pretending to be deexposed.   SHEET will be refreshed
afterward if it was exposed to begin with.  Returns the values of body.  If
Save-bits-p is true then the bits on the screen are saved to the bit array
before we go into the form.  This is a good idea unless you know that the
window is going to be completely redrawn.
Notes:  a) This macro doesn't seem to be much of a win on microExplorers.
        b) See also tv:without-window-ops-on-bit-array.
"
  `(let ((.sheet. ,sheet))
        (if (and nil (not (member .sheet. *Already-Messing-With-Screen-Arrays*))
	        (Tv:sheet-exposed-p .sheet.)
		(not (sys:mx-p))
	    )
	    (let ((.screen-array. (send .sheet. :Screen-Array))
		  (.bit-save-array. (send .sheet. :Bit-Array))
		  (.width. (tv:sheet-width .sheet.))
		  (.height. (tv:sheet-height .sheet.))
		  (*Already-Messing-With-Screen-Arrays*
		    (cons .sheet. *Already-Messing-With-Screen-Arrays*)
		  )
		 )
	         (check-type .bit-save-array. array)
		 (Prepare-sheet (.sheet.)
		   (if ,save-bits-p
		       (bitblt tv:alu-seta .width. .height. .screen-array.
			       0 0 .bit-save-array. 0 0
		       )
		   )
		   (tv:sheet-force-access (.sheet. nil)
		     (multiple-value-prog1
		      (letf (((symeval-in-instance .sheet. 'tv:screen-array)
			      .bit-save-array.
			     )
			     ((symeval-in-instance .sheet. 'tv:bit-array)
			      .screen-array.
			     )
			    )
			    ,@body
		      )
		      (send .sheet. :Refresh :use-saved-bits)
		     )
		   )
		 )
	    )
	    (progn ,@body)
	)
   )
)

(export 'tv:with-window-ops-on-bit-array 'tv)

(defmacro tv:without-window-ops-on-bit-array
	  ((sheet) &body body)
"Execute BODY with SHEET guaranteed not to be using
tv:with-window-ops-on-bit-array for its screen ops.
This is important if you are using tv:with-window-ops-on-bit-array
in a :refresh method (a generally good idea) because refresh is called
by set-edges and things get confused.  Thus if you do the above you should also
do (defwhopper (<my-window> :set-edges) (&rest args)
    (tv:without-window-ops-on-bit-array (self) (lexpr-continue-whopper args)))
"
 `(let ((*Already-Messing-With-Screen-Arrays*
	  (cons ,sheet *Already-Messing-With-Screen-Arrays*)
	)
       )
       ,@body
  )
)

(export 'tv:without-window-ops-on-bit-array 'tv)
;-------------------------------------------------------------------------------

;;; Hack the mouse process so that we always have a decent size of stack.

(defvar tv:*mouse-process-minimum-regular-pdl-size* 16000.)
(defvar tv:*mouse-process-minimum-special-pdl-size* 6000.)

;;; TI code.
tv:
(DEFUN tv:MOUSE-INITIALIZE (&OPTIONAL (SHEET DEFAULT-SCREEN) (force-new-p nil))
  ;;; JPR.
  (or sheet (setq sheet default-screen))
  (OR (and (not force-new-p)
	   (BOUNDP 'MOUSE-PROCESS)
	   (>= (array-total-size
		 (sys:sg-regular-pdl (send mouse-process :stack-group)))
	       4096))
      ;If first time loaded, initialize everything
      (SETQ mouse-process
	    (make-process
	      "Mouse"
	      :SPECIAL-PDL-SIZE *mouse-process-minimum-special-pdl-size*
	      :regular-pdl-size *mouse-process-minimum-regular-pdl-size*
	      :PRIORITY 30.
	      :WARM-BOOT-ACTION NIL)))
  ;; Above warm-boot-action prevents the process from starting up
  ;; until after these initializations have been completed.
  (SETQ MOUSE-WINDOW        NIL
        WINDOW-OWNING-MOUSE NIL
        MOUSE-X 0
        MOUSE-Y 0
        MOUSE-SHEET SHEET)
  (%SET-MOUSE-SCREEN SHEET)
  ;; Fill the mouse tracker's arrays with NIL instead of the garbage
  ;; that they contain initially.  At least interpreted ASET won't work otherwise.
  (LOOP FOR I FROM #o1640 BELOW #o1774
        DO (%P-dpb nil %%q-all-but-cdr-code (+ A-MEMORY-VIRTUAL-ADDRESS I)))
  ;; Set scaling and speed dependence.
  (MOUSE-SPEED-HACK 0.6 80. 0.8 104. 1 128. 1.3 192. 1.5 256. 1.8 320. 2.2 448. 2.5)
  ;; Make sure at least one blinker of each type exists
  (MOUSE-GET-BLINKER :CHARACTER)
  (MOUSE-GET-BLINKER :RECTANGLE-CORNER-BLINKER)
  (MOUSE-GET-BLINKER :RECTANGLE-BLINKER)
  (AND MOUSE-BLINKER (BLINKER-SET-VISIBILITY MOUSE-BLINKER NIL))
  (MOUSE-STANDARD-BLINKER)
  (MOUSE-WARP (- (SHEET-INSIDE-WIDTH MOUSE-SHEET)   8.)
              (- (SHEET-INSIDE-HEIGHT MOUSE-SHEET) 16.))
  ;; Call MOUSE-INPUT once to flush any pending motion and update variables, but don't wait.
  (SETQ MOUSE-BUTTONS-BUFFER-OUT-INDEX MOUSE-BUTTONS-BUFFER-IN-INDEX)
  (MOUSE-INPUT NIL)
  (SETQ MOUSE-X-SPEED 0
        MOUSE-Y-SPEED 0)
  ;; Start up the mouse process
  (SEND MOUSE-PROCESS :PRESET 'MOUSE-OVERSEER)
  (SEND MOUSE-PROCESS :RUN-REASON))


;;;; Reset mouse process.
(send tv:mouse-process :Interrupt 'eh:require-pdl-room
      tv:*Mouse-Process-Minimum-Regular-Pdl-Size*
      tv:*Mouse-process-minimum-special-pdl-size*)
