;;; -*- Mode:Common-Lisp; Package:System-Internals; Base:10 -*-




(defmacro ticl:With-Stack-Cons ((name car cdr) &body body)
"Makes a single cons on the stack called Name with the designated car and cdrs."
  `(with-stack-list* (,name ,car ,cdr) ,@body)
)

(export 'ticl:With-Stack-Cons 'ticl)

(defmacro ticl:With-Stack-Lists
	  ((&rest var-name-and-value-expressions) &body body)
  "Like with-stack-list only allows multiple stack lists like Let, e.g.
   (with-stack-lists ((name-1 val1-1 val1-2...) (name-2 val2-1 val2-2...) (...))
     ,@body)"
  (labels ((with-stack-lists-internal (vars-and-exprs body)
	     (if vars-and-exprs
		`(with-stack-list ,(first vars-and-exprs)
		   ,(with-stack-lists-internal (rest vars-and-exprs) body)
		 )
		`(progn ,@body)
	      )
	    )
	   )
    (with-stack-lists-internal var-name-and-value-expressions body)
  )
)

(export 'ticl:With-Stack-Lists 'ticl)

(defmacro ticl:With-Stack-Forms
	  ((&rest var-name-and-value-expressions) &body body)
  "Like with-stack-lists only each var/expr form has a type as well, e.g.
   (with-stack-forms ((cons name-1 val1-1 val1-2)
                      (list name-2 val2-1 val2-2...) (...))
     ,@body)"
  (labels ((with-stack-forms-internal (vars-and-exprs body)
	     (if vars-and-exprs
		`(,(ecase (first (first vars-and-exprs))
		     (list 'with-stack-list)
		     ((cons list*) 'with-stack-list*)
		     ((nconc append) 'ticl:With-Stack-Append)
		    )
		  ,(rest (first vars-and-exprs))
		  ,(with-stack-forms-internal (rest vars-and-exprs) body)
		 )
		`(progn ,@body)
	      )
	    )
	   )
    (with-stack-forms-internal var-name-and-value-expressions body)
  )
)

(export 'ticl:With-Stack-Forms 'ticl)

(defun Bq-1 (form)
"Interprets form in such a way that code to generate the stack-consed
equivalent of form is recorded.  The recorded forms are given to
with-stack-forms.
"
  (declare (special *lists*))
  (if (consp form)
      (let ((sym (gensym)))
	   (case (first form)
	     (list (push `(list ,sym ,@(mapcar #'Bq-1 (rest form)))
			  *lists*
		   )
		   sym
	     )
	     (list* (push `(list* ,sym ,@(mapcar #'Bq-1 (rest form)))
			   *lists*
		    )
		    sym
	     )
	     (cons (push `(cons ,sym ,@(mapcar #'Bq-1 (rest form)))
			  *lists*
		   )
		   sym
	     )
	     ((append nconc)
	      (push `(append ,sym ,@(mapcar #'Bq-1 (rest form)))
		    *lists*
	      )
	      sym
	     )
	     (quote form)
	     (otherwise form)
	   )
      )
      form
  )
)


(defun With-Stack-Backquote-Internal (names-and-expressions body environment)
"Internal function of with-stack-backquote"
  (if names-and-expressions
      (destructuring-bind ((name expression) . rest) names-and-expressions
        (let ((bq-list (macroexpand-all expression environment))
	      (*lists* nil)
	     )
	     (declare (special *lists*))
	     (let ((top-level-sym (Bq-1 bq-list)))
		 `(With-Stack-Forms ,(reverse *lists*)
		    (let ((,name ,top-level-sym))
			,(With-Stack-Backquote-Internal rest body environment)
		    )
		  )
	     )
	)
      )
     `(progn ,@body)
  )
)

(defmacro ticl:With-Stack-Backquote
	  ((&rest names-and-expressions) &body body &environment env)
"Defines the variable Name to have the value of the backquoted list Expression
throughout the body.  The backquoted list is stack consed.
"
  (gensym "BQ")
  (With-Stack-Backquote-Internal names-and-expressions body env)
)

(export 'ticl:With-Stack-Backquote 'ticl)

(defmacro ticl:With-Variable-Length-Stack-List ((name length) &body body)
"Creates a stack-consed list of length Length and calls it Name for the
duration of Body.
"
  `(let ((,name (%make-stack-list ,length))) ,@body)
)

(export 'ticl:With-Variable-Length-Stack-List 'ticl)

(defmacro ticl:With-Variable-Length-Stack-List* ((Name length) &body body)
"Creates a stack-consed list* of length Length and calls it Name for the
duration of Body.
"
  `(let ((,name (compiler:%make-stack-list* ,length))) ,@body)
)

(export 'ticl:With-Variable-Length-Stack-List* 'ticl)

(defmacro ticl:With-Stack-Append ((name &rest lists) &body body)
"Creates a list on the stack, which is the appended form of all of the
Lists and names it Name, which is visible throughout Body.  Note:  the
last list is not stack consed (unless you stack consed it already),
it is structure shared instead.
"
  (let ((pointer (gensym))
	(last-pointer (gensym))
	(last (gensym))
	(length (gensym))
	(lists-to-append (gensym))
       )
      `(let ((,last ,(first (last lists)))
	     (,length (+ ,@(loop for tail on lists when (rest tail)
				 collect `(length ,(first tail))
			   )
			   0
		      )
	     )
	    )
	    (if (> ,length 0)
		(With-Variable-Length-Stack-List* (,Name (+ 1 ,length))
		  (with-stack-list (,lists-to-append ,@(butlast lists))
		    (let ((,pointer ,name)
			  (,last-pointer nil)
			 )
			 (loop for list in ,lists-to-append do
			       (loop for element in list do
				     (setf (first ,pointer) element)
				     (setq ,last-pointer ,pointer)
				     (setq ,pointer (rest ,pointer))
			       )
			 )
			 (setf (rest ,last-pointer) ,last)
		    )
		  )
		  ,@body
		)
		(let ((,name ,last)) ,@body)
	    )
       )
  )
)

(export 'ticl:With-Stack-Append 'ticl)