;;; -*- Mode: Common-Lisp; Package: SPELLER; Fonts: cptfont,cptfontb; Base: 10 -*-

;;;
;;; Copyright (c) 1987 Douglas Johnson.  All rights reserved. 
;;; Copyright (c) 1985 Douglas Johnson.  All rights reserved. 
;;;


;;;*******************************************************
;;;                                                                                               *
;;;                                                                                               *
;;;                    The following routines deal with suffixation.      
;;;                                                                                               *
;;;                                                                                               *
;;;************************************************************************************************                                                                        

(DEFUN CHECK-SUFFIXES-P (WORD)
  "Checks to see if WORD is in the dictionary with an allowed suffix.
   Returns nil if it is not."
  (SELECTOR WORD STRING-END-EQUAL ("ness" (CHECK-P-SUFFIX WORD)) ("s" (CHECK-S-SUFFIX WORD))
     ("ed" (CHECK-D-SUFFIX WORD)) ("er" (CHECK-R-SUFFIX WORD)) ("ive" (CHECK-V-SUFFIX WORD))
     ("th" (CHECK-H-SUFFIX WORD)) ("ly" (CHECK-Y-SUFFIX WORD)) ("n" (CHECK-N-SUFFIX WORD))
     ("ing" (CHECK-G-SUFFIX WORD)) ("est" (CHECK-T-SUFFIX WORD))
     ("s'" (CHECK-QUOTE-SUFFIX WORD)) (OTHERWISE NIL))) 


(DEFUN CHECK-QUOTE-SUFFIX (WORD)
  "Check to see if WORD is valid with both m and s suffixes.  Then the
   construct WORDs' is valid."
  (LET* ((LENGTH (- (LENGTH WORD) 2))
	 (WORD2 (NSUB-STRING WORD 0 LENGTH *STRING1*)))
    (AND (CHECK-ROOT-WORD-P WORD2 "m") (CHECK-ROOT-WORD-P WORD2 "s")))) 


(DEFUN CHECK-S-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with one of the s suffixes.
   Returns t if it is, nil if not."
  (LET* ((LENGTH1 (1- (LENGTH WORD)))
	 (WORD1 (NSUB-STRING WORD 0 LENGTH1 *STRING1*)))
    (COND
      ((AND                             ;check the basic word unless it ends in y
	(NOT (EQL (GET-CHAR WORD LENGTH1) #\Y))
	(CHECK-ROOT-WORD-P WORD1 "s")))
      ((SELECTOR WORD1 STRING-END-EQUAL ("er" (CHECK-R-SUFFIX WORD1 "z"))
	  ("e" (CHECK-ES-SUFFIX WORD)) ("n" (CHECK-N-SUFFIX WORD1 "x"))
	  ("ing" (CHECK-G-SUFFIX WORD1 "j")) ("'" (CHECK-M-SUFFIX WORD))
	  ("y" (AND (> LENGTH1 2)       ;"ys" allowed only if a vowel is before the y
		  (VOWEL-P WORD1 (- LENGTH1 2)) (CHECK-ROOT-WORD-P WORD1 "s")))
	  (OTHERWISE NIL)))))) 


(DEFUN CHECK-ES-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with a es or ies ending.  
   Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((<= LENGTH 3) NIL)
      ((AND
	(POSITION (GET-CHAR WORD (- LENGTH 3)) (THE STRING (STRING "szxh")) :TEST #'CHAR-EQUAL)
	(CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) "s")));word with ES appended
      ((STRING-END-EQUAL WORD "ies")
       (AND (> LENGTH 4);"ies" allowed with y only if no vowel is before the y
	  (NOT (VOWEL-P WORD (- LENGTH 4)))
	  (CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*) "y")
			     "s")))))) 
     


(DEFUN CHECK-Y-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with a Y suffix.
   Returns t if it is, nil if not."
  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- (LENGTH WORD) 2) *STRING1*) "y")) 


(DEFUN CHECK-M-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with a M suffix.
   Returns t if it is, nil if not."
  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- (LENGTH WORD) 2) *STRING1*) "m")) 


(DEFUN CHECK-N-SUFFIX (WORD &OPTIONAL (SUFFIX "n"))
  "Checks to see if WORD is a valid word with a N-type 
   (N or X) suffix.  Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (OR
     (AND (STRING-END-EQUAL WORD "ication")
	(CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 7) *STRING1*) "y")
			   SUFFIX))
     (AND (STRING-END-EQUAL WORD "ion")
	(CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*) "e")
			   SUFFIX))
     (AND (STRING-END-EQUAL WORD "en")
	(CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) SUFFIX))))) 




(DEFUN CHECK-D-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with an D suffix.
   Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "yed");"yed" allowed only if character before y is a vowel
       (AND (> LENGTH 4) (VOWEL-P WORD (- LENGTH 4))
	  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) "d")))
      ((STRING-END-EQUAL WORD "ied");"ied" allowed only if character before y is not a vowel
       (AND (> LENGTH 4) (NOT (VOWEL-P WORD (- LENGTH 4)))
	  (CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*) "y")
			     "d")))
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) "d"));word with ED appended
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 1) *STRING1*) "d")))))        ;word ending in E, D appended


(DEFUN CHECK-V-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with an V suffix.
   Returns t if it is, nil if not."
  (LET* ((LENGTH (LENGTH WORD))
	 (SHORT-WORD (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*)))
    (COND
      ((CHECK-ROOT-WORD-P SHORT-WORD "v"));word with IVE appended
      ((CHECK-ROOT-WORD-P (NSTRING-APPEND SHORT-WORD "e") "v")))))  ;word with E dropped, IVE added


(DEFUN CHECK-G-SUFFIX (WORD &OPTIONAL (SUFFIX "g"))
  "Checks to see if WORD is a valid word with an valid
   g-type (G or J) suffix.  Returns t if it is, nil if not."
  (LET* ((LENGTH (LENGTH WORD))
	 (SHORT-WORD (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*)))
    (COND
      ((AND (NOT (STRING-END-EQUAL SHORT-WORD "e"));word does not end in e
	  (CHECK-ROOT-WORD-P SHORT-WORD SUFFIX)));check with ING appended
      ((CHECK-ROOT-WORD-P (NSTRING-APPEND SHORT-WORD "e") SUFFIX)))))  ;add an e and check again


(DEFUN CHECK-H-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with an H suffix.
   Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((AND (NOT (STRING-END-EQUAL WORD "yth"));do not allow "yth"
	  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) "h")));check word with TH appended
      ((STRING-END-EQUAL WORD "ieth")
       (CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 4) *STRING1*) "y") "h")))))  ;word ending in Y, ITH appended


(DEFUN CHECK-P-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with an P suffix.
   Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "yness");"yness" allowed only if character before y is a vowel
       (AND (> LENGTH 6) (VOWEL-P WORD (- LENGTH 6))
	  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 4) *STRING1*) "p")))
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 4) *STRING1*) "p"));word with NESS appended
      ((STRING-END-EQUAL WORD "iness");"iness" allowed only if character before y is not a vowel
       (AND (> LENGTH 6) (NOT (VOWEL-P WORD (- LENGTH 6)))
	  (CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 5) *STRING1*) "y")
			     "p"))))))   ;word ending in Y, INESS appended




(DEFUN CHECK-T-SUFFIX (WORD)
  "Checks to see if WORD is a valid word with an T suffix.
   Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "yest");"yest" allowed only if character before y is a vowel
       (AND (> LENGTH 5) (VOWEL-P WORD (- LENGTH 5))
	  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*) "t")))
      ((STRING-END-EQUAL WORD "iest");"iest" allowed only if character before y is not a vowel
       (AND (> LENGTH 5) (NOT (VOWEL-P WORD (- LENGTH 5)))
	  (CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 4) *STRING1*) "y")
			     "t")))
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*) "t"));word with EST appended
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) "t")))))        ;word ending in E, ST appended





(DEFUN CHECK-R-SUFFIX (WORD &OPTIONAL (SUFFIX "r"))
  "Checks to see if WORD is a valid word with an R suffix.
   Returns t if it is, nil if not."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "yer");"yer" allowed only if character before y is a vowel
       (AND (> LENGTH 4) (VOWEL-P WORD (- LENGTH 4))
	  (CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) SUFFIX)))
      ((STRING-END-EQUAL WORD "ier");"ier" allowed only if character before y is not a vowel
       (AND (> LENGTH 4) (NOT (VOWEL-P WORD (- LENGTH 4)))
	  (CHECK-ROOT-WORD-P (NSTRING-APPEND (NSUB-STRING WORD 0 (- LENGTH 3) *STRING1*) "y")
			     SUFFIX)))
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 2) *STRING1*) SUFFIX));word with ER appended
      ((CHECK-ROOT-WORD-P (NSUB-STRING WORD 0 (- LENGTH 1) *STRING1*) SUFFIX)))))        ;word ending in E, R appended


;;;********************************************************************************
;;; Comment out these functions.  They are the beginnings of user editable 
;;; dictionaries.  If that feature ever gets implemented, they may be useful.
;;;********************************************************************************


(DEFSUBST ADD-V-SUFFIX (WORD)
  "Returns a word created by adding an 'ive' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "e")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ive")
    (STRING-APPEND WORD "ive"))) 


(DEFSUBST ADD-H-SUFFIX (WORD)
  "Returns a word created by adding an 'th' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "y")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ieth")
    (STRING-APPEND WORD "th"))) 


(DEFSUBST ADD-N-SUFFIX (WORD)
  "Returns a word created by adding an 'n' type suffix to WORD."
  (SELECTOR WORD STRING-END-EQUAL
     ("e" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ion"))
     ("y" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ication"))
     (OTHERWISE (STRING-APPEND WORD "en")))) 


(DEFSUBST ADD-X-SUFFIX (WORD)
  "Returns a word created by adding an 'ns' type suffix to WORD."
  (SELECTOR WORD STRING-END-EQUAL
     ("e" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ions"))
     ("y" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ications"))
     (OTHERWISE (STRING-APPEND WORD "ens")))) 


(DEFSUBST ADD-G-SUFFIX (WORD)
  "Returns a word created by adding an 'ing' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "e")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ing")
    (STRING-APPEND WORD "ing"))) 


(DEFSUBST ADD-J-SUFFIX (WORD)
  "Returns a word created by adding an 'ings' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "e")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ings")
    (STRING-APPEND WORD "ings"))) 


(DEFSUBST ADD-D-SUFFIX (WORD)
  "Returns a word created by adding an 'ed' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "d"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "ied"))
      (T (STRING-APPEND WORD "ed"))))) 


(DEFSUBST ADD-T-SUFFIX (WORD)
  "Returns a word created by adding an 'est' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "st"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "iest"))
      (T (STRING-APPEND WORD "est"))))) 


(DEFSUBST ADD-R-SUFFIX (WORD)
  "Returns a word created by adding an 'er' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "r"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "ier"))
      (T (STRING-APPEND WORD "er"))))) 


(DEFSUBST ADD-Z-SUFFIX (WORD)
  "Returns a word created by adding an 'ers' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "rs"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "iers"))
      (T (STRING-APPEND WORD "ers"))))) 


(DEFSUBST ADD-S-SUFFIX (WORD)
  "Returns a word created by adding an 's' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "ies"))
      ((POSITION (GET-CHAR WORD (- LENGTH 1)) (THE STRING (STRING "szxh")) :TEST #'CHAR-EQUAL)
       (STRING-APPEND WORD "es"))
      (T (STRING-APPEND WORD "s")))))

(DEFSUBST ADD-P-SUFFIX (WORD)
  "Returns a word created by adding an 'ness' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (IF (AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
	(STRING-APPEND (subseq WORD 0 (1- LENGTH)) "iness")
	(STRING-APPEND WORD "ness"))))
      



(DEFUN ADD-SUFFIX-TO-WORD (WORD SUFFIX)
  "Makes a new word by adding SUFFIX to WORD."
  (SELECTOR SUFFIX STRING-EQUAL ("y" (STRING-APPEND WORD "ly")) ("v" (ADD-V-SUFFIX WORD))
     ("n" (ADD-N-SUFFIX WORD)) ("x" (ADD-X-SUFFIX WORD)) ("h" (ADD-H-SUFFIX WORD))
     ("g" (ADD-G-SUFFIX WORD)) ("j" (ADD-J-SUFFIX WORD)) ("d" (ADD-D-SUFFIX WORD))
     ("t" (ADD-T-SUFFIX WORD)) ("r" (ADD-R-SUFFIX WORD)) ("z" (ADD-Z-SUFFIX WORD))
     ("s" (ADD-S-SUFFIX WORD)) ("p" (ADD-P-SUFFIX WORD)) ("m" (STRING-APPEND WORD "'s"))
     (OTHERWISE NIL)))

(DEFUN ADD-SUFFIXES (WORD SUFFIX-STRING)
  "Returns a list of words created by adding the suffixes
   in SUFFIX-STRING to WORD."
  (LET ((WORD-LIST NIL))
    (DOTIMES (N (LENGTH SUFFIX-STRING))
      (SETQ WORD-LIST
	    (NCONC WORD-LIST
		   (LIST (ADD-SUFFIX-TO-WORD WORD (STRING (GET-CHAR SUFFIX-STRING N)))))))
    WORD-LIST))

