;;; -*- Mode:Common-Lisp; Package:W; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	Rich Acuff
;;; at the Stanford University Knowledge Systems Lab in Feb '87.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15

;;;  Small, quick monitor windows that display time-varying data by
;;;  calling an update function to get new values periodically.

;;;----------------------------------------------------------------------

;;;  These functions are examples of how to use graphical monitors.

;;; They should get better docs

(defun net:pkts-transmitted ()
  "Number of packets sent over the first network controller."
  (loop for ctl in net:controller-list
	summing (send ctl :pkts-transmitted)
	)
  )

(defun monitor-net-transmissions (&optional (x-pos 812)
				            (y-pos 0))
  "Put up a monitor of network transmissions."
  (make-instance 'graphical-monitor-window
		 :update-fn #'net:pkts-transmitted
		 :name "Net Xmit"
		 :delta? t
		 :max 50
		 :interval 4
		 :position `(,x-pos ,y-pos)
		 :expose-p t
		 )
  )

(defun net:pkts-received ()
  "Number of packets received over the first network controller."
  (loop for ctl in net:controller-list
	summing (send ctl :pkts-received)
	)
  )

(defun monitor-net-receptions (&optional (x-pos 918)
			                 (y-pos 0))
  "Put up a monitor of network receptions."
  (make-instance 'graphical-monitor-window
		 :update-fn #'net:pkts-received
		 :name "Net Recv"
		 :delta? t
		 :max 50
		 :interval 4
		 :position `(,x-pos ,y-pos)
		 :expose-p t
		 )
  )

(defun si:total-disk-ops ()
  "Total number of disk operations (reads and writes)."
  (+ (read-meter '%count-disk-page-reads)
     (read-meter '%count-disk-page-writes)
     )
  )

(defun monitor-disk-ops (&optional (x-pos 706)
				   (y-pos 0))
  (make-instance 'graphical-monitor-window
		 :update-fn #'si:total-disk-ops
		 :name "Disk Ops"
		 :max 200
		 :delta? t
		 :interval 2
		 :position `(,x-pos ,y-pos)
		 :expose-p t
		 )
  )

(defun monitor-consing (&optional (x-pos 600) (y-pos 0))
  (make-instance 'graphical-monitor-window
		 :update-fn #'time:area-size
		 :name "Consing"
		 :max 2500
		 :delta? t
		 :interval 5
		 :position `(,x-pos ,y-pos)
		 :expose-p t
		 )
  )

;;;----------------------------------------------------------------------

;;;  The main part of the monitor

(defflavor graphical-monitor-mixin
   ((interval 2)
    (sample-count 50)
    update-fn
    (min 0)
    (max 100)
    delta?
    new-value			;most recent value from UPDATE-FN
    new-y			;next to be displayed value
    (last-y 0)			;previous displayed value (pixel)
    (last-value 0)		;previous value from UPDATE-FN
    sample-size			;number of pixels per update
    x-scale			;float x scale factor
    y-scale			;float y scale factor
  foo
    )
   ()
  (:documentation :mixin
"When combined with a window, provides a facility for monitoring
time-varying values (eg. number of disk operations) by moving the window
to the left, drawing a line from the old value to the new, and waiting.
The following settable instance variables control the behavior of the
display:

(INTERVAL 2) --  Time in seconds to wait between updates
(SAMPLE-COUNT 50)  --  Number of time slices to display at once
UPDATE-FN  --  A function of no args called to get the next value
(MIN 0)  --  The minimum displayed value
(MAX 100)  --  The maximum displayed value
(DELTA? NIL)  --  T => display changes in value instead of value
")
;  (:required-flavors graphics-mixin stream-mixin essential-window)
  (:initable-instance-variables)
  (:gettable-instance-variables)
  (:settable-instance-variables)
  )

(defmethod (graphical-monitor-mixin :compute-scaling) ()
  "Set up X-SCALE, Y-SCALE, and SAMPLE-SIZE based on SAMPLE-COUNT,
   MAX, MIN, and window size."
  (setf x-scale (/ (send self :inside-width) (float sample-count)))
  (setf y-scale (/ (send self :inside-height) (float (- max min))))
  (setf sample-size (truncate x-scale))
  )

;;;  Methods to make sure the scaling gets kept up to date.

(defmethod (graphical-monitor-mixin :after :init) (&rest ignore)
  "Call :COMPUTE-SCALING."
  (declare (ignore ignore))
  (send self :compute-scaling)
  )
(defmethod (graphical-monitor-mixin :after :set-edges) (&rest ignore)
  "Call :COMPUTE-SCALING."
  (declare (ignore ignore))
  (send self :compute-scaling)
  )
(defmethod (graphical-monitor-mixin :after :set-max) (&rest ignore)
  "Call :COMPUTE-SCALING."
  (declare (ignore ignore))
  (send self :compute-scaling)
  )
(defmethod (graphical-monitor-mixin :after :set-min) (&rest ignore)
  "Call :COMPUTE-SCALING."
  (declare (ignore ignore))
  (send self :compute-scaling)
  )
(defmethod (graphical-monitor-mixin :after :set-sample-count) (&rest ignore)
  "Call :COMPUTE-SCALING."
  (declare (ignore ignore))
  (send self :compute-scaling)
  )

;;; The real loop
(defmethod (graphical-monitor-mixin :monitor) ()
  "Monitor the value of the function stored in the UPDATE-FN instance
   variable.  See GRAPHICAL-MONITOR-MIXIN documentation for control
   details.  Does not normally return."
  (loop
    (send self :update)
    (sleep interval)
    )
  )

(defmethod (graphical-monitor-mixin :update) ()
  "Get the next value to display and send :UPDATE-DISPLAY."
  (setf new-value (funcall update-fn))
  (check-type new-value number)
  (setf new-y (- (send self :height)
		 (send self :bottom-margin-size)
		 1
		 (truncate (* y-scale
			     (- (if delta?
				    (- new-value last-value)
				    new-value)
				min)))))
  (send self :update-display)
  (setf last-y new-y)
  (setf last-value new-value)
  )

(defmethod (graphical-monitor-mixin :update-display) ()
  "Display newly computed values being monitored.  Move the display left
   and draw a new line."
  (send self :move-left)
  (send self :draw-new)
  )

(defmethod (graphical-monitor-mixin :move-left) ()
  "Moves the old picture to the left by SAMPLE-SIZE to allow more
   data.  Loses some data off the left edge."
  ;; Move the bits by SAMPLE-SIZE bits
  (send self :bitblt-within-sheet alu-seta
	(- (send self :width) sample-size)
	(send self :height)
	sample-size
	0 0 0)

  ;; Clear the new area
  (send self :bitblt-within-sheet alu-setz
	sample-size
	(send self :height)
	(- (send self :width) sample-size -1)
	0 0 0)
  )

(defmethod (graphical-monitor-mixin :draw-new) ()
  "Draw a new line from LAST-Y to NEW-Y in the newly cleared area of
   the monitor."
  (send self :draw-line
	(- (send self :inside-width) sample-size)
	last-y
	(1- (send self :inside-width))
	new-y
	1
	black
	alu-seta
	)
  )

;;;----------------------------------------------------------------------

;;;  This instantialble flavor is probably the most useful thing in this
;;;  file.

(defflavor graphical-monitor-window
	   ((label-font fonts:tvfont)
	    label-string
	    idx			;beginning index for number in label string
	    )
	   (graphical-monitor-mixin
	    graphics-mixin
	    stream-mixin
	    minimum-window
	    borders-mixin
	    box-label-mixin
	    label-mixin
	    process-mixin
	    )
  (:documentation
  "A graphics monitor tool built on TV:GRAPHICAL-MONITOR-MIXIN.
   Displays a graphical monitor of UPDATE-FN's value with a label, and
   the most recent value to the right of the label.")
  (:default-init-plist
    :save-bits t
    :deexposed-typeout-action :permit
    :blinker-p nil
    :borders 2
    :label-box-p t
    :inside-height 50
    :inside-width 100
    :process 'top-level-fn
    )
  )

(defmethod (graphical-monitor-window :process-top-level) ()
  "Send SELF a :MONITOR message."
  (send self :monitor)
  )

;;;  These make sure the label stays accurate
(defmethod (graphical-monitor-window :after :init) (&rest ignore)
  "Send :FIXUP-LABEL."
  (declare (ignore ignore))
  (send self :fixup-label)
  )
(defmethod (graphical-monitor-window :after :set-name) (&rest ignore)
  "Send :FIXUP-LABEL."
  (declare (ignore ignore))
  (send self :fixup-label)
  )

(defmethod (graphical-monitor-window :fixup-label) ()
  "Set things up for appending new values to the label."
  (setf label-string
	(with-output-to-string (s)
	  (format s "~A  " name)
	  )
	)
  (setf idx (+ (length name) 2))
  (send self :set-label `(:string ,label-string :font ,label-font))
  )

(defmethod (graphical-monitor-window :after :update-display) (&rest ignore)
  "Append the new value to the label."
  (declare (ignore ignore))
  (unless (= new-y last-y)
    (let ((tem idx))
      (with-output-to-string (s label-string tem)
	(format s "~3d" (if delta?
			    (- new-value last-value)
			    new-value))
	)
      )
    (send self :set-label `(:string ,label-string :font ,label-font))
    )
  )

;;;  The following allows the user to update parameters dynamically.

;;;Edited by Acuff                 31 Oct 87  12:18
;;;Edited by Acuff                 16 Nov 87  15:42
(defmethod (graphical-monitor-window :mouse-click) (mouse-char x y)
  "R => System menu, M => Move Window, M-2 => Clear Window, 
   L => Choose Variables"
  (cond ((= mouse-char (char-int #\mouse-L-1))
	 (process-run-function "Select Monitor Variables"
			       self :select-variables)
	 )
	((= mouse-char (char-int #\mouse-M-1))
	 (if (and (boundp '*dragging-enabled?*) *dragging-enabled?*)
	     (drag-window-from self x y)
	     (process-run-function
	       '(:name "Set Window Position" :priority 1)
	       #'(lambda (window) (mouse-set-window-position window))
	       self)
	     )
	 )
	((= mouse-char (char-int #\mouse-M-2))
	 (send self :clear-screen)
	 )
	((= mouse-char (char-int #\mouse-R-1))
	 (tv:mouse-call-system-menu)
	 )
	)	
  )

;;;Edited by Acuff                 31 Oct 87  14:35
(defmethod (graphical-monitor-window :deexposed-mouse-buttons) (mask x y)
  "Call standard :MOUSE-BUTTONS method."
  (send self :mouse-buttons mask x y)
  )

;;;Edited by Acuff                 31 Oct 87  14:35
(defmethod (graphical-monitor-window :deexposed-who-line-documentation-string) ()
  "Call normal :WHO-LINE-DOCUMENTATION-STRING."
  (send self :who-line-documentation-string)
  )

;;;Edited by Acuff                 31 Oct 87  14:35
(defmethod (graphical-monitor-window :who-line-documentation-string) ()
  "Doc for who line when mouse is over a GRAPHICAL-MONITOR-WINDOW."
  '(:mouse-L-1 "Change Monitor Control Variables"
    :mouse-M-1 "Move Window"
    :mouse-M-2 "Clear Window"
    :mouse-R-1 "System Menu")
  )

(defun constrain-positive (n)
  (if (plusp n)
      nil
      "Must be a positive number")
  )

(defmethod (graphical-monitor-window :select-variables) ()
  (let ((*min* min)
	(*max* max)
	(*sample-count* sample-count)
	(*interval* interval)
	(*delta?* delta?)
	(*update-fn* update-fn)
	)
    (declare (special *min* *max* *sample-count* *interval*
		      *delta?* *update-fn*))
    (tv:choose-variable-values
      `((*interval* "Interval (sec)"
	 :documentation "Number of seconds between updates, possibly less than 1"
	 :number)
	(*sample-count* "Sample Count"
	 :documentation "Number of samples to display at once."
	 :number
	 )
	(*delta?* "Display Changes?"
	 :documentation "Yes to follow changes in value, no to follow value"
	 :boolean)
	(*update-fn* "Update Function"
	 :documentation "Function that returns new integer values"
	 :any)
	(*max* "Maximum Displayed Value"
	       :documentation "Highest value that will be displayed"
	       :number)
	(*min* "Minimum Displayed Value"
	 :documentation "Lowest value that will be displayed"
	 :number)
	)
      :label (string-append (send self :name) " Control Parameters")
      :margin-choices '("Done"
			("Abort" (signal-condition eh:abort-object)))
      )
    (setf max *max*)
    (setf min *min*)
    (setf sample-count *sample-count*)
    (setf interval *interval*)
    (setf delta? *delta?*)
    (setf update-fn *update-fn*)
    )
  (send self :compute-scaling)
  )
	 
(defmethod (graphical-monitor-window :mouse-select) (&rest ignore)
  "Cause the window to be exposed when it's clicked on while partially
   visable."
  (declare (ignore ignore))
  (send self :expose)
  )

;;;----------------------------------------------------------------------

(defun top-level-fn (window)
  "Send :PROCESS-TOP-LEVEL to WINDOW."
  (send window :process-top-level)
  )

(compile-flavor-methods graphical-monitor-window)
