;;; -*- Mode:Common-Lisp; Package:EH; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;;; Maybe we should put in more checks for binding nil.
;;; Maybe we should also have some sort of cache flushing for the window
;;; debugger when we setq some variable.

(defmacro with-frame-environment ((sg frame) &body body)
 `(multiple-value-bind (start end)
      (catch-error (sg-frame-special-pdl-range ,sg ,frame) nil)
    (let ((rp (sg-regular-pdl ,sg)))
         (let ((function (rp-function-word rp ,frame))
	       (sp (sg-special-pdl ,sg))
	      )
	      ,@body
	 )
    )
  )
)

(defun Compute-Locals-Etc-Binding-Specs
       (vars-and-vals coerce-packages-p body-function)
  "Computes the actual symbol-macrolet type form for the body function so that
   we can have the right things bound."
  (let ((vars nil)
	(vals nil)
	(locs nil)
       )
       (declare (special vars vals))
       (loop for (var val loc) in vars-and-vals
	     when (and (symbolp var) (not (eq :***Not-This-One*** loc)))
	     do (if (and coerce-packages-p
			 (not (eq *package* (symbol-package var)))
		    )
		    (progn (push (intern (symbol-name var) *package*) vars)
			   (push val vals)
			   (push loc locs)
		    )
		    nil
		)
		(push var vars)
		(push val vals)
		(push loc locs)
       )
       (let ((*binding-macrolets*
	       (loop for var in vars
		     for loc in locs
		     when (not (eq :***Not-This-One*** loc))
		     Collect (list var (list 'contents loc))
	       )
	     )
	    )
	    (declare (special *binding-macrolets*))
	    (loop for (loc fun name) in vars-and-vals
		   when (locativep loc)
		   do (push (list :Function
				  #'(lambda (&rest args) (apply fun args))
				  name
			    )
			    *binding-macrolets*
		      )
	    )
	    (ignore vars vals *binding-macrolets*)
	    (funcall body-function vars vals *binding-macrolets*)
       )
  )
)

(defun Maybe-With-Locals-And-Args-Bound-Internal
       (coerce-packages-p all-p body-function)
  (if (and *error-sg* *current-frame*)
      (let ((vars-and-vals
	      (if all-p
		  (get-all-bindings-for-fef *error-sg* *current-frame*)
		  (With-Frame-Environment (*error-sg* *current-frame*)
		    (append (Get-Locals-For-This-Fef
			      *error-sg* *current-frame*
			      start end function rp sp
			    )
			    (loop for (var val loc) in
				  (Get-Args-For-This-Fef
				    *error-sg* *current-frame*
				    start end function rp sp
				  )
				  unless (get var 'special)
				  collect (list var val loc)
			    )
		    )
		  )
	      )
	    )
	   )
	   (Compute-Locals-Etc-Binding-Specs
	     vars-and-vals coerce-packages-p body-function
	   )
      )
      (let ((vars nil)
	    (vals nil)
	    (*binding-macrolets* nil)
	   )
	   (declare (special vars vals *binding-macrolets*))
	   (ignore vars vals)
	   (funcall body-function vars vals *binding-macrolets*)
      )
  )
)

(defmacro Maybe-With-Locals-And-Args-Bound
  ((&optional (coerce-packages-p t) (spec-bind-p t) (all-p t)) &body body)
"Binds up specials for all of the args, locals and specials in the frame.
 Binds VARS to the list of vars and VALS to the list of vals."
  (let ((body-part (if spec-bind-p `(progv vars vals ,@body) `(progn ,@body))))
      `(Maybe-With-Locals-And-Args-Bound-Internal
	 ,coerce-packages-p ,all-p
	 #'(lambda (vars vals *binding-macrolets*)
	     (declare (special vars vals *binding-macrolets*))
	     ,body-part
	   )
       )
  )
)


(defmethod (debugger-frame :Eval-This) (form)
  ;;; Evals with self etc. bound correctly.
  (eval form)
)

;;; TI Code.
(DEFMETHOD (DEBUGGER-FRAME :AROUND :FETCH-AND-EXECUTE) (CONT MT IGNORE)
  "Set a couple of variables and check for command aborts."
  (DECLARE (SPECIAL PKG))
  (SETQ PKG (SYMEVAL-IN-STACK-GROUP '*PACKAGE* *ERROR-SG*))
  (SETQ *package*
	; fix here by JPR to make sure that we don't bind to kwd package.
	(IF (and (packagep pkg) (not (equal (find-package 'keyword) pkg)))
	    PKG
	    (FIND-PACKAGE "USER")))
  (let ((me self))
  ;;; Fix here by JPR to make sure we can type in expressions.
    (let-if (not (equal *readtable* sys:common-lisp-readtable))
	    ((*readtable* sys:common-lisp-readtable))
    ;;; Bind up locals just in case we need them.  JPR.
      (Maybe-With-Locals-And-Args-Bound ()
	;;; Using ME makes sure that SELF is who we think it should be
	;;; by the time that this continuation gets called.
	(send me :Eval-This
	      `(CATCH 'QUIT
		 (FUNCALL-WITH-MAPPING-TABLE ,CONT ,MT :FETCH-AND-EXECUTE)))))))


(defun setup-locals-window (window sg frame rest-arg-p)
  (let* (list
         (rp (sg-regular-pdl sg))
         (function (rp-function-word rp frame))
         (sp (sg-special-pdl sg))
         start end self-value)
    ;; Print the locals if this is a fef
    (if (typep function 'compiled-function)
        (do ((n-locals (fef-number-of-locals function))
             (local-idx (sys:rp-local-offset sg rp frame))
             (i 0 (1+ i)))
;;	   (J (+ FRAME (RP-LOCAL-BLOCK-ORIGIN RP FRAME)) (1+ J)))
	    ;; * old way to find locals *
            ((>= i n-locals))
          (cond ((not (and rest-arg-p (zerop i)))
		 ;;;Don't show rest arg (local 0) twice
                 (push (list (local-name function i)   ; Local Name
;;			   (AREF RP J)
                             (aref rp (+ i local-idx)) ; Local Value
                             i)                        ; Local Number
                       list)))))
    (let ((closure-variables
	    (catch-error (get-lexical-environment-for-this-fef
			   sg frame start end function rp sp
			 )
			 nil
	    )
	  )
	 )
         (if closure-variables
	     (progn (push "" list)
		    (push "Closure Variables Visible in this Function:" list)
		    (loop for (var val loc) in (nreverse closure-variables)
			  do (push (list var val) list)
		    )
	     )
	     nil
	 )
    )
    ;; Print the specials if any
    (multiple-value-setq (start end)
                         (sg-frame-special-pdl-range sg frame))

    (multiple-value-bind (specials-list self-val)
	(compute-specials-for-locals-window function sg frame sp start end)
      (setq self-value self-val)
      (setq list (append specials-list list))
    )
    ;; if SELF is mentioned in this frame, include its instance variables:
    (if (and self-value
             (typep self-value 'instance))
        (let* ((self-flavor 
                 (si:instance-flavor self-value))
               (self-vars (si:flavor-all-instance-variables-slow self-flavor)))
          (push "" list)
          (push "Non-special instance variables of SELF:" list)
          (do ((sv self-vars (cdr sv))
               (i 1 (1+ i)))
              ((null sv))
            (cond ((not (si:assq-careful (car sv) list))
                   (push (list (car sv)
                               (multiple-value-bind (val error)       ;Value
                                  (catch-error (%instance-ref self-value i) nil)
                                 (if error "unbound" val)))
                         list))))))
    (send window :setup (list 'print-arg-or-local '(local "Local")
			      (nreverse list)))))

(defun Make-Function-Binding-Spec
       (local-function-name fdefinition from-function)
  (list (locf (symbol-function local-function-name))
	(if fdefinition
	    fdefinition
	    #'(lambda (&rest ignore)
		(beep)
		(format t
	       "~&Local function ~S has been inlined and cannot be found in ~S."
			local-function-name (function-name from-function)
		)
		(throw :Abort-Eval :non-found-function)
	      )
	)
	local-function-name
  )
)

(defun get-local-functions-for-this-fef (sg frame start end function rp sp)
  (ignore sg frame start end rp sp)
  (let ((dbis (sys:get-debug-info-struct function t)))
       (let ((locals (getf (sys:dbis-plist dbis) :Internal-Fef-Names)))
	    (loop for local in locals
		  for fun = (sys:fdefinition-safe
			      `(:Internal ,(function-name function) ,local)
			    )
		  collect (Make-Function-Binding-Spec local fun function)
	    )
       )
  )
)

(defun get-all-bindings-for-fef (sg frame)
"Returns an alist of all of the locals, args, IVs, lexical variables
 and specials bound in this frame."
  (With-Frame-Environment (sg frame)
    (let ((args (catch-error
		  (Get-Args-For-This-Fef sg frame start end function rp sp)
		  nil
		)
	  )
	 )
	 (append (catch-error
		   (Get-Flavors-Instance-Variables-For-This-Fef
		     sg frame start end function rp sp
		   )
		   nil
		 )
		 (catch-error
		   (Get-CLOS-Instance-Variables-For-This-Fef
		     sg frame start end function rp sp args
		   )
		   nil
		 )
		 (catch-error
		   (Get-Specials-For-This-Fef sg frame start end function rp sp)
		   nil
		 )
		 (catch-error
		   (Get-Locals-For-This-Fef sg frame start end function rp sp)
		   nil
		 )
		 args
		 (catch-error
		   (Get-Lexical-Environment-For-This-Fef
		     sg frame start end function rp sp
		   )
		   nil
		 )
		 (catch-error
		   (Get-Local-Functions-For-This-Fef
		     sg frame start end function rp sp
		   )
		   nil
		 )
	 )
    )
  )
)

(defun function-matches-p (function other-function)
"Is true if the function we are looking for (function) matches
the other function."
  (or (equal function other-function)
      (and (consp other-function)
	   (eq :Internal (first other-function))
	   (function-matches-p function (second other-function))
      )
  )
)

(defun bindings-for-function-in-sg (function sg eh-sg &optional (barf-p t))
"Returns an alist of all of the local, arg and special bindings for Function in
SG.  If barf-p then we barf if there is no frame for Function in SG.
Eh-SG is the stack group being used by the debugger (or nil).  If we know the
eh-sg then we look in it for the current frame and then look down the stack
from there, otherwise we look from the top of the stack."
  (declare (values bindings barf-p))
  (let ((current-frame
	  (if eh-sg
	      (let ((try (without-interrupts
			   (symeval-in-stack-group '*current-frame* eh-sg)
			 )
		    )
		   )
		   (if (numberp try)
		       try
		       nil
		   )
	      )
	      nil
	  )
	)
       )
       (let ((frame-and-function
		;;; Start at current frame if known, else top of stack.
	       (do ((frame (or current-frame (sg-innermost-frame sg))
			   ;;; Just in case anything goes wrong.
			   (catch-error (sg-next-frame sg frame) nil)
		    )
		    (rp (sg-regular-pdl sg))
		    (name))
		   ((null frame) nil)
		 (setq name (function-name (rp-function-word rp frame)))
		 (and (function-matches-p (function-name function) name)
		      (return (list frame (rp-function-word rp frame)))
		 )
	       )
	     )
	    )
	    (if (null frame-and-function) ;;; Couldn't find a frame.
	        (if eh-sg
		    ;;; Recurse, with no eh-sg so we look again from the top.
		    (bindings-for-function-in-sg function sg nil barf-p)
		    (if barf-p
			(ferror nil "Could not find frame for function ~S."
				function
			)
			(values nil t)
		    )
		)
		(values (get-all-bindings-for-fef sg (first frame-and-function))
			nil
		)
	    )
       )
  )
)

(defun bindings-for-function-in-halted-process (function &optional (barf-p t))
"Returns an alist of all of the local, arg and special bindings for Function in
the first halted seeming process that it finds that has a frame for Function.
If barf-p then we barf if there is no process with a frame for Function in SG."
  (declare (values bindings barf-p process))
  (loop for process in tv:all-processes
	when (and (typep process 'sys:process)
		  (send process :Initial-Stack-Group)
		  (not (equal (send process :Stack-Group)
			      (send process :Initial-Stack-Group)
		       )
		  )
	     )
	do (multiple-value-bind (bindings barf-p)
	       (without-interrupts
		 (Bindings-For-Function-In-Sg
		   function (send process :Initial-Stack-Group)
		   (send process :Stack-Group) nil
	         )
	       )
	     (if barf-p
		 nil
		 (return bindings nil process)
	     )
	   )
	finally (if barf-p
		    (ferror nil "Could not find process running ~S" function)
		    (return nil t nil)
		)
  )
)

(defun bindings-for-function-in-all-halted-processes
       (function &optional (barf-p t))
"Given a function returns a list of the form:
  ((bindings1 process1) (bindings2 process2)...)
for all of the processes that have a stack frame for Function.
The bindings are an alist mapping the names of all of the locals,
args and specials references/bound for that frame and their
associatyed values.  If barf-p is true then we signal an error
if something goes wrong, otherwise we return a true barf-p value.
"
  (declare (values list-of-binding-list-and-process barf-p))
  (let ((results ;;; Collect the list of binding/process pairs for
	         ;;; matching processes
	  (loop for process in tv:all-processes
		when (and (typep process 'sys:process)
			  (send process :Initial-Stack-Group)
			  (not (equal (send process :Stack-Group)
				      (send process :Initial-Stack-Group)
			       )
			  )
		     )
		collect (multiple-value-bind (bindings barf-p)
			    (without-interrupts
			      (Bindings-For-Function-In-Sg
				function (send process :Initial-Stack-Group)
				(send process :Stack-Group) nil
			      )
			    )
			  (if barf-p
			      :barf
			      (list bindings process)
			  )
			)
	  )
	)
       )
       (if barf-p
	   (cond ((not results)
		  (ferror nil "Could not find process running ~S" function)
		 )
		 ((member :barf results)
		  (ferror nil "A problem occured in finding bindings for ~S"
			  function
		  )
		 )
		 (t results)
	   )
	   (values (remove :Barf results)
		   (or (not results) (member :Barf results))
	   )
       )
  )
)

(defun Get-Flavors-Instance-Variables-For-This-Fef
       (sg frame start end function rp sp)
"Gets an alist for the flavors instance variables for this fef, mapping the
iv names to their values."
  (ignore start end function rp sp)
  (let ((self-value (catch-error (symeval-in-stack-group 'self sg frame) nil))
	(list nil)
       )
       (if (and self-value (typep self-value 'instance))
	   (let* ((self-flavor 
		    (si:instance-flavor self-value))
		  (self-vars
		    (si:flavor-all-instance-variables-slow self-flavor)))
	     (do ((sv self-vars (cdr sv))
		  (i 1 (1+ i)))
		 ((null sv))
	       (cond ((not (si:assq-careful (car sv) list))
		      (push (list (car sv)
				  (multiple-value-bind (val error)	;Value
				      (catch-error (%instance-ref self-value i)
						   nil)
				    (if error "unbound" val))
				  (%instance-loc self-value i)) ; Locative
			    list))))))
       list
  )
)

(defun parent-function-name (function-name)
"Given a function name returns the name of the parent function if this an
internal type function."
  (if (and (consp function-name) (equal :Internal (first function-name)))
      (parent-function-name (second function-name))
      function-name
  )
)

(defun instance-variable-spec-list (instance specializer)
  (if (typep specializer 'ticlos:flavor-class)
      (loop for slot-name
	    in (sys:flavor-all-instance-variables
		 (get (clos:class-name specializer) 'sys:flavor)
	       )
	    ;;; Only return bound slots.
	    when (boundp-in-instance instance slot-name)
	    collect
	    (list slot-name ; name
		  (symeval-in-instance instance slot-name) ; value
		  (locf (symeval-in-instance instance slot-name)) ; locative
	    )
      )
      (loop for slot
	    in (clos:class-slots specializer)
	    for slot-name = (clos:slot-definition-name slot)
	    ;;; Only return bound slots.
	    when (clos:slot-boundp instance slot-name)
	    collect
	    (list slot-name ; name
		  (clos:slot-value instance slot-name) ; value
		  (ticlos:slot-location instance slot-name) ; locative
	    )
      )
  )
)

(defun Get-CLOS-Instance-Variables-For-This-Fef
       (sg frame start end function rp sp args)
"Returns an alist binding the names of all slots of all class specialized args
to methods to their values in tha appropriate argument instances.
"
  (ignore frame sg start end rp sp)
  (let ((parent (parent-function-name (function-name function))))
       (if (and (consp parent) (equal 'clos:method (first parent)))
	   ;;; We now know that this is a CLOS method.
	   (let ((method (tv:method-from-method-function-safe
			   (sys:fdefinition-safe parent)
			 )
		 )
		)
	        (let ((specializers (clos:method-specializers method))
		      (arg-names (arglist function))
		     )
		     (apply #'append
			    (loop for spec in specializers
				  for arg in arg-names
				  for instance = (second (assoc arg args))
				  when (and (typep spec 'clos:class)
					    (symbolp arg) (instancep instance)
				       )
				  collect
				    (Instance-Variable-Spec-List instance spec)
			    )
		     )
		)
	   )
	   nil
       )
  )
)

(defvar *not-these-specials*
	'(sys:keyword-garbage
	  ucl:kbd-input
	  *standard-input*
	  *standard-output*
	  *terminal-io*
	  *error-output*
	  *debug-io*
	  *trace-output*
	  *query-io*
	 )
  "A list of specials that it looks too dangerous to bind."
)

(defun get-specials-for-this-fef
       (sg frame start end function rp sp
	&aux (list nil) (self-value nil)
	(self-mapping-table-value nil) (local-specials nil)
       )
"Returns an alist of the special bindings appropriate for Frame."
  (ignore rp)
  (push (list 'self
	      (catch-error
		(symeval-in-stack-group 'self sg frame) nil)
	      :***Not-This-One***)
	list)
  (if (not self-mapping-table-value)
      (push (list 'sys:self-mapping-table
		  (catch-error
		    (symeval-in-stack-group
		      'sys:self-mapping-table sg frame) nil)
		  :***Not-This-One***)
	    List))
  (when start
    (do ((i start (+ i 2)))
	((>= i end))
      (if (eq 'self (symbol-from-value-cell-location (aref sp (1+ i))))
	  (setq self-value (aref sp i)))
      (if (eq 'sys:self-mapping-table
	      (symbol-from-value-cell-location (aref sp (1+ i))))
	  (setq self-mapping-table-value (aref sp i)))
      (pushnew (symbol-from-value-cell-location (aref sp (1+ i)))
	       local-specials :test #'eq)
      (let ((sym (symbol-from-value-cell-location (aref sp (1+ i)))))
	   (if (not (assoc sym list :Test #'eq))
	       (push (list sym					;Name
			   (multiple-value-bind (val error)	;Value
			       (catch-error (aref sp i) nil)
			     (if error "unbound" val))
			   (aref sp (1+ i)))			;Locative
		     list)
	       nil
	   )
      )
    )
  )
  (multiple-value-bind
     (referenced-ivars referenced-keywords problem
      referenced-functions referenced-generic-functions args returned-values
      locals specials-referenced specials-bound
     )
      (tv:ivars-and-messages-in-method function)
    (ignore referenced-ivars referenced-keywords problem
	    referenced-functions referenced-generic-functions args
	    returned-values locals specials-bound
    )
    (let ((unbound-specials
	    (loop for spec in specials-bound
		  unless (member spec local-specials)
		  collect spec
	    )
	  )
	 )
         (if unbound-specials
	     (loop for spec in unbound-specials
		   ;;; Sometimes we get keywords and (more dangerous)
		   ;;; sometimes we get sys:% vars like
		   ;;; %disk-switches, so don't frob these, it's too
		   ;;; risky.
		   unless (or (keywordp spec)
			      (member spec *Not-These-Specials* :Test #'eq)
			      (And (symbolp spec)
				   (position
				     #\% (the string (symbol-name spec))
				     :Test #'char=
				   )
			      )
			  )
		   do (push (value-for-locals-window spec sg frame) list)
	     )
	     nil
	 )
    )
    (let ((specials (if (typep function 'compiled-function)
			(loop for spec in specials-referenced
			      ;;; Sometimes we get keywords and (more dangerous)
			      ;;; sometimes we get sys:% vars like
			      ;;; %disk-switches, so don't frob these, it's too
			      ;;; risky.
			      unless (or (keywordp spec)
					 (member spec *Not-These-Specials*
						 :Test #'eq
					 )
					 (and (symbolp spec)
					      (position
						#\%
						(the string (symbol-name spec))
						:Test #'char=
					      )
					 )
				     )
			      collect (value-for-locals-window spec sg frame)
			)
			nil
		    )
	  )
	 )
         (if specials
	     (loop for (name value location) in specials
		   unless (assoc name list :Test #'eq)
		   do (push (list name value location) list)
	     )
	     nil
	 )
    )
  )
  list
)

(defun collect-lexical-variables-from-parent (function environment-location)
"Is passed a function object and the locative to its lexical environment.
It returns an alist for the closure variables for itself and all of 
its lexical parents."
  (let ((dbis (sys:get-debug-info-struct function))
	(environment
	  (catch-error (sys:find-structure-header environment-location) nil)
	)
       )
       (if dbis
	   (let ((parent
		   (getf (sys:dbis-plist dbis) :Lexical-Parent-Debug-Info)
		 )
		 (parent-env (third environment))
		)
	        (let ((vars (if parent
			        (getf (sys:dbis-plist parent)
				      :Variables-Used-In-Lexical-Closures
				)
				nil
		            )
		      )
		     )
		     (append (loop for var in vars
				   for loc on (rest (rest (rest environment)))
				   for val = (first loc)
				   collect (list var val (locf (first loc)))
			     )
			     (if parent
				 (collect-lexical-variables-from-parent
				   (sys:fdefinition-safe
				     (second (function-name function)) t
				   )
				   parent-env
				 )
				 nil
			     )
		     )
		)
	   )
	   nil
       )
  )
)

(defun get-lexical-environment-for-this-fef (sg frame start end function rp sp)
"Returns an alist of the lexical environment bindings appropriate for Frame."
  (ignore sp start end)
  (if (and (consp (function-name function))
	   (equal :Internal (first (function-name function)))
      )
      (let ((local-idx (sys:rp-local-offset sg rp frame)))
	   (let ((environment-locative
		   (if (>= (fef-number-of-locals function)
			   sys:lex-parent-env-reg
		       )
		       (aref rp (+ sys:lex-parent-env-reg local-idx))
		       nil
		   )
		 )
		)
	        (if (and environment-locative (locativep environment-locative))
		    (collect-lexical-variables-from-parent
		      function environment-locative
		    )
		    nil
		)
	   )
      )
      nil
  )
)

(defun get-locals-for-this-fef (sg frame start end function rp sp)
"Returns an alist of the local bindings appropriate for Frame."
  (ignore sp start end)
  (let ((list nil))
    (multiple-value-bind (rest-arg-value rest-arg-p lexpr-call)
	(sg-rest-arg-value sg frame)
      (ignore rest-arg-value rest-arg-p lexpr-call)
      (if (typep function 'compiled-function)
	  (do ((n-locals (fef-number-of-locals function))
	       (local-idx (sys:rp-local-offset sg rp frame))
	       (i 0 (1+ i)))
	      ((>= i n-locals))
	    (cond ((not (and rest-arg-p (zerop i)))
		   (push (list (local-name function i)		; Local Name
			       (aref rp (+ i local-idx))	; Local Value
			       (locf (aref rp (+ i local-idx)))); Local Location
			 list)))))
      (loop for (name value location) in list
	    when (and name (symbolp name))
	    collect
	      (if (string-equal "#'" (string name) :End2 2)
		  (let ((fname (find-symbol (subseq (string name) 2)
					    (symbol-package name))))
		       (let ((closure
			       (and fname (sys:fdefinition-safe
					    `(:Internal
					       ,(function-name function) ,fname)
					     t))))
			    (Make-Function-Binding-Spec
			      fname (and closure value) function)))
		  (list name value location))))))

(defun sg-rest-arg-location (sg frame)
  "Returns a locative to the rest arg in FRAME for SG."
  (let* ((rp (sg-regular-pdl sg))
	 (function (rp-function-word rp frame))
	 (nargs-supplied (rp-number-of-arguments rp frame))
	 (nargs-max nargs-supplied) nargs-min
	 lexpr-call-p rest-arg-p)
    (when (legitimate-function-p function)
      (multiple-value-setq (nargs-min nargs-max rest-arg-p)
	(args-desc function)))
    (when (and rest-arg-p
	       (typep function 'compiled-function)
	       (= nargs-max nargs-supplied)
	       (aref rp (rp-local-offset sg rp frame)))  ;local 0 is rest arg
      (setq lexpr-call-p t))
    (values
      ;; locative to rest arg:
      (cond ((and (typep function 'compiled-function)
		  rest-arg-p)  ;;dont look for a rest arg if there isn't one
	     ;; for a compiled function, local 0 is the rest arg
	     (locf (aref rp (rp-local-offset sg rp frame))))  ;local 0 
	     ;; for a lambda function, rest args should be on stack
	     ((> nargs-supplied nargs-max)
	      (locf (aref rp (+ (rp-argument-offset sg rp frame) 
				nargs-max))))
	     (t nil))
      rest-arg-p
      lexpr-call-p)))


(defun get-args-for-this-fef (sg frame start end function rp sp)
"Returns an alist of the arg bindings appropriate for Frame."
  (ignore sp start end)
  (let* ((list nil)
         (argument-index (sys:rp-argument-offset sg rp frame))
         (nargs-supplied (rp-number-args-supplied rp frame))
         (nargs-to-print (sg-number-of-spread-args sg frame))
         nargs-expected nargs-required
         lexpr-call rest-arg-p rest-arg-value)
    (if (or (typep function 'compiled-function)
            (and (typep function 'cons)
		 (member (first function) si:function-start-symbols)))
        (multiple-value-setq (nargs-required nargs-expected)
	  (si::args-desc function)))
    (multiple-value-setq (rest-arg-value rest-arg-p lexpr-call)
                         (sg-rest-arg-value sg frame))
    (dotimes (i nargs-to-print)
      (if (and nargs-expected (= i nargs-expected)) (return nil)) ;; Extraneous
      (let ((missing (and nargs-required
                          (> nargs-required nargs-supplied)
                          (>= i nargs-supplied))))
        (push (list (arg-name function i)             ;Arg name
                    (or missing (aref rp (+ argument-index i)))	;Arg Value
		    (or missing (locf (aref rp (+ argument-index i)))))
              list)))
    (if (or rest-arg-p lexpr-call)
        (push (list (or (and rest-arg-p (local-name function 0))  ;Rest Arg Name
			;;; Just in case this is an anonymous &rest (&key) arg.
			'*rest-arg*
		    )
                    rest-arg-value                            ;Rest Arg Value
		    (sg-rest-arg-location sg frame))          ;Rest ARG Locative
              list))
    (values (nreverse list) rest-arg-p))
)

(defun move-evalhook (form prelude-forms)
  (if (and (consp form) (equal 'evalhook (first form))
	   (consp (second form)) (equal 'quote (first (second form)))
      )
     `(evalhook ',(append prelude-forms (list (second (second form))))
		,@(rest (rest form))
      )
      (append prelude-forms (list form))
  )
)

(defmacro symbol-macrolet-or-macrolet ((&rest clauses) &body body)
  (let ((funs (loop for (key fun name) in clauses
		    for clause in clauses
		    when (eq :Function key)
		    collect `(,name (&rest args)
			      (cons 'funcall (cons ',fun args))
			     )
	      )
	)
	(syms (loop for (key) in clauses
		    for clause in clauses
		    when (not (eq :Function key))
		    collect clause
	      )
	)
       )
      `(macrolet (,@funs) (clos:symbol-macrolet (,@syms) ,@body))
  )
)

;;; TI code.
ucl:
(Defun ucl:CHECK-TOP-LEVEL (expression &Optional replace?)
  "Test for typing errors in expression.  Returns a string if an error is found, else NIL.  If
REPLACE?  is not NIL then do RPLACA's on symbols in the wrong package"
  (DECLARE (SPECIAL user:rh-error-search eh:*binding-macrolets*)) ;;; JPR
  (COND ((ATOM expression)
	 (WHEN (AND (SYMBOLP expression)
		    (With-Otl-And-Sg NIL (NOT (Boundp-Regardless expression)))) ;Check for atoms bound
;?		    (not (With-Otl-And-Sg NIL (Boundp-Regardless expression)))) ;Check for atoms bound
	   (LET ((new-atom (AND replace?
                                (With-Otl-And-Sg (*QUERY-IO*) (si:DWIMIFY-PACKAGE-0 expression 'Boundp-Regardless)))))
	     (IF new-atom
		 (VALUES NIL new-atom)
                 (PROGN (SETQ USER:RH-ERROR-SEARCH expression)
                        (FORMAT NIL "~S not bound at top level" expression))))))
	((AND (SYMBOLP (CAR expression)) (FBOUNDP (CAR expression)))
         ;;;Check for Function bound
	 (Check-Function-Arglist expression replace?))
	((AND (SYMBOLP (CAR expression))
	      (and (boundp 'eh:*binding-macrolets*) ;;; JPR
		   (member (CAR expression) eh:*binding-macrolets*
			   :Key #'third)));;;Check for Function bound
	 nil) ;;; Local function, so ok.
	;;Check for lambda expressions and substs, such as ((lambda (x) x) 43) or ((subst (x) x) 43).
	((AND (CONSP (CAR expression))
	      (OR (FUNCTIONP (CAR expression))
		  (AND (CONSP (CAR expression))
		       (SYMBOLP (CAAR expression))
		       (GET (CAAR expression) 'LAMBDA-MACRO))))
	 (Check-Function-Arglist expression replace?))
	;;Check for function in wrong package
	(replace?
	 (LET ((new-function (SI:DWIMIFY-PACKAGE-0 (CAR expression) 'FDEFINEDP)))
	   (IF (NULL new-function)
               (progn
                 (SETQ USER:RH-ERROR-SEARCH (CAR expression))
                 (FORMAT NIL "~S is not a function" (CAR expression)))
               (PROGN
                 (RPLACA expression new-function)
                 (Check-Function-Arglist expression replace?)))))
	(T (SETQ USER:RH-ERROR-SEARCH (CAR expression))
	   (FORMAT NIL "~S is not a function" (CAR expression)))))

;;; TI code.
ucl:
(Defmethod (ucl:Top-Level-Functions :HANDLE-TYPEIN-P) (expression type)
  (WHEN (EQ type 'CONS)
    ;; This binding is NOT wasteful; CHECK-TOP-LEVEL must be called before the check for FBOUNDP,
    ;; in case user corrects function's package.
    (LET ((error? (eh:Maybe-With-Locals-And-Args-Bound ()
		    (Check-Top-Level expression DWIMIFY-PACKAGE-P))))
      (VALUES (UNLESS error? Self) error?))))

;;; TI code.
ucl:
(Defmethod (ucl:Top-Level-Functions :EXECUTE) (ignore)    ;dan*
  ;; Flag indicating EVALing from UCL top level.
  (Let ((SI:LISP-TOP-LEVEL-INSIDE-EVAL T))
    (with-stack-list (ll 'quote -)
      (with-stack-list (l 'evalhook ll nil nil '(and (boundp '*env* ) *env*))
	;;; Bind up the locals.  JPR.
	(eh:Maybe-With-Locals-And-Args-Bound (t nil)
	  (let ((*default-top-level-function-execute-specials*
;		  (progn #+ignore append #+ignore (loop for var in eh:vars
;				for val in eh:vals
;				when (not (eq val 'eh:--unbound--))
;				collect var
;			  )
			  *default-top-level-function-execute-specials*
;		  )
		)
	       )
	       (ucl:with-otl-and-sg
		 *default-top-level-function-execute-specials*
		 (si:*eval
		   (eh:Move-Evalhook
		     l `(eh:Symbol-Macrolet-Or-Macrolet
			  ,eh:*binding-macrolets*))))))))))


;;; TI Code.
ucl:
(defmethod (ucl:top-level-symbols :Handle-Typein-P) (expression type)
  (when (not (member type '(cons :Implicit-List) :Test #'equal))
    ;; Kludge.  The other modes sometimes take an atom and change it to a list to simplify things.
    (when (consp expression)
      (setq expression (car expression)))
    ;;; Bind up the locals.  JPR.
    (eh:Maybe-With-Locals-And-Args-Bound ()
      (multiple-value-bind (error new-value)
	  (check-top-level expression dwimify-package-p)
	(when new-value (setq - new-value
			      expression new-value))
	;; This NOT doesn't work inside the with-otl.  Why?
	(values (unless (and (symbolp expression)
			     (not (with-otl-and-sg
				    eh:vars (boundp-regardless expression))))
		  self)
		error)))))

;;; TI code.
ucl:
(defmethod (ucl:Top-Level-Symbols :Execute) (ignore)    ;dan*
  ;; Kludge.  The other modes sometimes take an atom and change it to a list to simplify things.
  (when (consp -)
    (setq - (car -)))
  ;; Flag indicating EVALing from UCL top level.
  (let ((si:lisp-top-level-inside-eval T))
    (with-stack-list (ll 'quote -)
      (with-stack-list (l 'evalhook ll nil nil '(and (boundp '*env* ) *env*))
	;;; Bind up the locals.  JPR.
	(eh:Maybe-With-Locals-And-Args-Bound ()
	  (let ((*default-top-level-symbols-execute-specials*
;		  (append (loop for var in eh:vars
;				for val in eh:vals
;				when (not (eq val 'eh:--unbound--))
;				collect var
;			  )
			  *default-top-level-symbols-execute-specials*
;		  )
		)
	       )
	    (ucl:with-otl-and-sg
	      *default-top-level-symbols-execute-specials* 
	      (si:*eval
		(eh:Move-Evalhook
		  l `(eh:Symbol-Macrolet-Or-Macrolet
		       ,eh:*binding-macrolets*))))))))))




;-------------------------------------------------------------------------------


(defun get-bindings-for-function-in-one-or-more-processes
       (function all-processes-p)
"Returns an alist of bindings for function for all locals, args and specials
in a halted erroring process.  If all-processes-p is true then we check all
processes for this function and if there are multiple matches then we get a
menu, otherwise we just return the first matching process.  If something goes
wrong then we return T for the barf-p value, else nil.
"
  (declare (values bindings barf-p))
  (if all-processes-p
      (multiple-value-bind (bindings-and-processes barf-p)
	  (Bindings-For-Function-In-All-Halted-Processes function nil)
	(cond ((= (length bindings-and-processes) 1)
	       (first bindings-and-processes)
	      )
	      (barf-p (values nil t))
	      (t (let ((result
			 (w:menu-choose
			  (loop for (bindings process) in bindings-and-processes
				collect (list (format nil "~S" process)
					      :Value bindings
					)
			  )
			  :Label "Evaluate in which process?"
			 )
		       )
		      )
		      (if result (values result nil) (values nil t))
		 )
	      )
	)
      )
      (eh:Bindings-For-Function-In-Halted-Process
	function nil
      )
  )
)

(defun set-lex-reg (sg frame number value)
  (let ((eh:*erring-sg* sg)
	(eh:*erring-frame* frame)
       )
       (declare (special eh:*erring-sg* eh:*erring-frame*))  
       (multiple-value-bind (ignore location)
	   (sg-frame-local-value eh:*erring-sg* eh:*erring-frame* number)
	 (if location (rplacd location value))
	 value
       )
  )
)

(defun eval-form-maybe-with-lexical-environment (form)
  (condition-case ()
      (multiple-value-list (catch :Abort-Eval (si:*eval form)))
    (eh:no-lexical-superior
     (beep)
     (format t "Cannot find lexical environment for local function.")
     '(:No-Lex-Env)
    )
  )
)

(defun zwei:Evaluate-Expression-Binding-Args-Locals-And-Specials (form)
"Evaluates Form in such a way that specials are bound for all locals, args and
specials used in this function in an erroring stack frame in some distant
process.  If si:*force-defvar-init* is true (i.e. if we are in c-m-sh-e mode)
then we look in all processes for a matching frame and menu if there are
multiple matching frames, otherwise we just pick the first matching frame.
If condition is supplied then this is a condition that was signaled presumably
during some external evaluation of Form.  If something goes wrong then we
resignal this condition.
"
  (multiple-value-bind (bindings barf-p)
      (let ((function
	      (send (zwei:line-node (zwei:bp-line (zwei:point))) :Name)
	    )
	   )
	   (if (sys:fdefinition-safe function t nil)
	       (get-bindings-for-function-in-one-or-more-processes
		 function si:*force-defvar-init*
	       )
	       (values nil t)
	   )
      )
    (if barf-p
	;;; Then reevaluate the form so that we get the condition evaluated
	;;; in the right context.
	(si:*eval form)
	(let ((symbol-bindings nil)
	      (location-bindings nil)
	     )
	     (loop for (thing binding) in bindings
		   for entry in bindings
		   if (symbolp thing) do (push entry symbol-bindings)
		   else do (push entry location-bindings)
	     )
	     ;;; These bindings will be undone when we exit this frame.
	     (progv (mapcar #'first  symbol-bindings)
		    (mapcar #'second symbol-bindings)
	       (loop for (loc value) in location-bindings
		     do (bind loc value)
		     finally (return
			       (values-list
				 (Eval-Form-Maybe-With-Lexical-Environment
				   form
				 )
			       )
			     )
	       )
	     )
	)
	;;; Old version before we could do local functions.
;	(progv (mapcar #'first bindings)
;	       (mapcar #'second bindings)
;	  (si:*eval form)
;	)
    )
  )
)

(defvar zwei:*try-to-eval-in-debugger-context-p* :On-Error
"Causes Zmacs/Debugger eval hack to work:
  :on-error - if we get an unbound variable when evaling and expr
  t - always
  nil - never
"
)

;;; TI code.
zwei:
(DEFUN zwei:COMPILE-INTERVAL-PROCESS-FN (FORM)
  ;; 8/25/86 DNG - Merged functions COMPILE-INTERVAL-PROCESS-BASIC-FORM,
  ;;		COMPILE-BUFFER-FORM and EVAL-PRINT.  Use new function
  ;;		COMPILE-TOP-LEVEL-FORM.   Print results of compile as well as
  ;;		eval.  Don't use COMPILE-DRIVER when evaluating.  Eliminated
  ;;		function COMPILE-INTERVAL-PREPROCESS-FN.
  (DECLARE (SPECIAL COMPILE-P PRINT-RESULTS-STREAM COMPILE-PROCESSING-MODE))
  (LET ((VALUES-TO-PRINT
	  (MULTIPLE-VALUE-LIST
	    (IF COMPILE-P
		(LET ((COMPILE-VALUE NIL))
		  (COMPILER:COMPILE-DRIVER
		    FORM
		    #'(LAMBDA (FORM TYPE)
			(SETQ COMPILE-VALUE 
			      (IF (EQ COMPILE-P T) ;use the normal Lisp compiler
				  (IF (EQ TYPE 'SPECIAL)
				      (SI:*EVAL FORM)
				    (COMPILER:COMPILE-TOP-LEVEL-FORM
				      FORM 'COMPILER:COMPILE-TO-CORE #'SI:*EVAL
				      (IF (EQ TYPE 'MACRO)
					  'COMPILER:MACRO-COMPILE
					COMPILE-PROCESSING-MODE)))
				(FUNCALL COMPILE-P FORM))) )
		    NIL NIL T)
		  COMPILE-VALUE)
		;;; Condition handler put in by JPR to bind up args and locals
		;;; if we are in the debugger for this function somewhere.
		(if sys:*force-defvar-init*
		    (si:*eval form)
		    (case *try-to-eval-in-debugger-context-p*
		      (:on-error
		        (condition-case (condition)
			    (si:*eval form)
			  ((sys:unbound-variable sys:cell-contents-error)
			   (Evaluate-Expression-Binding-Args-Locals-And-Specials
			     form
			   )
			  )
			)
		      )
		      (nil (si:*eval form))
		      (otherwise
		       (Evaluate-Expression-Binding-Args-Locals-And-Specials
			 form
		       )
		      )
		    )
		)
	    )
	  )
	)
       )
    (WHEN PRINT-RESULTS-STREAM
      (LET-IF (EQ PRINT-RESULTS-STREAM *QUERY-IO*)
	      ((*PRINT-LENGTH* 5) (*PRINT-LEVEL* 2))
	(DOLIST (VAL VALUES-TO-PRINT)
	  (FORMAT PRINT-RESULTS-STREAM "~&~S" VAL))))))


zwei:
(defcom com-evaluate-region-in-debugger-context
	"Evaluate the current region or defun in the context of the debugger.
Result is typed out in the echo area.
If there is a region, it is evaluated.
Otherwise, the current or next defun is evaluated." ()
  (let ((zwei:*try-to-eval-in-debugger-context-p* t))
    (zwei:com-evaluate-region))
   dis-none)

(zwei:set-comtab zwei:*standard-comtab*
		 '(#\s-sh-e zwei:com-evaluate-region-in-debugger-context)
      (zwei:make-command-alist '(zwei:com-evaluate-region-in-debugger-context))
)