;;; -*- Mode:Common-Lisp; Package:Sys; Base:10; Fonts:(TvFont) -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;; This system requires the Explorer-36xx compatibility pacakge to work on Symbolics 36xx
;;; machines.  Non-Stanford sites will probably have to change the following:
#+Symbolics
(Make-System 'Tools :Noconfirm :Silent :Nowarn :No-Reload-System-Declaration)

#+Symbolics
(load-tools '(Explorer-36xx))

;;; This file contains a definition of a tool, which allows the running of
;;; command files "In the batch".  It does it by creating a process,
;;; which spends most of its time asleep, waking up every twenty minutes or so
;;; and looking to see if the time is between the times specified for legal
;;; batch activity.  If it is not then the process goes back to sleep.  If the
;;; time is ok then it wakes up and puts out a query to the user asking if the
;;; batch stream can start up.  If this question times out then the batch stream
;;; starts up, otherwise the process goes back to sleep.

;-------------------------------------------------------------------------------

;;; The following are the variables, which denothe the default times for the
;;; batch processor to start and stop its operation.

#+TI
(Intern '*batch-batch-process-start-time* 'TICL)

(defvar *batch-batch-process-start-time* "Today, 3:00"
"The time at which it is acceptable for the batch processor to start
 processing.
"
)

#+Symbolics
(let ((error-output 'si:null-stream))
     (Globalize "*BATCH-BATCH-PROCESS-START-TIME*")
)


#+TI
(Intern '*batch-batch-process-stop-time* 'TICL)

(defvar *batch-batch-process-stop-time*  "Today, 6:00"
"The time at which it is necessary for the batch processor to stop processing."
)

#+Symbolics
(let ((error-output 'si:null-stream))
     (Globalize "*BATCH-BATCH-PROCESS-STOP-TIME*")
)

;-------------------------------------------------------------------------------

;;; The following are the default names for the batch processor's command file
;;; name and log file name.

#+TI
(Intern '*Batch-input-command-file-name* 'TICL)

(defvar *Batch-input-command-file-name* "BATCH"
"The default name for the batch processor's command file."
)

#+Symbolics
(let ((error-output 'si:null-stream))
     (Globalize "*BATCH-INPUT-COMMAND-FILE-NAME*")
)


#+TI
(Intern '*Batch-log-file-name* 'TICL)

(defvar *Batch-log-file-name* "BATCH"
"The default name for the batch processor's log file."
)

#+Symbolics
(let ((error-output 'si:null-stream))
     (Globalize "*BATCH-LOG-FILE-NAME*")
)

;-------------------------------------------------------------------------------

(defconstant batch-process-name "Batch Process"
"The name of the process used to run the batch processor"
)

;-------------------------------------------------------------------------------

(defconstant batch-process-normal-sleep-interval 1000
"The number of seconds for which the batch processor normally goes to sleep
 before trying the time again to see whether it can run.
"
)

(defconstant one-minute 3600 "The number of clock ticks in a minute.")

#+TI
(Intern '*All-Batch-Processor-Commands* 'TICL)

(defparameter *All-Batch-Processor-Commands* nil
"The AList of all of the command types that the batch processor understands."
)

#+Symbolics
(let ((error-output 'si:null-stream))
     (Globalize "*ALL-BATCH-PROCESSOR-COMMANDS*")
)

(defun add-command-file-command-processor (tag function)
"This procedure is passed a tag name, which introduces a new command type in the
command file processor and a function, which is to be called by the comand file
processor in the event of the command being found.
"
    (Setq *All-Batch-Processor-Commands*
	  (Cons (List tag function)
		(remove (assoc tag *All-Batch-Processor-Commands*)
			*All-Batch-Processor-Commands*
		)
	  )
    )
)


#+TI
(Intern 'add-command-file-command-processor 'TICL)

#+Symbolics
(let ((error-output 'si:null-stream))
     (Globalize "ADD-COMMAND-FILE-COMMAND-PROCESSOR")
)

;-------------------------------------------------------------------------------

;;; The following are the default set of commands understood by the command file
;;; processor.

(add-command-file-command-processor :Input-File 'process-with-new-input-file)

(add-command-file-command-processor :Command-File
	'process-with-new-command-file
)


;-------------------------------------------------------------------------------


(defflavor fs:stream-of-yeses () (si:input-stream)
    (:Documentation :Special-Purpose
"This is a flavor of input stream, which always generates a stream of Y
 characters.
"
    )
)

(defmethod (fs:stream-of-yeses :Tyi) (Ignore)
"Whenever a character is read from this stream it is always a Y character."
    #\Y
)

(defmethod (fs:stream-of-yeses :UnTyi) (old-char)
"On this sort of stream when you untyi a Y character is always returned."
    (Ignore old-char)
    #\Y
)

(defflavor fs:interactive-stream-of-yeses () (fs:stream-of-yeses)
    (:Documentation :Special-Purpose
"This is an interactive version of the flavor fs:stream-of-yeses."
    )
)

(defmethod (fs:interactive-stream-of-yeses :Listen) ()
"When you listen to this sort of stream it always returns a Y character."
    #\Y
)

(defmethod (fs:interactive-stream-of-yeses :Tyi-no-hang) (eof)
"Wen you do a Tyi-no-hang on this stream it always returns a Y character."
    (Ignore eof)
    #\Y
)

(defmethod (fs:interactive-stream-of-yeses :Rubout-Handler)
    (options function &Rest args)
"This is null rubout handler for this flavor, since it always returns with a Y
 character.
"
    (Ignore options function args)
    #\Y
)

(defmethod (fs:interactive-stream-of-yeses :Beep) (type)
"Beeping on this sort of stream always causes nothing to happen and a Y
 character to be returned.
"
    (Ignore type)
    #\Y
)


;-------------------------------------------------------------------------------


;;; The following are the definitions of the functions, which know how to
;;; process the commands defined as the default command file processor command
;;; processing functions.


(defun process-the-command-file (command-file input-stream)
"This is the procedure, which processes command files in the batch stream.  It
 is passed an open stream, which is the current command file and an input
 stream, which is a stream, from which replies for input will come.  Forms are
 read from the command file and processed.  If the form read in is :Command-File
 then this means that a command file indirection is being made.  If the form is
 :Input-File then a new input file is being selected for the current context.
 The form that follws either of these is a string denoting the pathname of the
 file in question.
"
    (let ((form (read command-file nil :end-of-file )))
	 (if (equal form :end-of-file)
	     (close command-file)
	     (let ((thing-to-do (assoc form *All-Batch-Processor-Commands*)))
		  (if (equal nil thing-to-do)
		      (progn (catch-error (eval form) t)
		 	     (process-the-command-file command-file
						       input-stream
			     )
		      )
		      (funcall (symbol-function (Second thing-to-do))
			       command-file input-stream
		      )
		  )
	     )
	 )
    )
)


(defun process-with-new-input-file (command-file input-stream)
"This funciton causes the command file's user input to be redirected elsewhere.
It is passed the current command file.  The next form in the command file must
represent the pathname of the new input file.  If the file is opened
successfully then the processing of the command file continues in this context,
otherwise a stream is opened, which is a stream of Y characters.
"
    (Ignore input-stream)
    (let ((input-file-name (read command-file nil :end-of-file )))
	 (with-open-file (input-file input-file-name :Direction :Input
				     :If-Does-Not-Exist nil
			 )
	     (if (equal nil input-file)
		 (with-open-stream
		   (yes-stream (make-instance 'fs:interactive-stream-of-yeses))
		   (format t
	   "!!!!!!! Input file not found.  Using stream of Ys. !!!!!!!~%"
	           )
		   (let ((*standard-input* yes-stream))
			(process-the-command-file command-file yes-stream)
		   )
		 )
		 (let ((*standard-input* input-file))
		      (process-the-command-file command-file input-file)
		 )
	     )
	 )
    )
)


(defun apply-with-open-file-and-bindings (file-name warn-p function &rest args)
"Opens the file named by File-Name and binds specials to the file attribute
 bindings.  If warn-p it true then a message is printed if the file is not
 found.  Within this context Function is called with the new stream and Args.
"
  (let ((path (fs:default-pathname file-name)))
       (with-open-file (stream path :Direction :Input
			       :If-Does-Not-Exist nil
		       )
	     (if (equal nil stream)
		 (if warn-p
		     (format t "!!!!!! Input file ~S not found. !!!!!!~%"
			     file-name
		     )
		     nil
		 )
		 (progn (fs:read-attribute-list path stream)
		        (multiple-value-bind (vars values)
			    (fs:file-attribute-bindings path)
			  (progv vars values (apply function stream args))
			)
		 )
	     )
        )
   )
)



(defun process-with-new-command-file (command-file input-stream)
"This is the function, which causes a new command file to be selected as the
 current command file.  It is passed the current command file and the input
 stream for user replies.  The next form in the command file should represent
 the pathname of the new command file.  If the new file is opened successfully
 then it is used.  Otherwise the new one is ignored and the processing of the
 old one continues.  After the indirected commmand file is finished the
 processing of the old one continues, using the input file, which was current
 when the function was entered, rather than any that might have been selected by
 the indirect command files.
"
  (apply-with-open-file-and-bindings (read command-file nil :end-of-file) t
    #'process-the-command-file input-stream
  )
  (process-the-command-file command-file input-stream)
)


;-------------------------------------------------------------------------------


(defun run-the-batch-process ()
"This is the procedure, which is called once the system is fully committed to
 running the batch process.  First of all it checks to see whether there is a
 user logged in or not.  If the user id is the empty string (noone logged in) or
 the user id has SERVER in it (the logged in user is one of the servers) then it
 logs out and logs in as LispM.  After this it tries to open a command file.
 This is found by looking for a file called *Batch-input-command-file-name*.Lisp, 
 which defaults to the user's homedir and must be of type LISP.  
 If it finds one then it opens it and
 goes ahead.  If it doesn't find one then it gives up.  If it goes ahead then it
 opens a log file called *Batch-log-file-name*.Log, also defaulting the user-homedir,
 binding standard and error output to it.  It then processes the command file.
"
  (if (or (equal "" User-id) (search "SERVER" User-id))
      (progn (logout) (login 'LispM))
  )
  (apply-with-open-file-and-bindings
    
    ;; used to just default to user-homedir or "lispm;"
    (Send (merge-pathnames *Batch-input-command-file-name* (fs:user-homedir))
	  :New-Type :Lisp
    )
    nil
  #'(lambda (command-stream)
      (with-open-file
	(Log-File (Send (merge-pathnames *Batch-log-file-name* (fs:user-homedir))
			:New-Type "LOG"
		  )
		  :Direction :Output :If-Exists :New-Version
		  :If-Does-Not-Exist :Create
	)
        (let ((*standard-output* Log-File)
	      (*error-output* Log-File)
	     )
	     (with-open-stream
		(yes-stream (make-instance 'fs:interactive-stream-of-yeses))
		(let ((*standard-input* yes-stream))
		     (process-the-command-file command-stream yes-stream)
		)
	     )
	)
      )
    )
  )
)


(defvar *notify-before-batch-runs-p* t
"When true the batch processor puts out a notification before running."
)


(defun Batch-Process ()
"This is the top level of the batch processor after the system has decided that
 the time is ripe for batch processing It prompts the user to see whether it is
 ok to start up fully and if the user request times out then it starts properly.
"
  (if *notify-before-batch-runs-p*
      (let ((go-ahead t))
	   (with-timeout (one-minute)
	     (#+TI tv:careful-notify #+Symbolics tv:notify tv:initial-lisp-listener #+TI t
	       "Can I run the batch process now?  Hit any key to stop me.  Timeout after 60 seconds."
	     )
	     (if (and tv:selected-window (send tv:selected-window :any-tyi))
		 (setq go-ahead nil))
	   )
	   (if go-ahead (run-the-batch-process) nil)
      )
      (run-the-batch-process)
  )
  nil
)



(defvar *Batch-Processor-Enabled* t
"Whether the batch processor is enabled or not."
)


(defun Batch-process-initial-function ()
"This is the initial function for the batch processor.  It loops looking at the
 time.  If the time is within the bounds of *batch-batch-process-start-time* and
 *batch-batch-process-stop-time* then it starts up the batch process activity,
 otherwise it goes to sleep for another while.  WHen the batch process activity
 is finished it goes to sleep for an interval, which makes sure that the batch
 stream does not start up again before *batch-batch-process-stop-time*.
"
  (loop while t do
    (if (> (time:get-universal-time)
           (time:parse-universal-time *batch-batch-process-start-time* 0 nil t)
	)
        (if (and (< (time:get-universal-time)
		    (time:parse-universal-time *batch-batch-process-stop-time*
					       0 nil t
		    )
		 )
		 *Batch-Processor-Enabled*
	    )
	    (progn (Batch-Process)
		   (Sleep (- (time:parse-universal-time
			         *batch-batch-process-stop-time*
			     )
			     (time:get-universal-time)
			  )
		   )
	    )
	    nil
	)
	nil
    )
    (Sleep batch-process-normal-sleep-interval)
  )
)


(defun find-process-named (name)
"Given a name, which is a string, this function returns the first process in the
 list of active processes, which has the same name or nil if it does not find
 one.
"
    (remove nil (mapcar #'(lambda (a-process)
	      			(equal (Send a-process :Name) name)
			  )
			  tv:all-processes
		)
    )
)



(defun initialise-batch-evaluator ()
"This procedure initialises the batch process.  If looks for a process called
 the batch process.  If it finds one then the batch process is already running
 so it does nothing.  Otherwise it starts up the batch process.
"
   (let ((old-process (find-process-named batch-process-name)))
	(if (equal nil old-process)
	    (process-run-function
	       (List :Name batch-process-name)
	       #'Batch-Process-Initial-Function
	    )
	)
   )
)


;-------------------------------------------------------------------------------


;;; The following code creates initialisation forms, which cause the batch
;;; process to be started up again in the event of a warm or a cold boot.

(Eval-When (Load Eval)
    (Add-Initialization batch-process-name
			'(initialise-batch-evaluator)
			'(:Warm :Normal)
    )
    (Add-Initialization batch-process-name
			'(initialise-batch-evaluator)
			'(:Login :Normal)
    )
)


;-------------------------------------------------------------------------------


