;;; -*- Mode:Common-Lisp; Package:SYSTEM; Base:10 -*-

;;;  This file does whatever is necessary to make a world load for the
;;;  TI Explorer at Stanford KSL.  The user has to do the parts at the
;;;  end by hand, since the file can't be loaded if they are
;;;  uncommented.  

;;;  When bringing up a new world from TI, it is necessary to go through
;;;  the patches, testing each one, and test each of the tools.

(defun compute-ksl-world-partition-comment ()
  (let ((products
	  (get-relevant-systems-list *default-product-description-verbosity*)))
    (let ((release (find-if #'(lambda (x)
				(and (instancep x)
				     (search "xplorer system release"
					     (send x :name)
					     :test #'string-equal)))
			    products))
	  (system (find-if #'(lambda (x)
			       (and (consp x)
				    (string-equal "SYSTEM" (first x))))
			   products))
	  (ksl-ver (find-if #'(lambda (x)
				(search "KSL-PATCHES" (symbol-name x)
					:test #'string-equal
					)
				)
			    *features*))
	  (ksl (find-if #'(lambda (x)
			    (and (consp x)
				 (string-equal "KSL-PATCHES" (first x))))
			products))
	  )
      (multiple-value-bind (sec min hr day mon)
	  (time:decode-universal-time (time:get-universal-time))
	(declare (ignore sec min hr))
	(format nil "~D.~D(~D), KSL~D ~D.~D, ~0:2D-~A"
		(send release :major-version)
		(send release :minor-version)
		;;Minor version of SYSTEM system
		(loop for patch in (fourth system) maximize (first patch))
		;;Meta-version of KSL-PATCHES system
		(subseq (symbol-name ksl-ver) (length "KSL-PATCHES-"))
		;;Major version of KSL-PATCHES system
		(second ksl) 
		;;Minor version of KSL-PATCHES system
		(loop for patch in (fourth ksl) maximize (first patch))
		day
		(time:month-string mon :short)
		)
	)
      )
    )
  )

(defun make-ksl-world ()
  (let ((patches? t)
;;RDA: Now built into worlds
;	(clos? t)
	(ip? t)
	(rpc? t)
	(nfs? t)
	(tools? t)
	(band-clean? t)
	(compile-flavor-methods? t)
	(gc-and-save? nil)
	partition
	unit
	)
    (declare (special tv:*screen-saver-time-delay*))
    (setq tv:*screen-saver-time-delay* 10)
    ;;RDA: No screen saver while building world
    (let-globally ((tv:*screen-saver-time-delay* nil))
      (with-timeout
	(7200 (format *query-io* "Timeout -- defaulting the rest to 'Y'"))
	(setf patches? (y-or-n-p "Load System Patches? "))
;      (setf clos? (y-or-n-p "Load CLOS? "))
	(setf ip? (y-or-n-p "Load IP? "))
	(setf rpc? (y-or-n-p "Load RPC? "))
	(setf nfs? (if (si:mx-p) nil (y-or-n-p "Load NFS? ")))
	(setf tools? (y-or-n-p "Load KSL-Standard-Tools? "))
	(setf compile-flavor-methods?
	      (y-or-n-p "Optimize Flavors?"))
	(setf band-clean? (y-or-n-p "Run the band cleaner?"))
	(setf gc-and-save? (y-or-n-p "GC and disk save when done?"))
	(setf partition (if gc-and-save? (prompt-and-read :string-trim "~&Partition Name: ") nil))
	(setf unit (if gc-and-save? (prompt-and-read :integer "~&Unit: ") nil))
	)
      
      (when patches? (load-patches :noconfirm))
;    (when clos? (make-system :clos :noconfirm :nowarn :silent :safe))
      (when ip?   (make-system :ip   :noconfirm :nowarn :silent :safe))
      (when rpc?  (make-system :rpc  :noconfirm :nowarn :silent :safe))
      (when nfs?
	(make-system :nfs  :noconfirm :nowarn :silent :safe)
;;;RDA: The server now comes with the client
;	;;RDA: Work around a bug in the distributed world
;	(remprop :nfs-server :source-file-name)
;	(make-system :nfs-server :noconfirm :nowarn :silent :safe)
	)
      
      ;;  Load patches for this world.  Note that users should still do this
      ;;  in their login-init files to get the most up to date version.  A
      ;;  lot of tool loading and such that used to be done in this file is
      ;;  now done in KSL-CUSTOMIZATIONS, a module of KSL-Patches.
      (make-system :ksl-patches :safe :noconfirm :nowarn :silent)
      
      ;; Load tools commonly used in the KSL
      (when tools?
	(make-system :ksl-standard-tools :safe :nowarn :noconfirm :silent))
      ;;; This should speed things up.
      (when (and compile-flavor-methods? (fboundp 'compile-flavors-and-classes))
	(compile-flavors-and-classes))
      
      ;;; This should make things smaller
      (sys:load-if "SYS:BAND-TOOLS;BAND-CLEANER")
      (when band-clean?
	(sys:band-cleaner :verbose T :Doc-Strings nil))

      ;; Quiet the system.
      (notify-all-servers "Making a world load")       ;Say bye.
      
      (when gc-and-save?
	(let ((comment (compute-ksl-world-partition-comment)))
	  (sys:full-gc :duplicate-pnames t)
	  (sys:with-sys-host-accessible 
	    (disk-save partition unit :partition-comment comment :no-query t)
	    )
	  )
	)
      )
    )
  )

(send-if-handles tv:selected-window :force-kbd-input "(sys:make-ksl-world")
