;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:TV; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER ADOBE-COURIER14B HL12B CPTFONTBI) -*-

;1;; File "3WRITE-PBM*".*
;1;; Write an arbitrary bitmap section to a PBM or PGM file.*
;1;; PBM is "Portable Bit Map," an image conversion utility by Jef Poskanzer.*
;1;; This code is portable common lisp, with lispm-specific efficiency hacks.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;    5 Feb 90*	1Jamie Zawinski*	1Created.*
;1;;*  125 Apr 90*	1Jamie Zawinski *	1Added code to write PGM files as well as PBM.*
;1;;*


(defun 4write-pbm *(bitmap file &optional (compact-p t)
		    (x 0) (y 0) (w (array-dimension bitmap 1)) (h (array-dimension bitmap 0)))
  "2Write the specified portion of the bitmap to a PBM (Portable Bit Map) file.
 If COMPACT-P is true, then the file written out will use 8 bits per byte 
 instead of storing the bitmap as ASCII.  This is faster and more compact,
 but won't transit through the mail well.*"
  (with-open-file (stream file :direction :output :characters (not compact-p) :byte-size 8)
    (if compact-p
	(write-pbm-2 bitmap stream x y w h)
	(write-pbm-1 bitmap stream x y w h))
    (truename stream)))


(defun 4write-pbm-1 *(bitmap stream &optional
		      (x 0) (y 0) .
		      #+LISPM ((w (array-dimension bitmap 1)) (h (array-dimension bitmap 0))) ;1 Lispms are col-major.*
		      #-LISPM ((w (array-dimension bitmap 0)) (h (array-dimension bitmap 1)))
		      )
  "2Write a normal (P1) Portable Bit Map file.*"
  (declare (optimize speed (safety 0))
	   (fixnum x y w h))
  (format stream "3P1~%~D ~D~%*" w h)
  (let* ((tick 0))
    (declare (fixnum tick))
    (dotimes (j h)
      (declare (fixnum j))
      (dotimes (i w)
	(declare (fixnum i))
	(write-char (if (= 0 #+LISPM (aref bitmap (+ y j) (+ x i))	;1 Lispms are col-major.*
			     #-LISPM (aref bitmap (+ x i) (+ y j)))
			#\0 #\1)
		    stream)
	(when (> (incf tick) 68)
	  (write-char #\Newline stream)
	  (setq tick 0)))
      (when (< w 69)
	(terpri stream)
	(setq tick 0))))
  nil)


;1;;; The portable version of WRITE-PBM-2.*

#-LISPM
(defun 4write-pbm-2 *(bitmap stream &optional
		      (x 0) (y 0) .
		      #+LISPM ((w (array-dimension bitmap 1)) (h (array-dimension bitmap 0))) ;1 Lispms are col-major.*
		      #-LISPM ((w (array-dimension bitmap 0)) (h (array-dimension bitmap 1)))
		      )
  "2Write a compact (P4) Portable Bit Map file.*"
  (declare (optimize speed (safety 0))
	   (fixnum x y w h))
  ;1;*
  ;1; Write the string to the file (remember, the stream we have is :characters nil).*
  (let* ((string (format nil "3P4~%~D ~D~%*" w h)))
    (dotimes (i (length string))
      (nsubstitute (int-char 10) #\Newline string :test #'char=) ;1 just in case...*
      (write-byte (char-int (schar string i)) stream)))
  (dotimes (j h)
    (declare (fixnum j))
    (dotimes (i (ceiling w 8))
      (declare (fixnum i))
      (let* ((byte 0))
	(dotimes (k 8)
	  (setq byte (dpb #-LISPM (aref bitmap (+ x (* 8 i) k)  (+ y j))
			  #+LISPM (aref bitmap (+ y j)  (+ x (* 8 i) k))
			  #-LISPM (byte 1 k)
			  #+LISPM (byte 1 (- 7 k))
			  byte)))
	(write-byte byte stream))))
  nil)


;1;;; The Mondo-Efficient, Lispm-specific version of WRITE-PBM-2.*

#+LISPM
(defvar 4*bit-flippage* *(let* ((a (make-array 256 :element-type '(unsigned-byte 8))))
			   (dotimes (i 256)
			     (let* ((flip-byte 0))
			       (dotimes (j 8)
				 (setq flip-byte (dpb (ldb (byte 1 (- 7 j)) i) (byte 1 j) flip-byte)))
			       (setf (aref a i) flip-byte)))
			   a)
  "2As always, a table for quickly reversing the order of bits in a byte.*")

#+LISPM
(proclaim '(type (array (unsigned-byte 8) (256)) *bit-flippage*))


#+LISPM
(defun 4write-pbm-2 *(bitmap stream &optional (x 0) (y 0) (w (array-dimension bitmap 1)) (h (array-dimension bitmap 0)))
  "2Write a compact (P4) Portable Bit Map file.*"
  (declare (optimize speed (safety 0))
	   (fixnum x y w h))
  ;1; Width must be a multiple of 32 (i.e., scanlines word-aligned), or we can't use any neat tricks.*
  (setq w (+ 32 (* 32 (floor (1- w) 32))))
  ;1;*
  ;1; Write the string to the file, being careful to write it as ascii (remember, the stream we have is :characters nil).*
  (let* ((string (format nil "3P4~%~D ~D~%*" w h)))
    (nsubstitute (int-char 10) #\Newline string :test #'char=)
    (dotimes (i (length string))
      (write-byte (char-int (schar string i)) stream)))
  ;1;*
  ;1; Make a copy of the bitmap portion we are going to write - we must flip the bits around in it before writing it*
  ;1; out (since the PBM format stores its bits in the opposite order of Explorer memory), and we do this in place.*
  (let* ((bm2 (make-array (list h w) :element-type 'bit)))
    (bitblt tv:alu-seta w h bitmap x y bm2 0 0)
    (setq bitmap bm2))
  ;1;*
  ;1; Make a one-dimensional array displaced to the array we are going to write.  March through it and flip the*
  ;1; bits-within-a-byte around.  Then hand it to :string-out, which slams it to the file (using the array itself as an*
  ;1; output buffer).*
  (let* ((length (ceiling (* w h) 8))
	 (a (make-array length :element-type '(unsigned-byte 8) :displaced-to bitmap)))
    ;1; *(dotimes (i (length a))
    ;1;*  (setf (aref a i) (aref (the (array (unsigned-byte 8) (256)) *bit-flippage*)
    ;1;*                         (aref a i))))
    ;1; Thanks to Paul Fuqua for telling me about this miscop which does the above, but seven times faster...*
    (sys:%buffer-char-map a 0 (length a) a 0 (length a) *bit-flippage* #xFF 0)
    (send stream :string-out a))
  nil)


;1;; Portable Gray Maps*

(defun 4write-pgm *(bitmap file &optional (compact-p t)
		    (x 0) (y 0) (w (array-dimension bitmap 1)) (h (array-dimension bitmap 0)))
  "2Write the specified portion of the bitmap to a PGM (Portable Gray Map) file.
 If COMPACT-P is true, then the file written out will use one-byte-per-element
 instead of storing the bitmap as ASCII digits.  This is faster and more compact,
 but won't transit through the mail well.  
  If the bitmap has more than 8 bits per pixel, then it will be written in the dumb
 ASCII format no matter what.*"
  (with-open-file (stream file :direction :output :characters (not compact-p) :byte-size 8)
    (if compact-p
	(write-pgm-2 bitmap stream x y w h)
	(write-pgm-1 bitmap stream x y w h))
    (truename stream)))

(defun 4bits-per-elt *(array)
  "2Returns the number of bits-per-element in the array.*"
  #+EXPLORER
  (or (sys:array-bits-per-element (sys:%p-ldb sys:%%array-type-field (sys:follow-structure-forwarding array)))
      (error "3not a numeric array*"))
  #-EXPLORER		;1 5Yuck.**
  (cond ((typep array '(array (unsigned-byte 1))) 1)
	((typep array '(array (unsigned-byte 2))) 2)
	((typep array '(array (unsigned-byte 4))) 4)
	((typep array '(array (unsigned-byte 8))) 8)
	(t (dotimes (i 64)
	     (when (typep array `(array (unsigned-byte ,i)))
	       (return i)))
	   (error "3can't determine bits-per-elt.*"))))


(defun 4write-pgm-1 *(bitmap stream &optional
		      (x 0) (y 0) .
		      #+LISPM ((w (array-dimension bitmap 1)) (h (array-dimension bitmap 0))) ;1 Lispms are col-major.*
		      #-LISPM ((w (array-dimension bitmap 0)) (h (array-dimension bitmap 1)))
		      )
  "2Write a normal (P2) Portable Gray Map file.*"
  (declare (optimize speed (safety 0))
	   (fixnum x y w h))
  (format stream "3P2~%~D ~D~%~D~%*" w h (1- (expt 2 (bits-per-elt bitmap))))
  (let* ((tick 0))
    (declare (fixnum tick))
    (dotimes (j h)
      (declare (fixnum j))
      (dotimes (i w)
	(declare (fixnum i))
	(princ #+LISPM (aref bitmap (+ y j) (+ x i))	;1 Lispms are col-major.*
	       #-LISPM (aref bitmap (+ x i) (+ y j))
	       stream)
	(write-char #\Space stream)
	(when (> (incf tick 4) 68)
	  (write-char #\Newline stream)
	  (setq tick 0)))
      (when (< w 69)
	(terpri stream)
	(setq tick 0))))
  nil)

(defun 4write-pgm-2 *(bitmap stream &optional
		      (x 0) (y 0) .
		      #+LISPM ((w (array-dimension bitmap 1)) (h (array-dimension bitmap 0))) ;1 Lispms are col-major.*
		      #-LISPM ((w (array-dimension bitmap 0)) (h (array-dimension bitmap 1)))
		      )
  "2Write a compact (P5) Portable Gray Map file.*"
  (declare (optimize speed (safety 0))
	   (fixnum x y w h))
  (let ((bits (bits-per-elt bitmap)))
    ;1; If it's more than 8 bits, we must write it in the verbose format.*
    (when (> bits 8) (return-from 4WRITE-PGM-2* (write-pgm-1 bitmap stream x y w h)))
    (format stream "3P5~C~D ~D ~D~C*" (int-char 10) w h (1- (expt 2 bits)) (int-char 10)))
  (dotimes (j h)
    (declare (fixnum j))
    (dotimes (i w)
      (declare (fixnum i))
      (write-byte #+LISPM (aref bitmap (+ y j) (+ x i))	;1 Lispms are col-major.*
		  #-LISPM (aref bitmap (+ x i) (+ y j))
		  stream)))
  nil)
