;;; -*- Mode:Common-Lisp; Package:(TETRIS); Base:10 -*-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	Explorer Double Tetris
;
;	Author:		John Nguyen
;
;	Address:	545 Technology Square, Room 626
;			Cambridge, MA  02139
;			(617) 253-6028
;
;	E-mail:		johnn@hx.lcs.mit.edu
;
;	(based on an early version by Joe Morrison)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconstant *block-size* 24)

(defconstant *score-row* 80)
(defconstant *lines-row* 120)
(defconstant *unnice-lines-row* 160)
(defconstant *level-row* 200)

(defconstant *next-piece-x-offset* 30)
(defconstant *next-piece-y-offset* 260)

(defconstant *normal-bottom-delay* 30)

#+MIT (require "tetris-font" "sys:mit.hacks;tetris-font")
#-MIT (or (load "sys:fonts;tetris-font.xld#>" :if-does-not-exist nil)
	  (load "sys:mit.hacks;tetris-font.xld#>" :if-does-not-exist t))


(defstruct options
  board-width board-height initial-level initial-lines play-mode delay nice-lines
  key-left key-right key-rotate key-drop key-next key-swap use-other-keys)

;;;==================================================
;;; Function to define the Tetris shapes:
;;;==================================================

(defvar *tetris-shapes*
	'((((-1  0) ( 0  0) ( 1  0) ( 0  1))    ; XXX
	   (( 0 -1) ( 0  0) ( 0  1) ( 1  0))    ;  X
	   ((-1  0) ( 0  0) ( 1  0) ( 0 -1))
	   ((-1  0) ( 0 -1) ( 0  0) ( 0  1)))
	  (((-1  0) ( 0  0) ( 1  0) ( 2  0))    ; XXXX
	   (( 0 -1) ( 0  0) ( 0  1) ( 0  2))
	   ((-1  0) ( 0  0) ( 1  0) ( 2  0))
	   (( 0 -1) ( 0  0) ( 0  1) ( 0  2)))
	  (((-1  0) ( 0  0) ( 0 -1) ( 1 -1))    ;  XX
	   ((-1 -1) (-1  0) ( 0  0) ( 0  1))    ; XX
	   ((-1  0) ( 0  0) ( 0 -1) ( 1 -1))
	   ((-1 -1) (-1  0) ( 0  0) ( 0  1)))
	  (((-1 -1) ( 0 -1) ( 0  0) ( 1  0))    ; XX
	   (( 0 -1) ( 0  0) (-1  0) (-1  1))    ;  XX
	   ((-1 -1) ( 0 -1) ( 0  0) ( 1  0))
	   (( 0 -1) ( 0  0) (-1  0) (-1  1)))
	  (((-1  0) (-1  1) ( 0  0) ( 1  0))    ; XXX
	   (( 0 -1) ( 0  0) ( 0  1) ( 1  1))    ; X
	   ((-1  0) ( 0  0) ( 1  0) ( 1 -1))
	   ((-1 -1) ( 0 -1) ( 0  0) ( 0  1)))
	  (((-1  0) ( 0  0) ( 1  0) ( 1  1))    ; XXX
	   (( 0 -1) ( 0  0) ( 0  1) ( 1 -1))    ;   X
	   ((-1 -1) (-1  0) ( 0  0) ( 1  0))
	   (( 0 -1) ( 0  0) ( 0  1) (-1  1)))
	  (((-1  0) (-1 -1) ( 0  0) ( 0 -1))    ; XX
	   ((-1  0) (-1 -1) ( 0  0) ( 0 -1))    ; XX
	   ((-1  0) (-1 -1) ( 0  0) ( 0 -1))
	   ((-1  0) (-1 -1) ( 0  0) ( 0 -1)))))

(defvar *shape-bottoms* (make-array (list (length *tetris-shapes*) 4 4) :initial-element nil))
(defun init-shape-bottoms ()
  (dotimes (n (length *tetris-shapes*))
    (dotimes (rot 4)
      (let ((coord-list (nth rot (nth n *tetris-shapes*))))
	(dolist (coord coord-list)
	  (let* ((x (car coord))
		 (y (second coord))
		 (low-y (aref *shape-bottoms* n rot (1+ x))))
	    (setf (aref *shape-bottoms* n rot (1+ x))
		  (if low-y (max low-y y) y))))))))
(init-shape-bottoms)

(defvar *shape-y-offsets* '(0 0 1 1 0 0 1))
(defvar *shape-x-offsets* '(1 1 1 1 1 1 2))

(defvar *shape-asymmetric-rotations* '(4 2 2 2 4 4 1))

(defconstant *level-lines* `(10 20 30 40 50 60 70 90 105 ,most-positive-fixnum))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defflavor shape
	   (char
	    type
	    rotation
	    coord-list
	    x-pos
	    y-pos
	    x-offset
	    y-offset
	    array
	    delay
	    board
	    board-width
	    board-height
	    flash
	    flash-count
	    flash-length
	    draw?
	    )
	   ()
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)

(defvar *temp* 1)
(defun my-random (ignore)
  (if (>= (incf *temp*) (length *tetris-shapes*))
      (setq *temp* 0))
  *temp*)

(defmethod (shape :reset) ()
  (setq draw? t)
  (setq type (random (length *tetris-shapes*)))
  (setq char type)
  (setq board-width (send board :board-width)
	board-height (send board :board-height))
  (setq delay nil)
  (setq rotation 0)
  (setq x-pos (nth type *shape-x-offsets*)
	y-pos (nth type *shape-y-offsets*))
  (setq array (send (send board :window) :screen-array))
  (setq x-offset *next-piece-x-offset*
	y-offset *next-piece-y-offset*)
  (setq flash nil)
  (send self :init-coord-list)
  )

(defmethod (shape :start) ()
  (setq x-pos (1- (lsh board-width -1))
	y-pos 0)
  (setq x-offset (send board :x-offset)
	y-offset (send board :y-offset)))

(defmethod (shape :init-coord-list) ()
  (setq coord-list (nth rotation (nth type *tetris-shapes*))))

(defmethod (shape :draw-block) (x y alu screen-array)
  (when (and (<= 0 x (- board-width 1))
	     (<= 0 y (- board-height 1)))
    (w:%draw-character fonts:tetris char nil
		       (+ x-offset (* *block-size* x))
		       (+ y-offset (* *block-size* y))
		       alu
		       screen-array)))

(defmethod (shape :draw) (alu &optional screen-array)
  (when draw?
    (mapc #'(lambda (coord)
	      (send self :draw-block
		    (+ x-pos (first coord)) (+ y-pos (second coord))
		    alu (or screen-array array)))
	  coord-list)))

(defmethod (shape :flash-on) ()
  (setq flash :on
	flash-count 3
	flash-length 7))

(defmethod (shape :flash-off) ()
  (setq flash nil))

(defmethod (shape :maybe-flash) ()
  (when flash
    (when (zerop (decf flash-count))
      (case flash
	(:on (send self :draw w:erase)
	     (setq flash :off))
	(:off (send self :draw w:normal)
	      (setq flash :on)))
      (setq flash-count 3)
      (when (zerop (decf flash-length))
	(send board :maybe-swap-pieces)))))

(defmethod (shape :check) (blocks)
  ; this let is needed for lexical binding
  (let ((bwidth (send self :board-width))
	(bheight (send self :board-height)))
    (apply #'and (mapcar #'(lambda (coord)
			     (check-coord (+ (first coord) x-pos)
					  (+ (second coord) y-pos)
					  bwidth
					  bheight
					  blocks))
			 coord-list))))

(defun check-coord (x y board-width board-height blocks)
  (and (<= 0 x (1- board-width))
       (< y board-height)
       (or (< y 0) (= (aref blocks y x) 0))))		; Not on another piece!

(defmethod (shape :check-down) (blocks)
  (incf y-pos 1)
  (let ((result (send self :check blocks)))
    (decf y-pos 1)
    result))

(defmethod (shape :down) (blocks)
  (if (send self :check-down blocks)
      (progn
	(send self :draw w:erase)
	(incf y-pos 1)
	(send self :draw w:normal)
	(setq delay nil)
	t)
      (unless delay
	(setq delay t))))

(defmethod (shape :down-all) (blocks)
  (loop for rows from -1
	while (and (not delay) (send self :down blocks))
	finally
	(return rows)))

(defmethod (shape :check-horiz) (blocks delta)
  (incf x-pos delta)
  (let ((result (send self :check blocks)))
    (decf x-pos delta)
    result))

(defmethod (shape :horiz) (blocks delta)
  (if (send self :check-horiz blocks delta)
      (progn
	(send self :draw w:erase)
	(incf x-pos delta)
	(send self :draw w:normal)
	(when (send self :check-down blocks)
	  (setq delay nil))
	t)
      nil))

(defmethod (shape :left) (blocks)
  (send self :horiz blocks -1))

(defmethod (shape :right) (blocks)
  (send self :horiz blocks 1))

(defmethod (shape :check-rotate) (blocks)
  (setq rotation (mod (1+ rotation) 4))
  (send self :init-coord-list)
  (let ((result (send self :check blocks)))
    (setq rotation (mod (1- rotation) 4))
    (send self :init-coord-list)
    result))

(defmethod (shape :rotate) (blocks)
  (if (send self :check-rotate blocks)
      (progn
	(send self :draw w:erase)
	(setq rotation (mod (1+ rotation) 4))
	(send self :init-coord-list)
	(send self :draw w:normal)
	(when (send self :check-down blocks)
	  (setq delay nil))
	t)
      nil))

(defmethod (shape :add-to-blocks) (blocks)
  (send self :draw w:normal)
  (block add-to-blocks
    (mapc #'(lambda (coord)
	      (if (< (+ (second coord) y-pos) 0)
		  (return-from add-to-blocks nil)
		  (setf (aref blocks (+ (second coord) y-pos) (+ (car coord) x-pos))
			1)))
	  coord-list)
    t))

(defmethod (shape :add-to-tblocks) (blocks)
  (mapc #'(lambda (coord)
	    (unless (< (+ (second coord) y-pos) 0)
	      (setf (aref blocks (+ (car coord) x-pos) (- board-height (+ (second coord) y-pos) 1))
		    1)))
	coord-list))

(defmethod (shape :remove-from-tblocks) (blocks)
  (mapc #'(lambda (coord)
	    (unless (< (+ (second coord) y-pos) 0)
	      (setf (aref blocks (+ (car coord) x-pos) (- board-height (+ (second coord) y-pos) 1))
		    0)))
	coord-list))

(defmethod (shape :line-range) ()
  (let ((coords (mapcar #'(lambda (c) (+ (second c) y-pos)) coord-list)))
    (values (apply #'min coords) (apply #'max coords))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;==================================================
;;; The physical Tetris display:
;;;==================================================

(defflavor board
	   (window				; The physical lispM window to use
	    stats-window
	    board-height
	    board-width
	    x-offset
	    y-offset
	    last-timeout			; Last time the block dropped
	    drop-delay				; Time between drops (60th of a sec)
	    bottom-delay                        ; Time at bottom
	    blocks				; The blocks in place already
	    block-rows
	    line-status
	    shape			   ; The current shape that's falling
	    next-shape			   ; The next shape
	    mode
	    show-next
	    count
	    score
	    level
	    num-lines
	    unnice-lines
	    key-left
	    key-right
	    key-rotate
	    key-drop
	    key-next
	    key-swap
	    swappable?
	    options
	    opponent
	    auto-player
;	    (process nil)
	    )
	   ()
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)


;;;==================================================
;;; Initializing the display:
;;;==================================================

(defmethod (board :after :init) (&rest ignore)
  (setq shape (make-instance 'shape :board self))
  (setq next-shape (make-instance 'shape :board self))
  (setq auto-player (make-instance 'auto-player :board self))
  (setq blocks (make-array (list 32 32) :type 'art-1b :initial-element 0))
  (setq block-rows (make-array 32 :type 'art-32b :displaced-to blocks))
  (setq line-status (make-array 32))
  )


(defmethod (board :reset) (x-off y-off op)
  (setq board-width (options-board-width options)
	board-height (options-board-height options))
  (setq opponent op)
  (let ((width (* *block-size* board-width))
	(height (* *block-size* board-height)))
    (setq x-offset x-off
	  y-offset y-off)
    (let ((x1 (- x-offset 5))
	  (y1 (- y-offset 5))
	  (width1 (+ width 8))
	  (height1 (+ height 8)))
      (send window :draw-rectangle x1 y1 width1 height1 3)
      (send window :draw-rectangle x1 y1 width1 height1 1 w:black w:erase)))

  ;; Initialize the time (used for input):
  (setf last-timeout (time))
  
  (send self :set-level (- (char-int (options-initial-level options)) 48))

  ;; Initialize the "blocks world":

  (w:%draw-rectangle 32 32 0 0 w:erase blocks)
  
  (setq bottom-delay
	(if (eq (options-play-mode options) :advanced)
	    (lsh *normal-bottom-delay* -1)
	    *normal-bottom-delay*))

  (when (eq (options-play-mode options) :auto)
    (send auto-player :initialize)
;    (unless process
;      (setq process (process-run-function '(:name "Auto Tetris"); :priority -10 :quantum 10)
;					  auto-player :wait)))
    )

  (let ((key-options (if (and (options-use-other-keys options) opponent)
			 (send opponent :options)
			 options)))
    (setq key-left (options-key-left key-options)
	  key-right (options-key-right key-options)
	  key-rotate (options-key-rotate key-options)
	  key-drop (options-key-drop key-options)
	  key-next (options-key-next key-options)
	  key-swap (options-key-swap key-options)))

  (send self :setup-blocks (options-initial-lines options))

  (setq score 0)
  (setq num-lines 0)
  (setq unnice-lines 0)
  (send stats-window :update-score score)
  (send stats-window :update-lines num-lines unnice-lines)

  (setq swappable? nil)
  (setq show-next nil)
  (send next-shape :reset)
  (send self :start-new-object)
  (setq mode :normal)

)

(defmethod (board :after :set-level) (l)
  (send stats-window :update-level l)
  (setq drop-delay (floor (* (+ 3 (* 5 (- 9 l)))
			     (if (eq (options-play-mode options) :advanced) .5 1)))))

(defmethod (board :up-level) ()
  (when (< level 9)
    (send self :set-level (1+ level))))

(defmethod (board :setup-blocks) (lines)
  (loop for y from (- board-height lines) below board-height
	do
	(loop with cnt = 0
	      until (= cnt 4)
	      for x = (random board-width)
	      when (zerop (aref blocks y x))
	      do
	      (send self :add-block x y)
	      (incf cnt))))

(defmethod (board :check-timeout) ()
  (let ((time (time)))
    (when (> (time-difference time last-timeout)
	     (if (send shape :delay) bottom-delay drop-delay))
      (setf last-timeout time)
      t)))


;;;==================================================
;;; Start a new object falling:
;;;==================================================

(defmethod (board :start-new-object) ()
  (setq swappable? t)
  (when show-next
    (send stats-window :draw-next-piece next-shape w:erase))
  (let ((old-shape shape))
    (setq shape next-shape)
    (setq next-shape old-shape))
  (send shape :start)
  (send next-shape :reset)
  (when show-next
    (send stats-window :draw-next-piece next-shape w:normal))
  (if (send shape :check blocks)
      (progn
	(send shape :draw w:normal)
	(setq mode :normal))
      (progn
	(loop until (send shape :check blocks) do
	      (send shape :set-y-pos (1- (send shape :y-pos))))
	(send shape :draw w:normal)
	(setq mode :end)))
  (when (eq (options-play-mode options) :auto)
;    (send process :preset auto-player :new-piece)))
    (send auto-player :new-piece)))

(defmethod (board :toggle-show-next) ()
  (when show-next
    (send stats-window :draw-next-piece next-shape w:erase))
  (setq show-next (not show-next))
  (when show-next
    (send stats-window :draw-next-piece next-shape w:normal))
  )

(defmethod (board :swap-pieces) (this-shape that-shape)
  (let ((this-type (send this-shape :type))
	(that-type (send that-shape :type))
	(this-rot (send this-shape :rotation))
	(that-rot (send that-shape :rotation)))
    (send this-shape :set-type that-type)
    (send that-shape :set-type this-type)
    (send this-shape :set-char that-type)
    (send that-shape :set-char this-type)
    (send this-shape :set-rotation that-rot)
    (send that-shape :set-rotation this-rot)
    (send this-shape :init-coord-list)
    (send that-shape :init-coord-list)))

(defmethod (board :start-swap-pieces) ()
  (when (and opponent swappable?
	     (eq mode :normal)
	     (eq (send opponent :mode) :normal))
    (setq swappable? nil)
    (let* ((that-shape shape)
	   (this-shape (send opponent :shape)))
      (send this-shape :flash-on)
      (send that-shape :flash-on))))

(defmethod (board :maybe-swap-pieces) ()
  (let* ((that-shape shape)
	 (this-shape (send opponent :shape)))
    (when (and (eq mode :normal)
	       (eq (send opponent :mode) :normal))
      (send this-shape :draw w:erase)
      (send that-shape :draw w:erase)
      (send self :swap-pieces this-shape that-shape)
      (if (and (send this-shape :check blocks)
	       (send that-shape :check (send opponent :blocks)))
	  (progn
	    (setq swappable? nil)
	    (send opponent :set-swappable? nil))
	  (progn
	    (send self :swap-pieces this-shape that-shape)))
      (send this-shape :draw w:normal)
      (send that-shape :draw w:normal))
    (send this-shape :flash-off)
    (send that-shape :flash-off)))

(defmethod (board :drop-object) ()
  (unless (send shape :down blocks)
    (send self :add-object-to-blocks 0)))

(defmethod (board :drop-object-all) ()
  (let ((bonus (send shape :down-all blocks)))
    (send self :add-object-to-blocks bonus))
  )

(defmethod (board :move-object-left) ()
  (send shape :left blocks))

(defmethod (board :move-object-right) ()
  (send shape :right blocks))

(defmethod (board :rotate-object) ()
  (send shape :rotate blocks))

(defmethod (board :add-object-to-blocks) (bonus)
  (when opponent
    (send shape :flash-off)
    (send (send opponent :shape) :flash-off))
  (if (send shape :add-to-blocks blocks)
      (multiple-value-bind (line-top line-bottom)
	  (send shape :line-range)
	(incf score (floor (* (+ 4 (* level 2) bonus (- board-height (send shape :y-pos)))
			      (if show-next .75 1))))
	(if (send self :find-full-lines line-top line-bottom)
	    (progn
	      (setq mode :flash)
	      (setq count 12))
	    (send self :start-new-object))
	(send stats-window :update-score score))
      (setq mode :end)))

(defmethod (board :flash) ()
  (if (> count 0)
      (progn
	(if (evenp count)
	    (loop for row from 0 below board-height
		  when (aref line-status row)
		  do
		  (w:%draw-rectangle (* *block-size* board-width) *block-size*
				     x-offset (+ y-offset (* *block-size* row)) w:opposite
				     (send window :screen-array))))
	(decf count))
      (progn
	(loop for row from 0 below board-height
	      when (aref line-status row)
	      do
	      (w:%draw-rectangle (* *block-size* board-width) *block-size*
				 x-offset (+ y-offset (* *block-size* row)) w:erase
				 (send window :screen-array)))
	(setq mode :clear-wait)
	(setq count 6))))

(defmethod (board :clear-wait) ()
  (if (> count 0)
      (decf count)
      (progn
	(send self :clear-lines)
	(send self :start-new-object))))

(defmethod (board :clear-lines) ()
  (loop with line-count = 0
	for row from 0 below board-height
	when (aref line-status row)
	do
	(incf num-lines)
	(incf line-count)
	(when (and (< level 9)
		   (>= num-lines (nth level *level-lines*))
		   (or (not opponent) (>= (send opponent :num-lines) num-lines)))
	  (send self :up-level)
	  (when (and opponent (< (send opponent :level) level))
	    (send opponent :up-level)))
	(send self :move-rows 0 1 (- row))
	(send self :blank-row 0)
	(when (> line-count (options-nice-lines options))
	  (incf unnice-lines)
	  (when opponent
	    (send opponent :add-row))))
  (send stats-window :update-lines num-lines unnice-lines))

(defmethod (board :move-rows) (from to height)
  (bitblt w:normal board-width height blocks 0 from blocks 0 to)
  (bitblt w:normal (* *block-size* board-width) (* *block-size* height)
	  (send window :screen-array) x-offset (+ y-offset (* *block-size* from))
	  (send window :screen-array) x-offset (+ y-offset (* *block-size* to))))

(defmethod (board :blank-row) (row)
  (w:%draw-line 0 row (1- board-width) row w:erase t blocks)
  (w:%draw-rectangle (* *block-size* board-width) *block-size*
		     x-offset (+ y-offset (* *block-size* row)) w:erase
		     (send window :screen-array)))

(defmethod (board :add-block) (x y)
  (w:%draw-character fonts:tetris (random (length *tetris-shapes*)) nil
		     (+ x-offset (* x *block-size*))
		     (+ y-offset (* y *block-size*))
		     w:normal (send window :screen-array))
  (setf (aref blocks y x) 1))

(defmethod (board :add-row) ()
  (send shape :draw w:erase)
  (dotimes (i board-width)
    (when (plusp (aref blocks 0 i))
      (setq mode :end)))
  (send self :move-rows 1 0 (- board-height 1))
  (send self :blank-row (- board-height 1))
  (dotimes (row (1- board-height))
    (setf (aref line-status row)
	  (aref line-status (1+ row))))
  (setf (aref line-status (1- board-height)) nil)
  (let ((hole (random board-width)))
    (dotimes (i board-width)
      (when (/= i hole)
	(send self :add-block i (- board-height 1)))))
  (unless (send shape :check blocks)
    (if (< (send shape :y-pos) 1)
	(setq mode :end)
	(send shape :set-y-pos (1- (send shape :y-pos)))))
  (send shape :draw w:normal))

;;;==================================================
;;; Find all the lines that are full (i.e. that should
;;; disappear):
;;;==================================================

(defmethod (board :find-full-lines) (top bottom)
  (array-initialize line-status nil)
  (loop with flag = nil and full-value = (1- (ash 1 board-width ))
	for row from bottom downto top do
	(when (= (aref block-rows row) full-value)
	  (setq flag t)
	  (setf (aref line-status row) t))
	finally
	(return flag)))
		      
(defmethod (board :peek-char) ()
  (send window :listen))

(defmethod (board :tick) (char)
  (block tick
    (case mode
      (:normal
       (when (eq (options-play-mode options) :auto)
	 (let ((time (time)))
	   (loop with go = t
		 while (and (zerop (time-difference (time) time))
			    go)
		 do
		 (setq go (and (not (send self :peek-char))
			       (send auto-player :tick))))))
       (select char
	 (key-left (send self :move-object-left))
	 (key-right (send self :move-object-right))
	 (key-rotate (send self :rotate-object))
	 (key-drop (send self :drop-object-all))
	 (key-next (send self :toggle-show-next))
	 (key-swap (send self :start-swap-pieces))
	 )
       (send shape :maybe-flash)
       (when (send self :check-timeout) (send self :drop-object)))
      (:flash
       (send self :flash))
      (:clear-wait
       (send self :clear-wait))
      (:end
       (let ((width (* *block-size* board-width))
	     (height (* *block-size* board-height)))
	 (loop for i from 0 below (lsh height -1) do
	       (w:%draw-line x-offset (+ y-offset i)
			     (+ x-offset width -1) (+ y-offset i)
			     w:opposite t (send window :screen-array))
	       (w:%draw-line x-offset (+ y-offset (- height i 1))
			     (+ x-offset width -1) (+ y-offset (- height i 1))
			     w:opposite t (send window :screen-array))
	       (tv:delay 3000)
	       )
	 (process-sleep 10)
	 (w:%draw-rectangle width height x-offset y-offset w:opposite (send window :screen-array))
	 )
       (return-from tick nil)))
    t))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor options-window (options other-options)
	   (w:choose-variable-values-pane)
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables
  (:default-init-plist
    :scroll-bar-side :right
    :scroll-bar-mode :maximum
    :margin-choices nil
    :variables nil
    :value-tab 80
    :name-font fonts:cptfont
    :value-font fonts:hl12
    :string-font fonts:hl12b
    :unselected-choice-font fonts:tr10
    :selected-choice-font fonts:tr10b
    :label nil
    :stack-group current-stack-group))

(defun options-check-width (window var old-value new-value)
  (ignore var old-value)
  (let ((window-width (send (send (send window :superior) :graphics-pane) :inside-width)))
    (when (> (* *block-size* new-value) window-width)
      (format nil "Width must be less than ~d" (floor window-width *block-size*)))))

(defun options-check-height (window var old-value new-value)
  (ignore var old-value)
  (let ((window-height (send (send (send window :superior) :graphics-pane) :inside-height)))
    (when (> (* *block-size* new-value) window-height)
      (format nil "Height must be less than ~d" (floor window-height *block-size*)))))

(defun options-check-level (window var old-value new-value)
  (ignore window var old-value)
  (unless (<= #\0 new-value #\9)
    (format nil "Level must be between 0 and 9")))

(defmethod (options-window :initialize) (keys)
  (setq options
	(make-options :board-height 20
		      :board-width 10
		      :initial-level #\5
		      :initial-lines 0
		      :play-mode :normal
		      :nice-lines 1
		      :delay 1
		      :key-left (first keys)
		      :key-right (second keys)
		      :key-rotate (third keys)
		      :key-drop (fourth keys)
		      :key-next (fifth keys)
		      :key-swap (sixth keys)
		      :use-other-keys nil
	))
  (send self :set-variables
	`(""
	  (,(locf (options-board-width  options)) " Width"
	   :documentation "Width of playing board" :constraint options-check-width
	   :typep (integer 5 32))
	  (,(locf (options-board-height options)) " Height"
	   :documentation "Height of playing board" :constraint options-check-height
	   :typep (integer 5 32))
	  ""
	  (,(locf (options-initial-level options)) " Level"
	   :documentation "Initial level" :constraint options-check-level :character)
	  (,(locf (options-initial-lines options)) " Lines"
	   :documentation "Initial lines with random blocks" :choose (0 4 7 10 13))
	  ""
	  (,(locf (options-play-mode options)) " Mode"
	   :documentation "Active or deactivate board" :choose (:normal :advanced :auto :inactive))
	  ""
	  (,(locf (options-delay options)) " Delay"
	   :documentation "Delay for piece movement in auto mode (in 1/60 sec)"
	   :typep (integer 0 20))
	  ""
	  (,(locf (options-nice-lines options)) " Nice"
	   :documentation "Number of cleared lines that do not show up on opponent"
	   :choose (0 1 2 3 4))
	  ""
	  (,(locf (options-key-left   options)) " Left"
	   :documentation "Move left character" :character)
	  (,(locf (options-key-right  options)) " Right"
	   :documentation "Move right character" :character)
	  (,(locf (options-key-rotate options)) " Rotate"
	   :documentation "Rotate character" :character)
	  (,(locf (options-key-drop   options)) " Drop"
	   :documentation "Drop piece character" :character)
	  (,(locf (options-key-next   options)) " Next"
	   :documentation "Toggle show next piece character" :character)
	  (,(locf (options-key-swap   options)) " Swap"
	   :documentation "Swap pieces character" :character)
	  (,(locf (options-use-other-keys options)) " Use other"
	   :documentation "Use key definitions of other board" :boolean)
	  )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor tetris-window
	   (board1
	    board2
	    options1
	    options2
	    )
	   (w:window-with-typeout-mixin
	    w:list-mouse-buttons-mixin
	    w:notification-mixin
	    w:window)
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables
  (:default-init-plist
    :typeout-window '(w:typeout-window)
    :label nil
    :blinker-p nil
    :save-bits t
  ))

(defmethod (tetris-window :initialize) (o1 o2)
  (setq options1 o1 options2 o2)
  (send w:typeout-window :make-complete)
  (setq board1 (make-instance 'board)
	board2 (make-instance 'board))
  (send board1 :set-window self)
  (send board2 :set-window self)
  (send board1 :set-stats-window (send (send self :superior) :stats1-pane))
  (send board2 :set-stats-window (send (send self :superior) :stats2-pane))
  (send board1 :set-options options1)
  (send board2 :set-options options2)
  )

(defmethod (tetris-window :center-string) (string y font)
  (send self :string-out-centered-explicit string
	(w:sheet-inside-left) y (w:sheet-inside-right)
	nil font))

(defmethod (tetris-window :run) ()
  (let ((*terminal-io* self)
	(*debug-io* (send self :typeout-window)))
    (error-restart ((error sys:abort) "Tetris top level.")
      (when (send (send self :typeout-window) :bottom-reached)
	(send (send self :typeout-window) :deactivate))
      (send self :select)
      (send (send (send self :superior) :stats1-pane) :erase-next-piece)
      (send (send (send self :superior) :stats2-pane) :erase-next-piece)
      (send self :clear-screen)
      (send self :center-string "Double" (- (lsh w:height -1) 30) fonts:43vxms)
      (send self :center-string "Tetris" (+ (lsh w:height -1) 30) fonts:43vxms)
      (do-forever
	(send self :center-string "Press RETURN to begin"
	      (- w:height 30) fonts:tr18)
	(loop for char = (send self :any-tyi)
	      until (and (atom char) (equal (int-char char) #\return))
	      do
	      (when (and (consp char) (eq (car char) :variable-choice))
		(w:choose-variable-values-process-message
		  (second char)
		  char)))
	(send self :game)
	))))

(defmethod (tetris-window :game) ()
  (send self :clear-screen)
  (cond ((and (eq (options-play-mode options1) :inactive)
	      (eq (options-play-mode options2) :inactive))
	 nil)
	((eq (options-play-mode options2) :inactive)
	 (send self :setup-board board1 options1)
	 (send board2 :set-mode nil)
	 (send self :play))
	((eq (options-play-mode options1) :inactive)
	 (send self :setup-board board2 options2)
	 (send board1 :set-mode nil)
	 (send self :play))
	(t
	 (when (send self :setup-both-boards)
	   (send self :play)))))

(defmethod (tetris-window :setup-board) (board options)
  (send board :reset
	(lsh (- w:width (* *block-size* (options-board-width options))) -1)
	(lsh (- w:height (* *block-size* (options-board-height options))) -1)
	nil))

(defmethod (tetris-window :setup-both-boards) ()
  (let* ((max-height (max (options-board-height options1)
			  (options-board-height options2)))
	 (top (lsh (- w:height (* *block-size* max-height)) -1))
	 (left (floor (- w:width
			 (* *block-size* (+ (options-board-width options1)
					    (options-board-width options2))))
		      3)))
    (if (or (minusp top) (minusp left))
	(progn
	  (send self :center-string "Boards are too large!"
		(floor w:height 3) fonts:tr18)
	  nil)
	(progn
	  (send board1 :reset
		left
		(+ top (* *block-size* (- max-height (options-board-height options1))))
		board2)
	  (send board2 :reset
		(+ left (* *block-size* (options-board-width options1)) left)
		(+ top (* *block-size* (- max-height (options-board-height options2))))
		board1)
	  t))
    ))
  
(defmethod (tetris-window :play) ()
  (loop with go = t
	while go
	do
	(process-allow-schedule)
	(let ((char (when (and (or (send board2 :mode) (eq (send board1 :mode) :normal))
			       (or (send board1 :mode) (eq (send board2 :mode) :normal)))
		      (read-char-no-hang))))
	  (setq go (and (or (not (send board1 :mode)) (send board1 :tick char))
			(or (not (send board2 :mode)) (send board2 :tick char))))
	  (when (equal char #\p) (send self :tyi))))
  (send (send (send self :superior) :stats1-pane) :erase-next-piece)
  (send (send (send self :superior) :stats2-pane) :erase-next-piece)
  (send self :center-string "Game Over"	10 fonts:43vxms)
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor stats-window
	   ()
	   (w:window)
  (:default-init-plist
    :blinker-p nil
    :save-bits t
    :label nil
    :font-map (list fonts:tr18b fonts:tr18)))

(defmethod (stats-window :initialize) (number)
  (send self :string-out-centered-explicit (format nil "~d" number)
	(w:sheet-inside-left) 10 (w:sheet-inside-right)
	nil fonts:43vxms)
  (send self :set-current-font fonts:tr18)
  (send self :set-cursorpos 10 *score-row*)
  (format self "Score")
  (send self :set-cursorpos 10 *lines-row*)
  (format self "Lines")
  (send self :set-cursorpos 10 *unnice-lines-row*)
  (format self "% ~c" (if (= number 1) #\ #\))
  (send self :set-cursorpos 10 *level-row*)
  (format self "Level")
  (send self :set-current-font fonts:tr18b))

(defmethod (stats-window :update-score) (score)
  (send self :set-cursorpos 70 *score-row*)
  (send self :clear-eol)
  (format self "~7d" score)
  )

(defmethod (stats-window :update-lines) (lines unnice-lines)
  (send self :set-cursorpos 70 *lines-row*)
  (send self :clear-eol)
  (format self "~7d" lines)
  (send self :set-cursorpos 70 *unnice-lines-row*)
  (send self :clear-eol)
  (format self "    ~3d" (round (* unnice-lines 100) (max lines 1)))
  )

(defmethod (stats-window :update-level) (level)
  (send self :set-cursorpos 70 *level-row*)
  (send self :clear-eol)
  (format self "~7d" level)
  )

(defmethod (stats-window :draw-next-piece) (shape alu)
  (send shape :draw alu w:screen-array))

(defmethod (stats-window :erase-next-piece) ()
  (w:%draw-rectangle (* *block-size* 4) (* *block-size* 3)
		      *next-piece-x-offset*
		      *next-piece-y-offset*
		      w:erase w:screen-array))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun start-tetris (window)
  (send window :run))

(defflavor tscreen
	   (graphics-pane
	    options1-pane
	    options2-pane
	    stats1-pane
	    stats2-pane
	    )
	   (w:process-mixin
	    w:select-mixin
	    w:alias-for-inferiors-mixin
	    w:inferiors-not-in-select-menu-mixin
	    w:bordered-constraint-frame-with-shared-io-buffer)
  :gettable-instance-variables
  :settable-instance-variables
  (:default-init-plist
    :process '(start-tetris)
    :panes
    '((graphics tetris-window)
      (stats1 stats-window)
      (stats2 stats-window)
      (options1 options-window)
      (options2 options-window)
      )
    :constraints
    '((main (dummy)
	 ((dummy :horizontal (:even) (player1 graphics player2)
		((player1 :vertical (160) (stats1 options1)
		 ((stats1 0.45))
		 ((options1 :even))))
		((player2 :vertical (160) (stats2 options2)
		 ((stats2 0.45))
		 ((options2 :even))))
		((graphics :even))
		))))
    ))

(defmethod (tscreen :after :init) (ignore)
  (setq graphics-pane (send self :get-pane 'graphics)
	options1-pane (send self :get-pane 'options1)
	options2-pane (send self :get-pane 'options2)
	stats1-pane (send self :get-pane 'stats1)
	stats2-pane (send self :get-pane 'stats2)
	)
  (send self :set-configuration 'main)
  (send graphics-pane :select)
  (send graphics-pane :clear-screen)
  (send options1-pane :initialize '(#\j #\l #\k #\space #\n #\i))
  (send options2-pane :initialize '(#\4 #\6 #\5 #\0 #\1 #\8))
  (send options1-pane :set-other-options (send options2-pane :options))
  (send options2-pane :set-other-options (send options1-pane :options))
  (send graphics-pane :initialize
	(send options1-pane :options) (send options2-pane :options))
  (send stats1-pane :initialize 1)
  (send stats2-pane :initialize 2)
  )

(defmethod (tscreen :run) ()
  (send graphics-pane :run))

(defmethod (tscreen :or :verify-new-edges) (ignore ignore ignore ignore)
  nil)

(defflavor tetris-frame () (tscreen))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor auto-player
	   (real-blocks
	    blocks
	    tblocks
	    block-rows
	    block-columns
	    line-status
	    status-bits
	    real-levels
	    levels
	    shape
	    board
	    width
	    height
	    shape-heights
	    col
	    rot
	    good-col
	    good-rot
	    min-score
	    mode
	    moves
	    num-moves
	    delay
	    delay-time
	    )
	   ()
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)

(defmethod (auto-player :after :init) (&rest ignore)
  (setq real-blocks (make-array (list 32 32) :type 'art-1b)
	blocks (make-array (list 32 32) :type 'art-1b)
	tblocks (make-array (list 32 32) :type 'art-1b))
  (setq block-rows (make-array 32 :type 'art-32b :displaced-to blocks))
  (setq block-columns (make-array 32 :type 'art-32b :displaced-to tblocks))
  (setq real-levels (make-array 32 :type 'art-fix))
  (setq levels (make-array 32 :type 'art-fix))
  (setq line-status (make-array 32))
  (setq moves (make-array 50))
  (setq shape-heights (make-array 4))
  )

(defmethod (auto-player :initialize) ()
  (setq width (send board :board-width)
	height (send board :board-height))
  (setq shape (copy (send board :shape)))
  (send shape :set-draw? nil)
  (send shape :set-delay nil)
  (setq delay (options-delay (send board :options)))
  )

(defmethod (auto-player :new-piece) ()
  (bitblt w:normal 32 height (send board :blocks) 0 0 real-blocks 0 0)
  (w:%draw-rectangle 32 32 0 0 w:erase tblocks)
  (dotimes (column width)
    (dotimes (row height)
      (setf (aref tblocks column (- height row 1))
	    (aref real-blocks row column))))
  (array-initialize line-status nil)
  (setq status-bits 0)
  (send self :find-levels real-levels)     ; need this and hole-penalties for levels to be right
  (send self :hole-penalties real-levels)
  (send shape :set-type (send (send board :shape) :type))
  (send shape :set-board-width (send (send board :shape) :board-width))
  (send shape :set-board-height (send (send board :shape) :board-height))
  (setq min-score most-positive-fixnum
	good-rot 0
	good-col 0)
  (setq col 0
	rot 0)
  (setq mode :normal))

(defmethod (auto-player :tick) ()
  (case mode
    (:normal
     (send self :analyze-position)
     (when (>= (incf rot) (nth (send shape :type) *shape-asymmetric-rotations*))
       (setq rot 0)
       (when (>= (incf col) width)
	 (send self :move-shape)
	 (setq mode :move)))
     t)
    (:move
     (when (>= (time-difference (time) delay-time) delay)
       (setq delay-time (time))
       (unless (send board :peek-char)
	 (send (send board :window) :force-kbd-input (aref moves (decf num-moves)))
	 (if (zerop num-moves)
	     (setq mode :sleep))))
     nil)
    (:sleep nil)))

(defmethod (auto-player :analyze-position) (&aux (hole-count 0))
  (block analyze-position
    (send shape :set-x-pos col)
    (send shape :set-y-pos 3)
    (send shape :set-rotation rot)
    (send shape :init-coord-list)
    (send shape :set-delay nil)
    (when (send shape :check real-blocks)
      (bitblt w:normal 32 height real-blocks 0 0 blocks 0 0)
      (let ((min-height height)
	    (shape-type (send shape :type)))
	(dotimes (i 4)
	  (setf (aref shape-heights i)
		(let ((shape-bottom (aref *shape-bottoms* shape-type rot i)))
		  (when shape-bottom
		    (let ((value (- (aref real-levels (+ col i -1)) shape-bottom 1)))
		      (setq min-height (min min-height value))
		      value)))))
	(dotimes (i 4)
	  (when (aref shape-heights i)
	    (let ((holes (- (aref shape-heights i) min-height)))
	      (incf hole-count holes))))
	(send shape :set-y-pos min-height))
      (send shape :add-to-blocks blocks)
      (let ((full-lines (send self :find-full-lines)))
	(when (and (zerop full-lines) (>= hole-count 2))
	  (return-from analyze-position nil)))
      (send shape :add-to-tblocks tblocks)
      (let ((score (send self :analyze-blocks)))
	(send shape :remove-from-tblocks tblocks)
	(when (< score min-score)
	  (setq min-score score
		good-rot rot
		good-col col))))))

(defmethod (auto-player :move-shape) ()
  (let ((options (send board :options))
	(shape-x (send (send board :shape) :x-pos)))
    (setq num-moves 0)
    (send self :add-move (options-key-drop options))
    (if (< good-col shape-x)
	(dotimes (i (- shape-x good-col))
	  (send self :add-move (options-key-left options)))
	(dotimes (i (- good-col shape-x))
	  (send self :add-move (options-key-right options))))
    (dotimes (i good-rot)
      (send self :add-move (options-key-rotate options)))
    (setq delay-time (time))
    ))

(defmethod (auto-player :add-move) (char)
  (setf (aref moves num-moves) char)
  (incf num-moves))
	  
(defmethod (auto-player :analyze-blocks) ()
  (let (hole-penalty cliff-penalty)

    (send self :find-levels levels)

    (setq hole-penalty (send self :hole-penalties levels))

    (setq cliff-penalty (send self :cliff-penalties))

    (+ hole-penalty
       cliff-penalty
       (* (send shape :y-pos) -4)
       )))
  
(defmethod (auto-player :find-full-lines) ()
  (let ((count 0)
	(full-value (1- (ash 1 width))))
    (setq status-bits 0)
    (dotimes (row height)
      (if (= (aref block-rows row) full-value)
	  (progn
	    (incf count)
	    (incf status-bits (ash 1 (- height row 1)))
	    (setf (aref line-status row) t))
	  (setf (aref line-status row) nil)))
    count))

; find-levels only finds the number of "on" blocks after rows have been cleared

(defmethod (auto-player :find-levels) (arg-levels)
  (dotimes (column width)
    (let ((h (- height (integer-length (logxor (aref block-columns column) status-bits)))))
      (setf (aref arg-levels column) h)
      ))
  )

; hole-penalties actually increments each entry in levels for

(defmethod (auto-player :hole-penalties) (arg-levels)
  (let ((penalty 0))
    (dotimes (column width)
      (let* ((adjusted-column (logxor (aref block-columns column) status-bits))
	     (h (integer-length adjusted-column))
	     (hole-bits (logxor adjusted-column (1- (ash 1 h)))))
	(loop until (zerop hole-bits)
	      do
	      (let ((hole (integer-length hole-bits)))
		(if (aref line-status (- height hole))
		    (incf (aref arg-levels column))
		    (incf penalty (+ 40 (* 2 hole (min (- h hole) 3)))))
		(decf h)
		(setq hole-bits (- hole-bits (ash 1 (1- hole))))))))
    penalty))

(defsubst level-diff-penalty (l1 l2)
  (- (max 8 (lsh (abs (- l1 l2)) 3)) 5))

(defmethod (auto-player :cliff-penalties) ()
;  (format t "~%")
  (let ((penalty 0)
	(pen2 0)
	(pen3 0))
    (loop with last-level = 0
	  and next-level = (aref levels 1)
	  and level 
	  and flag = :down
	  for column from 0 below width
	  do
	  (setq level (aref levels column))
	  (when (> column 0)
	    (cond ((> level last-level)
		   (when (eq flag :up)
		     (incf penalty pen2)
		     (setq pen2 0)
		     (setq flag nil))
		   (incf penalty (lsh (level-diff-penalty level last-level)
				      (if (eq flag :down) -2 0))))
		  ((< level last-level)
		   (incf pen2 (level-diff-penalty level last-level))
		   (setq flag :up))))
	  (when (and (>= (- level last-level) 2)
		     (>= (- level next-level) 2))
	    (let* ((diff-last (- level last-level))
		   (diff-next (- level next-level))
		   (diff-min (min diff-last diff-next)))
	      (incf pen3 (* (+ (- (lsh diff-min 1) (if (= diff-min 2) 3 0))
			       (if (or (/= diff-last 2) (/= diff-next 2)) 2 1))
			    7))))
	  (setq last-level level)
	  (setq next-level (if (<= column (- width 3)) (aref levels (+ column 2)) 0)))
    (incf penalty (lsh pen2 -2))
    (+ pen3 penalty)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(add-initialization "Tetris System Key"
		    '(w:add-system-key #\space 'tetris:tetris-frame "Tetris" t)
		    '(:once))

(compile-flavor-methods shape board tetris-window options-window stats-window
			tscreen tetris-frame)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

