;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:USER -*-

;;; File "SUBDIR"
;;; Some code fragments for walking directory trees.
;;;
;;; ChangeLog:
;;;
;;;  long ago	Jamie Zawinski	Created.
;;;  10 Apr 89	Jamie Zawinski	Cleaned up.
;;;


(defun do-subdirectories (function directory &optional (directories-before-their-files-p t))
  "  Call FUNCTION for every file under the DIRECTORY.
  DIRECTORY is a wildcarded pathname.
  If FUNCTION returns NIL, and if the pathname that function was invoked with is
   itself a directory, do not recurse down it.
  FUNCTION's first argument will be a pathname; subsequent arguments are keyword-value pairs.
  Your function should take &ALLOW-OTHER-KEYS, as well as any of the following keywords:

    :ACCOUNT <string>
    :AUTHOR <string>
    :BLOCK-SIZE <number>
    :BYTE-SIZE <number>
    :CREATION-DATE <universal-date>
    :DELETED <boolean>
    :DONT-DELETE <boolean>
    :DONT-DUMP <boolean>
    :DONT-REAP <boolean>
    :DUMPED <boolean>
    :GENERATION-RETENTION-COUNT <number>
    :LENGTH-IN-BLOCKS <number>
    :LENGTH-IN-BYTES <number>
    :LINK-TO <string>
    :NOT-BACKED-UP <boolean>
    :OFFLINE <boolean>
    :PHYSICAL-VOLUME <string>
    :PROTECTION <string>
    :READER <string>
    :REFERENCE-DATE <universal-date>
    :TEMPORARY <boolean>

  You must take &KEY ... &ALLOW-OTHER-KEYS or &REST because this is not an exhaustive list;
  any keyword can be present on a file's plist.

  DIRECTORIES-BEFORE-THEIR-FILES-P says whether we call FUNCTION on a directory-file before or
  after we call it on all of the files in the directory.  Which is better depends on what you
  are doing - if you are deleting files, you want to do files before directories, for example.

  But if you want your function to be able to return NIL meaning that you do not want to
  descend this directory, then DIRECTORIES-BEFORE-THEIR-FILES-P must be T.
  "

  (setq directory (merge-pathnames directory (make-pathname :name :WILD :type :WILD :version :WILD)))
  
  (catch-error-restart (condition "Give up on descending ~A" directory)
    (let* ((file-lists (cdr (fs:directory-list directory)))
	   (supplied-name (pathname-name directory))
	   (supplied-type (pathname-type directory))
	   (supplied-vers (pathname-version directory)))
      (dolist (file-list file-lists)
	(let* ((pathname (car file-list))
	       (plist (cdr file-list))
	       (directory-p (getf plist :DIRECTORY)))
	  
	  (cond ((not directory-p)			; Just a normal file.
		 (apply function pathname plist))
		
		(t					; It's a directory.
		 (let* ((new-pathname (make-pathname :host (pathname-host pathname)
						     :device (pathname-device pathname)
						     :directory (append (pathname-directory pathname)
									(list (pathname-name pathname)))
						     :name supplied-name
						     :type supplied-type
						     :version supplied-vers)))
		   (cond (directories-before-their-files-p
			  (let* ((descend-p (apply function pathname plist)))
			    (when descend-p
			      (do-subdirectories function new-pathname t))))
			 (t
			  (do-subdirectories function new-pathname nil)
			  (apply function pathname plist)))))))))))


(defun print-dir-tree (&optional (top "LM:~;"))
  "A trivial example of the use of DO-SUBDIRECTORIES."
    (do-subdirectories #'(lambda (pathname &rest ignore)
			   (let* ((string (file-namestring pathname))
				  (depth (or (and (consp (pathname-directory pathname))
						  (length (pathname-directory pathname)))
					     0)))
			     (format t "~&~vT~A" (* 3 depth) string))
			   t)  ; Return T so that it descends.
		       top))



(defun print-dir-tree-2 ()
  "  A different (but still trivial) example of the use of DO-SUBDIRECTORIES.
  This one shows how to bind specials over the scope of a directory-walk."
  (labels ((walker (pathname &key directory &allow-other-keys)
	     (let* ((depth (if (boundp 'depth) (1+ depth) 0))
		    (string (file-namestring pathname)))
	       (declare (special depth))
	       (format t "~&~vT~A" (* 3 depth) string)
	       (when directory
		 (do-subdirectories #'walker (send pathname :pathname-as-directory))))
	     nil))    ; Always return NIL, so that it never automatically descends.
    
    (do-subdirectories #'walker "LM:~;")))



#+COMMENT
(defun rm-2.0 (pathname)
  "Walk the directory tree under PATHNAME, and delete and expunge all files which were a part of release 2.0 or 2.1."
  (do-subdirectories #'(lambda (file &key author &allow-other-keys)
			 (when (or (string= "REL2.0" author)
				   (string= "REL2.1" author)
				   (string= "REL2" author)
				   )
			   (print file)
			   (send file :delete-and-expunge))
			   t)
		       pathname
		       nil		; We want to do files before directories.
		       )
  nil)


(defun copy-machine (from to)
  "Copy all files on the machine FROM to the machine TO, excluding files by certain authors.
  If a directory already exists on the destination machine, do not copy the files in it."
  (let ((as ()))
    (do-subdirectories #'(lambda (file &key directory author &allow-other-keys)
			   (pushnew author as :test #'string=)
			   (cond (directory
				  #+COMMENT
				  (cond ((probe-file (make-pathname :host (pathname-host to) :defaults file))
					 (format t "~&directory ~A exists.~%" (make-pathname :host (pathname-host to)
											     :defaults file))
					 nil)
					(t
					 (let* ((name (pathname-name file)))
					   (cond ((or (string= name author)  ; DirName=AuthName means a homedir.
						      (find "GALLEY" name)
						      (find "PEGASYS" name))
						  (format t "~&skipping ~a." file)
						  nil)
						 (t t))))))
				 (t (copy-file file to :create-directories t :verbose t)
				    t)))
		       from)
    as))


(defun re-owner-directories (path old-owner new-owner)
  "Make all directory files under PATH which are owned by OLD-OWNER be owned by NEW-OWNER instead."
  (do-subdirectories #'(lambda (file &key directory author &allow-other-keys)
			 (when (and directory (string-equal author old-owner))
			   (format t "~A:~90t~A = ~A~%" file author new-owner)
			   (fs:change-file-properties file t :author new-owner)))
		     path))

(defun mark-backed-up-by-owner (path owner)
  "Mark all files under PATH which are owned by OWNER as being backed up."
  (do-subdirectories #'(lambda (file &key author not-backed-up &allow-other-keys)
			 (when (and not-backed-up (string-equal author owner))
			   (format t "~&~a" file)
			   (fs:change-file-properties file t :not-backed-up nil)))
		     path))

(defun mark-undeletable-by-owner (path owner)
  "Mark all files under PATH which are owned by OWNER as being undeletable."
  (do-subdirectories #'(lambda (file &key author dont-delete &allow-other-keys)
			 (when (and (not dont-delete) (string-equal author owner))
			   (format t "~&~a" file)
			   (fs:change-file-properties file t :dont-delete t)))
		     path))



(defun compare-hosts (directory host2 &optional show-dates-p)
  "Given a directory, and a host on a different machine that DIRECTORY, compare the corresponding files on the two machines.
  This will tell you if: 
     a file exists on one machine and not the other;
     a file on one machine has a greater version than the other;
     the files have different sizes; or
     the files have different write dates (if SHOW-DATES-P).
  "
  (do-subdirectories #'(lambda (path &key &allow-other-keys)
			 (let* ((other (make-pathname :defaults path :host host2))
				(pnew (make-pathname :defaults path :version :newest))
				(onew (make-pathname :defaults other :version :newest))
				(pv nil)
				(ov nil)
				(ok t))
			   (cond ((not (probe-file onew))
				  (setq ok nil)
				  (format t "~&~A does not exist." other))
				 ((/= (setq pv (pathname-version (probe-file pnew)))
				      (setq ov (pathname-version (probe-file onew))))
				  (format t "~&~A version changed from ~D to ~D." pnew pv ov))
				 ((and show-dates-p (/= (file-write-date path) (file-write-date other)))
				  (format t "~&~A has a different write-date." other))
				 ((/= (file-length path) (file-length other))
				  (format t "~&~A has a different size." other))
				 )
			   ok))
		     directory))


(defun all-authors (path)
  "Returns a list of all of the authors of all of the files under the directory tree at PATH."
  (let* ((authors '()))
    (do-subdirectories #'(lambda (ignore &key author &allow-other-keys)
			   (pushnew author authors :test #'equal)
			   t)
		       path)
    authors))


(defun find-author (path authors)
  "Prints out all files under PATH which are owned by an author in the list AUTHORS."
  (do-subdirectories #'(lambda (pathname &key author &allow-other-keys)
			 (when (member author authors :test #'string-equal)
			   (format t "~&~s ~A~%" author pathname))
			 t)
		     path))


(defun find-zero-length (path)
  "Prints out the names of all files under path which are zero-length."
  (do-subdirectories #'(lambda (pathname &key (length-in-bytes 0) &allow-other-keys)
			 (when (zerop length-in-bytes)
			   (format t "~&~A~%" pathname))
			 t)
		     path))


(defun recursive-size (top-directory &optional verbose)
  "  Walks the directory tree under DIRECTORY, and returns the total size in bytes of all of the files.
  If VERBOSE is T, then a subtotal will be printed for each directory under the tree.
    The size printed for each directory includes the size of its subdirectories.
  If VERBOSE is :ALL, then the size will be printed for each file under the tree, along with a subtotal
    for each directory.
  The total size of the tree is returned."
  
  (check-type verbose (member nil t :all))
  (setq top-directory (merge-pathnames top-directory
				       (make-pathname :defaults top-directory :name :wild :type :wild :version :wild)))
  (let* ((grand-total 0)
	 (supplied-name (pathname-name top-directory))
	 (supplied-type (pathname-type top-directory))
	 (supplied-vers (pathname-version top-directory)))
    
    (labels ((walker (pathname &key directory length-in-bytes &allow-other-keys)
	       (let* ((depth (if (boundp 'depth) (1+ depth) 0))
		      (string (file-namestring pathname)))
		 (declare (special depth subtotal))
		 (when length-in-bytes (incf grand-total length-in-bytes))
		 (cond (directory
			(let* ((total 0))
			  (let* ((subtotal length-in-bytes))
			    (declare (special subtotal))
			    (when (eq verbose :all)
			      (format t "~&~v@T~a ~50,10t ~10D" (* 3 depth) string length-in-bytes))
			    (do-subdirectories #'walker (make-pathname :name supplied-name
								       :type supplied-type
								       :version supplied-vers
								       :defaults (send pathname :pathname-as-directory)))
			    (case verbose
			      (nil nil)
			      (:all (format t "~&~50t ~10D total~2%" subtotal))
			      (t    (format t "~&~10D~v@T~a" subtotal (+ 2 (* 3 depth))
					    (directory-namestring (send pathname :pathname-as-directory))))
			      )
			    (setq total subtotal))
			  (when (boundp 'subtotal)
			    (incf subtotal total))))
		       
		       (t
			(when (boundp 'subtotal) (incf subtotal length-in-bytes))
			(when (eq verbose :all)
			  (format t "~&~v@T~A ~50,10t ~10D" (* 3 depth) string length-in-bytes)))))
	       
	       nil))    ; Always return NIL, so that it never automatically descends.
      
      (do-subdirectories #'walker top-directory)
      grand-total)))



;;; Comparing two directory trees.

(defun pathnames-congruent-p (pathname-1 pathname-2 &optional (compare-names t) (compare-types t) (compare-versions nil))
  (and (or (not compare-names)
	   (string= (pathname-name pathname-1) (pathname-name pathname-2)))
       (or (not compare-types)
	   (or (string= (pathname-type pathname-1) (pathname-type pathname-2))))
       (or (not compare-versions)
	   (equal (pathname-version pathname-1) (pathname-version pathname-2)))))

(defun pathname-lessp (pathname-1 pathname-2)
  (or (string< (pathname-name pathname-1) (pathname-name pathname-2))
      (and (string= (pathname-name pathname-1) (pathname-name pathname-2))
	   (string< (pathname-type pathname-1) (pathname-type pathname-2)))))


(defun safe-car (x) (if (consp x) (car x) x))

(defun insert-pathname-comparison (new-path position comparison-tree n-trees)
  (unless (dolist (vector comparison-tree nil)
	    (when (dotimes (i n-trees nil)
		    (when (aref vector i)
		      (return (if (getf (cdr (aref vector i)) :directory)
				  (pathnames-congruent-p (safe-car new-path)
							 (safe-car (aref vector i))
							 t nil nil)
				  (pathnames-congruent-p (safe-car new-path)
							 (safe-car (aref vector i))
							 t t nil)))))
	      (setf (aref vector position) new-path)
	      (return t)))
    #-TI (push (make-array n-trees) comparison-tree)
    #+TI (push (make-array n-trees :type 'ART-Q-LIST) comparison-tree)
    (setf (aref (car comparison-tree) position) new-path))
  comparison-tree)


(defun compare-dir-trees (function directories-before-their-files-p &rest directories)
  "This function lets you compare, in parallel, two or more directories.
  FUNCTION is a function of N+1 arguments, where N is the number of directories you pass in.
  The directory trees are walked down in parallel, and FUNCTION is called with the vital statistics
  of the corresponding files.  The first argument to FUNCTION is a number, which is our
  current depth under the passed-in directories.  Each subsequent argument to FUNCTION is a list,
  whose CAR is a pathname.  The CDR of the list is a property list, where the elements may be one of

    :ACCOUNT <string>
    :AUTHOR <string>
    :BLOCK-SIZE <number>
    :BYTE-SIZE <number>
    :CREATION-DATE <universal-date>
    :DELETED <boolean>
    :DONT-DELETE <boolean>
    :DONT-DUMP <boolean>
    :DONT-REAP <boolean>
    :DUMPED <boolean>
    :GENERATION-RETENTION-COUNT <number>
    :LENGTH-IN-BLOCKS <number>
    :LENGTH-IN-BYTES <number>
    :LINK-TO <string>
    :OFFLINE <boolean>
    :PHYSICAL-VOLUME <string>
    :PROTECTION <string>
    :READER <string>
    :REFERENCE-DATE <universal-date>
    :TEMPORARY <boolean>

  The CDR of the list will be NIL if the file does not actually exist.
  "
  
  (let* ((depth (if (numberp (car directories))
		    (pop directories)
		    0)))
    (setq directories
	  (map 'vector #'(lambda (directory)
			   (merge-pathnames directory (make-pathname :name :WILD :type :WILD :version :WILD)))
	       directories))
    (let* ((dir-set '())
	   (length (length directories)))
      (dotimes (i length)
	(let* ((list (and (aref directories i)
			  (cdr (condition-call (condition)
				   (fs:directory-list (aref directories i))
				 ((condition-typep condition 'FS:FILE-LOOKUP-ERROR)
				  nil))))))
	  (dolist (plist list)
	    (setq dir-set (insert-pathname-comparison plist i dir-set length)))))
      (setq dir-set (sort dir-set #'pathname-lessp :key #'(lambda (x) (safe-car (some #'identity x)))))
      
      (dolist (vec dir-set)
	(let* ((some-desc-list (some #'identity vec))
	       (some-plist (cdr some-desc-list))
	       (some-path (car some-desc-list))
	       (directory-p (getf some-plist :directory)))
	  ;;
	  ;; If an element of the vector is NIL (the file doesn't exist) put a dummy in there so that
	  ;; the sub-function knows what file it is that doesn't exist.
	  ;;
	  (dotimes (i (length vec))
	    (when (null (aref vec i))
	      (let* ((parent-dir (aref directories i))
		     (path (make-pathname :host (pathname-host parent-dir)
					  :device (pathname-device parent-dir)
					  :directory (pathname-directory parent-dir)
					  :name (pathname-name some-path)
					  :type (cond ((and directory-p (send parent-dir :unspecific-type-is-default))
						       :UNSPECIFIC)
						      ((and (eq (pathname-type some-path) :UNSPECIFIC)
							    (not (send parent-dir :unspecific-type-is-default)))
						       :DIRECTORY)
						      (t (pathname-type some-path)))
					  ; :type (pathname-type some-path)
					  :version :newest)))
		(setf (aref vec i) (list path)))))
	  ;;
	  ;; On the Explorer, the vector is an ART-Q-LIST array, so we can just lay a list-header over it and treat
	  ;; it as a list, rather than consing a new list to pass in as a rest-arg.
	  ;;
	  (when (or (not directory-p) directories-before-their-files-p)
	    #-TI (apply function depth (coerce vec 'list))
	    #+TI (apply function depth (g-l-p vec)))
	  
	  (when directory-p
	    (flet ((path-desc-to-wild-dir (list parent)
		     (let* ((pathname (car list)))
		       (make-pathname :host (pathname-host pathname)
				      :device (pathname-device pathname)
				      :directory (append (pathname-directory pathname)
							 (list (pathname-name pathname)))
				      ;; Use the same wildcarded name and type as the parent.
				      :name (pathname-name parent)
				      :type (pathname-type parent)
				      :version :newest))))
	      (apply #'compare-dir-trees function directories-before-their-files-p (1+ depth)
		     (map 'list #'path-desc-to-wild-dir vec directories)))
	    (unless directories-before-their-files-p
	      #-TI (apply function depth (coerce vec 'list))
	      #+TI (apply function depth (g-l-p vec))))
	  )))))


(defun update-directory-trees (directories &key query (verbose t))
  "Given two or more directories which would, in an ideal world, contain exactly the same files, copy the newest version
 of the corresponding files around to the other directories.  :VERBOSE is passed to COPY-FILE; :QUERY is whether to ask
 before each call to COPY-FILE.  The determination of which file is newest is made by comparing write-dates."
  (apply #'compare-dir-trees
	 #'(lambda (ignore &rest file-lists)
	     (unless (getf (cdr (car file-lists)) :directory) ; Don't do it for directories.
	       (setq file-lists (sort file-lists #'> :key #'(lambda (x) (getf (cdr x) :creation-date 0))))
	       (let* ((newest (car file-lists))
		      (newest-path (car newest))
		      (newest-date (getf (cdr newest) :creation-date))
		      (newest-date-string (and query (time:print-universal-time newest-date nil))))
		 (dolist (other (cdr file-lists))
		   (let* ((path (car other))
			  (date (getf (cdr other) :creation-date)))
		     (unless (eql date newest-date)
		       (when (or (null query)
				 (y-or-n-p
				   "~&Copy ~A (created at ~A)~%  to ~A (~:[which doesn't exist~;created at ~:*~A~])? "
				   newest-path newest-date-string
				   path (and date (time:print-universal-time date nil))))
			 (copy-file (car newest) (car other) :verbose verbose))))))))
	 t directories))


;(defun show-dir-comparison (indent &rest dirs)
;  (let* ((path (car (some #'identity dirs))))
;    (format t "~&~v@t~a.~a~30t" (* 5 indent)
;	    (pathname-name path) (pathname-type path))
;    (dolist (p dirs)
;      (format t "~4a" (not (null p))))))

;(defun show-dir-comparison-2 (indent &rest dirs)
;  (format t "~&~v@t----" (* 5 indent))
;  (dolist (x dirs)
;    (format t "~&~C~v@t ~a" (if (cdr x) #\Space #\*) (* 5 indent) (car x))))

;(defun show-dir-compare-dates (ignore &rest dirs)
;  (let* ((some-nonexistant (notevery #'cdr dirs))
;	 (dates-differ (or some-nonexistant
;			   (let* ((date (getf (cdr (car dirs)) :creation-date)))
;			     (dolist (x (cdr dirs) nil)
;			       (unless (eql (getf (cdr x) :creation-date) date)
;				 (return t)))))))
;    (when dates-differ
;      (dolist (list dirs)
;	(cond ((null (cdr list))
;	       (format t "~&File ~A does not exist." (car list)))
;	      (t
;	       (format t "~&File ~A has creation date ~100t ~A."
;		       (car list)
;		       (time:print-universal-time (getf (cdr list) :creation-date) nil))))))))



;;; The TI COPY-DIRECTORY function is very buggy.
;;; One day I hope to reimplement it using DO-SUBDIRECTORIES.
;;; That day has not yet come, but this is some code in that direction...  It doesn't work.
;;;

;(defun new-copy-directory (from to &key (verbose t) (create-directories :query))
;  (do-subdirectories #'(lambda (file)
;			 (unless (string-equal (pathname-type file) "DIRECTORY")
;			   (ticl:copy-file file to :verbose verbose :create-directories create-directories)))
;		     from))

;#+COMMENT
;(defun new-copy-directory (from to &key (verbose t) (create-directories :query)
;		                    (version nil) (copy-creation-date t) (copy-author t))
;  (setq to (merge-pathnames to from nil))
;  (let* ((source-directory (pathname-directory from))
;	 (target-directory (pathname-directory to))
;	 length)
;    (when (eq source-directory :root) (setq source-directory '()))
;    (when (eq target-directory :root) (setq target-directory '()))
;    (setq length (length source-directory))
;    (labels ((maybe-create-directory (target-pathname file-target-directory)
;	       (let* ((dir (append file-target-directory (list (pathname-name target-pathname))))
;		      (dir-path (make-pathname :directory dir
;					       :name :unspecific
;					       :type :unspecific
;					       :version :unspecific
;					       :defaults target-pathname)))
;		 (cond ((probe-file target-pathname)
;			t)
;		       (create-directories
;			(when (or (not (eq create-directories :query))
;				  (yes-or-no-p "Create the directory ~A? " target-pathname))
;			  (when verbose (format t "~&Creating directory ~A." dir-path))
;			  ;(fs:create-directory dir-path)
;			  t))
;		       (t (format t "~&Directory ~A does not exist." dir-path)
;			  nil))))
;	     (copy-one (file)
;	       (let* ((file-directory (pathname-directory file)))
;		 (when (eq file-directory :ROOT) (setq file-directory '()))
;		 (unless (equal source-directory
;				(subseq file-directory 0 length))
;		   (error "Directories inconsistent!  ~S ~S." source-directory file-directory))
;		 (let* ((file-target-directory (append target-directory (subseq file-directory length)))
;			(target-pathname (make-pathname :directory file-target-directory
;							:name (pathname-name file)
;							:type (pathname-type file)
;							:version (or version (pathname-version file))
;							:defaults to)))
;		   (cond ((string-equal (pathname-type file) "DIRECTORY")
;			  (maybe-create-directory target-pathname file-target-directory))
;			 (t
;			  (when verbose
;			    (format t "~&Copying ~A~16,16Tto ~A..." file target-pathname))
;			  ;(ticl:copy-file file target-pathname
;			  ;		:copy-creation-date copy-creation-date
;			  ;		:copy-author copy-author
;			  ;		:create-directories create-directories
;			  ;		:verbose nil)
;			  (when verbose (format t " copied.~%"))
;			  t))))))
;;      (maybe-create-directory ... )
;      (do-subdirectories #'copy-one from))))
