;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI -*-

;;; File "SPELLER-ADDITIONS"
;;; Additions to the Zwei spelling corrector.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 22 Dec 88  Jamie Zawinski    Created.
;;; 28 Dec 88  Jamie Zawinski    Made LOAD-DICTIONARY use an already-existent dictionary if one exists.
;;; 31 May 89  Jamie Zawinski 	 When you add a new word to the dictionary, it it is already there (but with too few 
;;;				  suffixes) the multiple-choice menu will now default to the existant values.
;;;				 Trailing ' characters are now ditched, so ``FOO'' is checked as the word FOO instead
;;;				  of as the word FOO''.
;;;




(defcom com-correct-spelling-multiple-dictionaries
	"Like COM-CORRECT-SPELLING, but allows the user to specify multiple dictionaries."
	()
  (let* ((speller:*current-document* (user-select-dicts "Select the dictionaries to use:")))
    (com-correct-spelling)))



(defvar *alternate-dictionaries* '()
  "A list of dictionaries to supplement the main dictionary.")


(defun load-dictionary (pathname)
  "Loads a dictionary from the given file, and puts it on the *ALTERNATE-DICTIONARIES* list."
  (let* ((name (pathname-name pathname))
	 (dictionary (or (lisp:find name *alternate-dictionaries* :test #'string-equal :key #'(lambda (x) (send x :name)))
			 (make-instance 'SPELLER:DICTIONARY :name name))))
    (with-lock (speller:*dictionary-lock*)  ; get the dictionary lock.
      (send dictionary :load pathname) ; this adds it to the alt-dicts list.
      dictionary)))



(defvar *default-dict-names* '("Main")
  "A list of the names of the dictionaries that USER-SELECT-DICTS should default to.")


(defun user-select-dicts (&optional (prompt "Select Dictionaries:") (defaults *default-dict-names*) (name "User Dict"))
  "Creates and returns an instance of SPELLER::DOCUMENT-DICTIONARY that contains the
   dictionaries that the user has selected."
  (unless (boundp 'speller:*main-dictionary*) (speller:editor-init-speller))
  (let* ((all-dicts (cons speller:*main-dictionary* *alternate-dictionaries*))
	 (items (cons "" (append all-dicts '(""))))
	 (default-dicts (delete nil
				(mapcar #'(lambda (x)
					    (lisp:find x all-dicts :test #'string-equal :key #'(lambda (d) (send d :name))))
					defaults))))
    (multiple-value-bind (dicts selected-p)
			 (w:menu-choose items
					:highlighting t :highlighted-items default-dicts
					:label prompt
					:menu-margin-choices '(:doit)
					:columns 1
					)
      (unless selected-p (signal-condition EH:ABORT-OBJECT))
      (let* ((doc (make-instance 'speller:document-dictionary :name name)))
	(dolist (dict dicts)
	  (send doc :add-dictionary dict))
	doc))))


(defvar *last-dict-selected* nil "Used as the default-default by USER-SELECT-DICT.")

(defun user-select-dict (&optional (prompt "Dictionary?") (ok-if-does-not-exist t) main-ok (default *last-dict-selected*))
  "Returns a dictionary that the user has selected, perhaps creating it."
  (declare (values dictionary created-p))
  (unless prompt (setq prompt "Dictionary?"))
  (let* ((completion-alist (mapcar #'(lambda (x) (cons (send x :name) x)) *alternate-dictionaries*))
	 (real-prompt (if default
			  (format nil "~A (default ~A)" prompt (send default :name))
			  prompt))
	 (value (completing-read-from-mini-buffer real-prompt
						  (if main-ok
						      (cons (cons "Main" speller:*main-dictionary*) completion-alist)
						      completion-alist)
						  (if ok-if-does-not-exist 'ZWEI:MAYBE nil)))
	 (created-p nil)
	 (dict
	   (cond ((equal value "") default)
		 ((stringp value)
		  (let* ((dict (make-instance 'SPELLER:DICTIONARY
					      :name value
					      :path (make-pathname :name value :type "dictionary" :version :NEWEST
								   :defaults (user-homedir-pathname)))))
		    (setq created-p t)
		    (push dict *alternate-dictionaries*)
		    dict))
		 (t (cdr value)))))
    (setq *last-dict-selected* dict)
    (values dict created-p)))


(defun speller:add-local-word (word &optional ignore)
  "Prompt the user for a dictionary, and add the word to it."
  (catch-all
    (multiple-value-bind (dict created-p) (user-select-dict (format nil "Add ~S to which dictionary?" word))
      (when (or created-p
		(and (not (member dict (send speller:*current-document* :dictionary-list) :test #'eq))
		     (y-or-n-p "Use the dictionary ~S for the remainder of this document?" (send dict :name))))
	(send speller:*current-document* :add-dictionary dict))
      (multiple-value-bind (root-word suffixes) (query-about-suffixes word)
	(send dict :add-word root-word suffixes)))))



(defvar *read-string-comtab* (make-comtab :comtab-keyboard-array '((#\Help . com-read-string-help))
					  :comtab-indirect-to *mini-buffer-comtab*))

(defun read-string-from-minibuffer (prompt &optional (initial-completion "") (initial-pos (length initial-completion))
						     *read-string-help*)
  (declare (special *read-string-help*))
  (multiple-value-bind (ignore ignore interval)
		       (edit-in-mini-buffer *read-string-comtab* initial-completion initial-pos (list prompt))
    (string-interval interval)))


(defcom com-read-string-help "Help from within READ-STRING-FROM-MINIBUFFER." ()
  (declare (special *read-string-help*))
  (format t "~&You are typing to the minibuffer.  Return or End will exit.~2%")
  (when (and (boundp '*read-string-help*) *read-string-help*)
    (princ *read-string-help*))
  (terpri)
  DIS-NONE)



(defconstant ALL-SUFFIX-FLAGS "vnxhygjdtrzspm"
  "A string of all the suffix flags that are defined.")


(defun query-about-suffixes (word)
  "Returns two values, a word and a string of the suffixes which may be applied to it."
  (cond ((not (fquery `(:help-function ,#'(lambda (&rest ignore)
					    (format t "~2&A ``complex'' word is one which may have suffixes applied to ~
                                                       it, or which is itself the suffixed form of another word.~2%")))
		      "Is ~S a complex word? " word))
	 (values word ""))
	(t
	 (let* ((real-word
		  (read-string-from-minibuffer "Type the root word:" word (length word)
		    "If this word is a suffixed form, as ``locally'' is to ``local'', then type the un-suffixed word.")))
	   (when (string= real-word "") (setq real-word word))
	   (let* ((alist '())
		  (defaults '()))
	     (dotimes (i (length ALL-SUFFIX-FLAGS))
	       (let* ((flag (schar ALL-SUFFIX-FLAGS i))
		      (mod-word (speller:add-suffix-to-word real-word (string flag)))
		      (menu-item (and mod-word `(,mod-word :value ,flag))))
		 (when mod-word
		   (push menu-item alist)
		   (when (speller:check-word-p mod-word) (push menu-item defaults)))))
	     (setq alist (nreverse alist))
	     (let* ((selected (w:menu-choose alist
					     :highlighting t
					     :highlighted-items defaults
					     :menu-margin-choices '(:DOIT)
					     :label (format nil "Select the forms of ~S which are valid." real-word))))
	       (values real-word (coerce selected 'simple-string))))))))


;;; grrr...
(defmethod (speller:dictionary :set-path) (pathname)
  (setq speller:path pathname))


(defcom com-save-dictionary
	"Prompt for the name of a dictionary, and the name of a file to write it to."
	()
  (unless *alternate-dictionaries* (barf "There are no user dictionaries."))
  (let* ((dict (user-select-dict "Save which dictionary?" nil))
	 (default-path (send dict :path))
	 (pathname (read-defaulted-pathname (format nil "Save ~S to what file?" (send dict :name))
					    default-path nil :NEWEST :WRITE)))
    (setf (send dict :path) pathname)
    (send dict :dump)
    (format *query-io* "~&Wrote ~A.~%" pathname))
  DIS-NONE)


(defcom com-show-dictionary
	"Prompt for the name of a dictionary, and dump its words to standard output."
	()
  (let* ((dict (user-select-dict "Show which dictionary?" nil t))
	 (hash (send dict :spell-hash))
	 (sort-p (if (> (hash-table-size hash) 300)  ; If it's really big, ask before sorting.
		     (y-or-n-p "Dictionary ~S has ~D entries.  Sort it first?" (send dict :name) (hash-table-size hash))
		     t)))
    (format t "Words in ~S:~2%" (send dict :name))
    (flet ((dump (word suffix)
	     (format t "~& ~A~{ ~15,15t~A~}" word (delete nil (speller:add-suffixes word suffix)))))
      (if sort-p
	(let* ((list (sort (maphash-return #'cons hash) #'string< :key #'car)))
	  (dolist (cons list) (dump (car cons) (cdr cons))))
	(maphash #'dump hash))))
  (format t "~2%Done.")
  DIS-NONE)




(set-comtab *STANDARD-COMTAB*
            '(#\M-C-$  com-correct-spelling-multiple-dictionaries)
	    (make-command-alist '(com-correct-spelling-multiple-dictionaries
				  com-save-dictionary
				  com-show-dictionary
				  )))



;;;; From MAIN.LISP
;;;
;;;  Modified this to realize that FOO'S is one word, but ``FOO'' is the word FOO instead of the word FOO''
;;;  This makes it lose on FOOS' but I can live with that.
;;;

SPELLER:
(DEFUN EDITOR-CHECK-WORD-P (WORD)
  "ZWEI interface to check if WORD is correctly spelled.  WORD assumed to contain only a-z and ' ."
  (setq word (string-trim "'" word))
  (LET ((LEN (LENGTH WORD)))
    (IF (EQL 1 LEN);accept all single letters
      T
      (WHEN (<= LEN '#,(ARRAY-TOTAL-SIZE *STRING*));reject very long words
	(SETF (FILL-POINTER *STRING*) LEN)
	(DOTIMES (I LEN)
	   (SETF (AREF *STRING* I) (CHAR-DOWNCASE (AREF WORD I))));remove case and fonts from word
	(CHECK-WORD-P *STRING*)))))



;;;; From MAIN.LISP
;;;
;;; This method was putting the string "nil" into dictionaries as suffix data - that's wrong.
;;;
SPELLER:
(DEFMETHOD (DICTIONARY :ADD-WORD) (WORD &OPTIONAL (SUFFIX NIL))
  "Adds WORD to the dictionary with valid SUFFIXes.
   Returns WORD."
  (SETF (GETHASH (STRING-DOWNCASE WORD) SPELL-HASH) (if suffix (STRING-DOWNCASE SUFFIX) ""))
  WORD)


;;; 
;;; We need this method to add SELF to the *ALTERNATE-DICTIONARIES* list if it is not the Main dict.
;;;
SPELLER:
(DEFMETHOD (DICTIONARY :LOAD) (&OPTIONAL (PATHNAME PATH))
  "Loads a dictionary from PATHNAME if specified."
  (SETQ PATH PATHNAME)
  (load pathname :verbose nil)
  (unless (string-equal speller:name "Main") (pushnew self zwei:*alternate-dictionaries*))
  (SETQ SPELL-HASH *TEMPORARY-HASH*))


;;; These were commented out in SUFFIX.LISP, but I'd like to use them.

SPELLER:
(progn

(DEFSUBST ADD-V-SUFFIX (WORD)
  "Returns a word created by adding an 'ive' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "e")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ive")
    (STRING-APPEND WORD "ive"))) 


(DEFSUBST ADD-H-SUFFIX (WORD)
  "Returns a word created by adding an 'th' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "y")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ieth")
    (STRING-APPEND WORD "th"))) 


(DEFSUBST ADD-N-SUFFIX (WORD)
  "Returns a word created by adding an 'n' type suffix to WORD."
  (SELECTOR WORD STRING-END-EQUAL
     ("e" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ion"))
     ("y" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ication"))
     (OTHERWISE (STRING-APPEND WORD "en")))) 


(DEFSUBST ADD-X-SUFFIX (WORD)
  "Returns a word created by adding an 'ns' type suffix to WORD."
  (SELECTOR WORD STRING-END-EQUAL
     ("e" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ions"))
     ("y" (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ications"))
     (OTHERWISE (STRING-APPEND WORD "ens")))) 


(DEFSUBST ADD-G-SUFFIX (WORD)
  "Returns a word created by adding an 'ing' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "e")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ing")
    (STRING-APPEND WORD "ing"))) 


(DEFSUBST ADD-J-SUFFIX (WORD)
  "Returns a word created by adding an 'ings' type suffix to WORD."
  (IF (STRING-END-EQUAL WORD "e")
    (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- (LENGTH WORD))) "ings")
    (STRING-APPEND WORD "ings"))) 


(DEFSUBST ADD-D-SUFFIX (WORD)
  "Returns a word created by adding an 'ed' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "d"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "ied"))
      (T (STRING-APPEND WORD "ed"))))) 


(DEFSUBST ADD-T-SUFFIX (WORD)
  "Returns a word created by adding an 'est' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "st"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "iest"))
      (T (STRING-APPEND WORD "est"))))) 


(DEFSUBST ADD-R-SUFFIX (WORD)
  "Returns a word created by adding an 'er' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "r"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "ier"))
      (T (STRING-APPEND WORD "er"))))) 


(DEFSUBST ADD-Z-SUFFIX (WORD)
  "Returns a word created by adding an 'ers' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((STRING-END-EQUAL WORD "e") (STRING-APPEND WORD "rs"))
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "iers"))
      (T (STRING-APPEND WORD "ers"))))) 


(DEFSUBST ADD-S-SUFFIX (WORD)
  "Returns a word created by adding an 's' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (COND
      ((AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
       (STRING-APPEND (SUBSEQ (STRING WORD) 0 (1- LENGTH)) "ies"))
      ((POSITION (GET-CHAR WORD (- LENGTH 1)) (THE STRING (STRING "szxh")) :TEST #'CHAR-EQUAL)
       (STRING-APPEND WORD "es"))
      (T (STRING-APPEND WORD "s")))))

(DEFSUBST ADD-P-SUFFIX (WORD)
  "Returns a word created by adding an 'ness' type suffix to WORD."
  (LET ((LENGTH (LENGTH WORD)))
    (IF (AND (STRING-END-EQUAL WORD "y") (NOT (VOWEL-P WORD (- LENGTH 2))))
	(STRING-APPEND (subseq WORD 0 (1- LENGTH)) "iness")
	(STRING-APPEND WORD "ness"))))
      

(DEFUN ADD-SUFFIX-TO-WORD (WORD SUFFIX)
  "Makes a new word by adding SUFFIX to WORD."
  (SELECTOR SUFFIX STRING-EQUAL ("y" (STRING-APPEND WORD "ly")) ("v" (ADD-V-SUFFIX WORD))
     ("n" (ADD-N-SUFFIX WORD)) ("x" (ADD-X-SUFFIX WORD)) ("h" (ADD-H-SUFFIX WORD))
     ("g" (ADD-G-SUFFIX WORD)) ("j" (ADD-J-SUFFIX WORD)) ("d" (ADD-D-SUFFIX WORD))
     ("t" (ADD-T-SUFFIX WORD)) ("r" (ADD-R-SUFFIX WORD)) ("z" (ADD-Z-SUFFIX WORD))
     ("s" (ADD-S-SUFFIX WORD)) ("p" (ADD-P-SUFFIX WORD)) ("m" (STRING-APPEND WORD "'s"))
     (OTHERWISE NIL))) 


(DEFUN ADD-SUFFIXES (WORD SUFFIX-STRING)
  "Returns a list of words created by adding the suffixes
   in SUFFIX-STRING to WORD."
  (LET ((WORD-LIST NIL))
    (DOTIMES (N (LENGTH SUFFIX-STRING))
      (SETQ WORD-LIST
	    (NCONC WORD-LIST
		   (LIST (ADD-SUFFIX-TO-WORD WORD (STRING (GET-CHAR SUFFIX-STRING N)))))))
    WORD-LIST))

 )
