;;; -*- Mode: Lisp; Base: 10.; Package: USER; Syntax: Common-lisp -*-

;;; Search a file (or files) for one or more strings.  The file(s) can be specified
;;; as a string with wildcards (e.g. ">gallagher>*.lisp", or ">foo>**>exotic.lisp")
;;; or as a list of pathnames or strings (no wildcards).

;;; Written by Kevin Q. Gallagher
;;; Modified by David R. Forster, July 8th, 1986:
;;;    disjunction of strings in search, use only LM wildcard mechanisms, not our own.
;;; Modified by David R. Forster, September 22nd, 1986:
;;;    specify output file, tag output lines with numbers, if desired.
;;; Modified by David R. Forster, October 9th, 1986:
;;;    documentation added
;;; Modified by David R. Forster, June 18th, 1987:
;;;    pathname lists added
;;; Modified by David R. Forster, July 22nd, 1988:
;;;    pathname lists can contain wildcarded pathnames now
;;;    added function SEARCH-SYSTEM

(provide "search-files")

(defun search-files (pathname-or-list strings &optional &key numbered output window)
  "SEARCH-FILES searches all files denoted by the wildcarded PATHNAME-OR-LIST
   and outputs each line matching any of the strings in the list STRINGS,
   together with an indication as to which file it was found in.
   NUMBERED: if T, the output includes line numbers.
   OUTPUT:   if non-NIL, it is assumed to be a file name, and output is redirected to it.
   WINDOW:   if non-NIL, it must be a number or list of two numbers giving the
             number of lines to display above and below a matched string.
   Examples:
     (SEARCH-FILES \"local:>foo>**>*x*.lisp\" '(\"hiho\" \"hee-haw\") :numbered t)
     (SEARCH-FILES \"lm:bar;*x*.lisp#>\" '(\"hiho\" \"hee-haw\") :window '(3 5))"
  (let* ((files (if (typep pathname-or-list 'cons)
		    (mapcan #'fs:directory-list pathname-or-list)
		    (fs:directory-list pathname-or-list)))
	 (above (cond ((consp window)   (first window))
		      ((numberp window) (floor window 2))
		      (t		0)))
	 (below (cond ((consp window)   (second window))
		      ((numberp window) (ceiling window 2))
		      (t		1)))
	 (*search-file-lines* (make-array (list above)
					  :element-type 'string
					  :initial-element ""))
	 (strings (if (consp strings)
		      strings
		      (list strings))))
    (declare (special *search-file-lines*))
    (flet ((search-files-in-list (pathname-list strings numbered above below)
	     (dolist (file pathname-list)
	       (let ((filename (if (consp file)
				   (and (not (getf file :directory))
					(car file))
				   file)))
		 (when filename
		   (search-for-string-in-file filename strings numbered above below))
		 ))))
      (if output
	  (with-open-file (*standard-output* output :direction :output)
	    (declare (special *standard-output*))
	    (search-files-in-list files strings numbered above below))
	  (search-files-in-list files strings numbered above below)))))


(defun search-system (system-name strings &optional &key numbered output window)
  "SEARCH-SYSTEM searches all source files used by the system denoted by
   SYSTEM-NAME and outputs each line matching any of the strings in the list
   STRINGS, together with an indication as to which file it was found in.
   NUMBERED: if T, the output includes line numbers.
   OUTPUT:   if non-NIL, it is assumed to be a file name, and output is redirected to it.
   WINDOW:   if non-NIL, it must be a number or list of two numbers giving the
             number of lines to display above and below a matched string.
   Examples:
     (SEARCH-FILES \"local:>foo>**>*x*.lisp\" '(\"hiho\" \"hee-haw\") :numbered t)
     (SEARCH-FILES \"lm:bar;*x*.lisp#>\" '(\"hiho\" \"hee-haw\") :window '(3 5))"

   (let ((system-structure (find-system system-name)))
     (flet ((search-system-components (system-structure strings numbered above below)
	      (dolist (module (system:system-modules system-structure))
		(dolist (component (system:module-components module))
		  (if (not (pathnamep (car component)))
		      (format t "~&I don't know how to search a system component like ~s~%"
			      component)
		      (let ((source-file (probe-file (make-pathname :defaults (car component)
								    :type :lisp))))
			(when source-file
			  (search-for-string-in-file
			     source-file strings numbered above below))))
		  ))
	      ))
       (if (null system-structure)
	   (format t "~&System ~s is unknown.~%" system-name)
	   (progn
	      (when (keywordp system-structure)
		; system hasn't yet been properly defined/loaded
		(make-system system-name :nowarn :silent :print-only)
		(setf  system-structure (find-system system-name)))
	      (if (not (typep system-structure 'sys::system))
		  (format t "~&I don't know how to search a system like ~s~%"
			  system-structure)
		  (let* ((above (cond ((consp window)   (first window))
				      ((numberp window) (floor window 2))
				      (t		0)))
			 (below (cond ((consp window)   (second window))
				      ((numberp window) (ceiling window 2))
				      (t		1)))
			 (*search-file-lines* (make-array (list above)
							  :element-type 'string
							  :initial-element ""))
			 (strings (if (consp strings)
				      strings
				      (list strings))))
		    (declare (special *search-file-lines*))
		    (if output
			(with-open-file (*standard-output* output :direction :output)
			  (declare (special *standard-output*))
			  (search-system-components system-structure strings numbered above below))
			(search-system-components system-structure strings numbered above below))))
	      )))
       ))

 
(defun search-for-string-in-file (pathname strings numbered above below)
  (declare (special *standard-output* *search-file-lines*))
  (flet ((display-line (line current-line-number)
	   (if numbered
	      (format *standard-output* "~&~d: ~a~%" current-line-number line)
	      (format *standard-output* "~&~a~%" line))))
    (with-open-file (stream pathname :direction :input)
      (do* ((pathname-displayed nil)
	    (eof-marker (gensym))
	    (lines-to-print -1)
	    (previous-line -1)
	    (line (read-line stream nil eof-marker)
		  (read-line stream nil eof-marker))
	    (current-line-number 1 (1+ current-line-number)))
	   ((eq line eof-marker)
	    nil)
	(dolist (str strings)
	  (when (string-search str line)
	    (unless pathname-displayed
	      (format *standard-output* "~&~%~a~%" pathname)
	      (setf pathname-displayed t))
	    (when (< lines-to-print 0)
	      (if (<= current-line-number above)
		  (dotimes (n (1- current-line-number))
		    (display-line (aref *search-file-lines* n) (1+ n)))
		  (dotimes (n above)
		    (display-line
		      (aref *search-file-lines* (mod (+ 1 previous-line n) above))
		      (+ (- current-line-number above) n)))))
	    (setf lines-to-print below)
	    (return)))
	(when (> lines-to-print 0)
	  (display-line line current-line-number))
	(decf lines-to-print)
	(when (> above 0)
	  (setf previous-line (mod (1+ previous-line) above))
	  (setf (aref *search-file-lines* previous-line) line))
	))))
