;;; -*- Mode:Lisp; Package:USER; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B HL12BI); Syntax:Common-Lisp -*-

;1;; File "*READ-BDF-FONT1"*
;1;; Reads X11 5Bitmap Distribution Format* files, or Andrew 5Font DataBase* files, and creates TI screen-fonts from the contents.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;    21 Apr 89*	1Jamie Zawinski*	1 Created.*
;1;;    23 Apr 89*	1Jamie Zawinski *	1 Added ability to read Andrew files as well.*
;1;;*

;1;; 5Synopsis:**
;1;;*
;1;; The function 5PARSE-BDF-FILE* returns a 5BDF* structure representing the contents of the X11 font-file given it.*
;1;; The function 5PARSE-FDB-FILE* returns a 5BDF* structure representing the contents of the Andrew font-file given it.*
;1;; The function 5BDF-FONT-TO-TI-FONT* will create a TI screen-font from the data in a BDF structure.*
;1;; Screen fonts can be saved to fasl files with the font editor, or with the function 5COMPILER:FASD-FONT*.*
;1;; The function 5CONVERT-FONT-DIRECTORY* will suck in BDF or FDB files from one directory, and spit XLD files into another.*
;1;;*
;1;; There are over six hundred 5BDF* and 5FDB* font files in the X Windows distribution.  Look in the subdirectories of*
;1;;*
;1;;*	3core.src/fonts/bdf/*					1; BDF files*
;1;;*	3contrib/fonts/bdf/*					1; BDF files*
;1;;*	3contrib/toolkits/andrew/overhead/fonts/fonts/*	1; FDB files*
;1;;*
;1;; Many of these fonts are public domain; all are freely redistributable.*
;1;;*
;1;; To see a list of all of the fonts available, go to the directory 3X.V11R3* and type*
;1;;*
;1;;*	3find . \( -name '*.fdb' -o -name '*.bdf' \) -print | more*
;1;;*



(defmacro 4stringcase *(string-form &body clauses)
  "2Sort of like CASE - clause-tests are strings.  If STRING-FORM begins with a clause-test, then the clause is evaluated.
  As a shorthand, the TEST parts of the clauses can be symbols - they are coerced to strings at compile-time.*"
  (let* ((var (gensym))
	 (varl (gensym)))
    `(let* ((,var ,string-form)
	    (,varl (length ,var)))
       (cond ,@(mapcar #'(lambda (clause)
			   (let* ((key (car clause))
				  (body (cdr clause))
				  (test (if (member key '(t otherwise))
					    key
					    `(string= ,var ,(string key) :end1 (min ,varl ,(length (string key)))))))
			     (cons test body)))
		       clauses)))))



(defstruct 4(bdf *(:print-function %print-bdf))
  "2This structure represents the contents of the font file.*"
  (name "" :type string)
  (size -1 :type number) ;1 In points.*
  (xres -1 :type fixnum) ;1 Intended horizontal resolution of display device in dots per inch.*
  (yres -1 :type fixnum) ;1 Intended vertical resolution of display device in dots per inch.*
  (bbox-width  -1 :type fixnum)	;1 *
  (bbox-height -1 :type fixnum)	;1 The default bounding box of characters in this font.*
  (bbox-left    0 :type fixnum)	;1 *
  (bbox-bottom  0 :type fixnum)	;1 *
  (ascent  -1 :type fixnum)	;1 The default number of bits from the baseline to the top of a character.*
  (descent -1 :type fixnum)	;1 The default number of bits from the bottom of a character to the baseline.*
  (default-char 0 :type fixnum)		;1 The index of a generic character.*
  (chars #() :type vector)		;1 Elements of this array are NIL, or BDF-CHAR structures.*
  (unencoded-chars () :type list)	;1 These are the characters for which no mapping was provided.*
  (copyright nil :type (or null string))
  (plist nil :type list)		;1 Other things that were specified in the file.*
  )


(defstruct 4(bdf-char *(:print-function %print-bdf-char))
  "2This structure represents one character of the font.*"
  (name "" :type string)		;1 The name of this character - may be something like "slash" or "cedilla" or "7".*
  (code nil :type (or null fixnum))	;1 The index of this character in a font, or NIL if unspecified.*
  (bbox-width  -1 :type fixnum)	;1 The horizontal distance that this characer consumes - it may actually be wider.*
  (bbox-height -1 :type fixnum)	;1 The distance from the top of this character to its baseline.*
  (bbox-left    0 :type fixnum)	;1 The amount of space to be inserted before the leftmost bits of the character.  May be negative.*
  (bbox-bottom  0 :type fixnum)	;1 The amount of space to be inserted below the bottommost bits of the character.  May be negative.*
  (ascent  -1 :type fixnum)	;1 The number of bits from the baseline to the top of the character.*
  (descent -1 :type fixnum)	;1 The number of bits from the bottom of the character to the baseline.*
  (width -1 :type fixnum)	;1 The real width of this character - no bits fall outside of this.*
  (bits nil :type array)	;1 An array, one element for each scanline.  Each element is itself an array, of type *(unsigned-byte 8)1.*
  )


(defun 4%print-bdf *(struct stream ignore)
  (format stream "3#<~S ~A ~D>*" (type-of struct) (bdf-name struct) (sys:%pointer struct)))

(defun 4%print-bdf-char *(struct stream ignore)
  (format stream "3#<~S ~A (~A) ~D>*" (type-of struct)
	  (bdf-char-name struct) (or (bdf-char-code struct) "3unencoded*")
	  (sys:%pointer struct)))



(defun 4parse-bdf-file *(file)
  "2Returns a BDF (Bitmap Distribution Font) structure read from FILE.*"
  (with-open-file (stream file :direction :input :characters t)
    (parse-bdf-stream stream)))


(defun 4parse-fdb-file *(file)
  "2Returns a BDF (Bitmap Distribution Font) structure read from FILE.*"
  (with-open-file (stream file :direction :input :characters t)
    (parse-fdb-stream stream)))


(defun 4bdf-readline *(stream &optional (eof-error-p t) eof-value)
  "2Reads and returns a line from STREAM, discarding all lines that begin with ``COMMENT'' or ``$Comment''.*"
  (loop
    (let* ((line (read-line stream eof-error-p eof-value)))
      (when (eq line eof-value) (return line))
      (setq line (string-right-trim '(#.(make-char 13)) line)) ;1 5Hack.*  Some of the 5X11* BDF files have Linefeeds in them...*
      (unless (or (string= "3COMMENT*" line :end2 7)
		  (string= "3$Comment*" line :end2 8))
	(return line)))))


(defun 4bdf-looking-for *(prefix-string stream &optional what (radix 10))
  "2Discards all lines that do not begin with *PREFIX-STRING2.
  If *WHAT2 is *NIL2, returns *T2.
  If *WHAT2 is* :STRING2, returns the subsequence of the line after the *PREFIX-STRING2.
  If *WHAT2 is a string, then ensures that the subsequence of the line after the *PREFIX-STRING2 is* STRING= 2to* WHAT2.
  If *WHAT2 is an integer, then returns N values, which are integers parsed out of the line read in (this uses *RADIX2).*"
  
  (let* ((pl (length prefix-string)))
    (loop
      (let* ((line (bdf-readline stream nil nil))
	     (ll (length line)))
	(unless line (error "3End of file while looking for ~S*" prefix-string))
	(when (string= prefix-string line :end2 (min pl ll))
	  
	  (cond ((eq what :string)	;1 Return the part of LINE after the prefix.*
		 (return (subseq line (1+ pl))))

		((stringp what)		;1 Make sure the rest of LINE matches.*
		 (assert (string= what line :start2 (1+ pl)) () "3Line ~S does not end in ~S.*" line what)
		 (return T))

		((numberp what)		;1 Parse and return N integers in RADIX.*
		 (let* ((i pl)
			(result '())
			n)
		   (dotimes (j what)
		     (multiple-value-setq (n i) (parse-integer line :start i :junk-allowed t :radix radix))
		     (push n result))
		   (return (apply #'values (nreverse result)))))
		
		(t (return T))))))))


(defun 4bdf-parse-bitmap-line *(string)
  "2Given a string of hexadecimal characters, returns an array of 8-bit integers.*"
  (let* ((i 0) (j 0)
	 (l (length string))
	 (h (ceiling l 2))
	 (conversion "0123456789ABCDEF")
	 (a (make-array h :element-type '(unsigned-byte 8))))
    (declare (fixnum i j l h)
	     (optimize speed))
    (when (oddp (length string)) (setq string (string-append string #\0) l (1+ l)))    ;1 bozos.*
    (loop
      (let* ((c1 (char-upcase (char string i)))
	     (c2 (char-upcase (char string (1+ i)))))
	(declare (string-char c1 c2))
	(setf (aref a j)
	      (+ (* 16 (position (the string-char c1) (the string conversion)))
	       (position (the string-char c2) (the string conversion)))))
      (incf i 2)
      (incf j)
      (when (= i l) (return a)))))


(defun 4parse-bdf-stream *(stream)
  "2Returns a BDF (Bitmap Distribution Font) structure read from STREAM.*"
  (let* ((bdf (make-bdf)))
    (bdf-looking-for "3STARTFONT*" stream "32.1*")
    (setf (bdf-name bdf) (bdf-looking-for "3FONT*" stream :string))
    (multiple-value-bind (size xres yres) (bdf-looking-for "3SIZE*" stream 3)
      (setf (bdf-size bdf) size
	    (bdf-xres bdf) xres
	    (bdf-yres bdf) yres))
    (multiple-value-bind (w h left bot) (bdf-looking-for "3FONTBOUNDINGBOX*" stream 4)
      (setf (bdf-bbox-width bdf) w
	    (bdf-bbox-height bdf) h
	    (bdf-bbox-left bdf) left
	    (bdf-bbox-bottom bdf) bot))
    (let* ((nprops (bdf-looking-for "3STARTPROPERTIES*" stream 1)))
      (dotimes (i nprops)
	(let* ((line (bdf-readline stream)))
	  (stringcase line
	    (ENDPROPERTIES (return))
	    (FONT_ASCENT   (setf (bdf-ascent bdf)       (parse-integer line :start 11 :junk-allowed t)))
	    (FONT_DESCENT  (setf (bdf-descent bdf)      (parse-integer line :start 12 :junk-allowed t)))
	    (DEFAULT_CHAR  (setf (bdf-default-char bdf) (parse-integer line :start 12 :junk-allowed t)))
	    (COPYRIGHT     (setf (bdf-copyright bdf)    (string-trim '(#\") (subseq line 10))))
	    (t (let* ((end (position #\Space line))
		      (keyword (intern (string-upcase (subseq line 0 (or end (length line)))) "3KEYWORD*"))
		      (value (if end (string-trim '(#\") (subseq line (1+ end))) 'T)))
		 (push keyword (bdf-plist bdf))
		 (push value (bdf-plist bdf))))
	    ))))
    
    (assert (bdf-ascent bdf)  () "3Must have FONT_ASCENT property.*")
    (assert (bdf-descent bdf) () "3Must have FONT_DESCENT property.*")
    
    (let* ((nchars (bdf-looking-for "3CHARS*" stream 1))	  ;1 This is number of chars in file, 6not* highest char-code!*
	   (char-array (make-array 255 :adjustable t)))
      (setf (bdf-chars bdf) char-array)
      (dotimes (charno nchars)
	(let* ((name (bdf-looking-for "3STARTCHAR*" stream :string))
	       (bdf-char (make-bdf-char))
	       enc1 enc2
	       wx wy
	       bw bh bl bb
	       code)
	  (multiple-value-setq (enc1 enc2)   (bdf-looking-for "3ENCODING*" stream 2))
	  (multiple-value-setq (wx   wy)     (bdf-looking-for "3SWIDTH*" stream 2))
	  (multiple-value-setq (wx   wy)     (bdf-looking-for "3DWIDTH*" stream 2))
	  (multiple-value-setq (bw bh bl bb) (bdf-looking-for "3BBX*" stream 4))
	  
	  (when (eql -1 enc1) (setq enc1 nil))
	  (when (eql -1 enc2) (setq enc2 nil))
	  
	  (setf (bdf-char-name bdf-char) name
		code (or enc1 enc2)
		(bdf-char-code bdf-char) code)
	  (cond (code
		 (when (>= code (array-dimension char-array 0))
		   (adjust-array char-array (1+ code)))
		 (setf (aref char-array code) bdf-char))
		(t ;1;;*(warn "3No encoding for character ~S.*" name)
		   (push bdf-char (bdf-unencoded-chars bdf))))
	  
	  (setf (bdf-char-bbox-width  bdf-char) bw
		(bdf-char-bbox-height bdf-char) bh
		(bdf-char-bbox-left   bdf-char) bl
		(bdf-char-bbox-bottom bdf-char) bb)
	  
	  (setf (bdf-char-ascent bdf-char)    (+ bh bb)	;1 Metrics.Ascent*
		(bdf-char-descent bdf-char)   (- bb)	;1 Metrics.Descent*
		(bdf-char-width bdf-char)     wx	;1 Metrics.CharacterWidth*
		)
	  (bdf-looking-for "3BITMAP*" stream)
	  (let* ((bits (make-array bh :element-type 'integer)) ;1 not necessarily fixnum.*
		 (i 0))
	    (setf (bdf-char-bits bdf-char) bits)
	    (loop
	      (let* ((line (bdf-readline stream)))
		(stringcase line
		  (ENDCHAR (return))
		  (t (setf (aref bits i) (bdf-parse-bitmap-line line))
		     (incf i)))))))))
      (bdf-looking-for "3ENDFONT*" stream)
      (setf (bdf-plist bdf) (nreverse (bdf-plist bdf)))
      bdf))


;magic
;rotation
;MaxWbase
;MaxNewline
;FontRepresentationType
;NIcons

(defmacro 4extract-fdb-digits *(x-var y-var string)
  (let ((space (gensym))
	(comma (gensym))
	(str (gensym)))
    `(let* ((,str ,string)
	    (,space (position #\Space (the string ,str) :test #'char=))
	    (,comma (position #\, (the string ,str) :start ,space :test #'char=)))
       (setq ,x-var (parse-integer ,str :start (1+ ,space) :end ,comma)
	     ,y-var (parse-integer ,str :start (1+ ,comma))))))


(defun 4parse-fdb-stream *(stream)
  "2Returns a BDF (Bitmap Distribution Font) structure read from the Andrew-format FDB file on the end of STREAM.*"
  (let* ((bdf (make-bdf))
	 line)

    (let* (nw-to-origin-x nw-to-origin-y
	   n-to-s-x n-to-s-y
	   w-to-e-x w-to-e-y)
      (loop
	(setq line (bdf-readline stream))
	(stringcase line
	  ("3$fontname*"		(setf (bdf-name bdf) (subseq line 10)))
	  ("3$pointsize*"	(setf (bdf-size bdf) (parse-integer line :start 11)))
	  ("3$MaxNWtoOrigin*"	(extract-fdb-digits nw-to-origin-x nw-to-origin-y line))
	  ("3$MaxNtoS*"		(extract-fdb-digits n-to-s-x n-to-s-y line))
	  ("3$MaxWtoE*"		(extract-fdb-digits w-to-e-x w-to-e-y line))
	  ("3$MaxWbase*"		nil)  ;1 unused*
	  ("3$MaxNewline*"	nil)  ;1 unused*
	  
	  ("$character"     (return))
	  
	  (t (let* ((space (position #\Space line :test #'char=))
		    (keyword (intern (string-upcase (subseq line 1 (or space (length line)))) "3KEYWORD*"))
		    (value (if space (string-trim '(#\") (subseq line (1+ space))) nil)))
	       (push keyword (bdf-plist bdf))
	       (push value (bdf-plist bdf))
	       ))))
      (setf (bdf-ascent bdf)      nw-to-origin-y
	    (bdf-descent bdf)     (- n-to-s-y nw-to-origin-y)
	    (bdf-bbox-width bdf)  w-to-e-x
	    (bdf-bbox-height bdf) n-to-s-y
	    (bdf-bbox-left bdf)   (- nw-to-origin-x)
	    (bdf-bbox-bottom bdf) (- nw-to-origin-y n-to-s-y)
	    ))
    
    (assert (bdf-ascent bdf)  () "3Must have FONT_ASCENT property.*")
    (assert (bdf-descent bdf) () "3Must have FONT_DESCENT property.*")

    (let* ((char-array (make-array 255 :adjustable t)))
      (setf (bdf-chars bdf) char-array)
      (block EOF
	(loop
	  ;1; At this point we have already read a line starting with 3"$character"* - so, we increment the line at the end, instead of beginning.*
	  (multiple-value-bind (code stop) (parse-integer line :start 11 :junk-allowed t)
	    (when (minusp code) (setq code nil))
	    (let* ((name (subseq line stop))
		   (bdf-char (make-bdf-char :name name :code code)))
	      (cond (code
		     (when (>= code (array-dimension char-array 0))
		       (adjust-array char-array (1+ code)))
		     (setf (aref char-array code) bdf-char))
		    (t (warn "3No encoding for character ~S.*" name)
		       (push bdf-char (bdf-unencoded-chars bdf))))
	      (let* ((box-w 0)
		     (box-h 0)
		     (origin-x 0)
		     (origin-y 0)
		     spacing-x
		     spacing-y
		     (raster-p nil))
		(loop
		  (setq line (bdf-readline stream))
		  (stringcase line
		    ("3$box*"	 (extract-fdb-digits box-w box-h line))		;1 This is the size of the box in which the bits lie.*
		    ("3$spacing*" (extract-fdb-digits spacing-x spacing-y line)) ;1 This is how much space the character consumes.*
		    ("3$origin*"	 (extract-fdb-digits origin-x origin-y line))	;1 This is the left offset.*
		    ("3$raster*"	   (return (setq raster-p t)))
		    ("3$character*" (return (setq raster-p nil)))
		    ("3$end*"	   (return-from EOF))
		    ))
		(unless spacing-x (setq spacing-x box-w))
		(unless spacing-y (setq spacing-y 0))
		(assert (and box-w box-h origin-x origin-y spacing-x spacing-y) () "3Oops.  Invalid size parameters...*")
		(setf (bdf-char-bbox-width  bdf-char) spacing-x
		      (bdf-char-bbox-height bdf-char) box-h
		      (bdf-char-bbox-left   bdf-char) (- origin-x)
		      (bdf-char-bbox-bottom bdf-char) (- origin-y box-h)
		      
		      (bdf-char-ascent  bdf-char)   origin-y
		      (bdf-char-descent bdf-char)   (bdf-char-bbox-bottom bdf-char)
		      (bdf-char-width bdf-char)     (bdf-char-bbox-width  bdf-char)
		      )

		(when raster-p
		1   *;1; At this point we have already read a line that was 3"$raster"**
		  (let* ((bits (make-array box-h :element-type 'integer)) 	;1 not necessarily fixnum.*
			 (i 0))
		    (setf (bdf-char-bits bdf-char) bits)
		    (loop
		      (setq line (bdf-readline stream))
		      (stringcase line
			3 *("3$character*" (return))
			  ("3$end*"        (return-from EOF))
			  (t (setf (aref bits i) (bdf-parse-bitmap-line line))
			     (incf i))))))))))))
    (setf (bdf-plist bdf) (nreverse (bdf-plist bdf)))
    bdf))



(defun 4bdf-font-to-ti-font *(bdf &optional name)
  "2Converts BDF-FONT structures into Explorer screen fonts.
  The name of the new font is taken from the font unless NAME is supplied.
  The symbol in the FONTS: package which names the new font is returned.
  You can save this font to a binary file using the Font Editor, FED.*"
  (check-type bdf bdf)
  (check-type name (or null string))
  (let* ((font-name (intern (or name (string-upcase (string3 *(bdf-name bdf)))) "3FONTS*"))
	 (fd (fed:font-get-fd font-name)))	; Lookup or create a Font Descriptor.

    (let* ((line-spacing (bdf-bbox-height bdf))				  ;1 Vertical distance between baselines.*
	   (baseline     (+ (bdf-bbox-height bdf) (bdf-bbox-bottom bdf))) ;1 Vertical distance from top of characters in this font.*
									  ;1 The baseline is what is aligned for different fonts.*
	   (blinker-height line-spacing)	;1 Height of a "blinker" in this font.*
	   (space-width (bdf-bbox-width bdf))	;1 Width of a space.*
	   (blinker-width space-width) 		;1 Width of a "blinker" in this font.*
	   
	   (vert-resolution (* (bdf-xres bdf) 10))   ;1 Dots per inch, times ten.*
	   (horiz-resolution (* (bdf-yres bdf) 10))  ;1 Dots per inch, times ten.*
	   )
      (setf (fed:fd-line-spacing fd) line-spacing
	    (fed:fd-baseline fd)     baseline
	    (fed:fd-blinker-height fd) blinker-height
	    (fed:fd-blinker-width  fd) blinker-width
	    (fed:fd-space-width fd)    space-width
	    (fed:fd-vert-resolution fd)  vert-resolution
	    (fed:fd-horiz-resolution fd) horiz-resolution)
      
      (let* ((chars (bdf-chars bdf)))
	(dotimes (ci (length chars))
	  (let* ((bdf-char (aref chars ci)))
	    (when bdf-char
	      (when (> ci 255)
		(format t "3~&Warning: Not converting character ~S: code ~D is too large.*"
			(bdf-char-name bdf-char) ci)
		(return))
	      (let* ((code (bdf-char-code bdf-char))
		     (width (bdf-char-width bdf-char))
		     (left-kern (- (bdf-char-bbox-left bdf-char)))
		     (descend (- (bdf-char-bbox-bottom bdf-char)))	;1 How tall the descender is.*
		     (cd (aref fd code)))		;1 Get the Char Descriptor, if it exists.*
		(unless cd				;1 Create it and store it, otherwise.*
		  (setq cd (fed:make-char-descriptor :make-array (:type 'ART-4B :dimensions '(0 0))))
		  (fed:fd-store-cd fd cd code))
		(setf (fed:cd-char-width cd) width
		      (fed:cd-char-left-kern cd) left-kern)
		
		;1; Copy the bits into the Char Descriptor.*
		(let* ((bits (bdf-char-bits bdf-char))
		       (nrows (length bits))
		       (max-nbytes 0)
		       (offset 0))
		  (dotimes (i nrows) (setf max-nbytes (max max-nbytes (length (aref bits i)))))	;1 Find max-bytes in this char.*
		  (adjust-array cd (list (max nrows (+ baseline descend)) (* 8 max-nbytes)))	;1 Resize the CD.*

		  (setq offset (- baseline nrows))
		  (incf offset descend)
		  (dotimes (i offset) (dotimes (j (* 8 nrows)) (setf (aref cd i j) 0)))		;1 Clear out area above offset.*
		  
;		  (terpri);1#########*
		  (dotimes (i nrows)
		    (let* ((this-row-bits (aref bits i))
			   (nbytes (length this-row-bits))
			   (col 0))
;		      (format t "3~%~3D: (~3d)  *" (+ offset i) col);1#########*
		      (dotimes (j nbytes)
			(let* ((this-byte (aref this-row-bits j)))
			  (dotimes (k 8)
;			    (princ (ldb (byte 1 (- 7 k)) this-byte));1#########*
			    (setf (aref cd (+ offset i) col)
				  (ldb (byte 1 (- 7 k)) this-byte))
			    (incf col))
;			  (princ "3 *");1########*
			  ))
		      )))
		))))))
    
    ;1; Turn the Font Descriptor into a real font.*
    (set font-name (fed:font-descriptor-into-font fd))
    font-name))




(defun 4convert-bdf-name *(bdf &optional prefix)
  "2Adobe gives their fonts insanely long names.  This function is for turning things like
  ``5-Adobe-Helvetica-Bold-R-Normal--24-240-75-75-P-130-ISO8859-1*'' into ``5Adobe-HL24B*''.*"
  
  (let* ((name (bdf-name bdf)))
    (cond ((> (length (string name)) 15)
	   (let* ((plist (bdf-plist bdf))
		  (full-name (getf plist :FAMILY_NAME name))
		  (size      (getf plist :PIXEL_SIZE ""))
		  (style     (getf plist :ADD_STYLE_NAME ""))
		  (slant     (getf plist :SLANT ""))
		  (weight    (getf plist :WEIGHT_NAME ""))
		  (foundry   (getf plist :FOUNDRY (or prefix ""))))
	     (setq full-name (substitute #\- #\Space full-name))
	     (when (string-equal full-name "3.bdf*" :start1 (max 0 (- (length full-name) 4)))     ;1 Bozos.*
	       (setq full-name (subseq full-name 0 (max 0 (- (length full-name) 4)))))
	     (cond ((string-equal full-name "3Helvetica*") (setq full-name "3HL*"))
		   ((string-equal full-name "3Times*")	  (setq full-name "3TR*"))
		   ((string-equal full-name "3Times*")	  (setq full-name "3TR*"))
		   ((string-equal full-name "3New-Century-Schoolbook*")   (setq full-name "3Century*"))
		   )
	     (cond ((string-equal slant "3R*") (setq slant ""))
		   ((string-equal slant "3O*") (setq slant "3I*")))
	     (cond ((string-equal weight "3Normal*") (setq weight nil))
		   ((string-equal weight "3Medium*") (setq weight nil))
		   ((string-equal weight "3Demi*")   (setq weight nil))
		   ((string-equal weight "3Light*")  (setq weight "3L*"))
		   ((string-equal weight "3Bold*")   (setq weight "3B*"))
		   ((string-equal weight "3Heavy*")  (setq weight "3H*")))
	     (setq name (string-append (format nil "3~A-~A~A~A~A*" foundry full-name size (or weight "") slant style)))))
	  (t
	   (when (string-equal name "3.bdf*" :start1 (max 0 (- (length name) 4)))     ;1 Bozos.*
	     (setq name (subseq name 0 (max 0 (- (length name) 4)))))
	   (when prefix (setq name (format nil "3~A-~A*" prefix name)))))
    name))


(defun 4convert-font-directory *(source destination &optional name-prefix)
  "2Reads all of the font files in the SOURCE directory, and writes XLD files to the destination.*"
  (setq source (merge-pathnames source (make-pathname :name :wild :type :wild :version :newest)))
  (setq destination (merge-pathnames (make-pathname :name :UNSPECIFIC :type :UNSPECIFIC :version :NEWEST
						    :defaults destination)
				     source :newest))
  (let* ((all-symbols '()))
    (dolist (source-pathname (directory source))
      (let* ((bdf (if (string-equal "3FDB*" (pathname-type source-pathname))
		      (parse-fdb-file source-pathname)
		      (parse-bdf-file source-pathname)))
	     (real-name (bdf-name bdf))
	     (font-name (string-upcase (convert-bdf-name bdf name-prefix)))
	     (font-symbol (bdf-font-to-ti-font bdf font-name))
	     (pathname1 *(make-pathname :name font-name :type :XLD :version :NEWEST :defaults destination))
	     (truename (compiler:fasd-font font-symbol pathname)))
	(setf (get font-symbol 'bdf) bdf)
	(push font-symbol all-symbols)
	(format t "3~&Wrote ~A for ~A.*" truename real-name)
	))
    all-symbols))


(defun 4display-all-fonts *()
  (let ((list nil))
    (do-local-symbols (x (find-package "3FONTS*") nil)
      (and (boundp x)
	   (typep (symbol-value x) 'font)
	   (push x list)))
    (setq list (sort list #'string-lessp))
    (dolist (font list)
      (condition-call (c)
	  (progn (w:display-font (tv:font-evaluate font))
		 (read-char))
	((condition-typep c 'sys:abort) t)))))



(pushnew '(:BDF . :FUNDAMENTAL) fs::*file-type-mode-alist*)	;1 So that the editor doesn't default to Common Lisp mode...*
(pushnew '(:FDB . :FUNDAMENTAL) fs::*file-type-mode-alist*)
(pushnew ':BDF fs::*copy-file-known-text-types*)		;1 So copy-file knows to do character translation.*
(pushnew ':FDB fs::*copy-file-known-text-types*)
