;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

;;; QIX: not just a Videogame.  A Philosophy.
;;;
;;; ChangeLog:
;;;
;;; 12 May 87  Jamie Zawinski  Created.
;;; 19 May 87  Jamie Zawinski  Got the user-drawn lines not able to intersect selves.
;;; 20 May 87  Jamie Zawinski  Wrote FILL-CONCAVE-POLYGON and POINT-IN-POLYGON.
;;;                            Broke up QIX-MAIN-LOOP into subfunctions.
;;; 22 May 87  Jamie Zawinski  Started converting to use defstructs for lines, rects, polygons...
;;; 23 May 87  Jamie Zawinski  Started writing POLYGON-TO-RECTS.
;;;  5 Jun 87  Jamie Zawinski  Reimplemented the LINE structure.
;;;  6 Jun 87  Jamie Zawinski  Changed the actual QIX code to use LINEs.
;;;  8 Jun 87  Jamie Zawinski  Got BUILD-THIRD-LINE working, worked on POLYGON-TO-RECTS some more.
;;;  7 Jul 87  Jamie Zawinski  Took a completely new tactic with breaking polygons into rectangles, and got it working.
;;;  8 Jul 87  Jamie Zawinski  Added some highly unstable code to keep the Qix inside a given polygon.
;;; 10 Jul 87  Jamie Zawinski  Defined LINE-AND-RECT-INTERSECT, RECT-AND-POLYGON-INTERSECT, etc.
;;; 27 Jul 88  Jamie Zawinski  Documented some.
;;; 22 Nov 88  Jamie Zawinski  Added color.
;;;
;;;

;;; Functionality remaining to implement:
;;;
;;;   o  allow the user to move cursor along closed borders when not drawing.
;;;   o  know how much of the board the user has captured.
;;;   o  not detect qix contacts when user is not drawing and has not yet STARTED drawing.
;;;   o  realize when user is in a dead-end, so we can bug out.
;;;   o  release and track SPARKS when: user stops drawing before completion; when user is at dead-end.
;;;   o  variable speed qixes and user-drawing.
;;;


;;;
;;; LINES and dealing with them.
;;;
;;; The LINE structure is capable of holding a large amount of data on a line, such as slope, intercepts, etc.
;;; but often this information is not needed.  So it is not calculated until it is requested.  But, once it has been 
;;; calculated, it is cached away in the line structure, and subsequent requests for it return the cached value.
;;;
;;; No user code should call functions beginning with %LINE-; these functions are the actual defstruct accessors.
;;; They should DEFINITELY NOT be SETFed, because that could invalidate cached data.  If you want to change the endpoints
;;; of a line use the function CHANGE-LINE-ENDPOINTS.
;;;
;;; Functions of note:
;;;
;;;   LINE-X and LINE-X1
;;;   LINE-Y and LINE-Y1
;;;   LINE-X2
;;;   LINE-Y2
;;;   LINE-LENGTH
;;;   LINE-X-INTERCEPT     ; These functions will return 
;;;   LINE-Y-INTERCEPT     ; either a number or the
;;;   LINE-SLOPE           ; keyword :INFINITE
;;;   LINE-MIDPOINT-X
;;;   LINE-MIDPOINT-Y
;;;
;;;   MAKE-LINE
;;;   MAKE-FULL-LINE       ; Returns a line, making sure ALL data is calculated.
;;;
;;;   Functions comparing lines:
;;;
;;;   LINES-COLINEAR
;;;   LINE-SEGMENTS-COLINEAR
;;;   LINES-INTERSECT-P
;;;

(defstruct (line (:constructor %make-line (x1 y1 x2 y2 &optional color x-intercept y-intercept length slope
					      midpoint-x midpoint-y))
		 (:print-function %print-line)
		 (:conc-name "%LINE-"))
  (x1 0 :type number)
  (y1 0 :type number)
  (x2 0 :type number)
  (y2 0 :type number)
  (length      nil :type (or number symbol))
  (slope       nil :type (or rational symbol))
  (x-intercept nil :type (or number symbol))
  (y-intercept nil :type (or number symbol))
  (midpoint-x  nil :type (or number symbol))
  (midpoint-y  nil :type (or number symbol))
  (color       nil :type (or number null))
  )

(proclaim '(inline
	     line-x line-y line-x1 line-y1 line-x2 line-y2   ;; These functions are the same as the low-level accessors
	     line-horizontal-p line-vertical-p
	     line-length line-slope
	     line-x-intercept line-y-intercept
	     line-midpoint-x line-midpoint-y
	     line-color
	     ))

;; Define user-level synonyms for the read-only slots of LINE structs.
;;
(defun line-x (line) (%line-x1 line))
(defun line-y (line) (%line-y1 line))
(defun line-x2 (line) (%line-x2 line))
(defun line-y2 (line) (%line-y2 line))
;;; Define LINE-X1 and LINE-Y1 to be exactly the same as LINE-X and LINE-Y.
;;; These are functions so that they are funcallable.
(defun line-x1 (line) (line-x line))
(defun line-y1 (line) (line-y line))

(defun line-horizontal-p (line)
  (= (line-y line) (line-y2 line)))

(defun line-vertical-p (line)
  (= (line-x line) (line-x2 line)))

(defun line-slope (line)
  (or (%line-slope line)
      (setf (%line-slope line)
	    (if (= (line-x line) (line-x2 line))
		:infinite
		(/ (- (line-y2 line) (line-y line))
		   (- (line-x2 line) (line-x line)))))))

(defun line-length (line)
  (or (%line-length line)
      (setf (%line-length line)
	    (if (line-vertical-p line)
		(- (line-y2 line) (line-y line))
		(sqrt (float (+ (expt (- (line-y2 line) (line-y line)) 2)
				(expt (- (line-x2 line) (line-x line)) 2))
			     1.s0))))))

(defun line-x-intercept (line)
  (or (%line-x-intercept line)
      (setf (%line-x-intercept line)
	    (cond ((line-vertical-p line)
		   (line-x line))
		  ((line-horizontal-p line)
		   :infinite)
		  (t (float (- (line-x line)
			       (* (/ (line-slope line)) (line-y line)))
			    1.s0))))))

(defun line-y-intercept (line)
  (or (%line-y-intercept line)
      (setf (%line-y-intercept line)
	    (cond ((line-vertical-p line)
		   :infinite)
		  ((line-horizontal-p line)
		   (line-y line))
		  (t (float (- (line-y line)
			       (* (line-slope line) (line-x line)))
			    1.s0))))))

(defun line-midpoint-x (line)
  (or (%line-midpoint-x line)
      (setf (%line-midpoint-x line)
	    (+ (line-x line)
	       (float (/ (- (line-x2 line) (line-x line)) 2)
		      1.s0)))))

(defun line-midpoint-y (line)
  (or (%line-midpoint-y line)
      (setf (%line-midpoint-y line)
	    (+ (line-y line)
	       (float (/ (- (line-y2 line) (line-y line)) 2)
		      1.s0)))))


(defun line-color (line) (%line-color line))

(defun %print-line (struct stream depth)
  (declare (ignore depth))
  (let* ((slope (%line-slope struct)))
    (format stream "#<LINE ~D,~D - ~D,~D  ~A>"
		   (line-x1 struct) (line-y1 struct) (line-x2 struct) (line-y2 struct)
		   (cond ((null slope)  "?? slope")
			 ((eq slope :infinite) "vertical")
			 ((zerop slope) "horizontal")
			 (t  (string-append "slope " (princ-to-string slope)))))))

(defun make-line (x y x2 y2 &optional color)
  "Create and return a LINE structure, with its endpoints sorted in ascending X order."
  ;; Sort the endpoints in a logical fasion: sort by ascending X, of if X are same, sort by ascending Y.
  (when (or (> x x2)
	    (and (= x x2) (> y y2)))
    (rotatef x x2) (rotatef y y2))
  (%make-line x y x2 y2 color))

(defun make-full-line (x y x2 y2 &optional color)
  "Returns a LINE structure, with its endpoints sorted in ascending-x order and all of its slots calculated."
  (let* ((line (make-line x y x2 y2 color)))
    (line-slope line)
    (line-length line)
    (line-x-intercept line)
    (line-y-intercept line)
    (line-midpoint-x line)
    (line-midpoint-y line)
    line))

(defun change-line-endpoints (line x y x2 y2)
  "Given a LINE structure, change its endpoints.  This causes cached information about the line to be discarded."
  (when (or (> x x2)
	    (and (= x x2) (> y y2)))
    (rotatef x x2) (rotatef y y2))
  (setf (%line-x1 line) x
	(%line-x2 line) x2
	(%line-y1 line) y
	(%line-y2 line) y2
	(%line-slope line) nil
	(%line-length line) nil
	(%line-x-intercept line) nil
	(%line-y-intercept line) nil
	(%line-midpoint-x line) nil
	(%line-midpoint-y line) nil)
  line)

(defsetf line-x (line) (newval)
  `(change-line-endpoints ,line ,newval (line-y ,line) (line-x2 ,line) (line-y2 ,line)))
(defsetf line-y (line) (newval)
  `(change-line-endpoints ,line (line-x ,line) ,newval (line-x2 ,line) (line-y2 ,line)))
(defsetf line-x2 (line) (newval)
  `(change-line-endpoints ,line (line-x ,line) (line-y ,line) ,newval (line-y2 ,line)))
(defsetf line-y2 (line) (newval)
  `(change-line-endpoints ,line (line-x ,line) (line-y ,line) (line-x2 ,line) ,newval))

(defsetf line-x1 (line) (newval) `(setf (line-x ,line) ,newval))
(defsetf line-y1 (line) (newval) `(setf (line-y ,line) ,newval))

(defun lines-colinear-p (linea lineb)
  "Returns T if the line segments LINEA and LINEB lie on the same line; the segments need not share any points."
  (and (eql (line-slope linea) (line-slope lineb))
       (eql (line-x-intercept linea) (line-x-intercept lineb))))

(defun lines-adjascent-p (linea lineb)
  "T if the lines share endpoints."
  (or (and (= (line-x1 linea) (line-x1 lineb))
	   (= (line-y1 linea) (line-y1 lineb)))
      (and (= (line-x2 linea) (line-x2 lineb))
	   (= (line-y2 linea) (line-y2 lineb)))
      (and (= (line-x2 linea) (line-x1 lineb))
	   (= (line-y2 linea) (line-y1 lineb)))
      (and (= (line-x1 linea) (line-x2 lineb))
	   (= (line-y1 linea) (line-y2 lineb)))))

(defun line-segments-colinear-p (linea lineb)
  "Returns T if the line segments LINEA and LINEB lie on the same line and share one or more points."
  (and (lines-colinear-p linea lineb)
       (or (>= (line-x linea) (line-x lineb)  (line-x2 linea))
	   (>= (line-x linea) (line-x2 lineb) (line-x2 linea))
	   (>= (line-x lineb) (line-x linea)  (line-x2 lineb))
	   (>= (line-x lineb) (line-x2 linea) (line-x2 lineb)))))



;; M = slope 
;; C = Y intercept.
;; Y=M*X+C                       Eq of a line
;; C=Y-M*X
;; C=Y1-((Y2-Y1)/(X2-X1))*X1
;; C=Y2-((Y2-Y1)/(X2-X1))*X2
;; Y2-((Y2-Y1)/(X2-X1))*X2 = Y1-((Y2-Y1)/(X2-X1))*X1    Plug in endpoints; they satisfy.
;;    Plug in the two points on the other line into the equation:
;;                                              (slope        * x)+((y2 -   slope    ) * x2)
;;      for (X3,Y3)  compute:  magic-number = ((Y2-Y1)/(X2-X1))*X3+(Y2-((Y2-Y1)/(X2-X1))*X2)
;;      for (X4,Y4)  compute:  magic-number = ((Y2-Y1)/(X2-X1))*X4+(Y2-((Y2-Y1)/(X2-X1))*X2)
;; If the magic-numbers have different signs, then the points lie on opposite sides of the line.
;; If the endpoints of both lines lie on opposite sides of the other line, then the lines intersect.


(defun lines-intersect-p (linea lineb &optional (endpoints-count t) (colinear-counts t))
"If ENDPOINTS-COUNT is NIL, then the lines will not be considered to intersect if only one of the endpoints of one line
 touches the other line anywhere.
If COLINEAR-COUNTS it NIL, then colinear lines will not be considered to intersect."
  (declare (inline rects-overlap-p))
  (or (and colinear-counts (line-segments-colinear-p linea lineb))
      (flet ((magic-number (x y line)
		   (- (* y (- (line-x2 line) (line-x line)))
		      (* x (- (line-y2 line) (line-y line)))
		      (- (* (line-y2 line) (line-x line)))
		      (* (line-y line) (line-x2 line)))))
;	       (+ (* (line-slope line) x)
;		  (* (line-x2 line) (- (line-y2 line) (line-slope line))))))
	(let* ((a-intersects-b (* (magic-number (line-x1 linea) (line-y1 linea) lineb)
				  (magic-number (line-x2 linea) (line-y2 linea) lineb)))
	       (b-intersects-a (* (magic-number (line-x1 lineb) (line-y1 lineb) linea)
				  (magic-number (line-x2 lineb) (line-y2 lineb) linea))))
	  (if endpoints-count
	      (and (not (plusp a-intersects-b)) (not (plusp b-intersects-a)))
	      (and (minusp a-intersects-b) (minusp b-intersects-a)))))))


(defun sort-lines-for-contiguity (lines)
  "Given a list of line segments, returns a copy of that list sorted such that endpoint-adjascent lines are adjascent
in the list.  This assumes that the line segments form a closed path: if there are any line segments that do not
connect, a list of them is returned (unsorted) as a second value."
  (let* ((px (line-x1 (car lines)))
	 (py (line-y1 (car lines)))
	 (lines-to-go (copy-list lines))
	 (result-lines nil))
    (do* ()
	 (())
      ;; LINE is the first line found containing the point PX,PY.
      ;; There may not be more than two such lines.
      (let* ((line (find-if #'(lambda (l)
				(or (and (= px (line-x1 l)) (= py (line-y1 l)))
				    (and (= px (line-x2 l)) (= py (line-y2 l)))))
			    lines-to-go)))
	;; If LINE is NIL, then either we are done, or we have found a break in the chain.  Bug out.
	(unless line (return-from SORT-LINES-FOR-CONTIGUITY (nreverse result-lines) lines-to-go))
	;; If LINE exists, then remove it from the list of lines, to search and put it on the done-list.
	(setq lines-to-go (delete line lines-to-go :test #'eq))
	(push line result-lines)
	(if (and (= px (line-x1 line)) (= py (line-y1 line)))
	    (setq px (line-x2 line)  py (line-y2 line))
	    (setq px (line-x1 line)  py (line-y1 line)))))))


;;;
;;; RECTANGLES and dealing with them.
;;;
;;; The following code does not use a cartesian coordinate system; X increases to the right and Y increases downward.
;;;

(defstruct (rect (:constructor %make-rect (x1 y1 x2 y2 width height area color))
		 (:print-function %print-rect)
		 (:conc-name "%RECT-"))
  (x1 0 :type number)
  (y1 0 :type number)
  (x2 0 :type number)
  (y2 0 :type number)
  (width  0 :type number)
  (height 0 :type number)
  (area   0 :type number)
  (color  nil :type (or number null))
  )

(proclaim '(inline
	     rect-x rect-y rect-x1 rect-y1 rect-x2 rect-y2
	     rect-left rect-top rect-right rect-bottom
	     rect-width rect-height rect-area rect-color))

(defun rect-x (rect) (%rect-x1 rect))
(defun rect-y (rect) (%rect-y1 rect))
(defun rect-x2 (rect) (%rect-x2 rect))
(defun rect-y2 (rect) (%rect-y2 rect))
(defun rect-width (rect)  (%rect-width rect))
(defun rect-height (rect) (%rect-height rect))
(defun rect-area (rect) (%rect-area rect))
;;; Define RECT-X1 and RECT-Y1 to be exactly the same as RECT-X and RECT-Y.
;;; These are functions so that they are funcallable.
(defun rect-x1 (rect) (rect-x rect))
(defun rect-y1 (rect) (rect-y rect))

(defun rect-left (rect) (rect-x rect))
(defun rect-right (rect) (rect-x2 rect))
(defun rect-top (rect) (rect-y1 rect))
(defun rect-bottom (rect) (rect-y2 rect))

;;; Changing one slot on a rectangle necessatates changing others:
;;; Changing X or Y means X2 or Y2 must change (and vice versa).
;;; Changing WIDTH or HEIGHT means that AREA and X2 or Y2 must change (and vice versa).
;;; It is not legal to change AREA, except by changing WIDTH or HEIGHT.
;;;
(defsetf rect-x (rect) (newval)
  `(setf (%rect-x1 ,rect) ,newval
	 (%rect-x2 ,rect) (+ ,newval (rect-width ,rect))))
(defsetf rect-y (rect) (newval)
  `(setf (%rect-y1 ,rect) ,newval
	 (%rect-y2 ,rect) (+ ,newval (rect-height ,rect))))

(defsetf rect-x1 (rect) (newval) `(setf (rect-x ,rect) ,newval))
(defsetf rect-y1 (rect) (newval) `(setf (rect-y ,rect) ,newval))
(defsetf rect-left (rect) (newval) `(setf (rect-x ,rect) ,newval))
(defsetf rect-right (rect) (newval) `(setf (rect-x2 ,rect) ,newval))
(defsetf rect-top (rect) (newval) `(setf (rect-y ,rect) ,newval))
(defsetf rect-bottom (rect) (newval) `(setf (rect-y2 ,rect) ,newval))

(defsetf rect-x2 (rect) (newval)
  `(setf (%rect-x2 ,rect) ,newval
	 (%rect-x1 ,rect) (- ,newval (rect-width ,rect))))
(defsetf rect-y2 (rect) (newval)
  `(setf (%rect-y2 ,rect) ,newval
	 (%rect-y1 ,rect) (- ,newval (rect-height ,rect))))

(defsetf rect-width (rect) (newval)
  `(setf (%rect-width ,rect) ,newval
	 (%rect-x2 ,rect) (+ (rect-x ,rect) ,newval)
	 (%rect-area ,rect) (* (rect-width ,rect) (rect-height ,rect))))
(defsetf rect-height (rect) (newval)
  `(setf (%rect-height ,rect) ,newval
	 (%rect-y2 ,rect) (+ (rect-y ,rect) ,newval)
	 (%rect-area ,rect) (* (rect-width ,rect) (rect-height ,rect))))


(defun rect-color (rect) (%rect-color rect))

(defun %print-rect (struct stream depth)
  (declare (ignore depth))
  (format stream "#<RECTANGLE ~D x ~D at ~D,~D>"
		   (rect-width struct) (rect-height struct) (rect-x1 struct) (rect-y1 struct)))

(defun make-rect (x y width height &optional color)
  "Create and return a RECTANGLE structure, with its calculated slots filled in.
Note that the code dealing with rectangles does not work in a cartesian coordinate system; 
X increases to the right and Y increases downward."
  ;; It's ok to create a rectangle with negative width or height, but we want the endpoints sorted.
  (let* ((x2 (+ x width))
	 (y2 (+ y height)))
    (psetq x  (min x x2)
	   x2 (max x x2))
    (psetq y  (min y y2)
	   y2 (max y y2))
    (setq width  (- x2 x)
	  height (- y2 y))
    (%make-rect x y x2 y2 width height (* width height) color)))


(defun point-in-rect (x y rect &optional (edges-count t))
  (if edges-count
      (and (<= (rect-x1 rect) x (rect-x2 rect))
	   (<= (rect-y1 rect) y (rect-y2 rect)))
      (and (< (rect-x1 rect) x (rect-x2 rect))
	   (< (rect-y1 rect) y (rect-y2 rect)))))

(defun line-and-rect-intersect (line rect &optional (endpoints-count t) (edges-count t))
  (or (point-in-rect (line-x1 line) (line-y1 line) rect edges-count)
      (point-in-rect (line-x2 line) (line-y2 line) rect edges-count)
      (lines-intersect-p line (make-line (rect-x1 rect) (rect-y1 rect) (rect-x1 rect) (rect-y2 rect))
			 endpoints-count edges-count)
      (lines-intersect-p line (make-line (rect-x1 rect) (rect-y1 rect) (rect-x2 rect) (rect-y1 rect))
			 endpoints-count edges-count)
      (lines-intersect-p line (make-line (rect-x2 rect) (rect-y1 rect) (rect-x2 rect) (rect-y2 rect))
			 endpoints-count edges-count)
      (lines-intersect-p line (make-line (rect-x1 rect) (rect-y2 rect) (rect-x2 rect) (rect-y2 rect))
			 endpoints-count edges-count)))

;;; ## These are all untested.

(defun line-and-polygon-intersect (line poly &optional (endpoints-count t) (edges-count t))
  (dolist (rect (polygon-rects poly))
    (when (line-and-rect-intersect line rect endpoints-count edges-count)
      (return t))))

(defun rect-and-polygon-intersect (rect poly &optional (edges-count t))
  (dolist (prect (polygon-rects poly))
    (when (rects-overlap-p rect prect edges-count)
      (return t))))

(defun polygons-intersect-p (polya polyb &optional (edges-count t))
  (declare (inline rects-and-polygon-intersect))
  (dolist (recta (polygon-rects polya))
    (when (rect-and-polygon-intersect recta polyb edges-count)
      (return t))))

(defun rects-overlap-p (recta rectb &optional (adjascent-counts t))
  "Returns T if the rectangles share any points.
If ADJASCENT-COUNTS is T, then rectangles with adjascent edges count as overlapping too."
  (not (if adjascent-counts
	   (or (<= (rect-x2 rectb) (rect-x1 recta))
	       (<= (rect-x2 recta) (rect-x1 rectb))
	       (<= (rect-y2 rectb) (rect-y1 recta))
	       (<= (rect-y2 recta) (rect-y1 rectb)))
	   (or (< (rect-x2 rectb) (rect-x1 recta))
	       (< (rect-x2 recta) (rect-x1 rectb))
	       (< (rect-y2 rectb) (rect-y1 recta))
	       (< (rect-y2 recta) (rect-y1 rectb))))))


;;;
;;; POLYGONS and dealing with them.
;;;
;;; In QIX, the only kind of polygons we have to deal with are those composed of horizontal and vertical lines.
;;; So all of the following code assumes the polygons to be such.
;;;

(defstruct (polygon (:constructor %make-polygon (lines rects points area))
		    (:print-function %print-poly))
  (lines  nil :type list) ; A list of the line segments composing this polygon.
  (rects  nil :type list) ; A list of the rectangles composing this polygon.
  (points nil :type list) ; A list of the points composing this polygon.
  (area 0 :type number)   ; The area of this polygon.
  )

(defun %print-poly (struct stream depth)
  (declare (ignore depth))
  (format stream "#<POLYGON ~D:  ~D points, ~D lines, ~D rectangles, area=~D>"
		 (sys:%pointer struct)
		 (length (polygon-points struct)) (length (polygon-lines struct))
		 (length (polygon-rects struct)) (polygon-area struct)))

(defun make-polygon (lines)
  "Given a list of contiguous LINEs, creates and returns a POLYGON structure, with its calculated slots filled in."
  ;; Sort the lines.
  (multiple-value-bind (sorted-lines other-lines) (sort-lines-for-contiguity lines)
    (if other-lines
	(error "Cannot make a polygon out of these lines: they do not form one and only one closed figure.")
	(setq lines sorted-lines)))
  ;; Get the points.
  (let* ((points (delete-duplicates (mapcan #'(lambda (l) (list (cons (line-x1 l) (line-y1 l))
								(cons (line-x2 l) (line-y2 l))))
					    lines)
				    :test #'equal))
	 (rects (polygon-to-rects lines))
	 (area 0))
    (dolist (r rects) (incf area (rect-area r)))
    (%make-polygon lines rects points area)))

(defun intersection-point (linea lineb)
  "Given two lines which are known to intersect, return as multiple values the X and Y of their common point.
It is an error for the two lines to not intersect, or to be colinear."
  (cond ((lines-colinear-p linea lineb)
	 (error "~S and ~S are colinear." linea lineb))
	((and (line-vertical-p linea) (line-horizontal-p lineb))        ; Very easy if the lines are right-angles.
	 (values (line-x1 linea) (line-y1 lineb)))
	((and (line-horizontal-p linea) (line-vertical-p lineb))        ;       "
	 (values (line-x1 lineb) (line-y1 linea)))
	(t (error "I haven't written code for intersection of lines not at right angles."))))
	 

(defun find-intersection (line lines)
  "LINE is some line, and LINES is a list of lines.
If LINE intersects any of LINES, return as multiple values the line intersected and the X,Y of the intersection."
  (dolist (lineb lines)
    (when (and (lines-intersect-p line lineb)
	       (not (lines-colinear-p line lineb)))
      (multiple-value-bind (x y) (intersection-point line lineb)
	(return-from FIND-INTERSECTION lineb x y)))))


;;; Turning Polygons into Rectangles.
;;;
;;;
;;;
;;; Here's how we break a rectiliniar polygon into rectangles:
;;;
;;;      Given a polygon:                          Get the left-edge lines.
;;;                                                A left-edge is a vertical line which has the "inside" of the polygon
;;;                                                to its right, and the "outside" to its left.
;;;
;;;           ---------------                                 *--------------
;;;           |             |                                 *             |
;;;           |     ----    |                                 *     ---*    |
;;;           |     |  |    --------                          *     |  *    --------
;;;           |     |  |           |                          *     |  *           |
;;;  ----------     |  |           |                 *--------*     |  *           |
;;;  |              |  -------------                 *              |  *------------
;;;  |              |                                *              |
;;;  |              |                                *              |
;;;  |               --------                        *               --------
;;;  |                      |                        *                      |
;;;  ------                 |                        *----*                 |
;;;       |                 |                             *                 |
;;;       -------------------                             *------------------
;;;
;;; Calculate the greatest common denominator of the lengths of all of the vertical lines (right edge and left edge).
;;; This is the minimum height that one of the resultant rectangles will have (in the worst case, this is 1).
;;;
;;; For each of the left-edge lines, step down them in GCD increments.
;;; Find the nearest right-edge line which is to the right of the section of the left-edge line we are looking at.
;;; Like so:
;;;
;;;    Given this segment, get this line             and produce this rectangle.  The rect is GCD tall.
;;;           |             |                                      |
;;;          \ /           \ /                                    \ /
;;;
;;;           *-------------*                                 ===============
;;;           *             *                                 ===============
;;;           |     ----    *                                 |     ----    |
;;;           |     |  |    *-------                          |     |  |    --------
;;;           |     |  |           |                          |     |  |           |
;;;  ----------     |  |           |                 ----------     |  |           |
;;;  |              |  -------------                 |              |  -------------
;;;  |              |                                |              |
;;;  |              |                                |              |
;;;  |               --------                        |               --------
;;;  |                      |                        |                      |
;;;  ------                 |                        ------                 |
;;;       |                 |                             |                 |
;;;       -------------------                             -------------------
;;;
;;; When adding a rectangle, if there is a rectangle adjascently above it with the same width, merge the two.
;;; instead of doing this,                             do this:
;;;
;;;           ---------------                                 ---------------
;;;           |             |                                 |             |
;;;           |     ----    |                                 |     ----    |
;;;           |     |  |    --------                          |     |  |    --------
;;;           |     |  |           |                          |     |  |           |
;;;  ================  |           |                 ================  |           |
;;;  ||            ||  -------------                 ||            ||  -------------
;;;  ================                                ||            ||
;;;  ||            ||                                ||            ||
;;;  ================--------                        ================--------
;;;  |                      |                        |                      |
;;;  ------                 |                        ------                 |
;;;       |                 |                             |                 |
;;;       -------------------                             -------------------
;;;
;;; And the result looks like this:
;;;
;;;           ===============               This is not the smallest number of rectangles necessary to fill the
;;;           ||           ||               area, but it is not an out-of-hand number.
;;;           =========||==||
;;;           ||   ||  ||==||=======        We decide whether a line is a left-edge by the rule that:
;;;           ||   ||  ||         ||        if a ray crosses an even number of lines, then the ray's origin is inside the
;;;  ================  ||         ||        figure.
;;;  ||            ||  =============
;;;  ||            ||                       The origin of the ray is chosen to be some point on a vertical line,
;;;  ||            ||                       and the ray is drawn perpendicular to the right.
;;;  ========================
;;;  ||                    ||               We offset this point by half of the GCD vertically, so that we don't have to
;;;  ========================               deal with the ray being coliniar to some horizontal edge of the figure.
;;;       ||               ||
;;;       ===================
;;;


(defun polygon-to-rects (lines &optional color)
  "Where LINES form a closed rectilinear polygon, this function returns a list of rectangles composing that polygon.
This will only work if the lines are horizontal or vertical, and their lengths are integers."
  (let* ((v-lines (sort (remove-if-not 'line-vertical-p lines) #'< :key 'line-x))
	 (gcd-v (apply #'gcd (mapcar #'(lambda (x) (floor (line-length x))) v-lines)))
	 (half-gcd-v (floor gcd-v 2))
	 (left-edge-lines nil)
	 (rects nil))
    
    (dolist (line v-lines)
      (let* ((middle-of-line (+ (line-y line)
				(floor (line-length line) 2)
				half-gcd-v))     ; Skew it a bit so we don't have to worry about point-adjascency.
	     (lines-this-level (remove-if-not #'(lambda (x) (<= (line-y x) middle-of-line (line-y2 x)))
					      v-lines))
	     (lines-to-right (nthcdr (1+ (position line lines-this-level)) lines-this-level)))
	;;
	;; If an arbitrary line drawn perpendicular to LINE crosses an odd number of lines, then LINE is a left-edge.
	(when (oddp (length lines-to-right))
	  (push line left-edge-lines))))
    ;;
    ;; Given sorted vertical lines, we want to build GCD rectangles out vertically, as far as possible at each GCD Y pos.
    ;; If a rectangle to be added is the same width and X origin as the rect above it, the rect above is enlarged, rather
    ;; than adding a new rect.  Thusly, this algorithm fills the polygon with the least number of rectangles possible.
    ;;
    (dolist (v-line left-edge-lines)
      (let* ((lines-to-right (nthcdr (1+ (position v-line v-lines)) v-lines)))
	(dotimes (gcd-y (floor (line-length v-line) gcd-v))
	  (let* ((x (line-x v-line))
		 (y (+ (line-y v-line) (* gcd-v gcd-y)))
		 (nearest-line (find-if #'(lambda (l) (<= (line-y l)
							  (+ y half-gcd-v)  ; Skew it a bit, as above.
							  (line-y2 l)))
					lines-to-right))
		 (distance (when nearest-line (- (line-x nearest-line) x)))
		 (last-rect-pushed (car rects)))
	    (when nearest-line
	      (if (and last-rect-pushed
		       (= distance (rect-width last-rect-pushed))
		       (= y (rect-bottom last-rect-pushed)))
		  (incf (rect-height last-rect-pushed) gcd-v)
		  (push (make-rect x y distance gcd-v color) rects)))))))
    rects))

#|

(setq test-lines (list (make-line 10 10 50 10)
		       (make-line 50 10 50 30)
		       (make-line 50 30 30 30)
		       (make-line 30 30 30 50)
		       (make-line 30 50 10 50)
		       (make-line 10 50 10 10)))


(setq test-lines (list (make-line 10 10 40 10)
		       (make-line 40 10 40 30)
		       (make-line 40 30 30 30)
		       (make-line 30 30 30 20)
		       (make-line 30 20 20 20)
		       (make-line 20 20 20 40)
		       (make-line 20 40 40 40)
		       (make-line 40 40 40 50)
		       (make-line 40 50 10 50)
		       (make-line 10 50 10 10)))

(setq test-lines (list (make-line 100 100 400 100)
		       (make-line 400 100 400 300)
		       (make-line 400 300 300 300)
		       (make-line 300 300 300 200)
		       (make-line 300 200 200 200)
		       (make-line 200 200 200 400)
		       (make-line 200 400 400 400)
		       (make-line 400 400 400 500)
		       (make-line 400 500 100 500)
		       (make-line 100 500 100 100)))

(defun tq (&optional filled-p)
  (send *qix-window* :select)
  (let* ((p (make-polygon test-lines)))
    (draw-qix-polygon *qix-window* p tv:alu-xor filled-p)
    (qix-internal *qix-window* (list (build-qix-list 110 110 110 110 10)) 10 50 p)
    ))

(defun tl (lines)
  (send *terminal-io* :clear-screen)
  (dolist (l lines)
    (draw-qix-line *terminal-io* l tv:alu-xor))
  (send *terminal-io* :increment-cursorpos 0 200))

(defun tr (rects)
  (send *terminal-io* :clear-screen)
  (dolist (r rects)
    (draw-qix-rect *terminal-io* r tv:alu-xor))
  (send *terminal-io* :increment-cursorpos 0 200))

 |#

(defun point-in-polygon (x y polygon &optional (edges-count t))
  (dolist (rect (polygon-rects polygon) nil)
    (when (point-in-rect x y rect edges-count)
      (return T))))

(defun line-intersects-polygon (line poly &optional (endpoints-count t) (adjascent-counts t))
  (dolist (lineb (polygon-lines poly) nil)
    (when (lines-intersect-p line lineb endpoints-count adjascent-counts)
      (return-from LINE-INTERSECTS-POLYGON t))))

(defun polygons-overlap (polya polyb &optional (adjascent-counts t))
  (dolist (recta (polygon-rects polya))
    (dolist (rectb (polygon-rects polyb))
      (when (rects-overlap-p recta rectb adjascent-counts)
	(return-from POLYGONS-OVERLAP t)))))


(defun divide-line (line x y)
  "Returns as two values two new lines, if the point was on the line."
  (if (and (<= (line-x1 line) x (line-x2 line))
	   (<= (line-y1 line) y (line-y2 line)))
      (values (make-line (line-x1 line) (line-y1 line) x y (line-color line))
	      (make-line x y (line-x2 line) (line-y2 line) (line-color line)))
      (values line nil)))


(defun divide-polygon (poly x y)
  "Given a polygon and a point on that polygon, break the polygon at that point.
  That is, if a point is in the middle of a line, divide the line in two at that point.
  This is destructive."
  (dolist (line (copy-list (polygon-lines poly)))
    (multiple-value-bind (a b) (divide-line line x y)
      (when b
	(setf (polygon-lines poly) (delete line (polygon-lines poly)))
	(push b (polygon-lines poly))
	(push a (polygon-lines poly))
	(push (cons x y) (polygon-points poly))
	)))
  (multiple-value-bind (sorted-lines other-lines) (sort-lines-for-contiguity (polygon-lines poly))
    (if other-lines
	(error "Cannot make a polygon out of these lines: they do not form one and only one closed figure.")
	(setf (polygon-lines poly) sorted-lines)))
  (setf (polygon-rects poly) (polygon-to-rects (polygon-lines poly) (rect-color (car (polygon-rects poly)))))
  poly)
  

;;;
;;; QIX ITSELF
;;;

(setq *print-circle* t)   ; this is awfully important.

(defvar *qix-window* nil "The window in which the game is played.")

(proclaim '(inline draw-qix-line draw-qix-rect))

(defun draw-qix-line (window line &optional (alu tv:alu-transp))
  "Draw a line on WINDOW representing the LINE structure."
  (let* ((color (or (line-color line) (tv:sheet-foreground-color window))))
    (when (eq alu tv:alu-setz) (setq color (tv:sheet-background-color window)
				     alu tv:alu-seta))
    (send window :draw-line (line-x1 line) (line-y1 line) (line-x2 line) (line-y2 line) 1 color alu)))

(defun draw-qix-rect (window rect &optional (alu tv:alu-transp) filled-p)
  "Draw a hollow or filled rectangle on WINDOW representing the RECT structure."
  (let* ((color (or (rect-color rect) (tv:sheet-foreground-color window))))
    (when (eq alu tv:alu-setz) (setq color (tv:sheet-background-color window)
				     alu tv:alu-seta))
    (let* ((x (rect-x rect))
	   (y (rect-y rect))
	   (w (rect-width rect))
	   (h (rect-height rect)))
      (if filled-p
	  (send window :draw-filled-rectangle x y (1- w) (1- h) color alu)
	  (send window :draw-rectangle x y w h 1 color alu)))))


(defun draw-qix-polygon (window poly &optional (alu tv:alu-transp) filled-p)
  "Draw the rectangles composing the polygon on the window."
  (dolist (rect (polygon-rects poly))
    (draw-qix-rect window rect alu filled-p)))


(defvar *qix-line-colors* (mapcar #'cdr w:color-alist))

(defun build-qix-list (x y x2 y2 length)
  "Builds and returns a circular list for use with QIX.
This list has LENGTH elements and is made of LINE structures, each specifying the endpoints of one line of the QIX.
The QIX always contains at most LENGTH lines."
  (let* ((list (list nil)))
    (dotimes (l length)
      (let* ((color (nth (random (length *qix-line-colors*)) *qix-line-colors*)))
	(push (make-line x y x2 y2 color) list)))
    (setf (nthcdr length list) list)  ; Tail-link it.
    list))

(defun qix-test (&key (how-many 1) (qix-length 100) (move-inc 10) (max-width 100))
  "Animate some QIX on the *QIX-WINDOW*.
HOW-MANY   how many QIX to run.
QIX-LENGTH how many lines each QIX is composed of.
MOVE-INC   max number of pixels a new line may be grown from the active edge of a QIX.
MAX-WIDTH  max length in pixels a line segment may be."
  (unless *qix-window*
    (setq *qix-window* (make-instance 'w:window :edges-from :mouse :label nil  :name "Qix Window")))
  (let* ((win *qix-window*))
    (send win :select)
    (dolist (x (send win :blinker-list)) (send x :set-visibility :off))
    (let* ((mid-x (floor (send win :inside-width) 2))
	   (mid-y (floor (send win :inside-height) 2))
	   (half-width (floor max-width 2))
	   (qixes nil))
      (dotimes (x how-many)
	(push (build-qix-list (- mid-x half-width) (- mid-y half-width)
			      (+ mid-x half-width) (+ mid-y half-width)
			      qix-length)
	      qixes))
      (qix-internal win qixes move-inc max-width))
    (send *qix-window* :clear-input)
    (send *qix-window* :deactivate)
    )
  nil)

(defun qix-internal (sheet qixes &optional (move-inc 10) (max-width 50) constraining-polygon)
  "Animate the QIXes until there is some keyboard activity.
SHEET is the sheet to animate the QIX on.
QIXes is a list of QIXes of the type made by BUILD-QIX-LIST.
MOVE-INC is the maximum number of pixels away from the old head that a new line may grow.
MAX-WIDTH is the maximum length in pixels that a line may be.
CONSTRAINING-POLYGON, if non-NIL, is the polygon that the QIX must always remain inside of."
  (let* ((start-time tv:kbd-last-activity-time))
    (do* ()
	 ((/= start-time tv:kbd-last-activity-time)
	  nil)
      (setq qixes (qix-internal-once sheet qixes move-inc max-width constraining-polygon)))))


(defun qix-internal-once (sheet qixes &optional (move-inc 10) (max-width 50) constraining-polygon)
  "Animate each of the QIXes one generation.

SHEET is the sheet to animate the QIX on.
QIXes is a list of QIXes of the type made by BUILD-QIX-LIST.
MOVE-INC is the maximum number of pixels away from the old head that a new line may grow.
MAX-WIDTH is the maximum length in pixels that a line may be.
CONSTRAINING-POLYGON, if non-NIL, is the polygon that the QIX must always remain inside of.

Returns a new pointer to QIXES which you should pass back into this function to get the next generation."
  
  (let* ((width (- (tv:sheet-inside-width sheet) 10))
	 (height (- (tv:sheet-inside-height sheet) 10)))
    ;; Iterate over all of the QIXes we've been given, keeping track of where they are in the list.
    (do* ((l (length qixes))
	  (count 0 (1+ count))
	  (qix-list (car qixes) (nth count qixes)))  ; ## do we really need to use NTH?
	 ((= count l)
	  qixes)
      ;; POPping on a circular list rotates it toward the front.
      (pop (nth count qixes))
      (tv:without-interrupts
	(let* ((head (second qix-list))    ; HEAD is the line to update and draw.
	       (prev (first qix-list)))    ; PREV was the HEAD last time through.
	  ;; Erase the TAIL of the QIX.  What is now the HEAD is actually the LAST link of the QIX.
	  (draw-qix-line sheet head TV:ALU-SETZ)
	  ;; Transform the TAIL into a new HEAD.  Calculate the HEAD based on the PREV head.
	  ;; Recalculate until both ends are in the polygon.
	  ;;
	  ;; ## Major problems here:
	  ;;    We keep iterating until we get one that works - what if we CAN'T get one that works?  Hung forever.
	  ;;    Also, we assume that if both endpoints are in the polygon, the whole line is.  FALSE.
	  ;;    We should calculate only once, make it as complicated as necessary, but this iteration is a mondo BAD idea.
	  ;;
	  (do ((done-once nil t))
	      ((and done-once
		    (or (null constraining-polygon)
			(not (line-intersects-polygon head constraining-polygon nil nil)))
		    ))
	    (change-line-endpoints head
	      ; Aesthetic: no QIX point can move more than 5 pixels at a time.
	      (round (max 5 (min width  (+ (line-x prev) (* move-inc (1- (random 2.0s0)))))))
	      (round (max 5 (min height (+ (line-y prev) (* move-inc (1- (random 2.0s0)))))))
	      (round (max 5 (min width  (max (- (line-x head) max-width)
					     (min (+ max-width (line-x head))
						  (+ (line-x2 prev)  (* move-inc (1- (random 2.0s0)))))))))
	      (round (max 5 (min height (max (- (line-y head) max-width)
					     (min (+ max-width (line-y head))
						  (+ (line-y2 prev) (* move-inc (1- (random 2.0s0)))))))))))
	  ;; draw the new HEAD.
	  (draw-qix-line sheet head TV:ALU-SETA))))))



(defvar *cursor-x* nil "The x position of the user's player in the qix window.")
(defvar *cursor-y* nil "The x position of the user's player in the qix window.")
(defvar *drawing* nil "Whether and what kind of drawing is going on - this is NIL, :FAST, or :SLOW.")
(defvar *direction* nil "Which direction the drawing is taking place in - this is NIL, :UP, :DOWN, :LEFT, or :RIGHT.")
(defvar *walls* nil "This is a list of four LINEs, the outermost walls of the playfield.")
(defvar *walls-polygon* nil "This is a polygon made of *WALLS*.")
(defvar *lines-already-drawn* nil "This is a list of the lines drawn and completed by the user.")
(defvar *lines-being-drawn* nil "This is a list of lines in the current path that the user is drawing.")
(defvar *polygons-drawn* nil "This is a list of the polygons the user has placed.")
(defvar *needs-refreshed* nil "Setting this to T will cause everything to be refreshed.")
(defvar *qixes* "This is a list of The Enemy.  There is one qix here initially, and more are added on higher levels.")

(defparameter *qix-ratio* 10 "A speed factor - the qix moves this many times faster than the user.")
(defparameter *cursor-slow-inc* 1 "How fast slow-drawing goes.")
(defparameter *cursor-fast-inc* 3 "How fast fast-drawing goes.")


(defun handle-user-input (window)
  "Handle characters typed by the user.
  Control-L means redisplay everything.
  Control-<cursor-key> means draw quickly in the direction specified.
  Meta-<cursor-key> means draw slowly in the direction specified.
  <cursor-key> means draw at the same speed in the direction specified.  Must already be drawing.
  Once we are drawing, cannot change speed."
  
  (let* ((char-typed-p (listen window))
	 (char (read-char-no-hang window))
	 (was-drawing *drawing*))
    (case char
	  (nil  (setq char-typed-p nil))
	  (#\c-L (setq *needs-refreshed* t))
	  (#\     (setq *direction* :up    ))
	  (#\     (setq *direction* :left  ))
	  (#\     (setq *direction* :down  ))
	  (#\     (setq *direction* :right ))
	  
	  (#\c-   (setq *direction* :up    *drawing* (or *drawing* :fast)))
	  (#\c-   (setq *direction* :left  *drawing* (or *drawing* :fast)))
	  (#\c-   (setq *direction* :down  *drawing* (or *drawing* :fast)))
	  (#\c-   (setq *direction* :right *drawing* (or *drawing* :fast)))
	  
	  (#\m-   (setq *direction* :up    *drawing* (or *drawing* :slow)))
	  (#\m-   (setq *direction* :left  *drawing* (or *drawing* :slow)))
	  (#\m-   (setq *direction* :down  *drawing* (or *drawing* :slow)))
	  (#\m-   (setq *direction* :right *drawing* (or *drawing* :slow)))
	  (t    (setq char-typed-p nil) (beep)))
    (let* ((xsign 0) (ysign 0) (inc 0))
      (ecase *direction*
	     (:LEFT  (setq xsign -1))
	     (:RIGHT (setq xsign 1))
	     (:UP    (setq ysign -1))
	     (:DOWN  (setq ysign 1))
	     (nil    nil))
      (ecase *drawing*
	     (:FAST (setq inc *cursor-fast-inc*))
	     (:SLOW (setq inc *cursor-slow-inc*))
	     (nil    nil))
      (when was-drawing
	(incf *cursor-x* (* xsign inc))
	(incf *cursor-y* (* ysign inc))))))


(defun handle-cursor-drawing (old-cursor-x old-cursor-y old-direction)
  "When the user is drawing, increment the cursor position and current-line length accordingly."
  ;;
  ;; Make there be some line if there isn't one.
  ;;
  (unless *lines-being-drawn* (push (make-line old-cursor-x old-cursor-y *cursor-x* *cursor-y*) *lines-being-drawn*))
  ;;
  ;; If they tried to reverse direction, barf.
  ;;
  (when (or (and (eq *direction* :left)  (eq old-direction :right))
	    (and (eq *direction* :right) (eq old-direction :left))
	    (and (eq *direction* :up)    (eq old-direction :down))
	    (and (eq *direction* :down)  (eq old-direction :up)))
    (beep)
    (throw 'MOVE-ERROR nil))
  
  (cond ;;
        ;; Add a new line to the list if the direction has changed.
        ;;
        ((and (not (and (line-horizontal-p (car *lines-being-drawn*))
			(line-vertical-p (car *lines-being-drawn*))))
	      (or (and (line-vertical-p (car *lines-being-drawn*))
		       (/= (line-x (car *lines-being-drawn*)) *cursor-x*))
		  (and (line-horizontal-p (car *lines-being-drawn*))
		       (/= (line-y (car *lines-being-drawn*)) *cursor-y*))))
	 ;; TAIL-P is T if the first line touches the second line at (X,Y), and is NIL if it touches at (X2,Y2).
	 ;; We need to know this to decide which end of the first line the new line should be adjascent to.
	 ;;
	 (let* ((tail-p (and (second *lines-being-drawn*)
			     (or (and (= (line-x (car *lines-being-drawn*)) (line-x (second *lines-being-drawn*)))
				      (= (line-y (car *lines-being-drawn*)) (line-y (second *lines-being-drawn*))))
				 (and (= (line-x (car *lines-being-drawn*)) (line-x2 (second *lines-being-drawn*)))
				      (= (line-y (car *lines-being-drawn*)) (line-y2 (second *lines-being-drawn*)))))))
		(new-line (if tail-p
			      (make-line (line-x2 (car *lines-being-drawn*)) (line-y2 (car *lines-being-drawn*))
					 *cursor-x* *cursor-y*)
			      (make-line (line-x (car *lines-being-drawn*)) (line-y (car *lines-being-drawn*))
					 *cursor-x* *cursor-y*))))
	   ;; Before adding the new line, make sure it doesn't cross its own path.
	   (dolist (line (cdr *lines-being-drawn*))
	     (when (lines-intersect-p line new-line)
	       (throw 'MOVE-ERROR nil)))
	   (push new-line *lines-being-drawn*)
	   (draw-qix-line *qix-window* new-line)))
	;;
	;; Barf if the new line would have taken them out of the playfield.
	;;
	;; ## this isn't working right - it lets you get out of the playfield, and keeps you there...
;	((and (not (point-in-polygon *cursor-x* *cursor-y* *walls-polygon*))
;	      (not (point-in-polygon old-cursor-x old-cursor-y *walls-polygon*)))
;	 (beep :vt100-beep)
;	 (throw 'MOVE-ERROR nil))
	;;
	;; Modify the old line.
	;;
	(t (let* ((maybe-line (make-line (line-x (car *lines-being-drawn*)) (line-y (car *lines-being-drawn*))
					 *cursor-x* *cursor-y*)))
	     ;; ##  Inefficiently consing a line here; MAYBE-LINE is always thrown away.
	     ;;
	     ;; This next clause makes it impossible to draw a line which overlaps a line already drawn.
	     (dolist (line (cddr *lines-being-drawn*))
	       (when (lines-intersect-p line maybe-line)
		 (throw 'MOVE-ERROR nil)))
	     
	     (let* ((l (car *lines-being-drawn*)))
	       (if (and (= old-cursor-x (line-x1 l)) (= old-cursor-y (line-y1 l)))
		   (change-line-endpoints l *cursor-x* *cursor-y* (line-x2 l) (line-y2 l))
		   (change-line-endpoints l (line-x l) (line-y l) *cursor-x* *cursor-y*))
	       (draw-qix-line *qix-window* l)
	       ))))
  nil)

;;; This function MUST draw with XOR.
(defun draw-head (x y)
  "Draw some glyph at XY representing the active head of the user's line."
  (send *qix-window* :string-out-explicit #.(string #\Escape) x y nil nil fonts::cptfont tv:alu-xor))

(defun handle-cursor-movement ()
  (let* ((on-some-line nil)
	 (test-line (make-line *cursor-x* *cursor-y* *cursor-x* *cursor-y*)))   ; ## This uncooly conses a line.
    (dolist (l (append *lines-already-drawn* *walls*))
      (when (lines-intersect-p test-line l) (setq on-some-line t) (return)))
    (unless on-some-line (throw 'MOVE-ERROR nil))))

(defun perhaps-closure (window)
  (when *lines-being-drawn*
    (let* ((connections 0))
      (dolist (l (append *lines-already-drawn* *walls*))
	(let* ((active-line (car *lines-being-drawn*))
	       (origin-line (car (last *lines-being-drawn*))))
	  (when (lines-intersect-p l active-line) (incf connections))
	  (when (and (not (= (length *lines-being-drawn*) 1))
		     (lines-intersect-p l origin-line))
	    (incf connections))
	  (when (>= connections 2) ; should never be greater than 2, but just in case...
	    (let* ((dest (make-polygon *walls* ))) ; (append *lines-already-drawn* *walls*))))
	      (divide-polygon dest (line-x1 active-line) (line-y1 active-line))
	      (divide-polygon dest (line-x2 active-line) (line-y2 active-line))
	      (divide-polygon dest (line-x1 origin-line) (line-y1 origin-line))
	      (divide-polygon dest (line-x2 origin-line) (line-y2 origin-line))
	      (let* ((poly (make-polygon (polygon-lines dest))))
		(push poly *polygons-drawn*)
		(draw-qix-polygon window poly)
		(setq *lines-already-drawn* (append *lines-already-drawn* *lines-being-drawn*)
		      *lines-being-drawn* nil
		      *drawing* nil)
		(return)))))))))


(defun perhaps-hit (qix-length)
  (let* ((nth-line (floor qix-length 5)))
    (dolist (q *qixes*)
      (do* ((rest-of-lines q (nthcdr nth-line rest-of-lines))
	    (count 0 (1+ count))
	    (line (car rest-of-lines) (car rest-of-lines)))
	   ((> count 0))
	(dolist (line2 *lines-being-drawn*)
	  (declare (inline lines-intersect-p))
	  (when (lines-intersect-p line line2)
	    (return-from PERHAPS-HIT t))))))
  nil)


(defun qix-main-loop ()
  (unless *qix-window*
    (setq *qix-window* (make-instance 'tv:window :edges-from :mouse
						 :label nil  :name "Qix Window")))
  (let* ((window *qix-window*)
	 (margins 15)
	 (width (- (send window :inside-width) (* 2 margins)))
	 (height (- (send window :inside-height) (* 2 margins)))
	 (move-inc 10) (max-width 100) (qix-length 10)
	 (mid-x (floor width 2)))
    (setq *cursor-x* mid-x
	  *cursor-y* height
	  *lines-already-drawn* nil
	  *lines-being-drawn* nil
	  *polygons-drawn* nil
	  *walls* (list (make-line margins margins width margins) (make-line width margins width height)
			(make-line margins height width height) (make-line margins margins margins height))
	  *walls-polygon* (make-polygon *walls*)
	  *drawing* nil
	  *direction* nil
	  *needs-refreshed* t
	  *qixes* (list (build-qix-list mid-x 10 mid-x 10 qix-length)))  ; only one qix.
    (send window :select)
    (loop
      ;;
      ;; Iterate over all of the QIXes, keeping track of where they are in the list.
      (dotimes (n *qix-ratio*)
	(setq *qixes* (qix-internal-once window *qixes* move-inc max-width )));*walls-polygon*)))
      (let* ((old-cursor-x *cursor-x*)
	     (old-cursor-y *cursor-y*)
	     (old-direction *direction*))
	(draw-head old-cursor-x old-cursor-y)
	;;
	;; Deal with the user.
	(handle-user-input *qix-window*)
	;;
	;; Erase the old lines.
	(when *needs-refreshed*
	  (send window :clear-screen)
	  (dolist (l (append *lines-being-drawn* *lines-already-drawn* *walls*))
	    (draw-qix-line *qix-window* l tv:alu-setz)))
	;;
	;; Calculate the results of what the user did.
	;;
	(let* ((completed-successfully nil))
	  (catch 'MOVE-ERROR
	    (if *drawing*
		(progn (handle-cursor-drawing old-cursor-x old-cursor-y old-direction)
		       (perhaps-closure window))
		(handle-cursor-movement))
	    (setq completed-successfully t))
	  (unless completed-successfully
	    (setq *cursor-x* old-cursor-x  *cursor-y* old-cursor-y  *direction* old-direction)))
	;;
	;; Redraw lines.
	(when *needs-refreshed*
	  (setq *needs-refreshed* nil)
	  (dolist (l (append *lines-being-drawn* *lines-already-drawn* *walls*))
	    (draw-qix-line window l))))
      (draw-head *cursor-x* *cursor-y*)
      ;;
      ;; Deal with hits.
      (let* ((hit-p (perhaps-hit qix-length)))
	(when hit-p (beep :vt100-beep)))
      )))
