;;; -*- Mode:Common-Lisp; Package:ZWEI; Base:10 -*-

;; want to be able to define a list of keyword/subject pairs such that all
;; messages of this form or of the form <"RE: " + subject> will be ignored

(defvar kill-filter-list
	'()
   "List of tedious subjects and tedious people in mailing lists (keywords).
Structure: ((keyword subject person) ...)  --  see kill-list-filter for semantics.")


(define-mail-filter kill-list-filter "Kill List"
   "Sort out messages with tedious subjects or from tedious people on a
per-list (keyword) basis.  Uses Kill-Filter-List, with structure:
\((keyword subject person) ...)  --  KEYWORD must be a keyword (matching a
specific list) or T (matching all lists), SUBJECT must be a string or
NIL (in which case it is ignored), and PERSON must be a string or NIL (in
which case it is ignored).  If both Subject and Person are NIL, then the
entry is ignored."
   (kill-list-filter-function msg))


(defvar minimum-mail-subject-match-length 5
   "Minimum length of subject heading in a mail message which is required
to match when `re:' prefixes have been stripped.")

(defun kill-list-filter-function (msg)
   ;; N.B.: accessing the FROM field is complicated, for no good reason (IMHO), so we
   ;; cheat by using :summary-from in the :plist of the msg.
   ;; see (define-system-mail-filter FROM ...), Zwei::Msg-Header-Has-Search-String and
   ;; Zwei::Make-Header-Search-Filter for details about the `right' way to do it.
   (let* ((msg-keys (send msg :keywords))
	  (msg-subject (string-remove-fonts (send msg :name)))
	  msg-no-re-subject
	  (msg-status (send msg :status))
	  (msg-from (flet ((get-status-entry (name status)
			     (let ((entry (cadr (assoc name status))))
			       (and (consp entry)
				    (eq (car entry) :interval)
				    (string-interval (send (cadr entry) :first-bp)
						     (send (cadr entry) :last-bp))))))
		      (or (get-status-entry :reply-to msg-status)
			  (get-status-entry :sender msg-status))))
	  (msg-person (string-remove-fonts
			 (or (getf (send msg :plist) :summary-from)
			     msg-from))))
     (do ((subject msg-subject
		   (string-trim '(#\space #\tab #\:) (subseq subject 3))))
	 ((not (string-equal "re:" subject :end2 3))
	  (setf msg-no-re-subject subject)))
     (find-if 
	#'(lambda (kill-entry
		   &aux (keys (first kill-entry))
			(subject (second kill-entry))
			(person (third kill-entry)))
	    (and (or subject person)
		 (or (eq keys t)
		     (intersection keys msg-keys))
		 (or (null subject)
		     (string-equal subject msg-subject)
		     (let ((end (max minimum-mail-subject-match-length
				     (min (length subject) (length msg-no-re-subject)))))
		       (string-equal subject msg-no-re-subject :end1 end :end2 end)))
		 (or (null person)
		     (string-equal person msg-person))))
	kill-filter-list)))

(set-comtab *read-mail-comtab*  '(#\s-k com-add-kill-filter-list-entry))
(set-comtab *read-mail-comtab*  '(#\s-s com-add-subject-to-kill-filter-list))
(set-comtab *read-mail-comtab*  '(#\s-f com-add-person-to-kill-filter-list))
(set-comtab *read-mail-comtab*  '(#\s-w com-write-kill-filter-list))
(set-comtab *read-mail-comtab*  '(#\s-r com-read-kill-filter-list))

(defvar kill-filter-list-pathname nil
   "Default file from which to read kill filter list")

(defcom com-read-kill-filter-list
	      "Read kill-filter-list from a file"
	      ()
  (let ((file (read-defaulted-pathname
		 "File from which to read kill filter list:"
		 (or kill-filter-list-pathname
		     (merge-pathnames "news.kill-filter-list"
				      *nntp-profile-pathname*)))))
    (declare (special kill-filter-list-pathname))
    (setf kill-filter-list-pathname file)
    (with-open-file (in file :direction :input)
      (setf kill-filter-list (read in)))
    dis-none))

(defcom com-write-kill-filter-list
	      "Write kill-filter-list to a file"
	      ()
  (let ((file (read-defaulted-pathname
		 "File to which to write kill filter list:"
		 (or kill-filter-list-pathname
		     (merge-pathnames "news.kill-filter-list"
				      *nntp-profile-pathname*))))
	(*print-circle* nil))
    (declare (special kill-filter-list-pathname
		      *print-circle*))
    (setf kill-filter-list-pathname file)
    (with-open-file (out file :direction :output)
      (write kill-filter-list :stream out))
    dis-none))

(defcom com-add-kill-filter-list-entry
	"Declare messages of this type to be ignorable"
	()
  (in-mail-context (:require-message t)
    (add-kill-filter-list-entry *msg*)
    dis-none))

(defcom com-add-person-to-kill-filter-list
	"Declare messages from this person to be ignorable"
	()
  (declare (special *msg*))
  (in-mail-context (:require-message t)
    (add-kill-filter-list-entry *msg* :ignore-person t)
    dis-none))

(defcom com-add-subject-to-kill-filter-list
	"Declare messages about this subject to be ignorable"
	()
  (in-mail-context (:require-message t)
    (add-kill-filter-list-entry *msg* :ignore-subject t)
    dis-none))

(defun add-kill-filter-list-entry (msg &key ignore-subject ignore-person)
   (let ((msg-keys (cons t (send msg :keywords)))
	 (msg-subject (string-remove-fonts (send msg :name)))
	 (msg-person (string-remove-fonts (getf (send msg :plist) :summary-from))))
     (do ((subject msg-subject
		   (string-trim '(#\space #\tab #\:) (subseq subject 3))))
	 ((not (string-equal "re:" subject :end2 3))
	  (setf msg-subject subject)))
     (when (and (cdr msg-keys)
		*numeric-arg-p*)
       (setf msg-keys (w::multiple-menu-choose msg-keys)))
     (when (member t msg-keys)
       (setf msg-keys t))
     (when msg-keys
       (unless (or ignore-subject ignore-person)
	 (setf ignore-subject (w:mouse-y-or-n-p (format nil "Ignore messages about ~a?"
							msg-subject))
	       ignore-person  (w:mouse-y-or-n-p (format nil "Ignore messages from ~a?"
							msg-person))))
       (unless ignore-subject (setf msg-subject nil))
       (unless ignore-person  (setf msg-person nil))
       (push (list msg-keys msg-subject msg-person) kill-filter-list))))
