;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:RECORD -*-

;;; File "MOUSE-RECORD"
;;; Defines a facility for transcribing and playing back mouse and keyboard events.
;;; This bashes some TI code, so portability is severely limited...
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 15 Jul 88  Jamie Zawinski    Created.
;;; 21 Jul 88  Jamie Zawinski    Added file interface.
;;; 22 Jul 88  Jamie Zawinski    Made the saved original functions be in function cells instead of value cells.
;;;                               That way we can make them be inline, and save that extra funcall.
;;;                              Added *LOCK-OUT*.
;;;

(in-package "RECORD" :nicknames '("REC"))


;;; Synopsis:
;;; =========
;;;
;;;   START-RECORDING (name)
;;;    From the first moment at which all keys are up after this is called, all further mouse and keyboard actions
;;;    will be remembered.  This event record will be called NAME, which should be a keyword.
;;;    The recording will terminate when you hit the unlabeled key in the middle of the arrow cluster.
;;;
;;;   PLAYBACK (name)
;;;    Plays back a recording make with START-RECORDING.  All mouse and keyboard action will be exactly duplicated -
;;;    the system software cannot tell the difference between an actual keystroke/mouse-click and one simulated by
;;;    this function.  The delay between output events is exactly the same as it was when they were input, so you get
;;;    to see how slowly you really type.
;;;    The playback will terminate when all events are output, or when the user aborts it by hitting the unlabeled key.
;;;
;;;
;;; The above functions return immediately - the actual recording and playback take place in the guts of the keyboard
;;; and mouse processes.
;;;
;;;
;;;   WRITE-RECORD-FILE (name file)
;;;    Dumps the event record called NAME to a file in a binary format.  The name of the file defaults to NAME, and its
;;;    type defaults to "EVENTS".
;;;
;;;   READ-RECORD-FILE (file)
;;;    Reads a file produced with WRITE-RECORD-FILE.  FILE's type defaults to "EVENTS".
;;;
;;; 
;;; Example of a typical session:
;;; =============================
;;;
;;;   Type SYSTEM L.
;;;   Call #'REC:START-RECORDING with the name of the action you are about to record.
;;;   Do your thing.
;;;   Hit the unlabeled key in the middle of the arrow cluster.
;;;   Type SYSTEM L, and call #'PLAYBACK to test the recording.
;;;   Call #'REC:WRITE-RECORD-FILE to save it.
;;;


(export '(start-recording stop-recording playback write-record-file read-record-file *lock-out*
	  install-recording-code uninstall-recording-code))


;;;  Defining a Queue structure.


(defstruct (queue (:print-function %print-queue)
		  (:constructor make-queue ()))
  (list () :type list)   ; This is the items in the queue, earliest first.
  (last () :type list)   ; This is a pointer to the last cons of the LIST, so we can add to the end of LIST very quickly.
  )

(defun %print-queue (struct stream depth)
  (declare (ignore depth))
  (if (queue-empty-p struct)
      (format stream "#<QUEUE empty ~S>" (sys:%pointer struct))
      (format stream "#<QUEUE ~D items ~D>" (length (queue-list struct)) (sys:%pointer struct))))

(defun queue-empty-p (queue)
  "T if the QUEUE structure has no entries."
  (null (queue-list queue)))

(defun push-queue (item queue)
  "Place ITEM in the QUEUE structure.  These are first-in last-out."
  (let* ((new-last (cons item nil)))
    ;; Can't RPLACD on NIL, so we must special-case an empty queue.
    (cond ((queue-last queue)
	   (rplacd (queue-last queue) new-last)
	   (setf (queue-last queue) new-last))
	  (t (setf (queue-last queue) new-last)
	     (setf (queue-list queue) (queue-last queue)))))
  item)

(defun pop-queue (queue)
  "Remove and return the oldest item in the queue, or NIL if there are no items.
  The second value is whether the queue was empty."
  (declare (values item eof-p))
  (let* ((item (pop (queue-list queue)))
	 (eof nil))
    (when (null (queue-list queue))
      (setf (queue-last queue) nil)
      (setq eof t))
    (values item eof)))

(defun peek-queue (queue)
  "Returns two values: the oldest element in the queue, and whether the queue is empty."
  (declare (values item eof-p))
  (values (car (queue-list queue))
	  (null (queue-list queue))))

(defun clear-queue (queue)
  "Remove all of the elements in the queue."
  (setf (queue-list queue) nil
	(queue-last queue) nil)
  queue)


;;; Event structures.


(defstruct (event (:constructor make-event (delay)))
  (delay  0   :type fixnum)
  )

(defstruct (kbd-event (:constructor make-kbd-event (delay hardware-code))
			   (:include event))
  (hardware-code 0 :type fixnum)
  )

(defstruct (mouse-event (:constructor make-mouse-event (delay buttons x y))
			     (:include event))
  (buttons 0 :type fixnum)
  (x       0 :type fixnum)
  (y       0 :type fixnum)
  )


;;; The event queue.

(defvar *event-queue* (make-queue) "The QUEUE structure of events.")

(eval-when (load eval compile)

(proclaim '(inline any-events push-event pop-kbd-event pop-mouse-event peek-kbd-event peek-mouse-event last-event))

(defun any-events ()
  (not (queue-empty-p *event-queue*)))

(defun push-event (event)
  (push-queue event *event-queue*))

(defun pop-kbd-event ()
  "Pops an event from the event queue if the next available event is a keyboard event."
  (let* ((event (peek-queue *event-queue*)))
    (when (and event
	       (kbd-event-p event))
      (pop-queue *event-queue*))))

(defun pop-mouse-event ()
  "Pops an event from the event queue if the next available event is a mouse event."
  (let* ((event (peek-queue *event-queue*)))
    (when (and event
	       (mouse-event-p event))
      (pop-queue *event-queue*))))

(defun peek-kbd-event ()
  "Returns the next available event in the event queue, if it is a keyboard event."
  (let* ((event (peek-queue *event-queue*)))
    (when (and event
	       (kbd-event-p event))
      event)))

(defun peek-mouse-event ()
  "Returns the next available event in the event queue, if it is a mouse event."
  (let* ((event (peek-queue *event-queue*)))
    (when (and event
	       (mouse-event-p event))
      event)))

(defun last-event ()
  "Returns the last (most recently written) event in the queue."
  (car (queue-last *event-queue*)))


  ) ;closes eval-when


(defun describe-events (&optional events)
  "Describe the events in the queue."
  (let* ((time 0))
    (dolist (e (or events (queue-list *event-queue*)))
      (let* ((delay (event-delay e))
	     (ftime (float (/ (incf time delay) 60))))
	(cond ((kbd-event-p e)
	       (let* ((n (kbd-event-hardware-code e)))
		 (format t "~&~O~10t ~A~40t~D~55t~D~%"
			 n
			 (case n
			   (#o400000 "generic down")
			   (#o400200 "generic up")
			   (t (let* ((ch (sys:ti-lookup (logand #o177 n)))
				     (down-p (logbitp 7 n)))
				(if down-p
				    (format nil "~S down" ch)
				    (format nil "~S up" ch)))))
			 ftime delay)))
	      ((mouse-event-p e)
	       (format t "~&~3,'0b at ~d,~d~40t~D~55t~D~%"
		       (mouse-event-buttons e)
		       (mouse-event-x e)
		       (mouse-event-y e)
		       ftime delay))
	      (t (format t "~&DON'T KNOW WHAT THIS IS!! ~S~%" e))
	      ))))
  nil)


;;; installation

(eval-when (load eval compile)

(defmacro copy-function (new old &optional doc)
  "Make the symbol NEW have the same function definition as OLD.
  NEW will be proclaimed inline.
  NEW is not redefined if it already has a definition."
  `(unless (fboundp ',new)
     (proclaim '(inline ,new))
     (setf (symbol-function ',new) (symbol-function ',old)
	   (documentation ',new 'function) ',doc)
     ',new))

(copy-function original-kbd-hardware-char-available tv:kbd-hardware-char-available
	       "The original version of this function, before we redefined it.")
(copy-function original-kbd-get-hardware-char tv:kbd-get-hardware-char
	       "The original version of this function, before we redefined it.")
(copy-function original-mouse-buttons tv:mouse-buttons
	       "The original version of this function, before we redefined it.")

 ) ; closes eval-when


(defun install-recording-code ()
  "Install the record/playback code into the guts of the keyboard and mouse processes."
  (setf #'tv:kbd-hardware-char-available 'record-patch-kbd-hardware-char-available)
  (setf #'tv:kbd-get-hardware-char 'record-patch-kbd-get-hardware-char)
  (setf #'tv:mouse-buttons 'record-patch-mouse-buttons)
  t)


(defun uninstall-recording-code ()
  "Remove the record/playback code from the guts of the keyboard and mouse processes.  This is for emergencies..."
  (setf #'tv:kbd-hardware-char-available #'original-kbd-hardware-char-available)
  (setf #'tv:kbd-get-hardware-char #'original-kbd-get-hardware-char)
  (setf #'tv:mouse-buttons #'original-mouse-buttons)
  nil)


(defun all-keys-up-p ()
  "T if no keyboard keys or mouse buttons are depressed, NIL otherwise."
  (and (zerop tv:mouse-last-buttons)
       (every #'zerop (the bit-vector si::kbd-key-state-array))))

(defun wait-until-all-keys-up ()
  "This function does not return until there are no depressed keyboard keys and mouse buttons."
  (process-wait "Keys Up" 'all-keys-up-p))


;;; keeping track of the mode.

(defvar *record-playback-mode* nil "One of NIL, :RECORD, or :PLAYBACK.")

(defmacro recording ()
  "T if in record mode."
  `(eq :RECORD *record-playback-mode*))

(defmacro playing ()
  "T if in playback mode."
  `(eq :PLAYBACK *record-playback-mode*))

(defmacro record-mode ()
  "Make the current mode be :RECORD."
  `(setq *record-playback-mode* :record))

(defmacro playback-mode ()
  "Make the current mode be :PLAYBACK."
  `(setq *record-playback-mode* :playback))

(defmacro no-mode ()
  "Make the current mode be NIL."
  `(setq *record-playback-mode* nil))


;;; inits for playing and recording.

(defvar *last-event-dump-time* nil "The 60th-second time at which the last event was dumped.")
(defvar *last-event-read-time* nil "The 60th-second time at which the last event was recorded.")

(defvar *last-written-mouse-event* nil "The last mouse event we output.")
(defvar *recordings* (make-hash-table) "Hash table of recording names and their recordings.")

(defvar *lock-out* t
  "If this is T, then when playing back, the real keyboard and mouse are disabled.
 If NIL, then the playback and the usr may interfere with each other.")

(defvar *bugout-key-pressed* nil
  "This is set to T when the unlabeled key in the middle of the arrow cluster goes down.
  When this goes T, as soon as all keys are up, the recording is stopped.")

(defun playback (&optional name)
  "This function will cause the keyboard process to begin playing back the current event history."
  (cond ((recording) (format t "~&Error - can't playback during recording.~%"))
	((playing) (format t "~&Error - can't playback recursively.~%"))
	(t
	 (format t "~&;;; Starting playback.")
	 (wait-until-all-keys-up)
	 (setq *last-event-dump-time* nil)
	 (setq *last-written-mouse-event* nil)
	 (setq *bugout-key-pressed* nil)
	 (when name
	   (setf (queue-list *event-queue*) (gethash name *recordings*)))
	 (playback-mode)
	 t)))

(defun stop-playing ()
  "This function stops the keyboard process from playing back events.  This gets called automatically when
  the playback is over - you shouldn't need to call it yourself."
  (when (playing)
    (no-mode)
    (setq *last-event-dump-time* nil)
    (setq *last-written-mouse-event* nil)
    (clear-queue *event-queue*) ; in case this is called by hand.
    (if *bugout-key-pressed*
	(tv:notify nil "Playback aborted.")
	(tv:notify nil "Playback finished."))
    nil))

(defvar *current-recording-name* nil "The name of the recording we are making.")

(defun start-recording (name)
  "Tell the keyboard process to start remembering the keys typed."
  (cond ((playing) (format t "~&;;; Error - can't record while playing.~%"))
	((recording) (format t "~&;;; Error - already recording.~%"))
	(t
	 (format t "~&;;; Starting to record.")
	 (wait-until-all-keys-up)
	 (clear-queue *event-queue*)
	 (setq *last-event-read-time* nil)
	 (setq *current-recording-name* name)
	 (setq *bugout-key-pressed* nil)
	 (record-mode)
	 (tv:mouse-buttons t)  ; get an initial mouse position recorded.
	 T)))

(defun stop-recording (&optional notify)
  "Tell the keyboard process to stop remembering the keys typed."
  (cond ((playing) ; ## (format t "~&Error - we are not recording, can't stop.")
	           nil)
	((recording)
	 (if notify
	     (tv:notify nil "Stopping recording of ~A." *current-recording-name*)
	     (format t "~&;;; Stopping recording of ~A." *current-recording-name*))
	 (wait-until-all-keys-up)
	 (setq *last-event-read-time* nil)
	 (setq *bugout-key-pressed* nil)
	 (no-mode)
	 (save-current-record)
	 nil)
	(t ; ## (format t "~&;;; Error - not recording.~%")
	   nil)))


(defun save-current-record ()
  (when *current-recording-name*
    (setf (gethash *current-recording-name* *recordings*)
	  (queue-list *event-queue*))
    t))


(defun ok-to-dump (event &optional set)
  "T if it is time to dump the event represented by the EVENT.
  SET is NIL if this is called from a peeking function, and is T if we are really going to write the event.
  When it is T, some variables are updated so that we know we are processing the next event."
  (declare (optimize (speed 3) (safety 0)))
  (when event
    (let* ((delay (kbd-event-delay (the event event)))
	   (now (time:time))
	   (time-since-last-play (when *last-event-dump-time*
				   (time:time-difference now *last-event-dump-time*)))
	   (ok-to-dump (or (null time-since-last-play)
			   (<= delay time-since-last-play))))
      (when (and ok-to-dump set)
	(setq *last-event-dump-time* now))
      ok-to-dump)))



;;; recording and playing the keyboard.

(eval-when (load eval compile)

(proclaim '(inline maybe-bugout))

(defun maybe-bugout (raw-code)
  (when (= (logand #o177 raw-code) SYS:SCAN-CODE-HOME)
    (setq *bugout-key-pressed* t))
  (when (and *bugout-key-pressed* (all-keys-up-p))
    (cond ((recording) (stop-recording t))
	  ((playing) (stop-playing)))))

 ) ;eval-when


(defun playback-kbd-hardware-char-available ()
  "When we are in playback mode, this says whether an enqueued character should now be processed.
  This is not just whether there is a character in the queue - it also takes into account the recorded
  delay between characters."
  (when (playing)
    (maybe-play-mouse) ; hook for the mouse playback.
    (if (any-events)
	(ok-to-dump (peek-kbd-event))
	(progn (stop-playing) nil))))


(defun playback-kbd-get-hardware-char ()
  "When we are in playback mode, this returns a new character off of the queue, or NIL if it is not appropriate.
  Like PLAYBACK-KBD-HARDWARE-CHAR-AVAILABLE, this takes into account the recorded delay between characters."
  (when (playing)
    (maybe-play-mouse) ; hook for the mouse playback.
    (if (any-events)
	(let* ((event (pop-kbd-event)))
	  (when (and event (ok-to-dump event t))
	    (kbd-event-hardware-code event)))
	(progn (stop-playing) nil))))


(defun record-next-hardware-char (hardware-char)
  "When in record mode, push the hardware char onto the event history."
  (when (recording)
    (maybe-bugout hardware-char)
    (let* ((now (time:time))
	   (time-since-last-recording (if *last-event-read-time*
					  (time:time-difference now *last-event-read-time*)
					  0))
	   (event (make-kbd-event time-since-last-recording hardware-char)))
      (push-event event)
      (setq *last-event-read-time* now)))
  hardware-char)


(defun record-patch-kbd-get-hardware-char ()
  "Intended as a replacement for TV:KBD-GET-HARDWARE-CHAR."
  (tv:without-interrupts
    (let* ((SYS:DEFAULT-CONS-AREA SYS:WORKING-STORAGE-AREA)) ; We create permanent data under here, so get into a good area!
      (cond ((playing)
	     (let* ((typed-char (original-kbd-get-hardware-char)))
	       (when typed-char (maybe-bugout typed-char))
	       (or (playback-kbd-get-hardware-char)
		   (if *lock-out*
		       nil
		       typed-char))))
	    ((recording)
	     (record-next-hardware-char (original-kbd-get-hardware-char)))
	    (t
	     (original-kbd-get-hardware-char))))))


(defun record-patch-kbd-hardware-char-available ()
  "Intended as a replacement for TV:KBD-HARDWARE-CHAR-AVAILABLE."
  (tv:without-interrupts
    (cond ((playing)
	   (or (playback-kbd-hardware-char-available)
	       (if *lock-out*
		   (let* ((typed-char (original-kbd-get-hardware-char)))
		     (when typed-char (maybe-bugout typed-char))
		     nil)
		   (original-kbd-hardware-char-available))))
	  (t
	   (original-kbd-hardware-char-available)))))


;;; ## nothing uses this - for debugging.
(defmacro without-kbd-process (&body body)
  "Execute BODY with the keyboard process inhibited.  When BODY terminates, the keyboard process is re-enabled.
  If BODY gets an error, the keyboard process is re-enabled before the error handler is entered.
  WARNING!!  If the BODY goes into an endless loop, you are HOSED."
  `(progn
     (send tv:kbd-process :arrest-reason :kbd-inhibit)
     (unwind-protect (condition-bind ((t #'(lambda (&rest ignore)
					     (send tv:kbd-process :revoke-arrest-reason :kbd-inhibit)
					     nil)))
		       ,@body)
       (send tv:kbd-process :revoke-arrest-reason :kbd-inhibit))))



;;; Recording and playing the mouse

(defvar *never-record-mouse* nil "For debugging - if T, no mouse events will be recorded.")


;;; This is called from the keyboard process.
;;; The keyboard process constantly spins about, polling the keyboard.
;;; The mouse process doesn't - it is idle until somebody wakes it.
;;; So we have a hook in the keyboard process that will wake the mouse process when necessary.
;;;
(defun maybe-play-mouse ()
  "If we are playing back, and the next event is a mouse event, wake the mouse process."
  (when (playing)
    (let* ((event (peek-mouse-event)))
      (when event (setq tv:mouse-wakeup t)))))



(defun record-mouse-buttons (buttons microsecond-time x y)
  "When in record mode, push the mouse info onto the event history."
  (declare (ignore microsecond-time)) ; ## what to do... see comment about time below.
  (when (recording)
    (let* ((last-event (car (queue-last *event-queue*)))
	   (same-p (and last-event
			(mouse-event-p last-event)
			(= buttons (mouse-event-buttons last-event))
			(= x (mouse-event-x last-event))
			(= y (mouse-event-y last-event)))))
      (unless same-p  ; Don't push an event if it is the same as the last event we pushed.
	(let* ((now (time:time))
	       (time-since-last-recording (if *last-event-read-time*
					      (time:time-difference now *last-event-read-time*)
					      0))
	       (event (make-mouse-event time-since-last-recording buttons x y)))
	  (setq *last-event-read-time* now)
	  (push-event event)))))
  nil)


(defun playback-mouse-buttons (peek)
  "If we are in play mode, make it such that the next call to TV:MOUSE-BUTTONS will return the values in the
  topmost event on the queue."
  (declare (ignore peek))
  (when (playing)
    (if (any-events)
	(let* ((event (peek-mouse-event)))
	  (cond ((and event (ok-to-dump event t))
		 (pop-mouse-event))
		((or event *lock-out*)
		 (setq event *last-written-mouse-event*)))
	  (when event
	    (let* ((buttons (mouse-event-buttons event))
		   (x (mouse-event-x event))
		   (y (mouse-event-y event)))
	      (mouse-buffer-push buttons x y))  ; ## letting time default to "now" for lack of a better idea...
	    (setq *last-written-mouse-event* event))
	  nil)
	(progn (stop-playing) nil))))

;;;
;;; ## Here is a Winning Feature.
;;;
;;; ## Mouse events come in tagged with a time from the microsecond clock.
;;; ## Keyboard events come in tagged with a time from the sixtieth-second clock.
;;;
;;; ## The microsecond clock and the sixtieth-second clock do not have the same zero-point or rollover-point.
;;; ## Because of this, it is not possible to convert one to the other.
;;; ## One can make a correspondence between the time and jtime now, but that doesn't say anything about what is was
;;; ## a half a second ago - one of the clocks may have turned over while the other didn't.
;;;

(defun mouse-buffer-push (buttons x y &optional (time (time:fixnum-microsecond-time)))
  "BUTTONS is a fixnum with a bit set for each button depressed.  TIME is in microseconds.
   Warp the mouse, and make it such that the next time TV:MOUSE-BUTTONS is called, BUTTONS will be returned."
  (tv:without-interrupts   ;; Run without-interrupts so the mouse process doesn't interfere.
    (tv:mouse-warp x y)
    (setq tv:mouse-buttons-in-progress buttons
	  tv:mouse-last-buttons-time time
	  tv:mouse-last-buttons-x x
	  tv:mouse-last-buttons-y y
	  )
    ;; Tell the mouse-input handler that there is some input ready.
    (setq tv:mouse-wakeup t)
    ))


(defun record-patch-mouse-buttons (&optional (peek (not (eq tv:current-process tv:mouse-process))))
  "Intended as a replacement for TV:MOUSE-BUTTONS."
  (declare (values mouse-last-buttons mouse-last-buttons-time mouse-x mouse-y))
  (when (playing) (playback-mouse-buttons peek))
  (multiple-value-bind (mouse-last-buttons mouse-last-buttons-time mouse-x mouse-y)
		       (original-mouse-buttons peek)
    (when (and (recording) (not *never-record-mouse*))
      (record-mouse-buttons mouse-last-buttons mouse-last-buttons-time mouse-x mouse-y))
    (values mouse-last-buttons mouse-last-buttons-time mouse-x mouse-y)))


;;; Saving recordings to a file.

(defun write-fixnum (n stream)
  "Write the number on the stream in as many 8-bit bytes as a fixnum occupies."
  (let* ((bytes-in-a-fixnum #.(ceiling (integer-length MOST-POSITIVE-FIXNUM) 8)))  ; this is 3 - fixnums are 24 bits.
    (dotimes (i bytes-in-a-fixnum)
      (let* ((byte-field (ldb (byte 8 (* 8 (- bytes-in-a-fixnum (1+ i))))
			      n)))
	(write-byte byte-field stream))))
  n)

(defun read-fixnum (stream)
  "Read and return a number from the stream in as many 8-bit bytes as a fixnum occupies."
  (let* ((n 0)
	 (bytes-in-a-fixnum #.(ceiling (integer-length MOST-POSITIVE-FIXNUM) 8)))  ; this is 3 - fixnums are 24 bits.
    (dotimes (i bytes-in-a-fixnum)
      (setq n (dpb (read-byte stream)
		   (byte 8 (* 8 (- bytes-in-a-fixnum (1+ i))))
		   n)))
    n))

(defun write-word (n stream)
  "Writes the number on the stream as 16 bits."
  (write-byte (ldb (byte 8 8) n) stream)
  (write-byte (ldb (byte 8 0) n) stream)
  n)

(defun read-word (stream)
  "Writes the number on the stream as 16 bits."
  (+ (* 256 (read-byte stream))
     (read-byte stream)))


(defun write-event (event stream)
  (etypecase event
    (KBD-EVENT
     (write-byte (char-code #\K) stream)
     (write-fixnum (event-delay event) stream)
     (write-fixnum (kbd-event-hardware-code event) stream)
     )
    (MOUSE-EVENT
     (write-byte (char-code #\M) stream)
     (write-fixnum (event-delay event) stream)
     (write-byte (mouse-event-buttons event) stream)  ; always 3 bits, write 8.
     (write-word (mouse-event-x event) stream)  ; no screen is wider than 65k bits...
     (write-word (mouse-event-y event) stream)
     ))
  event)

(defun read-event (stream)
  (let* ((code (read-byte stream)))
    (ecase code
      (#.(char-code #\K)
	 (let* ((delay (read-fixnum stream))
		(hardware-code (read-fixnum stream)))
	   (make-kbd-event delay hardware-code)))
      (#.(char-code #\M)
	 (let* ((delay (read-fixnum stream))
		(buttons (read-byte stream))
		(x (read-word stream))
		(y (read-word stream)))
	   (make-mouse-event delay buttons x y))))))


(defun write-record-file (record-name file)
  "Dump the event record called RECORD-NAME to FILE.  
  FILE's name defaults to the RECORD-NAME, and it's type defaults to EVENTS."
  (setq file (merge-pathnames file
			      (merge-pathnames (make-pathname :name (string record-name)
							      :type "EVENTS"))))
  (let* ((events (gethash record-name *recordings*)))
    (unless events (error "~S is not a record name." record-name))
    (with-open-file (stream file :direction :output :characters nil :byte-size 8)
      (princ "Event Record File." stream)
      (terpri stream)
      (let* ((*print-case* :upcase))
	(princ record-name stream))
      (terpri stream)
      (dolist (e events)
	(write-event e stream))
      (truename stream))))


(defun read-record-file (file)
  "Read an event record from the file.  This file should have been produced with WRITE-RECORD-FILE.
  Returns the name of the event record that was read.  If there is already an event record of that name defined,
  it is overridden.
  FILE's type defaults to EVENTS."
  (setq file (merge-pathnames file (merge-pathnames (make-pathname :type "EVENTS"))))
  
  (with-open-file (stream file :direction :input :characters nil :byte-size 8)
    (loop
      (when (= #.(char-code #\Newline) (read-byte stream))
	(return)))
    (let* ((string (make-array 255 :element-type 'string-char :fill-pointer 0))
	   name
	   (events '()))
      (loop
	(let* ((ch (code-char (read-byte stream))))
	  (if (char= #\Newline ch)
	      (return)
	      (vector-push ch string))))
      (setq name (intern string "KEYWORD"))
      (do ()
	  ((not (listen stream)))
	(push (read-event stream) events))
      (setq events (nreverse events))
      (setf (gethash name *recordings*) events)
      name)))


(install-recording-code)
