;;; -*- Mode:Common-Lisp; Package:ZWEI; Base:10; Patch-File:T -*-

;;; Code to put a Percent Through File notation in the Zmacs modeline, just like every other Emacs has...


(defvar *file-percentage-string* "<Top> " "A string showing the percentage in the file of the point.")

(defun find-every-window-of-flavor (flavor)
  "Returns a list of all previously or currently selected windows whose flavor includes FLAVOR."
  (let* ((result nil))
    ;; Flet this since we use it twice.
    (flet ((push-window-if-ok (w)
	     (when w
	       (cond ((and (typep w flavor) (send w :name-for-selection)
			   (push w result)))
		     (t (let ((wss (send w :selection-substitute)))
			  (when (and wss (typep wss flavor) (send wss :name-for-selection))
			    (push wss result))))))))
      ;; Iterate over all windows.
      (dolist (w (send tv:default-screen :inferiors))
	(push-window-if-ok w))
      (nreverse result))))



(defvar *percent-tally-tick* 0 "The last time the percent marker in the modeline was updated.")

(setf (get 'update-percent-string 'command-hook-priority) 9)

(defun update-percent-string (&optional ignore)
  "Call UPDATE-PERCENT-STRING-INTERNAL if more than one second has passed since the last time we have called it.
  This function is called every time some keyboard event occurs in Zmacs, which is why we fake a scheduler like this."
  (let* ((time (get-universal-time)))
    (when (< *percent-tally-tick* time) ; Don't do anything unless the clock has ticked (1 second).
      (ignore-errors
	;; ZMACS Redisplay really sucks.  Sometimes the point and the interval-first-bp are not in the same buffer.
	;; So, COUNT-LINES will signal an error.  So we ignore errors and try again a second from now.
	(update-percent-string-internal)
	(setq *percent-tally-tick* time))
      t)))

(defun update-percent-string-internal (&optional ignore)
  " Updates the variable *FILE-PERCENTAGE-STRING* to be a string saying what percentage the point is at in the buffer.
 This is not a very accurate number, but it is usually close."
  (cond ((bp-= (point) (interval-first-bp *interval*))
	 (setq *file-percentage-string* "<Top> "))
	((bp-= (point) (interval-last-bp *interval*))
	 (setq *file-percentage-string* "<Bottom> "))
	(t
	 (let* ((*print-base* 10)  ; ZetaLisp buffers are in base 8...
		(point (point))
		(first-bp (interval-first-bp *interval*))
		(last-bp (interval-last-bp *interval*))
		(point-lines (1- (count-lines first-bp point)))        ; Total lines at point.
		(interval-lines (1- (count-lines first-bp last-bp)))   ; Total lines in buffer. (sometimes 0)
		(this-line-ratio (if (zerop interval-lines) 0
				     (/ point-lines interval-lines)))      ; The ratio through the buffer that this line is.
		(next-line-ratio (if (zerop interval-lines) 0
				     (/ (1+ point-lines) interval-lines))) ; The ratio through the buffer that next line is.
		(this-char-ratio (if (zerop (length (bp-line point))) 0    ; The ratio through this line that the cursor is.
				     (/ (bp-index point) (length (bp-line point)))))
		;;
		;; If the beginning of this line is N% through, and the beginning of the next line is M% through, and
		;; the cursor is L% through the line, then the cursor is N+(L*(M-N))% through the buffer.
		;;
		;; However, the values for M and N here are not very accurate, since we have no efficient way of knowing
		;; character counts, only line counts.  We can only hope that the average length of the lines before the
		;; point is the same as the average length of lines after the point.
		;;
		(percent  (truncate (* 100 (+ this-line-ratio
					      (* this-char-ratio (- next-line-ratio this-line-ratio))))))
		(new-string (concatenate 'simple-string "<" (princ-to-string percent) "%> ")))
	   ;;
	   ;; ZMACS decides whether to update the modeline by EQness, so don't change it unless we have to.
	   (unless (string= new-string *file-percentage-string*)
	     (setq *file-percentage-string* new-string))))))


(defun set-up-percentage-tally ()
  "Make the modeline of all existant ZMACSes display the point's position in the buffer in percent."
  (turn-off-percentage-tally)
  ;;
  ;; Set it in every existant editor.
  (dolist (window (find-every-window-of-flavor 'zwei::zmacs-frame))
    (let* ((sg (send (send window :process) :stack-group)))
      (eh::set-in-stack-group '*mode-line-list* sg
			      (cons '*file-percentage-string* (eh::symeval-in-stack-group '*mode-line-list* sg)))
      (eh::set-in-stack-group '*post-command-hook* sg
			      (cons 'update-percent-string (eh::symeval-in-stack-group '*post-command-hook* sg)))
      ))
  ;;
  ;; Set it globally.
  (when (boundp-globally '*mode-line-list*)
    (setq-globally *mode-line-list*
		   (cons '*file-percentage-string* (symeval-globally '*mode-line-list*))))
  (when (boundp-globally '*post-command-hook*)
    (setq-globally *post-command-hook*
		   (cons 'update-percent-string (symeval-globally '*post-command-hook*))))
  ;;
  ;; Make sure newly created editors get it.
  (advise INITIALIZE-TOP-LEVEL-EDITOR :after :turn-on-mail-modeline nil
    (unless (member '*file-percentage-string* *mode-line-list*)
      (setq *mode-line-list* (cons '*file-percentage-string* *mode-line-list*))
      (command-hook 'update-percent-string '*post-command-hook*)))
  nil)


(defun turn-off-percentage-tally ()
  ;;
  ;; Remove the variable from every existant editor.
  (dolist (window (find-every-window-of-flavor 'zwei::zmacs-frame))
    (let* ((sg (send (send window :process) :stack-group)))
      (eh::set-in-stack-group '*mode-line-list* sg
			      (delete '*file-percentage-string*
				      (eh::symeval-in-stack-group '*mode-line-list* sg)))
      (eh::set-in-stack-group '*post-command-hook* sg
			      (delete 'update-percent-string
				      (eh::symeval-in-stack-group '*post-command-hook* sg)))))
  ;;
  ;; Remove it at top level.
  (when (boundp-globally '*mode-line-list*)
    (setq-globally *mode-line-list* (delete '*mail-p-modeline-string* (symeval-globally '*mode-line-list*))))
  nil)
