;;; -*- Mode:Lisp; Package:(MAZE); Syntax:Common-Lisp -*-

;;; File "MAZE"
;;; Generates random mazes.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 21 Dec 88  Jamie Zawinski    Created.
;;;


(in-package "MAZE")
(export '(generate display *branch-chance*) "MAZE")


;;; GENERATE constructs a maze with exactly one path through it.  The start and end points do not necessarily
;;; lie on the borders of the maze.  If the WINDOW argument to GENERATE is non-NIL, then the maze will be drawn
;;; as it is being generated.
;;;
;;; DISPLAY will draw the previously-calculated maze.
;;;
;;; The variable *BRANCH-CHANCE* controls the complexity of the maze; it is an integer from 0 to 100.
;;; The size of the maze is determined by the dimensions of the array *SEGMENTS-BITMAP*.
;;;
;;; Try this: 
;;;
;;; (PROGN (SEND TV:SELECTED-WINDOW :CLEAR-SCREEN) (GENERATE TV:SELECTED-WINDOW 5 5))
;;;
;;;
;;; At the end of this file is some incomplete code for a MazeWar-like view of the maze.


;;; The algorithm:
;;;
;;; Start with a point somewhere on the grid.
;;; For each point ("live head"):
;;;   If all neighbors are occupied, remove this point from the list of live heads.
;;;   Otherwise:
;;;    1: pick a random direction from the set of unoccupied neighbor grid cells.
;;;       Mark the current point as having an opening in that direction.
;;;       Modify the current point to point at the picked neighbor, and make the picked neighbor be marked as 
;;;       having an opening back towards where we came from.  
;;;    2: If there are still some unoccupied neighbors, then pick a random number.
;;;       If this number is less than *BRANCH-CHANCE*, then pick a direction from the set of unoccupied neighbors,
;;;       and add a new "live head" for that position.
;;;
;;; The first cell to "die" is considered the "start point", and the last cell to "die" is considered the end point.
;;; This guarentees that the path between them will be the longest path between any two dead-ends in the maze.
;;; But we could assign the start and end points to any two dead-ends in the maze, because there will be exactly one
;;; path between them wherever they are.  This path will pass through the first point we picked, so it's probably not
;;; a good idea for that to always be at the center of the maze.  It turns out that the first point to die is often
;;; near the initial point, at least with *BRANCH-CHANCE* around 12.
;;;

;;;
;;; This could be modified to make a multi-level maze by adding more bits to the SEGMENT type -
;;; there could be 12 directions: NORTH-TOP, NORTH-MID, NORTH-BOT, EAST-TOP, etc.
;;; These could be displayed as thin paths on top of thicker paths.


(defconstant EMPTY #b0000)
(defconstant NORTH #b1000)
(defconstant EAST  #b0100)
(defconstant SOUTH #b0010)
(defconstant WEST  #b0001)

(deftype segment () '(integer 0 #b1111))


(defvar *segments-bitmap* (make-array '(100 100) :element-type 'segment :initial-element EMPTY)
  "The array intowhich we generate.  The bits of the cells represent north, east, south, and west.")

(defvar *branch-chance* 12 "Percent chance for a head to split.")

(defvar *live-heads* '() "A list of conses of x,y which are growth sites.")


(defun legal-directions (x y max-x max-y)
  "Returns a four-bit quantity representing the directions in which it is legal for the head at XY to grow.
 A direction is illegal if the cell in that direction is already allocated (non-zero).
 MAX-X and MAX-Y should be the dimensions of the *SEGMENTS-BITMAP*."
  (declare (optimize (speed 3) (safety 0))
	   (fixnum x y max-x max-y))
  (let* ((result EMPTY))
    (declare (type segment result))
    ;; WEST
    (when (and (plusp x) (zerop (aref *segments-bitmap* (1- x) y)))
      (setq result (logior result WEST)))
    ;; NORTH
    (when (and (plusp y) (zerop (aref *segments-bitmap* x (1- y))))
      (setq result (logior result NORTH)))
    ;; EAST
    (when (and (< x (1- max-x))
	       (zerop (aref *segments-bitmap* (1+ x) y)))
      (setq result (logior result EAST)))
    ;; SOUTH
    (when (and (< y (1- max-y))
	       (zerop (aref *segments-bitmap* x (1+ y))))
      (setq result (logior result SOUTH)))
    result))



(defun pick-direction (x y legal-directions howmany)
  "Randomly pick from the directions intowhich it is legal to move.
  HOWMANY is the LOGCOUNT of LEGAL-DIRECTIONS - it is passed in so we don't have to calculate it here.
  Returns five values:
   WHICH, a number from 0 to HOWMANY, the direction we picked.
   DIRECTION, one of the constants NORTH, SOUTH, EAST, or WEST.
   OPPOSITE-DIRECTION, same as above, but in the other direction.
   NEWX and NEWY, which are the position of the chosen neighbor of XY."
  (declare (values which direction opposite-direction newx newy)
	   (optimize (speed 3) (safety 0))
	   (fixnum x y howmany)
	   (type segment legal-directions))
  (let* ((which (random howmany))
	 (direction 0))
    (declare (fixnum which)
	     (type segment direction))
    (do* ((i 0 (1+ i))
	  (c 0))
	 ((= i 4))
      (declare (fixnum i c))
      (when (logbitp i legal-directions)
	(incf c))
      (when (= (1- c) which)
	(setq direction (ash 1 i))
	(return)))
    (let* ((newx x)
	   (newy y)
	   (opposite-direction 0))
      (ecase direction
	(#.NORTH (decf newy)
		 (setq opposite-direction SOUTH))
	(#.SOUTH (incf newy)
	         (setq opposite-direction NORTH))
	(#.EAST  (incf newx)
	         (setq opposite-direction WEST))
	(#.WEST  (decf newx)
	         (setq opposite-direction EAST)))
      (values which direction opposite-direction newx newy))))


(defun walk-head (head max-x max-y)
  "Given a cons of X and Y into *SEGMENTS-BITMAP*, move the active site represented.
  A new head may be added if the *BRANCH-CHANCE* is met.
  HEAD will be destructively modified to point to the new position.

  Returns two values: whether this head has died, and the head it has spawned, if any."
  (declare (values died-p branched-head)
	   (optimize (speed 3) (safety 0))
	   (cons head)
	   (fixnum max-x max-y)
	   (inline legal-directions))

  (let* ((x (car head))
	 (y (cdr head))
	 (current-value (aref *segments-bitmap* x y))
	 (legal-directions (legal-directions x y max-x max-y))
	 (howmany (logcount legal-directions))
	 (stopping (zerop legal-directions))
	 (branching (and (not stopping) (/= 1 howmany)
			 (< (random 100) *branch-chance*)))
	 (branch-head nil))
    (declare (fixnum x y howmany)
	     (type segment current-value legal-directions))
    
    (cond (stopping
	   (setq *live-heads* (delete head *live-heads* :test #'eq)))
	  
	  (branching
	   (multiple-value-bind (which direction opposite-direction newx newy)
				(pick-direction x y legal-directions howmany)
	     (declare (ignore which)
		      (fixnum newx newy)
		      (type segment direction opposite-direction))
	     
	     (when (and (< newx max-x)
			(< newy max-y))
	       (setf (aref *segments-bitmap* newx newy) opposite-direction)
	       (setq current-value (logior current-value direction))
	       (setf (aref *segments-bitmap* x y) current-value)
	       (setq legal-directions (logand legal-directions (lognot direction)))
	       (decf howmany)
	       (setq branch-head (cons newx newy))
	       (push branch-head *live-heads*)))))
    (unless stopping
      (multiple-value-bind (which direction opposite-direction newx newy)
			   (pick-direction x y legal-directions howmany)
	     (declare (ignore which)
		      (fixnum newx newy)
		      (type segment direction opposite-direction))
	     
	(when (and (< newx max-x)
		   (< newy max-y))
	  (setf (aref *segments-bitmap* newx newy) opposite-direction)
	  (setq current-value (logior current-value direction))
	  (setf (aref *segments-bitmap* x y) current-value)
	  (setf (car head) newx
		(cdr head) newy))))
    (values stopping branch-head)))



(defun generate (&optional window (x-off 0) (y-off 0) (seg-w 10) (seg-h 10))
  "Fill the *SEGMENTS-BITMAP* with a randomly generated maze.
  Returns four values, the starting and ending coordinates of a legal path through the maze."
  (declare (values start-x start-y end-x end-y))
  (let* ((initial-x (random (array-dimension *segments-bitmap* 0)))
	 (initial-y (random (array-dimension *segments-bitmap* 1)))
	 (initial-head (cons initial-x initial-y))
	 (max-x (array-dimension *segments-bitmap* 0))
	 (max-y (array-dimension *segments-bitmap* 1)))
    (setq *live-heads* (list initial-head))
    (dotimes (i max-x)
      (dotimes (j max-y)
	(setf (aref *segments-bitmap* i j) 0)))
    (let* ((start-x nil)
	   (start-y nil)
	   (end-x nil)
	   (end-y nil)
	   (i 0))
      (block MAIN
	(loop
	  (if *live-heads*
	      (dolist (head *live-heads*)
		(incf i)
		(let* ((old-x (car head))
		       (old-y (cdr head)))
		  (let* ((stopping (walk-head head max-x max-y)))
		    (if stopping
			(cond ((null start-x)
			       (setq start-x old-x start-y old-y))
			      ((null *live-heads*)
			       (setq end-x old-x end-y old-y))))
		    (when window (display-cell old-x old-y seg-w seg-h (aref *segments-bitmap* old-x old-y)
					       window x-off y-off)))))
	      (return-from MAIN))))
      (unless end-x (setq end-x initial-x))  ; If *BRANCH-CHANCE* is absurdly low, we may not branch, so
      (unless end-y (setq end-y initial-y))  ; return the initial point as the end point.
      (when window
	(display-cell start-x start-y seg-w seg-h (aref *segments-bitmap* start-x start-y) window x-off y-off t t)
	(display-cell end-x end-y seg-w seg-h (aref *segments-bitmap* end-x end-y) window x-off y-off t t))
      (values start-x start-y end-x end-y)
      )))


(defun display-cell (grid-x grid-y w h value window &optional (start-x 0) (start-y 0) (all-sides t) draw-x)
  "X and Y are the position in the maze grid; W and H are the size all cells are scaled to; VALUE is of type SEGMENT."
  (let* ((x (+ start-x (* w grid-x)))
	 (y (+ start-y (* h grid-y))))
    (cond (nil;(zerop value)
	   (send window :draw-rectangle w h x y)
	   )
	  (t
	   (when (zerop (logand NORTH value))
	     (send window :draw-line x y (+ x w) y))
	   (when (zerop (logand WEST value))
	     (send window :draw-line x y x (+ y h)))
	   (when all-sides
	     (when (zerop (logand SOUTH value))
	       (send window :draw-line x (+ y h) (+ x w) (+ y h)))
	     (when (zerop (logand EAST value))
	       (send window :draw-line (+ x w) y (+ x w) (+ y h))))))
    (when draw-x
      (send window :draw-line x y (+ x w) (+ y h))
      (send window :draw-line (+ x w) y x (+ y h)))
    ))


(defun display (&optional (draw-x 0) (draw-y 0) (seg-w 10) (seg-h 10) (window tv:selected-window)
		(x 0) (y 0) width height)
  (let* ((max-w (array-dimension *segments-bitmap* 0))
	 (max-h (array-dimension *segments-bitmap* 1))
	 (w (or width  (- max-w x)))
	 (h (or height (- max-h y)))
	 (dw (* seg-w w))
	 (dh (* seg-h h)))
    (send window :draw-line draw-x        draw-y        (+ draw-x dw) draw-y)
    (send window :draw-line draw-x        draw-y        (+ draw-x dh) draw-y)
    (send window :draw-line draw-x        (+ draw-y dh) (+ draw-x dw) (+ draw-y dh))
    (send window :draw-line (+ draw-x dw) draw-y        (+ draw-x dw) (+ draw-y dh))
    (dotimes (i w)
      (dotimes (j h)
	(when (and (<= 0 (+ x i) max-w) (<= 0 (+ y j) max-h))
	  (let* ((value (aref *segments-bitmap* (+ x i) (+ y j))))
	    (display-cell i j seg-w seg-h value tv:selected-window draw-x draw-y nil nil)))))))


(defun maze-screenhack (&optional (window tv:selected-window))
  (loop
    (send window :clear-screen)
    (let* ((w (+ 10 (random 20))))
      (generate window 5 5 w w))
    (sleep 3)))

(when (boundp 'tv:*screen-saver-hacks-list*)
  (pushnew 'maze-screenhack tv:*screen-saver-hacks-list*))


;;; rat's eye view.


(defun draw-left-wall (x y w h window open-p nowalls)
  (let* ((perspective 0.2)
	 (x2 (round (+ x (* w perspective))))
	 (y2 (round (+ y (* h perspective))))
	 (y3 (round (+ y (* h (- 1.0 perspective)))))
	 (y4 (+ y h)))
    (unless nowalls
      (send window :draw-line x y x y4))
    (send window :draw-line x2 y2 x2 y3)
    (cond (open-p
	   (send window :draw-line x y2 x2 y2)
	   (send window :draw-line x y3 x2 y3))
	  (t
	   (send window :draw-line x y x2 y2)
	   (send window :draw-line x y4 x2 y3)))))


(defun draw-right-wall (x y w h window open-p nowalls)
  (let* ((perspective 0.2)
	 (x2 (round (+ x (* w (- 1.0 perspective)))))
	 (y2 (round (+ y (* h perspective))))
	 (y3 (round (+ y (* h (- 1.0 perspective)))))
	 (x4 (+ x w))
	 (y4 (+ y h)))
    (unless nowalls
      (send window :draw-line x4 y x4 y4))
    (send window :draw-line x2 y2 x2 y3)
    (cond (open-p
	   (send window :draw-line x4 y2 x2 y2)
	   (send window :draw-line x4 y3 x2 y3))
	  (t
	   (send window :draw-line x4 y4 x2 y3)
	   (send window :draw-line x4 y x2 y2)))))


(defun draw-front-wall (x y w h window)
  (let* ((x1 (round (+ x (* w 0.2))))
	 (y1 (round (+ y (* h 0.2))))
	 (x2 (round (+ x (* w 0.8))))
	 (y2 (round (+ y (* h 0.8)))))
    (send window :draw-line x1 y1 x2 y1)
    (send window :draw-line x1 y2 x2 y2)))


(defun rat-draw-maze-1 (x y width height maze-x maze-y facing window nowalls)
  (if (typep window 'tv:graphics-mixin)
      (send window :draw-rectangle width height x y (send window :erase-aluf))
      (send window :draw-filled-rectangle x y width height (send window :erase-aluf)))
  (let* ((room (aref *segments-bitmap* maze-x maze-y))
	 (north-open (plusp (logand NORTH room)))
	 (south-open (plusp (logand SOUTH room)))
	 (east-open (plusp (logand EAST room)))
	 (west-open (plusp (logand WEST room))))
    (ecase facing
      (#.NORTH
       (draw-left-wall  x y width height window west-open nowalls)
       (draw-right-wall x y width height window east-open nowalls)
       )
      (#.SOUTH
       (draw-left-wall  x y width height window east-open nowalls)
       (draw-right-wall x y width height window west-open nowalls)
       )
      (#.EAST
       (draw-left-wall  x y width height window north-open nowalls)
       (draw-right-wall x y width height window south-open nowalls)
       )
      (#.WEST
       (draw-left-wall  x y width height window south-open nowalls)
       (draw-right-wall x y width height window north-open nowalls)
       ))
    (cond ((plusp (logand facing room))
	   t)
	  (t
	   (draw-front-wall x y width height window)
	   nil))))


(defun rat-draw-maze (x y width height maze-x maze-y facing &optional depth (window tv:selected-window))
  (dotimes (i (or depth MOST-POSITIVE-FIXNUM))
    (unless (rat-draw-maze-1 x y width height maze-x maze-y facing window (= i 0))
      (return))
    (incf x (round (* width 0.2)))
    (incf y (round (* height 0.2)))
    (setq width (round (* width .6)))
    (setq height (round (* height .6)))
    (ecase facing
      (#.NORTH (decf maze-y))
      (#.SOUTH (incf maze-y))
      (#.EAST  (incf maze-x))
      (#.WEST  (decf maze-x))))
  nil)


(defun hand-to-compass (direction facing)
  (ecase direction
    (:LEFT  (ecase facing (#.NORTH WEST) (#.WEST SOUTH) (#.SOUTH EAST) (#.EAST NORTH)))
    (:RIGHT (ecase facing (#.NORTH EAST) (#.WEST NORTH) (#.SOUTH WEST) (#.EAST SOUTH)))
    (:AHEAD facing)
    (:BACK  (ecase facing (#.NORTH SOUTH) (#.WEST EAST) (#.SOUTH NORTH) (#.EAST WEST)))))


(defun test ()
  (let* ((mx 20)
	 (my 20)
	 (facing NORTH))
    (loop
      (send tv:selected-window :clear-screen)
      (rat-draw-maze 430 100 200 200 mx my facing)
      (display 100 100 10 10 tv:selected-window (- mx 10) (- my 10) 20 20)
      (display-cell 10 10 10 10 #b1111 tv:selected-window 100 100 t t)
      (send tv:selected-window :set-cursorpos 510 200)
      (format t "~S" (find facing '(north south east west) :key 'symbol-value))
      (send tv:selected-window :set-cursorpos 0 400)
      (let* ((ch (read-char))
	     (move-p nil))
	(case ch
	  (#\Up-Arrow   (setq move-p t))
	  (#\Left-Arrow (setq facing (hand-to-compass :left facing)))
	  (#\Right-Arrow (setq facing (hand-to-compass :right facing)))
	  (t (beep)))
	(when move-p
	  (cond ((zerop (logand facing (aref *segments-bitmap* mx my)))
		 (beep))
		(t
		 (ecase facing
		   (#.NORTH (decf my))
		   (#.SOUTH (incf my))
		   (#.EAST  (incf mx))
		   (#.WEST  (decf mx)))
		 )))))))

