;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:ZWEI; VSP:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B) -*-

;1;; File "3MAIL-MODELINE*"*
;1;; Written and maintained by Jamie Zawinski.*
;1;;*
;1;; This file adds some utility to the TI mail system.*
;1;;*
;1;; 3MAIL:PROBE-INBOXES* has been redefined with more functionality:*
;1;;  If the variable3 *UNIX-INBOX-PATHNAME** is set, and the file or files are not on the inbox-probe list, then*
;1;;   it/they are added.*
;1;;  The variable3 MAIL:*NOTIFY-ON-MAIL-ARRIVAL** says whether a3 TV:NOTIFY* notification should be sent *
;1;;   when new mail arrives (default is T).*
;1;;  The function now returns two values:  a list of inbox files which have gained mail, and a list of inbox files*
;1;;   which have mail.*
;1;;*
;1;; If you call the function *TURN-ON-MAIL-MODELINE1, then the Zmacs modeline will tell you when you have mail.*
;1;; The variable 3ZWEI:*MAIL-MODELINE-CK-INTERVAL* *is how many seconds there is between each update*
;1;; of the modeline. * 1The default value is 30.  Before this feature, it was not possible to have your inbox probed *
;1;; more often than every five minutes, because the mailer daemon did the probing.*
;1;; The function *TURN-OFF-MAIL-MODELINE1 removes this.*
;1;;*
;1;; This file also fixes a bug in 3MAIL:DEFAULT-FROM-ADDRESS* so that the user can specify a return-address in*
;1;; *MAIL::*MAIL-USER-PERSONAL-NAME*1.  It used to be that5 "My Name" *mapped to5 "My Name <my address>"*, but*
;1;; that5 "My Name <my address>" *mapped to 5"My Address"*, which was silly.*
;1;;*
;1;; ( The mail-abbrev code that used to be here has been moved to its own file, 3MAIL-ABBREVS.LISP*. )*
;1;;*


;1;; 5ChangeLog:**
;1;;*
;1;;   17 Aug 87*	1Jamie Zawinski*	1Created.*
;1;;   14 Sep 87*	1Jamie Zawinski*	1Added the signature string.*
;1;;   25 Sep 87*	1Jamie Zawinski*	1Gee, toggling of Abbrev Mode was broken.  Fixed it.*
;1;;   26 Oct 87*	1Jamie Zawinski*	1Added an *IGNORE-ERRORS1 around the* CKMAIL1 in* MAIL-MODELINE-UPDATE-PROCESS1, since*
;1;;*				1 the network was sometimes flaking out in the *OPEN1 for* PROBE-FILE1...*
;1;;   23 Jun 88*	1Jamie Zawinski*	1Got *CKMAIL1 working for inboxes on unix machines.*
;1;;   29 Jul 88*	1Jamie Zawinski*	1Augumented* MAIL:PROBE-INBOXES1, and defined* CKMAIL1 to call that.*
;1;;    3 Sep 88*	1Jamie Zawinski*	1Made *MAIL-MODELINE-UPDATE-PROCESS1 not ever prompt for the user's password.*
;1;;   20 Oct 88*	1Jamie Zawinski*	1Sometimes *MAIL-MODELINE-UPDATE-PROCESS1 would hang forever in the "Directory" state.*
;1;;*				1 So I added a *WITH-TIMEOUT1.*
;1;;   12 Dec 88*	1Jamie Zawinski*	1Made *NIL1 pathnames not get on probe list, and made inboxes be forgotten at logout.*
;1;;   13 Feb 89*	1Jamie Zawinski*	1Made the toggling of abbrev and fill mode not happen when not necessary.*
;1;;   17 Feb 89*	1Jamie Zawinski*	1Patched the function *DEFAULT-FROM-ADDRESS1, removed the* SET-FROM-LINE1 hack.*
;1;;   27 Jul 89*	1Jamie Zawinski *	1Deleted obsolete mail-abbrevs code.*
;1;;    7 Sep 89*	1Jamie Zawinski *	1Cleaned up *MAIL:PROBE-INBOXES1.*
;1;;*  113 Sep 89*	1Jamie Zawinski *	1Fixed *MAIL:PROBE-INBOXES1 one more time - it was silently failing on the first attempt at a new inbox.*


(defvar 4mail:*notify-on-mail-arrival** t "2If T, send a notification when new mail arrives.*")


(defun 4probe-inboxes-avoid-password-query* (host &rest ignore)
  (let* ((uname (cdr (assoc host fs:user-unames)))
	 (rest (and uname (fs:lookup-password-etc uname host))))
    (cond ((and uname (car rest))
	   (apply #'values uname rest))
	  (t
	   (throw 'MAIL:PROBE-INBOX-ATTEMPT nil)))))


(defun 4mail:probe-inboxes* ()
  "2  Look at all of the known inboxes and see if any of them have mail or have new mail.
  When a file is noticed to have gained mail, and *MAIL:*NOTIFY-ON-MAIL-ARRIVAL* 2is non-NIL, this function
  will send a notification about the arrival.
  Returns two values:
    A list of those inboxes which have changed since the last time this function was called, and
    a list of those inboxes which are not empty.*"
  (declare (values inboxes-with-new-mail inboxes-with-any-mail))
  (guarentee-pathname-on-probe-list *unix-inbox-pathname*) ;1 Somebody's got to do this...*
  (let* ((inboxes-with-mail nil)
	 (inboxes-with-new-mail nil))
    (condition-call-if (not mail:*debug-mailer*) (condition)
	(dolist (cons mail:*inbox-probe-list*)
	  (let* ((path (first cons))		;1 The inbox we are checking.*
		 (previous-date (second cons))  ;1 The write-date of this file the last time we looked (or NIL if we've*
		 )				;1  never looked, or it didn't used to exist).*
	    (when path
	      (catch 'MAIL:PROBE-INBOX-ATTEMPT   ;1 If the remote machine wants a password, bug out.*
		(let* ((fs:*generic-login-function* 'probe-inboxes-avoid-password-query)
		       (dirlist (fs:directory-list path :noerror))
		       (file-plist (and (consp dirlist) (cdr (second dirlist)))))
		  
		  (unless (or (errorp dirlist) (null dirlist))  ;1 we couldn't read the file for some reason.  Do nothing.*
		    (let* ((new-date (getf file-plist :creation-date))
			   (new-size (getf file-plist :length-in-bytes))
			   ;1;*
			1    *;1; When the file exists and has something in it, it "has mail".*
			   ;1; When the file has mail, and its date is newer than the last time we looked (or we never looked before, or*
			   ;1; the file didn't exist when we last looked) then it "has new mail".*
			   ;1;*
			   (has-mail (and new-size (plusp new-size)))
			   (has-new-mail (and has-mail (or (null previous-date)
							   (and new-date (> new-date previous-date))))))
		      (and new-date (setf (second cons) new-date))
		      (when has-mail (pushnew path inboxes-with-mail :test #'equalp))
		      (when has-new-mail
			(pushnew path inboxes-with-new-mail :test #'equalp)
			(when mail:*notify-on-mail-arrival* (tv:notify nil "3New mail in ~A*" path)))
		      )))))))
      ((mail::handle-condition-p condition)
       condition))
    
    (values inboxes-with-new-mail inboxes-with-mail)))


(defun 4guarentee-pathname-on-probe-list* (&optional (pathname-or-pathnames *unix-inbox-pathname*))
  "2Given a pathname or list of pathnames, make sure they are all on the inbox-probe list.*"
  (cond ((listp pathname-or-pathnames)
	 (dolist (path pathname-or-pathnames)
	   (guarentee-pathname-on-probe-list path)))
	(t
	 (unless (pathnamep pathname-or-pathnames)
	   (setq pathname-or-pathnames (pathname pathname-or-pathnames)))
	 (unless (member pathname-or-pathnames mail:*inbox-probe-list*
			 :key #'car :test #'equalp)
	   (mail:add-mail-inbox-probe pathname-or-pathnames)))))


(defun 4ckmail* ()
  "2Whether the logged in user has unread mail.  See also *#'MAIL:PROBE-INBOXES2.*"
  (multiple-value-bind (new-mail old-mail) (mail:probe-inboxes)
    (declare (ignore new-mail))
    (not (null old-mail))))

(add-initialization "3Forget Inboxes*" '(setq mail:*inbox-probe-list* nil) :logout)



;1;;2 Zmacs Modeline Mail Notifications.**
;1;;*
;1;; This code causes the modeline to say whether you have mail.*


(defvar 4*mail-p-modeline-string** "" "2String to be displayed at the end of the modeline saying whether there's mail.*")
(defvar 4*mail-modeline-update-process** nil  "2Holds the process that is updating the modeline to say whether there is mail.*")
(defvar 4*mail-modeline-ck-interval** 30 "2The modeline is updated to show the state of mail every this-many seconds.*")

(defun 4mail-modeline-update-process* ()
  (loop
    (ignore-errors
      (catch 'MAIL:PROBE-INBOX-ATTEMPT   ;1 If the remote machine wants a password, bug out.*
	(process-wait "3Wait until Logged In*"
		      #'(lambda ()
			  (not (or (equalp user-id "") (equalp user-id "3File Server*")))))
	;1; Bind* FS:*GENERIC-LOGIN-FUNCTION* 1so that if the system wanted to prompt the user for a password, we bug out.*
	(let ((fs:*generic-login-function* 'probe-inboxes-avoid-password-query))
	  (with-timeout (3600)
	    (if (ckmail)
		(setq *mail-p-modeline-string* (string-append "3    *" (string-capitalize user-id) "3 has mail.*"))
		(setq *mail-p-modeline-string* "")))
	  (sleep *mail-modeline-ck-interval*))))))


(defun 4find-every-window-of-flavor* (flavor)
  "2Returns a list of all existant windows which are of type *FLAVOR2.*"
  (let* ((result nil))
    ;1; Iterate over all windows.*
    (dolist (w (send tv:default-screen :inferiors))
      (when w
	(cond ((and (typep w flavor) (send w :name-for-selection)
		    (push w result)))
	      (t (let ((wss (send w :selection-substitute)))
		   (when (and wss (typep wss flavor) (send wss :name-for-selection))
		     (push wss result)))))))
    (nreverse result)))


(defun 4turn-on-mail-modeline* ()
  (turn-off-mail-modeline)
  (setq *mail-modeline-update-process* (process-run-function "3Mail Modeline Update*" #'mail-modeline-update-process))
  ;1;*
  ;1; Set it in every existant editor.*
  (dolist (window (find-every-window-of-flavor 'ZWEI::ZMACS-FRAME))
    (let* ((sg (send (send window :process) :stack-group)))
      (eh::set-in-stack-group '*mode-line-list* sg
			      (append (eh::symeval-in-stack-group '*mode-line-list* sg)
				      '(*mail-p-modeline-string*)))))
  ;1;*
  ;1; Set it globally.*
  (when (boundp-globally '*mode-line-list*)
    (setq-globally *mode-line-list*
		   (append (symeval-globally '*mode-line-list*)
			   '(*mail-p-modeline-string*))))
  ;1;*
  ;1; Make sure newly created editors get it.*
  (advise INITIALIZE-TOP-LEVEL-EDITOR :after :turn-on-mail-modeline nil
    (unless (member '*mail-p-mode-line-string* *mode-line-list*)
      (setq *mode-line-list* (append *mode-line-list* '(*mail-p-modeline-string*)))))
  nil)


(defun 4turn-off-mail-modeline* ()
  ;1;*
  ;1; Kill the process.*
  (when *mail-modeline-update-process*
    (send *mail-modeline-update-process* :kill)
    (setq *mail-modeline-update-process* nil))
  ;1;*
  ;1; Remove the variable from every existant editor.*
  (dolist (window (find-every-window-of-flavor 'zwei::zmacs-frame))
    (let* ((sg (send (send window :process) :stack-group)))
      (eh::set-in-stack-group '*mode-line-list* sg
			      (delete '*mail-p-modeline-string*
				      (eh::symeval-in-stack-group '*mode-line-list* sg)))))
  ;1;*
  ;1; Remove it at top level.*
  (when (boundp-globally '*mode-line-list*)
    (setq-globally *mode-line-list* (delete '*mail-p-modeline-string* (symeval-globally '*mode-line-list*))))
  nil)


;1;;; Signature string.*

(defvar 4*send-mail-signature-string** nil)
(forward-value-cell '*send-mail-signature-string* 'mail::*mail-user-personal-name*)


;1;; Fixed this function so that the user could specify a return-address in *MAIL::*MAIL-USER-PERSONAL-NAME*
;1;; It used to be that "My Name" mapped to "My Name <my address>", but that "My Name <my address>" mapped to "My Address".*
;1;;*
MAIL:
(defun 4DEFAULT-FROM-ADDRESS* (&optional errorp)
  
  ;1;Would like to use fs:user-personal-name-first-name-first as a "fallback" when the personal*
  ;1;name is not specified in some other way, but this variable gets mangled for unknown*
  ;1;reasons, so it's use has been removed*
  
  (fs:force-user-to-login)
  (let* ((addr-string (or *user-mail-address*
			  (name:lookup-attribute-value user-id :user :mail-address)))
	 ;1; SPR#5420: Default personal name from user object if variable no set.   \/\/*
	 (personal-name (or *mail-user-personal-name*
			    (name:lookup-attribute-value user-id :user :personal-name)))
	 user-supplied-p
	 address
	 temp-string)
    
    (cond ((or (null addr-string) (equal addr-string ""))
	   (setq temp-string (xstring-append nil user-id "3@*" (send fs:user-login-machine :name)))
	   (setq address (parse-address temp-string 0 nil errorp :mailbox))
	   (deallocate-xstring temp-string))
	  (t
	   (setq user-supplied-p t)
	   (setq address (parse-address addr-string 0 nil errorp :mailbox))))
    (when (and (basic-address-p address)
	       personal-name
	       (stringp personal-name)
	       (not (equal personal-name "")))
      (cond ((string-search-set '(#\< #\> #\@ #\, #\; #\:) personal-name)
	     (setq address (parse-address personal-name 0 nil errorp :mailbox)))
	    (t
	     (setq temp-string (xstring-append nil personal-name "3 <*" (send address :string-for-message) "3>*"))
	     (setq address (parse-address temp-string 0 nil errorp :mailbox))
	     (deallocate-xstring temp-string))))
    (values
      (send address :canonical-address)
      user-supplied-p)))
