;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:W; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER ADOBE-COURIER14B HL12B HL12I) -*-

;1;; File "3LIVE-LISTENER*"*
;1;; Defines a new flavor of Lisp Listener which has mouse sensitivity similar to Symbolics listeners.*
;1;; Copyright (c) 1989 University of California, Berkeley.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;    27 Jul 89*	1Jamie Zawinski*	1Created.*
;1;;*   131 Jul 89*	1Jamie Zawinski*	1Commented, made commands type-sensitive and configurable.*
;1;;*   11 Aug 89*	1Jamie Zawinski*	1Made it work with variable-width fonts, moved it to the W package.*
;1;;*   12 Aug 89*	1Jamie Zawinski*	1Converted to use the UCL.*
;1;;*   13 Aug 89*	1Jamie Zawinski*	1Implemented 5:read-or-click*.*
;1;;*   17 Aug 89*	1Jamie Zawinski*	1Got the 5:prompt-and-read* method working.*
;1;;*   18 Aug 89*	1Peter Norvig*	1Added some commands.*
;1;;*  121 Aug 89*	1Jamie Zawinski*	1Fixed the bug where it was not possible to bind two things to one mouse button in the same command*
;1;;*				1 table; the command table is no longer automatically sorted in alphabetical order, so which command gets*
;1;;*				1 the button is determined by which one appears earlier in the table.*
;1;;*				1Made user-typed items mousable - that is, you can now do Control-C/Meta-C with the mouse.*
;1;;*  122 Aug 89*	1Jamie Zawinski*	1Made 5fquery* output be mousable - clicking right while being queried brings up a menu of choices.*
;1;;*  128 Aug 89*	1Jamie Zawinski*	1Added a presentation-like ability, by having command tables on the item-type-alist.  *
;1;;*				1Reimplemented input history commands this way (nuking various hacks), and made text items of the form*
;1;;*				1 2<< While compiling FOO >>* mousable.*
;1;;*   18 Sep 89*	1Jamie Zawinski*	1Suggestions from Rice: fixed 5:mouse-click* method to send a 5:mouse-select* if self is not the *
;1;;*				1 selected window, and moved all of the methods/commands to 5live-listener-mixin* so that making*
;1;;*				1 other kinds of listeners might be easier.  Added a conditional patch to KSL's incorrect version of*
;1;;*				1 5tv:grind-into-list-make-item*.*
;1;;*  113 Sep 89*	1Jamie Zawinski*	1Added both the live listener and the old listener to the Create option of the System Menu.*
;1;;*  129 Sep 89*	1Jamie Zawinski*	1Added Release-6 / CLOS support.*
;1;;*   12 Oct 89*	1Jamie Zawinski*	1Made 5live-listener-pprin1* take a window as an argument.*
;1;;*   16 Oct 89*	1Jamie Zawinski*	1Added 5:copy-to-kill-ring* command, and made prompts print more often.*
;1;;*  127 Oct 89*	1Jamie Zawinski*	1Cleaned up synonym-stream handling a bit.*
;1;;*				1Made notifications be printed mousably.*
;1;;*  130 Oct 89*	1Jamie Zawinski*	1Added the 5:describe-host* command.*
;1;;*  117 Nov 89*	1Jamie Zawinski*	1Major reimplementation, dividing things up into many more mixins.*
;1;;*  121 Nov 89*	1Jamie Zawinski *	1Added mousability to all break loops.  Made all calls to 5pprint* produce mousable objects.*
;1;;*  122 Nov 89*	1Jamie Zawinski *	1Sometimes my mouse was locking up when over a string.  I think this was because while computing*
;1;;*				1 the wholine info, we were checking the type of that string against 5maybe-address-string* and*
;1;;*				1 the namespace was freaking out on it, instead of returning a nice graceful error.  Made*
;1;;*				1 5maybe-address-string* be more careful about what it tries to parse.*
;1;;*				1Made mousability work while in the debugger as well; someone was calling 5:tyi* instead of 5:any-tyi*.*
;1;;*    12 Dec 89*	1Jamie Zawinski *	1Made Mouse-Left return objects even if in a Typeout window.*
;1;;*   111 Jan 90*	1Jamie Zawinski *	1Tweaked 5pathname-parsable *some more...  arrrrrggghh!!*
;1;;*   114 Mar 90*	1Jamie Zawinski *	1Made items be in inverse-video when active.*
;1;;*

(export '(live-listener prin1-mousably-mixin live-listener-UCL-commands-mixin live-listener-synchronous-UCL-mixin
	  live-listener-asynchronous-mixin LL-command *LL-lisp-command-table* *LL-history-command-table*
	  *default-live-listener-font-map*
	  ))


;1;; This file defines some new flavors which enable mouse-sensitivity of objects printed with 5print, prin1, princ, pprint,* etc.*
;1;; Clicking right on* 1the object brings up a menu of all operations which are* 1applicable to that type of object - that is, there are*
;1;; operations one can perform* 1on hash tables that one cannot perform on symbols, and* 1vice versa. * 1Actions can be tied to*
;1;; clicking or double-clicking Left or Middle;* 1for example, in some of the window flavors defined in this file, the default actions are*
;1;; for Mouse-Left* 1to* 1return* 1the object,* 1Mouse-M to describe it,* 1and Mouse-M-2 to inspect it.*
;1;;*
;1;; Two major changes are made by this file.  First, there is a new flavor of Lisp Listener, called 5w:live-listener*, which has the*
;1;; mouse-sensitivity described above.  This is bound to 5System-L*, so to use it just spawn a new Lisp Listener.*
;1;; The second change is that Zmacs break-loops also have this mouse-sensitivity.  Again, this will not exhibit itself until you*
;1;; make a new Zmacs; old instances will not have this feature.*

;1;; 5Implementing Live Listener Commands:**
;1;; 5=============================**
;1;;*
;1;; There are many commands already defined for manipulating Lisp objects, but defining new ones is fairly easy. * 1To define a new one,*
;1;; simply use 5defcommand*, giving it a 5:command-flavor* of 5W:LL-Command*.*  1If you want your new command to only be applicable to*
;1;; certain* 1types, then provide a type specifier with the5 :applicable-type** 1keyword in the keyword list passed to5 defcommand*; when*
;1;; deciding* 1whether your command can be used on a given object, the* 1object will be tested against your type specifier with 5typep*. * 
;1;; By using the* 5satisfies1 type specifier, you can have commands** 1which are only applicable in very restricted circumstances; for*
;1;; instance, the* 6Makunbound1 command shows up on the menu only of** 1objects which are bound symbols, and are not declared to be*
;1;; constants. * 1And the* 6Inspect Flavor1 command can be used on** 1instances, flavors, and symbols which name flavors.*
;1;;*
;1;; It is possible to have commands which are active when not over any object.  To do this, give the command an 5:applicable-type**
;1;; of NIL.  NIL is the type to which no objects belong, so we use it to represent the set of no objects.  However, since an empty set*
;1;; is a subset of all sets, to specify a command which is applicable both when over an object and when not over an object, we use a*
;1;; slightly hacked specification, like 5(or nil integer)*.  Though 5NIL* is a subtype of 5integer*, only a specification which explicitly names*
;1;; NIL will be available over non-objects.  Commands which are invoked when not over an object are called with no arguments; *
;1;; commands which are invoked when over an object are invoked with one argument, the object.*
;1;;*
;1;; For prunings which are too sophisticated or to inefficient to represent with a type specifier, you may define a new flavor whose*
;1;; parent is 5ll-command* and specialize its 5:applicable?* method; the default version of this method simply calls 5typep*.  Just pass the*
;1;; name of this new flavor into 5defcommand* instead of 5LL-Command.**
;1;;*
;1;; The Live-listener mouse commands are looked up in UCL command tables which are enumerated on the 5mouse-command-tables**
;1;; instance variable of one of the mixins.  To use Live Listeners for applications other than Lisp typein, simply define a new set of*
;1;; commands, and give your instance a new set of command-tables.*
;1;;*
;1;; It is possible to have an object specify its set of commands, as opposed to having the commands specify their set of objects.*
;1;; This is accomplished by having a command table (or a symbol naming one) in the fourth position of an element on the 5item-type-alist*.*
;1;; The input history commands are implemented this way; look at that for examples.*
;1;;*
;1;; All mousable items on Live Listener windows are printed in Font 1, the second element in the window's font map.  You can use*
;1;; the 5:set-font-map* method to change the output.  Live Listeners cope well with variable width fonts.  Likewise, all items typed*
;1;; by the user at top-level are printed in font 2, the third element in the window's font map.*
;1;;*
;1;; You can prompt the user for one of the objects that is printed on the screen with the 5:mouse-read-type* method of*
;1;; 5live-listener*.  Pass this method a type specifier; only those objects which are of that type will be selectable.  When the user*
;1;; clicks left on one of them, it will be returned to you.  *
;1;;*
;1;; Also, calling 5prompt-and-read* will let the user either type a value, or mouse on one.  Only those values which are appropriate*
;1;; will be mousable.*
;1;;*
;1;; When 5fquery* is called, clicking right will bring up a menu of the Fquery options.  Selecting from this menu is the same as typing*
;1;; the corresponding character or string.*
;1;;*

;1;; 5Implementing New Kinds of Live Listeners:**
;1;; 5================================**
;1;;*
;1;; The following flavors are defined here.  To implement a new style of Live Listener (for example, one which interfaced to a radically *
;1;; different kind of command loop, or one which looked up commands in a different way) you must define your own version of one or*
;1;; more of the following mixins.  *
;1;;*
;1;; 4prin1-mousably-mixin**			1Anything printed with 5prin1* or 5princ* will be mousable.  When the mouse is clicked on*
;1;;*						1a printed object, a 5:process-print-blip* method is invoked with the mouse-sensitive *
;1;;*						1item, the button which selected it, and the mouse position.*
;1;;*
;1;; 4live-listener-UCL-commands-mixin**		1Defines methods for getting UCL commands applicable to a given printed object.*
;1;;*						1A 5:process-print-blip* method is defined here; it finds and invokes the first UCL *
;1;;*						1command (from a list on the window instance) which is applicable to the printed object.*
;1;;*						1The command is ``invoked'' by calling 5:mouse-click-invoke-command*.*
;1;;*						1Though UCL commands are used to represent the commands, the window need not*
;1;;*						1contain a UCL mixin, or indeed be running any command loop at all.*
;1;;*
;1;;*						1This defines a 5:mouse-click-invoke-command* which ``executes'' a command by deferring*
;1;;*						1it.  An 5:execute* blip is stuffed on the input buffer of the window, with the intention that*
;1;;*						1when the command loop (any command loop) is again ready to process input, that blip*
;1;;*						1will be read and invoked.  A handler for 5:execute* blip is not provided at this level, *
;1;;*						1though one is provided in break loops, debuggers, and the Zmacs top-level.*
;1;;*
;1;; 4live-listener-synchronous-UCL-mixin**	1Provides a handler for the 5:execute* blip; makes user-typed things be mousable.*
;1;;*						1To use this mixin, you must include a UCL command loop mixin in the window.*
;1;;*
;1;; 4live-listener**				1All of the above, bundled up into a 5w:lisp-listener*.*
;1;;*
;1;; 4asynchronous-live-listener**		1This is like 5live-listener*, except that commands happen as they are invoked, instead*
;1;;*						1of being queued up; you may find this more to your liking.*
;1;;*

;1;; 5Misfeatures:* 6(i.e., bugs)**
;1;;5 =========**
;1;;*
;1;;     3 **	1The blinkers which follow printed objects don't follow list structure properly (that is, like the Inspector).  The code which does*
;1;; *	1this for the inspector is tightly tied into the fact that the inspector frames are text-scroll-windows, so I can't just hook into*
;1;;*	1that already-existing code.*
;1;;*
;1;;     3 **	1When printing large compound objects, sometimes the regions are not over the text of the object they represent; I think this*
;1;;*	1is a bug in the TI mousable-item code.*
;1;;*
;1;;     3 **	1Our use of the 3NIL* type in applicable-type specifications is kind of silly.*
;1;;*
;1;;     3 **	1We imcompatibly change some flavors, so old instances do not get these features.  Not a lot can be done about this.*
;1;;*	1Build it into the core and you probably won't notice.*
;1;;*
;1;;     3 **	1Inspect should have some inspect-specific commands; for instance, inspecting an object from an inspect break loop should not*
;1;;*	1create another inspect frame.*
;1;;*
;1;;     3 **	1The Inspect listener window (not the break loop) should also be sensitive.*
;1;;*
;1;;     3 **	1User input (the command history) is not mousable on anything except 5w:live-listener* windows; this is because in order to *
;1;;*	1fontify and mousify the text, we need to wrap the 5:handle-typein-input* method of the UCL.  This is easy for windows in which*
;1;;*	1the UCL application instance and the window itself are the same thing, but for random typeout windows, the UCL instance and *
;1;;*	1the window are distinct, leading to much confusion.  We can't just wrap the 5:handle-typein-input* method of 5sys:ucl-break*, because*
;1;;*	1we can't assume that all ucl-breaks are running on windows.  We can't just wrap a method on the window, because the method *
;1;;*	1isn't there.*

(defvar 4*default-live-listener-font-map* *(list FONTS:CPTFONT FONTS:CPTFONTI FONTS:CPTFONTB FONTS:TR8B)
  "2This is the default font map for windows on which printed objects are mousable.*")

(defvar 4*ll-lisp-command-table** nil "2The table in which the lisp-context mouse commands live.*")
(defvar 4*ll-history-command-table** nil "2The table in which the history-context mouse commands live.*")


(defstruct 4(user-input-descriptor* (:constructor make-4user-input-descriptor* (object printed-representation)))
  "2This structure represents a command or form typed by the user.
  We don't store just a list, because we want to preserve whitespace and capitalization.
  We don't store just a string, because we don't want to have to call READ all the time.
  This must be its own datatype so that a different set of commands can apply to the input history, but not to random printed objects.*"
  object			;1 The unevaluated object typed by the user; usually a list, but may be a string, an instance, etc.*
  printed-representation	;1 The exact string that they typed, whitespace and all.*
  )


(defvar 4*live-listener-item-type-alist**
	'((:PRINTED-OBJECT :printed-object    (live-listener-wholine-function) nil)
	  (:USER-INPUT     :user-input	      (live-listener-wholine-function) *ll-history-command-table*)
	  (FUNCTION-NAME   function-name      (live-listener-wholine-function) *ll-lisp-command-table*)
	  )
  "2The alist for *LIVE-LISTENER2 windows; calls to *W:ADD-TYPEOUT-ITEM-TYPE2 are made with this to add mouse commands.*")


(defmacro 4in-font *((window font &optional ok-if-not-in-font-map) &body body)
  "2For the duration of body, the current font of the window will be FONT.  The font is reset to what it was before after body completes.*"
  (let* ((f (gensym))
	 (w (gensym)))
    `(let* ((,w ,window)
	    (,f (send ,w :current-font)))
       (unwind-protect (progn
			 (send ,w :set-current-font ,font ,ok-if-not-in-font-map)
			 ,@body)
	 (send ,w :set-current-font ,f t)))))


;1;; 4prin1-mousably-mixin**			1Anything printed with 5prin1* or 5princ* will be mousable.  Handle clicks youself.*


(defflavor 4prin1-mousably-mixin *() ()
  (:included-flavors w:basic-mouse-sensitive-items w:essential-window)
  (:default-init-plist :item-type-alist *live-listener-item-type-alist*)
;1 These mess up the calls to build-command-table, so screw it.*
;  (:required-flavors stream-mixin essential-mouse)
;  (:required-methods :process-print-blip)
  (:documentation :mixin
   "2Mixing this flavor into a window output that is printed with PRIN1 or PRINC be mousable.
  The output methods of the window are modified substantially to hook this in.
  Clicks on the mouse-sensitive items representing printed objects to be interpreted by calling
  the :PROCESS-PRINT-BLIP method, which is not defined here.*")
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(prin1-mousably-mixin :item-for-output*) (object start-x start-y end-x end-y &optional type max-x)
  "2Dump a mouse-sensitive item on SELF which is large enough to cover the OBJECT.
   START-X,Y and END-X,Y are the pixel positions at which the first and last characters in the printed representation of OBJECT landed.
   If the object occupied multiple lines, then multiple mouse-sensitive items will be output so that the object is completely covered.*"

  (flet ((item (x y x2 y2)
	   (send self :primitive-item-outside (or type :PRINTED-OBJECT) object x y x2 y2)
;	   (send self :draw-rectangle x y (- x2 x) (- y2 y))
	   ))
    
    (cond ((= start-y end-y)	;1 5Items which fit entirely on one line.**
	   (item start-x start-y end-x (+ end-y tv:line-height -1)))
	  
	  (t			;1 5Items which occupy multiple lines.**
	   ;1;*
	   ;1; We draw one of four possible polygons:*
	   ;1;*
	   ;1;*		----------------------------		              ----------
	   ;1; sx = lm*	|                          |	1ex = rm*	--------------|        |
	   ;1; ex = rm*	|                          |		|                      |
	   ;1;*		----------------------------		------------------------
	   ;1;*
	   ;1;*		----------------------------		              ----------
	   ;1; sx = lm*	|                          |	1else*	--------------|        |
	   ;1;*		|            |-------------1-*		|                      |
	   ;1;*		--------------             		|        |--------------
	   ;1;*							----------
	   (let* ((nlines (floor (- end-y start-y) tv:line-height))
		  (right (or max-x (- tv:width tv:right-margin-size))))
	     ;1;*
	     ;1; Add a rectangle for the top section, if the top section's left edge does not line up with the middle section's.*
	     ;1; If they line up, then just make the middle section taller.*
	     ;1;*
	     (if (<= start-x tv:left-margin-size)
		 (setq nlines (1+ nlines)
		       start-y (- start-y tv:line-height))
		 (item start-x start-y right (+ start-y tv:line-height -1)))
	     ;1;*
	     ;1; Add a rectangle for the bottom section, if the bottom section's right edge does not line up with the middle section's.*
	     ;1; If they line up, then just make the middle section taller.*
	     ;1;*
	     (if (>= end-x (- right tv:left-margin-size))
		 (incf nlines)
		 (item tv:left-margin-size end-y end-x (+ end-y tv:line-height -1)))
	     ;1;*
	     ;1; Add a rectangle for the middle section, perhaps encompassing the top and/or bottom sections as well.*
	     ;1;*
	     (when (plusp nlines)
	       (item tv:left-margin-size (+ start-y tv:line-height) right (+ start-y (* nlines tv:line-height))))
	     )))))


(defvar 4*princ-string-mousable-p* *nil
  "2If T, then strings printed with PRINC will be mousable; otherwise, they will be just static text.  It's a good idea for this to be NIL,
 because otherwise seemingly random parts of the output of FORMAT will be mousable.*")

(defmethod 4(prin1-mousably-mixin :print)* (exp prindepth escape)
  "2Anything written on SELF using PRINT, PRIN1, or PRINC goes through this method; we ensure that the printed items are mousable, and 
  that they are printed in font #1 (the second font in the window's font map).*"
  (let* ((x w:cursor-x)
	 (y w:cursor-y)
	 (font (aref (send self :font-map) 1)))	;1 Mousable objects are always printed in font #1.*
    (cond ((or (consp exp)
	       (and (arrayp exp)
		    (not (stringp exp))
		    *print-array*))
	   ;1;*
	   ;1; If the object is a compound object (that is, a list, or an array with 5*print-array** on) then it is necessary for our caller*
	   ;1; (up in the guts of the printer code) to print the object.  We must call compute-motion to determine how big it will be.*
	   ;1;*
	   (multiple-value-bind (end-x end-y)
				(tv:sheet-compute-motion self x y (if escape (prin1-to-string exp) (princ-to-string exp))
							 0 nil nil 0 nil nil nil font tv:line-height)
	     (send self :item-for-output exp x y end-x end-y))
	1    *;1; Return NIL, meaning the caller should print this object.*
	   nil)
	  
	  ((and (stringp exp)
		(or *princ-string-mousable-p*
		    (not escape)))
	   ;1; If this is a string, and print-escape is off, then there is little point in making it mousable.  When one calls 5princ* on a string,*
	   ;1; the string is usually not intented to be interpreted as an 6object*; it is just text.  What you see is what you get.*
	   ;1; This usually-reasonable behavior can be overridden by binding 5*princ-string-mousable-p** to non-nil.*
	   nil)
	  
	  (t
	   ;1;*
	   ;1; If the object is not compound, then we can print it ourself, and know what area it took up by watching 5w:cursor-6[*x6,*y6]** change.*
	   ;1;*
	   (in-font (self 1) (sys:print-object exp prindepth self '(:string-out)))
	   (let* ((new-x w:cursor-x)
		  (new-y w:cursor-y))
	     (send self :item-for-output exp x y new-x new-y))
	1    *;1; Return T, meaning the caller should not print this object since we just did.*
	   t))))


;1;; If CLOS is around, we need to do things slightly differently.  In the CLOS version of 5print-object*, we don't hook in via the 5:print**
;1;; method, but rather by specializing the internal printing function.*
;1;;*
;1;; One small problem with this is that we want to dispatch on the second argument, the stream.  When the stream is a live-listener,*
;1;; we want to do things differently.  But, the stream argument may be a synonym-stream, not the window itself.  So, we wrap *
;1;; 5print-object* so that if it is passed a synonym-stream, it recurses on the real stream at the end of that chain.  Eventually it will*
;1;; be calling itself with a non-synonym stream, and we can dispatch on the type.

4#+CLOS**
(ticlos:defmethod 4ticlos:print-object :around* ((object t) (stream symbol))
  "2Expand synonym streams into real streams before doing type-dispatching on the second argument.*"
  (let* ((syn-stream (sys:follow-syn-stream stream)))
    (if (neq syn-stream stream)
	(ticlos:print-object object syn-stream)    ;1 Can't do call-next-method here - args have changed.*
	(ticlos:call-next-method))))


;1;; Now, we should be able to just do this.  But down in the CLOS printer code, the Flavors printer code is called.  So a CLOS method like*
;1;; this one will be called on a top-level object, but not on sub-objects of it (like a list or vector's* 1contents).  In an ideal world, when CLOS*
;1;; was loaded, we would need only this clos-method, and not the 5:print* flavors-method of 5live-listener-synchronous-UCL-mixin*.  But with *
;1;; the current setup, we must provide a CLOS hook that calls our Flavors hook.*
;1;;*
;4#+CLOS*
;(ticlos:defmethod4 ticlos:print-object :around* ((object t) (stream w:live-listener-synchronous-UCL-mixin))
;  "2Wrap a mouse-sensitive item around the object which is printed.*
;2  This implementation requires that calling 5ticlos:print-object* with a list causes this method to be invoked recursively.*"
;  (let* ((x (send stream :cursor-x))
;	 (y (send stream :cursor-y))
;	 (value nil))
;    (in-font (stream 1)
;      (when (ticlos:next-method-p)
;	(setq value (ticlos:call-next-method))))
;    (let* ((new-x (send stream :cursor-x))
;	   (new-y (send stream :cursor-y)))
;      (send stream :item-for-output object x y new-x new-y))
;    value))

4#+CLOS*
(ticlos:defmethod 4ticlos:print-object :around* ((object t) (stream w:prin1-mousably-mixin))
  "2Invoke the 5:print* flavors-method of 5w:live-listener-synchronous-UCL-mixin* to print this object.*"
  (or (send stream :print object si:*prindepth* *print-escape*)
      (when (ticlos:next-method-p) (ticlos:call-next-method))))


(defun 4live-listener-pprin1 *(object &optional (x-offset 0) width (window self))
  "2This function is used to pretty-print returned values on WINDOW; we ensure that the printed items are mousable, and that they
  are printed in font #1 (the second font in this window's font map).

  Unfortunately, if someone explicitly calls PPRINT, it does not go through* 2this method.  The PPRINT code does not have method-hooks
  in it like the PRIN1 code does.*"

  (multiple-value-bind (strings print-locs list-locs)
		       (tv:grind-into-list object (or width sys:pp-line-length (send window :size-in-characters)) t)
    ;1;*
    ;1; First put down the regions for the lists in the output.*
    ;1;*
    (let* ((lm (tv:sheet-left-margin-size window))
	   (y (tv:sheet-cursor-y window))
	   (lh (tv:sheet-line-height window))
	   (font (aref (send window :font-map) 1)))	;1 Mousable objects are always printed in font #1.*
      (dolist (list list-locs)
	(let* ((loc (pop list))
	       (start-char   (pop list))
	       (start-line-n (pop list))
	       (end-char     (pop list))
	       (end-line-n   (pop list))
	       (start-line (nth start-line-n strings))
	       (end-line   (nth end-line-n   strings))
	       (start-x (+ lm (tv:sheet-string-length window start-line 0 start-char nil font)))
	       (end-x   (+ lm (tv:sheet-string-length window end-line   0 end-char nil font)))
	       (start-y (+ y (* start-line-n lh)))
	       (end-y   (+ y (* end-line-n   lh)))
	       (real-object (if (locativep loc) (contents loc) object)))
	  (send window :item-for-output real-object (+ x-offset start-x) start-y (+ x-offset end-x) end-y)))
1        *;1;
        *;1; The put down the regions for the atoms in the output.
        *;1;*
      (do* ()
	   ((null strings))
1           *;1; First put down the regions for this line.*
	(dolist (list (car print-locs))
	  (let* ((loc (first list))
		 (start (third list))
		 (end (fourth list))
		 (start-pixel (tv:sheet-string-length window (car strings) 0 start nil font))
		 (pixel-width (tv:sheet-string-length window (car strings) start (min end (length (car strings))) nil font))
		 (real-object (if (locativep loc) (contents loc) object)))
	    (send window :item-for-output real-object
		  (+ lm x-offset start-pixel) y
		  (+ lm x-offset start-pixel pixel-width) y)))
	;1; Then print the text.*
	;1;*
	(in-font (window 1)
	  (tv:sheet-open-blinkers window)
	  (setf (tv:sheet-cursor-x window) (max x-offset (tv:sheet-cursor-x window)))
	  (write-line (car strings) window))
	(pop strings)
	(pop print-locs)
	(incf y lh)
	))))

;1;; Place advice on 5pprin1* so that if it is called on a5 *window including 5prin1-mousably-mixin*, it will *
;1;; use 5live-listener-pprin1* instead of the normal 5pprin1*.  This is kind of a hack.*
;1;;*
(compiler-let ((sys:compile-encapsulations-flag t))
  (sys:advise 4sys:pprin1 :around 'mousability* nil
    (let* ((stream (second sys:arglist)))
      (when (or (null stream) (eq stream t)) (setq stream *standard-output*))
      (if (and stream
	       (typep (setq stream (sys:follow-syn-stream stream))
		      'prin1-mousably-mixin))
	  (live-listener-pprin1 (car sys:arglist) 0 nil stream)
	  :do-it)))
  )


(defmethod 4(prin1-mousably-mixin :print-notification-on-self*) (time string window-of-interest)
  "2Print a notification on self mousably.*"
  (lock-sheet (self)
    (tv:output-before-rubout-handler (self)
      (in-font (self 0)
	(send self :fresh-line)
	(send self :beep 'tv:notify)
	(let* ((notification-x1 w:cursor-x)
	       (notification-y1 w:cursor-y)
	       time-x1 time-y1 time-x2 time-y2
	       string-x1 string-y1 string-x2 string-y2
	       )
	  (write-char #\[ self)
	  (setq time-x1 w:cursor-x time-y1 w:cursor-y)
	  (time:print-brief-universal-time time self)
	  (setq time-x2 w:cursor-x time-y2 w:cursor-y)
	  (write-char #\Space self)
	  ;1; Write the string mousably.*
	  (setq string-x1 w:cursor-x string-y1 w:cursor-y)
	  (let ((end (length string)))
	    (or (zerop end)
		(send self :string-out string 0
		      (if (= (aref string (1- end)) #\Newline)
			  (1- end)
			  end))))
	  (setq string-x2 w:cursor-x string-y2 w:cursor-y)
	  (write-char #\] self)
	  (when window-of-interest
	    (send self :item-for-output window-of-interest notification-x1 notification-y1 w:cursor-x w:cursor-y))
	  (send self :item-for-output time time-x1 time-y1 time-x2 time-y2)
	  (send self :item-for-output string string-x1 string-y1 string-x2 string-y2)
	  (terpri self))))))


;1;; Clicks, at the most general level.*
;1;;*
;1;; The5 :mouse-click4 **method of5 prin1-mousably-mixin *looks up mouse-sensitive item, determines whether it represents a printed object,*
;1;; and if so, invokes the 5:process-print-blip* method with the mouse-sensitive item, the button which selected it, and the mouse position.*
;1;; This method is defined in a higher context.*
;1;;*
;1;; Returning 5nil* from the 5:mouse-click* method of 4prin1-mousably-mixin* means that the 5:mouse-click* method of the next mixed-in flavor*
;1;; will be executed (which happens to be 5basic-mouse-sensitive-items*).  Returning non-nil means it will not be.  This is because the*
;1;; 5:mouse-click* method of 5essential-mouse* was defined with that open sore called 6:OR method combination.*  The less you deal with this*
;1;; thing, the happier you will be.*
;1;;*
(defmethod 4(prin1-mousably-mixin :mouse-click)* (button x y)
  "2When the mouse is clicked, dispatch to the printed-object click-handler.*"
  "2Look up the button in the mouse-command-tables, and invoke the appropriate command with the clicked-on object.*"
  (cond
    ;1; If we are not the selected window, then interpret this click as a mouse-select.  Otherwise, process the click.*
    ((not (eq self tv:selected-window))
     (send self :mouse-select)
     t)
    (t
     (let* ((item (send self :mouse-sensitive-item x y)))
       (cond ((and item (not (eq (tv:typeout-item-type item) :PRINTED-OBJECT)))
	      ;1; There is an item, but it is not one produced by 3PRIN1* or 3PRINC*.*
	      ;1; Return NIL and let the 3:MOUSE-CLICK* method for 3BASIC-MOUSE-SENSITIVE-ITEMS* handle it.*
	      nil)
	     (t
	      ;1; There is no item, or there is an item produced with 3PRIN1* or 3PRINC*.*
	      ;1; If there is an entry on our item-type alist for it, and there is a command table there, then invoke one of those commands.*
	      (send self 4:process-print-blip* item button x y)
	      t))))))

(defmacro 4with-region-hilighted *((typeout-item window) &body body)
  "2Arrange for the given mouse-sensitive region to be in inverse-video for the duration of BODY.*"
  (let* ((item (gensym))
	 (w (gensym)))
    `(let* ((,item ,typeout-item)
	    (,w ,window))
       (unwind-protect
	   (progn (when ,item
		    (tv:prepare-sheet (,w)
		      (sys:%draw-rectangle (- (w:typeout-item-right ,item) (w:typeout-item-left ,item) 1)
					   (- (w:typeout-item-bottom ,item) (w:typeout-item-top ,item) 1)
					   (w:typeout-item-left ,item) (w:typeout-item-top ,item) w:opposite ,w)))
		  ,@body)
	 (when ,item
	   (tv:prepare-sheet (,w)
	     (sys:%draw-rectangle (- (w:typeout-item-right ,item) (w:typeout-item-left ,item) 1)
				  (- (w:typeout-item-bottom ,item) (w:typeout-item-top ,item) 1)
				  (w:typeout-item-left ,item) (w:typeout-item-top ,item) w:opposite ,w)))))))




;1;; 4live-listener-UCL-commands-mixin**		1Defines methods for getting UCL commands applicable to a given printed object.*
;1;;*						1Represents mouse-operations with UCL commands (without necessarily using the UCL).*

(defflavor 4live-listener-UCL-commands-mixin*
	   ((mouse-command-tables '(*ll-lisp-command-table*))
	    (restrict-to-type	  't))
	   ()
  (:included-flavors w:basic-mouse-sensitive-items w:minimum-window w:stream-mixin w:essential-mouse)
  (:documentation :mixin
    "2This flavor enables the use of type-sensitive UCL commands on printed objects.
 The UCL command-loop need not be running in a window using this.  Once the user has selected
 an object and an operation on it, this mixin invokes the method :MOUSE-CLICK-INVOKE-COMMAND,
 which is not defined here.*")
;1 These mess up the calls to build-command-table, so screw it.*
;  (:required-flavors prin1-mousably-mixin)
;  (:required-methods :mouse-click-invoke-command)  ;1 Once it gets a command to invoke, it passes it to this.*
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


;1;; Selection of items.*
;1;;*
;1;; The 5:process-print-blip* method of 5live-listener-UCL-commands-mixin* finds a list of UCL command objects which apply to the given*
;1;; mouse-sensitive item (containing an object that was printed with 5prin1* or 5princ*), and invokes the first one that applies to that object*
;1;; with that mouse-button.  (Menus of commands are implemented as a command which invokes other commands).*
;1;;*
;1;; The command is ``invoked'' by passing it to the 5:mouse-click-invoke-command* method, which is defined in a higher context.*
;1;;*
(defmethod 4(live-listener-UCL-commands-mixin :process-print-blip)* (item button x y &optional commands)
  "2Look up the button in the mouse-command-tables, and invoke the appropriate command with the clicked-on object.*"
  (declare (ignore x y))
  (with-region-hilighted (item self)
  ;1; Query all commands as to whether they apply to this item.*
  (let* ((no-object-p (null item))
	 (object (tv:typeout-item-item item))
	 (commands (or commands
		       (if no-object-p
			   (send self :commands-for-no-object)
			   (send self :commands-for-object object)))))
    (block DONE
      (dolist (command commands)
	(when (or ;1 *guarenteed-applicable
		  (if no-object-p
		      (send command :applicable-to-none?)
		      (send command :applicable? object)))
	  (let* ((chs (send command :keys)))
	    (dolist (ch chs)
	      (when (consp ch) (setq ch (car ch)))
	      (when (char= ch button)
		(send self 4:mouse-click-invoke-command* command object item button)
		(return-from DONE t))))))
      (if (char= button #\Mouse-R-1)	;1 If nothing bound to Mouse-R, do the system-menu-boogie.*
	  (tv:mouse-call-system-menu)
	  (beep))))))


(defun 4live-listener-wholine-function *(&optional (item nil itemp))
  "2Given any object, returns a wholine documentation list describing it.  
 This assumes the selected window contains LIVE-LISTENER-UCL-COMMANDS-MIXIN.*"
  (let* ((object (second item))
	 (doc (and itemp
		   (typecase object
		     (NULL       "3This is the symbol NIL.*")
		     (HASH-TABLE (format nil "3This is a hash table of ~D element~:P.*" (hash-table-count object)))
		     (USER-INPUT-DESCRIPTOR  "3This is user input.*")
		     (STRUCTURE  (format nil "3This is a structure of type ~S*" (type-of object)))
		     (SEQUENCE   (let* ((type (type-of object))
					(length (if (consp object) (list-length object) (length object))))
				   (when (eq type 'CONS) (setq type 'LIST))
				   (if length
				       (format nil "3This is a ~S of length ~S.*" type length)
				       "3This is a circular list.*")))
		     (CONDITION	 (format nil "3This is a CONDITION of type ~S.*" (type-of object)))
		     (INSTANCE   (format nil "3This is an instance of the ~S flavor.*" (type-of object)))
		     (ARRAY      (format nil "3This is a ~D~{x~D~} ~S.*"
					 (array-dimension object 0) (cdr (array-dimensions object)) (type-of object)))
		     (SYMBOL     (if (symbol-package object)
				     (format nil "3~S is a symbol from the ~A package.*" object (symbol-package object))
				     (format nil "3~S is a symbol with no home package.*" object)))
		     (t (format nil "3This is an object of type ~S.*" (type-of object)))))))
    (let* ((list (and doc (list :DOCUMENTATION doc)))
	   (commands (cond ((user-input-descriptor-p object)
			    ;1; ## This is an unfortunate hack, since the user-input commands use a different set of command tables.*
			    (send tv:selected-window :commands-for-object object '(*ll-history-command-table*)))
			   (itemp
			    (send tv:selected-window :commands-for-object object))
			   (t (send tv:selected-window :commands-for-no-object))))
	   ML1 ML2 MM1 MM2 MR1)
      (dolist (command commands)
	(let* ((chs (send command :keys)))
	  (dolist (ch chs)
	    (when (consp ch) (setq ch (car ch)))
	    (case ch
	      (#\Mouse-L-1 (unless ML1 (setq ML1 (send command :description))))
	      (#\Mouse-L-2 (unless ML2 (setq ML2 (send command :description))))
	      (#\Mouse-M-1 (unless MM1 (setq MM1 (send command :description))))
	      (#\Mouse-M-2 (unless MM2 (setq MM2 (send command :description))))
	      (#\Mouse-R-1 (unless MR1 (setq MR1 (send command :description))))
	      ))))
      (when MR1 (push MR1 list) (push :MOUSE-R-1 list))
      (when MM2 (push MM2 list) (push :MOUSE-M-2 list))
      (when MM1 (push MM1 list) (push :MOUSE-M-1 list))
      (when ML2 (push ML2 list) (push :MOUSE-L-2 list))
      (when ML1 (push ML1 list) (push :MOUSE-L-1 list))
      (nconc list
	     (if MR1
		 '(:Mouse-R-2 "3System Menu*")
		 '(:Mouse-R-1 "3System Menu*"))))))


(defwhopper 4(live-listener-UCL-commands-mixin :who-line-documentation-string*) ()
  "2Returns a wholine list for this window.  If the mouse is over a sensitive item, then that item provides the documentation.
  If there is a docstring available from the automatic lisp-typein help, then we use that.  Otherwise, we use the documentation
  for no-items.*"
  (or (let* ((ucl:*enable-wholine-help-documentation-p* nil))	;1 First try with this NIL, so that we get the doc that comes out*
	(continue-whopper))					;1  of basic-mouse-sensitive-items...*
      (and ucl:*enable-wholine-help-documentation-p*		;1 Then try with it non-NIL, so that we can get the doc that comes*
	   (continue-whopper))					;1  out of lisp-help-mixin...*
      (live-listener-wholine-function)				;1 Otherwise, compute doc for click-on-no-item.*
      ))

(defwhopper 4(live-listener-UCL-commands-mixin :mouse-click)* (button x y)
  (let* ((item (send self :mouse-sensitive-item x y)))
    (cond ((and item (not (eq (tv:typeout-item-type item) :PRINTED-OBJECT)))
	   ;1; There is an item, but it is not one produced by 3PRIN1* or 3PRINC*.*
	   ;1; If there is an entry on our item-type alist for it, and there is a command table there, then invoke one of those commands.*
	   (let* ((list (assoc (tv:typeout-item-type item) tv:item-type-alist :test #'eq))
		  (comtab (fourth list)))
	     (when (and (symbolp comtab) (boundp comtab) (typep (symbol-value comtab) 'UCL:COMMAND-TABLE))
	       (setq comtab (symbol-value comtab)))
	     (if (or (typep comtab 'UCL:COMMAND-TABLE))
		 (let* ((commands (g-l-p (send comtab :commands))))
		   (send self :process-print-blip item button x y commands)
		   t)
		 (continue-whopper button x y))))
	  (t (continue-whopper button x y)))))


;1;;; Operations on UCL Commands.*

(defmethod 4(ucl:command :print-self)* (stream &rest ignore)
  "2Dammit.*"
  (format stream "3#<~S ~S ~D>*" (type-of self) (send self :name) (sys:%pointer self)))


(defflavor 4ll-command*
	   ((applicable-type 't))
	   (ucl:command)
  (:documentation :combination "2A flavor of command with an 3:APPLICABLE-TYPE* slot, and an 3:APPLICABLE?* method.*")
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defmethod 4(ll-command :applicable?)* (object)
  "2Whether this command may be applied to the given object.*"
  (and ucl::active?
       (ignore-errors (typep object applicable-type))))


(defmethod 4(ll-command :applicable-to-none?)* ()
  "2Whether this command may be applied to no objects (that is, to objects that don't exist).*"
  (and ucl::active?
       (let* ((at (send self :applicable-type)))
	 (or (null at)
	     (and (consp at)
		  (eq (car at) 'OR)
		  (member NIL at :test #'eq))))))
	 

(defmethod 4(live-listener-UCL-commands-mixin :commands-for-object)* (&optional (object nil objp) command-tables)
  "2Returns a list of the live-listener commands which are applicable to OBJECT.*"
  (let* ((result '()))
    (dolist (table-name (or command-tables mouse-command-tables))
      (let* ((table (symbol-value table-name))
	     (commands (g-l-p (send table :commands))))
	(dolist (command commands)
	  (when (if objp
		    (send command :send-if-handles :applicable? object)
		    (send command :send-if-handles :applicable-to-none?))
	    (push command result)))))
    (nreverse result)))


(defmethod 4(live-listener-UCL-commands-mixin :commands-for-no-object)* ()
  "2Returns a list of the live-listener commands which are applicable to no object, that is, to objects of type NIL.*"
  (send self :commands-for-object))

;1;;*
;1;; We define our own flavor of command, which allows a new initialization parameter, 3:APPLICABLE-TYPE*.*
;1;; However, the set of init keywords acceptable to 3DEFCOMMAND* is 6hard coded* into the function.  This is silly;*
;1;; the set of init keywords allowed should be determined by the flavor.  To get around this, we advise the*
;1;; errant function to allow this new keyword.*
;1;;*
(eval-when (load eval compile)

(sys:advise ucl:command-options-error-check :before ALLOW-APPLICABLE-TYPE-KEYWORD nil
  (setq arglist (copy-list arglist))
  (let* ((val (car (remf (car arglist) :applicable-type))))
    (when (and val (consp val) (eq (car val) 'QUOTE))
      (warn "3Do not quote the value of the *:APPLICABLE-TYPE3 keyword!*"))))
 ) ;1 closes eval-when*



;1;; Menus of commands.*
;1;;*
;1;; When the 5:menu-of-commands* method of5 live-listener-UCL-commands-mixin* is invoked, it will pop up a menu of the given commands*
;1;; (which are assumed to be applicable).  If a command is selected, then the 5:menu-invoke-command* method is called, in much the same*
;1;; manner as the 5:mouse-click-invoke-command* method is called from 5:mouse-click.**
;1;;*
;1;; This method is intended to be called from commands; so, if a command's name has a 4NOT-ON-LL-MENU* property, it will be excluded*
;1;; from the menu.  In this way, menu-invoking commands can avoid listing themselves.*
;1;;*
(defmethod 4(live-listener-UCL-commands-mixin :menu-of-commands)*
	   (command-list object &optional (object-supplied-p t) return-menu)
  "2Pops up a menu of the given commands operating on the given object.  If object-supplied-p is NIL, then that means that these commands
 should be invoked on no object.  If return-menu is non-NIL, then the menu descriptor list is returned rather than invoking the menu.
 Selections from the menu will invoke the commands synchonously or asynchronously, depending on what kind of window this is.*"
  (setq command-list (delete-if #'(lambda (x) (get x 'NOT-ON-LL-MENU))
				command-list
				:key #'(lambda (x) (ignore-errors (third (send x :definition))))))
  (let* ((menu (mapcar #'(lambda (command)
			   (list (send command :name) :VALUE command
				 :DOCUMENTATION (send command :documentation)))
		       command-list))
	 (type (if (consp object) 'LIST (type-of object)))
	 (label (cond ((not object-supplied-p) "Operations on no objects")
		      ((typecase object ;1;; - norvig*
			 ((or symbol fixnum float character bit complex) t)
			 ((or string bit-vector) (< (length object) 15))
			 (bignum (< (integer-length object) 45))
			 (ratio (< (+ (integer-length (numerator object))
				      (integer-length (denominator object))) 45)))
		       (format nil "Operations on the ~(~S~) ~S" type object))
		      ((typep object 'user-input-descriptor) "2Operations on this user input*")
		      (t (format nil "Operations on this ~(~S~)" type)))))
    (if return-menu	;1 They want to do it themselves...*
	(values label menu)
	(let* ((command (w:menu-choose menu :label label)))
	  (when command
	    (send self 4:menu-invoke-command* command object object-supplied-p))))))


(defmethod 4(live-listener-UCL-commands-mixin :menu-of-command-table)*
	   (command-table object &optional (object-supplied-p t) return-menu)
  "2Pops up a menu of all of the commands in the given command table (not just the applicable ones).  If object-supplied-p is NIL, then that
 means that these commands should be invoked on no object.  If return-menu is non-NIL, then the menu descriptor list is returned rather than
 invoking the menu.  Selections from the menu will be executed, synchronously or asynchronously as appropriate.*"
  (send self :menu-of-commands (g-l-p (send command-table :commands))
	object object-supplied-p return-menu))

(defmethod 4(live-listener-UCL-commands-mixin :menu-invoke-command)* (command object object-p)
  "2Given an instance of 3LL-COMMAND* which was produced via popup menu on the given mouse-sensitive item, invoke it.
  Do this by calling the :mouse-click-invoke-command method, passing :MENU as the button.*"
  (send self :mouse-click-invoke-command command object object-p :menu))




;1;;*
;1;; The 5:mouse-click-invoke-command* method of 5live-listener-UCL-commands-mixin* ``executes'' a command by deferring it.*
;1;; An 5:execute* blip is stuffed on the input buffer of the window, and when the command loop (any command loop) is ready,*
;1;; it will presumably handle the blip.*
;1;;*
(defmethod 4(live-listener-UCL-commands-mixin :mouse-click-invoke-command)* (command object object-p button)
  "2Invoke the command by stuffing an 5:execute* blip on this window's input buffer.*"
  (declare (ignore button))
  ;1;*
  ;1; Push an 5:execute* blip onto the input buffer.* 1For normal break loops, this will be read and processed by the 5sys:ucl-break* that is running.*
  ;1; Zmacs happens to use 5:execute* in it's command loop as well, so if Zmacs is in the Keyboard state, we still win.*
  ;1;*
  (send self :force-kbd-input
	(list :execute #'(lambda (window command object object-p)
			   (let* ((*terminal-io* window)
				  (def (send command :definition)))
			     (with-region-hilighted ((and (consp object-p) object-p) window)
			       (with-stack-list (args object)
				 (unless object-p (setq args nil))
				 (if (consp def)	;1 a method.*
				     (lexpr-send window (third def) args)
				     (apply def args))))))
	      self command object object-p)))


;1;; 4live-listener-synchronous-UCL-mixin**	1Provides a handler for the 5:execute* blip; makes user-typed things be mousable.*

(defflavor 4live-listener-synchronous-UCL-mixin *()
	   (live-listener-UCL-commands-mixin prin1-mousably-mixin)
  (:included-flavors w:basic-mouse-sensitive-items ucl:basic-command-loop)
;  (:required-flavors stream-mixin ucl:command-loop-mixin)
  (:default-init-plist
    :print-function 'live-listener-pprin1
    :mouse-command-tables '(*ll-lisp-command-table*))
  (:documentation :mixin
   "2Defines a :MOUSE-CLICK-INVOKE-COMMAND method which arranges for clicks on the mouse-sensitive items
 representing printed objects to be interpreted by the UCL process running in this window.*")
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(live-listener-synchronous-UCL-mixin :after :init*) (ignore)
  "2Enable typeout-execute blips, so that we can process mouse-click commands synchronously.*"
  (push '(:EXECUTE :HANDLE-EXECUTE) ucl:blip-alist))

(defmethod 4(live-listener-synchronous-UCL-mixin :handle-execute)* (&optional blip)
;  (format t "3~&ll-synch handling :execute*");1##*
  (unless blip (setq blip UCL:KBD-INPUT))
  (apply (second blip) (cddr blip)))


(defmethod (4live-listener-synchronous-UCL-mixin :after* :4execute-command*) ()
  "2If we have just executed a command issued from the mouse, and there is no further input pending, print a prompt.*"
  (when (eq UCL:KBD-INPUT 'UCL:COMMAND-CAME-FROM-QUEUE)
    (unless (or ucl:command-execution-queue (listen self))
      (send self :handle-prompt))))


;1;; Making user-typed things be mousable.*
;1;;*
;1;; We wrap the input-handler of UCL live listener windows so that user input is also mousable (giving commands on the input history, etc).*
;1;;*
(defmethod 4(live-listener-synchronous-UCL-mixin :handle-typein-input)* (&optional (untyi-first-char? t))
  "2This is just like 3(:method w:lisp-listener :handle-typein-input)*, except that we remember the cursor positions of the
 beginning and end of the user input, so that we can make a mouseable region around it.*"
  (when untyi-first-char?
    (tv:unread-any UCL:KBD-INPUT *standard-input*))
  (let* ((original-read ucl:read-function)
	 start-x start-y)
    (flet ((read-function ()
	     "2Snarf the values of *TV:RUBOUT-HANDLER-STARTING-2[*X2,*Y2] from within the scope of the rubout handler.*"
	     (unwind-protect (funcall (or original-read ucl:*default-read-function*))
	       (setq start-x tv:rubout-handler-starting-x
		     start-y tv:rubout-handler-starting-y))))
      (unwind-protect (progn (setf ucl:read-function #'read-function)
			     (in-font (self 2)
			       (ucl:handle-read-function)))
	(setf ucl:read-function original-read)
	(let* ((input-object (make-user-input-descriptor - (coerce tv:rubout-handler-buffer 'SIMPLE-STRING))))
	  (when (and start-x start-y)
	    (incf start-x tv:left-margin-size)
	    (incf start-y tv:top-margin-size)
	    (multiple-value-bind (end-x end-y final-index max-x)
				 (tv:sheet-compute-motion self start-x start-y tv:rubout-handler-buffer
							  0 nil nil 0 nil nil nil (aref tv:font-map 2)
							  tv:line-height)
	      (declare (ignore final-index))
	      (send self :item-for-output input-object
		    start-x start-y end-x end-y :USER-INPUT max-x)
	      ))))))
  (ucl:process-typein))


(defwhopper 4(live-listener-synchronous-UCL-mixin :break*) (&rest args)
  "2Run break loops in font #0.*"
  ;1; 5## *This isn't enough.  We intercept 5Break* and 5Meta-Break*, but not 5Control-Break* and 5Meta-Control-Break* - where do those come from?*
  (in-font (self 0) (lexpr-continue-whopper args)))

(defwhopper 4(live-listener-synchronous-UCL-mixin :restore-rubout-handler-buffer*) (string &optional pointer)
  "2When redrawing the rubout handler buffer after a break, draw it in font #2.*"
  (in-font (self 2) (continue-whopper string pointer)))



;1;; 4live-listener-asynchronous-mixin**		1Provides a printed-object click-handler that spawns a process to invoke the command.*


(defflavor 4live-listener-asynchronous-mixin* ()
	   (live-listener-UCL-commands-mixin prin1-mousably-mixin)
  (:default-init-plist
    :mouse-command-tables '(*ll-lisp-command-table*))
  (:documentation :mixin
   "2Defines a :MOUSE-CLICK-INVOKE-COMMAND method which arranges for clicks on the mouse-sensitive items
 representing printed objects to be executed in a spawned process.*")
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

;1;;*
;1;; The 5:mouse-click-invoke-command* method of 5live-listener-asynchronous-mixin* executes a command by spawning a process for it.*
;1;;*
(defmethod 4(live-listener-asynchronous-mixin :mouse-click-invoke-command)* (command object object-p button)
  "2Given an instance of 3LL-COMMAND* which was produced via the given button on the given mouse-sensitive item, invoke it.*"
  (declare (ignore button))
  (process-run-function (format nil "3Execute ~A*" (send command :name))
			#'(lambda (window command object object-p)
			    (let* ((*terminal-io* window)
				   (def (send command :definition)))
			      (with-region-hilighted ((and (consp object-p) object-p) window)
				(with-stack-list (args object)
				  (unless object-p (setq args nil))
				  (if (consp def) ;1 a method.*
				      (lexpr-send window (third def) args)
				      (apply def args))))))
			self command object object-p)
  t)



;1;; 4live-listener**				1A UCL Lisp Listener that executes mouse-commands synchronously.*

(defflavor 4live-listener*
	   ()
	   (prin1-mousably-mixin live-listener-synchronous-UCL-mixin w:lisp-listener)
  (:documentation :combination "2A flavor of Lisp Listener in which much output is mousable.*")
  (:default-init-plist :font-map *default-live-listener-font-map*)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


;1;; 4asynchronous-live-listener**		1A UCL Lisp Listener that executes mouse-commands asynchronously, by spawning*
;1;;*						1a background process.*

(defflavor 4asynchronous-live-listener*
	   ()
	   (prin1-mousably-mixin live-listener-asynchronous-mixin w:lisp-listener)
  (:documentation :combination "2A flavor of Lisp Listener in which much output is mousable; mouse-commands execute asynchronously.*")
  (:default-init-plist
    :print-function 'live-listener-pprin1
    :font-map *default-live-listener-font-map*
    )
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)





;1;; The command definitions for the 4*ll-lisp-command-table**.  *
;1;; These commands are appropriate for general Lisp objects, and will work either synchronously or asynchronously.*
;1;;*

(defprop 4:menu-of-all-applicable-commands *t not-on-ll-menu)  ;1 This brings up the menu, so don't show it.*

(defcommand 4(live-listener-UCL-commands-mixin :menu-of-all-applicable-commands*)
	    (&optional (object nil objp) return-menu)
  '(:names "2Menu of All Applicable Commands*"
	   :description "2Menu of operations on this object*"
	   :documentation "2Compute and pop up a menu of all commands which may operate on this object.*"
	   :keys #\Mouse-R :command-flavor LL-COMMAND :applicable-type (or T NIL))
  (let* ((commands (if objp
		       (send self :commands-for-object object)
		       (send self :commands-for-no-object))))
    (send self :menu-of-commands commands object objp return-menu)))


(defcommand 4(live-listener-UCL-commands-mixin :describe-printed-object*) (object)
  '(:description "2Describe this object*"
    :documentation "2Describe this object, with the output going to the window.*"
    :names "2Describe*" :keys #\Mouse-M :command-flavor 4LL-COMMAND*)
  (describe object)
  )


(defcommand 4(live-listener-UCL-commands-mixin :return-printed-object*) (object)
  '(:description "2Return this object*"
    :documentation "2This command causes OBJECT to be pprinted, and placed in **2 just as if it had been returned.*"
    :names "2Return*" :keys #\Mouse-L :command-flavor LL-COMMAND)
  (cond ((send self :operation-handled-p :return-object)
	 (send self :return-object object))
	(t (format t "2~&~A doesn't handle the :RETURN-OBJECT message.*" self)
	   (beep))))


(defcommand 4(live-listener-UCL-commands-mixin :inspect-printed-object*) (&optional (object nil objp))
  '(:description "2Inspect this object*"
    :documentation "2Examine this object in the inspector (using 3INSPECT**).*"
    :names "2Inspect*" :keys #\Mouse-M-2
    :command-flavor LL-COMMAND :applicable-type (or T NIL))
  (if objp (inspect* object) (inspect*)))

(defsubst 4flavor-name-p *(symbol) (get symbol 'sys:flavor))

(defcommand 4(live-listener-UCL-commands-mixin :inspect-flavor*) (&optional (object nil objp))
  '(:description "2Inspect this object's flavor*"
    :documentation "2Examine the flavor of this object (or the flavor named by this object, if this object is a symbol) in the Flavor Inspector.*"
    :names "2Inspect Flavor*"
    :applicable-type (or SYS:FLAVOR SYS:INSTANCE
			 (and SYMBOL (satisfies FLAVOR-NAME-P))
			 NIL)
    :command-flavor LL-COMMAND)
  (if objp (inspect-flavor object) (inspect-flavor)))


(defcommand 4(live-listener-UCL-commands-mixin :stuff-printed-object-input*) (object &optional string)
  '(:description "2Stuff input*" :documentation "2Stuffs the printed representation of OBJECT onto the input buffer.*"
3    *:names "2Stuff Input*" :command-flavor 4LL-COMMAND*)
  ;1;*
  ;1; If *OBJECT1 contains unreadable elements, then the rubout handler will issue warnings.  If the last character in the printed representation*
  ;1; of *OBJECT1 is a close-parenthesis, then it is not pushed onto the input buffer, as that might complete the input before the user wanted to.*
  ;1;*
  (unless string (setq string (write-to-string object :escape t :radix *read-base* :pretty nil :level nil :length nil
					       :case :downcase :gensym t :array t)))
  (when (char= #\) (char string (1- (length string))))
    (if (array-has-fill-pointer-p string)
	(decf (fill-pointer string))
	(adjust-array string (1- (length string)))))
1   *;1; We must do the buffer stuffing in another process, because otherwise long strings will overflow the buffer.  We need things
   *;1; to be taken off of the buffer by this process at the same time another process is stuffing them.*
  (process-run-function "3Buffer Stuffer*"
			#'(lambda (window str) (send window :force-kbd-input (the string str)))
			self string))


(defcommand 4(live-listener-UCL-commands-mixin :stuff-gensym-input*) (object)
  '(:description "2Stuff Gensym*"
    :documentation "2Gentemp a variable, and set it to OBJECT; then stuff the printed representation of the variable onto the input buffer.*"
    :names "2Stuff Gensym*" :command-flavor 4LL-COMMAND*)
  (let* ((var (gentemp "3TMP*")))
    (set var object)
    (send self :force-kbd-input #\Space)
    (send self :stuff-printed-object-input var)))


(defcommand 4(live-listener-UCL-commands-mixin :copy-to-kill-ring*) (object)
  '(:description "2Copy to Kill Ring*" :documentation "2Copies the printed representation of OBJECT onto the Zmacs kill ring.*"
3    *:names "2Copy to Kill Ring*" :command-flavor 4LL-COMMAND*)
  (let* ((zwei:*batch-undo-save* t)) ;1 don't try to record undo info - we're not in Zmacs.*
    (zwei:kill-interval (zwei:create-interval (prin1-to-string object)) nil nil t t)))


(defcommand 4(live-listener-UCL-commands-mixin :makunbound*) (object)
  '(:description "2Makunbound*" :documentation "2Make the indicated symbol be unbound.*" :names "2Makunbound*"
    :applicable-type (and SYMBOL (satisfies BOUNDP) (not (satisfies CONSTANTP)))
    :command-flavor LL-COMMAND)
  (makunbound object))


;1;; Commands on Functions*

(defcommand 4(live-listener-UCL-commands-mixin :trace*) (fn)
  '(:description "2Trace this function*" :documentation "2Trace this function.*" :names "2Trace Function*"
    :command-flavor LL-COMMAND :applicable-type (and SYMBOL FUNCTION))
  (format t "~&~{~a ~}traced~&" (sys:trace-1 fn)))

(defcommand 4(live-listener-UCL-commands-mixin :compile-function*) (fn)
  '(:description "2Compile this function*" :documentation "2Compile this function*" :names "2Compile Function*"
    :command-flavor LL-COMMAND :applicable-type (and SYMBOL FUNCTION))
  (format t "~&~a compiled.~&" (compile fn)))

(defcommand 4(live-listener-UCL-commands-mixin :disassemble*) (fn)
  '(:description "2Disassemble this function*"
    :documentation "2Show the assembly code for this function.*"
    :names "2Disassemble*" :command-flavor LL-COMMAND :applicable-type FUNCTION)
  (fresh-line)
  (disassemble fn))

(defcommand 4(live-listener-UCL-commands-mixin :arglist*) (fn)
  '(:description "2Show arguments of this function.*"
    :documentation "2Show the arguments for this function.*"
    :names "2Arglist*" :command-flavor LL-COMMAND :applicable-type (and SYMBOL FUNCTION))
  (zwei:print-arglist fn t) (terpri))


;1;; Commands on Pathnames.*
;1;; These commands work on pathnames, or on strings which are parsable as pathnames.*

(defun 4pathname-parsable *(string)
  "2T if string can be parsed into a valid pathname.*"
  (declare (string string))
  (and (not (string= "" string))
       ;1; ## There are some contortions in here because apparently calling 5parse-namestring* on arbitrary strings*
       ;1; ## can make the namespace freak - starts spitting out notifications like mad, or locks up the mouse...*
       (< (length string) 128)
       (some #'alpha-char-p string)
       (every #'graphic-char-p string)
       (let* ((colon (position #\: string :test #'char-equal))
	      (sp (and colon (position-if #'(lambda (x) (or (char-equal x #\space) (char-equal x #\tab)))
					  string :end colon))))
	 (and colon (not sp) (< colon 30)
	      (condition-call () (parse-namestring string)
		(t nil))))))


(deftype 4pathname-string *() '(and string (satisfies pathname-parsable)))

(defcommand 4(live-listener-UCL-commands-mixin :compile-file*) (file)
  '(:description "2Compile this file*" :documentation "2Compile this file*" :names "2Compile*"
    :command-flavor LL-COMMAND :applicable-type (or PATHNAME PATHNAME-STRING))
  (compile-file file))

(defcommand 4(live-listener-UCL-commands-mixin :load*) (file)
  '(:description "2Load this file*" :documentation "2Load this file*" :names "2Load*"
    :command-flavor LL-COMMAND :applicable-type (or PATHNAME PATHNAME-STRING))
  (load file))

(deftype 4function-spec *() '(or function (satisfies sys:function-spec-p)))

(defcommand 4(live-listener-UCL-commands-mixin :ed*) (&optional object)
  '(:description "2Edit the definition of this object*"
    :documentation "2Invoke the editor on the given object by calling ED with it.*" :names "2Invoke Editor*"
    :applicable-type (or SYMBOL FUNCTION-SPEC PATHNAME PATHNAME-STRING NIL)
    :command-flavor LL-COMMAND)
  (ed object))


(defvar 4*ll-describe-file-props-to-lose* *'(:byte-size :length-in-blocks :length-in-bytes :author :creation-date
					   :not-backed-up :characters :dired-echo)
  "2The :describe-file command prints a Dired-like line describing the file, followed by a table of the file's properties.
  This variable enumerates the properties which should be excluded from that table, since they are presented in a different form 
  in the Dired string.*")

(defcommand 4(live-listener-UCL-commands-mixin :describe-file*) (object)
  '(:description "2Describe a file on disk.*" :names "2Describe file*"
    :documentation "2Print some useful information about the file which this pathname represents.*"
    :applicable-type (or PATHNAME PATHNAME-STRING) :command-flavor LL-COMMAND)
  (setq object (merge-pathnames (pathname object)
				(make-pathname :defaults object :name :WILD :type :WILD :version :NEWEST)))

  (let* ((wild-p (send object :wild-p))
	 (list (ignore-errors (fs:directory-list object)))
	 (stream-width (truncate (send self :width) (send self :char-width))))
    
    (cond ((null (cdr list))
	   (if wild-p
	       (format t "3~&There are no matches to the wildcarded pathname ~A.~%*" object)
	       (format t "3~&The file ~A does not exist.~%*" object)))
	  (t
	   (when wild-p
	     (format t "3~&Pathname ~A matches ~D file~:P.~%*" object (length (cdr list))))
	   (dolist (desc (cdr list))
	     (using-resource (stream-list-buffer zwei:stream-list-buffer-pool stream-width)
	       (let* ((output-string (zwei:default-list-one-file
				       desc stream-list-buffer :string-for-printing
				       stream-width))
		      (props (copy-list (cdr desc))))
		 (fresh-line)
		 (multiple-value-bind (sx sy) (send self :read-cursorpos)
		   (multiple-value-bind (dx dy) (tv:sheet-compute-motion self sx sy output-string)
		     (princ (string-trim '(#\Space #\Tab) output-string))
		     (send self :item-for-output desc sx sy dx dy)))
		 (terpri)
		 (dolist (prop 4*ll-describe-file-props-to-lose**)
		   (remf props prop))
		 (when (zerop (getf props :version-limit 0)) (remf props :version-limit))
		 (when props
		3   *;(format t "3  Additional properties of file ~A:~%*" (car desc))
		    (send self :show-symbol-plist props 4 t)))))))
    ))


(defcommand 4(live-listener-UCL-commands-mixin :describe-file-stream*) (object)
  '(:description "2Describe Open File*" :names "2Describe Open File*"
    :documentation "2Print some useful information about this stream, which is open to a file.*"
    :keys #\Mouse-M-1 :command-flavor LL-COMMAND
    :applicable-type SYS:FILE-STREAM-MIXIN)
  (multiple-value-bind (pathname direction bytes percent) (send object :who-line-information)
    (format t "3~&This ~S is open for ~(~A~) to the file ~A~%*" (type-of object) direction pathname)
    (format t "3Byte size: ~D; character translation: ~A.~%*"
	    (send object :get :byte-size)
	    (if (send object :get :characters) "3Yes*" "3No*"))
    (case direction
      (:INPUT  (format t "3~:D byte~:P have been read (~D percent).~%*" bytes percent))
      (:OUTPUT (format t "3~:D byte~:P have been written.~%*" bytes)))
    ))

;1;; Two pretty bogus types - something is of type 5maybe-address-string* if it is a string and contains an at-sign.*
;1;; Something is of type 5maybe-net-host-name* if it is a string ending with 5.EDU*, 5.COM*, 5.UUCP*, 5.ARPA*, or 5.CS.NET*.*
;1;; These are not supposed to be accurate; they are just a first guess at whether a string might be intended for a *
;1;; particular purpose.  If we guess wrong, no harm done.*
;1;;*
(deftype 4MAYBE-ADDRESS-STRING *()
  "2If something is of this type, it is a string which may be intended for use as a mail address (it has an at-sign in it).*"
  '(and string (satisfies maybe-address-string)))

(deftype 4MAYBE-NET-HOST-NAME *()
  "2If something is of this type, it is a string which might be the name of a host (it ends with .edu, etc).*"
  '(and string (satisfies maybe-net-host-name)))

(defun 4maybe-address-string *(string)
  (position #\@ (the string string) :test #'char-equal))

(defun 4maybe-net-host-name *(string)
  (declare (string string) (optimize speed (safety 0)))
  (and (not (maybe-address-string string))
       (let* ((l (length string)))
	 (dolist (suffix '("3.edu*" "3.com*" "3.arpa*" "3.mil*" "3.gov*" "3.cs.net*" "3.uucp*"))
	   (declare (simple-string suffix))
	   (let* ((l2 (length suffix)))
	     (when (and (> l l2) (string-equal string suffix :start1 (- l l2) :end1 l :start2 0 :end2 l2))
	       (return suffix)))))))


(defcommand 4(live-listener-UCL-commands-mixin :describe-host*) (host)
  '(:description "2Describe Host*" :names "2Describe Host*"
    :documentation "2Print some useful information about the host which this host-object represents.*"
    :keys #\Mouse-M-1 :command-flavor LL-COMMAND
    :applicable-type (or NET:HOST PATHNAME MAIL:ADDRESS MAYBE-NET-HOST-NAME MAYBE-ADDRESS-STRING))
  (when (typep host 'mail:address) (setq host (send host :address-string)))
  (cond ((pathnamep host) (setq host (pathname-host host)))
	((stringp host)
	 (setq host (net:parse-host (subseq host (1+ (or (position #\@ host) -1)))))))
  (format t "3~&Host ~A, in domain ~A of site ~A*" host (send host :domain) (send host :site))
  (format t "3~&System Type: ~A*" (send host :system-type))
  (format t "3~&Addresses: *")
  (dolist (cons (send host :addresses))
    (format t "3~13t~8A *" (car cons))
    (funcall (get (car cons) :network-address-printer) (cadr cons) *standard-output* nil)
    (terpri))
  (format t "3~&Services: ~{~13t~S~%~}*" (send host :service-list))
  (format t "3~&Attributes:~%*")
  (send self :show-symbol-plist (send host :host-attributes) 13 t)
  (format t "3~&Properties:~%*")
  (send self :show-symbol-plist (send host :property-list) 13 t)
  )


(defcommand 4(live-listener-UCL-commands-mixin :finger-host*) (host)
  '(:description2 "Finger Host*" :names "2Finger Host*"
    :documentation "2Open a finger connection to this host to see who is logged on.*"
    :command-flavor LL-COMMAND
    :applicable-type (or NET:HOST PATHNAME MAIL:ADDRESS MAYBE-NET-HOST-NAME MAYBE-ADDRESS-STRING))
  (when (typep host 'mail:address) (setq host (send host :address-string)))
  (cond ((pathnamep host) (setq host (pathname-host host)))
	((and (stringp host) (maybe-net-host-name host)) (setq host (net:parse-host host)))
	((stringp host) (setq host (net:parse-host (subseq host (1+ (or (position #\@ host) -1)))))))
  (let* ((finger-args (if (eq (send host :system-type) :LISPM)
			  "3/W*"
			  "")))
    (format t "3~&Fingering ~A@~A...*" finger-args host)
    (with-open-stream (finger-stream (send host :show-users :connect finger-args))
      (fresh-line)
      (if (typep finger-stream 'condition)
	  (princ finger-stream)
	  (sys:stream-copy-until-eof finger-stream *standard-output*)))))


(defcommand 4(live-listener-UCL-commands-mixin :finger-address*) (host)
  '(:description2 "Finger User*" :names "2Finger User*"
    :documentation "2Open a finger connection to this host to see who this addressee is.*"
    :command-flavor LL-COMMAND
    :applicable-type (or NET:HOST PATHNAME MAIL:ADDRESS MAYBE-ADDRESS-STRING))
  (let* ((user ""))
    (when (typep host 'mail:address) (setq host (send host :address-string)))
    (cond ((pathnamep host) (setq host (pathname-host host)))
	  ((stringp host) (let* ((pos (position #\@ host)))
			    (setq user (if pos (subseq host 0 pos) "")
				  host (net:parse-host (subseq host (if pos (1+ pos) 0)))))))
    (format t "3~&Fingering ~A@~A...*" user host)
    (with-open-stream (finger-stream (send host :show-users :connect user))
      (fresh-line)
      (if (typep finger-stream 'condition)
	  (princ finger-stream)
	  (sys:stream-copy-until-eof finger-stream *standard-output*)))))


;1;; Commands on various other things...*

(defcommand 4(live-listener-UCL-commands-mixin :show-hash-table-contents*) (object)
  '(:description "2Show this hash table's contents*" :names "2Show Contents*" :keys #\Mouse-M-2
    :documentation "2Print the contents of the indicated hash table on the window.*" 
    :applicable-type HASH-TABLE :command-flavor LL-COMMAND)
  (check-type object hash-table)
  (format t "3~&~S has ~D element~:P.~%*" object (hash-table-count object))
  (let* ((tab 35)
	 (tab-pixels (* tab (tv:sheet-char-width self)))
	 (w (- (send self :size-in-characters) tab 1)))
    (flet ((dump (key val &rest ignore)
	     (prin1 key)
	     (format t "3~vT*" tab)
	     (live-listener-pprin1 val tab-pixels w)))
      (maphash #'dump object))
    ))


(defcommand 4(live-listener-UCL-commands-mixin :show-symbol-plist*) (object &optional (indent 1) quiet)
  '(:description "2Show properties*" :names "2Show Properties*"
    :documentation "2Prints the contents of the indicated object's property list on the window.*"
    :applicable-type (or SYMBOL SYS:PROPERTY-LIST-MIXIN)
    :command-flavor LL-COMMAND)
  (setq indent (make-string indent :initial-element #\Space))
  (let* ((plist (if (consp object) object (symbol-plist object))))  ;1 Symbol-plist works on instances as well.*
    (unless quiet (format t "3~&~D propert~:@P:~%*" (round (length plist) 2)))
    (let* ((tab 35)
	   (tab-pixels (* tab (tv:sheet-char-width self)))
	   (w (- (send self :size-in-characters) tab 1)))
      (do* ((rest plist (cddr rest)))
	   ((null rest))
	(let* ((key (car rest))
	       (val (cadr rest)))
	  (princ indent)
	  (prin1 key)
	3  *(format t "3~vT*" tab)
	   (live-listener-pprin1 val tab-pixels w))))
    ))


(defcommand 4(live-listener-UCL-commands-mixin :print-integer-as-time*) (object &optional (print object))
  '(:description "2Print as3 *Universal Time*" :names "2Print as Time*"
    :documentation "2Interpret this integer as a Universal Time.*"
    :applicable-type INTEGER	;1 not necessarily a fixnum.*
    :command-flavor LL-COMMAND)
  (fresh-line) (princ print) (princ "3 = *")
  (time:print-universal-time object)
  (fresh-line))

(defcommand 4(live-listener-UCL-commands-mixin :print-integer-as-time-interval*) (object &optional (print object))
  '(:names "2Print as time interval (seconds)*" :description "2Print as time interval (seconds)*"
    :documentation "2Interpret this integer as a time interval in seconds.*"
    :applicable-type INTEGER	;1 not necessarily a fixnum.*
    :command-flavor LL-COMMAND)
  (fresh-line) (princ print) (princ "3 = *")
  (if (minusp object) (princ "3minus *"))
  (time:print-interval-or-never (abs object))
  (fresh-line))

(defun 4ticks-to-ut *(ticks)
  "2Given a (relatively recent) time in 60ths of a second, return the corresponding Universal Time in seconds.*"
  (let (now now60)
    (without-interrupts
      (setq now (get-universal-time)
	    now60 (time:time)))
    (let* ((offset60 (time:time-difference now60 ticks)))
      (- now (round offset60 60)))))

(defcommand 4(live-listener-UCL-commands-mixin :print-fixnum-as-time-60ths*) (object)
  '(:description "2Print as3 *time (ticks)*" :names "2Print as Time (ticks)*"
    :documentation "2Interpret this integer as a (relatively recent) time from the 60th-second clock.*"
    :applicable-type (INTEGER 0 #.MOST-POSITIVE-FIXNUM)	;1 The 60-clock always returns positive fixnums.*
    :command-flavor LL-COMMAND)
  (send self :print-integer-as-time (round object 60) object))

(defcommand 4(live-listener-UCL-commands-mixin :print-fixnum-as-time-interval-60ths*) (object)
  '(:names "2Print as time interval (ticks)*" :description "2Print as time interval in 60ths of a second*"
    :documentation "2Interpret this integer as a time interval in ticks, or 60ths of a second.*"
    :applicable-type FIXNUM :command-flavor LL-COMMAND)
  (send self :print-integer-as-time-interval (round object 60) object))

(defcommand 4(live-listener-UCL-commands-mixin :print-integer-as-IP-address*) (object &optional (print object))
  '(:names "2Print as Internet address*" :description "2Print as Internet address*"
    :documentation "2Interpret this integer as a 32-bit IP address.*"
    :applicable-type (INTEGER #.(expt 2 24))	;1 larger than address 1.0.0.0.*
    :command-flavor LL-COMMAND)
  (fresh-line) (princ print) (princ "3 = *")
  (net:print-network-address object :IP self)
  (terpri))


;1;; The command definitions for the 4*ll-history-command-table**.  *
;1;; These commands apply to the user input history.  They go in their own command table because they may be useful in an application *
;1;; for which the general Lisp-object commands are inappropriate.*
;1;;*

(defcommand 4(live-listener-synchronous-UCL-mixin :mouse-yank-previous-input*) (object)
  '(:names "2Stuff Previous Input*"
	   :description "2Stuff previous input*"
	   :documentation "2Insert a previously-typed form as if it were being typed now.*"
	   :keys #\Mouse-L :command-flavor LL-COMMAND
	   :applicable-type 4USER-INPUT-DESCRIPTOR*)
  (send self :stuff-printed-object-input object (user-input-descriptor-printed-representation object)))


(defcommand 4(live-listener-synchronous-UCL-mixin :mouse-return-previous-input*) (object)
  '(:names "2Return Previous Input*" :keys #\Mouse-M
    :description "2Return previous input unevaluated*" :documentation "2Return as a value a previously-typed form.*"
    :command-flavor LL-COMMAND :applicable-type 4USER-INPUT-DESCRIPTOR*)
  (send self :return-printed-object (user-input-descriptor-object object)))


(defprop 4:menu-of-all-input-element-commands *t not-on-ll-menu)  ;1 This brings up the menu, so don't show it.*

(defcommand 4(live-listener-synchronous-UCL-mixin :menu-of-all-input-element-commands*) (input-element)
  '(:names "2Menu of All Applicable Input Element Commands*"
    :description "2Menu of operations on this object*"
    :documentation "2Compute and pop up a menu of all commands which may operate on this object.*"
    :keys #\Mouse-R :command-flavor LL-COMMAND
    :applicable-type USER-INPUT-DESCRIPTOR)
  (send self :menu-of-command-table *ll-history-command-table* input-element))


;1;;; The Printed-Objects command tables.*
;1;;; *
;1;;; Note that the order of the commands in these command tables is important.  *
;1;;; Commands which apply to more restricitve types should preceed commands which apply to less restrictive types.*
;1;;; This is because, in the set of applicable commands, the frontmost command which wants a mouse button gets it.*
;1;;; So, if both 5Describe Object* and 5Describe File Stream* are applicable, 5Describe File Stream* should be first.*
;1;;;*

(build-command-table '*ll-lisp-command-table* 'live-listener-UCL-commands-mixin
  '(
    :menu-of-all-applicable-commands	;1 bound to Mouse-R-1*
    :describe-file-stream		;1 bound to Mouse-M-1*
    :describe-host			;1 bound to Mouse-M-1*
    :describe-printed-object		;1 bound to Mouse-M-1*
    :show-hash-table-contents		;1 bound to Mouse-M-2*
    :inspect-printed-object		;1 bound to Mouse-M-2*
    :return-printed-object		;1 bound to Mouse-L-1*
    :stuff-printed-object-input

    :finger-host
    :finger-address
    :describe-file
    :stuff-gensym-input
    :copy-to-kill-ring
    :makunbound
    :ed
    :compile-file
    :load
    :trace
    :disassemble
    :compile-function
    :arglist
    :show-symbol-plist
    :inspect-flavor
    :print-integer-as-time
    :print-integer-as-time-interval
    :print-fixnum-as-time-60ths
    :print-fixnum-as-time-interval-60ths
    :print-integer-as-IP-address
    )
  :init-options '(:documentation "2The set of Live Listener commands that are useful for manipulating general Lisp objects.*"
		  :table-sorts		(:COMMAND nil :NAME ucl:sort-name-table :KEY ucl:sort-key-table)
		  :command-lookup-fun	ucl:unsorted-lookup-command
		  ))


;1;; The Input History command table.*
;1;;*
;1;; Commands on this command table are applicable to typeout-items of type 3:USER-INPUT*, which are added for user-typed expressions.*
;1;;*
(build-command-table '*ll-history-command-table* 'live-listener-synchronous-UCL-mixin
  '(
    :menu-of-all-input-element-commands	;1 bound to Mouse-R-1*
    :mouse-return-previous-input  	;1 bound to Mouse-M-1*
    :mouse-yank-previous-input	  	;1 bound to Mouse-L-1*
    tv:rh-com-menu-pop-up-input-ring
    )
  :init-options '(:documentation "2The set of Live Listener commands that are applicable to the input history.*"
		  :table-sorts		(:COMMAND nil :NAME ucl:sort-name-table :KEY ucl:sort-key-table)
		  :command-lookup-fun	ucl:unsorted-lookup-command
		  ))


;1;; Making 5Prompt-and-Read* use the mouse on UCL Live Listeners.*
;1;;*

(defwhopper 4(live-listener-synchronous-UCL-mixin :mouse-sensitive-item)* (x y)
  "2If the RESTRICT-TO-TYPE instance variable is not T, then don't let a mouse-sensitive item be hilighted unless it satisfies that type.*"
  (let* ((item (continue-whopper x y)))
    (cond ((or (null item)
	       (eq restrict-to-type 'T)
	       (typep (tv:typeout-item-item item) restrict-to-type))
	   item)
	  (t nil))))

(defmethod 4(live-listener-synchronous-UCL-mixin :mouse-read-type)* (type)
  "2Returns some object, which satisfies the given type specifier.
  This object will be selected from among the mousable objects printed on the screen.
  Only those objects which satisfy the type will be selectable.
  This method does not allow keyboard typein at all.*"
  (let* ((old restrict-to-type))
    (unwind-protect
	(tv:with-mouse-grabbed
	  (setq restrict-to-type type)
	  (loop
	    (process-wait "3Buttons Up*" #'(lambda () (zerop (tv:mouse-buttons t))))
	    (multiple-value-bind (dx dy down up x y) (tv:mouse-input t)
	      (declare (ignore dx dy down up))
	      (let* ((buttons (tv:mouse-buttons nil)))
		(send self :mouse-moves x y)
		(cond ((logbitp 0 buttons) ;1 Button 0 down.*
		       (let* ((item (send self :mouse-sensitive-item x y)))
			 (if item
			     (return (tv:typeout-item-item item))
			     (beep))))
		      ((plusp buttons)     ;1 Some other button down.*
		       (beep)))))))
      (setq restrict-to-type old))))


(defcommand 4(live-listener-synchronous-UCL-mixin :read-or-click-complete)* (object)
  '(:description "2Select this object*" :names "2Select this object*"
    :documentation "2Select this object.  This is the only command installed when within the scope of 3:READ-AND-CLICK*.*"
    :keys #\Mouse-L :command-flavor LL-COMMAND)
  (when (user-input-descriptor-p object) (setq object (user-input-descriptor-object object))) ;1 ## hack.*
  (throw 'MOUSE-SELECT object))

(defvar 4*read-or-click-command-table*)*
(build-command-table '4*read-or-click-command-table** 'live-listener-synchronous-UCL-mixin
  '(:read-or-click-complete)
  :init-options '(:documentation "2This is the command table in place during the scope of :READ-OR-CLICK;
 it contains only one command, which is bound to Mouse-L, and throws to a catcher in the method.*"))



;(defmethod 4(live-listener-synchronous-UCL-mixin :preempted-handle-blip)* (blip)
;  "2If BLIP is a list of the form 3( :typeout-execute* <ucl-command-instance> 3[* <args> 3]* )*, then we invoke the given command with*
;2  the given arguments just as the UCL would do; it would be cleaner if we could just stuff this blip onto the command-execution-queue, but *
;2  this method is for use in cases where the command-execution-queue is not usable because we are already within the scope of a read.*
;2  In other words, this is a hack.*"
;  (when (and (consp blip)
;	     (eq (car blip) :typeout-execute)
;	     (typep (second blip) 'LL-COMMAND))
;    (let* ((command (second blip))
;	   (args (cddr blip))
;	   (def (send command :definition)))
;      (setq tv:old-typeahead nil)
;      (send self :clear-input)
;      (if (consp def)
;	  (lexpr-send self (third def) args)
;	  (apply def args)))
;    t))

(defmethod 4(live-listener-synchronous-UCL-mixin :preempted-handle-blip)* (blip)
  "2If BLIP is a list of the form 3( :execute* <function> 3[* <args> 3]* )*, then we invoke it.  It would be cleaner if we could just stuff
  this blip onto the command-execution-queue, but this method is for use in cases where the command-execution-queue is not usable
  because we are already within the scope of a read.*
 2In other words, this is a hack.*"
  (when (and (consp blip) (eq (car blip) :execute))
    (send self :handle-execute blip)
    t))

(defun 4allow-click-of-type *(type read-function &optional (command-tables '(4*read-or-click-command-table**)))
  "2  Given a function of one argument, an input stream, read some input by calling that function.
  The user may also click on any mousable item on the screen which is of the type TYPE; clicking on
  an item causes a throw out of the passed-in read-function.
  The first value returned is the input from the user; the second value is how the input was obtained.
  This is :MOUSE if the user clicked on something, and may be other values, such as :DEFAULT,
  depending on the keyword passed to PROMPT-AND-READ.*"
  (declare (:self-flavor live-listener-synchronous-UCL-mixin)
	   (values object read-type-flag))
  (let* ((old-type restrict-to-type)
	 (old-tables mouse-command-tables)
	 (old-com ucl:command-entry)
	 (typein-completed nil)
	 blip read-type-flag)
    ;1; ## wholine doc hack.  Lose the Lisp-Help-Mixin documentation when inside of Fquery or something.*
    (when (typep self 'tv:lisp-help-mixin) (sys:set-in-instance self 'tv:who-line-string nil))
    
    (setq blip
	  (catch 'MOUSE-SELECT
	    (unwind-protect
		(progn
		  (setq restrict-to-type type
			mouse-command-tables command-tables)
		  (loop
		    (multiple-value-setq (blip read-type-flag) (funcall read-function self))
		    (unless (send self :preempted-handle-blip blip)
		      (setq typein-completed t)
		      (return blip))))
	      (setq restrict-to-type old-type
		    mouse-command-tables old-tables
		    ucl:command-entry old-com))))
    (unless typein-completed (setq read-type-flag :MOUSE))
    (values blip read-type-flag)))


(defun 4live-listener-preemptable-rubout-handler *()
  "2This is an internal function used by the :PROMPT-AND-READ method of LIVE-LISTENER-SYNCHRONOUS-UCL-MIXIN windows; it is just like
  the function *TV:ALTERNATE-RUBOUT-HANDLER2, except that it guarentees that all input is preemptable.*"
  (declare (:self-flavor live-listener-synchronous-UCL-mixin))
  (let* ((already-preemptable (assoc :preemptable tv:rubout-handler-options :test #'eq)))
    
    (if already-preemptable
	;1; If it's preemptable already, we don't have to do anything special.*
	(tv:alternate-rubout-handler)
	;1;*
	;1; Otherwise, we have to bend over backwards.*
	;1;*
	;1; o  First we cons the preemptable option onto the rubout handler options list.*
	;1; o  then we invoke *TV:ALTERNATE-RUBOUT-HANDLER1.*
	;1; o  If *TV:ALTERNATE-RUBOUT-HANDLER1 returns a blip that we know how to handle (meaning that this blip came from a mouse click)*
	;1;    then we handle it.  The handling of the blip will probably cause a throw; if it doesn't, then we go back and call ARH again.*
	;1; o  If ARH simply returned its value, then we return it.*
	;1; o  If ARH returned its value by throwing to *TV:RETURN-FROM-RUBOUT-HANDLER1, then we must throw the value to that tag again.*
	;1;*
	(with-stack-list* (tv:rubout-handler-options '(:preemptable :preemptable) tv:rubout-handler-options)
	  (loop
	    (let* ((completed-normally nil)
		   (result (catch 'TV:RETURN-FROM-RUBOUT-HANDLER
			     (prog1 (tv:alternate-rubout-handler)
				    (setq completed-normally t)))))
	      (if (consp result)
		  (or (send self :preempted-handle-blip result)
		      (return result))
		  (if completed-normally
		      (return result)
		      (throw 'TV:RETURN-FROM-RUBOUT-HANDLER result)))))))))


(defmethod 4(live-listener-synchronous-UCL-mixin :prompt-and-read)* (option format-string &rest format-args)
  "2Calling the PROMPT-AND-READ function in a Live-Listener will invoke this method.  Typein is just as for
  PROMPT-AND-READ, but certain items on the screen will be mousable as well; clicking on them will return them from
  the PROMPT-AND-READ function.*"
  (let* ((option-type (if (consp option)
			  (car option)
			  option))
	 (function (get option-type 'sys:prompt-and-read-function))
	 (sys:prompt-and-read-format-string format-string)
	 (sys:prompt-and-read-format-args format-args)
	 ;1;*
	 ;1; We need a new property associated with the prompt-and-read keywords - a normal typep type-specifier, so that*
	 ;1; the mouse handler can decide which items should be mousable.  These are defined below.*
	 (click-typespec (get option-type 'prompt-and-read-mousable-type nil))
	 ;1;*
	 ;1; We need a slightly different rubout handler, to assure that mouse-clicks will preempt typein.*
	 (TV:STREAM-MIXIN-RUBOUT-HANDLER '4live-listener-preemptable-rubout-handler*)
	 )
    ;1;*
    ;1; If the option was specified as 3(:NUMBER :OR-NIL T)* instead of simply 3:NUMBER*, then the mouse typespec should*
    ;1; be 3(OR NULL NUMBER)* instead of simply 3NUMBER*.*
    ;1;*
    (when (and (consp option) (getf (cdr option) :or-nil))
      (setq click-typespec `(OR NULL ,click-typespec)))
    
    (allow-click-of-type click-typespec
      #'(lambda (stream)
	  (cond ((get option-type 'sys:prompt-and-read-no-rubout-function)
		 (send (get option-type 'sys:prompt-and-read-no-rubout-function) option stream))
		((null function)
		 (ferror () "~S is not a known PROMPT-AND-READ option keyword." option-type))
		(t
		 (let ((rh-options (get option-type 'sys:prompt-and-read-rubout-options
					'((:prompt sys:prompt-and-read-prompt-function)
					  ;(:activation zlc:memq (#\End #\Newline))
					  (:activation = #\Newline)
					  ))))
		   (if sys:rubout-handler-options
		       (setf rh-options (cons sys:rubout-handler-options rh-options)))
		   (push '(:preemptable t) rh-options)
		   (send stream :rubout-handler rh-options function option *query-io*))))
	  ))))


;1;;*
;1;; Converting prompt-and-read keywords to real type specifiers.*
;1;;*

(defprop :string		string			prompt-and-read-mousable-type)
(defprop :string-or-nil		(or string null)	prompt-and-read-mousable-type)
(defprop :character-list	string			prompt-and-read-mousable-type)
(defprop :string-trim		string			prompt-and-read-mousable-type)
(defprop :delimited-string	string			prompt-and-read-mousable-type)
(defprop :delimited-string-or-nil string		prompt-and-read-mousable-type)

(defprop :pathname		pathname		prompt-and-read-mousable-type)
(defprop :pathname-or-end	pathname		prompt-and-read-mousable-type)
(defprop :pathname-or-nil	(or pathname null)	prompt-and-read-mousable-type)

(defprop :number		number			prompt-and-read-mousable-type)
(defprop :integer		integer			prompt-and-read-mousable-type)
(defprop :small-fraction	(or (float 0.0 1.0) (rational 0 1))	prompt-and-read-mousable-type)

(defprop :read			t			prompt-and-read-mousable-type)
(defprop :expression		t			prompt-and-read-mousable-type)
(defprop :expression-or-end	t			prompt-and-read-mousable-type)
(defprop :eval-read		t			prompt-and-read-mousable-type)
(defprop :eval-form		t			prompt-and-read-mousable-type)
(defprop :eval-sexp		t			prompt-and-read-mousable-type)
(defprop :eval-read-or-end	t			prompt-and-read-mousable-type)
(defprop :eval-form-or-end	t			prompt-and-read-mousable-type)
(defprop :eval-sexp-or-end	t			prompt-and-read-mousable-type)

(defprop :boolean		(member T NIL)		prompt-and-read-mousable-type)
(defprop :character		character		prompt-and-read-mousable-type)

;1;; Don't know what to do about:*  :string-list, :date, :assoc, :choose


;1;; Making 5Fquery* use the mouse on UCL Live Listeners.*
;1;;*

(defwhopper 4(live-listener-synchronous-UCL-mixin :loop)* ()
  (send self :set-current-font 0)
  (condition-bind ((:fquery #'(lambda (condition window)
				(lexpr-send window :fquery (get condition :options)
					    (send condition :format-string) (send condition :format-args)))
			    self))
    (with-stack-list* (tv:kbd-intercepted-characters '(#\Break live-listener-break)
						     '(#\Meta-Break live-listener-error-break)
						     tv:kbd-intercepted-characters)
      (continue-whopper))))

(defun 4live-listener-break* (char &rest ignore)
  (in-font (tv:selected-window 0)
    (tv:kbd-intercept-break char)))

(defun 4live-listener-error-break* (char &rest ignore)
  (in-font (tv:selected-window 0)
    (tv:kbd-intercept-error-break char)))


(defcommand 4(live-listener-synchronous-UCL-mixin :fquery-menu)* (&rest ignore)
  '(:description "2Fquery command menu*" :names "2Fquery command menu*"
    :documentation "2Pop up a menu of responses to an FQUERY.*"
    :active-in-display? nil :keys #\Mouse-R
    :applicable-type (or T NIL) :command-flavor LL-COMMAND)
  (declare (special fquery-options))
  (lexpr-send self :fquery-menu-internal fquery-options format:fquery-format-string format:fquery-format-args))


(defmethod 4(live-listener-synchronous-UCL-mixin :fquery-menu-internal)* (options format-string &rest format-args)
  (let* ((readline-p (eq :READLINE (getf options :TYPE)))
	 (format-control (if readline-p
			     "3Select an answer.~%This is the same as typing ``~A'' followed by a newline.*"
			     "3Select an answer.~%This is the same as typing ~@:C.*")))
    (w:menu-choose (mapcar #'(lambda (choice)
			       (let* ((item-name (if (consp (car choice))
						     (second (car choice))
						     (second choice)))
				      (text (second choice)))
				 (list item-name
				       :KBD (if readline-p (string-append text #\Newline) text)
				       :DOCUMENTATION
				       (format nil format-control text))))
			   (or (getf options :choices) sys:y-or-n-p-choices))
		   :label (if format-string (apply #'format nil format-string format-args) nil)
		   )))


(defvar 4*ll-fquery-command-table*)*
(build-command-table '4**ll-fquery4-command-table** 'live-listener-synchronous-UCL-mixin
  '(:fquery-menu)
  :init-options '(:documentation "2This is the command table in place during the scope of 3FQUERY*;
 it contains only one command, which is bound to Mouse-R, and stuffs characters on the input buffer.*"))


(defmethod 4(live-listener-synchronous-UCL-mixin :tyi*) (&optional ignore)
  "2This is just like 3(:method stream-mixin :tyi)* except that command blips are transparently handled.*"
  (do (ch)
      (())
    (cond ((numberp (setq ch (send self :any-tyi)))
	   (return ch))
	  ((characterp ch)
	   (return (char-int ch)))
	  ((and (consp ch)
		(eq (car ch) :mouse-button)
		(eql (int-char (cadr ch)) #\Mouse-R-1))
	   (tv:mouse-call-system-menu))
	  ((and (consp ch)
		(send self :preempted-handle-blip ch)))
	  )))


(defmethod 4(live-listener-synchronous-UCL-mixin :fquery)* (fquery-options fquery-format-string &rest fquery-format-args)
  (declare (special fquery-options))
  (with-stack-list* (fquery-options :condition nil fquery-options)
    (declare (special fquery-options))
    (values :NEW-VALUE
	    (allow-click-of-type nil
	      #'(lambda (ignore)
		  ;1; We need a slightly different rubout handler, to assure that mouse-clicks will preempt typein.*
		  (let* ((TV:STREAM-MIXIN-RUBOUT-HANDLER 'live-listener-preemptable-rubout-handler))
		    (apply 'fquery fquery-options fquery-format-string fquery-format-args)))
	      '(*ll-fquery-command-table*)
	      ))))


;1;; Make break-loops running in any application's typeout window have mousability.*
;1;;*

(defflavor 4live-listener-ucl-break-mixin *()
	   (prin1-mousably-mixin live-listener-UCL-commands-mixin)
  (:default-init-plist :font-map *default-live-listener-font-map*)
  (:documentation :mixin "2When mixed into a random-typeout-window, break-loops running in it wil be mousable.*")
  )


(defmethod 4(live-listener-ucl-break-mixin :return-object)* (object)
  "2Find UCL which is running in the process of this window, and send it the :return-object method if it handles it.
  It it doesn't handle it, or if we can't find a UCL, beep and print an error message.*"
  (let* ((proc (send self :process))
	 (sg (and proc (send proc :stack-group)))
	 (ucl (and sg (symeval-in-stack-group 'ucl:this-application sg))))
    (cond ((null ucl) (format t "2~&Can't find a UCL for ~A*" self) (beep))
	  ((send ucl :operation-handled-p :return-object)
	   (send ucl :return-object object))
	  (t (format t "2~&~A doesn't handle the :RETURN-OBJECT message.*" ucl)))))


(defmethod 4(ucl:basic-command-loop :return-object*) (object)
  "2Make a nonlocal exit within this UCL so that the given object is immediately returned.*"
  (declare (special UCL:COMMAND-ENTRY))  ;1 Bound by the UCL, which we must be within the scope of.*
  (assert (boundp 'ucl:command-entry) () "2UCL:COMMAND-ENTRY is unbound; we are likely not running synchronously.*")
  ;1;*
  ;1; The default behavior is to only print the results of ``abnormal'' commands - Lisp typein commands qualify, so returned values are*
  ;1; printed.  However, this command does not qualify as ``abnormal'' because it came in as a command object.  So we set the*
  ;1; command-entry to NIL so that we are in abnormal-mode, 5:handle-results* will print the value passed it (and store it in **1).*
  ;1;*
  (setq ucl:command-entry nil)
  ;1;*
  ;1; At this point, we could just return 5object*, and it would be placed in **1 and printed.  But it would be nice if the prompt was printed*
  ;1; again after the returned value; to do this, we explicitly call 5:handle-results* with our value (instead of just returning and letting it be*
  ;1; called there).  And since we've called 5:handle-results*, we can't simply return, or it will be called again, so we throw back to top-level.*
  ;1;*
  (send self :handle-results (list object))
  (throw 'UCL:COMMAND-ABORT nil))



;1;; Modify the 5UCL-Break* flavor (which is what gets created when typing #\Break at :tyi) and the 5UCL-debugger* flavor*
;1;; (which is what gets created to handle any error) to have an 5:execute* blip so we can force them to synchronously *
;1;; execute arbitrary code.*
;1;;*
(defmethod 4(sys:ucl-break :after :init)* (ignore)
  (push '(:execute :handle-execute) ucl:blip-alist))

(defmethod 4(sys:ucl-break :handle-execute)* (&optional blip)
;  (format t "3~&sys:ucl-break handling :execute*");1##*
  (unless blip (setq blip UCL:KBD-INPUT))
  (apply (second blip) (cddr blip)))

(defmethod 4(eh:ucl-debugger :after :init)* (ignore)
  (push '(:execute :handle-execute) ucl:blip-alist))

(defmethod 4(eh:ucl-debugger :handle-execute)* (&optional blip)
;  (format t "3~&eh:ucl-debugger handling :execute*");1##*
  (unless blip (setq blip UCL:KBD-INPUT))
  (apply (second blip) (cddr blip)))


;1;; Originally defined in 5Sys: EH; UCL-Debugger.Lisp#>**
;1;; This method was (intentionally it would seem) calling 5:TYI* instead of 5:ANY-TYI*.  Doing this prevented our 5:execute* blips *
;1;; from being seen.  Using 5:ANY-TYI* doesn't seem to hurt anything, so we do that instead.*
;1;;*
EH:(defmethod 4(ucl-debugger :fetch-input*) ()
  "2Redefines (:METHOD UCL:BASIC-COMMAND-LOOP :FETCH-INPUT) which gets the
initial (keystroke) input from the user on each interation.
Changed to always prompt after every debugger command, ie, not just typein input;
Changed to set *reading-command**"
  (when (eq ucl:typein-handler :handle-typein-input)
    (send self :handle-prompt))
  (let ((*reading-command* t))
    (declare (special *reading-command*))
    (send *standard-input* :any-tyi)))		;1; <-- [jwz] Changed this.  Use :any-tyi instead of :tyi*


;1;; Originally defined in 5Sys: Debug-Tools; Peek.Lisp#>**
;1;; Need it to handle 5:execute* blips as well.*
;1;;*
TV:(DEFMETHOD 4(peek-frame :AFTER :INIT*) (IGNORE)
  "2Set up the pane variables, which are used in the defcommands.*"
  (push '(:execute :handle-execute) ucl:blip-alist)		;1; <--- [jwz] added this.*
  (SETQ peek-pane (SEND SELF :get-pane 'PEEK)
	typeout-pane (SEND peek-pane :typeout-window)
	mode-pane (SEND SELF :get-pane 'menu)
	cmds-pane (SEND SELF :get-pane 'cmds))
  (SEND SELF :select-pane peek-pane)
  (colorize-peek-frame))

(defmethod 4(tv:peek-frame :handle-execute)* (&optional blip)
;  (format t "3~&tv:peek-frame handling :execute*");1##*
  (unless blip (setq blip UCL:KBD-INPUT))
  (apply (second blip) (cddr blip)))


;1;; Many applications use this flavor for their typeout windows.  Make break-loops running on it be mousable.*
;1;; Originally defined in 5Sys:Window;Typwin.Lisp#>**
(defflavor 4tv::typeout-window*
	   ((tv::label nil) (tv::borders nil))
	   (live-listener-UCL-break-mixin	;1 <---- [jwz] added this.*
	    tv:basic-typeout-window
	    tv:notification-mixin tv:window)
  )

;1;; Zmacs uses this flavor for its typeout windows.  Make break-loops running on it be mousable.*
;1;; It's not enough to just redefine 5tv:typeout-window*, because 5live-listener-UCL-break-mixin* must preceed* 5tv:basic-mouse-sensitive-items1.**
;1;; There's probably a way to work around this by specifying the method-combination differently, but life's too short.*
;1;; Originally defined in 5Sys:Window;Typwin.Lisp#>**
(defflavor 4tv::typeout-window-with-mouse-sensitive-items*
	   ((tv::label nil) (tv::borders nil))
	   (live-listener-UCL-break-mixin	;1 <---- [jwz] added this.*
	    tv::basic-mouse-sensitive-items
	    tv::typeout-window)
  (:documentation :combination "2Typeout window with item operations.*"))


;1;; Useful for debugging.*
;(defmethod 4(basic-mouse-sensitive-items :get-item-list)* () tv:item-list)

;1;;*
;1;; Add the Live-Listener items to the Zmacs item-list.  Zmacs won't listen to us until we do.*
;1;; This should probably be fixed more generally.*
(eval-when (load eval)
  (dolist (item-spec (reverse *live-listener-item-type-alist*))
    (pushnew item-spec zwei:*typeout-command-alist* :test #'equalp))
  )


;1;; Another useful Zmacs patch.  If a typeout window is up, and you select from a mouse-sensitive item, it should not*
;1;; immediately go away - you might want to select from another.  This is how List Buffers works, because the typeout*
;1;; window is the last thing done (no code gets executed after you hit space) but other things, like compilation warnings,*
;1;; ditch the window at any input.*

ZWEI:(DEFUN 4CHECK-FOR-TYPEOUT-WINDOW-TYPEOUT* (&OPTIONAL WAIT-IF-EXPOSED)
  "2If the typeout window is incomplete, wait until an input character is available.
WAIT-IF-EXPOSED non-NIL says do so if typeout window is exposed
even if it is not incomplete.*"
  (SYS:WITH-SUGGESTIONS-MENUS-FOR ZWEI:CHECK-FOR-TYPEOUT-WINDOW-TYPEOUT
    (COND ((SEND *STANDARD-OUTPUT* :SEND-IF-HANDLES :NEVER-FLUSH-TYPEOUT))
	  ((IF WAIT-IF-EXPOSED
	       (SEND *STANDARD-OUTPUT* :EXPOSED-P)
	       (TYPEOUT-WINDOW-INCOMPLETE-P *STANDARD-OUTPUT*))
	   (WITHOUT-IO-BUFFER-OUTPUT-FUNCTION
	     ;1;; jwz: changed this loop to process any blips rather than exiting and deferring them.*
	     (DO ((CHAR (W:READ-ANY) (W:READ-ANY)))
		 ((not (consp char))
		  (OR (EQL CHAR #\SPACE)
		      ;1; If it's not a space, unread it.  That will*
		      ;1; prevent immediate redisplay.*
		      (W:UNREAD-ANY CHAR)))
	       (when (consp char)
		 (cond ((AND ;1; Ignore requests to select current window*
			     (EQ (FIRST CHAR) 'SELECT-WINDOW)
			     (EQ (SECOND CHAR) *WINDOW*))
			nil)
		       (t
			(let* ((int *interval*))
			  (apply #'zwei:process-special-command char)
			  (unless (eq *interval* int) (return)))	;1 hack - return if selected buffer changes.*
			)))))))))


;1;; Make command-typein windows in most applications be mousable.*

;1;; This doesn't really work.  Applications tend to special-case mouse clicks in non-uniform ways, so we need to patch*
;1;; into them on an application-by-application basis.*

;1;; Originally defined in 5Sys: Input-Editor; RH-Help.Lisp#>**
;1;;*
;(defflavor 4ucl:command-and-lisp-typein-window* ()  
;	   (prin1-mousably-mixin
;	    live-listener-synchronous-UCL-mixin
;	    tv:notification-mixin
;	    tv:auto-scrolling-mixin
;	    tv:lisp-help-mixin
;	    tv:package-maintenance-mixin
;            tv:preemptable-read-any-tyi-mixin
;	    w:window)
;  (:DEFAULT-INIT-PLIST
;    :save-bits T)
;  (:DOCUMENTATION :COMBINATION "
;2Suggested command typein window for the Universal Command* 2Loop.*  2Provides for special wholine help*
;2on lisp typein.*  2An application with the* 2COMMAND-TYPEIN-MIXIN should include* 2this either in a*
;2constraint frame or as a mixin. * 2In the* 2case of* 2a constraint frame, *TERMINAL-IO* should be bound to*
;2the pane.*"))



;1;; 5Pretty-printer bug fix.*  Submitted as SPR 10795.*
;1;;*
;1;; The function *TV:GRIND-INTO-LIST1 returns lists describing a pretty-printed data structure.*
;1;; Among the information returned is: what line a particular object was printed on; what character positions it occupied; and a*
;1;; locative pointer to the object itself.*
;1;;*
;1;; The problem was, when **PRINT-ARRAY*1 was true, and a structure like this was printed:*
;1;;*
;1;;3    #(A B C)**
;1;;*
;1;; The information returned was of the following form:*
;1;;*
;1;;     character positions 0-7:  the array 3#(A B C)**	1<-- this is correct.*
;1;;     character positions 2-2:  the array 3#(A B C)**	1<-- this should be ``the symbol A''*
;1;;     character positions 4-4:  the array 3#(A B C)**	1<-- this should be ``the symbol B''*
;1;;     character positions 6-6:  the array 3#(A B C)**	1<-- this should be ``the symbol C''*
;1;;*
;1;; A result of this bug was that, in both the inspector and in the code in this file, mousing on an element of a vector returned *
;1;; the entire vector instead of the selected element.  The bug was in the internal pprinter function 3SYS::PP-OBJIFY-VECTOR*,*
;1;; which is fixed below.*
;1;;*


SYS::
(defun pp-objify-vector (object location  currlevel)
  ;;CLM for PHD 10/20/87 Fixed SPR#??? "No leading space before ..."
  (declare (fixnum currlevel))
  "Makes a PP-Obj for a vector."
  (if (or (not *print-array*)
	  ;; phd 4/7/86 leave the printing of bit-vectors to the regular print.
	  (typep object 'bit-vector))
      (pp-objify-atom object location)
      (do* ((index 0 (1+ index))
	    (terminus (length (the vector object)))
	    (total-length 2)
	    (result (list pp-sharp-open-paren-obj))
	    (splice result))
	   ((or (and *print-length* (>= index *print-length*)) (= index terminus))
	    (cond ((/= index terminus)
		   (end-cons splice pp-dotdotdot-obj)
		   (setq total-length (+ 4 total-length))))
	    (end-cons splice pp-close-paren-obj)
	    (make-pp-obj :type 'complex
			 :length (1+ total-length)
			 :object result
			 :location location))
	(declare (fixnum index total-length terminus))
	(cond ((> index 0)
	       (end-cons splice pp-space-obj)
	       (setq total-length (1+ total-length))))
	;1;*
	;1; This next form used to be*
	;1;*	3(end-cons splice (pp-objify (aref object index) location (1+ currlevel)))*
	;1; which is incorrect; 3PP-OBJIFY* was being called with an element of the array, but the locative of the whole array, rather*
	;1; than the locative of that element of the array.  This caused a bug in the inspector, where mousing on an element of a*
	;1; pretty-printed array would inspect the whole array rather than the selected element of it.  -- jwz, 31 aug 89.*
	;1;*
	(end-cons splice (pp-objify (aref object index) (locf (aref object index)) (1+ currlevel))))))



;1;; 5Prompt-and-Read bug fix.*  Submitted as SPR 10793; reported fixed on 22-nov-89.*
;1;;*
;1;; The function 3PROMPT-AND-READ-INTERNAL* was structured in the form*
;1;;*
;1;;* (or (send *query-io* :send-if-handles :prompt-and-read2 ... *)
;1;;*       2< ... normal body of prompt-and-read ... >*
;1;;*       )
;1;;*
;1;; the intent being that **QUERY-IO*1 could handle the 3:PROMPT-AND-READ* message specially.  However, the code should have*
;1;; been structured in the form*
;1;;*
;1;;*   (cond ((send *query-io* :operation-handled-p :prompt-and-read)
;1;;*          (send *query-io* :prompt-and-read2 ... *))
;1;;*         (t
;1;;*           2< ... normal body of prompt-and-read ... >*
;1;;*          )
;1;;*
;1;; because there 6is a difference* between not handling an operation, and handling it but returning 3NIL*.  In the first case, if the user*
;1;; typed 3NIL* at the prompt, they would be prompted a second time!*
;1;;*


SYS:
(defun prompt-and-read-internal (option rubout-handler-options format-string &optional format-args)
  ;5; 11/20/87 CLM - Fix to use ZLC:MEMQ instead of MEMQ.  The function was looking for MEMQ in the*
  ;5;                SYS package, which used to work.  At some time MEMQ must have been exported to*
  ;5;                SYS, but this is no longer the case.*
  ;1;  8/7/89 JWZ  - Fixed to use *:operation-handled-p1 instead of *:send-if-handles1, so that the *:prompt-and-read1 method on*
  ;1;                  a window is allowed to return NIL!*
  (if (send *query-io* :operation-handled-p :prompt-and-read)
      (lexpr-send *query-io* :prompt-and-read option format-string format-args)
      (let* ((option-type (if (consp option)
			      (car option)
			      option))
	     (function (get option-type 'prompt-and-read-function))
	     (prompt-and-read-format-string format-string)
	     (prompt-and-read-format-args format-args))
	(cond
	  ((get option-type 'prompt-and-read-no-rubout-function)
	   (send (get option-type 'prompt-and-read-no-rubout-function) option *query-io*))
	  ((null function)
	   (ferror () "~S is not a known PROMPT-AND-READ option keyword." option-type))
	  ((send *query-io* :operation-handled-p :rubout-handler)
	   (let ((rh-options (get option-type 'prompt-and-read-rubout-options
				  '((:prompt prompt-and-read-prompt-function)
				    (:activation zlc:memq (#\END #\NEWLINE))))))
	     (if rubout-handler-options
		 (setf rh-options (cons rubout-handler-options rh-options)))
	     (send *query-io* :rubout-handler rh-options function option *query-io*)))
	  (t (funcall function option *query-io*))))))


;1;; 5Insert Character bug fix.**
;1;;*
;1;; The rubout handler has this silly habit of passing around 5art-64b* arrays as if they were strings.  This has bad effects in general, but*
;1;; in our case, it causes 5:insert-string* to always compute font widths from font #0, regardless of what the current font is (because*
;1;; the function 5tv:sheet-string-length* treats 5art-64b* arrays as 5art-fat-string*s and uses the high bits of the character as a font index -*
;1;; and of course, these bits are always zero).*
;1;;*

(defwhopper 4(live-listener-synchronous-UCL-mixin :insert-string*) (string &rest args)
  (unless (or (stringp string) (characterp string) (numberp string))
    (setq string (coerce string 'SIMPLE-STRING)))
  (lexpr-continue-whopper string args))


;1;; 5KSL Inspector Enhancements Bug Fix.*  Reported to Rice; certainly fixed by now.*
;1;; The KSL code has a copy of this function which is outdated/incorrect.  Here is the correct version, installed only if the KSL code is loaded.*
;1;;*

TV::
(DEFUN 4working-*GRIND-INTO-LIST-MAKE-ITEM (THING LOC ATOM-P LEN)
  (LET ((IDX (IF GRIND-INTO-LIST-STRING
	       (ARRAY-ACTIVE-LENGTH GRIND-INTO-LIST-STRING)
	       0)))
;      (PUSH (LIST thing (IF (CONSP loc) (CAR loc) loc) atom-p len) user:xx)
    (COND
      (ATOM-P
         ;; An atom -- make an item for it.
;       (PUSH (LIST LOC :LOCATIVE IDX (+ IDX  LEN)) ;(FLATSIZE THING)))
       (PUSH (LIST LOC :LOCATIVE IDX (+ IDX  (if (stringp (IF (CONSP loc) (CAR loc) loc) )
					       (+ 1 (or (position #\cr (IF (CONSP loc) (CAR loc) loc))
						   (1- (flatsize (IF (CONSP loc) (CAR loc) loc)))))
					       len)))
	     (CAR GRIND-INTO-LIST-ITEMS)))
      (T
       ;; Printing an interesting character
       (CASE THING
	 ('sys:start-of-object
	  ;; Start of a list.
	  (PUSH (LIST LOC IDX GRIND-INTO-LIST-LINE () ()) GRIND-INTO-LIST-LIST-ITEM-STACK))
	 ('sys:end-of-object
	  ;; Closing a list.
	  (LET ((ITEM (POP GRIND-INTO-LIST-LIST-ITEM-STACK)))
		;; 1+ is to account for close-paren which hasn't been
		;; typed yet. in rel2 next line was (1+ idx)
	    (SETF (FOURTH ITEM) IDX)
	    (SETF (FIFTH ITEM) GRIND-INTO-LIST-LINE)
	    (PUSH ITEM GRIND-INTO-LIST-LIST-ITEMS))))))))

(eval-when (load eval)
  (when (intersection '(:ksl-patches-7 :ksl-patches-8 :ksl-patches-9) *features*) ;1 Only define when *KSL[7-9]1 is loaded.*
    (deff TV:GRIND-INTO-LIST-MAKE-ITEM 'TV:WORKING-GRIND-INTO-LIST-MAKE-ITEM)))



;1;; 5Installation.**
;1;;*
;1;; Add the live listener (and the old listener) to the Create option of the System Menu.*
;1;;*
(setq tv:default-window-types-item-list
      (set-difference tv:default-window-types-item-list '("3Lisp*" "3Old Lisp Listener*" "3Lisp Listener*"
							  "3Asynch Lisp Listener*")
		      :key #'nse:safe-car :test #'string-equal))

(tv:add-window-type "3Old Lisp Listener*" 'w:lisp-listener "2The old-style Lisp interactor, with no mouse sensitivity.*")
(tv:add-window-type "3Lisp Listener*" 'w:live-listener "2A flavor of Lisp interactor in which most items are mousable.*")
;(tv:add-window-type "3Asynch Lisp Listener*" 'w:asynchronous-live-listener
;		    "2A different flavor of Lisp interactor in which most items are mousable.*")

(w:add-system-key #\L 'LIVE-LISTENER "2Evaluate Lisp forms, making output mouse sensitive.*" t nil "2Live Listener*")

(compile-flavor-methods LIVE-LISTENER
			ASYNCHRONOUS-LIVE-LISTENER
			LL-COMMAND
			TV::TYPEOUT-WINDOW
			TV::TYPEOUT-WINDOW-WITH-MOUSE-SENSITIVE-ITEMS
			)
