;;; -*- Mode:Lisp; Syntax: Common-Lisp; Base:10. -*-
;;
;; Put a mark at the beginning of a line with text on it, put two 
;; marks and point at the beginning of a subsequent empty line, and
;; do a C-X T (exchange regions.)  This fixes what happens.
;; By Fritz Mueller <mueller@sumex-aim.stanford.edu>
;;
;;  From ZMACS;INSERT.LISP#4 on 12/13/90 by FritzM.  
;;
;;  This fixes a pathological case where the source and destination
;;  overlap and that overlap is an empty line.  Normally such an overlap
;;  was handled appropriately, but when inserting stuff ending with an
;;  empty line ("CR") into a position at the front of a line the code
;;  special cases: it assumes it can just "push down" the old
;;  destination line and splice in a new line to hold the start of the
;;  inserted text.  In the case of a source/destination overlap, this
;;  caused the copying loop to be diverted onto the splice, such that
;;  the termination condition of the loop could never be met.  The fix:
;;  delay the splice until after the copying operation.
;;

;;; First arg is a BP.  Second is an interval, or second&third are an ordered range.
;;; Insert the stuff from the interval at the BP.
(DEFUN INSERT-INTERVAL (AT-BP FROM-BP &OPTIONAL TO-BP IN-ORDER-P)
  "Insert a copy of an interval into text at AT-BP.
Either pass the interval to insert as the second argument,
or pass a pair of BPs that delimit the interval.
AT-BP is left pointing before the inserted text unless it is of type :MOVES.
The value is a BP pointing after the inserted text."
  (AND (NOT *BATCH-UNDO-SAVE*)
       *UNDO-SAVE-SMALL-CHANGES*
       (UNDO-SAVE-NEW-SMALL-CHANGE AT-BP AT-BP))
  (MUNG-BP-INTERVAL AT-BP)
  (GET-INTERVAL FROM-BP TO-BP IN-ORDER-P)
  (LET ((AT-LINE (BP-LINE AT-BP))
	(AT-INDEX (BP-INDEX AT-BP))
	(FROM-LINE (BP-LINE FROM-BP))
	(FROM-INDEX (BP-INDEX FROM-BP))
	(TO-LINE (BP-LINE TO-BP))
	(TO-INDEX (BP-INDEX TO-BP)))
    (IF (EQ FROM-LINE TO-LINE)
	;; Insert within AT-LINE.
	(INSERT-WITHIN-LINE AT-LINE AT-INDEX FROM-LINE FROM-INDEX TO-INDEX)
	(LET ((AT-LINE-LENGTH (LINE-LENGTH AT-LINE))
	      (FROM-LINE-LENGTH (LINE-LENGTH FROM-LINE))
	      (ARRAY-TYPE (IF (OR (EQ (ARRAY-TYPE TO-LINE) 'ART-FAT-STRING)
				  (EQ (ARRAY-TYPE FROM-LINE) 'ART-FAT-STRING))
			      'ART-FAT-STRING
			      (ARRAY-TYPE AT-LINE)))
	      FIRST-LINE
	      LAST-LINE)
	  (COND ((AND (ZEROP TO-INDEX)
		      (ZEROP AT-INDEX))
		 ;;Inserting stuff ending with CR at front of line
		 ;;implies we can just shove down the old line
		 (SETQ LAST-LINE AT-LINE)
		 ;; But then we can't use it as the first line.
		 (SETQ FIRST-LINE (CREATE-LINE ARRAY-TYPE
					       (- FROM-LINE-LENGTH FROM-INDEX)
					       (BP-NODE AT-BP)))
		 (COPY-ARRAY-PORTION FROM-LINE FROM-INDEX FROM-LINE-LENGTH
				     FIRST-LINE 0 (- FROM-LINE-LENGTH FROM-INDEX))
		 (SETF (LINE-PLIST FIRST-LINE) (LINE-PLIST FROM-LINE))
		 ;; Transfer bps from the front of AT-LINE to FIRST-LINE.
		 (DOLIST (BP (LINE-BP-LIST AT-LINE))
		   (AND (ZEROP (BP-INDEX BP))
			(EQ (BP-STATUS BP) :NORMAL)
			(MOVE-BP BP FIRST-LINE 0))))
		(T
		 ;; Otherwise, keep the beginning of the line we are inserting in,
		 ;; and make a new line for the tail end of the string.
		 (SETQ FIRST-LINE AT-LINE)
		 (SETQ LAST-LINE (CREATE-LINE ARRAY-TYPE
					      (+ TO-INDEX (- AT-LINE-LENGTH AT-INDEX))
					      (BP-NODE AT-BP)))
		 ;; Copy the first part of TO-LINE into the LAST-LINE.
		 (COPY-ARRAY-PORTION TO-LINE 0 TO-INDEX
				     LAST-LINE 0 TO-INDEX)
		 ;; Figure out whether AT-LINE is being changed at all.
		 (OR (AND (ZEROP FROM-LINE-LENGTH)
			  (= AT-INDEX (LINE-LENGTH AT-LINE)))
		     (MUNG-LINE AT-LINE))
		 ;; Copy the second part of AT-LINE to LAST-LINE.
		 (COPY-ARRAY-PORTION AT-LINE AT-INDEX AT-LINE-LENGTH
				     LAST-LINE TO-INDEX (+ (- AT-LINE-LENGTH AT-INDEX) TO-INDEX))
		 ;; Copy FROM-LINE into AT-LINE.
		 (SET-LINE-LENGTH AT-LINE (+ AT-INDEX (- FROM-LINE-LENGTH FROM-INDEX)))
		 (DO ((FF FROM-INDEX (1+ FF))
		      (AT AT-INDEX (1+ AT))
		      (|16B-P| (EQ (ARRAY-TYPE AT-LINE) 'ART-FAT-STRING))
		      (CH))
		     ((>= FF FROM-LINE-LENGTH))
		   (COND ((NOT (OR (< (CHAR-INT (SETQ CH (AREF FROM-LINE FF))) 400)
				   |16B-P|))
			  (SET-LINE-ARRAY-TYPE AT-LINE 'ART-FAT-STRING)
			  (SETQ |16B-P| T)))
		   (SETF (AREF AT-LINE AT) CH))
		 ;; Relocate buffer pointers.
		 (DOLIST (BP (LINE-BP-LIST AT-LINE))
		   (LET ((I (BP-INDEX BP)))
		     (COND ((OR (> I AT-INDEX)
				(AND (= I AT-INDEX)
				     (EQ (BP-STATUS BP) :MOVES)))
			    (MOVE-BP BP LAST-LINE (+ (- I AT-INDEX) TO-INDEX))))))))
	  (DO ((PREV-LINE FIRST-LINE THIS-LINE)
	       (THIS-LINE)
	       (NODE (BP-NODE AT-BP))
	       (THE-LINE-BEYOND (LINE-NEXT AT-LINE))
	       (ORIGINAL-LINE (LINE-NEXT FROM-LINE) (LINE-NEXT ORIGINAL-LINE)))
	      ((EQ ORIGINAL-LINE TO-LINE)

	       ;; Splice top of copy in now.  --FritzM. 12/13/90
	       (SETF (LINE-PREVIOUS FIRST-LINE) (LINE-PREVIOUS AT-LINE))
	       (AND (LINE-PREVIOUS AT-LINE)
		    (SETF (LINE-NEXT (LINE-PREVIOUS AT-LINE)) FIRST-LINE))

	       (AND THE-LINE-BEYOND
		    (SETF (LINE-PREVIOUS THE-LINE-BEYOND) LAST-LINE))
	       (SETF (LINE-NEXT LAST-LINE) THE-LINE-BEYOND)
	       (SETF (LINE-NEXT PREV-LINE) LAST-LINE)
	       (SETF (LINE-PREVIOUS LAST-LINE) PREV-LINE))
	    (SETQ THIS-LINE (COPY-LINE ORIGINAL-LINE NODE))
	    (SETF (LINE-NEXT PREV-LINE) THIS-LINE)
	    (SETF (LINE-PREVIOUS THIS-LINE) PREV-LINE))
	  (CREATE-BP LAST-LINE TO-INDEX))))) 
