;;; -*- Mode:Common-Lisp; Package:PRINTER; Fonts:(MEDFNT TR12BI TR12B); Base:10; Patch-file:T -*-

;1*********************************************************************************
;
;1  This file contains all the necessary patches to get the print-file function and*
;1    screen dumps to work for the Imagens.  TERM-Q does the right thing, and*
;1    so does the Hardcopy option from the system menu.*
;
;1  In addition, PRINT-IMPRESS-FILE can be used to print out raw imPRESS files.*
;
;1  For print-file, some options will not work, but there are other things one can do:*
;1    Rather than :LINES, give IMAGEN:*PIXELS-DOWN-A-PAGE* the desired value.*
;1    Rather than :PRINT-WIDE, modify IMAGEN:*PIXELS-IN-A-LINE* to change the*
;1       print width.*
;1    :CPI and :LPI won't work.*
;1    IMAGEN:*TOP-MARGIN-HEIGHT* and IMAGEN:*LEFT-MARGIN-WIDTH* can*
;1      also be changed.*
;
;1  When printing screen dumps, the Image File question is not relevant since this version*
;1    merely uses a string for the temporary image file.*
;
;1  The IMAGEN:*IMAGEN-TIMEOUT* variable has been set at 180 (or 3 minutes).*
;1  Since this stuff will be done by a background process, the potentially long wait*
;1    shouldn't matter, and it should reduce the possibility that the Imagen is busy*
;1    for the entire wait.*
;
;1  JNN: John Nguyen -- July 1987*
;
;1*********************************************************************************


(DEFVAR IMAGEN:*IMAGEN-PRINTER-NAME* (PRINTER:GET-DEFAULT-PRINTER)
  "2Default Imagen printer*")

(DEFVAR IMAGEN:*IMAGEN-REMOTE-PORT* 35.
  "2TCP remote port for Imagen printers*")

(DEFVAR IMAGEN:*IMAGEN-TIMEOUT* nil
  "2Wait for Imagen to be free on open.  NIL to wait indefinitely*")

(DEFVAR IMAGEN:*BIGGEST-MANUAL-MAGNIFICATION* 1000.
  "2Biggest size of the magnified array before we force imPRESS magnification
rather than magnify manually*")

;1 Map of Lisp fonts to Imagen fonts*
(SETQ IMAGEN:*LISP-TO-IMAGEN-FONT-MAPPING*
      '(("CPTFONT"."CMASC09") ("MEDFNT"."CMASC10") ("COURIER"."CMASC10")
	("WIDER-MEDFNT"."CMASC10") ;1 All fixed fonts are CMASC*
	("CPTFONTB"."CMB10") ("CPTFONTI"."CMSSS") ("CPTFONTBI"."CMBTI")
	("HL10"."CMSS10") ("HL10B"."CMB10") ("HL12"."CMSSB")
	("HL12B"."CMB10") ("HL12I"."CMSSS") ("HL12BI"."CMBTI")
	("TR10"."CMSS10") ("TR12"."CMSSB")
	("TR10B"."CMB10") ("TR12B"."CMB10")
	("TR10BI"."CMBTI") ("TR10BI"."CMBTI")
	("TR10I"."CMSSS") ("TR12I"."CMSSS")
	("MEDFNB"."CMB10")
	)
      )


;1*********************************************************************************
;
;1    This function for printing raw imPRESS files -- JNN*
;
;1*********************************************************************************

(DEFUN GLOBAL::PRINT-IMPRESS-FILE (FILE &REST OPTIONS)
  "2Print an imPRESS format file directly to an Imagen printer.  The options are
the same as those in PRINT-FILE.*"
  (APPLY #'PRINT-FILE FILE :PAGE-HEADING " IMPRESS " OPTIONS))
(EXPORT 'GLOBAL::PRINT-IMPRESS-FILE 'USER)


;1*********************************************************************************
;
;1     The following functions are for actual communication to the Imagens*
;
;1*********************************************************************************


(DEFUN IMAGEN:COPY-STREAM-TO-IMAGEN (FROM-STREAM &KEY (IMAGEN-NAME IMAGEN:*IMAGEN-PRINTER-NAME*)
				     (CHARACTERS  T))
  "2Establish a connection to printer IMAGEN-NAME, then send the
contents of FROM-STREAM to it.*"
  (IMAGEN:COPY-STREAMS-TO-IMAGEN (LIST FROM-STREAM)
				 :IMAGEN-NAME IMAGEN-NAME :CHARACTERS CHARACTERS))

(DEFUN IMAGEN:COPY-STREAMS-TO-IMAGEN (FROM-STREAMS &KEY (IMAGEN-NAME IMAGEN:*IMAGEN-PRINTER-NAME*)
				      (CHARACTERS  T))
  "2Establish a connection to printer IMAGEN-NAME, then send the
contents of streams in FROM-STREAMS to it.*"
  (WITH-OPEN-STREAM (TO-STREAM (IP:OPEN-STREAM IMAGEN-NAME
					       :REMOTE-PORT IMAGEN:*IMAGEN-REMOTE-PORT*
					       :TIMEOUT     IMAGEN:*IMAGEN-TIMEOUT*
					       :CHARACTERS  CHARACTERS))
    (LOOP FOR FROM-STREAM IN FROM-STREAMS DO
	  (STREAM-COPY-UNTIL-EOF FROM-STREAM TO-STREAM))))


;1*********************************************************************************
;
; 1The following are patches to the Release 2 Imagen code (from SYS: IMAGEN; IMAGENP);*
;1 without these changes things don't work.*
;
;1*********************************************************************************


(DEFMETHOD (IMAGEN-PRINTER :FINISH-IMAGEN-DOCUMENT) ()
  "2Finish Imagen document: output 255 and set doc string to nil*"
  (SEND PRINTER-STREAM :TYO 255)
  (SETQ IMAGEN:DOCUMENT-STRING-ALREADY-SENT-P NIL))

(DEFMETHOD (IMAGEN-PRINTER :TYO-RAW) (CHAR)
  (WRITE-BYTE CHAR PRINTER-STREAM))


;1*********************************************************************************
;
;1 The following are changes to the Release 2 Imagen code (from SYS: IMAGEN; IMAGENP) which*
; 1make various "improvements" to the original code.*
;
;1*********************************************************************************

;
;1 Change the :PRINT-TEXT-FILE method to not use a temporary file, but rather a string buffer*
;1 Re-added & modified the stuff at the end to make pages print out in reverse order -- JNN*
;
(DEFMETHOD (IMAGEN-PRINTER :PRINT-TEXT-FILE) (FILE-STREAM
						       &OPTIONAL FONTS PAGE-ORDER
						       &AUX LAST-PAGE-END BUFFER)
  "2Using the font information in the file attribute line, print the
file looking as much as possible like the file looked on the screen.
Page-reverse the printing by first creating the string stream to send the Imagen
and remembering where all the page starts are, then printing that stream page
by page backwards.*"
  ;1;*
  ;1; Establish our buffer file, then "print" to it, remembering the positions of the pages' starts*
  ;1;    in the list page-start-list...*
  ;1;*
  (WITH-OPEN-STREAM (TEMP-RAW-FILE (MAKE-STRING-OUTPUT-STREAM))
1     *(SETQ IMAGEN:TRUE-PRINT-STREAM			1     *;1 Save our printer's stream.*
	  PRINTER-STREAM)
    (SETQ PRINTER-STREAM TEMP-RAW-FILE)1     *;1 Point all "printing" at our buffer file.*

    ;1; Setup fonts...*
    (UNLESS (CONSP FONTS)			1     *;1 Make sure we have a list of fonts to work with.*
      (SETF FONTS (LIST FONTS)))
    (SETQ IMAGEN:WYSIWYG-P
	  (OR IMAGEN:*WYSIWYG-DEFAULT*		    ;1Decide whether or not to create WYSIWYG fonts.*
	      (STRING-EQUAL (FIRST FONTS) "WYSIWYG")))
    (WHEN (OR IMAGEN:WYSIWYG-P			    ;1 Get the file's fonts if in WYSIWYG mode*
	      (NULL (FIRST FONTS)))		1     *;1    or none supplied by caller.*
      (SETF FONTS (GET-FONT-ATTRIBUTE-LIST FILE-STREAM)))
    (IF (NULL (FIRST FONTS))			1     *;1 If no font specification,*
	(IMAGEN:SETUP-IMAGEN-FONTS '(:CPTFONT))	1     *;1    force CPTFONT use,*
	;1; else...*
	(IMAGEN:SETUP-IMAGEN-FONTS FONTS))		1     *;1    otherwise use what was specified.*

    ;1; Get on with the printing...*
    (SEND SELF :START-NEW-PAGE)			1     *;1 Setup to start the first page of printing.*
    (LOOP FOR IN-CHAR = (TYI FILE-STREAM ())	1     *;1 Then read in the file to be printed,*
	  UNTIL (NULL IN-CHAR) DO		1     *;1    character by character.*
	  (COND
	    ((eql (int-char IN-CHAR) #\epsilon)	1     *;1 If an epsilon, it might be a font switch.*
	     (SETQ IN-CHAR			1     *;1 Get the character following.*
		   (SEND FILE-STREAM :TYI))
	     (UNLESS				    ;1 If font-handling is active for this printing *
	       (AND (NOT (NULL FONTS))		1     *;1    and 'tis a good font switch,*
		    (SEND SELF :SET-CURRENT-FONT-FAMILY IN-CHAR))   ;1 then do it.*
	       (SEND SELF :TYO-CHAR #\epsilon)	1     *;1 Otherwise print the epsilon*
	       (SEND FILE-STREAM :UNTYI IN-CHAR)))  ;1 and let the next char be scanned again.*
	    (T (SEND SELF :TYO-CHAR IN-CHAR))))	1     *;1 Print anything else.*
    (SEND SELF :STRING-OUT-RAW (IMAGEN:IMPRESS-ENDPAGE))1    *;1 Finish off the last page.*	1   *
    (SETQ BUFFER (GET-OUTPUT-STREAM-STRING TEMP-RAW-FILE))
    (SETQ LAST-PAGE-END	(LENGTH BUFFER)))	1     *;1 Get position of end of last page.*
	  ;1;*
	  ;1;  Now copy the buffered pages to the Imagen in reverse order...*
          ;1;  We must reverse them if we want them printed in the correct order.*
          ;1;  Added back this part so that Imagen output is in the correct order -- JNN*
	  ;1;*
  (UNLESS (EQ PAGE-ORDER :REVERSE)
    (SETQ IMAGEN:PAGE-START-LIST (NREVERSE IMAGEN:PAGE-START-LIST)))
  (LET ((PAGE-BUFFER
	  (MAKE-ARRAY
	    (LOOP FOR PAGE-START IN IMAGEN:PAGE-START-LIST WITH PREVIOUS-PAGE-START = LAST-PAGE-END
		  MAXIMIZE (- PREVIOUS-PAGE-START PAGE-START) DO
		  (SETQ PREVIOUS-PAGE-START PAGE-START))
	    :ELEMENT-TYPE '(UNSIGNED-BYTE 8) :FILL-POINTER 0)))
    (SETQ PRINTER-STREAM IMAGEN:TRUE-PRINT-STREAM)  ;1 Now we talk to the real printer!*
    (SEND SELF :START-IMAGEN-DOCUMENT)		1     *;1 Start the document.*
    (DOTIMES (COPY IMAGEN:NCOPIES)		    ;1 Do multiple copies*
      (LOOP FOR PAGE-START IN IMAGEN:PAGE-START-LIST	    ;1 Then read the pages in in reverse order*
	    WITH PAGE-END = LAST-PAGE-END DO	1     *;1    and print them.*
	    (WITH-INPUT-FROM-STRING (TEMP-IN BUFFER :START PAGE-START)
	      (SEND TEMP-IN
		    :STRING-IN () PAGE-BUFFER 0 (- PAGE-END PAGE-START))
	      (SEND IMAGEN:TRUE-PRINT-STREAM
		    :STRING-OUT PAGE-BUFFER)
	      (SETQ PAGE-END PAGE-START))))
    (SEND SELF :FINISH-IMAGEN-DOCUMENT)))	1     *;1 Finish off print.*


;
;1 Modify the :READ-POINTER message to measure the length of the buffer -- JNN*
;
(DEFMETHOD (IMAGEN-PRINTER :START-NEW-PAGE) ()
  2"Send a form feed and possibly print a page heading"*
  (UNLESS (AND (= IMAGEN:VERTICAL-POSITION IMAGEN:START-OF-TEXT)
	       (NOT (= IMAGEN:LAST-CHAR #\FF)))
      (SEND SELF :STRING-OUT-RAW (IMAGEN:IMPRESS-ENDPAGE))  ;1 Finish this page.*
      (LET (BUFFER)
	(SETQ BUFFER (GET-OUTPUT-STREAM-STRING PRINTER-STREAM))
	(SETQ IMAGEN:PAGE-START-LIST (CONS (LENGTH BUFFER) IMAGEN:PAGE-START-LIST))
	(SEND PRINTER-STREAM :STRING-OUT BUFFER))
      (SEND SELF :STRING-OUT-RAW		1     *;1 Start the next page.*
	    (STRING-APPEND
	      (IMAGEN:IMPRESS-SET-ABS-V
		:NEW-V IMAGEN:*TOP-MARGIN-HEIGHT*)  ;1 1/2" top margin.*
	      (IMAGEN:IMPRESS-SET-ABS-H
		:NEW-H IMAGEN:*LEFT-MARGIN-WIDTH*)  ;1 1/2" left-hand *
	      (IMAGEN:IMPRESS-SET-BOL
		:LINE-BEGIN IMAGEN:*LEFT-MARGIN-WIDTH*)))
      (SETQ IMAGEN:HORIZONTAL-POSITION 0)
      (SETQ IMAGEN:VERTICAL-POSITION 0)
      (WHEN PAGE-WAIT
	(FORMAT T "~&Type any character when ready for next page:")
	(TYI))
      (IF PAGE-HEADING
	(SEND SELF :PRINT-PAGE-HEADING))
      (SETQ IMAGEN:START-OF-TEXT IMAGEN:VERTICAL-POSITION)
      (SETQ IMAGEN:HORIZONTAL-POSITION 0)
      (SEND SELF :SWITCH-PRINTER-TO-FONT IMAGEN:CURRENT-FONT-DESCRIPTOR)))


;1*********************************************************************************
;
;1 The patched definition of :BEFORE :PRINT-TEXT-FILE (in printer patch 2.5)*
;1 messes up IMPRINT-FILE (and probably other stuff); disappear it!*
;1 Added the same for :BEFORE :PRINT-RAW-FILE -- JNN*
;
;1*********************************************************************************


(DEFMETHOD (BASIC-PRINTER :BEFORE :PRINT-TEXT-FILE) (&REST IGNORE)
  "2New method which only resets the page count*"
  (SETQ PAGE-COUNT 0))

(DEFMETHOD (BASIC-PRINTER :BEFORE :PRINT-RAW-FILE) (&REST IGNORE)
  "2New method which only resets the page count*"
  (SETQ PAGE-COUNT 0))



;1*********************************************************************************
;
;1 This is a patch to allow the print-file function to use the Imagens -- JNN*
;
;1*********************************************************************************


(DEFMETHOD (FILE-PRINT-REQUEST :SEND-REMOTE-PRINT-FILE-REQUEST) (PRINTER-HOST)
  "2Send to a remote Print Server on host the print request's filename, printer name, and printer options*"
  (CASE (SEND PRINTER-HOST :SYSTEM-TYPE)
    (:LISPM (SEND SELF :LISPM-REMOTE-PRINT-FILE-REQUEST PRINTER-HOST))
    ((:VAX :VMS :VMS4 :VMS5) (SEND SELF :VAX-REMOTE-PRINT-FILE-REQUEST PRINTER-HOST))
    (:LMFS (SEND SELF :SYMBOLICS-REMOTE-PRINT-FILE-REQUEST PRINTER-HOST))
    (:IMAGEN (SEND SELF :IMAGEN-REMOTE-PRINT-FILE-REQUEST PRINTER-HOST))
    (T (SEND SELF :FOREIGN-REMOTE-PRINT-FILE-REQUEST PRINTER-HOST))))


(DEFMETHOD (FILE-PRINT-REQUEST :IMAGEN-REMOTE-PRINT-FILE-REQUEST) (PRINTER-HOST)
  "2Print file or stream FILE-NAME to a remote Imagen printer.
The value of instance variable LINES and PAGE-HEADING determines
whether the file is text or imPRESS.*"
  (IF (LISTP PAGE-HEADING) (SETQ PAGE-HEADING (CAR (LAST PAGE-HEADING))))
  (LET* ((STRING-STREAM (MAKE-STRING-OUTPUT-STREAM))
	 (PRT (SEND SELF :MAKE-PRINTER-DEVICE STRING-STREAM))
	 IN-STREAM)
    (IF (AND LINES (NOT (STRING-EQUAL " IMPRESS " PAGE-HEADING)))
	(WITH-OPEN-FILE (FILE FILE-NAME)
	  (SEND PRT :START-DOCUMENT (AND HEADER FILE-NAME) USER-NAME COPIES)
	  (SEND PRT :PRINT-TEXT-FILE FILE FONT-LIST (GET PRINT-DEVICE :PAGE-ORDER))
	  (WITH-INPUT-FROM-STRING (IN-STREAM (GET-OUTPUT-STREAM-STRING STRING-STREAM))
	    (IMAGEN:COPY-STREAM-TO-IMAGEN IN-STREAM :IMAGEN-NAME PRINTER-HOST)))
	(UNWIND-PROTECT
	    (PROGN (SETQ IN-STREAM (IF (STREAMP FILE-NAME)
				       FILE-NAME
				       (OPEN FILE-NAME :DIRECTION :INPUT :ELEMENT-TYPE '(MOD 256))))
		   (SEND PRT :START-DOCUMENT
			 (AND HEADER (IF (STREAMP FILE-NAME) "TI SCREEN DUMP" FILE-NAME))
			 USER-NAME COPIES)
		   (SEND PRT :START-IMAGEN-DOCUMENT)
		   (WITH-INPUT-FROM-STRING (STRING-STREAM
					     (CONCATENATE 'STRING
							  (FORMAT NIL "@document(copies ~D)" COPIES)
							  (GET-OUTPUT-STREAM-STRING STRING-STREAM)))
		     (IMAGEN:COPY-STREAMS-TO-IMAGEN (LIST STRING-STREAM IN-STREAM)
						    :IMAGEN-NAME PRINTER-HOST)))
	  (UNLESS (STREAMP FILE-NAME)
	    (SEND IN-STREAM :CLOSE))))
    (AND DELETE-AFTER (NOT (STREAMP FILE-NAME))
	 (DELETE-FILE FILE-NAME))
    (TV:NOTIFY () "Printed ~A to ~A"
	       (IF (STREAMP FILE-NAME) "SCREEN DUMP" FILE-NAME)
	       PRINTER-HOST)))


;1*********************************************************************************
;
;1 These are patches to get rid of the temp file when printing bitmaps -- JNN*
;
;1*********************************************************************************


(DEFMETHOD (ARRAY-PRINT-REQUEST :HANDLE-REMOTE-REQUEST) (PRINTER-HOST)
  "2Printer is on a remote host.  Copy the screen image into a temporary file
and send its file request to the print server for remote printer.*"
  (COND
    ((NOT *ALLOW-SENDING-OF-REMOTE-PRINT-REQUESTS*)
     (NOTIFY-USER-AT-HOST
      (FORMAT () "Spooling of ~A to printer ~A on ~A not allowed" SCREEN-NAME (CAR PRINT-DEVICE)
	      (GET PRINT-DEVICE :HOST))
      SENDER-HOST))
    ((AND (TYPEP BITMAP-ARRAY 'ARRAY) (EQ (ARRAY-TYPE BITMAP-ARRAY) 'ART-1B))
     (NOTIFY-USER-AT-HOST
      (FORMAT () "Spooling ~A print request to printer ~A on ~A" SCREEN-NAME (CAR PRINT-DEVICE)
	      (GET PRINT-DEVICE :HOST))
      SENDER-HOST)
     (LET ((TEMP-FILE (IF (EQL (SEND PRINTER-HOST :SYSTEM-TYPE) :IMAGEN)
			  (SEND SELF :COPY-ARRAY-INTO-TEMP-STREAM)
			  (SEND SELF :COPY-ARRAY-INTO-TEMP-FILE)))
	   FILE-REQUEST)
       (UNLESS (ERRORP TEMP-FILE)
	 (SETQ FILE-REQUEST
	       (MAKE-INSTANCE 'FILE-PRINT-REQUEST :PRINT-DEVICE PRINT-DEVICE :FILE-NAME
			      TEMP-FILE :HEADER-NAME SCREEN-NAME :HEADER HEADER :USER-NAME
			      USER-NAME :SENDER-HOST SENDER-HOST :COPIES COPIES :LINES ()
			      :DELETE-AFTER T
			      ))
	 (SEND FILE-REQUEST :SEND-REMOTE-PRINT-FILE-REQUEST PRINTER-HOST))))
    (T
     (NOTIFY-USER-AT-HOST
      (FORMAT () "~A not a bit array in print request queue entry: ~A" BITMAP-ARRAY SELF)
      SENDER-HOST))))


(DEFMETHOD (ARRAY-PRINT-REQUEST :COPY-ARRAY-INTO-TEMP-STREAM) (&AUX PRINTER)
  "2Copy bitmap array into a temporary stream and returning the input stream*"
  (LET ((FILE-STREAM (MAKE-STRING-OUTPUT-STREAM)))
    (SETQ PRINTER (SEND SELF :MAKE-PRINTER-DEVICE FILE-STREAM))
    (SEND PRINTER :PRINT-BITMAP BITMAP-ARRAY WIDTH HEIGHT START-X START-Y ORIENTATION
	  DOTS-PER-INCH)
    (MAKE-STRING-INPUT-STREAM (GET-OUTPUT-STREAM-STRING FILE-STREAM))))


;1*********************************************************************************
;
;1  Fixed the painfully slow version of screen dumps by using BITBLT rather*
;1    than loops, and using Imagen magnification whenever possible -- JNN*
;
;1*********************************************************************************

(DEFMETHOD (PRINTER::IMAGEN-PRINTER :PRINT-BITMAP)
	   (BITMAP-ARRAY
	    &OPTIONAL WIDTH HEIGHT (START-X 0) (START-Y 0) (ROTATION NIL) IGNORE
	    &AUX REAL-MAG MAGNIFICATION (POWER-OF-TWO 0))
  "2Copy the bitmap array to an Imagen print stream.*"

  (IF (NULL WIDTH) (SETQ WIDTH (ARRAY-DIMENSION BITMAP-ARRAY 1)))
  (IF (NULL HEIGHT) (SETQ HEIGHT (ARRAY-DIMENSION BITMAP-ARRAY 0)))
  (SEND SELF :START-IMAGEN-DOCUMENT)

  ;1; Calculate magnification, top/bottom margins, and left/right margines based on rotation and the*
  ;1; size of the area to be printed...*
  (IF (OR (NOT ROTATION) (EQL ROTATION :BEST))
      (SETQ ROTATION (IF (> WIDTH HEIGHT) :LANDSCAPE :PORTRAIT)))
  (IF (EQL ROTATION :LANDSCAPE)
      (SETQ MAGNIFICATION (MIN (FLOOR *PRINTABLE-PIXELS-ACROSS-A-PAGE* HEIGHT)
			       (FLOOR *PRINTABLE-PIXELS-DOWN-A-PAGE* WIDTH)))
      (SETQ MAGNIFICATION (MIN (FLOOR *PRINTABLE-PIXELS-DOWN-A-PAGE* HEIGHT)
			       (FLOOR *PRINTABLE-PIXELS-ACROSS-A-PAGE* WIDTH))))

  ;1 Use Imagen magnification as much as possible*
  (LOOP WHILE (AND (< POWER-OF-TWO 2) (ZEROP (LOGAND MAGNIFICATION 1))) DO
	(INCF POWER-OF-TWO)
	(SETQ MAGNIFICATION (/ MAGNIFICATION 2)))

  ;1 Use Imagen magnification totally if magnified array is too large*
  (LOOP WHILE (AND (OR (> (* MAGNIFICATION HEIGHT) IMAGEN:*BIGGEST-MANUAL-MAGNIFICATION*)
		       (> (* MAGNIFICATION WIDTH) IMAGEN:*BIGGEST-MANUAL-MAGNIFICATION*))
		   (> MAGNIFICATION 1)) DO
	(DECF MAGNIFICATION)
	(LOOP WHILE (AND (< POWER-OF-TWO 2) (ZEROP (LOGAND MAGNIFICATION 1))) DO
	      (INCF POWER-OF-TWO)
	      (SETQ MAGNIFICATION (/ MAGNIFICATION 2))))

  ;1 Send Imagen magnification*
  (UNLESS (ZEROP POWER-OF-TWO)
    (SEND SELF :STRING-OUT-RAW (IMAGEN:IMPRESS-SET-MAGNIFICATION :POWER POWER-OF-TWO)))
  (SETQ REAL-MAG (LSH MAGNIFICATION POWER-OF-TWO))

  ;1 Compute margins*
  (IF (EQL ROTATION :LANDSCAPE)
      (SEND SELF :STRING-OUT-RAW
	    (CONCATENATE 'STRING
			 (IMAGEN:IMPRESS-SET-HV-SYSTEM :ORIGIN 2     ;1 top-left origin after rotation.*
						       :AXES 2	1  *;1 clockwise axes.*
						       :ORIENTATION 5)1  *;1 Rotate 90 degress.*
			 (IMAGEN:IMPRESS-SET-ABS-H
			   :NEW-H (+ 25 (FLOOR (- *PRINTABLE-PIXELS-DOWN-A-PAGE*
						  (* REAL-MAG WIDTH)) 2)))
			 (IMAGEN:IMPRESS-SET-ABS-V
			   :NEW-V (+ 125 (FLOOR (- *PRINTABLE-PIXELS-ACROSS-A-PAGE* 
						   (* REAL-MAG HEIGHT)) 2)))))
      (SEND SELF :STRING-OUT-RAW
	    (CONCATENATE 'STRING
			 (IMAGEN:IMPRESS-SET-ABS-H
			   :NEW-H (+ 125 (FLOOR (- *PRINTABLE-PIXELS-ACROSS-A-PAGE*
						   (* REAL-MAG WIDTH)) 2)))
			 (IMAGEN:IMPRESS-SET-ABS-V
			   :NEW-V (+ 25 (FLOOR (- *PRINTABLE-PIXELS-DOWN-A-PAGE*
						  (* REAL-MAG HEIGHT)) 2))))))
  (IF (= MAGNIFICATION 1)
      (IMAGEN:OUTPUT-BITMAP-TO-PRINTER BITMAP-ARRAY HEIGHT WIDTH START-X START-Y)
      (IMAGEN:OUTPUT-MAGNIFIED-BITMAP-TO-PRINTER BITMAP-ARRAY HEIGHT WIDTH START-X START-Y
						 MAGNIFICATION))
  (SEND SELF :FINISH-IMAGEN-DOCUMENT))


;1 Output a bitmap whose START-X and WIDTH values are multiples of 8 so*
;1   that reversed-bytes can be computed easily.*
;

(DEFUN IMAGEN:OUTPUT-BITMAP-TO-PRINTER (BITMAP-ARRAY HEIGHT WIDTH START-X START-Y
					&AUX (IMAGEN-HEIGHT (CEILING HEIGHT 32))
					(IMAGEN-WIDTH (CEILING WIDTH 32)))
  "2Convert the bitmap to imPRESS format and output to print stream.*"
  (DECLARE (:SELF-FLAVOR PRINTER::IMAGEN-PRINTER))
  
  ;1; Adjust array so that we can use BITBLT on it*
  (UNLESS (ZEROP (LOGAND (ARRAY-DIMENSION BITMAP-ARRAY 1) 31))
    (SETQ BITMAP-ARRAY (ADJUST-ARRAY BITMAP-ARRAY
				     (LIST (LSH (CEILING WIDTH 32) 5) HEIGHT))))

  ;1;   Send out the BITMAP command and its initial arguments...*
  (SEND SELF :STRING-OUT-RAW
	(IMAGEN:IMPRESS-BITMAP :OPERATION-TYPE 7    ;1 "OR" operation.*
			       :HSIZE IMAGEN-WIDTH	1     *;1 # of patches wide.*
			       :VSIZE IMAGEN-HEIGHT))	1     *;1 # of patches high.*

  ;1;   Actually output the bitmap to the printer...*
  (LET* ((TEMP-ARRAY (MAKE-ARRAY (LIST 32 32) :ELEMENT-TYPE '(MOD 2)))
	 (TEMP-1D (MAKE-ARRAY 128 :ELEMENT-TYPE '(MOD 256) :DISPLACED-TO TEMP-ARRAY)))
    (LOOP FOR Y FROM 0 BELOW (LSH IMAGEN-HEIGHT 5) BY 32 DO
	  (LOOP FOR X FROM 0 BELOW (LSH IMAGEN-WIDTH 5) BY 32 DO
		(BITBLT TV:ALU-SETA 32 32
			BITMAP-ARRAY (+ START-X X) (+ START-Y Y)
			TEMP-ARRAY 0 0)
		(SEND SELF :STRING-OUT-RAW
		      (MAP 'STRING #'(LAMBDA (X)
				       (AREF IMAGEN:*FLIPPED-BITS-TABLE* X)) TEMP-1D))))))

;1 Magnify a bitmap*

(DEFUN IMAGEN:OUTPUT-MAGNIFIED-BITMAP-TO-PRINTER
       (BITMAP-ARRAY HEIGHT WIDTH START-X START-Y MAGNIFICATION
	&AUX
	(MAGNIFIED-HEIGHT (* HEIGHT MAGNIFICATION))
	(MAGNIFIED-WIDTH (* WIDTH MAGNIFICATION))
	MAGNIFIED-BITMAP-HEIGHT
	MAGNIFIED-BITMAP-WIDTH)
  "2Output a magnified copy of the bitmap to the print stream*"

  (DECLARE (:SELF-FLAVOR PRINTER::IMAGEN-PRINTER))

  (SETQ MAGNIFIED-BITMAP-HEIGHT (LSH (CEILING MAGNIFIED-HEIGHT 32) 5))
  (SETQ MAGNIFIED-BITMAP-WIDTH (LSH (CEILING MAGNIFIED-WIDTH 32) 5))
  ;1; Adjust array so that we can use BITBLT on it*
  (UNLESS (ZEROP (LOGAND (ARRAY-DIMENSION BITMAP-ARRAY 1) 31))
    (SETQ BITMAP-ARRAY (ADJUST-ARRAY BITMAP-ARRAY
				     (LIST (LSH (CEILING WIDTH 32) 5) HEIGHT))))
  (USING-RESOURCE
    (MAGNIFIED-BITMAP-ARRAY PRINTER::SCREEN-IMAGE-BIT-ARRAY
			    MAGNIFIED-BITMAP-WIDTH MAGNIFIED-BITMAP-HEIGHT)
1     *;1; First, zero out the entire magnified bitmap array...*
    (BITBLT TV::ALU-SETZ (ARRAY-DIMENSION MAGNIFIED-BITMAP-ARRAY 1)
	    (ARRAY-DIMENSION MAGNIFIED-BITMAP-ARRAY 0) MAGNIFIED-BITMAP-ARRAY 0 0
	    MAGNIFIED-BITMAP-ARRAY 0 0)
    ;1; Then build a magnified version of the given array...*
    ;1; Use BITBLT rather than the loops used by TI -- JN*
    (LOOP FOR X FROM 0 BELOW WIDTH DO
	  (LOOP FOR XX FROM 0 BELOW MAGNIFICATION DO
		(BITBLT TV::ALU-SETA 1 HEIGHT
			BITMAP-ARRAY (+ X START-X) START-Y
			MAGNIFIED-BITMAP-ARRAY (+ (* X MAGNIFICATION) XX) 0)))
    (LOOP FOR Y FROM (1- HEIGHT) DOWNTO 0 DO
	  (LOOP FOR YY FROM 0 BELOW MAGNIFICATION DO
		(BITBLT TV::ALU-SETA (* WIDTH MAGNIFICATION) 1
			MAGNIFIED-BITMAP-ARRAY 0 Y
			MAGNIFIED-BITMAP-ARRAY 0 (+ (* Y MAGNIFICATION) YY))))
1      *;1; Finally, actually output the bitmap to the printer...*
    (IMAGEN:OUTPUT-BITMAP-TO-PRINTER
      MAGNIFIED-BITMAP-ARRAY MAGNIFIED-BITMAP-HEIGHT MAGNIFIED-BITMAP-WIDTH 0 0)))


;1*********************************************************************************
;
;1 New flavor definition for namespace printer type -- JNN*
;
;1*********************************************************************************

(DEFFLAVOR IMPRESS () (IMAGEN-PRINTER))

(COMPILE-FLAVOR-METHODS IMAGEN-PRINTER IMPRESS)

(ADD-INITIALIZATION "Cache printer attributes"
		    '(LIST-PRINTERS :local nil)
		    '(:warm))