;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-

;;; GREYNETIC: the world's most astounding moving sculpture.  Now in technicolor!!
;;;
;;; ChangeLog:
;;;
;;; 12 May 87  Jamie Zawinski  Created.
;;;  6 Oct 87  Jamie Zawinski  Added the QUIT-WHEN-MOUSE-MOVED option, and made it restore the contents of the main screen.
;;;  7 Nov 87  Jamie Zawinski  Opened the mouse blinker during the save/restore blitting so that it doesn't leave ghosts.
;;; 24 Oct 88  Jamie Zawinski  Added color support.  Regretfully removed GREYNETIC-ON-SCREEN - it's real hard in color.
;;;  1 Dec 88  Jamie Zawinski  Got non-pastels working much more efficiently, re-implemented GREYNETIC-ON-SCREEN.
;;; 30 Dec 88  Jamie Zawinski  Made GREYNETIC-INTERNAL work on TV windows as well as W windows, and incidentally sped it up
;;;                             by about two times.
;;; 15 Apr 89  Jamie Zawinski  Moved the definition of GREYNETIC-SCREENHACK to this file from NEW-SCREENHACKS.LISP.

;;;
;;; GREYNETIC (&optional pastel-p x y w h)
;;;  Generate random grey (or color) rectangles on a window of size X Y W H.
;;;  If the size is not specified and the window doesn't already exist, the user is asked to specify it with the mouse.
;;;  If we are on a color system and PASTEL-P is non-NIL, then the colors will be soothing Miami-vice tones...
;;;
;;; GREYNETIC-ON-SCREEN ()
;;;  Generate random grey (or color) rectangles on the default screen directly.
;;;  Stops when the mouse is moved.  It is not possible to generate pastels with this one.
;;;


(defvar *greynetic-window* nil)

(defun greynetic (&optional pastel-p x y w h)
  "Generate random grey (or color) rectangles on a window of size X Y W H.
  If the size is not specified and the window doesn't already exist, the user is asked to specify it with the mouse.
  If we are on a color system and PASTEL-P is non-NIL, then the colors will be soothing Miami-vice tones..."
  (make-random-state t)
  (unless *greynetic-window*
    (setq *greynetic-window* (make-instance 'w:window :edges-from (if (and x y w h)
								      (list x y (+ x w) (+ y h))
								      :mouse)
					    :label nil  :name "Greynetic")))
  (unless (or (not (and x y w h))
	      (equalp (list x y (+ x w) (+ y h))
		      (send *greynetic-window* :edges)))
    (send *greynetic-window* :set-edges x y (+ x w) (+ y h)))
  (let* ((win tv:selected-window))
    (send *greynetic-window* :select)
    (dolist (x (send *greynetic-window* :blinker-list)) (send x :set-visibility :off))
    (greynetic-internal *greynetic-window*
			(send *greynetic-window* :width)
			(send *greynetic-window* :height)
			nil TV:ALU-SETA pastel-p)
    (send *greynetic-window* :clear-input)
    (send *greynetic-window* :deactivate)
    (send win :select))
  nil)


(defun greynetic-on-screen (&optional pastel-p)
  "Generate random grey (or color) rectangles on the default screen directly.
  Stops when the mouse is moved.  It is not possible to generate pastels with this one."
  (let* ((screen tv:default-screen)
	 (doc-win tv:who-line-documentation-window)
	 (obg (tv:sheet-background-color doc-win))
	 (color-p (tv:color-system-p doc-win))
	 (font FONTS:CMR10))
    (unwind-protect
	(progn
	  (send doc-win :set-background-color (if color-p W:RED W:BLACK))
	  (send doc-win :clear-screen)
	  (sheet-display-x-y-centered-string doc-win "GREYNETIC: move the mouse to flush."
					     0 0 (tv:sheet-width doc-win) (tv:sheet-height doc-win)
					     font (if color-p TV:ALU-TRANSP TV:ALU-XOR) 0 nil
					     (tv:font-char-height font) (if color-p W:CYAN W:WHITE))
	  (greynetic-internal screen (tv:sheet-inside-width screen) (tv:sheet-inside-height screen) t W:ALU-SETA pastel-p))
      (send doc-win :set-background-color obg)
      (send screen :refresh)
      (send doc-win :refresh)))
  (values))

(defun greynetic-internal (window width height &optional quit-when-mouse-moved (alu W:ALU-SETA) stipple)
  "GREYNETIC the sheet using BITBLT.
 If QUIT-WHEN-MOUSE-MOVED is T, it exits at any motion.  If NIL, it exits when a character is typed or a button is clicked."
  (unless (w:color-system-p window) (setq stipple t))  ; If we don't have color, we *must* use stipple patterns.
  (let* ((color-p (w:color-system-p window))
	 (patterns (if stipple
		       (vector W:100%-WHITE W:12%-GRAY W:25%-GRAY W:33%-GRAY W:50%-GRAY
			       W:66%-GRAY W:75%-GRAY W:88%-GRAY W:100%-BLACK)
		       (vector W:100%-BLACK)))
	 (numpatterns (length patterns))
	 (numcolors 255)
	 (start-time tv:kbd-last-activity-time)
	 (start-mx tv:mouse-x)
	 (start-my tv:mouse-y))
    (do* ()
	 ((or (/= start-time tv:kbd-last-activity-time)
	      (and quit-when-mouse-moved
		   (or (/= start-mx tv:mouse-x) (/= start-my tv:mouse-y))))
	  (when quit-when-mouse-moved (setq tv:kbd-last-activity-time (time:time))))
      (declare (inline random))
      (let* ((x (random (- width 20)))
	     (y (random (- height 20)))
	     (width (+ 10 (random (- width x 10))))
	     (height (+ 10 (random (- height y 10))))
	     (color-number (if color-p (random numcolors) W:BLACK))
	     (pattern (svref patterns (random numpatterns))))
	(declare (fixnum x y width height color-number))
	(sys:without-interrupts
	  (if stipple
	      ;; If we want stipple patterns, then we must use this method, which calls the microcode triangle primitive.
	      (tv:prepare-sheet (window)
		(bitblt alu width height
			pattern (rem y (array-dimension pattern 1)) (rem x (array-dimension pattern 0))
			(tv:sheet-screen-array window) x y))
	      ; The above is faster than the following method call, and it also works on both TV windows and W windows.
	      ;(send window :draw-filled-rectangle x y width height color-number alu t pattern)
	      
	      ;; If we want solids, we can call the microcode rectangle primitive directly, and really fly.
	      (tv:prepare-color (window color-number)
		(tv:prepare-sheet (window)
		  (sys:%draw-rectangle width height x y alu window)
		  ))))))))


;;; Greynetic's interface to the ScreenSaver.

(defun greynetic-screenhack (window)
  (send window :clear-screen)
  (greynetic-internal window (tv:sheet-inside-width window) (tv:sheet-inside-height window)))

(when (boundp 'tv:*screen-saver-hacks-list*)
  (pushnew 'GREYNETIC-SCREENHACK tv:*screen-saver-hacks-list*))
