;;; -*- Mode:Common-Lisp; Package:TV; Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B TR12BI ILSYMBOLS10); Base:10 -*-

;1;; File "*EDIT-PRINT-QUEUE1"*
;1;; Adds a mode to PEEK for displaying and editing the Print Queue across the net.*
;1;; Written and maintained by Jamie Zawinski.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;    4 Nov 88*	1Jamie Zawinski*	1Created.*
;1;;    9 Nov 88*	1Jamie Zawinski*	1Got it working.*
;1;;   25 Jan 89*	1Jamie Zawinski *	1Made the Host line say whether a connection has been established.*
;1;;   26 Jan 89*	1Jamie Zawinski *	1Made the percent through the file be displayed as well.*
;1;;*				1Redesigned it so that network access takes place in a background process instead of under redisplay.*
;1;;*				Added documentation.
;1;;   31 Jan 89*	1Jamie Zawinski *	1Closing the connection to one host and opening it to another was getting an (albeit graceful) error.*
;1;;*				1 Made the background process be reset before the connection is closed, and made the background process*
;1;;*				1 realize that this is what was happening, so that it doesn't close the connection again.*
;1;;     2 Feb 89*	1Jamie Zawinski *	1Made 4REMOTE-PRINT-QUEUE-SERVER* ignore lost connection conditions, so that these processes don't*
;1;;*				1 keep hanging around in the "selected" state.*
;1;;    15 Feb 88*	1Jamie Zawinski*	1Fixed the bug where you weren't able to view the print-queue on the local machine.... oops.*
;1;;*				1Fixed bug where file percent and bytes were printing wrong on local machine.... oops.*
;1;;*				1Made the "select host" menu note which host was the local machine, if LM has a printer attached.*
;1;;*


;1;; In order to install itself in 5PEEK*, this file redefines the flavors 5BASIC-PEEK* and5 PEEK-FRAME*,*
;1;; and the variable 5PEEK-MODES*.  It also rebuilds the5 UCL-PEEK-MENU* and5 PEEK-MODE-CMD-TABLE* so that*
;1;; the change to5 PEEK-MODES* takes effect.*
;1;;*
;1;; 5Design Notes:**
;1;;*
;1;; It used to be that the function *SNARF-QUEUE1 went to the net-connection and updated the queue.  This function is called every few seconds*
;1;; under *TV:SCROLL-MAINTAIN-LIST1, which is to say, under the window's redisplay.*
;1;; *
;1;; There were problem with this.  Since network access was happening under redisplay (!) a network waits would keep the window system*
;1;; locked up for longer than necessary.  Also, if the remote server died, the error was signalled while the sheet is locked - which means an*
;1;; error message couldn't come out until 5Term Control-Clear-Input* was typed.*
;1;;*
;1;; The current implementation is much sounder.  Each Print Queue Editor has a background process which performs the network access for it.*
;1;; The function *SNARF-QUEUE1 merely 5enable*s this process, which then wakes up, reads from the remote server, and 5arrest*s itself again.*
;1;;*
;1;; If a network error occurs, it is handled like any other - in the background stream of the Print Queue Client process.*
;1;;*
;1;;*
;1;; How do we get the information from the server to the client compactly?*
;1;; The first time a connection is made, the entire remote print-queue is dumped to the client as a list of forms.*
;1;; Elements of this list are forms which evaluate to *PRINTER:PRINT-REQUEST1 structures duplicating the ones on the server.*
;1;;*
;1;; Every few seconds, the server is polled.  It replies with the ID numbers of all of the requests currently in the queue.*
;1;; The client notices when ID numbers are missing from the queue which had been present, and deletes the corresponding local copy of the*
;1;; print request.  When an ID appears which the client has not yet seen, it contacts the server again, and the server replies with a form to*
;1;; reproduce the request of the indicated number.  So each print request is transmitted from the server to the client exactly once, and all of*
;1;; the IDs are transmitted often.*
;1;;*
;1;; There are two kinds of operations one can perform on a print queue: change the information in a print request, and change the position of*
;1;; the request in the queue.  It is not useful to do either of these on the first element of the queue, because it has already begun being printed.*
;1;; However, it is possible to kill the topmost job before it completes.*
;1;;*
;1;;*
;1;; The following information is displayed for every file in the queue:*
;1;;*
;1;;    7h2  the name of the file***
;1;;    7h2  the user and host from which it originated***
;1;;    7h2  the name of the printer it is destined for***
;1;;    7h2  whether a header page is printed***
;1;;    7h2  the size of the font it will be printed in***
;1;;    7h2  how many copies are being printed***
;1;;    7h2  whether and what the page heading is***
;1;;*
;1;; For all but the frontmost file, all of the above items are mouse-sensitive.*
;1;; Clicking on the file name brings up a menu allowing the user to send this job to the front or back, or to delete it.*
;1;; Clicking on any other item allows the user to edit it.*
;1;;*
;1;; The number of bytes and percent shipped of the frontmost file is also displayed.  This updates every few seconds like everything else.*
;1;; There is an option for clearing the entire queue, and for deleting all requests whose IDs fall between a lower and upper bound.*
;1;;*
;1;; If the server connection is lost, the user is notified, and the local machine's queue is displayed (for lack of a better thing).*
;1;;*
;1;; All commands function exactly the same on a remote queue and the local queue.  It is possible to have multiple instances of PEEK, with*
;1;; each one looking at a different (or same) queue.*
;1;;*


(defstruct 4(remote-print-queue-descriptor*
	     (:print-function %print-pq-desc)
	     (:conc-name "3PQ-DESC-*"))
  "2This is a structure which represents the state of a remote print queue, and the connection to its server.*"
  
  (host   nil :type (or null net:host))   ;1 The machine connected to.  *NIL1 = local machine.*
  (conn   nil :type (or null chaos:conn)) ;1 The connection object.*
  (stream nil :type (or null stream))     ;1 A stream to the connection.*
  (queue  nil :type list)                 ;1 A list of *PRINT-REQUEST1 structures, representing the queue on the* HOST1.*
  (file-info () :type list)		  ;1 The percent-shipped and byte-count of the file at the front of the queue.*
  
  (connection-process nil :type (or null process))    ;1 The process that handles network communication.*
  (connection-status nil :type (member :ERROR :OPENING :EXITING NIL))
  
  (lock nil)    ;1 To prevent overlapping access of the net connection.  I'm not sure this is necessary any more, but it doesn't hurt.*
  )


(defsubst 4pq-desc-in-error-p *(pq-desc)
  (eq :ERROR (pq-desc-connection-status pq-desc)))

(defsubst 4pq-desc-exiting-p *(pq-desc)
  (eq :EXITING (pq-desc-connection-status pq-desc)))

(defsubst 4pq-desc-opening-p *(pq-desc)
  (eq :OPENING (pq-desc-connection-status pq-desc)))


(defun 4%print-pq-desc* (struct stream depth)
  (declare (ignore depth))
  (princ "3#<*" stream)
  (prin1 (type-of struct) stream)
  (princ "3 for *" stream)
  (princ (or (pq-desc-host struct) "3LM*") stream)
  (case (pq-desc-connection-status struct)
    (NIL nil)
    (:ERROR   (princ "3 - in error*" stream))
    (:OPENING (princ "3 - opening*" stream))
    (:CLOSING (princ "3 - closing*" stream))
    (t (format stream "3 - ~S*" stream)))
  (princ "3>*" stream))


(defun 4pq-desc* (peek-pane)
  "2Returns the* REMOTE-PRINT-QUEUE-DESCRIPTOR2 in use by this Print Queue Editor pane.*"
  (send peek-pane :pq-desc))


(defmacro 4locking-pq-conn* ((pq-desc) &body body)
  "2Hold a lock on the* REMOTE-PRINT-QUEUE-DESCRIPTOR2, so that we can safely multitask.*"
  `(with-lock ((pq-desc-lock ,pq-desc) :whostate "3PQ Lock*")
     ,@body))


(defun 4open-pq-conn* (pq-desc)
  "2Open a connection to the remote Print Queue server on the host of the* PQ-DESC2, if it is not already open.
  No net-connection is made if the host is NIL, meaning local machine.*"
  (locking-pq-conn (pq-desc)
    (let* ((host (pq-desc-host pq-desc)))
      (when (and host (not (pq-desc-conn pq-desc)))  ;1 When it is a remote host, and there is no connection yet.*
	(setf (pq-desc-connection-status pq-desc) :OPENING) ;1 This is set to *NIL1 after the first read, in *PRINT-QUEUE-CLIENT-INTERNAL1.*
	(let* ((conn (chaos:connect host "3REMOTE-PRINT-QUEUE*"))
	       (stream (chaos:make-stream conn)))
	  (setf (pq-desc-conn pq-desc) conn
		(pq-desc-stream pq-desc) stream
		(pq-desc-queue pq-desc) nil
		(pq-desc-connection-status pq-desc) nil)))))
  pq-desc)


(defun 4close-pq-conn* (pq-desc &optional error-p)
  "2Close the connection in the PQ-DESC if it is open, and clear the PQ-DESC structure.
  If *ERROR-P2 is non-NIL, then mark the structure as being in a bad way.*"
  (setf (pq-desc-connection-status pq-desc)
	(if error-p :ERROR :EXITING))
  (unwind-protect
      (locking-pq-conn (pq-desc)
	;1;*
	;1; Allow the background process to run once more.*
	(send (pq-desc-connection-process pq-desc) :reset)
	(send (pq-desc-connection-process pq-desc) :revoke-arrest-reason :PRINT-QUEUE-WAIT)
	
	(setf (pq-desc-host pq-desc) nil
	      (pq-desc-queue pq-desc) nil
	      (pq-desc-file-info pq-desc) nil)
	(condition-call (condition)
	    (cond ((pq-desc-stream pq-desc)
		   (unwind-protect
		       (progn
			 (write-line "3BYE*" (pq-desc-stream pq-desc))
			 (close (pq-desc-stream pq-desc)))
		     (setf (pq-desc-stream pq-desc) nil
			   (pq-desc-conn pq-desc) nil)))
		  ((pq-desc-conn pq-desc)
		   (unwind-protect (chaos:close-conn (pq-desc-conn pq-desc))
		     (setf (pq-desc-conn pq-desc) nil))))
	  ;1;*
	  ;1; Trap errors which occur while trying to close a lost connection.  Who cares?  We know it's gone!*
	  ((condition-typep condition 'NET:4BAD-CONNECTION-STATE*)
	   nil)))
    (setf (pq-desc-connection-status pq-desc) nil))
  pq-desc)


(defmacro 4with-conn* ((stream-var pq-desc &optional string print-request)
		     &body body)
  "2Initialize the* PQ-DESC2 to communicate with a remote print queue server, if it is not already open.
  If the* PQ-DESC2 is pointing to the local machine, the *BODY2 is not executed.
  If* STRING2 is supplied, then *STRING2 and a newline will be written before body executes.
  If* PRINT-REQUEST2 is supplied as well, then the request number of the print request (and a newline)
    will be written after* STRING2 but before* BODY2.
  A* FORCE-OUTPUT2 will be done after* BODY2 terminates, but if *BODY2 wishes to read from the stream, it must do a *FORCE-OUTPUT2 first.
  *BODY2's values are returned.*"
  (let* ((pqd (gensym)))
    `(let* ((,pqd ,pq-desc))
       (locking-pq-conn (,pqd)
	 (open-pq-conn ,pqd)		         ;1 Open a connection if not already open.*
	 (let* ((,stream-var (pq-desc-stream ,pqd)))
	   (when ,stream-var	                 ;1 When not connected to the local machine:*
	     
	     ;1; Write the string if supplied.*
	     ,@(when string `((write-line ,string ,stream-var)
			      ,@(when print-request
				  `((princ (send ,print-request :request-no) ,stream-var)
				    (terpri ,stream-var)))))
	     
	     ;1; Execute the body and force-output.*
	     (unwind-protect (progn ,@body)
	       (send ,stream-var :force-output))))))))



;1;; Taken from 3SYS:DEBUG-TOOLS;PEEK.LISP#40.**
;1;; Added an instance variable, PQ-DESC, to hold a *REMOTE-PRINT-QUEUE-DESCRIPTOR1 structure.*
;1;; This wants to be in an instance variable so that there can be multiple instances of 5PEEK* looking at different print queues.*
;1;;*
(DEFFLAVOR 4BASIC-PEEK* ((NEEDS-REDISPLAY NIL)
		       (pq-desc (make-remote-print-queue-descriptor)))     ;1 5New*  - JWZ.*
   (SCROLL-MOUSE-MIXIN
    BORDERS-MIXIN
    scroll-bar-mixin
    SCROLL-WINDOW-WITH-TYPEOUT FULL-SCREEN-HACK-MIXIN)
  :SETTABLE-INSTANCE-VARIABLES
  :GETTABLE-INSTANCE-VARIABLES
  (:DEFAULT-INIT-PLIST :LABEL "3Peek*"
		       :TRUNCATION NIL)
  (:DOCUMENTATION :SPECIAL-PURPOSE 2"The actual peek window.  This has the capability3 *to display in a PEEK display mode.*"))


;1;; Taken from 3SYS:DEBUG-TOOLS;PEEK.LISP#40.**
;1;; Added *PQ-CONN1 to the *:BLIP-ALIST1.*
;1;;*
(DEFFLAVOR 4peek-frame*
	   ((peek-pane nil)			;1These variables can be used in the commands to*
	    (typeout-pane nil)			;1access the respective panes.*
	    (mode-pane nil)
	    (cmds-pane nil)
	    )
	   (UCL:COMMAND-LOOP-MIXIN
	    TV:STREAM-MIXIN
	    FRAME-DONT-SELECT-INFERIORS-WITH-MOUSE-MIXIN
	    BORDERED-CONSTRAINT-FRAME-WITH-SHARED-IO-BUFFER)
  :GETTABLE-INSTANCE-VARIABLES
  :SETTABLE-INSTANCE-VARIABLES
  :INITTABLE-INSTANCE-VARIABLES
  :SPECIAL-INSTANCE-VARIABLES
  (:DEFAULT-INIT-PLIST
    :active-command-tables '(peek-mode-cmd-table peek-cmd-table peek-other-cmd-table)
    :all-command-tables '(peek-mode-cmd-table peek-cmd-table peek-other-cmd-table)
    :menu-panes '((menu ucl-peek-menu)
		  (cmds ucl-peek-cmd-menu))
    :blip-alist '((:MENU :handle-menu-input)
		  (:DIRECT-COMMAND-ENTRY :handle-direct-command-input)
		  (:MOUSE-BUTTON :peek-handle-blip)
	          (TELNET        :peek-handle-blip)
	          (QSEND         :peek-handle-blip)
	          (EH            :peek-handle-blip)
	          (INSPECT       :peek-handle-blip)
	          (DESCRIBE      :peek-handle-blip)
		  (PQ-CONN       :peek-handle-blip)         ;1 5NEW* - JWZ.*
		  (HOST:HOST-STATUS   :peek-handle-blip))       ;1; LS 11/06/86*
    :save-bits :delayed
    :typein-handler nil
    :basic-help '(peek-documentation-cmd)))



;1;; Taken from 3SYS:DEBUG-TOOLS;PEEK.LISP#40.**
;1;; Added handling of the *PQ-CONN1 blip.*
;1;;*
(DEFMETHOD 4(PEEK-FRAME :PEEK-HANDLE-BLIP*) ()
  "2Handle special Peek blips.*"
  (LET ((BLIP UCL::KBD-INPUT))
    (DECLARE (SPECIAL PEEK-PANE))
    (CASE (CAR BLIP)
      (TELNET (TELNET (CADR BLIP)))
      (QSEND (QSEND (CADR BLIP))                      ;1; CHANGED 11/06/86 - LS - GOT RID OF EXTRA @*
             (SEND PEEK-PANE :SET-NEEDS-REDISPLAY T)
	     (SEND *STANDARD-OUTPUT* :MAKE-COMPLETE))
      (EH (EH (CADR BLIP)))
      (INSPECT (INSPECT (CADR BLIP)))
      (DESCRIBE (DESCRIBE (CADR BLIP)))
      (HOST:HOST-STATUS
       (if (cadr blip)
	   (HOST:HOST-STATUS (CADR blip))
	   (host:host-status)))    ;1;; NEW - LS 11/06/86*
      (:MOUSE-BUTTON (BEEP))

      (PQ-CONN                                                  ;1 5New* - JWZ.*
       (close-pq-conn (send peek-pane :pq-desc))
       (setf (pq-desc-host (send peek-pane :pq-desc)) (second blip))
       (when (pq-desc-host (send peek-pane :pq-desc))
	 (setf (pq-desc-connection-status (send peek-pane :pq-desc)) :OPENING))
       (pq-complete-redisplay peek-pane) ;1 Display an empty queue right away.*
       (open-pq-conn (send peek-pane :pq-desc)))

      (OTHERWISE
       (FERROR () "3Peek method :peek-handle-blip doesn't know what to do with blip ~A~%*" BLIP)))))


(defcommand 4peek-print-queue-cmd* ()
  '(:DESCRIPTION "2Examine or edit the Print Queue on this or another host.*"
    :NAMES ("3Print Queue*")
    :KEYS (#\q))
  (declare (special ucl-peek-menu peek-pane mode-pane))
  (let ((menu-item (dolist (item ucl-peek-menu)
		     (when (eq self (third item)) (return item)))))
    (send mode-pane :set-highlighted-items (list menu-item))
    (send peek-pane :set-label "3Print Queue*")
    (pq-complete-redisplay peek-pane)))


;1;; Taken from 3SYS:DEBUG-TOOLS;PEEK.LISP#40.**
;1;; Added *PEEK-PRINT-QUEUE-CMD1.*
;1;;*
(DEFPARAMETER 4PEEK-MODES*
   '(PEEK-WINDOW-HIERARCHY-CMD
      PEEK-PROCESSES-CMD
      PEEK-SERVERS-CMD
      PEEK-COUNTERS-CMD
      PEEK-AREAS-CMD
      PEEK-FILE-SYSTEM-CMD
      PEEK-NETWORK-CMD
      peek-histo-cmd
      peek-print-queue-cmd        ;1 5New* - JWZ.*
      ))


;1;; Taken from 3SYS:DEBUG-TOOLS;PEEK.LISP#40.*  This is unchanged, but must be in this file for the previous changes to take effect.*
(BUILD-COMMAND-TABLE 'peek-mode-cmd-table 'peek-frame
  peek-modes
  :INIT-OPTIONS
  '(:NAME "3Peek modes*"))


;1;; Taken from 3SYS:DEBUG-TOOLS;PEEK.LISP#40*.  This is unchanged, but must be in this file for the previous changes to take effect.*
(BUILD-MENU 'ucl-peek-menu 'peek-frame
  :default-item-options '(:font fonts:cptfont)
  :item-list-order peek-modes) 



;1;; The new display.*

(defvar 4*pq-columns**
	'(("3   ID:*" . 7)
	  ("3File Name:*" . 34)
	  ("3Requested By:*" . 17)
	  ("3Printer:*" . 14)
	  ("3Header:*" . 9)
	  ("3Copies:*" . 10)
	  ("3CPI:*" . 7)
	  ("3LPI:*" . 7)
	  ("3LPP:*" . 7)
	  ("3Page Header:*" . 12)
	  )
  "2A list of conses, where each cons represents one column of the Print Queue display.
  The CARs are the name, and the CDRs are how many spaces they occupy.*")


(defun 4pq-spawn* (function print-request peek-pane)
  "2Calls *PROCESS-RUN-FUNCTION2, applying *FUNCTION2 to *PRINT-REQUEST2 and *PEEK-PANE2.*"
  (process-run-function "3Print Queue Manager menu*"
			function print-request peek-pane))


(defun 4print-request-pretty-slots* (print-request slot)
  "2Returns the value of some slot of the print requests.
  Certain slots are special-cased to return prettier values.*"
  (case slot
    (:USER-AND-HOST (format nil "3~A@~A*" (send print-request :user-name) (send print-request :sender-host)))
    (:FILE-NAME     (pathname (send print-request :file-name)))
    (:PRINTER-NAME 
     (let* ((dev-list (send print-request :print-device))
	    (printer-name (if (stringp dev-list)
			      dev-list
			      (car dev-list))))
       (or printer-name "3<no printer>*")))
    (:HEADING-BRIEF (let* ((h (send print-request :page-heading)))
		      (cond ((eq h t) "3Yes*")
			    ((eq h nil) "3No*")
			    (t "3<string>*"))))
    (t (send print-request slot))))


(defun 4scroll-parse-print-request* (peek-pane print-request &optional (active-p t))
  "2Returns a* SCROLL-ITEM2 representing the print request.  If* ACTIVE-P2 is* NIL2, it will not be mousable.*"
  (let* ((list *pq-columns*))
    (flet ((item (mouse-list message width &optional (format-string "3~A*"))
	     "2Returns a list acceptable to *TV:SCROLL-PARSE-ITEM.2  This list may or may not be mouse-sensitive3,* depending on *ACTIVE-P."
	     (let* ((list `(:FUNCTION print-request-pretty-slots
				      (,print-request ,message) ,width (,format-string))))
	       (if active-p
		   `(:MOUSE ,mouse-list ,@list)
		   list))))
      
      (tv:scroll-parse-item
	`(:string ,(format nil "3~5:<~D~>  *" (send print-request :request-no)) ,(cdr (pop list)))
	
	(item `(nil :eval (pq-spawn 'pq-item-menu ',print-request ',peek-pane)
		    :documentation (:MOUSE-L-1 "3Menu of useful things to do to this print request.*"))
	      :FILE-NAME
	      (cdr (pop list)))
	
	(item `(nil :eval (pq-spawn 'pq-change-sender ',print-request ',peek-pane)
		    :documentation (:MOUSE-L-1 "3Alter the originator of this print request.*"))
	      :USER-AND-HOST
	      (cdr (pop list)))

	(item `(nil :eval (pq-spawn 'pq-change-printer ',print-request ',peek-pane)
		    :documentation (:MOUSE-L-1 "3Alter the destination printer of this print request.*"))
	      :PRINTER-NAME
	      (+ (cdr (pop list)) 1))       ;1 Add 1 to move it to the right of it's column heading (looks better).*

	(item `(nil :eval (pq-spawn 'pq-toggle-header ',print-request ',peek-pane)
		    :documentation (:MOUSE-L-1 "3Toggle whether a header page is printed.*"))
	      :HEADER
	      (- (+ (cdr (pop list)) 3) 1)  ;1 Add 3 to move it to the right, and subtract 1 to compensate for above.*
	      "3~:[No~;Yes~]*")
	
	(item `(nil :eval (pq-spawn 'pq-change-copies ',print-request ',peek-pane)
		    :documentation (:MOUSE-L-1 "3Change the number of copies printed.*"))
	      :COPIES
	      (- (cdr (pop list)) 3))       ;1 Subtract 3 to compensate for above.*
	
	(item `(nil :eval (pq-spawn 'pq-change-font-size ',print-request ',peek-pane)
		    :documentation
		    (:MOUSE-L-1 "3Change the characters per inch/line, and lines per page of this print request.*"))
	      :CPI
	      (cdr (pop list)))
	
	(item `(nil :eval (pq-spawn 'pq-change-font-size ',print-request ',peek-pane)
		    :documentation
		    (:MOUSE-L-1 "3Change the characters per inch/line, and lines per page of this print request.*"))
	      :LPI
	      (cdr (pop list)))
	
	(item `(nil :eval (pq-spawn 'pq-change-font-size ',print-request ',peek-pane)
		    :documentation
		    (:MOUSE-L-1 "3Change the characters per inch/line, and lines per page of this print request.*"))
	      :LINES
	      (+ (cdr (pop list)) 2)       ;1 Add 2 to move it to the right of it's column heading (looks better).*
	      )
	
	(item `(nil :eval (pq-spawn 'pq-change-page-heading ',print-request ',peek-pane)
		    :documentation (:MOUSE-L-1 "3Change the page heading of this print request.*"))
	      :HEADING-BRIEF
	      (- (cdr (pop list)) 2)       ;1 ditto.*
	      )
	))))


(defun 4pq-header-line-entry* ()
  "2Create a scrolling-item that displays the column heading for the Print Queue editor.*"
  (apply #'tv:scroll-parse-item (mapcar #'(lambda (x) (list :STRING (car x) (cdr x)))
					*pq-columns*)))


(defun 4pq-which-machine* (peek-pane)
  "2Returns a string describing which machine we are connected to, whether the connection is complete, and whether we are in a bad state.*"
  (let* ((desc (pq-desc peek-pane))
	 (host   (pq-desc-host desc))
	 (conn   (pq-desc-conn desc))
	 (stream (pq-desc-stream desc))
	 (error  (pq-desc-in-error-p desc))
	 (opening (pq-desc-opening-p desc))
	 )
    (cond (error   (format nil "3~A (error)*" (or host "3LM*")))
	  (opening (format nil "3~A (working)*" (or host "3LM*")))
	  ((null host) "3LM*")
	  ((or (null conn) (null stream)) (format nil "3~A (not connected)*" host))
	  (t host)
	  )))


(defun 4pq-file-percent *(peek-pane)
  "2Returns a string describing the number of bytes and percent of the file at the top of the queue which has been shipped.*"
  (let* ((pq-desc (pq-desc peek-pane))
	 (bytes (first (pq-desc-file-info pq-desc)))
	 (percent (second (pq-desc-file-info pq-desc))))
    (if (or (null bytes) (null percent)
	    (and (zerop bytes) (zerop percent)))
	""
	(format nil "3~3D% ~D*" percent bytes))))


(defun 4pq-entries* (peek-pane &rest ignore)
  "2Returns a list of SCROLL-ITEMs representing the state of the examined print queue.*"
  (list ()
	(tv:scroll-parse-item
	  "3Print Queue on host *"
	  `(:MOUSE (nil :eval (pq-spawn 'pq-change-queue nil ,peek-pane)
			:documentation (:MOUSE-L-1 "3Click here to examine a different queue.*"))
	    :FUNCTION pq-which-machine (,peek-pane) nil ("3~A:*")))
	(tv:scroll-parse-item "")
	(tv:scroll-parse-item "3Commands:*")
	(tv:scroll-parse-item "3  *"
	  `(:mouse (nil :eval (pq-spawn 'pq-cancel nil ',peek-pane)
			:documentation (:MOUSE-L-1 "3Give up on printing the current job.*"))
		   :string "3Cancel Current Job*" 20))
	(tv:scroll-parse-item "3  *"
	  `(:mouse (nil :eval (pq-spawn 'pq-reset-queue nil ',peek-pane)
			:documentation (:MOUSE-L-1 "3Throw away all pending print requests.*"))
		   :string "3Reset Queue*" 14))
	(tv:scroll-parse-item "3  *"
	  `(:mouse (nil :eval (pq-spawn 'pq-cancel-block nil ',peek-pane)
			:documentation (:MOUSE-L-1 "3Give up on printing the current job.*"))
		   :string "3Cancel Multiple Jobs*" 23))
	(tv:scroll-parse-item "")
	(tv:scroll-parse-item "")
	(tv:scroll-parse-item
	  "3Current Print Request:  *"
	  `(:FUNCTION pq-file-percent (,peek-pane) nil ("3~A*"))
	  )
	(pq-header-line-entry)
	(tv:scroll-parse-item "")
	(tv:scroll-maintain-list
	  #'(lambda ()
	      (sys:without-interrupts
		(let* ((pq-desc (pq-desc peek-pane)))
		  (snarf-queue pq-desc)
		  (if (pq-desc-queue pq-desc)
		      (list (car (pq-desc-queue pq-desc)))
		      '("3none*")))))
	  #'(lambda (print-request)
	      (if (stringp print-request)
		  (tv:scroll-parse-item print-request)
		  (scroll-parse-print-request peek-pane print-request nil)))
	  )
	(tv:scroll-parse-item "")
	(tv:scroll-parse-item "")
	(tv:scroll-parse-item "3Pending Print Requests:*")
	(pq-header-line-entry)
	(tv:scroll-parse-item "")
	(tv:scroll-maintain-list
	  #'(lambda ()
	      (let* ((pq-desc (pq-desc peek-pane)))
		(cdr (pq-desc-queue pq-desc))))
	  #'(lambda (print-request) (scroll-parse-print-request peek-pane print-request))
	  )
	(tv:scroll-parse-item "")
	))


(defun 4pq-complete-redisplay *(peek-pane)
  "2Completely recalculate the scroll-items of the Print Queue Editor.*"
  (send peek-pane :set-display-item (pq-entries peek-pane)))


;1;; For *CHOOSE-VARIABLE-VALUES1.*
;1;;*
(defprop 4:HOST* (princ cvv-read-host) w:choose-variable-values-keyword)
(defprop 4:T-OR-NIL-OR-STRING* (princ cvv-read-t-or-nil-or-string) w:choose-variable-values-keyword)

(defun 4cvv-read-host* (stream)
  "2Read in and validate a host name.*"
  (let* ((string (read-line stream)))
    (net:parse-host string)))

(defun 4cvv-read-t-or-nil-or-string* (stream)
  "2Read a line which is T, NIL, or else a string.*"
  (let* ((string (read-line stream))
	 (trim (string-trim '(#\Space #\Tab) string)))
    (cond ((string-equal trim "3T*") t)
	  ((string-equal trim "3NIL*") nil)
	  (t string))))


;1;;; The mouse-invoked commands.*


(defun 4pq-change-sender* (print-request peek-pane)
  "2Let the user do some editing with *CHOOSE-VARIABLE-VALUES2, and ship the results to the remote server.*"
  (let* ((*print-base* 10)
	 (*read-base* 10)
	 (*nopoint t)
	 (new-sender (send print-request :user-name))
	 (new-host (send print-request :sender-host)))
    (declare (special new-sender new-host))
    (w:choose-variable-values
      '((new-sender "3User ID*"
		    :documentation "3Mark this print request as having been queued by a different User-ID.*" :STRING)
	(new-host   "3Host:*"
		    :documentation "3Mark this print request as having been queued from a different host.*" :HOST)
	)
      :label (format nil "3Origin of print request ~A*" (send print-request :file-name))
      :margin-choices `(,w:margin-choice-completion-string
			(,w:margin-choice-abort-string (signal-condition eh:*abort-object*))))
    (change-request-sender print-request (pq-desc peek-pane) new-sender new-host)
    print-request))


(defun 4pq-change-printer* (print-request peek-pane)
  "2Let the user select a new printer, and ship the results to the remote server.*"
  (let* ((objects (remove-if #'(lambda (x) (getf (third x) :*ALIAS-OF*))
			     (name:list-objects-from-attributes :class :printer)))
	 (menu-list (mapcar #'(lambda (list)
				(let* ((name (car list))
				       (host (getf (third list) :HOST)))
				  `(,(format nil "3~A on host ~A*" name (net:parse-host host))
				    :value ,name
				    :documentation "3Select a printer.*")))
			    objects))
	 (new-printer-name (w:menu-choose menu-list :label "3Select a printer*")))
    (when new-printer-name
      (change-request-device print-request (pq-desc peek-pane) new-printer-name))
    print-request))


(defun 4pq-toggle-header* (print-request peek-pane)
  "2Toggle the header flag of the request, and tell the remote server about it.*"
  (change-request-header print-request (pq-desc peek-pane) (not (send print-request :header)))
  print-request)


(defun 4pq-change-copies* (print-request peek-pane)
  "2Let the user do some editing with *CHOOSE-VARIABLE-VALUES2, and ship the results to the remote server.*"
  (let* ((*print-base* 10)
	 (*read-base* 10)
	 (*nopoint t)
	 (new-copies (send print-request :copies)))
    (declare (special new-copies))
    (w:choose-variable-values
      '((new-copies "3Copies*" :documentation "3Change the number of copies of this document printed.*"
		    :POSITIVE-FIXNUM))
      :label (format nil "3Copies to print of print request ~A*" (send print-request :file-name))
      :margin-choices `(,w:margin-choice-completion-string
			(,w:margin-choice-abort-string (signal-condition eh:*abort-object*))))
    (change-request-copies print-request (pq-desc peek-pane) new-copies)
    print-request))


(defun 4pq-change-font-size* (print-request peek-pane)
  "2Let the user do some editing with *CHOOSE-VARIABLE-VALUES2, and ship the results to the remote server.*"
  (let* ((*print-base* 10)
	 (*read-base* 10)
	 (*nopoint t)
	 (new-cpi (send print-request :CPI))
	 (new-lpi (send print-request :LPI))
	 (new-lines (send print-request :LINES)))
    (declare (special new-cpi new-lpi new-lines))
    (w:choose-variable-values
      '((new-cpi   "3Characters per Inch*" :DECIMAL)
	(new-lpi   "3Lines per Inch*" :DECIMAL)
	(new-lines "3Lines per Page*" :POSITIVE-FIXNUM))
      :label (format nil "3Change the font and page size of print request ~A*" (send print-request :file-name))
      :margin-choices `(,w:margin-choice-completion-string
			(,w:margin-choice-abort-string (signal-condition eh:*abort-object*))))
    (change-request-font-size print-request (pq-desc peek-pane) new-cpi new-lpi new-lines)
    print-request))


(defun 4pq-change-page-heading* (print-request peek-pane)
  "2Let the user do some editing with *CHOOSE-VARIABLE-VALUES2, and ship the results to the remote server.*"
  (let* ((*print-base* 10)
	 (*read-base* 10)
	 (*nopoint t)
	 (new-heading (send print-request :page-heading)))
    (declare (special new-heading))
    (w:choose-variable-values
      '((new-heading  "3Page Heading*" :T-OR-NIL-OR-STRING))
      :label (format nil "3Change the page heading of print request ~A.~%~
                          Must be T, meaning default heading, NIL, meaning no heading, or a string.*"
		     (send print-request :file-name))
      :extra-width 40
      :margin-choices `(,w:margin-choice-completion-string
			(,w:margin-choice-abort-string (signal-condition eh:*abort-object*))))
    (change-request-page-heading print-request (pq-desc peek-pane) new-heading)
    print-request))



(defun 4pq-change-queue* (ignore peek-pane)
  "2Select a host to display the print queue of, and connect to it.*"
  ;1;*
  ;1; Go to the network namespace to find the *NAME:OBJECT1s of all of the known printers.*
  (let* ((objects (remove-if #'(lambda (x) (getf (third x) :*ALIAS-OF*))
			     (name:list-objects-from-attributes :class :printer)))
	 list)
    (dolist (obj objects)
      (let* ((printer-name (car obj))
	     (host (getf (third obj) :HOST)))
	(when host
	  ;1;*
	  ;1; When the printer has a host, add it to the list.*
	  ;1; Elements of *LIST1 are of the form ( <2host*> <2printer-1*> <2printer-2*> ... ) to handle hosts with multiple printers.*
	  (setq host (net:parse-host host))
	  (let* ((cons (assoc host list :test #'eq)))
	    (if cons
		(push printer-name (cdr cons))
		(push (list host printer-name) list))))))
    (let* (menu-list)
      ;1;*
      ;1; Iterate over *LIST1 to build something suitable for passing to *W:MENU-CHOOSE1.*
      (dolist (cons list)
	(let* ((host (car cons))
	       (printers (cdr cons))
	       (plural-p (cdr printers))
	       (local-p (eq host si:local-host))
	       (string (if plural-p
			   (format nil "3~A, with printers ~A~{, ~A~}*" host (car printers) (cdr printers))
			   (format nil "3~A, with printer ~A*" host (car printers)))))
	  (when local-p (setq string (string-append "3Local Host *" string)))
	  (push `(,string :value ,host :documentation "3View the print queue of this host.*") menu-list)))
      (setq menu-list (nreverse menu-list))
      ;1;*
      ;1; If the local machine has a printer attached to it, it is already on the list.  But we want it to be there even if it doesn't have a printer.*
      (unless (member (net:parse-host "3LM*") list :key #'car)
	(push '("3Local Machine*" :value nil :documentation "3View the print queue of the local machine.*") menu-list))
      
      (multiple-value-bind (value selected-p) (w:menu-choose menu-list :label "3Select a machine:*")
	(when selected-p
	  (when (eq value (net:parse-host "3LM*")) (setq value nil))
	  (send peek-pane :force-kbd-input `(PQ-CONN ,value))
	  )))))


(defun 4pq-cancel* (ignore peek-pane)
  "2Tell the remote print server to cancel the current job.*"
  (cancel-current-job (pq-desc peek-pane)))

(defun 4pq-reset-queue* (ignore peek-pane)
  "2Tell the remote print server to cancel all pending jobs.*"
  (delete-all-queued (pq-desc peek-pane)))

(defun 4pq-cancel-block* (ignore peek-pane)
  "2Tell the remote print server to cancel a group of jobs chosen with *W:CHOOSE-VARIABLE-VALUES2.*"
  (let* ((*print-base* 10)
	 (*read-base* 10)
	 (*nopoint t)
	 (start 0) (end 0)
	 (pq-desc (pq-desc peek-pane)))
    (declare (special start end))
    (sys:without-interrupts
      (when (cdr (pq-desc-queue pq-desc))
	(setq start (send (cadr (pq-desc-queue pq-desc)) :request-no))
	(setq end (send (car (last (pq-desc-queue pq-desc))) :request-no))))
    (w:choose-variable-values
      '((start "3Starting ID*" :POSITIVE-FIXNUM)
        (end "3Ending ID*" :POSITIVE-FIXNUM))
      :label "3Type the range of Print Request IDs to delete.*"
      :margin-choices `(,w:margin-choice-completion-string
			(,w:margin-choice-abort-string (signal-condition eh:*abort-object*))))
    (when (> start end) (rotatef start end))
    (delete-block pq-desc start end)
    (values start end)))


(defun 4pq-item-menu* (print-request peek-pane)
  "2Pop up a menu of commands for operating on a print-request.*"
  (let* ((option (w:menu-choose
		   '(("3Request to Front*" :value :FRONT :documentation "3Move this request to the front of the queue.*")
		     ("3Request to Back*" :value :BACK :documentation "3Move this request to the end of the queue.*")
		     ("3Delete Request*" :value :DELETE :documentation "3Remove this print request.*"))
		   :label (format nil "3~D: ~A*"
				  (send print-request :request-no)
				  (namestring (send print-request :file-name)))
		   :columns 1))
	 (pq-desc (pq-desc peek-pane)))
    (ecase option
      (:FRONT (request-to-front print-request pq-desc))
      (:BACK  (request-to-back print-request pq-desc))
      (:DELETE (delete-request print-request pq-desc))
      (NIL nil)))
  print-request)



;1;; Functions for changing the state of a request locally and on the foreign host.*
;1;; These functions operate only on the contents of a request, not on its position in the queue.*
;1;; *


(defun 4change-request-sender* (print-request pq-desc new-sender new-host)
  ;1; Change it locally.*
  (sys:without-interrupts
    (setf (send print-request :user-name) new-sender)
    (setf (send print-request :sender-host) new-host))
  ;1; Change it remotely, if necessary.*
  (with-conn (s pq-desc "3SENDER*" print-request)
    (write-string new-sender s)
    (write-char #\@ s)
    (princ new-host s)
    (terpri s))
  print-request)

(defun 4change-request-device* (print-request pq-desc new-printer-name)
  ;1; Change it locally.*
  (sys:without-interrupts
    (setf (send print-request :print-device) (printer:get-printer-device new-printer-name)))
  ;1; Change it remotely, if necessary.*
  (with-conn (s pq-desc "3DEVICE*" print-request)
    (write-line new-printer-name s))
  print-request)

(defun 4change-request-header* (print-request pq-desc new-value)
  ;1; Change it locally.*
  (sys:without-interrupts
    (setf (send print-request :header) new-value))
  ;1; Change it remotely, if necessary.*
  (with-conn (s pq-desc "3HEADER*" print-request)
    (write-line (if new-value "3Y*" "3N*") s))
  print-request)

(defun 4change-request-copies* (print-request pq-desc new-copies)
  ;1; Change it locally.*
  (sys:without-interrupts
    (setf (send print-request :copies) new-copies))
  ;1; Change it remotely, if necessary.*
  (with-conn (s pq-desc "3COPIES*" print-request)
    (princ new-copies s)
    (terpri s))
  print-request)

(defun 4change-request-font-size* (print-request pq-desc new-cpi new-lpi new-lines)
  ;1; Change it locally.*
  (sys:without-interrupts
    (setf (send print-request :cpi) new-cpi
	  (send print-request :lpi) new-lpi
	  (send print-request :lines) new-lines))
  ;1; Change it remotely, if necessary.*
  (with-conn (s pq-desc "3SIZE*" print-request)
    (format s "3~D~%~D~%~D~%*" new-cpi new-lpi new-lines))
  print-request)

(defun 4change-request-page-heading* (print-request pq-desc new-heading)
  ;1; Change it locally.*
  (sys:without-interrupts
    (setf (send print-request :page-heading) new-heading))
  ;1; Change it remotely, if necessary.*
  (with-conn (s pq-desc "3PHEADING*" print-request)
    (prin1 new-heading s)
    (terpri s)))


;1;; Functions for changing the position of a request in the queue, both locally and on the foreign host.*
;1;; *

(defun 4delete-request* (print-request pq-desc)
  (let (local-p)
    (sys:without-interrupts
      (setq local-p (null (pq-desc-host pq-desc)))
      (when (and local-p printer:*print-queue*)
	(setf (cdr printer:*print-queue*)
	      (delete print-request (cdr printer:*print-queue*) :test #'eq))))
    (when (pq-desc-queue pq-desc)
      (setf (cdr (pq-desc-queue pq-desc))
	    (delete print-request (cdr (pq-desc-queue pq-desc)) :test #'eq)))
    (unless local-p
      (with-conn (s pq-desc "3DELETE*" print-request))))
  print-request)


(defun 4request-to-front* (print-request pq-desc)
  (let (local-p)
    (sys:without-interrupts
      (setq local-p (null (pq-desc-host pq-desc)))
      (when (and local-p printer:*print-queue*)
	(setf (cdr printer:*print-queue*)
	      (cons print-request
		    (delete print-request (cdr printer:*print-queue*) :test #'eq)))))
    (when (pq-desc-queue pq-desc)
      (setf (cdr (pq-desc-queue pq-desc))
	    (cons print-request
		  (delete print-request (cdr (pq-desc-queue pq-desc)) :test #'eq))))
    (unless local-p
      (with-conn (s pq-desc "3TOFRONT*" print-request))))
  print-request)


(defun 4request-to-back* (print-request pq-desc)
  (let (local-p)
    (sys:without-interrupts
      (setq local-p (null (pq-desc-host pq-desc)))
      (when (and local-p printer:*print-queue*)
	(setf (cdr printer:*print-queue*)
	      (append (delete print-request (cdr printer:*print-queue*) :test #'eq)
		      (cons print-request nil)))))
    (when (pq-desc-queue pq-desc)
      (setf (cdr (pq-desc-queue pq-desc))
	    (append (delete print-request (cdr (pq-desc-queue pq-desc)) :test #'eq)
		    (cons print-request nil))))
    (unless local-p
      (with-conn (s pq-desc "3TOBACK*" print-request))))
  print-request)


(defun 4delete-all-queued* (pq-desc)
  (let (local-p)
    (sys:without-interrupts
      (setq local-p (null (pq-desc-host pq-desc)))
      (when (and local-p printer:*print-queue*)
	(setf (cdr printer:*print-queue*) nil)))
    (when (pq-desc-queue pq-desc)
      (setf (cdr (pq-desc-queue pq-desc)) nil))
    (unless local-p
      (with-conn (s pq-desc "3CLEAR*"))))
  pq-desc)


(defun 4cancel-current-job* (pq-desc)
  (let (local-p)
    (sys:without-interrupts
      (setq local-p (null (pq-desc-host pq-desc)))
      (when local-p
	(send printer:*print-daemon* :reset)))
    (unless local-p
      (with-conn (s pq-desc "3CANCEL*"))))
  pq-desc)


(defun 4delete-block* (pq-desc start end)
  (let (local-p)
    (sys:without-interrupts
      (setq local-p (null (pq-desc-host pq-desc)))
      (when (and local-p printer:*print-queue*)
	(setf (cdr printer:*print-queue*)
	      (delete-if #'(lambda (print-request)
			     (<= start (send print-request :request-no) end))
			 (cdr printer:*print-queue*)))))
    (when (pq-desc-queue pq-desc)
      (setf (cdr (pq-desc-queue pq-desc))
	    (delete-if #'(lambda (print-request)
			   (<= start (send print-request :request-no) end))
		       (cdr (pq-desc-queue pq-desc)))))
    (unless local-p
      (with-conn (s pq-desc "3DELBLOCK*")
	(format s "3~D~%~D~%*" start end))))
  pq-desc)


;1;; Functions for querying the remote server about the state and contents of the queue.*
;1;; *

(defun 4get-remote-ids* (pq-desc)
  "2Returns an ordered list of the IDs of the requests on the remote host's queue.*"
  (let* ((stream (pq-desc-stream pq-desc)))
    (format stream "3IDS~%*")
    (send stream :force-output)
    (read stream)))


(defun 4get-request* (id pq-desc)
  "2Make a new request object, which is just like the request numbered ID on the remote host.*"
  (let* ((stream (pq-desc-stream pq-desc)))
    (format stream "3GET~%~D~%*" id)
    (send stream :force-output)
    (eval (read stream))))


(defun 4get-request-info *(pq-desc)
  "2Returns two values: the number of bytes read and the percent read of the file at the front of the queue.
  If there is no file, or it is not open, returns two zeros.*"
  (declare (values bytes-read percent-read))
  (cond ((null (pq-desc-host pq-desc))  ;1 Local machine*
	 (let* ((request (car printer:*print-queue*))
		(pathname (and request (send request :send-if-handles :file-name)))
		p d (bytes nil) (percent nil))
	   (when pathname (multiple-value-setq (p d bytes percent) (pathname-who-line-info pathname)))
	   (values (or bytes 0) (or percent 0))))
	(t
	 (with-conn (s pq-desc "3INFO*")
	   (send s :force-output)
	   (let* ((bytes (parse-integer (read-line s)))
		  (percent (parse-integer (read-line s))))
	     (values bytes percent))))))


(defvar *debugging-print-queue-editor*4 *nil
 "2If T, then the Print Queue Client process will signal errors instead of just NOTIFYing about them and resetting.*")


(defun 4print-queue-client-top-level *(pq-desc)
  "2This is the heart of the Print Queue Client process.  It loops forever.
   First, go to the remote contact and make our copy of the print queue look just like the remote one.
   Then arrest this process (that is, *TV:CURRENT-PROCESS2) so that we stop running.
   This means that each time this process is sent the *:REVOKE-ARREST-REASON2 message, it runs through
   the its loop once and then stops until it is unarrested again.*"
  (loop
    (let* ((completed-normally nil))
      (unwind-protect
	  (block MAIN
	    ;1;*
	    ;1; If an error occurs during *PRINT-QUEUE-CLIENT-INTERNAL1, then set the *IN-ERROR-P1 flag before it is signalled so that*
	    ;1; the display updates even if the user doesn't handle the error.*
	    (condition-bind ((SYS:ERROR #'(lambda (ignore pqd)
					    (setf (pq-desc-connection-status pqd) :ERROR)
					    nil)
					pq-desc))
	      (progn
		(condition-call-if (not *debugging-print-queue-editor*) (condition)
		    (print-queue-client-internal pq-desc)
		  (T
		   (tv:notify nil "3Process ~A got an error:~%~A.*" (send tv:current-process :name) condition)
		   (return-from MAIN)))
		
		(setq completed-normally t)))
	    
	    (send TV:CURRENT-PROCESS :arrest-reason :PRINT-QUEUE-WAIT))
	(unless (or completed-normally
		    (pq-desc-exiting-p pq-desc)
		    (pq-desc-in-error-p pq-desc))
	  ;1; If we get here, then *PRINT-QUEUE-CLIENT-INTERNAL1 got an error.  Ditch the connection.*
	  ;1; Don't do it if the connection is currently being closed, or another process is handling the error.*
	  (close-pq-conn pq-desc :ERROR))))
    ))



(defun 4print-queue-client-internal* (pq-desc)
  "2Refresh the local cache of the print queue on the remote host (or the local host, if LM...)*"
  (let* ((local-p (null (pq-desc-host pq-desc))))
    
    (when (or local-p (pq-desc-stream pq-desc))  ;1 If we have a host and no stream, we're not really attached yet.  Give up.*
      ;1;
        *;1; Get the queue.
        *;1;*
      (cond (local-p
	     (sys:without-interrupts
	       (setf (pq-desc-queue pq-desc) (copy-list printer:*print-queue*))))	;1 ## maybe optimize.*
	    (t
	     (let* ((remote-ids (get-remote-ids pq-desc))
		    (cache (pq-desc-queue pq-desc))
		    (new-cache '()))
	       (dolist (id remote-ids)
		 (let* ((existing (find id cache :key #'(lambda (x) (send x :request-no)))))
		   (if existing
		       (push existing new-cache)
		       (let* ((r (get-request id pq-desc)))
			 (when r (push r new-cache))))))
	       (setq new-cache (nreverse new-cache))
	       (setf (pq-desc-queue pq-desc) new-cache))))
      (setf (pq-desc-connection-status pq-desc) nil)
1        *;1;
        *;1; Get the file info.
        *;1;*
      (multiple-value-bind (bytes percent) (get-request-info pq-desc)
	(if (consp (pq-desc-file-info pq-desc))
	    (setf (first (pq-desc-file-info pq-desc)) bytes
		  (second (pq-desc-file-info pq-desc)) percent)
	    (setf (pq-desc-file-info pq-desc) (list bytes percent))))))
  pq-desc)


(defun 4snarf-queue *(pq-desc)
  "2Wake up the Print Queue Client process of this Remote Print Queue Descriptor to refresh our image of the remote print queue.*"
  (unless (pq-desc-connection-process pq-desc)
    (let* ((proc (make-process "3Print Queue Client*")))
      (process-preset proc 'PRINT-QUEUE-CLIENT-TOP-LEVEL pq-desc)
      (send proc :arrest-reason :PRINT-QUEUE-WAIT)  ;1 Create it idle.*
      (send proc :run-reason :ENABLE)
      (setf (pq-desc-connection-process pq-desc) proc)))
  (send (pq-desc-connection-process pq-desc) :revoke-arrest-reason :PRINT-QUEUE-WAIT))


(defun 4pathname-who-line-info *(pathname)
  "2Returns the Who-line information for the first open stream to PATHNAME, or NIL.*"
  (declare (values pathname direction bytes percent))
  (setq pathname (pathname pathname))
  (dolist (stream (send tv:who-line-file-state-sheet :open-streams))
    (let* ((sp (send stream :pathname)))
      (when (and sp (equal pathname (pathname sp)))
	(return (send stream :who-line-information))))))



;1;; Remote Server.  This is the code that listens to commands for queue manipulation from *
;1;; the controlling host.*


(defun 4dumb-print* (obj stream)
  "2for printing a sexpr in a more compact (yet still READable) form than a *PRIN12 or* PPRINT2.
  Does nothing more than collapse *QUOTE2 into one character.*"
  (typecase obj
    (LIST
     (cond ((and (eq (car obj) 'QUOTE)
		 (eql (length obj) 2))
	    (princ #\' stream)
	    (dumb-print (second obj) stream))
	   (t
	    (princ #\( stream)
	    (dolist (x obj)
	      (dumb-print x stream))
	    (princ #\) stream))))
    (t (prin1 obj stream)
       (princ #\Space stream))))


(defun 4dump-request* (print-request stream)
  "2Write a readable form of PRINT-REQUEST onto STREAM.*"
  (let* ((list (send print-request :fasd-form)) ;1 Get an EVALable form for reproducing the request.*
	 (*package* (find-package "3LISP*"))
	 (*print-base* 10)
	 (*print-array* t)
	 (*print-pretty* nil))
    (dumb-print list stream)))


(defun 4dump-queue* (stream &optional ids-only)
  "2Dump a readable form of the entire print queue to *STREAM2.
  If* IDS-ONLY2 is T, then write a list of id-numbers.
  Otherwise, write a list of forms to reproduce the request objects.*"
  (princ #\( stream)
  (dolist (x (sys:without-interrupts (copy-list printer:*print-queue*)))
    (if ids-only
	(princ (send x :request-no) stream)
	(dump-request x stream))
    (terpri stream))
  (princ #\) stream)
  nil)


(defun 4service-request* (request stream)
  "2REQUEST is a keyword, the action requested.  STREAM is where it came from, and where to send the reply.*"
  (ecase request
    (:IDS
     (dump-queue stream t))
    
    (:GET
     (let* ((id (parse-integer (read-line stream)))
	    (request (find id printer:*print-queue* :key #'(lambda (x) (send x :request-no)))))
       (if request
	   (dump-request request stream)
	   (princ "3()*" stream))))
    
    (:INFO
     (let* ((request (car printer:*print-queue*))
	    (pathname (and request (send request :send-if-handles :file-name)))
	    p d
	    (bytes nil)
	    (percent nil))
       (when pathname (multiple-value-setq (p d bytes percent) (pathname-who-line-info pathname)))
       (format stream "3~A~%~A~%*" (or bytes 0) (or percent 0))))
    
    (:DELETE
     (let* ((id (parse-integer (read-line stream))))
       (printer:cancel-print-request id nil)))

    (:SENDER
     (let* ((id (parse-integer (read-line stream)))
	    (string (read-line stream))
	    (at (position #\@ string :test #'char=))
	    (user (subseq string 0 at))
	    (host (net:parse-host (subseq string (1+ at))))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (send request :user-name) user
		 (send request :sender-host) host)))))

    (:DEVICE
     (let* ((id (parse-integer (read-line stream)))
	    (printer-name (read-line stream))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (send request :print-device) (printer:get-printer-device printer-name))))))

    (:HEADER
     (let* ((id (parse-integer (read-line stream)))
	    (header (read-line stream))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (let* ((real-header (cond ((string-equal header "3Y*") t)
				     ((string-equal header "3N*") nil)
				     (t (error "3Header was not Y or N.*")))))
	     (setf (send request :header) real-header))))))
    
    (:PHEADING
     (let* ((id (parse-integer (read-line stream)))
	    (heading (read stream))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (send request :page-heading) heading)))))
    
    (:COPIES
     (let* ((id (parse-integer (read-line stream)))
	    (copies (parse-integer (read-line stream)))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (send request :copies) copies)))))

    (:SIZE
     (let* ((id (parse-integer (read-line stream)))
	    (cpi (read-from-string (read-line stream)))  ;1 these might not be integers.*
	    (lpi (read-from-string (read-line stream)))
	    (lines (read-from-string (read-line stream)))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (send request :cpi) cpi
		 (send request :lpi) lpi
		 (send request :lines) lines)))))

    (:DELBLOCK
     (let* ((start (parse-integer (read-line stream)))
	    (end (parse-integer (read-line stream))))
       (sys:without-interrupts
	 (when printer:*print-queue*
	   (setf (cdr printer:*print-queue*)
		 (delete-if #'(lambda (print-request)
				(<= start (send print-request :request-no) end))
			    (cdr printer:*print-queue*)))))))
    
    (:CLEAR
     (sys:without-interrupts (when printer:*print-queue* (setf (cdr printer:*print-queue*) nil))))

    (:CANCEL
     (send printer:*print-daemon* :reset))

    (:TOFRONT
     (let* ((id (parse-integer (read-line stream)))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (cdr printer:*print-queue*)
		 (cons request
		       (delete request (cdr printer:*print-queue*) :test #'eq)))))))
    
    (:TOBACK
     (let* ((id (parse-integer (read-line stream)))
	    request)
       (sys:without-interrupts
	 (setq request (find id (cdr printer:*print-queue*) :key #'(lambda (x) (send x :request-no))))
	 (when request
	   (setf (cdr printer:*print-queue*)
		 (append (delete request (cdr printer:*print-queue*) :test #'eq)
			 (cons request nil)))))))
    )
  (send stream :force-output)
  request)


(add-initialization "3REMOTE-PRINT-QUEUE*"
		   '(process-run-function "3Remote Print Queue Server*" 'remote-print-queue-server)
		   () 'chaos:server-alist)


(defun 4remote-print-queue-server* ()
  "2Listens for a connection to the Remote Print Queue Server, and loops until the client hangs up.*"
  (let* ((conn (chaos:listen "3REMOTE-PRINT-QUEUE*"))
	 (USER-ID "3PRINTER*")
	 chaos-stream)
    (chaos:accept conn)
    (condition-call (condition)
	(unwind-protect
	    (progn
	      (setq chaos-stream (chaos:make-stream conn))
	      (loop
		(let* ((keyword (intern (read-line chaos-stream) "3KEYWORD*")))
		  (if (eq keyword :BYE)
		      (return-from REMOTE-PRINT-QUEUE-SERVER nil)
		      (service-request keyword chaos-stream)))))
	  (when chaos-stream (send chaos-stream :finish))
	  (when conn (chaos:close-conn conn)))
1        *;1;*
      ;1; Trap errors which occur while trying to close a lost connection.*
      ((condition-typep condition 'NET:4BAD-CONNECTION-STATE*)
       nil)))
  nil)
